/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include "layer.h"
#include "typ_msg.h"
#include "typ_32.h"
#include "nbgethost.h"

/* is a t_hash_cmp_proc.
   It compares 2 internet addresses. */
int			inaddr_cmp(addr1,addr2)
struct in_addr		*addr1;
struct in_addr		*addr2;
{
  t_u32			saddr1;
  t_u32			saddr2;
  
  saddr1 = (t_u32)(UNSAFE_HTONL(addr1->s_addr));
  saddr2 = (t_u32)(UNSAFE_HTONL(addr2->s_addr));
  return ((int)(saddr1 - saddr2));
}

/* converts a string to an internet address.
   It first tries to do an inet_addr(3) call, if it fails, it uses
   gethostbyname(3).
   Note that mba_inaddr must be aligned!
   Returns 0 if OK. Returns -ERR_MALFORMED if the number of dots
   in str is != 3. Returns -ERR_UNRESOLVABLE if both inet_addr(3)
   and gethostbyname(3) fail. Returns -ERR_NOMETHOD if address family
   is not AF_INET */
t_status		inaddr_from_str(str,mba_inaddr,resolve)
char			*str;		/* A.b.c.d or a host name */
struct in_addr		*mba_inaddr;	/* An aligned in_addr struct pointer */
t_boolean		resolve;	/* If FALSE,bypasses gethostbyname(3)
					   but doesn't return an error if
					   inet_addr(3) has returned -1 */
{
  struct hostent	*hostentry;

  if (!resolve)
    {
      if (indexcount(str,'.') != 3)
	return (-ERR_MALFORMED);
    }
  mba_inaddr->s_addr = inet_addr(str);
  if (mba_inaddr->s_addr == -1)
    {
      if (resolve)
	{
	  if ((hostentry = gethostbyname(str)) == NULL)
	    return (-ERR_UNRESOLVABLE);
	  if (hostentry->h_addrtype != AF_INET)
	    return (-ERR_NOMETHOD);
	  FBCOPY(hostentry->h_addr_list[0],mba_inaddr,hostentry->h_length);
	}
    }
  return (0);
}

/* converts-and-catenates an internet address to a bridled string.
   It uses gethostbyaddr(3) and reverts to inet_ntoa(3) if it fails.
   Note that mba_addr must be aligned!
   Returns 0 if OK, might return -ERR_BO on buffer overflow */
t_status		inaddr_to_str(mba_inaddr,str,max_len,resolve)
struct in_addr		*mba_inaddr;	/* Aligned in_addr struct pointer */
char			*str;		/* Valid string */
int			max_len;	/* Maximum length */
t_boolean		resolve;      /* If FALSE, bypasses gethostbyaddr(3) */
{
  struct hostent	*hostentry;
  
  if (resolve)
    if ((hostentry = gethostbyaddr((char *)mba_inaddr,
				   sizeof (struct in_addr),
				   AF_INET)) != NULL)
      return (str_cat_str(str,max_len,hostentry->h_name));
  return (str_cat_str(str,max_len,inet_ntoa(*mba_inaddr)));
}

/* is a t_msg_proc.
   Manages ip addresses. */
t_status		typ_inaddr_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("inaddr");
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	struct in_addr	inaddr;

	if (ed->b.len < sizeof (inaddr))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&inaddr,sizeof (struct in_addr));
	return (inaddr_to_str(&inaddr,
			      bs->str,
			      bs->max_len,
			      FALSE));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	struct in_addr	inaddr;
	t_status	status;	

	if (ed->b.len < sizeof (inaddr))
	  return (-ERR_TRUNC);
	if ((status = inaddr_from_str(str,
				      &inaddr,
				      FALSE)) < 0)
	  return (status);
	FBCOPY(&inaddr,ed->b.buf,sizeof (struct in_addr));
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages ip addresses from/to host names or A.B.C.D form.

   Warning: It modifies the SIGARLM signal handler for the process.
   It is intended to use with care! */
t_status		typ_inaddr_resolved_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("inaddr_resolved");
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	struct in_addr	inaddr;

	if (ed->b.len < sizeof (inaddr))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&inaddr,sizeof (struct in_addr));
	return (inaddr_to_str_alarm(&inaddr,
				    bs->str,
				    bs->max_len,
				    (t_boolean)layer_resolve));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	struct in_addr	inaddr;
	t_status	status;	

	if (ed->b.len < sizeof (inaddr))
	  return (-ERR_TRUNC);
	if ((status = inaddr_from_str_alarm(str,
					    &inaddr,
					    TRUE)) < 0)
	  return (status);
	FBCOPY(&inaddr,ed->b.buf,sizeof (struct in_addr));
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	
