/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "lay_tcp.h"
#include "lay_ip.h"
#include "lay_data.h"
#include "lay_tcpopt.h"
#include "typ_8.h"
#include "typ_16.h"
#include "typ_32.h"
#include "typ_port.h"
#include "typ_seqack.h"

t_mask_def		tcpflags_mask_defs[] = 
{
  {"fin",		TCP_FIN},
  {"syn",		TCP_SYN},
  {"rst",		TCP_RST},
  {"push",		TCP_PUSH},
  {"ack",		TCP_ACK},
  {"urg",		TCP_URG},
  {NULL,		0}
};

t_bit_field		tcp_off_bit_field =
{
  0,3
};	

t_bit_field		tcp_x2_bit_field =
{
  4,7
};	

/* computes a tcp checksum.
   It is portable and deals with alignment.
   Returns the checksum. */
t_u16			tcp_cksum(ip)
t_ip			*ip;
{
   t_tcp		*tcp;
   t_u16		*sptr;
   t_u16		len;
   unsigned long	sum;
   int			i;
   
   tcp = (t_tcp *)(((char *)ip) + ip_get_hl(ip) * 4);
   sum = 0;
   sptr = (t_u16 *)(&(ip->src));
   i = 0;
   while (i < IP_ALEN)
     {
       t_u16		u16;

       FBCOPY(sptr,&u16,sizeof (u16)); /* ALIGNMENT */
       sum += UNSAFE_HTONS(u16);
       sptr++;
       i++;
     }
   sptr = (t_u16 *)tcp;
   len = UNSAFE_NTOHS(ip->len) - ip_get_hl(ip) * 4;
   sum += IP_PROTO_TCP + len;
   if (len % 2)
   {
      ((char *)tcp)[len] = 0;
      len += 1;
   }
   len >>= 1;
   i = 0;
   while (i < len)
     {
       t_u16		u16;

       FBCOPY(sptr,&u16,sizeof (u16)); /* ALIGNMENT */
       sum += UNSAFE_HTONS(u16);
       sptr++;
       i++;
     }
   sum = (sum >> 16) + (sum & 0xffff);
   sum += (sum >> 16);
   return ((t_u16)(~sum & 0xffff));
}

VOID_FUNC		tcp_compute_sum(ip)
t_ip			*ip;
{
   t_tcp		*tcp;
   t_u16		zero;

   tcp = (t_tcp *)(((char *)ip) + ip_get_hl(ip) * 4);
   zero = 0;
   FBCOPY(&zero,&tcp->sum,sizeof (zero)); /* ALIGNMENT */
   safe_htons(tcp_cksum(ip),&tcp->sum);
}

VOID_FUNC		tcp_set_proto(ip)
t_ip			*ip;
{
  ip->p = IP_PROTO_TCP;
}

VOID_FUNC		tcp_set_sport(tcp,sport)
t_tcp			*tcp;
int			sport;
{
  safe_htons(sport,&tcp->sport);
}

int			tcp_get_sport(tcp)
t_tcp			*tcp;
{
  return (safe_ntohs(&tcp->sport));
}	

VOID_FUNC		tcp_set_dport(tcp,dport)
t_tcp			*tcp;
int			dport;
{
  safe_htons(dport,&tcp->dport);
}

int			tcp_get_dport(tcp)
t_tcp			*tcp;
{
  return (safe_ntohs(&tcp->dport));
}	

VOID_FUNC		tcp_set_seq(tcp,seq)
t_tcp			*tcp;
t_tcp_seq		seq;
{
  safe_htonl(seq,&(tcp->seq));
}

t_tcp_seq		tcp_get_seq(tcp)
t_tcp			*tcp;
{
  return (safe_ntohl(&(tcp->seq)));
}

VOID_FUNC		tcp_set_ack(tcp,ack)
t_tcp			*tcp;
t_tcp_seq		ack;
{
  safe_htonl(ack,&(tcp->ack));
}

t_tcp_seq		tcp_get_ack(tcp)
t_tcp			*tcp;
{
  return (safe_ntohl(&(tcp->ack)));
}

VOID_FUNC		tcp_set_off(tcp,off)
t_tcp			*tcp;
int			off;
{
  bit_field_u8_set(&(tcp->offx2),
		   tcp_off_bit_field.from,
		   tcp_off_bit_field.to,
		   off);
}

int			tcp_get_off(tcp)
t_tcp			*tcp;
{
  t_u8			*offp;
  
  offp = bit_field_u8_get(&(tcp->offx2),
			  tcp_off_bit_field.from,
			  tcp_off_bit_field.to);
  return ((int)(*offp));
}

VOID_FUNC		tcp_set_x2(tcp,x2)
t_tcp			*tcp;
int			x2;
{
  bit_field_u8_set(&(tcp->offx2),
		   tcp_x2_bit_field.from,
		   tcp_x2_bit_field.to,
		   x2);
}

int			tcp_get_x2(tcp)
t_tcp			*tcp;
{
  t_u8			*x2p;
  
  x2p = bit_field_u8_get(&(tcp->offx2),
			 tcp_x2_bit_field.from,
			 tcp_x2_bit_field.to);
  return ((int)(*x2p));
}

VOID_FUNC		tcp_set_flags(tcp,flags)
t_tcp			*tcp;
int			flags;
{
  tcp->flags = (u_char)flags;
}

int			tcp_get_flags(tcp)
t_tcp			*tcp;
{
  return (tcp->flags);
}

VOID_FUNC		tcp_set_win(tcp,win)
t_tcp			*tcp;
int			win;
{
  safe_htons(win,&tcp->win);
}

int			tcp_get_win(tcp)
t_tcp			*tcp;
{
  return (safe_ntohs(&tcp->win));
}	

VOID_FUNC		tcp_set_sum(tcp,sum)
t_tcp			*tcp;
int			sum;
{
  safe_htons(sum,&tcp->sum);
}

int			tcp_get_sum(tcp)
t_tcp			*tcp;
{
  return (safe_ntohs(&tcp->sum));
}

VOID_FUNC		tcp_set_urp(tcp,urp)
t_tcp			*tcp;
int			urp;
{
  safe_htons(urp,&tcp->urp);
}

int			tcp_get_urp(tcp)
t_tcp			*tcp;
{
  return (safe_ntohs(&tcp->urp));
}	

t_status		tcp_off(buf,len,off)
char			*buf;
int			len;
t_off			*off;
{
  t_tcp			*tcp;

  LAYER_TCP_CHECK(tcp,buf,len);
  (*off) = tcp_get_off(tcp) * 4;
  return (0);
}

t_status		tcp_sub(buf,len,sub_mp)
char			*buf;
int			len;
t_msg_proc		*sub_mp;
{
  t_tcp			*tcp;

  LAYER_TCP_CHECK(tcp,buf,len);
  (*sub_mp) = &lay_data_msg;
  return (0);
}

t_status		tcp_sum(buf,len,up_buf,up_len)
char			*buf;
int			len;
char			*up_buf;
int			up_len;
{
  t_tcp			*tcp;
  t_ip			*ip;

  LAYER_TCP_CHECK(tcp,buf,len);
  LAYER_IP_CHECK(ip,up_buf,up_len);
  if (ip_get_len(ip) != up_len)
    return (-ERR_BADLEN);
  if (ip_get_hl(ip) * 4 < IP_MINHLEN)
    return (-ERR_BADHEADER);
  tcp_compute_sum(ip);
  return (0);
}

t_status		tcp_has_opt(buf,len,hsd)
char			*buf;
int			len;
t_has_opt_data		*hsd;
{
  t_tcp			*tcp;

  LAYER_TCP_CHECK(tcp,buf,len);
  if (tcp_get_off(tcp) * 4 > TCP_MINHLEN)
    {
      hsd->has_opt = TRUE;
      hsd->opt_off = TCP_MINHLEN;
      hsd->opt_len = tcp_get_off(tcp) * 4 - TCP_MINHLEN;
      hsd->opt_mp = lay_tcpopt_msg;
    }
  else
    {
      hsd->has_opt = FALSE;
    }
  return (0);
}

t_field				tcp_fields[] = 
{
  {"sport",	OFFSET(t_tcp *,sport),		typ_nu16_msg,		NULL},
  {"Sport",	OFFSET(t_tcp *,sport),		typ_port_resolved_msg,	"tcp"},
  {"dport",	OFFSET(t_tcp *,dport),		typ_nu16_msg,		NULL},
  {"Dport",	OFFSET(t_tcp *,dport),		typ_port_resolved_msg,	"tcp"},
  {"seq",	OFFSET(t_tcp *,seq),		typ_nu32_msg,		NULL},
  {"ack",	OFFSET(t_tcp *,ack),		typ_nu32_msg,		NULL},
  {"x2",	OFFSET(t_tcp *,offx2),
   typ_u8bitfield_msg,	(VOID_PTR)(&tcp_x2_bit_field)},
  {"off",	OFFSET(t_tcp *,offx2),
   typ_u8bitfield_msg,	(VOID_PTR)(&tcp_off_bit_field)},
  {"flags",	OFFSET(t_tcp *,flags),		typ_u8_msg,		NULL},
  {"Flags",	OFFSET(t_tcp *,flags),	 typ_u8mask_msg,  tcpflags_mask_defs},
  {"win",	OFFSET(t_tcp *,win),		typ_nu16_msg,		NULL},
  {"sum",	OFFSET(t_tcp *,sum),		typ_nu16_msg,		NULL},
  {"urp",	OFFSET(t_tcp *,urp),		typ_nu16_msg,		NULL},
  NULL_FIELD
};

char				*tcp_itmpl = "\n\
<!--tcp_itmpl-->\n\
<table width=100%%%% bgcolor=\"%%tcpColor%%\">\n\
<tr>\n\
<td width=100%%%%>\n\
<small>\n\
<a href=\"extract(tcp[%i%])\">[Extract]</a>\n\
<a href=\"trunc(tcp[%i%])\">[Trunc]</a>\n\
<a href=\"paste(tcp[%i%])\">[Paste]</a>\n\
<a href=\"adaptlen(tcp[%i%])\">[Adapt len]</a>\n\
<a href=\"sum(tcp[%i%])\">[Sum]</a>\n\
 tcp\n\
</small>\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=50%%%%>\n\
<a href=\"setfield(tcp[%i%].Sport)\">%%tcp[%i%].Sport%%</a>\n\
(<a href=\"setfield(tcp[%i%].sport)\">%%tcp[%i%].sport%%</a>)\n\
</td>\n\
<td align=center width=50%%%%>\n\
<a href=\"setfield(tcp[%i%].Dport)\">%%tcp[%i%].Dport%%</a>\n\
(<a href=\"setfield(tcp[%i%].dport)\">%%tcp[%i%].dport%%</a>)\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=100%%%%>\n\
<a href=\"setfield(tcp[%i%].Seq)\">%%tcp[%i%].Seq%%</a>\n\
(<a href=\"setfield(tcp[%i%].seq)\">%%tcp[%i%].seq%%</a>)\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=100%%%%>\n\
<a href=\"setfield(tcp[%i%].Ack)\">%%tcp[%i%].Ack%%</a>\n\
(<a href=\"setfield(tcp[%i%].ack)\">%%tcp[%i%].ack%%</a>)\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=25%%%%>\n\
<a href=\"setfield(tcp[%i%].x2)\">%%tcp[%i%].x2%%</a>\n\
</td>\n\
<td align=center width=25%%%%>\n\
<a href=\"setfield(tcp[%i%].off)\">%%tcp[%i%].off%%</a>\n\
</td>\n\
<td align=center width=25%%%%>\n\
<a href=\"setfield(tcp[%i%].Flags)\">%%tcp[%i%].Flags%%</a>\n\
(<a href=\"setfield(tcp[%i%].flags)\">%%tcp[%i%].flags%%</a>)\n\
</td>\n\
<td align=center width=25%%%%>\n\
<a href=\"setfield(tcp[%i%].win)\">%%tcp[%i%].win%%</a>\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=50%%%%>\n\
<a href=\"setfield(tcp[%i%].sum)\">%%tcp[%i%].sum%%</a>\n\
</td>\n\
<td align=center width=50%%%%>\n\
<a href=\"setfield(tcp[%i%].urp)\">%%tcp[%i%].urp%%</a>\n\
</td>\n\
</tr>\n\
</table>\n\
";

t_status			lay_tcp_msg(msg,arg1,arg2)
t_msg				msg;
VOID_PTR			arg1;
VOID_PTR			arg2;
{
  t_status			status;

  switch (msg)
    {
      LAY_CLASS_GENERIC;
      LAY_NAME_ID_GENERIC(&lay_tcp_msg,"tcp");
      LAY_GET_FIELDS_GENERIC(tcp_fields);
      LAY_GET_FIELD_TYP_GENERIC(tcp_fields);
      LAY_GET_ITMPL_GENERIC(&lay_tcp_msg,tcp_itmpl);
    case LAY_GET_FIELD:
      {
	LAY_GET_FIELD_ARGS(gfd,bs);
	t_field			*field;
	t_field			*fields;

	field = NULL;
	fields = tcp_fields;
	while (fields->name)
	  {
	    if (!strcmp(fields->name,gfd->field))
	      field = fields;
	    fields++;
	  }
	if (!strcmp(gfd->field,"Seq") || !strcmp(gfd->field,"Ack"))
	  {
	    t_extract_data	ed;
	    
	    ed.b.buf = gfd->b.buf + TCP_SEQACK_OFF;
	    ed.b.len = gfd->b.len - TCP_SEQACK_OFF;
	    if (gfd->optional_id)
	      {
		t_seq_ack_data	sad;
		t_hash_elt	*he;
		t_tcp		*tcp;
		t_off		off;

		LAYER_TCP_CHECK(tcp,gfd->b.buf,gfd->b.len);
		if ((status = tcp_off(gfd->b.buf,
				      gfd->b.len,
				      &off)) < 0)
		  return (status);
		if (he = id_get(gfd->optional_id,
				&lay_ip_src_chan))
		  FBCOPY(&he->value,&sad.src,sizeof (sad.src));
		else
		  goto noipadvice;
		if (he = id_get(gfd->optional_id,
				&lay_ip_dst_chan))
		  FBCOPY(&he->value,&sad.dst,sizeof (sad.dst));
		else
		  goto noipadvice;
		sad.sport = safe_ntohs(&tcp->sport);
		sad.dport = safe_ntohs(&tcp->dport);
		sad.flags = tcp->flags;
		sad.datalen = gfd->b.len - off;
		sad.seq_wanted = !strcmp(gfd->field,"Seq");
		ed.data = &sad;
	      }
	    else
	      {
	      noipadvice:
		ed.data = NULL;
	      }
	    return (typ_msg(typ_seqack_msg,
			    TYP_EXTRACT,
			    &ed,
			    bs));
	    return (0);
	  }
	if (!field)
	  return (-ERR_NOENT);
	return (get_field_to_str(gfd->b.buf,
				 gfd->b.len,
				 field,
				 bs->str,
				 bs->max_len));
      }
    case LAY_SET_FIELD:
      {
	LAY_SET_FIELD_ARGS(gfd,str);
	t_field			*field;
	t_field			*fields;
	
	field = NULL;
	fields = tcp_fields;
	while (fields->name)
	  {
	    if (!strcmp(fields->name,gfd->field))
	      field = fields; 
	    fields++;
	  }
	if (!strcmp(gfd->field,"Seq") || !strcmp(gfd->field,"Ack"))
	  return (-ERR_NI);
	if (!field)
	  return (-ERR_NOENT);
	return (set_field_from_str(gfd->b.buf,
				   gfd->b.len,
				   field,
				   str));
      }
    case LAY_OFF:
      {
	LAY_OFF_ARGS(b,off);
	
	return (tcp_off(b->buf,b->len,off));
      }
    case LAY_SUB:
      {
	LAY_SUB_ARGS(b,mp);

	return (tcp_sub(b->buf,b->len,mp));
      }
    case LAY_SUM:
      {
	LAY_SUM_ARGS(sd,unused);
	
	return (tcp_sum(sd->b.buf,
			sd->b.len,
			sd->up.buf,
			sd->up.len));
      }
    case LAY_HAS_OPT:
      {
	LAY_HAS_OPT_ARGS(b,hsd);
	
	return (tcp_has_opt(b->buf,b->len,hsd));
      }
    }
  return (-ERR_NOMETHOD);
}
