/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "layer.h"
#include "lay_ip.h"
#include "lay_icmp.h"
#include "lay_udp.h"
#include "lay_tcp.h"
#include "lay_data.h"
#include "lay_ipopt.h"
#include "typ_8.h"
#include "typ_16.h"
#include "typ_inaddr.h"
#include "typ_ipproto.h"

t_assoc			ipfrag_assocs[] =
{
  {"mf",		(VOID_PTR)IPFRAG_MF},
  {"df",		(VOID_PTR)IPFRAG_DF},
  {NULL,		NULL},
};

/*
TOS Value       Description                             Reference
---------       --------------------------              ---------
  0000          Default                                 [RFC1349]
  0001          Minimize Monetary Cost                  [RFC1349]
  0010          Maximize Reliability                    [RFC1349]
  0100          Maximize Throughput                     [RFC1349]
  1000          Minimize Delay                          [RFC1349]
  1111          Maximize Security                       [RFC1455]
  */

t_assoc			iptos_assocs[] = 
{
  {"default",			(VOID_PTR)0x0},
  {"minimize_monetary_cost",	(VOID_PTR)0x1},
  {"maximize_reliability",	(VOID_PTR)0x2},
  {"maximize_throughput",	(VOID_PTR)0x4},
  {"minimize_delay",		(VOID_PTR)0x8},
  {"maximize_security",		(VOID_PTR)0x17},
  {NULL,		NULL}
};

t_bit_field	ip_v_bit_field =
{
  0,3
};	

t_bit_field	ip_hl_bit_field =
{
  4,7
};	

t_bit_field	ip_frag_bit_field = 
{
  0,2
};

t_bit_field_assocs_data	ip_frag_bfad = 
{
  &ip_frag_bit_field,
  ipfrag_assocs
};

t_bit_field	ip_off_bit_field =
{
  3,15
};

/* computes an internet checksum.
   It is portable and deals with alignment.
   Returns the checksum. */
t_u16			in_cksum(buf,nwords)
t_u16			*buf;
int			nwords;
{
  unsigned long		sum;
   
  sum = 0;
  while (nwords > 0)
    {
      t_u16		u16;
      
      FBCOPY(buf,&u16,sizeof (u16)); /* ALIGNMENT */
      sum += UNSAFE_HTONS(u16);
      buf++;
      nwords--;
    }
  sum = (sum >> 16) + (sum & 0xffff);
  sum += (sum >> 16);
  return ((t_u16)(~sum));
}

/* computes ip checksum. 
   It deals with alignment.
   It calls in_cksum(3). */
VOID_FUNC		ip_compute_sum(ip)
t_ip			*ip;
{
  t_u16			zero;

  zero = 0;
  FBCOPY(&zero,&(ip->sum),sizeof (zero)); /* ALIGNMENT */
  safe_htons(in_cksum((t_u16 *)ip,ip_get_hl(ip) * 2),&(ip->sum));
}

/* sets ip version */
VOID_FUNC		ip_set_v(ip,v)
t_ip			*ip;
int			v;
{
  bit_field_u8_set(&(ip->vhl),
		   ip_v_bit_field.from,
		   ip_v_bit_field.to,
		   v);
}

/* gets ip version.
   Returns the version */
int			ip_get_v(ip)
t_ip			*ip;
{
  t_u8			*vp;
  
  vp = bit_field_u8_get(&(ip->vhl),
			ip_v_bit_field.from,
			ip_v_bit_field.to);
  return ((int)(*vp));
}

/* sets ip header length */
VOID_FUNC		ip_set_hl(ip,hl)
t_ip			*ip;
int			hl;
{
  bit_field_u8_set(&(ip->vhl),
		   ip_hl_bit_field.from,
		   ip_hl_bit_field.to,
		   hl);
}

/* gets ip header length.
   Returns ip header length */
int			ip_get_hl(ip)
t_ip			*ip;
{
  t_u8			*hlp;

  hlp = bit_field_u8_get(&(ip->vhl),
			 ip_hl_bit_field.from,
			 ip_hl_bit_field.to);
  return ((int)(*hlp));
}

/* sets ip tos */
VOID_FUNC		ip_set_tos(ip,tos)
t_ip			*ip;
int			tos;
{
  ip->tos = (u_char)tos;
}

/* returns ip tos */
int			ip_get_tos(ip)
t_ip			*ip;
{
  return (ip->tos);
}

/* sets ip length.
   Note: deals with alignment */
VOID_FUNC		ip_set_len(ip,len)
t_ip			*ip;
int			len;
{
  safe_htons(len,&(ip->len));
}

/* gets ip length.
   Note: deals with alignment.
   Returns ip length. */
int			ip_get_len(ip)
t_ip			*ip;
{
  return (safe_ntohs(&ip->len));
}

/* sets ip id.
   Note: deals with alignment */
VOID_FUNC		ip_set_id(ip,id)
t_ip			*ip;
int			id;
{
  safe_htons(id,&ip->id);
}

/* gets ip id.
   Note: deals with alignment. 
   Returns ip id */
int			ip_get_id(ip)
t_ip			*ip;
{
  return (safe_ntohs(&ip->id));
}

/* sets ip off.
   Note: deals with alignment. */ 
VOID_FUNC		ip_set_off(ip,off)
t_ip			*ip;
int			off;
{
  t_u16			u16;

  FBCOPY(&ip->fragoff,&u16,sizeof (t_u16)); /* ALIGNMENT */
  bit_field_u16_set(&u16,
		    ip_off_bit_field.from,
		    ip_off_bit_field.to,
		    off);
  FBCOPY(&u16,&ip->fragoff,sizeof (t_u16)); /* ALIGNMENT */
}

/* gets ip off.
   Note: deals with alignement.
   Return ip off. */
int			ip_get_off(ip)
t_ip			*ip;
{
  t_u16			*offp;
  t_u16			u16;

  FBCOPY(&ip->fragoff,&u16,sizeof (t_u16)); /* ALIGNMENT */
  offp = bit_field_u16_get(&u16,
			   ip_off_bit_field.from,
			   ip_off_bit_field.to);
  return ((int)(*offp));
}

/* sets ip ttl */
VOID_FUNC		ip_set_ttl(ip,ttl)
t_ip			*ip;
int			ttl;
{
  ip->ttl = (u_char)ttl;
}

/* gets ip ttl */
int			ip_get_ttl(ip)
t_ip			*ip;
{
  return (ip->ttl);
}

/* sets ip proto */
VOID_FUNC		ip_set_p(ip,p)
t_ip			*ip;
int			p;
{
  ip->p = (u_char)p;
}

/* gets ip proto */
int			ip_get_p(ip)
t_ip			*ip;
{
  return (ip->p);
}

/* sets ip sum.
   Note: deals with alignment */
VOID_FUNC		ip_set_sum(ip,sum)
t_ip			*ip;
int			sum;
{
  safe_htons(sum,&ip->sum);
}

/* gets ip sum.
   Note: deals with alignement */
int			ip_get_sum(ip)
t_ip			*ip;
{
  return (safe_ntohs(&ip->sum));
}

/* sets ip src.
   Note: deals with alignment */
VOID_FUNC		ip_set_src(ip,src)
t_ip			*ip;
t_in_addr		*src;
{
  FBCOPY(src,&(ip->src),sizeof (struct in_addr));
}

/* gets ip dst.
   Note: deals with alignment */
VOID_FUNC		ip_get_src(ip,src)
t_ip			*ip;
t_in_addr		*src;
{
  FBCOPY(&(ip->src),src,sizeof (struct in_addr));
}

/* sets ip dst.
   Note: deals with alignment */
VOID_FUNC		ip_set_dst(ip,dst)
t_ip			*ip;
t_in_addr		*dst;
{
  FBCOPY(dst,&(ip->dst),sizeof (struct in_addr));
}

/* gets ip dst.
   Note: deals with alignment */
VOID_FUNC		ip_get_dst(ip,dst)
t_ip			*ip;
t_in_addr		*dst;
{
  FBCOPY(&(ip->dst),dst,sizeof (struct in_addr));
}

/* gets ip offset to sub layer.
   This procedure is called by lay_ip_msg(3).
   It checks standard header length (IP_MINHLEN) and returns header
   length even a if it is a bad value (e.g zero).
   Returns 0 if OK. Might return -ERR_TRUNC */
t_status		ip_off(buf,len,off)
char			*buf;
int			len;
t_off			*off;
{
  t_ip			*ip;

  ip = (t_ip *)buf;
  LAYER_IP_CHECK(ip,buf,len);
  (*off) = ip_get_hl(ip) * 4;
  return (0);
}

/* gets ip sub layer.
   This procedure is called by lay_ip_msg(3).
   It checks standard header length (IP_MINHLEN) and returns sub layer.
   It reverts to lay_data_msg(3) if it doesn't recognizes the protocol.
   Currently, only icmp, udp, tcp and ipip are recognized.
   Returns 0 if OK. Might return -ERR_TRUNC. */
t_status		ip_sub(buf,len,sub_mp)
char			*buf;
int			len;
t_msg_proc		*sub_mp;
{
  t_ip			*ip;

  ip = (t_ip *)buf;
  LAYER_IP_CHECK(ip,buf,len);
  switch (ip_get_p(ip))
    {
    case IP_PROTO_ICMP:
      (*sub_mp) = &lay_icmp_msg;
      return (0);
    case IP_PROTO_UDP:
      (*sub_mp) = &lay_udp_msg;
      return (0);
    case IP_PROTO_TCP:
      (*sub_mp) = &lay_tcp_msg;
      return (0);
    case 0:
      (*sub_mp) = &lay_ip_msg;
      return (0);
    }
  (*sub_mp) = &lay_data_msg;
  return (0);
}

/* performs an ip checksum.
   This procedure is called by lay_ip_msg(3) and corresponds to the 
   meta-checksum-mechanism of pkt_sum(3).
   It checks the standard header len (IP_MINHLEN) and calls ip_compute_sum(3).
   Returns 0 if OK. Returns -ERR_BADLEN if the given length is different
   from the one retrieved by ip_get_len(3). Returns -ERR_BADHEADER if
   ip_get_hl(3) is less than IP_MINHLEN. Might of course return -ERR_TRUNC. */
t_status		ip_sum(buf,len,up_buf,up_len)
char			*buf;
int			len;
char			*up_buf;
int			up_len;
{
  t_ip			*ip;

  LAYER_IP_CHECK(ip,buf,len);
  if (ip_get_len(ip) != len)
    return (-ERR_BADLEN);
  if (ip_get_hl(ip) * 4 < IP_MINHLEN)
    return (-ERR_BADHEADER);
  ip_compute_sum(ip);
  return (0);
}

/* adapts its "length" field to length of buffer.
   This procedure is called by lay_ip_msg(3).
   Returns 0 if OK. Might return -ERR_TRUNC */
t_status		ip_adapt_len(buf,len)
char			*buf;
int			len;
{
  t_ip			*ip;

  LAYER_IP_CHECK(ip,buf,len);
  ip_set_len(ip,len);
  return (0);
}

/* tells if ip packet has option.
   This procedure is called by lay_ip_msg(3).
   Returns 0 if OK. Might return -ERR_TRUNC. */
t_status		ip_has_opt(buf,len,hsd)
char			*buf;
int			len;
t_has_opt_data		*hsd;
{
  t_ip			*ip;

  LAYER_IP_CHECK(ip,buf,len);
  if (ip_get_hl(ip) * 4 > IP_MINHLEN)
    {
      hsd->has_opt = TRUE;
      hsd->opt_off = IP_MINHLEN;
      hsd->opt_len = ip_get_hl(ip) * 4 - IP_MINHLEN;
      hsd->opt_mp = lay_ipopt_msg;
    }
  else
    {
      hsd->has_opt = FALSE;
    }
  return (0);
}

t_field				ip_fields[] = 
{
  {"hl",	0,	typ_u8bitfield_msg,(VOID_PTR)(&ip_hl_bit_field)},
  {"v",		0,	typ_u8bitfield_msg,(VOID_PTR)(&ip_v_bit_field)},
  {"tos",	OFFSET(t_ip *,tos),	typ_u8_msg,		NULL},
  {"Tos",	OFFSET(t_ip *,tos),	typ_u8assoc_msg,	iptos_assocs},
  {"len",	OFFSET(t_ip *,len),	typ_nu16_msg,		NULL},
  {"id",	OFFSET(t_ip *,id),	typ_nu16_msg,		NULL},
  {"frag",	OFFSET(t_ip *,fragoff),
   typ_nu16bitfield_msg,	(VOID_PTR)(&ip_frag_bit_field)},
  {"Frag",	OFFSET(t_ip *,fragoff),
   typ_nu16bitfieldassoc_msg,	(VOID_PTR)(&ip_frag_bfad)},
  {"off",	OFFSET(t_ip *,fragoff),
   typ_nu16bitfield_msg,	(VOID_PTR)(&ip_off_bit_field)},
  {"ttl",	OFFSET(t_ip *,ttl),	typ_u8_msg,		NULL},
  {"p",		OFFSET(t_ip *,p),	typ_u8_msg,		NULL},
  {"P",		OFFSET(t_ip *,p),	typ_ipproto_msg,	NULL},
  {"sum",	OFFSET(t_ip *,sum),	typ_nu16_msg,		NULL},
  {"src",	OFFSET(t_ip *,src),	typ_inaddr_msg,		NULL},
  {"Src",	OFFSET(t_ip *,src),	typ_inaddr_resolved_msg,NULL},
  {"dst",	OFFSET(t_ip *,dst),	typ_inaddr_msg,		NULL},
  {"Dst",	OFFSET(t_ip *,dst),	typ_inaddr_resolved_msg,NULL},
  NULL_FIELD
};

char				*ip_itmpl = "\n\
<!--ip_itmpl-->\n\
<table width=100%%%% bgcolor=\"%%ipColor%%\">\n\
<tr>\n\
<td>\n\
<small>\n\
<a href=\"extract(ip[%i%])\">[Extract]</a>\n\
<a href=\"trunc(ip[%i%])\">[Trunc]</a>\n\
<a href=\"paste(ip[%i%])\">[Paste]</a>\n\
<a href=\"adaptlen(ip[%i%])\">[Adapt len]</a>\n\
<a href=\"sum(ip[%i%])\">[Sum]</a>\n\
 ip\n\
</small>\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=12%%%%><a href=\"setfield(ip[%i%].v)\">%%ip[%i%].v%%</a></td>\n\
<td align=center width=13%%%%><a href=\"setfield(ip[%i%].hl)\">%%ip[%i%].hl%%</a></td>\n\
<td align=center width=25%%%%><a href=\"setfield(ip[%i%].Tos)\">%%ip[%i%].Tos%%</a>(<a href=\"setfield(ip[%i%].tos)\">%%ip[%i%].tos%%</a>)</td>\n\
<td align=center width=50%%%% colspan=2><a href=\"setfield(ip[%i%].len)\">%%ip[%i%].len%%</a></td>\n\
</tr>\n\
<tr>\n\
<td align=center width=50%%%%><a href=\"setfield(ip[%i%].id)\">%%ip[%i%].id%%</a></td>\n\
<td align=center width=25%%%%><a href=\"setfield(ip[%i%].Frag)\">%%ip[%i%].Frag%%</a>(<a href=\"setfield(ip[%i%].frag)\">%%ip[%i%].frag%%</a>)</td>\n\
<td align=center width=25%%%%><a href=\"setfield(ip[%i%].off)\">%%ip[%i%].off%%</a></td>\n\
</tr>\n\
<tr>\n\
<td align=center width=33%%%%><a href=\"setfield(ip[%i%].ttl)\">%%ip[%i%].ttl%%</a></td>\n\
<td align=center width=33%%%%><a href=\"setfield(ip[%i%].P)\">%%ip[%i%].P%%</a>(<a href=\"setfield(ip[%i%].p)\">%%ip[%i%].p%%</a>)</td>\n\
<td align=center width=33%%%%><a href=\"setfield(ip[%i%].sum)\">%%ip[%i%].sum%%</a></td>\n\
</tr>\n\
<tr>\n\
<td align=center width=100%%%% colspan=4><a href=\"setfield(ip[%i%].Src)\">%%ip[%i%].Src%%</a>(<a href=\"setfield(ip[%i%].src)\">%%ip[%i%].src%%</a>)</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=100%%%% colspan=4><a href=\"setfield(ip[%i%].Dst)\">%%ip[%i%].Dst%%</a>(<a href=\"setfield(ip[%i%].dst)\">%%ip[%i%].dst%%</a>)</td>\n\
</tr>\n\
</table>\n\
";

int				lay_ip_src_chan;
int				lay_ip_dst_chan;

/* is a t_msg_proc.
   It manages ip layers. 
   It fills two channels in LAY_SUB_ID: lay_ip_src_chan
   and lay_ip_dst_chan which are used by sub layers (e.g lay_tcp_msg(3)). */
t_status			lay_ip_msg(msg,arg1,arg2)
t_msg				msg;
VOID_PTR			arg1;
VOID_PTR			arg2;
{
  t_status			status;

  switch (msg)
    {
      LAY_CLASS_GENERIC;
      LAY_NAME_ID_GENERIC(&lay_ip_msg,"ip");
      LAY_GET_FIELD_GENERIC(ip_fields);
      LAY_SET_FIELD_GENERIC(ip_fields);
      LAY_GET_FIELDS_GENERIC(ip_fields);
      LAY_GET_FIELD_TYP_GENERIC(ip_fields);
      LAY_GET_ITMPL_GENERIC(&lay_ip_msg,ip_itmpl);
    case LAY_OFF:
      {
	LAY_OFF_ARGS(b,off);
	
	return (ip_off(b->buf,b->len,off));
      }
    case LAY_SUB:
      {
	LAY_SUB_ARGS(b,mp);
	
	return (ip_sub(b->buf,b->len,mp));
      }
    case LAY_SUB_ID:
      {
	LAY_SUB_ID_ARGS(sid,mp);
	t_status		status;	
	t_ip			*ip;
	t_hash_elt		*he;
	t_buf			b;
	t_u32			inaddr;

	if (sid->b.len < IP_MINHLEN)
	  return (-ERR_TRUNC);
	ip = (t_ip *)(sid->b.buf);
	FBCOPY(&(ip->src),&inaddr,sizeof (inaddr));
	if ((status = id_override(sid->id,
				  &lay_ip_src_chan,
				  (VOID_PTR)inaddr)) < 0)
	  return (status);
	FBCOPY(&(ip->dst),&inaddr,sizeof (inaddr));
	if ((status = id_override(sid->id,
				  &lay_ip_dst_chan,
				  (VOID_PTR)inaddr)) < 0)
	  return (status);
	b.buf = sid->b.buf;
	b.len = sid->b.len;
	return (lay_msg(lay_ip_msg,
			LAY_SUB,
			&b,
			mp));
      }
    case LAY_SUM:
      {
	LAY_SUM_ARGS(sd,unused);
	
	return (ip_sum(sd->b.buf,
		       sd->b.len,
		       sd->up.buf,
		       sd->up.len));
      }
    case LAY_HAS_OPT:
      {
	LAY_HAS_OPT_ARGS(b,hsd);

	return (ip_has_opt(b->buf,b->len,hsd));
      }
    case LAY_ADAPT_LEN:
      {
	LAY_ADAPT_LEN_ARGS(b,unused);

	return (ip_adapt_len(b->buf,b->len));
      }
    }
  return (-ERR_NOMETHOD);
}
