/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "lay_icmp.h"
#include "lay_ip.h"
#include "lay_data.h"
#include "typ_8.h"
#include "typ_16.h"
#include "typ_32.h"
#include "typ_inaddr.h"
#include "typ_icmpcode.h"
#include "typ_time.h"

/*
Type    Name                                    Reference
----    -------------------------               ---------
  0     Echo Reply                               [RFC792]
  1     Unassigned                                  [JBP]
  2     Unassigned                                  [JBP]
  3     Destination Unreachable                  [RFC792]
  4     Source Quench                            [RFC792]
  5     Redirect                                 [RFC792]
  6     Alternate Host Address                      [JBP]
  7     Unassigned                                  [JBP]
  8     Echo                                     [RFC792]
  9     Router Advertisement                    [RFC1256]
 10     Router Selection                        [RFC1256]
 11     Time Exceeded                            [RFC792]
 12     Parameter Problem                        [RFC792]
 13     Timestamp                                [RFC792]
 14     Timestamp Reply                          [RFC792]
 15     Information Request                      [RFC792]
 16     Information Reply                        [RFC792]
 17     Address Mask Request                     [RFC950]
 18     Address Mask Reply                       [RFC950]
 19     Reserved (for Security)                    [Solo]
 20-29  Reserved (for Robustness Experiment)        [ZSu]
 30     Traceroute                              [RFC1393]
 31     Datagram Conversion Error               [RFC1475]
 32     Mobile Host Redirect              [David Johnson]
 33     IPv6 Where-Are-You                 [Bill Simpson]
 34     IPv6 I-Am-Here                     [Bill Simpson]
 35     Mobile Registration Request        [Bill Simpson]
 36     Mobile Registration Reply          [Bill Simpson]
 37-255 Reserved                                    [JBP]
 */

t_assoc				icmptype_assocs[] =
{
  {"echo_reply",		(VOID_PTR)ICMP_ECHO_REPLY},
  {"unreachable",		(VOID_PTR)ICMP_UNREACHABLE},
  {"source_quench",		(VOID_PTR)ICMP_SOURCE_QUENCH},
  {"redirect",			(VOID_PTR)ICMP_REDIRECT},
  {"alternate_host_address",	(VOID_PTR)6},
  {"echo_request",		(VOID_PTR)ICMP_ECHO_REQUEST},
  {"router_advertisement",	(VOID_PTR)ICMP_ROUTER_ADVERTISEMENT},
  {"router_selection",		(VOID_PTR)10},
  {"time_exceed",		(VOID_PTR)ICMP_TIME_EXCEED},
  {"parameter_problem",		(VOID_PTR)12},
  {"timestamp_request",		(VOID_PTR)ICMP_TS_REQUEST},
  {"timestamp_reply",		(VOID_PTR)ICMP_TS_REPLY},
  {"info_request",		(VOID_PTR)15},
  {"info_reply",		(VOID_PTR)16},
  {"mask_request",		(VOID_PTR)ICMP_MASK_REQUEST},
  {"mask_reply",		(VOID_PTR)ICMP_MASK_REPLY},
  {"traceroute",		(VOID_PTR)30},
  {"datagram_conversion_error",	(VOID_PTR)31},
  {"mobile_host_redirect",	(VOID_PTR)32},
  {"ipv6_where_are_you",	(VOID_PTR)33},
  {"ipv6_i_am_here",		(VOID_PTR)34},
  {"mobile_regist_request",	(VOID_PTR)35},
  {"mobile_regist_reply",	(VOID_PTR)36},
  {NULL,			NULL},
};

int		icmp_compute_sum(ip,icmp)
t_ip		*ip;
t_icmp		*icmp;
{
  int		datalen;
  t_u16		zero;

  zero = 0;
  FBCOPY(&zero,&(icmp->cksum),sizeof (zero)); /* ALIGNMENT */
  datalen = ip_get_len(ip) - ip_get_hl(ip) * 4;
  safe_htons(in_cksum((t_u16 *)icmp,datalen >> 1),&(icmp->cksum));
}

VOID_FUNC	icmp_set_proto(ip)
t_ip		*ip;
{
  ip_set_p(ip,IP_PROTO_ICMP);
}

VOID_FUNC	icmp_set_type(icmp,type)
t_icmp		*icmp;
int		type;
{
  icmp->type = type;
}

int		icmp_get_type(icmp)
t_icmp		*icmp;
{
  return (icmp->type);
}

VOID_FUNC	icmp_set_code(icmp,code)
t_icmp		*icmp;
int		code;
{
  icmp->code = code;
}

int		icmp_get_code(icmp)
t_icmp		*icmp;
{
  return (icmp->code);
}

VOID_FUNC	icmp_set_cksum(icmp,sum)
t_icmp		*icmp;
int		sum;
{
  safe_htons(sum,&(icmp->cksum));
}

int		icmp_get_cksum(icmp)
t_icmp		*icmp;
{
  return (safe_ntohs(&(icmp->cksum)));
}	

int		icmp_sum(buf,len,up_buf,up_len)
char		*buf;
int		len;
char		*up_buf;
int		up_len;
{
  t_icmp	*icmp;
  t_ip		*ip;

  LAYER_ICMP_CHECK(icmp,buf,len);
  LAYER_IP_CHECK(ip,up_buf,up_len);
  if (ip_get_len(ip) != up_len)
    return (-ERR_BADLEN);
  if (ip_get_hl(ip) * 4 < IP_MINHLEN)
    return (-ERR_BADHEADER);
  icmp_compute_sum(ip,icmp);
  return (0);
}

t_field				icmpmbz_fields[] = 
{
  {"mbz",	0,		typ_nu32_msg,			NULL},
  NULL_FIELD
};

char				*icmpmbz_itmpl = "\n\
<!--icmpmbz_tmpl-->\n\
<table width=100%%%% bgcolor=\"%%icmpmbzColor%%\">\n\
<tr>\n\
<td width=100%%%%>\n\
<small>\n\
<a href=\"extract(icmpmbz[%i%])\">[Extract]</a>\n\
<a href=\"trunc(icmpmbz[%i%])\">[Trunc]</a>\n\
<a href=\"paste(icmpmbz[%i%])\">[Paste]</a>\n\
 icmpmbz\n\
</small>\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=100%%%%>\n\
<a href=\"setfield(icmpmbz[%i%].mbz)\">%%icmpmbz[%i%].mbz%%</a>\n\
</td>\n\
</tr>\n\
</table>\n\
";

t_status			lay_icmpmbz_msg(msg,arg1,arg2)
t_msg				msg;
VOID_PTR			arg1;
VOID_PTR			arg2;
{
  t_status			status;
  
  switch (msg)
    {
      LAY_CLASS_GENERIC;
      LAY_NAME_ID_GENERIC(&lay_icmpmbz_msg,"icmpmbz");
      LAY_GET_FIELD_GENERIC(icmpmbz_fields);
      LAY_SET_FIELD_GENERIC(icmpmbz_fields);
      LAY_GET_FIELDS_GENERIC(icmpmbz_fields);
      LAY_GET_FIELD_TYP_GENERIC(icmpmbz_fields);
      LAY_GET_ITMPL_GENERIC(&lay_icmpmbz_msg,icmpmbz_itmpl);
    case LAY_OFF:
      {
	LAY_OFF_ARGS(b,off);
	
	(*off) = sizeof (t_u32);
	return (0);
      }
    case LAY_SUB:
      {
	LAY_SUB_ARGS(b,mp);
	
	(*mp) = &lay_ip_msg;
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}

t_field				icmpredirect_fields[] = 
{
  {"router",	0,		typ_inaddr_msg,			NULL},
  {"Router",	0,		typ_inaddr_resolved_msg,	NULL},
  NULL_FIELD
};

char				*icmpredirect_itmpl = "\n\
<!--icmpredirect_tmpl-->\n\
<table width=100%%%% bgcolor=\"%%icmpredirectColor%%\">\n\
<tr>\n\
<td width=100%%%%>\n\
<small>\n\
<a href=\"extract(icmpredirect[%i%])\">[Extract]</a>\n\
<a href=\"trunc(icmpredirect[%i%])\">[Trunc]</a>\n\
<a href=\"paste(icmpredirect[%i%])\">[Paste]</a>\n\
 icmpredirect\n\
</small>\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=100%%%%>\n\
<a href=\"setfield(icmpredirect[%i%].router)\">%%icmpredirect[%i%].router%%</a>\n\
(<a href=\"setfield(icmpredirect[%i%].Router)\">%%icmpredirect[%i%].Router%%</a>)\n\
</td>\n\
</tr>\n\
</table>\n\
";

t_status			lay_icmpredirect_msg(msg,arg1,arg2)
t_msg				msg;
VOID_PTR			arg1;
VOID_PTR			arg2;
{
  t_status			status;
  
  switch (msg)
    {
      LAY_CLASS_GENERIC;
      LAY_NAME_ID_GENERIC(&lay_icmpredirect_msg,"icmpredirect");
      LAY_GET_FIELD_GENERIC(icmpredirect_fields);
      LAY_SET_FIELD_GENERIC(icmpredirect_fields);
      LAY_GET_FIELDS_GENERIC(icmpredirect_fields);
      LAY_GET_FIELD_TYP_GENERIC(icmpredirect_fields);
      LAY_GET_ITMPL_GENERIC(&lay_icmpredirect_msg,icmpredirect_itmpl);
    case LAY_OFF:
      {
	LAY_OFF_ARGS(b,off);
	
	(*off) = sizeof (t_u32);
	return (0);
      }
    case LAY_SUB:
      {
	LAY_SUB_ARGS(b,mp);
	
	(*mp) = &lay_ip_msg;
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}

t_field				icmpmask_fields[] = 
{
  {"id",	0,		typ_nu16_msg,			NULL},
  {"seq",	2,		typ_nu16_msg,			NULL},
  {"mask",	4,		typ_inaddr_msg,			NULL},
  {"Mask",	4,		typ_inaddr_resolved_msg,	NULL},
  NULL_FIELD
};

char				*icmpmask_itmpl = "\n\
<!--icmpmask_tmpl-->\n\
<table width=100%%%% bgcolor=\"%%icmpmaskColor%%\">\n\
<tr>\n\
<td width=100%%%%>\n\
<small>\n\
<a href=\"extract(icmpmask[%i%])\">[Extract]</a>\n\
<a href=\"trunc(icmpmask[%i%])\">[Trunc]</a>\n\
<a href=\"paste(icmpmask[%i%])\">[Paste]</a>\n\
 icmpmask\n\
</small>\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=50%%%%>\n\
<a href=\"setfield(icmpmask[%i%].id)\">%%icmpmask[%i%].id%%</a>\n\
</td>\n\
<td align=center width=50%%%%>\n\
<a href=\"setfield(icmpmask[%i%].seq)\">%%icmpmask[%i%].seq%%</a>\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=100%%%%>\n\
<a href=\"setfield(icmpmask[%i%].Mask)\">%%icmpmask[%i%].Mask%%</a>\n\
(<a href=\"setfield(icmpmask[%i%].mask)\">%%icmpmask[%i%].mask%%</a>)\n\
</td>\n\
</tr>\n\
</table>\n\
";

t_status			lay_icmpmask_msg(msg,arg1,arg2)
t_msg				msg;
VOID_PTR			arg1;
VOID_PTR			arg2;
{
  t_status			status;
  
  switch (msg)
    {
      LAY_CLASS_GENERIC;
      LAY_NAME_ID_GENERIC(&lay_icmpmask_msg,"icmpmask");
      LAY_GET_FIELD_GENERIC(icmpmask_fields);
      LAY_SET_FIELD_GENERIC(icmpmask_fields);
      LAY_GET_FIELDS_GENERIC(icmpmask_fields);
      LAY_GET_FIELD_TYP_GENERIC(icmpmask_fields);
      LAY_GET_ITMPL_GENERIC(&lay_icmpmask_msg,icmpmask_itmpl);
    case LAY_OFF:
      {
	LAY_OFF_ARGS(b,off);
	
	(*off) = sizeof (t_u32);
	return (0);
      }
    case LAY_SUB:
      {
	LAY_SUB_ARGS(b,mp);
	
	(*mp) = &lay_data_msg;
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}

t_field				icmpts_fields[] = 
{
  {"id",	0,		typ_nu16_msg,			NULL},
  {"seq",	2,		typ_nu16_msg,			NULL},
  {"Originatets",4,		typ_time_msg,			NULL},
  {"Receivets",	8,		typ_time_msg,			NULL},
  {"Transmitts",12,		typ_time_msg,			NULL},
  NULL_FIELD
};

char				*icmpts_itmpl = "\n\
<!--icmpts_tmpl-->\n\
<table width=100%%%% bgcolor=\"%%icmptsColor%%\">\n\
<tr>\n\
<td width=100%%%%>\n\
<small>\n\
<a href=\"extract(icmpts[%i%])\">[Extract]</a>\n\
<a href=\"trunc(icmpts[%i%])\">[Trunc]</a>\n\
<a href=\"paste(icmpts[%i%])\">[Paste]</a>\n\
 icmpts\n\
</small>\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=50%%%%>\n\
<a href=\"setfield(icmpts[%i%].id)\">%%icmpts[%i%].id%%</a>\n\
</td>\n\
<td align=center width=50%%%%>\n\
<a href=\"setfield(icmpts[%i%].seq)\">%%icmpts[%i%].seq%%</a>\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=100%%%%>\n\
<a href=\"setfield(icmpts[%i%].Originatets)\">%%icmpts[%i%].Originatets%%</a>\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=100%%%%>\n\
<a href=\"setfield(icmpts[%i%].Receivets)\">%%icmpts[%i%].Receivets%%</a>\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=100%%%%>\n\
<a href=\"setfield(icmpts[%i%].Transmitts)\">%%icmpts[%i%].Transmitts%%</a>\n\
</td>\n\
</tr>\n\
</table>\n\
";

t_status			lay_icmpts_msg(msg,arg1,arg2)
t_msg				msg;
VOID_PTR			arg1;
VOID_PTR			arg2;
{
  t_status			status;
  
  switch (msg)
    {
      LAY_CLASS_GENERIC;
      LAY_NAME_ID_GENERIC(&lay_icmpts_msg,"icmpts");
      LAY_GET_FIELD_GENERIC(icmpts_fields);
      LAY_SET_FIELD_GENERIC(icmpts_fields);
      LAY_GET_FIELDS_GENERIC(icmpts_fields);
      LAY_GET_FIELD_TYP_GENERIC(icmpts_fields);
      LAY_GET_ITMPL_GENERIC(&lay_icmpts_msg,icmpts_itmpl);
    case LAY_OFF:
      {
	LAY_OFF_ARGS(b,off);
	
	(*off) = sizeof (t_u32);
	return (0);
      }
    case LAY_SUB:
      {
	LAY_SUB_ARGS(b,mp);
	
	(*mp) = &lay_data_msg;
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}

int		icmp_sub(buf,len,sub_mp)
char		*buf;
int		len;
t_msg_proc	*sub_mp;
{
  t_icmp	*icmp;	

  LAYER_ICMP_CHECK(icmp,buf,len);
  switch (icmp_get_type(icmp))
    {
    case ICMP_REDIRECT:
      (*sub_mp) = &lay_icmpredirect_msg;
      return (0);
    case ICMP_SOURCE_QUENCH:
    case ICMP_UNREACHABLE:
    case ICMP_TIME_EXCEED:
      (*sub_mp) = &lay_icmpmbz_msg;
      return (0);
    case ICMP_TS_REQUEST:
    case ICMP_TS_REPLY:
      (*sub_mp) = &lay_icmpts_msg;
      return (0);
    case ICMP_MASK_REQUEST:
    case ICMP_MASK_REPLY:
      (*sub_mp) = &lay_icmpmask_msg;
      return (0);
    }
  (*sub_mp) = &lay_data_msg;
  return (0);
}

t_field				icmp_fields[] = 
{
  {"type",	OFFSET(t_icmp *,type),	typ_u8_msg,			NULL},
  {"Type",	OFFSET(t_icmp *,type),	typ_u8assoc_msg,  icmptype_assocs},
  {"code",	OFFSET(t_icmp *,code),	typ_u8_msg,			NULL},
  {"Code",	OFFSET(t_icmp *,type),	typ_icmpcodeagainsttype_msg,NULL},
  {"cksum",	OFFSET(t_icmp *,cksum),	typ_nu16_msg,		NULL},
  NULL_FIELD
};

char				*icmp_itmpl = "\n\
<!--icmp_tmpl-->\n\
<table width=100%%%% bgcolor=\"%%icmpColor%%\">\n\
<tr>\n\
<td width=100%%%%>\n\
<small>\n\
<a href=\"extract(icmp[%i%])\">[Extract]</a>\n\
<a href=\"trunc(icmp[%i%])\">[Trunc]</a>\n\
<a href=\"paste(icmp[%i%])\">[Paste]</a>\n\
<a href=\"sum(icmp[%i%])\">[Sum]</a>\n\
 icmp\n\
</small>\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=33%%%%>\n\
<a href=\"setfield(icmp[%i%].Type)\">%%icmp[%i%].Type%%</a>\n\
(<a href=\"setfield(icmp[%i%].type)\">%%icmp[%i%].type%%</a>)\n\
</td>\n\
<td align=center width=33%%%%>\n\
<_tiny><a href=\"setfield(icmp[%i%].Code)\">%%icmp[%i%].Code%%</a></_tiny>\n\
(<a href=\"setfield(icmp[%i%].code)\">%%icmp[%i%].code%%</a>)\n\
</td>\n\
<td align=center width=34%%%%>\n\
<a href=\"setfield(icmp[%i%].cksum)\">%%icmp[%i%].cksum%%</a>\n\
</td>\n\
</tr>\n\
</table>\n\
";

t_status			lay_icmp_msg(msg,arg1,arg2)
t_msg				msg;
VOID_PTR			arg1;
VOID_PTR			arg2;
{
  t_status			status;
  
  switch (msg)
    {
      LAY_CLASS_GENERIC;
      LAY_NAME_ID_GENERIC(&lay_icmp_msg,"icmp");
      LAY_GET_FIELD_GENERIC(icmp_fields);
      LAY_SET_FIELD_GENERIC(icmp_fields);
      LAY_GET_FIELDS_GENERIC(icmp_fields);
      LAY_GET_FIELD_TYP_GENERIC(icmp_fields);
      LAY_GET_ITMPL_GENERIC(&lay_icmp_msg,icmp_itmpl);
    case LAY_OFF:
      {
	LAY_OFF_ARGS(b,off);
	
	(*off) = ICMP_HLEN;
	return (0);
      }
    case LAY_SUB:
      {
	LAY_SUB_ARGS(b,mp);
	
	return (icmp_sub(b->buf,b->len,mp));
      }
    case LAY_SUM:
      {
	LAY_SUM_ARGS(sd,unused);
	
	return (icmp_sum(sd->b.buf,
			 sd->b.len,
			 sd->up.buf,
			 sd->up.len));
      }
    }
  return (-ERR_NOMETHOD);
}
