/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/ShellP.h>
#include "xmg_i.h"
#include "xt_macro.h"
#include "Xmg.h"

extern Widget		toplevel;
extern XtAppContext	app_context;

/* this is an XtActionProc.
   It does nothing. */
VOID_FUNC		XmgActionNop(w,event,params,num_params)
Widget			w;
XEvent			*event;
String			*params;
Cardinal		*num_params;
{
  /* NOP */
}

/* this is an XtActionProc.
   It popdowns the widget. */
VOID_FUNC		XmgActionPopdown(w,event,params,num_params)
Widget			w;
XEvent			*event;
String			*params;
Cardinal		*num_params;
{
  XtPopdown(w);
}

/* this is an XtActionProc.
   It destroys the widget. */
VOID_FUNC		XmgActionDestroy(w,event,params,num_params)
Widget			w;
XEvent			*event;
String			*params;
Cardinal		*num_params;
{
  if (w == toplevel)
    {
      XtDestroyApplicationContext(XtWidgetToApplicationContext(w));
      exit(0);
    }
  else
    XtDestroyWidget(w);
}

/* this is an XtActionProc.
   It sets the focus to the widget. */
VOID_FUNC		XmgActionSetFocus(w,event,params,num_params)
Widget			w;
XEvent			*event;
String			*params;
Cardinal		*num_params;
{
  XtSetKeyboardFocus(XtParent(w),w);
}

/* acts like XtPopdown(3) but doesn't call XtRemoveGrab(3).
   (as it is called by another routine: see XmgActionPopdownAllMenus(3).) */
VOID_FUNC		XmgPopdown(widget)
Widget			widget;
{
  ShellWidget		shell_widget;

  shell_widget = (ShellWidget)widget;
  if (!XtIsShell(widget)) 
    {
      XtAppErrorMsg(XtWidgetToApplicationContext(widget),
		    "invalidClass","xmgPopdown","XmgToolkitError",
		    "XmgPopdown requires a subclass of shellWidgetClass",
		    (String *)NULL,
		    (Cardinal *)NULL);
    }
  if (shell_widget->shell.popped_up) 
    {
      XtGrabKind	grab_kind;

      grab_kind = shell_widget->shell.grab_kind;
      XtUnmapWidget(widget);
      XWithdrawWindow(XtDisplay(widget),
		      XtWindow(widget),
		      XScreenNumberOfScreen(XtScreen(widget)));
      if (grab_kind != XtGrabNone)
	{
#ifdef NOTDEF
	  XtRemoveGrab(widget);
#endif
        }
      shell_widget->shell.popped_up = FALSE;
      XtCallCallbacks(widget,XtNpopdownCallback,(XtPointer)&grab_kind);
    }
}

/* this is an XtActionProc.
 Should be used instead of MenuPopdown() on <BtnUp> translations in the
 top cascaded menu. */
VOID_FUNC		XmgActionPopdownAllMenus(w,event,params,num_params)
Widget			w;
XEvent			*event;
String			*params;
Cardinal		*num_params;
{
  int			i;
  CoreWidget		parent;

  parent = XtParent(w);
  i = 0;
  while (i < parent->core.num_popups)
    {
#ifdef NOTDEF
      XtCallActionProc(parent->core.popup_list[i],
		       "XtMenuPopdown",
		       event,
		       params,
		       *num_params);
#else
      XmgPopdown(parent->core.popup_list[i]);
#endif
      i++;
    }
  XtRemoveGrab(w);
}

/* this is an XtActionProc.
 Should be used on <EnterWindow> translations in cascaded menus. */
VOID_FUNC	XmgActionPopdownAllMenusAfterMe(w,event,params,num_params)
Widget			w;
XEvent			*event;
String			*params;
Cardinal		*num_params;
{
  int			i;
  CoreWidget		parent;
  Boolean		ok;

  ok = False;
  parent = XtParent(w);
  i = 0;
  while (i < parent->core.num_popups)
    {
      if (ok)
	{
#ifdef NOTDEF
	  XtCallActionProc(parent->core.popup_list[i],
			   "XtMenuPopdown",
			   event,
			   params,
			   *num_params);
#else
	  XmgPopdown(parent->core.popup_list[i]);
#endif
	}
      if (parent->core.popup_list[i] == w)
	ok = True;
      i++;
    }
}

XtActionsRec            XmgActions[] =
{
  {"XmgNop",		XmgActionNop},
  {"XmgPopdown",	XmgActionPopdown},
  {"XmgDestroy",	XmgActionDestroy},
  {"XmgSetFocus",	XmgActionSetFocus},
  {"XmgPopdownAllMenus",XmgActionPopdownAllMenus},
  {"XmgPopdownAllMenusAfterMe",XmgActionPopdownAllMenusAfterMe},
};

/* installs a WM_DELETE_WINDOW action which destroys the widget. */
VOID_FUNC		XmgSetDestroyOnDelete(w)
Widget			w;
{
  Atom			atom;

  atom = XInternAtom(XtDisplay(w),"WM_DELETE_WINDOW",False);
  XtOverrideTranslations(w,
               XtParseTranslationTable("<Message>WM_PROTOCOLS: XmgDestroy()"));
  XSetWMProtocols(XtDisplay(w),XtWindow(w),&atom,1);
}

/* installs a WM_DELETE_WINDOW action which popdowns the widget. */
VOID_FUNC		XmgSetPopdownOnDelete(w)
Widget			w;
{
  Atom			atom;

  atom = XInternAtom(XtDisplay(w),"WM_DELETE_WINDOW",False);
  XtOverrideTranslations(w,
                 XtParseTranslationTable("<Message>WM_PROTOCOLS: XmgPopdown()"));
  XSetWMProtocols(XtDisplay(w),XtWindow(w),&atom,1);
}

/* installs a <Key>Return action which does nothing. */
VOID_FUNC		XmgSetNopOnReturn(w)
Widget			w;
{
  XtOverrideTranslations(w,
                         XtParseTranslationTable("<Key>Return: XmgNop()"));
}

/* prints a warning message using XtWarning(3) */
#ifdef HAVE_STDARG_H
VOID_FUNC		XmgWarning(char *fmt,...)
#else
VOID_FUNC		XmgWarning(fmt,va_alist)
Widget			parent;
char			*fmt;
va_dcl
#endif
{
  va_list		ap;
  char			buf[BUFSIZ];
  
  assert(fmt);
#ifdef HAVE_STDARG_H
  va_start(ap,fmt);
#else
  va_start(ap);
  va_arg(ap,char *);
#endif
  vsnprintf(buf,sizeof (buf),fmt,ap);
  XtWarning(buf);
  va_end(ap);
}

/* centers a widget */
VOID_FUNC		XmgCenterWidget(w)
Widget			w;
{
  int			dwidth;
  int			dheight;
  Dimension		width;
  Dimension		height;
  Position		x;
  Position		y;

  dwidth = DisplayWidth(XtDisplay(w),DefaultScreen(XtDisplay(w)));
  dheight = DisplayHeight(XtDisplay(w),DefaultScreen(XtDisplay(w)));
  XtVaGetValues(w,
                XtNwidth,	&width,
                XtNheight,	&height,
                NULL);
  x = (dwidth - width)/2;
  y = (dheight - height)/2;
  XtVaSetValues(w,
                XtNx,		x,
                XtNy,		y,
                NULL);
}

/* tallies a widget with screen */
VOID_FUNC		XmgTallyWidgetWithScreen(w)
Widget			w;
{
  int			dwidth;
  int			dheight;
  Dimension		width;
  Dimension		height;
  Position		x;
  Position		y;
  Position		nx;
  Position		ny;
  Dimension		border_width;

  dwidth = DisplayWidth(XtDisplay(w),DefaultScreen(XtDisplay(w)));
  dheight = DisplayHeight(XtDisplay(w),DefaultScreen(XtDisplay(w)));
  XtVaGetValues(w,
		XtNx,		&x,
		XtNy,		&y,
                XtNwidth,	&width,
                XtNheight,	&height,
		XtNborderWidth,	&border_width,
                NULL);
  if ((x + width + border_width) > dwidth)
    nx = x - (x + width +border_width - dwidth);
  else
    nx = x;
  if ((y + height + border_width) > dheight)
    ny = y - (y + height + border_width - dheight);
  else
    ny = y;
  XtVaSetValues(w,
                XtNx,		nx,
                XtNy,		ny,
                NULL);
}

t_dict				*XmgPixmapDict = NULL;
t_dict				*XmgXFSDict = NULL;
t_dict				*XmgPixelDict = NULL;

/* initializes Xmg actions, XmgPixmapDict, XmgXFSDict, XmgPixelDict */
t_status			XmgInit(VOID_DECL)
{
  t_status			status;

  XtAppAddActions(app_context,XmgActions,XtNumber(XmgActions));
  if ((XmgPixmapDict = XMG_SMALL_DICT_NEW(&status)) == NULL)
    return (status);
  if ((XmgXFSDict = XMG_SMALL_DICT_NEW(&status)) == NULL)
    return (status);
  if ((XmgPixelDict = XMG_SMALL_DICT_NEW(&status)) == NULL)
    return (status);
}

/* adds a XFontStruct* to XmgXFSDict.
   Returns 0 if OK. Might return various errors */
t_status			XmgXFSDictAdd(name,xfs)
char				*name;
XFontStruct			*xfs;
{
  t_hash_elt			*he;

  if (he = dict_get(XmgXFSDict,name))
    {
      return ;
    }
  else
    return (dict_add(XmgXFSDict,
		     name,
		     xfs));
}

/* retrieves a XFontStruct* from XmgXFSDict.
   Returns 0 if OK. Might return various errors */
t_status			XmgXFSDictGet(name,xfs)
char				*name;
XFontStruct			**xfs;
{
  t_hash_elt			*he;

  if (he = dict_get(XmgXFSDict,name))
    {
      (*xfs) = (XFontStruct *)(he->value);
      return (0);
    }
  return (-ERR_NOENT);
}

/* adds a Pixel to XmgPixelDict.
   Returns 0 if OK, might return various errors */
t_status			XmgPixelDictAdd(name,cmap,pixel)
char				*name;
Colormap			cmap;
Pixel				pixel;
{
  t_hash_elt			*he;
  t_status			status;
  t_vec				*vec;
  XmgPixelRec			*xpr;
  
  vec = NULL;
  if (he = dict_get(XmgPixelDict,name))
    {
      vec = (t_vec *)(he->value);
      VEC_FOR(vec,XmgPixelRec *x)
	{
	  if (cmap == x->cmap)
	    {
	      return ;
	    }
	}
      VEC_ENDFOR;
    }
  if (!vec)
    {
      if ((vec = XMG_VEC_ONE_NEW(&status)) == NULL)
	return (status);
      if ((status = dict_add(XmgPixelDict,
			     name,
			     vec)) < 0)
	{
	  vec_delete(vec);
	  return (status);
	}
    }
  if ((xpr = XMG_ALLOC_PROC(sizeof (XmgPixelRec),
			    "Xmg",
			    "XmgPixelDictAdd:xpr",
			    &status)) == NULL)
    return (status);
  xpr->cmap = cmap;
  xpr->pixel = pixel;
  if ((status = vec_add(vec,xpr)) < 0)
    {
      XMG_FREE_PROC(xpr,
		    "Xmg",
		    "*:xpr");
      return (status);
    }
  return (0);
}

/* retrieves a Pixel from XmgPixelDict.
   Returns 0 if OK, might return various errors */
t_status			XmgPixelDictGet(name,cmap,pixel_ret)
char				*name;
int				cmap;
Pixel				*pixel_ret;
{
  t_hash_elt			*he;
  t_status			status;

  if (he = dict_get(XmgPixelDict,name))
    {
      t_vec			*vec;

      vec = (t_vec *)(he->value);
      VEC_FOR(vec,XmgPixelRec *xpr)
	{
	  if (cmap == xpr->cmap)
	    {
	      (*pixel_ret) = xpr->pixel;
	      return (0);
	    }
	}
      VEC_ENDFOR;
    }
  return (-ERR_NOENT);
}

/* adds a Pixmap to XmgPixmapDict.
   Returns 0 if OK. Might return various errors */
t_status			XmgPixmapDictAdd(name,depth,pixmap)
char				*name;
int				depth; /* Might not be AnyDepth */
Pixmap				pixmap;
{
  t_hash_elt			*he;
  t_status			status;
  t_vec				*vec;
  XmgPixmapRec			*xpr;

#ifdef NOTDEF
  printf("XmgPixmapDictAdd(%s,%d)\n",name,depth);
#endif
  if (depth == AnyDepth)
    return (-ERR_INVAL);
  vec = NULL;
  if (he = dict_get(XmgPixmapDict,name))
    {
      vec = (t_vec *)(he->value);
      VEC_FOR(vec,XmgPixmapRec *x)
	{
	  if (depth == x->depth)
	    {
	      return ;
	    }
	}
      VEC_ENDFOR;
    }
  if (!vec)
    {
      if ((vec = XMG_VEC_ONE_NEW(&status)) == NULL)
	return (status);
      if ((status = dict_add(XmgPixmapDict,
			     name,
			     vec)) < 0)
	{
	  vec_delete(vec);
	  return (status);
	}
    }
  if ((xpr = XMG_ALLOC_PROC(sizeof (XmgPixmapRec),
			    "Xmg",
			    "XmgPixmapDictAdd:xpr",
			    &status)) == NULL)
    return (status);
  xpr->depth = depth;
  xpr->pixmap = pixmap;
  if ((status = vec_add(vec,xpr)) < 0)
    {
      XMG_FREE_PROC(xpr,
		    "Xmg",
		    "*:xpr");
      return (status);
    }
  return (0);
}

/* retrieves a Pixmap from XmgPixmapDict.
   If depth equals AnyDepth, then the deepest pixmap is choosen.
   Returns 0 if OK. Might return various errors */
t_status			XmgPixmapDictGet(name,depth,pixmap_ret)
char				*name;
int				depth; /* Might be AnyDepth */
Pixmap				*pixmap_ret;
{
  t_hash_elt			*he;
  t_status			status;
  int				best_depth;
  Pixmap			best_pixmap;

#ifdef NOTDEF
  printf("XmgPixmapDictGet(%s,%d)\n",name,depth);
#endif
  best_depth = 0;
  if (he = dict_get(XmgPixmapDict,name))
    {
      t_vec			*vec;

      vec = (t_vec *)(he->value);
      VEC_FOR(vec,XmgPixmapRec *xpr)
	{
	  if (depth == xpr->depth)
	    {
	      (*pixmap_ret) = xpr->pixmap;
	      return (0);
	    }
	  if (xpr->depth > best_depth)
	    {
	      best_depth = xpr->depth;
	      best_pixmap = xpr->pixmap;
	    }
	}
      VEC_ENDFOR;
      if (depth == AnyDepth && VEC_COUNT(vec) > 0)
	{
	  (*pixmap_ret) = best_pixmap;
	  return (0);
	}
    }
  return (-ERR_NOENT);
}

/* force the named Pixmap from XmgPixmapDict to match depth.
   It recreates a new pixmap if needed. 
   Returns 0 if OK. Might return various errors */
t_status			XmgPixmapDictForceGet(display,
						      name,
						      depth,
						      pixmap_ret)
Display				*display;
char				*name;
int				depth;
Pixmap				*pixmap_ret;
{
  t_status			status;
  Window			root;
  int				x;
  int				y;
  unsigned int			width;
  unsigned int			height;
  unsigned int			border_width;
  unsigned int			real_depth;
  Pixmap			new_pixmap;
  
#ifdef NOTDEF
  printf("XmgPixmapDictForceGet(%s,%d)\n",name,depth);
#endif
  if ((status = XmgPixmapDictGet(name,
				 depth,
				 pixmap_ret)) == 0)
    return (0);
  if (depth == AnyDepth)
    return (-ERR_NOENT);
  if ((status = XmgPixmapDictGet(name,
				 AnyDepth,
				 pixmap_ret)) < 0)
    return (status);
  XGetGeometry(display,
	       *pixmap_ret,
	       &root,
	       &x,
	       &y,
	       &width,
	       &height,
	       &border_width,
	       &real_depth);
  if (real_depth == 1)
    {
      GC			gc;
      XGCValues			xgcv;

      new_pixmap = XCreatePixmap(display,
				 DefaultRootWindow(display),
				 width,
				 height,
				 depth);
      xgcv.foreground = BlackPixel(display,
				   DefaultScreen(display));
      xgcv.background = WhitePixel(display,
				   DefaultScreen(display));
      gc = XCreateGC(display,
		     DefaultRootWindow(display),
		     GCForeground|GCBackground,
		     &xgcv);
      XCopyPlane(display,
		 *pixmap_ret,
		 new_pixmap,
		 gc,
		 0,
		 0,
		 width,
		 height,
		 0,
		 0,
		 1);
      XFreeGC(display,gc);
      if ((status = XmgPixmapDictAdd(name,
				     depth,
				     new_pixmap)) < 0)
	{
	  XFreePixmap(display,new_pixmap);
	  return (status);
	}
      (*pixmap_ret) = new_pixmap;
      return (0);
    }
  return (-ERR_NI);
}

t_status			XmgDestroyPixmapDictWalk(he,unused)
t_hash_elt			*he;
VOID_PTR			unused;
{
  t_vec				*vec;

  vec = (t_vec *)(he->value);
  VEC_FOR(vec,XmgPixmapRec *xpr)
    {
      XFreePixmap(XtDisplay(toplevel),
		  xpr->pixmap);
      XMG_FREE_PROC(xpr,
		    "Xmg",
		    "*:xpr");
    }
  VEC_ENDFOR;
  vec_delete(vec);
  return (0);
}

t_status			XmgDestroyXFSDictWalk(he,unused)
t_hash_elt			*he;
VOID_PTR			unused;
{
  XFreeFont(XtDisplay(toplevel),
	    (XFontStruct *)(he->value));
  return (0);
}

t_status			XmgDestroyPixelDictWalk(he,unused)
t_hash_elt			*he;
VOID_PTR			unused;
{
  t_vec				*vec;

  vec = (t_vec *)(he->value);
  VEC_FOR(vec,XmgPixelRec *xpr)
    {
      XFreeColors(XtDisplay(toplevel),
		  xpr->cmap,
		  &(xpr->pixel),
		  1,
		  0);
      XMG_FREE_PROC(xpr,
		    "Xmg",
		    "*:xpr");
    }
  VEC_ENDFOR;
  vec_delete(vec);
  return (0);
}

/* destroys XmgPixmapDict, XmgXFSDict and XmgPixelDict freeing appropriate
 X resources (Pixmaps, XFontStructs and Pixels) */
VOID_FUNC			XmgDestroy(VOID_DECL)
{
  t_status			status;

  status = dict_walk(XmgPixmapDict,
		     (t_dict_walk_proc)XmgDestroyPixmapDictWalk,
		     NULL);
  dict_delete(XmgPixmapDict);
  status = dict_walk(XmgXFSDict,
		     (t_dict_walk_proc)XmgDestroyXFSDictWalk,
		     NULL);
  dict_delete(XmgXFSDict);
  status = dict_walk(XmgPixelDict,
		     (t_dict_walk_proc)XmgDestroyPixelDictWalk,
		     NULL);
  dict_delete(XmgPixelDict);
}

#ifdef DEBUG
/* shows a XRectangle.
   This is a debug function. */
VOID_FUNC			XmgShowRect(rect)
XRectangle			*rect;
{
  fprintf(stderr,"+%d+%d %dx%d\n",
	  rect->x,
	  rect->y,
	  rect->width,
	  rect->height);
}

t_status			XmgPixmapDictStatusWalk(he,unused)
t_hash_elt			*he;
VOID_PTR			unused;
{
  t_vec				*vec;

  vec = (t_vec *)(he->value);
  fprintf(stderr,"%s\n",he->key);
  VEC_FOR(vec,XmgPixmapRec *xpr)
    {
      fprintf(stderr,"\t%d %x\n",xpr->depth,xpr->pixmap);
    }
  VEC_ENDFOR;
  return (0);
}

/* shows XmgPixmapDict, XmgPixelDict and XmgXFSDict.
   This is a debug function. */
VOID_FUNC			XmgStatus(VOID_DECL)
{
  fprintf(stderr,"pixmaps:\n");
  dict_walk(XmgPixmapDict,
	    (t_dict_walk_proc)XmgPixmapDictStatusWalk,
	    NULL);
  fprintf(stderr,"pixels:\n");
  dict_show_sorted(XmgPixelDict);
  fprintf(stderr,"xfs:\n");
  dict_show_sorted(XmgXFSDict);
}
#endif
