/* ----------------------------------------------------------------------------
 * Copyright (C) 1995-1999 by Karim Kaschani
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * to rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
 * --------------------------------------------------------------------------*/

#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include "xlogin.h"
#include "net.h"





/* ----------------------------------------------------------------------------
 * removeXHost - remove single entry in current xhost table
 * ------------------------------------------------------------------------- */

void removeXHost(XHostData *xhosts, int entry, int number)
{
   int		i;

   if (xhosts != NULL) {
      if (xhosts[entry].addr != NULL) {
         free(xhosts[entry].addr);
         xhosts[entry].addr = NULL;
   
         for (i=entry+1; i<number; i++) {
             xhosts[i-1].addr = xhosts[i].addr;
             xhosts[i].addr = NULL;
         }
      }
   }
}





/* ----------------------------------------------------------------------------
 * freeXHosts - free current xhost table
 * ------------------------------------------------------------------------- */

void freeXHosts(XHostData *xhosts, int number)
{
   XHostData	*x;
   int		i;

   x = xhosts;

   if (xhosts != NULL) {
      for (i=0; i<number; i++) {
          if (x->addr != NULL) free(x->addr);
          x++;
      }
      
      free(xhosts);
   }
}





/* ----------------------------------------------------------------------------
 * getXHosts - get hosts of current xhost table
 * ------------------------------------------------------------------------- */

XHostData *getXHosts(Display *disply, int *number)
{
   XHostAddress		*x_p0, *x_p;		/* xhost table pointer    */
   Bool			status;			/* access control status  */
   int			i;
   XHostData		*xhosts, *x;		/* xhosts table           */

   /* ---------------------------------------------------- query XHost table */
   
   x_p = x_p0 = XListHosts(disply, number, &status);

   if (*number > 0 && x_p0 != NULL) {
      xhosts = x = (XHostData *) malloc(sizeof(XHostData) * (*number));

      for (i=0; i<*number; i++) {
          x->addr = (char *) malloc(sizeof(char) * 16);

          sprintf(x->addr, "%d.%d.%d.%d%c", (unsigned char) x_p->address[0],
                                            (unsigned char) x_p->address[1],
                                            (unsigned char) x_p->address[2],
                                            (unsigned char) x_p->address[3],
                                            '\0');
          x_p++;
          x++;
      }

      XFree(x_p0);
   } else {
      xhosts = NULL;
   }

   return(xhosts);
}





/* ----------------------------------------------------------------------------
 * updateXHost - update xhost table
 * ------------------------------------------------------------------------- */

void updateXHost(char *host_addr, int addHost)
{
    struct hostent		*hPtr;               /* host entry pointer   */
    XHostAddress 		*XHAddr;             /* XHostAddress pointer */
    Display			*disply;

    XHAddr = NULL;

    if ((hPtr = getHostEnt(host_addr)) != NULL) {

       /* only taking into account internet addresses */

       if (hPtr->h_addrtype == AF_INET && *(hPtr->h_addr_list) != NULL) {
          XHAddr = (XHostAddress *) malloc(sizeof(XHostAddress));

          XHAddr->family  = FamilyInternet;
          XHAddr->length  = hPtr->h_length;
          XHAddr->address = *(hPtr->h_addr_list);

	  /* open display to connect to local X server */

          if ((disply = XOpenDisplay(NULL)) != NULL) {

             /* update xhost table */

             if (addHost) {
                XAddHost(disply, XHAddr);
                fprintf(stderr, "updateXHost: %s being added ", host_addr);
                fprintf(stderr, "to access control list\n");
             } else {
                XRemoveHost(disply, XHAddr);
                fprintf(stderr, "updateXHost: %s being removed ", host_addr);
                fprintf(stderr, "from access control list\n");
             }

             XCloseDisplay(disply);
          } else {
             fprintf(stderr, "updateXHost: cannot connect to local X server ");
             fprintf(stderr, "- check local DISPLAY environment variable\n");
          }

          /* free reserved memory */

          XHAddr->address = NULL;
          free(XHAddr);
       } else {
          fprintf(stderr,
                  "updateXHost: access denied - %s is no internet address\n",
                  host_addr);
       }
    } else {
       fprintf(stderr,
               "updateXHost: access denied - unresolved host address %s\n",
               host_addr);
    }
}





/* ----------------------------------------------------------------------------
 * ExecXHost - execute xhost update procedure
 * ------------------------------------------------------------------------- */

int ExecXHost(char *host_addr, char *rc_path, int addHost)
{
   char		dotfile[MAX_CHARS];
   FILE		*filep;
   pid_t	pid;
   int		n;
   struct stat	files;

   (void) strcpy(dotfile,rc_path);
   (void) strcat(dotfile,"/.");
   (void) strcat(dotfile,host_addr);
   (void) strcat(dotfile,"@");
   (void) strcat(dotfile,hostname);

   /* ------------------------------------------------------- update dotfile */

   if (!(filep = fopen(dotfile, "r+"))) {
      if (addHost) {
         if (!(filep = fopen(dotfile, "w"))) {
            perror("ERROR: ");
            exit(2);
         } else {
            n = 1;
            fprintf(filep,"%d\n",n);
         }
      } else {
         fprintf(stderr,"ExecXHost: SESSION MISCOUNT.\n");
      }
   } else {
      fscanf(filep,"%d\n",&n);
      if (addHost)
         n++;
      else
         n--;
      rewind(filep);
      fprintf(filep,"%d\n",n);
   }

   fclose(filep);

   /* ----------------------------------------------- check xhost permission */

   if (stat(dotfile, &files) != 0) {
      perror("ERROR: ");
      exit(2);
   } else {
      if (files.st_mode & (S_IXGRP | S_IXOTH)) {
         return(FAIL);
      }
   }

   /* -------------------------------------------------------- execute xhost */

   if ((addHost && (n == 1)) || (!addHost && (n == 0))) {

      if (addHost) {
         fprintf(stderr, "ExecXHost: adding %s ...\n", host_addr);
      } else {
         fprintf(stderr, "ExecXHost: removing %s ...\n", host_addr);
      }

      updateXHost(host_addr, addHost);
   }
   
   if (n == 0)
      if (unlink(dotfile) != 0) {
         perror("ERROR: ");
         exit(2);
      }

   return(SUCCESS);
}





/* ----------------------------------------------------------------------------
 * ForceXHost - force xhost update procedure
 * ------------------------------------------------------------------------- */

int ForceXHost(char *host_addr, char *rc_path, int addHost, int lockfile)
{
   char		dotfile[MAX_CHARS];
   FILE		*filep;
   pid_t	pid;
   int		n;

   (void) strcpy(dotfile,rc_path);
   (void) strcat(dotfile,"/.");
   (void) strcat(dotfile,host_addr);
   (void) strcat(dotfile,"@");
   (void) strcat(dotfile,hostname);

   /* ------------------------------------------------------- update dotfile */

   if (lockfile) {

      /* -------------------------------------- enable/disable server access */
      
      if (access(dotfile, F_OK) != 0) {
         if (!(filep = fopen(dotfile, "w"))) {
            perror("ERROR: ");
            exit(2);
         } else {
            fprintf(filep,"0\n");
            fclose(filep);
         }
      }
      
      if (addHost) {

         /* ------------------------------------------- enable server access */

         if (chmod(dotfile, S_IRUSR | S_IWUSR | S_IXGRP) != 0) {
            perror("ERROR: ");
            exit(2);
         }
      } else {

         /* ------------------------------------------ disable server access */

         if (chmod(dotfile, S_IRUSR | S_IWUSR | S_IXOTH) != 0) {
            perror("ERROR: ");
            exit(2);
         }
      }
   } else {

      /* ------------------------------------------- automatic server access */

      if ((filep = fopen(dotfile, "r")) != NULL) {
         fscanf(filep,"%d\n",&n);
         fclose(filep);

         if (n > 0) {
            addHost = TRUE;
            if (chmod(dotfile, S_IRUSR | S_IWUSR) != 0) {
               perror("ERROR: ");
               exit(2);
            }
         } else {
            addHost = FALSE;
            if (unlink(dotfile) != 0) {
               perror("ERROR: ");
               exit(2);
            }
         }
      } else {
         addHost = FALSE;
      }
   }

   /* -------------------------------------------------------- execute xhost */

   if (addHost)
      fprintf(stderr, "ForceXHost: adding %s ...\n", host_addr);
   else
      fprintf(stderr, "ForceXHost: removing %s ...\n", host_addr);

   updateXHost(host_addr, addHost);
   
   return(SUCCESS);
}





/* ----------------------------------------------------------------------------
 * StatXHost - get xhost status
 * ------------------------------------------------------------------------- */

int StatXHost(char *host_addr, char *rc_path)
{
   char		dotfile[MAX_CHARS];
   struct stat	files;

   (void) strcpy(dotfile,rc_path);
   (void) strcat(dotfile,"/.");
   (void) strcat(dotfile,host_addr);
   (void) strcat(dotfile,"@");
   (void) strcat(dotfile,hostname);

   /* ----------------------------------------------- check xhost permission */

   if (stat(dotfile, &files) == 0) {
      if (files.st_mode & S_IXGRP)
         return(ENABLE);		/* server access enabled   */
      if (files.st_mode & S_IXOTH)
         return(DISABLE);		/* server access disabled  */
   }

   return(AUTOMATIC);			/* automatic server access */
}
