/* ----------------------------------------------------------------------------
 * Copyright (C) 1999 by Karim Kaschani
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * to rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
 * --------------------------------------------------------------------------*/

#include "net.h"
#include "strutils.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>

#ifdef SVR4
#  include <utmpx.h>
#else
#  include <utmp.h>
#endif

#include <X11/Xlib.h>





/* ----------------------------------------------------------------- defines */

#define TIMEOUT	1000

#define	TRUE	1
#define FALSE	0

#ifndef UTMP_FILE
#  define	UTMP_FILE	"/etc/utmp"
#endif

#ifndef WTMP_FILE
#  define	WTMP_FILE	"/etc/wtmp"
#endif

#if !defined (SVR4) && !defined (SYSV)
#  define	SEEK_SET	0
#  define	SEEK_CUR	1
#  define	SEEK_END	2
#endif





/* ----------------------------------------------------------------- globals */

static int	debug = FALSE;				/* debugging flag   */
static char	*Version = 				/* copyright notice */
"xDisplay Version 1.0\n\
\n\
Copyright (C) 1999 by Karim Kaschani\n\
\n\
Permission is hereby granted, free of charge, to any person obtaining a\n\
copy of this software and associated documentation files (the 'Software'),\n\
to deal in the Software without restriction, including without limitation\n\
to rights to use, copy, modify, merge, publish, distribute, sublicense,\n\
and/or sell copies of the Software, and to permit persons to whom the\n\
Software is furnished to do so, subject to the following conditions:\n\
\n\
The above copyright notice and this permission notice shall be included in\n\
all copies or substantial portions of the Software.\n\
\n\
THE SOFTWARE IS PROVIDED 'AS IS', WITHOUT WARRANTY OF ANY KIND, EXPRESS OR\n\
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,\n\
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL\n\
THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER\n\
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN\n\
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.";





/* ----------------------------------------------------------------------------
 * fprintd - print debugging information
 * --------------------------------------------------------------------------*/

void fprintd(FILE *fp, char *fmt, ...)
{
	va_list		ap;
	char		*p, *sval;
	int		ival;
	long		lval;
	double		dval;

	/* ------------------------------------------------ empty debug flag */

	if (! debug)
	   return;

	/* -------------------------------------------- interprete arguments */

	va_start(ap, fmt);

	for (p = fmt; *p; p++) {
	    if (*p != '%') {
	       fputc(*p, fp);
	       continue;
	    }

	    switch (*++p) {
	           case 'd': ival = va_arg(ap, int);
	                     fprintf(fp, "%d", ival);
	                     break;
	           case 'l': lval = va_arg(ap, long);
	                     fprintf(fp, "%ld", lval);
	                     break;
	           case 'f': dval = va_arg(ap, double);
	                     fprintf(fp, "%f", dval);
	                     break;
	           case 's': for (sval = va_arg(ap, char *); *sval; sval++)
	                         fputc(*sval, fp);
	                     break;
	           default : fputc(*p, fp);
	                     break;
	    }
	}
}





/* ----------------------------------------------------------------------------
 * empty - check for empty null string
 * ------------------------------------------------------------------------- */

int empty(char *string)
{
	int	status = TRUE;
	int	i;

	if (string != NULL) {
	   for (i=0; i<strlen(string); i++) {
	       if (string[i] != ' ') {
	          status = FALSE;
	          break;
	       }
	   }
	}

	return status;
}




 
/* ----------------------------------------------------------------------------
 * lookup - scan [uw]tmp(x) file for terminal entry
 * ------------------------------------------------------------------------- */

#ifdef SVR4
int lookup(char *file, char *terminal, int timeout, struct utmpx **tty)
#else
int lookup(char *file, char *terminal, int timeout, struct utmp **tty)
#endif
{
	char		ut_term[257];
	int		fd;
	size_t		bytes;
	time_t		seconds, sec_diff, *tloc=NULL;

#ifdef SVR4
	bytes = sizeof(struct utmpx);
	*tty   = (struct utmpx *) malloc(bytes);
#else
	bytes = sizeof(struct utmp);
	*tty   = (struct utmp *) malloc(bytes);
#endif

	seconds = time(tloc);

	/* -------------------------------------------- open [uw]tmp(x) file */

	if ((fd = open(file, O_RDONLY)) < 0) {
	   perror("xDisplay: ");
	   exit(1);
	}

	/* -------------------------------------------------------- seek EOF */

	if (lseek(fd, 0, SEEK_END) == -1) {
	   perror("xDisplay: ");
	   exit(1);
	}

	/* ------------------------------------------- seek preceeding entry */

	while (lseek(fd, - (off_t) bytes, SEEK_CUR) != -1) {

	      if (read(fd, *tty, bytes) == -1) {
	         perror("xDisplay: ");
	         exit(1);
	      }

	      /* ------------------ expand terminal entry in [uw]tmp(x) file */

	      if (strncmp((*tty)->ut_line, "/dev/", 5) != 0) {
	         (void) strcpy(ut_term, "/dev/");
	         (void) strcat(ut_term, (*tty)->ut_line);
	      } else {
	         (void) strcpy(ut_term, (*tty)->ut_line);
	      }

	      /* ------------- time difference = current time - logging time */

#ifdef SVR4
	      sec_diff = seconds - (*tty)->ut_tv.tv_sec;
	      (*tty)->ut_tv.tv_sec = sec_diff;
#else
	      sec_diff = seconds - (*tty)->ut_time;
	      (*tty)->ut_time = sec_diff;
#endif

	      fprintd(stderr, "* \n");
	      fprintd(stderr, "* %s Terminal: %s\n", file, ut_term);
	      fprintd(stderr, "* %s Hostname: %s\n", file, (*tty)->ut_host);
	      fprintd(stderr, "* %s Time: %l\n", file, sec_diff);

	      /* ----------------------------------------------- EOF reached */

	      if (lseek(fd, - (off_t) bytes, SEEK_CUR) == -1) {
	         fprintd(stderr, "* \n");
	         fprintd(stderr, "* EOF reached.\n");

	         close(fd);
	         free(*tty);

	         return -1;
	      }

	      /* --------------- expired entries (older than 1 hour) reached */

	      if (sec_diff > timeout && timeout > 0) {
	         fprintd(stderr, "* \n");
	         fprintd(stderr, "* TIMEOUT reached.\n");

	         close(fd);
	         free(*tty);

	         return -1;
	      }

	      /* ------------------------------------ found terminal entry ? */

	      if (StrCmp(terminal, ut_term) == 0) {
	         close(fd);
	         return 0;
	      }
	};
}





/* ----------------------------------------------------------------------------
 * isValidDisplay - check a hostname:displaynumber struct for validity
 * --------------------------------------------------------------------------*/

int isValidDisplay (char *host, char *dno, char *term)
{
    Display	*display;
    char	*DISPLAY;
    int		isValid = FALSE;

    if (host != NULL && dno != NULL) {
       DISPLAY = (char *) malloc(sizeof(char) * (strlen(host)+strlen(dno)+1));

       (void) strcpy(DISPLAY, host);
       (void) strcat(DISPLAY, dno);

       /* ------------------------ Suppress error messages from XOpenDisplay */

       if (term != NULL)
          (void) freopen("/dev/null", "a", stderr);

       if ((display = XOpenDisplay(DISPLAY)) != NULL) {
          XCloseDisplay(display);
          isValid = TRUE;
       }

       /* ---------------------------------------------- Enable stderr again */

       if (term != NULL)
          (void) freopen(term, "a", stderr);

       free(DISPLAY);
    }

    return isValid;
}





/* ----------------------------------------------------------------------------
 * xDisplay - evaluate the correct hostname:displaynumber struct by query of a
 *            pre-defined DISPLAY variable and the remote host's [uw]tmp or
 *            [uw]tmpx table
 * ------------------------------------------------------------------------- */

int main (int argc, char **argv)
{
	extern char	*optarg;
	char		*pos, *uwHost, *uwNo, *terminal;
	char		*envDisp, *envHost, *envNo;
	char		*defHost, *defNo;
	char		tmp[100];
	int		c;
	int		address = TRUE;
	int		error = FALSE;
	int		interactive = FALSE;
	int		ustat, wstat;
	extern int	opterr;

#ifdef SVR4
	struct utmpx 	*u, *w, *tty;
#else
	struct utmp 	*u, *w, *tty;
#endif

	/* -------------------------------------------------- initialization */

	uwHost = uwNo = envDisp = envHost = envNo = defHost = defNo = NULL;
	tty = NULL;

	/* -------------------------------------- parse command line options */

	opterr = 0;

	while ((c = getopt(argc, argv, "ainDvd:")) != EOF) {
	      switch (c) {
	             case 'a': address = TRUE;
	                       break;
	             case 'i': interactive = TRUE;
	                       break;
	             case 'n': address = FALSE;
	                       break;
	             case 'd': defNo = optarg;
	                       break;
	             case 'D': debug = TRUE;
	                       break;
	             case 'v': fprintf(stderr, "%s\n", Version);
	                       exit(0);
	                       break;
	             case '?': error = TRUE;
	                       break;
	      }
	}

	if (error) {
	   fprintf(stderr, "usage: xDisplay [-a] [-n] [-i] [-d :<displaynumber>] [-D]\n");
	   exit(2);
	}

	/* -------------------------------- get current terminal name if any */

	   if ((terminal = ttyname(STDIN_FILENO)) != NULL) {
	      fprintd(stderr, "* Current terminal: %s\n\n", terminal);
	   } else {
	      fprintf(stderr, "WARNING: Cannot determine current terminal.\n\n");
	   }

	/* ------------------------------- set-up default display parameters */

	defHost = (char *) malloc(sizeof(char));
	defHost = getLHostName(defHost);

	if (substAddr(&defHost) == -1) {
	   fprintf(stderr, "ERROR: CANNOT RESOLVE IP ADDRESS OF ");
	   fprintf(stderr, "LOCAL HOST '%s' - GIVING UP.\007\n", defHost);

	   exit(1);
	}

	if (defNo == NULL) {
	   defNo = (char *) malloc(sizeof(char) * 5);
	   (void) strcpy(defNo, ":0.0");
	}

	fprintd(stderr, "* Set-up default DISPLAY: %s%s\n", defHost, defNo);

	/* ----------------------------- read former DISPLAY variable if any */

	if ((envDisp = getenv("DISPLAY")) != NULL) {
	   char	*p;

	   /* ............................. check for valid syntax, i.e. ':' */

	   if ((p = strrchr(envDisp, ':')) != NULL) {
	      int l_Host, l_No;

	      l_No   = strlen(p);
	      l_Host = strlen(envDisp) - l_No;

	      envNo   = (char *) malloc(sizeof(char) * (l_No + 1));

	      (void) strcpy(envNo, p);

	      if (l_Host > 0) {
	         envHost = (char *) malloc(sizeof(char) * (l_Host + 1));

	         (void) StrNCpy(envHost, envDisp, l_Host);
	      } else {
	         envHost = (char *) malloc(sizeof(char) * (strlen(defHost)+1));
	         (void) strcpy(envHost, defHost);
	      }

	      fprintd(stderr, "* Found DISPLAY variable: %s%s\n",
	              envHost, envNo);

	      if (substAddr(&envHost) == -1) {
	         fprintd(stderr, "* \n");
	         fprintd(stderr, "* WARNING: Cannot resolve ip address of ");
	         fprintd(stderr, "host: %s\n",envHost);
	         fprintd(stderr, "*          Ignoring DISPLAY variable: %s%s\n",
	                 envHost, envNo);

	         free(envHost);
	         free(envNo);
	         envHost = NULL;
	         envNo   = NULL;
	      } else {

	         /* ...................... try to contact remote (!) Xserver */

	         if (StrCmp(envHost, defHost) != 0) {

	            fprintd(stderr, "* Verifying DISPLAY variable: %s%s ",
	                    envHost, envNo);

	            if (! isValidDisplay(envHost, envNo, terminal)) {
	               fprintd(stderr, "- ACCESS DENIED.\n");

	               free(envHost);
	               free(envNo);
	               envHost = envNo = NULL;
	            } else {
	               fprintd(stderr, "- OK.\n");
	            }
	         }
	      }
	   } else {
	      fprintd(stderr, "* Found corrupt DISPLAY variable: %s", envDisp);
	      fprintd(stderr, " - ignored.\n");
	   }
	}

	/* the DISPLAY variable is preferred over [uw]tmp? entry, unless it
	 * points to the local Xserver                                       */

	if (envHost != NULL && envNo != NULL &&
	   (StrCmp(envHost, defHost) != 0)) {

	   uwHost  = envHost;
	   uwNo    = envNo;
	   envHost = NULL;
	   envNo   = NULL;
	} else {

	   /* --------------------- lookup terminal entry in [uw]tmp(x) file */

	   if (terminal != NULL) {

#ifdef SVR4
	      ustat = lookup(UTMPX_FILE, terminal, 0, &u);
	      wstat = lookup(WTMPX_FILE, terminal, TIMEOUT, &w);

	      if (ustat == 0 && wstat == 0) {
	         if (u->ut_tv.tv_sec < w->ut_tv.tv_sec) {
#else
	      ustat = lookup(UTMP_FILE, terminal, 0, &u);
	      wstat = lookup(WTMP_FILE, terminal, TIMEOUT, &w);

	      if (ustat == 0 && wstat == 0) {
	         if (u->ut_time < w->ut_time) {
#endif
	            tty = u;
	            free(w);
	         } else {
	            tty = w;
	            free(u);
	         }
	      } else if (ustat == 0) {
	         tty = u;
	      } else if (wstat == 0) {
	         tty = w;
	      } else {
	         tty = NULL;
	      }
	   }

	   /* --------------------------------------- found terminal entry ? */

	   if (tty != NULL) {

	      /* .......................... got the name of the remote host? */
	      
              if (! empty(tty->ut_host)) {

	         /* separate display extension ":*" if any from hostname */

	         if ((pos = strrchr(tty->ut_host, ':')) != NULL) {
	            int l_Host, l_No;

	            l_No   = strlen(pos);
#ifdef SVR4
	            l_Host = tty->ut_syslen - l_No;
#else
	            l_Host = strlen(tty->ut_host) - l_No;
#endif
	            uwNo   = (char *) malloc(sizeof(char) * (l_No + 1));

	            (void) strcpy(uwNo, pos);

	            if (l_Host > 0) {
	               uwHost = (char *) malloc(sizeof(char) * (l_Host + 1));

	               (void) StrNCpy(uwHost, tty->ut_host, l_Host);
	            } else {
	               uwHost = (char *) malloc(sizeof(char) *
	                        (strlen(defHost)+1));
	               (void) strcpy(uwHost, defHost);
	            }
	         } else {
#ifdef SVR4
	            uwHost = (char *) malloc(sizeof(char) * tty->ut_syslen);
#else
	            uwHost = (char *) malloc(sizeof(char) *
	                      (strlen(tty->ut_host)+1));
#endif
	            (void) strcpy(uwHost, tty->ut_host);
	         }

	         free(tty);
	      } else {
	         uwHost = (char *) malloc(sizeof(char) * (strlen(defHost)+1));
	         (void) strcpy(uwHost, defHost);
	      }

	      /* workaround to determine the host address of a crippled host
	       * name entry by guessing its alias (hope it works) */
		 
	      fprintd(stderr, "* \n");
	      fprintd(stderr, "* Found remote host: %s", uwHost);

	      while (substAddr(&uwHost) == -1 &&
	            (pos = strrchr(uwHost, '.')) != NULL) {
	            (void) strcpy(pos, "\0");
	      }	            

	      fprintd(stderr, " (%s)\n", uwHost);

	      /* .......................... is display number still unknown? */

	      if (uwNo == NULL) {
	         if (StrCmp(uwHost, envHost) == 0 && envNo != NULL) {
	            uwNo  = (char *) malloc(sizeof(char) * (strlen(envNo)+1));
	            (void) strcpy(uwNo, envNo);
	         } else {
	            uwNo  = (char *) malloc(sizeof(char) * (strlen(defNo)+1));
	            (void) strcpy(uwNo, defNo);
	         }
	      }

	      fprintd(stderr, "* Retrieved DISPLAY: %s%s\n", uwHost, uwNo);

	      /* ......................... try to contact remote (!) Xserver */

	      if (StrCmp(uwHost, defHost) != 0) {

	         fprintd(stderr, "* Verifying DISPLAY: %s%s ", uwHost, uwNo);

	         if (! isValidDisplay(uwHost, uwNo, terminal)) {
	            fprintd(stderr, "- ACCESS DENIED.\n");

	            free(uwHost);
	            free(uwNo);
	            uwHost = uwNo = NULL;
	         } else {
	            fprintd(stderr, "- OK.\n");
	         }
	      }
	   }
	}

	/* --------------------------- Have we got a valid DISPLAY at least? */

	if (uwHost != NULL && uwNo != NULL) {
	   fprintd(stderr, "* \n");
	   fprintd(stderr, "* Using DISPLAY variable: %s%s\n", uwHost, uwNo);

	   /* ......................... get official host name, if requested */

	   if (!address) {
	      uwHost = getRHostName(uwHost);

	      fprintd(stderr, "* Resolved to DISPLAY: %s%s\n", uwHost, uwNo);
	   }

	   /* ................................................. print result */

	   fprintf(stdout, "%s%s\n", uwHost, uwNo);
	} else {
	   fprintf(stderr, "xDisplay: No valid DISPLAY found\007\n");

	   /* ................................. query user for valid display
	    *                           only if we are running on a terminal */

	   while (interactive && isatty(0) &&
	         (uwHost == NULL || uwNo == NULL)) {
	      fprintf(stderr, "          Enter DISPLAY [<host>:<number>]: ");
	      fgets(tmp, 100, stdin);
	      pos = strrchr(tmp, '\n');
	      strcpy(pos, "\0");


	      if ((pos = strrchr(tmp, ':')) != NULL) {
	         int l_Host, l_No;

	         l_No   = strlen(pos);
	         l_Host = strlen(tmp) - l_No;
	         uwNo   = (char *) malloc(sizeof(char) * (l_No + 1));

	         (void) strcpy(uwNo, pos);

	         if (l_Host > 0) {
	            uwHost = (char *) malloc(sizeof(char) * (l_Host + 1));

	            (void) StrNCpy(uwHost, tmp, l_Host);
	         } else {
	            uwHost = (char *) malloc(sizeof(char)*(strlen(defHost)+1));
	            (void) strcpy(uwHost, defHost);
	         }
	      }

	      /* ............................................ check validity */

	      if (! isValidDisplay(uwHost, uwNo, terminal)) {
	         fprintf(stderr, "   ERROR: Invalid DISPLAY\007\n");

	         if (uwHost != NULL) free(uwHost);
	         if (uwNo != NULL) free(uwNo);
	         uwHost = uwNo = NULL;
	      } else {
	         fprintf(stdout, "%s%s\n", uwHost, uwNo);
	      }
	   }
	}

	/* ------------------------------------------- free allocated memory */

	if (uwHost != NULL) free(uwHost);
	if (uwNo != NULL) free(uwNo);
	if (envHost != NULL) free(envHost);
	if (envNo != NULL) free(envNo);
	if (defHost != NULL) free(defHost);
	if (defNo != NULL) free(defNo);

	exit(0);
}
