/* reg.c - general purpose window region package */

/*  Copyright 1991 Mark Russell, University of Kent at Canterbury.
 *
 *  You can do what you like with this source code as long as
 *  you don't try to make money out of it and you include an
 *  unaltered copy of this message (including the copyright).
 */

char ups_reg_c_sccsid[] = "@(#)reg.c	1.18 16 Sep 1994 (UKC)";

#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#include <stdlib.h>

#include <local/ukcprog.h>
#include <mtrprog/alloc.h>
#include <local/wn.h>
#include <local/obj/mhole.h>

#include "reg.h"
#include "cursors.h"
#include "debug.h"
#include "tdr.h"
#include "ui.h"

#define XPOS_OR_WIDTH	0
#define YPOS_OR_HEIGHT	1

#define XPOS	XPOS_OR_WIDTH
#define YPOS	YPOS_OR_HEIGHT

#define WIDTH	XPOS_OR_WIDTH
#define HEIGHT	YPOS_OR_HEIGHT

struct Region {
	int wn;

	float prop;
	short minsize[2];
	short size[2];

	unsigned char flags;
	unsigned char cursor;

	unsigned char left_margin;
	unsigned char right_margin;
	unsigned char top_margin;
	unsigned char bottom_margin;

	const char **mhcaps;

	re_draw_proc_t draw_proc;
	re_input_proc_t input_proc;
	re_destroy_proc_t destroy_proc;
	re_accept_focus_proc_t accept_focus_proc;
	re_release_focus_proc_t release_focus_proc;

	char *focus_data;
	bool force_quit;

	char *data;

	Region *parent;
	Region *subregs;
	Region *next;
};

#define RF_VSPLIT	0x01	/* region is split vertically (cf horizontally) */
#define RF_SIZE_CHANGED	0x02	/* size of region has changed since last redraw */
#define RF_REDO_PROPS	0x04	/* proportions of subregs need re-normalising */
#define RF_FIXED_SIZE	0x08	/* prop was specified as 0.0 */

#define HH			/* Define this if you like Douglas Adams */

static Region *add_region PROTO((Region *par, int wn, double prop, int size));
static void resize PROTO((Region *par));
static void expose PROTO((Region *par, re_draw_reason_t expose_reason));
static void check_margin_range PROTO((int n));
static void set_win_size PROTO((Region *re, int *pos));
static void await_big_enough_window PROTO((Region *re));
static void renormalise_props PROTO((Region *par, bool skip_minsize_adjustment));
static void check_minsizes PROTO((Region *par));
static void set_minsize PROTO((Region *re, int *minsize));
static void update_window_positions PROTO((Region *par, int dim));
static int avail_pixels PROTO((Region *start, Region *lim, int dim));
static Region *prev_re PROTO((Region *re));
static void swap_regions PROTO((Region *sel_re, event_t *event));
static void swap_children PROTO((Region *re1, Region *re2));
static void rswap_regions PROTO((Region *sel_re, event_t *ev, bool going_up));
#ifdef HH
static void grey_out PROTO((Region *region, int wn,
			    int width, int height,
			    re_draw_reason_t draw_reason));
#endif /* HH */

/*  BUG: these should be packaged up with the root of a region tree in
 *       a regtree_t structure
 */
static bool Quit_event_loop;
static Region *Keyboard_re = NULL;
static Region *Default_keyboard_re = NULL;
static Region *Root_region;

void
re_set_data(re, data)
Region *re;
char *data;
{
	re->data = data;
}

void
re_set_cursor(re, cursor)
Region *re;
int cursor;
{
	if (cursor < 0 || cursor > 255)
		panic("bad cursor in sc");

	re->cursor = cursor;
}

void
re_set_mhcaps(re, left_caption, middle_caption, right_caption)
Region *re;
const char *left_caption, *middle_caption, *right_caption;
{
	re->mhcaps = (const char **)e_malloc(4 * sizeof(char *));
	re->mhcaps[0] = left_caption;
	re->mhcaps[1] = middle_caption;
	re->mhcaps[2] = right_caption;
	re->mhcaps[3] = "";	/* anachronism from Perq days */
}

char *
re_get_data(re)
Region *re;
{
	return re->data;
}

int
re_get_wn(re)
Region *re;
{
	return re->wn;
}

void
re_set_minsize(re, width, height)
Region *re;
int width, height;
{
	int minsize[2];

	/*  Composite regions minimum sizes must be derived from their
	 *  sub-regions.
	 */
	if (re->subregs != NULL)
		panic("setminsize botch");

	minsize[WIDTH] = width + re->left_margin + re->right_margin;
	minsize[HEIGHT] = height + re->top_margin + re->bottom_margin;

	set_minsize(re, minsize);
}

static void
set_minsize(re, minsize)
Region *re;
int *minsize;
{
	Region *par;

	/*  Propagate the new minumum size requirements up the tree.
	 */
	for (par = re->parent; par != NULL; par = par->parent) {
		int dim, otherdim, par_minsize[2];

		if (re->minsize[WIDTH] == minsize[WIDTH] &&
		    re->minsize[HEIGHT] == minsize[HEIGHT])
			return;

		dim = (par->flags & RF_VSPLIT) ? WIDTH : HEIGHT;
		otherdim = 1 - dim;

		par_minsize[WIDTH] = par->minsize[WIDTH];
		par_minsize[HEIGHT] = par->minsize[HEIGHT];

		par_minsize[dim] += minsize[dim] - re->minsize[dim];
		if (minsize[otherdim] > par_minsize[otherdim])
			par_minsize[otherdim] = minsize[otherdim];
		
		re->minsize[WIDTH] = minsize[WIDTH];
		re->minsize[HEIGHT] = minsize[HEIGHT];
		check_minsizes(re);

		re = par;
		minsize[WIDTH] = par_minsize[WIDTH];
		minsize[HEIGHT] = par_minsize[HEIGHT];
	}

	re->minsize[WIDTH] = minsize[WIDTH];
	re->minsize[HEIGHT] = minsize[HEIGHT];
	check_minsizes(re);

}

static void
check_minsizes(par)
Region *par;
{
	Region *re;
	int dim, otherdim, minsize[2];

	if (par->subregs == NULL)
		return;

	dim = (par->flags & RF_VSPLIT) ? WIDTH : HEIGHT;
	otherdim = 1 - dim;

	minsize[WIDTH] = minsize[HEIGHT] = 0;
	for (re = par->subregs; re != NULL; re = re->next) {
		minsize[dim] += re->minsize[dim];
		if (re->minsize[otherdim] > minsize[otherdim])
			minsize[otherdim] = re->minsize[otherdim];
	}

	if (minsize[dim] != par->minsize[dim])
		panic("dim botch in cms");
	if (minsize[otherdim] > par->minsize[otherdim])
		panic("odim botch in cms");
}

void
re_set_callbacks(re, draw_proc, input_proc, destroy_proc)
Region *re;
re_draw_proc_t draw_proc;
re_input_proc_t input_proc;
re_destroy_proc_t destroy_proc;
{
	re->draw_proc = draw_proc;
	re->input_proc = input_proc;
	re->destroy_proc = destroy_proc;
}

void
re_set_keyboard_focus_callbacks(re, accept_proc, release_proc)
Region *re;
re_accept_focus_proc_t accept_proc;
re_release_focus_proc_t release_proc;
{
	re->accept_focus_proc = accept_proc;
	re->release_focus_proc = release_proc;
}

Region *
re_make_region(wn)
int wn;
{
	return add_region((Region *)NULL, wn, 1.0, 0);
}

static void
check_margin_range(n)
int n;
{
	if (n < 0 || n > 255)
		panic("bad margin value in cmr");
}

void
re_set_margins(re, left, right, top, bottom)
Region *re;
int left, right, top, bottom;
{
	int ldelta, rdelta, tdelta, bdelta;

	check_margin_range(left);
	check_margin_range(right);
	check_margin_range(top);
	check_margin_range(bottom);


	ldelta = left - re->left_margin;
	rdelta = right - re->right_margin;
	tdelta = top - re->top_margin;
	bdelta = bottom - re->bottom_margin;

	re->flags |= RF_SIZE_CHANGED;
	wn_adjust_win_size(re->wn, ldelta, tdelta,
			   ldelta + rdelta, tdelta + bdelta);

	re->left_margin = left;
	re->right_margin = right;
	re->top_margin = top;
	re->bottom_margin = bottom;
}

static Region *
add_region(par, wn, prop, fixed_size)
Region *par;
int wn;
double prop;
int fixed_size;
{
	static const char *nullcaps[] = { "", "", "", "" };
	Region *new_re;

	new_re = (Region *)e_malloc(sizeof(Region));

	if (par == NULL)
		new_re->wn = wn;
	else {
		Region *re;

		for (re = par; re != NULL; re = re->parent)
			check_minsizes(re);
		new_re->wn = wn_create_subwin(wn, 0, 0, 0, 0,
						 WN_INPUT_OUTPUT);
	}


	/*  We set the minimum size to zero intially, so we can use
	 *  re_set_minsize to set the actual values as a delta.
	 */
	new_re->minsize[WIDTH] = new_re->minsize[HEIGHT] = 0;

	new_re->size[WIDTH] = new_re->size[HEIGHT] = 0;

	new_re->left_margin = new_re->right_margin = 0;
	new_re->top_margin = new_re->bottom_margin = 0;

	new_re->flags = RF_SIZE_CHANGED;
	new_re->cursor = CU_DEAD;
	new_re->mhcaps = nullcaps;

	new_re->subregs = NULL;
	new_re->prop = prop;
	
	new_re->draw_proc = NULL;
	new_re->input_proc = NULL;
	new_re->destroy_proc = NULL;
	new_re->accept_focus_proc = NULL;
	new_re->release_focus_proc = NULL;

	new_re->data = NULL;
	new_re->force_quit = FALSE;
	new_re->focus_data = NULL;
	new_re->subregs = NULL;
	new_re->next = NULL;
	new_re->parent = par;

	if (prop == 0.0)
		new_re->flags |= RF_FIXED_SIZE;

	wn_set_win_data(new_re->wn, (long)new_re);

	if (par != NULL) {
		Region *re, *last;
		int dim;

		dim = (par->flags & RF_VSPLIT) ? WIDTH : HEIGHT;

		last = NULL;
		for (re = par->subregs; re != NULL; re = re->next)
			last = re;
		if (last == NULL) {
			if (par->minsize[dim] != 0)
				panic("par ms botch");
			par->subregs = new_re;
		}
		else
			last->next = new_re;
		
		if (prop != 0.0)
			par->flags |= RF_REDO_PROPS;

		if (fixed_size != 0) {
			int minsize[2];

			minsize[dim] = fixed_size;
			minsize[1 - dim] = 0;
			set_minsize(new_re, minsize);
		}
	}

	{
		Region *re;

		for (re = new_re; re != NULL; re = re->parent)
			check_minsizes(re);
	}

	return new_re;
}

/*  Scale the re_prop fields of all the variable size children
 *  of par so that the total is reasonably near to 1.0
 */
static void
renormalise_props(par, skip_minsize_adjustment)
Region *par;
bool skip_minsize_adjustment;
{
	Region *re;
	double total_props, scale, average_minsize;
	int total_vpixels, total_minsize, standard_minsize, nvregs, dim;
	bool all_minsizes_equal;

	dim = (par->flags & RF_VSPLIT) ? WIDTH : HEIGHT;

	nvregs = total_minsize = 0;
	total_props = 0.0;
	all_minsizes_equal = TRUE;
	standard_minsize = 0;		/* to satisfy gcc */

	for (re = par->subregs; re != NULL; re = re->next) {
		total_props += re->prop;
		if ((re->flags & RF_FIXED_SIZE) == 0) {
			++nvregs;
			total_minsize += re->minsize[dim];
			if (nvregs == 0)
				standard_minsize = total_minsize;
			else if (re->minsize[dim] != standard_minsize)
				all_minsizes_equal = FALSE;
		}
	}

	if (nvregs == 0 || total_props < 0.0001)
		panic("props botch in rp");

	scale = 1.0 / total_props;
	for (re = par->subregs; re != NULL; re = re->next)
		re->prop *= scale;
	
	if (all_minsizes_equal || skip_minsize_adjustment)
		return;

	/*  Compensate for the minumum size allocations by adding
	 *  or subtracting to re_prop an amount proportional to the
	 *  deviation of each min_size from the average min_size.
	 */

	average_minsize = total_minsize / nvregs;
	total_vpixels = par->size[dim] - par->minsize[dim];

	for (re = par->subregs; re != NULL; re = re->next) {
		double minsize_dev;

		if (re->flags & RF_FIXED_SIZE)
			continue;
		minsize_dev = re->minsize[dim] - average_minsize;
		re->prop -= minsize_dev / total_vpixels;
		if (re->prop < 0.0)
			re->prop = 0.0;
	}

	renormalise_props(par, TRUE);
}

/*  Resize the children of par such that the total width and height
 *  of the children matches par->size.  Recursively update descendents.
 */
static void
resize(par)
Region *par;
{
	Region *re, *maxsize_re;
	int dim, otherdim;
	int total_vpixels, vpixels_left, maxsize_vpixels;

	if (par->size[WIDTH] < par->minsize[WIDTH])
		panic("width botch in rs");
	if (par->size[HEIGHT] < par->minsize[HEIGHT])
		panic("height botch in rs");

	if (par->subregs == NULL)
		return;

	if (par->flags & RF_REDO_PROPS) {
		renormalise_props(par, FALSE);
		par->flags &= ~RF_REDO_PROPS;
	}

	dim = (par->flags & RF_VSPLIT) ? XPOS_OR_WIDTH : YPOS_OR_HEIGHT;
	otherdim = 1 - dim;

	vpixels_left = total_vpixels = par->size[dim] - par->minsize[dim];
	
	/*  And each region shall recieve pixels according to its need ...
	 *  Remember the largest variable size region so we can feed it
	 *  any roundoff pixels.
	 */
	maxsize_re = NULL;
	maxsize_vpixels = 0;
	for (re = par->subregs; re != NULL; re = re->next) {
		int npixels, vpixels;

		npixels = re->minsize[dim];

		vpixels = total_vpixels * re->prop;
		npixels += vpixels;
		vpixels_left -= vpixels;

		if (vpixels > maxsize_vpixels) {
			maxsize_re = re;
			maxsize_vpixels = vpixels;
		}

		if (re->size[dim] != npixels ||
		    re->size[otherdim] != par->size[otherdim])
			re->flags |= RF_SIZE_CHANGED;

		re->size[dim] = npixels;
		re->size[otherdim] = par->size[otherdim];
	}

	/*  BUG: what if vpixels_left is negative and maxsize_re is already
	 *       at its minimum size?
	 */
	if (vpixels_left != 0) {
		if (maxsize_re == NULL)
			panic("vpix botch");
		maxsize_re->size[dim] += vpixels_left;
	}
	
	update_window_positions(par, dim);
}

/*  Recalculate positions based on the new sizes and recursively
 *  resize children.
 */
static void
update_window_positions(par, dim)
Region *par;
int dim;
{
	int pos[2];
	Region *re;

	pos[XPOS] = pos[YPOS] = 0;
	for (re = par->subregs; re != NULL; re = re->next) {
		set_win_size(re, pos);
		if (re->flags & RF_SIZE_CHANGED)
			resize(re);
		pos[dim] += re->size[dim];
	}

	/*  Sanity check
	 */
	if (pos[dim] != par->size[dim])
		panic("resize botch");
}

static void
set_win_size(re, pos)
Region *re;
int *pos;
{
	int x, y, width, height;

	x = pos[XPOS] + re->left_margin;
	y = pos[YPOS] + re->top_margin;
	width = re->size[WIDTH] - (re->left_margin + re->right_margin);
	height = re->size[HEIGHT] - (re->top_margin+re->bottom_margin);

	wn_change_win(re->wn, re->parent->wn, x, y, width, height);
}

static void
expose(par, expose_reason)
Region *par;
re_draw_reason_t expose_reason;
{
	Region *re;
	int width, height;
	re_draw_reason_t draw_reason;

	for (re = par->subregs; re != NULL; re = re->next)
		expose(re, expose_reason);

	if (par->draw_proc == NULL)
		return;

	if ((par->flags & RF_SIZE_CHANGED) && expose_reason == RE_EXPOSED) {
		par->flags &= ~RF_SIZE_CHANGED;
		draw_reason = RE_RESIZED;
	}
	else
		draw_reason = expose_reason;

	wn_get_window_size(par->wn, &width, &height);
	(*par->draw_proc)(par, par->wn, width, height, draw_reason);
}

void
re_set_exit_event_loop_flag()
{
	Quit_event_loop = TRUE;
}

bool
re_get_exit_event_loop_flag()
{
	return Quit_event_loop;
}

Region *
re_get_keyboard_focus_region()
{
	return Keyboard_re;
}

char *
re_get_keyboard_focus_data(re)
Region *re;
{
	return (re != NULL) ? re->focus_data : NULL;
}

void
re_clear_force_quit_flag(re)
Region *re;
{
	re->force_quit = FALSE;
}

bool
re_set_keyboard_focus(re, focus_data)
Region *re;
char *focus_data;
{
	Region *oldre;

	oldre = Keyboard_re;

	if (oldre == re && oldre->focus_data == focus_data)
		return TRUE;
	
	if (oldre != NULL) {
		if (oldre->release_focus_proc != NULL &&
		    !(*oldre->release_focus_proc)(oldre, oldre->focus_data,
						  oldre->force_quit)) {
			if (oldre->force_quit)
				panic("re did not quit when forced");

			oldre->force_quit = TRUE;
			return FALSE;
		}
		
		oldre->force_quit = FALSE;
		oldre->focus_data = NULL;
	}

	if (re == NULL)
		re = Default_keyboard_re;	/* (might also be NULL) */

	if (re != NULL) {
		re->focus_data = focus_data;
		
		if (re->accept_focus_proc != NULL)
			(*re->accept_focus_proc)(re, focus_data);
	}
	
	Keyboard_re = re;
	
	return TRUE;
}

void
re_set_default_keyboard_focus(re, focus_data)
Region *re;
char *focus_data;
{
	re->focus_data = focus_data;
	Default_keyboard_re = re;
}

void
re_set_rootwin(region)
Region *region;
{
	Root_region = region;
}

void
re_redraw_root(event_type, clear_first)
unsigned long event_type;
bool clear_first;
{
	if (event_type == EV_WINDOW_RESIZED) {
		await_big_enough_window(Root_region);
		resize(Root_region);
	}

	re_expose(Root_region, clear_first);
}

static void
await_big_enough_window(re)
Region *re;
{
	font_t *font;

	font = wn_get_sysfont();

	for (;;) {
		bool wide_enough, deep_enough;
		int win_width, win_height, mesg_width, mesg_height;
		const char *mesg;
		event_t event;

		wn_get_window_size(re->wn, &win_width, &win_height);

		wide_enough = win_width >= re->minsize[WIDTH];
		deep_enough = win_height >= re->minsize[HEIGHT];

		if (wide_enough && deep_enough) {
			re->size[WIDTH] = win_width;
			re->size[HEIGHT] = win_height;
			break;
		}

		if (wide_enough)
			mesg = "Window too short";
		else if (deep_enough)
			mesg = "Window too narrow";
		else
			mesg = "Window too small";
		
		mesg_width = wn_strwidth(mesg, font);
		mesg_height = font->ft_height;

		wn_set_area(re->wn, 0, 0, win_width, win_height, WN_BG);
		wn_tputs(re->wn, mesg, (win_width - mesg_width) / 2,
				          (win_height - mesg_height) / 2);
		wn_next_event(re->wn, EV_WINDOW_RESIZED | EV_WINDOW_EXPOSED,
									&event);
	}
}

void
re_event_loop(region)
Region *region;
{
	const int evmask = EV_BUTTON_UP | EV_BUTTON_DOWN | EV_MOUSE_MOVED | EV_KEY |
			   EV_INTERRUPT | EV_WINDOW_EXPOSED | EV_WINDOW_RESIZED;
	Region *root;
	int last_cursor;
	static const char **last_mhcaps;

	root = region;
	last_cursor = -1;
	
	for (Quit_event_loop = FALSE; !Quit_event_loop; ) {
		unsigned long resize_event;
		int cursor;
		Region *re;
		event_t event;

		resize_event = wn_get_resize_event(root->wn);
		if (resize_event != 0) {
			if (display_area_overlay_control(0, 0))
			  display_area_overlay_control(0, 2);
			re_redraw_root(resize_event, FALSE);
			td_record_refresh();
			/* reset, no paint */
			continue;
		}

		wn_next_event(WN_ANY, evmask, &event);

		/*  Cancel any displayed error message on a mouse ior key press
		 */
		if (event.ev_type & (EV_BUTTON_DOWN | EV_KEY))
			clear_message();

		/*  We want key presses anywhere in the window to go to the
		 *  typing line.
		 */
		if (Keyboard_re != NULL && event.ev_type == EV_KEY) {
			re = Keyboard_re;
		}
		else {
			re = (Region *)wn_get_win_data(event.ev_wn);

			cursor = (re != NULL) ? re->cursor : CU_DEAD;
			if (cursor != last_cursor) {
				set_bm_cursor(re->wn, cursor);
				last_cursor = cursor;
			}
			if (re->mhcaps != last_mhcaps) {
				mhdraw(re->mhcaps);
				last_mhcaps = re->mhcaps;
			}
		}

		if (re != NULL) {
			const int swmask = B_SHIFT_MASK | B_CONTROL_MASK;

			if (event.ev_type == EV_BUTTON_DOWN &&
					(event.ev_buttons & swmask) == swmask)
				swap_regions(re, &event);
			else if (re->input_proc != NULL)
			{
			  if (event.ev_type & (EV_BUTTON_DOWN | EV_KEY))
			  {
			    if (re->input_proc != menu_input &&
				display_area_overlay_control(0, 0))
			      display_area_overlay_control(0, 1); /* reset */
			  }
			  (*re->input_proc)(re, &event);
			}
		}
	}
}

void
re_expose(re, clear_first)
Region *re;
bool clear_first;
{

	if (clear_first)
		wn_set_area(re->wn, 0, 0,
				   re->size[WIDTH], re->size[HEIGHT], WN_BG);
	expose(re, RE_EXPOSED);
}

#ifdef __STDC__
void
re_divide(Region *re, re_orientation_t orientation, ...)
{
	va_list ap;

	va_start(ap, orientation);
#else
void
re_divide(va_alist)
va_dcl
{
	va_list ap;
	re_orientation_t orientation;
	Region *re;

	va_start(ap);
	re = va_arg(ap, Region *);
	orientation = va_arg(ap, re_orientation_t);
#endif

	switch (orientation) {
	case RE_VERTICAL:
		if (!(re->flags & RF_VSPLIT)) {
			if (re->subregs != NULL)
				panic("tried to change region split type");
			re->flags |= RF_VSPLIT;
		}
		break;
	case RE_HORIZONTAL:
		if (re->flags & RF_VSPLIT) {
			if (re->subregs != NULL)
				panic("tried to change region split type");
			re->flags &= ~RF_VSPLIT;
		}
		break;
	default:
		panic("bad orientation in re_divide");
	}

	for(;;) {
		Region **p_region;
		Region *new_re;
		int size;
		double prop;

		p_region = va_arg(ap, Region **);
		if (p_region == NULL)
			break;

		size = va_arg(ap, int);
		prop = va_arg(ap, double);

		new_re = add_region(re, re->wn, prop, size);
		if (*p_region != NULL) {
			Region *old_re;

			old_re = *p_region;
			new_re->draw_proc = old_re->draw_proc;
			new_re->input_proc = old_re->input_proc;
			new_re->destroy_proc = old_re->destroy_proc;
			new_re->cursor = old_re->cursor;
			new_re->mhcaps = old_re->mhcaps;
		}
		*p_region = new_re;
	}

	va_end(ap);
}

static Region *
prev_re(lim)
Region *lim;
{
	Region *re, *prev;

	prev = NULL;
	for (re = lim->parent->subregs; re != lim; re = re->next)
		prev = re;
	return prev;
}

static int
avail_pixels(start, lim, dim)
Region *start, *lim;
int dim;
{
	Region *re;
	int avail;

	avail = 0;
	for (re = start; re != lim; re = re->next)
		avail += re->size[dim] - re->minsize[dim];

	return avail;
}

#ifdef HH
static void
grey_out(re, wn, width, height, draw_reason)
Region *re;
int wn;
int width, height;
re_draw_reason_t draw_reason;
{
	switch (draw_reason) {
	case RE_EXPOSED:
	case RE_RESIZED:
		wn_shade_area(wn, 0, 0, width, height, WN_GREY2, R_RPL);
		break;
	case RE_RESIZING:
	case RE_UNDRAW:
		break;
	default:
		panic("bad dr in ld");
	}
}
#endif

static void
swap_regions(sel_re, ev)
Region *sel_re;
event_t *ev;
{
	int buttons;

	buttons = ev->ev_buttons & B_ANY;

	if (buttons == 0)
		return;
	wn_next_event(WN_ANY, EV_BUTTON_UP, ev);
	if (ev->ev_wn != sel_re->wn)
		return;

#ifdef HH
	if (buttons == B_MIDDLE && (Debug_flags & DBFLAG_MISC) != 0) {
		static Region *last_middle_re;

		/*  The DECstation 3100 compiler objects to the following
		 *  comparison unless the (noop) cast is in place ...
		 */
		if (sel_re->draw_proc == (re_draw_proc_t)grey_out)
			return;

		if (sel_re == last_middle_re) {
			errf("I did say not to press this button again ...");
			sel_re->draw_proc = grey_out;
			sel_re->input_proc = NULL;
			sel_re->destroy_proc = NULL;
			re_expose(sel_re, TRUE);
		}
		else {
			errf("Please do not press this button again");
			last_middle_re = sel_re;
		}
	}
#endif /* HH */

	if (buttons & (B_LEFT | B_RIGHT))
		rswap_regions(sel_re, ev, buttons == B_LEFT);
}

static void
rswap_regions(sel_re, ev, going_up)
Region *sel_re;
event_t *ev;
bool going_up;
{
	Region *re, *par;
	int dim;

	if ((par = sel_re->parent) == NULL)
		return;
	dim = (par->flags & RF_VSPLIT) ? XPOS_OR_WIDTH : YPOS_OR_HEIGHT;

	re = sel_re;
	do {
		re = going_up ? prev_re(re) : re->next;
	} while (re != NULL && re->size[dim] <= 1);

	if (re == NULL)
		rswap_regions(par, ev, going_up);
	else {
		swap_children(re, sel_re);
		wn_swap_wins(re->wn, sel_re->wn);
		update_window_positions(par, dim);
		re_expose(par, TRUE);
		wn_warp_mouse(ev->ev_wn, ev->ev_x, ev->ev_y);
	}
}

static void
swap_children(re1, re2)
Region *re1, *re2;
{
	Region **tab, *re, *par;
	int nregs, i, re1pos, re2pos;
	
	par = re1->parent;
	if (re2->parent != par)
		panic("par botch in sc");

	re1pos = re2pos = 0;	/* to satisfy gcc */
	nregs = 0;
	for (re = par->subregs; re != NULL; re = re->next) {
		if (re == re1)
			re1pos = nregs;
		if (re == re2)	
			re2pos = nregs;
		++nregs;
	}

	tab = (Region **)e_malloc(nregs * sizeof(Region *));
	for (i = 0, re = par->subregs; re != NULL; ++i, re = re->next)
		tab[i] = re;
	
	re = tab[re1pos];
	tab[re1pos] = tab[re2pos];
	tab[re2pos] = re;

	re = NULL;
	for (i = nregs - 1; i >= 0; --i) {
		tab[i]->next = re;
		re = tab[i];
	}
	par->subregs = re;

	free((char *)tab);
}

int
re_change_position(mv_re, delta)
Region *mv_re;
int delta;
{
	int dim, avail, total_vpixels, needed;
	bool going_up;
	Region *re, *exp_re, *par, *first_re, *last_re;

	par = mv_re->parent;

	dim = (par->flags & RF_VSPLIT) ? XPOS_OR_WIDTH : YPOS_OR_HEIGHT;

	going_up = delta < 0;
	if (going_up)
		delta = -delta;

	/*  Check that we have pixels to spare in the area that's
	 *  getting smaller.
	 */
	if (going_up)
		avail = avail_pixels(par->subregs, mv_re, dim);
	else
		avail = avail_pixels(mv_re->next, (Region *)NULL, dim);
	if (avail < delta)
		delta = avail;

	/*  Find a variable size region in the expanded area to put the
	 *  extra pixels.
	 */
	exp_re = mv_re;
	do {
		exp_re = going_up ? exp_re->next : prev_re(exp_re);
	} while (exp_re != NULL && (exp_re->flags & RF_FIXED_SIZE) != 0);

	if (exp_re == NULL || delta == 0)
		return 0;
	
	/*  Find out how far we have to go to get the pixels we want in
	 *  the area that's getting smaller.  We do this before doing
	 *  any actual size changes as we want to undraw (with the old
	 *  sizes) the regions that are going to move.
	 */
	re = mv_re;
	needed = delta;
	while (needed > 0) {
		int spare;

		re = going_up ? prev_re(re) : re->next;
		if (re == NULL)
			panic("pixel famine");

		spare = re->size[dim] - re->minsize[dim];
		if (spare != 0 && (re->flags & RF_FIXED_SIZE) == 0)
			needed -= spare;
	}

	if (going_up) {
		first_re = re;
		last_re = exp_re->next;
	}
	else {
		first_re = exp_re;
		last_re = re->next;
	}

	/*  OK, we have some pixels and we know which regions are going to
	 *  change.  We are definitely going to munge the display, so do the
	 *  undraw now.
	 */
	wn_updating_off(par->wn);
	for (re = first_re; re != last_re; re = re->next)
		expose(re, RE_UNDRAW);
	 
	total_vpixels = par->size[dim] - par->minsize[dim];

	/*  First do the compressed area.  We search back for a region with spare
	 *  pixels, and take all the spare pixels or delta, whichever is greater.
	 *  We keep going like this until we have all the pixels we want.
	 */
	re = mv_re;
	needed = delta;
	while (needed != 0) {
		int spare;

		re = going_up ? prev_re(re) : re->next;
		if (re == NULL)
			panic("pixel famine");

		spare = re->size[dim] - re->minsize[dim];
		if (spare == 0 || (re->flags & RF_FIXED_SIZE))
			continue;
		
		if (spare > needed)
			spare = needed;
		
		/*  Steal the pixels from the region, and change re_prop
		 *  to that the new proportions will be preserved on a
		 *  window size change.
		 */
		re->size[dim] -= spare;
		re->prop -= (double)spare / total_vpixels;
		re->flags |= RF_SIZE_CHANGED;

		needed -= spare;
	}

	/*  Add delta pixels to the region to be expanded and munge it's
	 *  proportion to match.
	 */
	exp_re->size[dim] += delta;
	exp_re->prop += (double)delta / total_vpixels;
	exp_re->flags |= RF_SIZE_CHANGED;

	/*  Update the window positions.
	 */
	update_window_positions(par, dim);

	for (re = first_re; re != last_re; re = re->next)
		expose(re, RE_RESIZING);
	wn_updating_on(par->wn);

	return going_up ? -delta : delta;
}

Region *
re_get_parent(region)
Region *region;
{
	return region->parent;
}
