/* ao_elfcore.c - extract information from an ELF core file */

/*  Copyright 1995 Mark Russell, University of Kent at Canterbury.
 *
 *  You can do what you like with this source code as long as
 *  you don't try to make money out of it and you include an
 *  unaltered copy of this message (including the copyright).
 */

char ups_ao_elfcore_c_sccsid[] = "@(#)ao_elfcore.c	1.1 24/5/95 (UKC)";

#include <mtrprog/ifdefs.h>
#include "ao_ifdefs.h"

#ifdef AO_ELF

#include <sys/types.h>
#include <elf.h>
#include <sys/procfs.h>
#include <stdlib.h>
#include <string.h>

#include <local/ukcprog.h>
#include <mtrprog/io.h>

#include "ups.h"
#include "ao_core.h"
#include "ao_elfpriv.h"
#include "ao_elfread.h"
#include "ao_elfcore.h"

#ifdef AO_USE_PROCFS
static void
get_prstatus_info(alloc_pool_t *ap, prstatus_t *pr,
		  int *p_signo, prgreg_t **p_regs)
{
	*p_signo = pr->pr_cursig;

	*p_regs = alloc(ap, sizeof(prgregset_t));
	memcpy(*p_regs, pr->pr_reg, sizeof(prgregset_t));
}

#endif
static void
get_psinfo_info(alloc_pool_t *ap, prpsinfo_t *ps,
		char **p_cmdname, const char **p_cmdline)
{
	char *cmdline, *args;
	
	cmdline = alloc(ap, sizeof(ps->pr_psargs) + 1);
	memcpy(cmdline, ps->pr_psargs, sizeof(ps->pr_psargs));
	cmdline[sizeof(ps->pr_psargs)] = '\0';

	args = strchr(cmdline, ' ');

	if (args != NULL) {
		*args++ = '\0';
		*p_cmdname = strsave(cmdline);
		*p_cmdline = args;
	}
	else {
		*p_cmdname = e_malloc(sizeof(ps->pr_fname) + 1);
		memcpy(*p_cmdname, ps->pr_fname, sizeof(ps->pr_fname));
		(*p_cmdname)[sizeof(ps->pr_fname)] = '\0';

		*p_cmdline = cmdline;
	}
}

#ifdef AO_USE_PROCFS
static bool
get_note_info(alloc_pool_t *ap, const char *path, int fd,
	      Elf32_Phdr *note_ph,
	      int *p_signo, char **p_cmdname, const char **p_cmdline,
	      prgreg_t **p_regs, prfpregset_t **p_p_fpregs)
#else
static bool
get_note_info(alloc_pool_t *ap, const char *path, int fd,
	      Elf32_Phdr *note_ph,
	      int *p_signo, char **p_cmdname, const char **p_cmdline)
#endif
{
	/*  These values come from the the Solaris 2 core(4) man page.
	    RGA for Solaris 2.6, NT_PSINFO = 13 in elf.h, so prepend UPS_.
	    The file now has :
	    NT_PRSTATUS     1 
	    NT_PRFPREG      2
	    NT_PRPSINFO     3
	 */
	enum {
		UPS_NT_STATUS = 1,
		UPS_NT_FPREGSET = 2,
		UPS_NT_PSINFO = 3,
		UPS_NT_TASKSTRUCT = 4, /* RGA added */
		UPS_NT_PLATFORM = 5,   /* RCB added */
		UPS_NT_AUXVEC = 6   /* RCB added */
	};
	static const char what[] = "ELF core file";
	char *buf, *cmdname;
	const char *cmdline;
	int signo;
	size_t pos;
#ifdef AO_USE_PROCFS
	prfpregset_t *fpregs;
	prgreg_t *regs;
#endif
		
	if (note_ph == NULL) {
		errf("No PT_NOTE section in ELF core file %s", path);
		return FALSE;
	}

	buf = e_malloc(note_ph->p_filesz);
	if (!read_chunk(path, "Elf executable", fd, "note section",
			(off_t)note_ph->p_offset, buf, note_ph->p_filesz)) {
		return FALSE;
	}

#ifdef AO_USE_PROCFS
	regs = NULL;
	fpregs = NULL;
#endif
	cmdname = NULL;
	cmdline = NULL;
	
	pos = 0;
	while (pos < note_ph->p_filesz) {
		int *ip;
		const char *name, *desc;
		int namesz, descsz, type;

		ip = (int *)&buf[pos];
		namesz = ip[0];
		descsz = ip[1];
		type = ip[2];
		name = (char *)&ip[3];
		desc = name + ((namesz + 3) / 4) * 4;

		switch (type) {
#ifdef AO_USE_PROCFS
		case UPS_NT_STATUS:
			get_prstatus_info(ap, (prstatus_t *)desc,
					  &signo, &regs);
			break;
		case UPS_NT_FPREGSET:
			fpregs = e_malloc(sizeof(prfpregset_t));
			memcpy(fpregs, desc, sizeof(prfpregset_t));
			break;
#endif
		case UPS_NT_PSINFO:
			get_psinfo_info(ap, (prpsinfo_t *)desc,
					&cmdname, &cmdline);
			break;
		case UPS_NT_TASKSTRUCT:
		case UPS_NT_PLATFORM:
		case UPS_NT_AUXVEC:
			break;
		default:
			errf("Unknown PT_NOTE type %d in ELF core file %s "
			     "- ignored", type, path);
			break;
		}
		
		pos = (desc - buf) + ((descsz + 3) / 4) * 4;
	}

	free(buf);
	
	if (cmdname == NULL) {
		errf("No prpsinfo_t item in note section of %s %s", what, path);
		return FALSE;
	}

#ifdef AO_USE_PROCFS
	if (regs == NULL) {
		errf("No prstatus_t item in note section of %s %s", what, path);
		free(cmdname);
		return FALSE;
	}
#endif

	*p_cmdname = cmdname;
#ifdef AO_USE_PROCFS
	*p_regs = regs;
	*p_p_fpregs = fpregs;
#endif
	*p_signo = signo;
	*p_cmdline = cmdline;
	return TRUE;
}

bool
elf_get_core_info(alloc_pool_t *ap, const char *corepath, int fd,
		  bool user_gave_core,
		  int *p_signo, char **p_cmdname, const char **p_cmdline,
		  Core_segment **p_segments, int *p_nsegments, char **p_regs)
{
	Elf32_Ehdr hdr;
	Elf32_Phdr *phtab, *note_ph;
	Core_segment *segments, *sc;
	unsigned i;
	int nsegments, signo;
	char *cmdname;
	const char *cmdline;
#ifdef AO_USE_PROCFS
	prgreg_t *regs;
	prfpregset_t *p_fpregs;
	Elf_core_regs *cr;
#endif
	
	if (!read_chunk(corepath, "", fd, "ELF header",
			(off_t)0, &hdr, sizeof(hdr)) ||
	    !check_elf_header_type(&hdr, corepath, ET_CORE, "debug from")) {
		return FALSE;
	}

	phtab = e_malloc((size_t)(hdr.e_phentsize * hdr.e_phnum));
	
	if (!read_chunk(corepath, "Elf executable", fd, "program header table",
			(off_t)hdr.e_phoff,
			phtab, (size_t)(hdr.e_phentsize * hdr.e_phnum))) {
		free(phtab);
		return FALSE;
	}

	note_ph = NULL;
	nsegments = 0;
	
	for (i = 0; i < hdr.e_phnum; ++i) {
		switch (phtab[i].p_type) {
		case PT_NOTE:
			note_ph = &phtab[i];
			break;
			
		case PT_LOAD:
			if (phtab[i].p_filesz != 0)
				++nsegments;
			break;

		default:
			break;
		}
	}

	if (nsegments == 0) {
		errf("No PT_LOAD segments in ELF core file %s", corepath);
		free(phtab);
		return FALSE;
	}

#ifdef AO_USE_PROCFS
	if (!get_note_info(ap, corepath, fd, note_ph,
			   &signo, &cmdname, &cmdline, &regs, &p_fpregs)) {
#else
	if (!get_note_info(ap, corepath, fd, note_ph,
			   &signo, &cmdname, &cmdline)) {
#endif
		free(phtab);
		return FALSE;
	}

	segments = alloc(ap, nsegments * sizeof(Core_segment));
	sc = segments;

	for (i = 0; i < hdr.e_phnum; ++i) {
		Elf32_Phdr *ph;

		ph = &phtab[i];

		if (ph->p_type == PT_LOAD && ph->p_filesz != 0) {
			sc->base = ph->p_vaddr;
			sc->lim = ph->p_vaddr + ph->p_memsz;
			sc->file_offset = ph->p_offset;

			++sc;
		}
	}

#ifdef AO_USE_PROCFS
	cr = (Elf_core_regs *)alloc(ap, sizeof(Elf_core_regs));
	cr->regtab = regs;
	cr->p_fpregs = p_fpregs;
#endif
	
	*p_signo = signo;
	*p_cmdname = cmdname;
	*p_cmdline = cmdline;
	*p_segments = segments;
	*p_nsegments = nsegments;
#ifdef AO_USE_PROCFS
	*p_regs = (char *)cr;
#endif
	
	return TRUE;
}

#endif /* AO_ELF */
