/*
 * $Id: pixy.c,v 3.0.1.1 1996/03/20 12:31:49 deejai!martin Exp $
 * Copyright 1996, Romano Giannetti. No guarantees or warantees or anything
 * are provided or implied in any way whatsoever. Use this program at your
 * own risk. Permission to use this program for any purpose is given,
 * as long as the copyright is kept intact.
 *
 * Romano Giannetti - Dipartimento di Ingegneria dell'Informazione
 *                    via Diotisalvi, 2  PISA
 * mailto:romano@iet.unipi.it
 * http://www.iet.unipi.it/~romano
 *
 * Adaptions to speed up the conversion processing, to auto-execute sxpm
 * and to use getopt() made by
 *     Martin Kraemer  <Martin.Kraemer@Mch.SNI.De>
 *
 * Revision 3.2  release: copying policy passed to GPL. See file COPYING
 * Added rgb_to_xpm script to distribution (Martin)
 * 
 * Revision 3.1  1996/03/23 romano
 * Re-coded weights and distance calculations (more coherent now)
 * added --max option
 * 
 *  $Log: pixy.c,v $
 * Revision 3.0.1.1  1996/03/20  12:31:49  deejai!martin
 * I added a --tint option. It allows the user to modify the colors
 * by adding a supplied tint value to the colors' hue.
 * --tint 180 results in a false color image (orange<->blue, magenta<->green)
 *
 * Revision 3.0  1996/03/20  11:34:37  romano
 * it has no sense to give different x and y weight.
 * We change the syntax to read only two numbers.
 * Well, it seem to me that the most logical solution is to have only
 * two weights , xy and z.
 *
 * Revision 2.4  1996/03/20  11:29:55  romano
 * Just noted that all cyan are tranformed in magentas... Well, just a typo.
 *
 * Revision 2.3  1996/03/20  11:28:55  romano
 * AAAAAAAARGH!!! BUGFIX!!!!! Silly me.
 *
 * Revision 2.2  1996/03/20  11:23:16  romano
 * Full support for weights.
 *
 * Revision 2.1.1.5  1996/03/18  09:01:05  martin
 * Usage() messages completed.
 *
 * Revision 2.1.1.4  1996/03/18  08:51:26  martin
 * I added several things:
 *   1) rudimentary RGB/HVC conversion choice - no weight yet, but selectable.
 *   2) getopt() processing of options, if available, use getopt_long().
 *   3) by default, "sxpm -nod" is executed to filter the converted output.
 *   4) the generated file can optionally be written to a given file name
 *      (this might even be the original file... beware!)
 *
 * Revision 2.1.1.1  1996/03/16  15:40:50  martin
 * This version features significant speed enhancements over the original
 * version. This is achieved by storing all informations about the colors
 * (RGB value, HVC value in cartesian coordinates, "none" bit and a pointer
 * to the XpmColor element) in a vector for each base_image's and my_image's
 * colors. The actual comparison needs to compute the squares of two color's
 * x/y/z distances only.
 *
 */
#define  HAVE_GETOPT_LONG       1

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <math.h>

#include <X11/Xlib.h>
#include <X11/xpm.h>
#include <X11/Xcms.h>

#if HAVE_GETOPT_LONG
#include <getopt.h>

struct option pixy_opts[] = 
{
    { "version", 0, 0, 'V' },
    { "verbose", 0, 0, 'v' },
    { "no_sxpm", 0, 0, 'n' },
    { "output",  1, 0, 'o' },
    { "rgb",     1, 0, 'r' },
    { "hvc",     1, 0, 'h' },
    { "max",     1, 0, 'm' },
    { NULL,      0, 0, 0 },
};
int option_index = 0;
#else
    extern int getopt(int argc, char *const argv[], const char *), optind;
    extern char *optarg;
#endif

#define cmap() DefaultColormap(display,screen)
#define ccc()  XcmsDefaultCCC(display,screen)

char	*prog;		/* program name for error messages */
Display *display;
int     screen;

typedef char BOOL;

/* Options... These are supposed to become more complete over time... */
static struct
{
    BOOL    verbose;
    BOOL    use_rgb_distance;
    BOOL    use_hvc_distance;
    BOOL    call_sxpm;
    char    *outfile;
    double  tint;
    double  max_dist;
} opt = { 0, 0, 0, 1, "-", 0.0, 0.0 };


/* This structure is used to quickly access the HVC and RGB values of the colors */
/* without repeatedly having to transform them. Two arrays are created, one      */
/* containing all the base image's colors, the other containing the processed    */
/* image's colors. A pointer to the XpmColor and a "none" flag are there, too.   */
typedef struct
{
    struct
    {
	/* cartesian coordinates in HVC (hue, value, chroma) color space */
	/* we don't actually need the HVC information, so we store only x/y/z */
        double x,
               y,
               z;
    } hvc_space;

    XColor rgb_space;		/* the RGB color as returned by XParseColor() */

    XpmColor* xpm_color;	/* Pointer to the XpmColor from XpmReadFileToXpmImage() */

    unsigned  is_none:1;	/* set if this is the "none" color */
} Color_Info;

/* This structure will contain the weight information */


typedef struct {
    
    double r,
           g,
	   b;
    double xy,
           z;
} Weight_Info;


/* Build a color table of Color_Info elements. Return its dimension */
unsigned
make_rgb_colortable(XpmImage *im, Color_Info **color_array_ptr, double tint) {

    int i,ris;
    unsigned col_in_image;
    Color_Info *color_array;

    col_in_image=(im->ncolors);
    *color_array_ptr = color_array =
	(Color_Info*) malloc(col_in_image * sizeof(Color_Info));

    if (color_array == NULL)
    {
	fprintf (stderr, "%s: Could not allocate array of %d color entries\n",
		 prog, col_in_image);
	exit(1);
    }

    /* loop thru all colors given in the Xpm */
    /* convert the colors (name strings as in xpm file) */
    /* to the X/Y/Z coordinates in HVC color space: */
    for (i=0; i<col_in_image; i++) {

	char *cname=(im->colorTable[i]).c_color;

	color_array[i].xpm_color = &im->colorTable[i];

	if (!strcasecmp(cname,"none")) {

	    color_array[i].is_none = 1;
	    /* no need to fill the other components of color_array[i] */

	} else {
	    XcmsColor xcms_color;

	    color_array[i].is_none = 0;

	    /* fprintf(stderr,"Parsing col %d, name %s\n",i,cname); */

	    ris=XParseColor(display, cmap(), cname, &color_array[i].rgb_space);
	    if (ris==0) {
		fprintf(stderr,"%s: Cannot parse color %s\n", prog, cname);
		exit(4);
	    }

	    if (opt.use_hvc_distance)
	    {
    /* Think of the HVC color space as a cylinder:                                 */
    /* -  the (vertical) z direction is the color's value (luminance or intensity) */
    /* -  the radius of the cylinder is the color's chroma (saturation)            */
    /* -  the angle of the horizontal vector is the color's hue (reddish/greenish/..) */
    /* In order to compare two colors, we transform the radius=value/angle=hue     */
    /* values into their equivalent X / Y coordinates and perform the distance     */
    /* calculation on the resulting X/Y/Z values.                                  */

		xcms_color.format=XcmsRGBFormat;
		xcms_color.spec.RGB.red   = color_array[i].rgb_space.red;
		xcms_color.spec.RGB.green = color_array[i].rgb_space.green;
		xcms_color.spec.RGB.blue  = color_array[i].rgb_space.blue;

		/* RGB -> HVC transformation */
		ris=XcmsConvertColors(ccc(),&xcms_color,1,XcmsTekHVCFormat,NULL);
		if (ris == XcmsFailure) {
		    fprintf(stderr,"%s: Cannot convert color %s to HVC\n", prog, cname);
		    exit(4);
		}

		/* polar -> cartesian transformation */
		color_array[i].hvc_space.x = xcms_color.spec.TekHVC.C *
					sin((tint+xcms_color.spec.TekHVC.H) * M_PI/180);
		color_array[i].hvc_space.y = xcms_color.spec.TekHVC.C *
					cos((tint+xcms_color.spec.TekHVC.H) * M_PI/180);
		color_array[i].hvc_space.z = xcms_color.spec.TekHVC.V;
	    }
	}
    }
    return col_in_image;
}


double square(double x) {
    return x*x;
}

double distance(Color_Info *col1, Color_Info *col2, Weight_Info *w) {

    /* by squaring the individual differences, */
    /* we determine the (square of) the distance in HVC or RGB space */
        double dst=0.0;
        /* normalize to 100^2=10000 distance between black and white,
	 * for the two metric and standard 1:1:1 or 1:1 weights. Note that 
	 * for rgb metric whatever weights you choose black and white ditance is
	 * the same. The same does NOT stand for hvc. Magic numbers here are:
	 * 655.35 == (2^16-1)/100 
	 * 2 is the factor so that w_xy=0.5, w_z=0.5 give black-white=100^2 */ 
        if (opt.use_rgb_distance) 
            dst += w->r * square((double) (col2->rgb_space.red   - col1->rgb_space.red)/655.35)
		+ w->g  * square((double) (col2->rgb_space.green - col1->rgb_space.green)/655.35)
		+ w->b  * square((double) (col2->rgb_space.blue  - col1->rgb_space.blue)/655.35);
        if (opt.use_hvc_distance) 
            dst += w->xy * 2 *square(col2->hvc_space.x - col1->hvc_space.x)
		+ w->xy * 2 * square(col2->hvc_space.y - col1->hvc_space.y)
		+ w->z * 2 * square(col2->hvc_space.z - col1->hvc_space.z);
        return dst;
}


/* Replace the color in my_color by the closest matching color from base_table */
double
substitute_color(Color_Info *base_table, int base_col_count,
		 Color_Info *my_color, Weight_Info *w) {

    int i, minind;
    double mindst=1e20;
    double dst;

    if (my_color->is_none) {
	return 0.0;		/* do not substitute the "none" color */
    }

    /* Loop over all base_table colors; find out which one is closest to my_color */
    minind = 0;
    for(i=0; i < base_col_count; i++) {
	if (base_table[i].is_none)	/* do not compare with the "none" color */
		continue;
	dst = distance (my_color, &base_table[i], w);
	if (dst < mindst ) {
	    mindst=dst;
	    minind=i;
	}
    }
   
    if (opt.max_dist==0.0 || (mindst <= opt.max_dist)) { 
	/* Yes, substitute */
	if (opt.verbose)
	  fprintf(stderr,"Subs: %3d: %16s with   %16s (dist %10.3f)\n",
		  minind, my_color->xpm_color->c_color, base_table[minind].xpm_color->c_color, mindst);

	/* Finally: replace the color string by the newly determined color string */
	my_color->xpm_color->c_color = base_table[minind].xpm_color->c_color;
    } else {
	/* Do not substitute */
	if (opt.verbose)
	  fprintf(stderr,"KEEP: %3d: %16s nearer %16s (dist %10.3f)\n",
		  minind, my_color->xpm_color->c_color, base_table[minind].xpm_color->c_color, mindst);
    }
    return mindst;
}

void
Usage (void)
{
    fprintf(stderr,"Usage: %s [<opts>] <basepixmapfile> <imagepixmapfile>\n", prog);
    fprintf(stderr,"Function: Reduce the colors in the image file to those used in the base file\n");
    fprintf(stderr,"Options:\n");
#if HAVE_GETOPT_LONG
    fprintf(stderr,"  --version      Display program version and exit\n");
    fprintf(stderr,"  --verbose      Verbose messages to stderr\n");
    fprintf(stderr,"  --no_sxpm      Do not fork \"sxpm -nod\" to reduce duplicates\n");
    fprintf(stderr,"  --output file  Save converted image in <file>, do not send to stdout\n");
    fprintf(stderr,"  --hvc hc:v     Use HVC distances. This is the default (using 1:1)\n");
    fprintf(stderr,"  --rgb r:g:b    Use RGB distances, weights are numbers, separated by ':'\n");
    fprintf(stderr,"  --tint t       Rotate the image's color hue by <t> degrees\n");
    fprintf(stderr,"  --max m        Substitute only if dist <= m\n");
#else
    fprintf(stderr,"  -V       Display program version and exit\n");
    fprintf(stderr,"  -v       Verbose messages to stderr\n");
    fprintf(stderr,"  -n       Do not fork \"sxpm -nod\" to reduce duplicates\n");
    fprintf(stderr,"  -o file  Save converted image in <file>, do not send to stdout\n");
    fprintf(stderr,"  -r r:g:b Use RGB distances, weights are numbers, separated by ':'\n");
    fprintf(stderr,"  -h hc:v  Use HVC distances. This is the default (using 1:1)\n");
    fprintf(stderr,"  -t t     Rotate the image's color hue by <t> degrees\n");
    fprintf(stderr,"  -m m     Substitute only if dist <= m\n");
#endif
}

int main (int argc, char *argv[]) {

    XpmImage	base_image,  my_image;
    XpmInfo	base_info,   my_info;
    Color_Info	*base_table, *my_table;
    /* We initialize to zero to get a right normalization */
    Weight_Info my_w={0.0,0.0,0.0, 0.0,0.0};
    int         ris,i;
    int         opt_char;
    double      sum=0.0;
    
    prog = strrchr(argv[0],'/');
    if (prog == NULL)
	prog = argv[0];
    else
	++prog;

#if HAVE_GETOPT_LONG
    while ((opt_char = getopt_long (argc, argv, "?Vvnh:o:r:t:", pixy_opts, &option_index)) != EOF)
#else
    while ((opt_char = getopt (argc, argv, "?Vvnh:o:r:t:")) != EOF)
#endif
    {
	switch (opt_char)
	{
	    case 'V':
		Usage();
		fprintf (stderr, "%s: pixy 3.0 alpha\n", prog);
		exit(0);

	    case 'v':	/* verbose logging */
		opt.verbose = 1;
		break;

	    case 'n':	/* no fork of sxpm */
		opt.call_sxpm = 0;
		break;

	    case 'o':	/* Output File */
		opt.outfile = optarg;
		break;

	    case 'r':	/* RGB instead of HVC */
		opt.use_rgb_distance = 1;
	        if (sscanf(optarg,"%lf:%lf:%lf",&my_w.r,&my_w.g,&my_w.b)!=3) {
		    Usage();
		    exit(1);
		}
		break;

	    case 'h':	/* set HVC weights */
		opt.use_hvc_distance = 1;
	        if (sscanf(optarg,"%lf:%lf",&my_w.xy,&my_w.z)!=2) {
		    Usage();
		    exit(1);
		}
	        break;

	    case 't':   /* Tint: shift the color by some degrees */
		if (sscanf(optarg,"%lf", &opt.tint) != 1) {
		    Usage();
		    exit(1);
		}
		break;

	    case 'm':   /* Max: max amount of distance for substitutions */
		if (sscanf(optarg,"%lf", &opt.max_dist) != 1) {
		    Usage();
		    exit(1);
		}
		break;

	    case '?':
	    default:
		Usage();
		exit(1);
	}
    }

    /* after the options, we expect exactly two file names */
    if (optind != argc-2)
    {
	Usage();
	exit(1);
    }

    /* Now build right weights... 
     * if no --rgb or --hvc specified, then it defaults to hc:v = 1:1
     */
    if (!opt.use_rgb_distance && !opt.use_hvc_distance) {
	my_w.xy=1.0;
	my_w.z=1.0;
	opt.use_hvc_distance=1;
    }
    /* Now we have sure some weight in. Let them normalize to 1 */
    sum=my_w.xy+my_w.z+my_w.r+my_w.g+my_w.b;
    my_w.r /= sum;
    my_w.g /= sum;
    my_w.b /= sum;
    my_w.xy /= sum;
    my_w.z /= sum;
    
    if (opt.verbose) {
	fprintf(stderr,"Weights:");
	if (opt.tint != 0.0)
	    fprintf(stderr," tint=%5.2f ", opt.tint);
	if (opt.use_rgb_distance)
	    fprintf(stderr," r-g-b=%5.2f:%5.2f:%5.2f ",my_w.r,my_w.g,my_w.b);
	if (opt.use_hvc_distance)
	    fprintf(stderr," xy-z(hc-v)=%5.2f:%5.2f ",my_w.xy,my_w.z);
	fprintf(stderr,"\n");
    }
    
    display=XOpenDisplay(NULL);
    if (display==NULL) {
	fprintf(stderr,"%s: Cannot open display\n", prog);
	exit(1);
    }
    screen = DefaultScreen(display);

    ris=XpmReadFileToXpmImage(argv[optind], &base_image, &base_info);
    if (ris!=XpmSuccess) {
	fprintf(stderr,"%s: Cannot open base pixmap %s\n", prog, argv[optind]);
	exit(2);
    }

    ris=XpmReadFileToXpmImage(argv[optind+1], &my_image, &my_info);
    if (ris!=XpmSuccess) {
	fprintf(stderr,"%s: Cannot open image pixmap %s\n", prog, argv[optind+1]);
	exit(2);
    }

    make_rgb_colortable(&base_image, &base_table, 0.0);
    make_rgb_colortable(&my_image,   &my_table,   opt.tint);

    for(i=0; i<my_image.ncolors; i++) {
	substitute_color(base_table, base_image.ncolors, &my_table[i], &my_w);
    }

    if (!opt.call_sxpm)
    {
	XpmWriteFileFromXpmImage(strcmp(opt.outfile,"-")==0 ? NULL : opt.outfile, 
				 &my_image, &my_info);
    }
    else
    {
	int pipe_fd[2];	/* communication father -> son */
	pipe(pipe_fd);

	switch (fork())
	{
	case -1:	/* error */
	    perror ("fork");
	    exit (1);

	case 0:		/* son */
	    close (pipe_fd[0]);	/* close reader side of pipe */
	    close (1);
	    dup(pipe_fd[1]);	/* copy writer side of pipe to stdout */
	    XpmWriteFileFromXpmImage(NULL, &my_image, &my_info);
	    exit (0);

	default:	/* father */
	    /* Our stdin is connected to the pipe. */
	    close (pipe_fd[1]);	/* close writer side of pipe */
	    close (0);
	    dup (pipe_fd[0]);	/* copy reader side of pipe to fd 0 */
	    close (pipe_fd[0]);	/* close unused copy of reader side */
	    if (opt.verbose)
		fprintf (stderr, "%s: calling \"sxpm -nod - -o %s\"\n", prog, opt.outfile);
	    execlp ("sxpm", "sxpm", "-nod", "-", "-o", opt.outfile, NULL);
	    perror ("exec");
	    exit (1);
	}
    }
    return 0;
}
