/* 
 * rusers.c 
 * 
 * Eric Dumas 
 * This file is largely inspired from the rusers.c for rusers command
 *
 * $Id:$
 *
 */

/*
 *  Copyright (c) 1993 John Brezak
 *  All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. The name of the author may not be used to endorse or promote products
 *     derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR `AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <netdb.h>
#include <stdio.h>
#include <string.h>

#include <rpc/rpc.h>
#include <rpc/pmap_clnt.h>

#include <arpa/inet.h>
#include <utmp.h>

#include <stdlib.h>


/*
 * For now we only try version 2 of the protocol. The current
 * version is 3 (rusers.h), but only Solaris and NetBSD seem
 * to support it currently.
 */

#ifdef SOLARIS
extern void bzero(caddr_t, size_t); 
#endif

#include <rpcsvc/rusers.h>	

#include <maprusers.h>


#define MAX_INT     0x7fffffff
#define HOST_WIDTH  20
#define LINE_WIDTH  8

#define MAX_HOST_SIZE 50
#define MAX_DATE_SIZE 32
#define MAX_IDLE_TIME 64
#define MAX_REMOTE    64
#define MAX_LOCAL     64

#define MAX_NAME      8

/* Struct for informations */
struct people_s
{
    char name[MAX_NAME];            /* login name    */
	char date[MAX_DATE_SIZE],       /* login date    */
         idle_time[MAX_IDLE_TIME],  /* idle time     */
         remote[MAX_REMOTE],        /* remote host   */
         local[MAX_LOCAL];          /* local tty use */
};

struct data_internal
{
    char   host[MAX_HOST_SIZE];     /* host name */
    int    nb_people;               /* nb login  */
    int    width;
    int    height;
    struct people_s people[50];
};

struct timeval timeout = { 5, 0 }; /* 5 seconds max 0 microseconds */

static int             nb_reply = 0;

static data_internal_t rusers_reply(data_internal_t list_host,char *replyp, 
                        struct sockaddr_in *raddrp);

/* Access functions */

void  set_rusers_width(data_internal_t list_host, int width)
{
    list_host[0].width = width;
}

int get_rusers_width(data_internal_t list_host)
{
    return list_host[0].width;
}

void  set_rusers_height(data_internal_t list_host, int height)
{
    list_host[0].height = height;
}
int get_rusers_height(data_internal_t list_host)
{
    return list_host[0].height;
}

char *get_idle_time(data_internal_t list_host, int n_host, int n_login)
{
    return list_host[n_host].people[n_login].idle_time;
}

char *get_remote(data_internal_t list_host, int n_host, int n_login)
{
    return list_host[n_host].people[n_login].remote;
}

char *get_local(data_internal_t list_host, int n_host, int n_login)
{
    return list_host[n_host].people[n_login].local;
}

char *get_date(data_internal_t list_host, int n_host, int n_login)
{
    return list_host[n_host].people[n_login].date;
}

char *get_name_host(data_internal_t list_host, int n)
{
    return list_host[n].host;
}

char *get_name_login(data_internal_t list_host, int n_host, int n_login)
{
    return list_host[n_host].people[n_login].name;
}

int get_nb_login(data_internal_t list_host, int n_host)
{
    return list_host[n_host].nb_people;
}

void set_nb_login(data_internal_t list_host, int n_host, int val)
{
    if (list_host == NULL) /* Should never occure */
        return;
    
    list_host[n_host].nb_people=val;
}

void delete_host_list(data_internal_t list_host)
{
    nb_reply = 0;
    if (list_host == NULL)
        return;
    
    free(list_host);
}


/* Check host */
data_internal_t onehost(data_internal_t list_host,int *flag, char *host)
{
	struct utmpidlearr  up;
	CLIENT             *rusers_clnt;
	struct sockaddr_in  addr;
	struct hostent     *hp;

	hp = gethostbyname(host);
    *flag = 1;
    
	if (hp == NULL) 
    {
		fprintf(stderr, "unknown host \"%s\"\n", host);
        *flag = 0;
        return list_host;
	}

    /* Connect to the host */
	rusers_clnt = clnt_create(host, RUSERSPROG, RUSERSVERS_IDLE, "udp");
	if (rusers_clnt == NULL) 
    {
        fprintf(stderr," --> %s : ", host);
		clnt_pcreateerror("Unable to Create Connexion");
        *flag = 0;        
        return list_host;
	}

	bzero((char *)&up, sizeof(up));

    /* Call rusersd  */
	if (clnt_call(rusers_clnt, RUSERSPROC_NAMES, xdr_void, NULL,
                  xdr_utmpidlearr, (caddr_t)&up, timeout) != RPC_SUCCESS) 
    {
        fprintf(stderr," --> %s : ", host);
		clnt_perror(rusers_clnt, "Enable to call");
        *flag = 0;        
        return list_host;
	}

	addr.sin_addr.s_addr = *(int *)hp->h_addr;
    
	list_host = rusers_reply(list_host,(char *)&up, &addr);

    /* Close connexion */
    clnt_destroy(rusers_clnt);
    
    return list_host;
}

/* Static */
static data_internal_t rusers_reply(data_internal_t list_host, char *replyp, 
                 struct sockaddr_in *raddrp)
{
	int                 x, 
                        idle;
	struct hostent     *hp;
	struct utmpidlearr *up = (struct utmpidlearr *)replyp;

	char *host;
	int days, hours, minutes, seconds;
	
	hp = gethostbyaddr((char *)&raddrp->sin_addr.s_addr,
                       sizeof(struct in_addr), AF_INET);
    
	if (hp != NULL)
		host = hp->h_name;
	else
		host = inet_ntoa(raddrp->sin_addr);
	
    if (nb_reply == 0)
        list_host = malloc(sizeof(struct data_internal));
    else
        list_host = realloc(list_host,
                           sizeof(struct data_internal)*(nb_reply+2));
    if (list_host == NULL)
    {
        fprintf(stderr," Can't malloc \n");
        exit(-1);
    }
    
    /* Hostname */
    sprintf(list_host[nb_reply].host,"%s", host);
    
    /* Nb login */
    list_host[nb_reply].nb_people = up->uia_cnt;
    
	for (x = 0; x < up->uia_cnt; x++) 
    {
        /* connection date */
		strncpy(list_host[nb_reply].people[x].date,
                &(ctime((time_t *)&(up->uia_arr[x]->ui_utmp.ut_time))[4]),
                sizeof(list_host[nb_reply].people[x].date)-1);

        list_host[nb_reply].people[x].date[strlen(list_host[nb_reply].people[x].date)-1]=0;
        
		idle = up->uia_arr[x]->ui_idle;
		sprintf(list_host[nb_reply].people[x].idle_time, "%d", 
                idle);

		if (idle == MAX_INT)
			strcpy(list_host[nb_reply].people[x].idle_time, "??");
		else 
            if (idle == 0)
                strcpy(list_host[nb_reply].people[x].idle_time, "");
            else 
            {
                seconds = idle;
                days = seconds/(60*60*24);
                seconds %= (60*60*24);
                hours = seconds/(60*60);
                seconds %= (60*60);
                minutes = seconds/60;
                seconds %= 60;

                if (idle > 60)
                    sprintf(list_host[nb_reply].people[x].idle_time, "%d:%d",
                            minutes, seconds);

                if (idle >= (60*60))
                    sprintf(list_host[nb_reply].people[x].idle_time, "%d:%d:%d",
                            hours, minutes, seconds);

                if (idle >= (24*60*60))
                    sprintf(list_host[nb_reply].people[x].idle_time, 
                            "%d days, %d:%d:%d",
                            days, hours, minutes, seconds);
            }

		strncpy(list_host[nb_reply].people[x].remote, 
                up->uia_arr[x]->ui_utmp.ut_host,
                sizeof(list_host[nb_reply].people[x].remote)-1);

		if (strlen(list_host[nb_reply].people[x].remote) != 0)
			sprintf(list_host[nb_reply].people[x].remote, "%.16s",
                    up->uia_arr[x]->ui_utmp.ut_host);

        strncpy(list_host[nb_reply].people[x].local, 
                list_host[nb_reply].host, 
                sizeof(list_host[nb_reply].people[x].local));

        list_host[nb_reply].people[x].local[HOST_WIDTH + LINE_WIDTH + 1 -
                        strlen(up->uia_arr[x]->ui_utmp.ut_line) - 1] = 0;
        strcat(list_host[nb_reply].people[x].local, ":");
        strcat(list_host[nb_reply].people[x].local, 
               up->uia_arr[x]->ui_utmp.ut_line);

        sprintf(list_host[nb_reply].people[x].name,"%.8s",
                up->uia_arr[x]->ui_utmp.ut_name);
	}
	
    nb_reply++;

	return list_host;
}



