/*----------------------------------------------------------------------------
--
--  Module:           xitInfo
--
--  Project:          xit - X Internal Toolkit
--  System:           <>
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Browse a structured info file.
--
--  Filename:         xitInfo.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-10-17
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitInfo.c, Version: 1.1, Date: 95/02/18 15:10:34";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <ctype.h>
#include <limits.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/List.h>
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/ScrolledW.h>
#include <Xm/Separator.h>
#include <Xm/Text.h>
#include <Xm/ScrollBar.h>

#include "xitRwPixB.h"

#include "System.h"
#include "LstLinked.h"
#include "Message.h"

#include "msgTopic.h"
#include "xitError.h"
#include "xitInfoBook.h"
#include "xitInfoFile.h"
#include "xitInfoPrint.h"
#include "xitInfoSrch.h"
#include "xitInfoToc.h"
#include "xitTools.h"
#include "xitInfo.h"

#include "xitBackIcon.bm"
#include "xitContentsIcon.bm"
#include "xitHelpIcon.bm"
#include "xitMasterBookIcon.bm"
#include "xitPrinterIcon.bm"
#include "xitSearchIcon.bm"

/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#define max( a, b )       (a < b ? b : a )

/* Number of sections within a chapter. */
#define  MAX_SECTIONS  100

/* Number of history items we keep. */
#define  MAX_HISTORY   50


/* Local widgets in the entry edit window. */
#define actionFr            dataLocalW[  0 ]
#define actionRc            dataLocalW[  1 ]
#define chapterLa           dataLocalW[  2 ]
#define commandRc           dataLocalW[  3 ]
#define helpTx              dataLocalW[  4 ]
#define infoPa              dataLocalW[  5 ]
#define menuBr              dataLocalW[  6 ]
#define pane1Fo             dataLocalW[  7 ]
#define pane2Fo             dataLocalW[  8 ]
#define pane3Fo             dataLocalW[  9 ]
#define sectionLa           dataLocalW[ 10 ]
#define sectionLi           dataLocalW[ 11 ]
#define dummySp             dataLocalW[ 12 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* History record. */
typedef struct {
  char  *book;
  char  *chapter_id;
  char  *subject;
} HISTORY_REC;


/* Info record. */
typedef struct {

  /* Save history record? */
  Boolean  save_history;

  /* Display the action buttons? */
  Boolean  use_action_buttons;

  /* Icons on the action buttons? */
  Boolean  use_button_icons;

  /* Help text as label? */
  Boolean  use_help_label;

  /* First and last position in the history record. */
  int  current_hist;

  /* Icon file to use. */
  char  *icon_file;

  /* The master book. */
  char  *master_book;

  /* The print command to use. */
  char  *print_script;

  /* File with system printers to use. */
  char  *system_pr_file;

  /* File with user printers to use. */
  char  *user_pr_file;

  /* The current chapter displayed. */
  char  curr_chapter_id[ 50 ];

  /* The info window. */
  Widget  infoW;

  /* The book we are reading. */
  XIT_IF_BOOK_REC  curr_book;

  /* History record. */
  HISTORY_REC  history[ MAX_HISTORY ];

  /* User and system books. */
  LST_DESC_TYPE  books[ 2 ];

  /* The book selection. */
  XIT_IB_HANDLE  book_handle;

  /* The info file. */
  XIT_IF_HANDLE  file_handle;

  /* The print window. */
  XIT_IP_HANDLE  print_handle;

  /* The search window. */
  XIT_IS_HANDLE  search_handle;

  /* The table of contents window. */
  XIT_IT_HANDLE  toc_handle;

  /* Callback and user data to inform our creator of specific actions. */
  XIT_IN_ACTION_CB  actionCB;
  void              *user_data;

} INFO_REC, *INFO_REC_REF;


/* Activation data. */
typedef struct {
  INFO_REC_REF     info_ref;
  XIT_IF_TAG_TYPE  tag_type;
  char             *param1;
  char             *param2;
  char             *param3;
  char             *param4;
} ACTIVATE_DATA_REC;
  

/* Info resources. */
typedef struct {
  Boolean  use_action_buttons;
  Boolean  use_button_icons;
  Boolean  use_help_label;
  char     *gen_print_command;
  char     *icon_file;
  char     *master_book;
  char     *system_books;
  char     *system_printer_file;
  char     *user_books;
  char     *user_printer_file;
} INFO_RESOURCES, *INFO_RESOURCES_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xitInfo";

/* Name of text domain. */
static char  *td = "Topic";

/* Info resources. */
static XtResource  sub_resources[] = {
  { "genPrintCmd", "GenPrintCmd", XtRString, sizeof( String ),
    XtOffset( INFO_RESOURCES_REF, gen_print_command ), 
    XtRString, "lpr -P" },

  { "iconFile", "IconFile", XtRString, sizeof( String ),
    XtOffset( INFO_RESOURCES_REF, icon_file ), 
    XtRString, "" },

  { "masterBook", "MasterBook", XtRString, sizeof( String ),
    XtOffset( INFO_RESOURCES_REF, master_book ),
    XtRString, "" },

  { "systemBooks", "SystemBooks", XtRString, sizeof( String ),
    XtOffset( INFO_RESOURCES_REF, system_books ),
    XtRString, "" },

  { "systemPrinterFile", "SystemPrinterFile", XtRString, sizeof( String ),
    XtOffset( INFO_RESOURCES_REF, system_printer_file ), 
    XtRString, "" },

  { "useActionButtons", "UseActionButtons", XtRBoolean, sizeof( Boolean ),
    XtOffset( INFO_RESOURCES_REF, use_action_buttons ), 
    XtRString, "True" },

  { "useButtonIcons", "UseButtonIcons", XtRBoolean, sizeof( Boolean ),
    XtOffset( INFO_RESOURCES_REF, use_button_icons ), 
    XtRString, "True" },

  { "useHelpLabel", "UseHelpLabel", XtRBoolean, sizeof( Boolean ),
    XtOffset( INFO_RESOURCES_REF, use_help_label ), 
    XtRString, "False" },

  { "userBooks", "UserBooks", XtRString, sizeof( String ),
    XtOffset( INFO_RESOURCES_REF, user_books ),
    XtRString, "" },

  { "userPrinterFile", "UserPrinterFile", XtRString, sizeof( String ),
    XtOffset( INFO_RESOURCES_REF, user_printer_file ), 
    XtRString, "" },
};


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  activateCB( Widget             widget, 
              ACTIVATE_DATA_REC  *button_data_ref,
              XtPointer          call_data );

static Boolean
  backHistory( INFO_REC_REF  info_ref,
               char          **book,
               char          **chapter_id,
               char          **subject );

static void 
  bookActionCB( XIT_IB_REASON  reason,
                void           *user_data,
                char           *book_name );

static void 
  booksCB( Widget        widget, 
           INFO_REC_REF  info_ref,
           XtPointer     call_data );

static void
  canGoBack( INFO_REC_REF  info_ref );

static void 
  closeCB( Widget        widget, 
           INFO_REC_REF  info_ref,
           XtPointer     call_data );

static void 
  createActionButtons( INFO_REC_REF   info_ref,
                       LST_DESC_TYPE  tag_list );

static Widget
  createInfoWindow( INFO_REC_REF     info_ref,
                    Widget           parent );

static void 
  destroyCB( Widget        widget, 
             INFO_REC_REF  info_ref,
             XtPointer     call_data );

static void 
  destroyActivateCB( Widget             widget, 
                     ACTIVATE_DATA_REC  *button_data_ref,
                     XtPointer          call_data );

static void
  displaySections( INFO_REC_REF  info_ref,
                   char          *chapter_id );

static void
  displaySectionSubject( INFO_REC_REF  info_ref,
                         char          *chapter_id,
                         char          *subject );

static void
  displaySectionText( INFO_REC_REF        info_ref,
                      XIT_IF_SECTION_REF  section_ref );

static void 
  doBackHistory( INFO_REC_REF  info_ref );

static void 
  doBackCB( Widget        widget,
            INFO_REC_REF  info_ref,
            XtPointer     call_data );

static void 
  doBook( INFO_REC_REF  info_ref );

static void 
  doContentsCB( Widget        widget,
                INFO_REC_REF  info_ref,
                XtPointer     call_data );

static void 
  doMasterBook( INFO_REC_REF  info_ref );

static void 
  doMasterBookCB( Widget        widget,
                  INFO_REC_REF  info_ref,
                  XtPointer     call_data );

static void 
  doPrint( INFO_REC_REF  info_ref );

static void 
  doPrintCB( Widget        widget,
             INFO_REC_REF  info_ref,
             XtPointer     call_data );

static void 
  doSearch( INFO_REC_REF  info_ref );

static void 
  doSearchCB( Widget        widget,
              INFO_REC_REF  info_ref,
              XtPointer     call_data );

static void 
  doToc( INFO_REC_REF  info_ref );

static
  char *expandFilename( char  *filename );

static Boolean
  jumpTo( INFO_REC_REF    info_ref,
          XIT_IF_SECTION  *section_rec );

static LST_DESC_TYPE
  parseBooks( char  *books );

static void 
  printActionCB( XIT_IP_REASON  reason,
                 void           *user_data );

static void 
  saveHistory( INFO_REC_REF  info_ref,
               char          *book,
               char          *chapter_id,
               char          *subject );

static void 
  scrollWindow( Widget  scrollBar,
                int     value );

static void 
  searchActionCB( XIT_IS_REASON  reason,
                  void           *user_data,
                  char           *book,
                  char           *chapter_id,
                  char           *title );

static void 
  sectionListCB( Widget                widget, 
                 INFO_REC_REF          info_ref,
                 XmListCallbackStruct  *call_data );

static void
  startProcess( Widget  parent,
                char    *command );

static void 
  switchToBook( INFO_REC_REF  info_ref,
                char          *new_book,
                Boolean       also_if_current );

static void 
  tocActionCB( XIT_IT_REASON  reason,
               void           *user_data,
               char           *chapter_id,
               char           *title );

static void 
  topCB( Widget          widget, 
         INFO_REC_REF    info_ref,
         XtPointer       call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XIT_IN_HANDLE
  xitInInitialize( Widget            parent,
                   XIT_IN_ACTION_CB  actionCB,
                   void              *user_data )
{

  /* Variables. */
  INFO_REC_REF  info_ref;

  static Boolean         resources_read = False;
  static INFO_RESOURCES  resource_rec;


  /* Code. */

  /* Create and initialize our private data. */
  info_ref = SysNew( INFO_REC );
  if( info_ref == NULL )
    return( NULL );


  /* Initialize data. */
  info_ref -> actionCB            = actionCB;
  info_ref -> user_data           = user_data;
  info_ref -> book_handle         = NULL;
  info_ref -> file_handle         = NULL;
  info_ref -> print_handle        = NULL;
  info_ref -> search_handle       = NULL;
  info_ref -> toc_handle          = NULL;
  info_ref -> curr_book.name[ 0 ] = '\0';
  info_ref -> books[ 0 ]          = NULL;
  info_ref -> books[ 1 ]          = NULL;
  info_ref -> save_history        = True;
  info_ref -> current_hist        = -1;


  /* Get the sub-resources internal to this widget. */
  if( ! resources_read ) {

    int   index;
    char  *file_ref;
    char  **expand_files[ 50 ];

    XtGetSubresources( parent, (XtPointer) &resource_rec,
                       (String) NULL, (String) NULL,
                       sub_resources, XtNumber( sub_resources ),
                       (ArgList) NULL, 0 );

    /* Expand any directory and filename references. */
    expand_files[  0 ] = &resource_rec.icon_file;
    expand_files[  1 ] = &resource_rec.system_printer_file;
    expand_files[  2 ] = &resource_rec.user_printer_file;
    expand_files[  3 ] = NULL;

    index = 0;
    do {
      if( expand_files[ index ] == NULL )
        break;

      file_ref = expandFilename( *(expand_files[ index ]) );
      *expand_files[ index ] = file_ref;

      index++;
    } while( True );

    resources_read = True;
  }

  info_ref -> icon_file          = resource_rec.icon_file;
  info_ref -> master_book        = resource_rec.master_book;
  info_ref -> print_script       = resource_rec.gen_print_command;
  info_ref -> system_pr_file     = resource_rec.system_printer_file;
  info_ref -> use_action_buttons = resource_rec.use_action_buttons;
  info_ref -> use_button_icons   = resource_rec.use_button_icons;
  info_ref -> use_help_label     = resource_rec.use_help_label;
  info_ref -> user_pr_file       = resource_rec.user_printer_file;


  /* Create the info window. */
  info_ref -> infoW = createInfoWindow( info_ref, parent );

  if( info_ref -> infoW == NULL ) {
    SysFree( info_ref );

    return( NULL );
  }


  /* Parse the supplied user and system books. */
  info_ref -> books[ 0 ] = parseBooks( resource_rec.user_books );
  info_ref -> books[ 1 ] = parseBooks( resource_rec.system_books );

  /* Search the current book. */
  if( LstLinkElements( info_ref -> books[ 0 ] ) > 0 ) {
    LstLinkGetFirst( info_ref -> books[ 0 ], (void *) &info_ref -> curr_book );

  } else if( LstLinkElements( info_ref -> books[ 1 ] ) > 0 ) {
    LstLinkGetFirst( info_ref -> books[ 1 ], (void *) &info_ref -> curr_book );

  } else {
    xitInDestroy( (XIT_IN_HANDLE) info_ref );

    return( NULL );
  } /* if */


  /* Initalize the info file. */
  info_ref -> file_handle = xitIfInitialize( info_ref -> curr_book.file );
  
  if( info_ref -> file_handle == NULL ) {
    xitInDestroy( (XIT_IN_HANDLE) info_ref );

    return( NULL );
  }


  /* Can we go back in time? */
  canGoBack( info_ref );


  return( (XIT_IN_HANDLE) info_ref );

} /* xitInInitialize */


/*----------------------------------------------------------------------*/

void
  xitInDestroy( XIT_IN_HANDLE  info_handle )
{

  /* Variables. */
  INFO_REC_REF  info_ref;


  /* Code. */

  if( info_handle == NULL )
    return;

  /* Our private data. */
  info_ref = (INFO_REC_REF) info_handle;

  /* Get rid of all windows. */
  if( info_ref -> infoW != NULL )
    XtDestroyWidget( info_ref -> infoW );


  return;

} /* xitInDestroy */


/*----------------------------------------------------------------------*/

void
  xitInDisplayInfo( XIT_IN_HANDLE  info_handle,
                    char           *book,
                    char           *chapter_id )
{

  /* Variables. */
  char            buffer[ 200 ];
  Arg             args[ 10 ];
  Cardinal        n;
  INFO_REC_REF    info_ref;
  XIT_IF_SECTION  section_rec;
  XIT_IF_STATUS   status;


  /* Code. */

  if( info_handle == NULL )
    return;

  /* Our private data. */
  info_ref = (INFO_REC_REF) info_handle;


  /* Book to use? */ 
  if( book != NULL )
    switchToBook( info_ref, book, True );


  /* Don't let our children spoil our size. */
  n = 0;
  XtSetArg( args[ n ], XmNallowShellResize, False ); n++;
  XtSetValues( info_ref -> infoW, args, n );


  /* Start at the toplevel? */
  if( chapter_id == NULL || strlen( chapter_id ) == 0 )
    strcpy( info_ref -> curr_chapter_id, 
            info_ref -> curr_book.first_chapter_id );
  else
    strcpy( info_ref -> curr_chapter_id, 
            chapter_id );

  /* Display the sections within the chapter. */
  displaySections( info_ref, info_ref -> curr_chapter_id );


  /* Display the text for the first section. */
  status = xitIfGetChapter( info_ref -> file_handle,
                            info_ref -> curr_chapter_id, 1,
                            &section_rec );

  displaySectionText( info_ref, &section_rec );


  /* Save in the history book. */
  saveHistory( info_ref, 
               info_ref -> curr_book.name,
               section_rec.chapter_id,
               section_rec.title );


  /* Set the window title. */
  sprintf( buffer, "%s  (%s)",
           msgDGetText( td, MINF_HELP_TITLE ), info_ref -> curr_book.name );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, buffer ); n++;
  XtSetValues( info_ref -> infoW, args, n );


  /* Make sure the info window is visable. */
  XtPopup( info_ref -> infoW, XtGrabNone );

  XRaiseWindow( XtDisplay( info_ref -> infoW ), 
                XtWindow(  info_ref -> infoW ) );

  XtMapWidget( info_ref -> infoW );


  return;

} /* xitInDisplyInfo */


/*----------------------------------------------------------------------*/

void
  xitInDisplayToc( XIT_IN_HANDLE  info_handle )
{

  /* Variables. */
  INFO_REC_REF    info_ref;


  /* Code. */

  if( info_handle == NULL )
    return;

  /* Our private data. */
  info_ref = (INFO_REC_REF) info_handle;


  /* Display the table of contents. */
  doToc( info_ref );


  return;

} /* xitInDisplyToc */


/*----------------------------------------------------------------------*/

static Boolean
  backHistory( INFO_REC_REF  info_ref,
               char          **book,
               char          **chapter_id,
               char          **subject )
{

  /* Variables. */
  HISTORY_REC  *history_ref;


  /* Code. */

  /* Any history? */
  if( info_ref -> current_hist < 1 )
    return( False );


  /* We don't want the current data. */
  history_ref = &info_ref -> history[ info_ref -> current_hist ];

  SysFree( history_ref -> book );
  SysFree( history_ref -> chapter_id );
  SysFree( history_ref -> subject );


  /* Fix counter. */
  info_ref -> current_hist--;


  /* Return data. */
  history_ref = &info_ref -> history[ info_ref -> current_hist ];

  *book       = SysNewString( history_ref -> book );
  *chapter_id = SysNewString( history_ref -> chapter_id );
  *subject    = SysNewString( history_ref -> subject );


  /* Can we go back in time? */
  canGoBack( info_ref );


  return( True );

} /* backHistory */


/*----------------------------------------------------------------------*/

static void
  canGoBack( INFO_REC_REF  info_ref )
{

  /* Variables. */
  Boolean   sensitive;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    mainW;
  Widget    tempW;


  /* Code. */

  mainW = XtNameToWidget( info_ref -> infoW, "InfoTlBase.InfoTlFo" );
  tempW = XtNameToWidget( mainW, "InfoPa.Pane2Fo.CommandRc.BackPb" );

  if( ! XtIsRealized( tempW ) )
    return;

  if( info_ref -> current_hist < 1 )
    sensitive = False;
  else
    sensitive=True;

  n = 0;
  XtSetArg( args[ n ], XmNsensitive, sensitive ); n++;
  XtSetValues( tempW, args, n );


  return;

} /* canGoBack */


/*----------------------------------------------------------------------*/

static void 
  createActionButtons( INFO_REC_REF   info_ref,
                       LST_DESC_TYPE  tag_list )
{

  /* Variables. */
  int                index;
  char               buffer[ 100 ];
  Arg                args[ 10 ];
  Cardinal           n;
  Widget             actionBu;
  Widget             actionRcW;
  Widget             mainW;
  Widget             tempW;
  XmString           xstr;
  ACTIVATE_DATA_REC  *button_data_ref;
  LST_STATUS         lst_status;
  XIT_IF_TAG_INFO    tag_info;


  /* Code. */

  if( ! info_ref -> use_action_buttons )
    return;

  mainW     = XtNameToWidget( info_ref -> infoW, "InfoTlBase.InfoTlFo" );
  actionRcW = XtNameToWidget( mainW, "InfoPa.Pane2Fo.ActionFr.ActionRc" );


  XtUnmapWidget( actionRcW );

  /* Delete any previous buttons. */
  index = 1;

  do {
    sprintf( buffer, "Action%dPb", index );

    tempW = XtNameToWidget( actionRcW, buffer );
    if( tempW == NULL )
      break;

    XtDestroyWidget( tempW );
    index++;
  } while( True );


  /* Add buttons for all the actions tags. */
  index = 1;

  lst_status = LstLinkCurrentFirst( tag_list );
  while( lst_status == LST_OK ) {

    (void) LstLinkGetCurrent( tag_list, &tag_info );

    /* Action button? */
    switch( tag_info.tag_type ) {

      case XIT_IF_TAG_ACTION_EXECUTE:
      case XIT_IF_TAG_ACTION_JUMP:

        sprintf( buffer, "Action%dPb", index );

        xstr = XmStringCreateLtoR( tag_info.param1, CS );

        n = 0;
        XtSetArg( args[ n ], XitRwNpixmapPosition, XitRwLEFT ); n++;
        XtSetArg( args[ n ], XmNlabelString, xstr ); n++;

        actionBu = XtCreateWidget( buffer, xitRwPixButtonWidgetClass,
                                   actionRcW, args, n );
        XmStringFree( xstr );

        /* Allocate button data. */
        button_data_ref = (ACTIVATE_DATA_REC *) 
                           SysMalloc( sizeof( ACTIVATE_DATA_REC ) );

        button_data_ref -> info_ref = info_ref;
        button_data_ref -> tag_type = tag_info.tag_type;
        button_data_ref -> param1   = SysNewString( tag_info.param1 );
        button_data_ref -> param2   = SysNewString( tag_info.param2 );
        button_data_ref -> param3   = SysNewString( tag_info.param3 );
        button_data_ref -> param4   = NULL;

        XtAddCallback( actionBu, XmNactivateCallback, 
                       (XtCallbackProc) activateCB,
                       (XtPointer) button_data_ref );
        XtAddCallback( actionBu, XmNdestroyCallback, 
                       (XtCallbackProc) destroyActivateCB,
                       (XtPointer) button_data_ref );

        /* Any pixmap? */
        if( strlen( tag_info.param2 ) > 0 && tag_info.param2[ 0 ] != ' ' ) {

          Pixel   background;
	  Pixel   foreground;
	  Pixmap  pixmap;

	  n = 0;
	  XtSetArg( args[ n ], XmNforeground, &foreground ); n++;
	  XtSetArg( args[ n ], XmNbackground, &background ); n++;
	  XtGetValues( actionBu, args, n );

	  pixmap = XmGetPixmap( XtScreen( mainW ), tag_info.param2,
				foreground, background );

	  if( pixmap != XmUNSPECIFIED_PIXMAP ) {
	    n = 0;
	    XtSetArg( args[ n ], XmNlabelPixmap, pixmap ); n++;
	    XtSetValues( actionBu, args, n );
	  }

	} /* if */

	XtManageChild( actionBu );

	index++;
        break;

      default:
        break;

    } /* switch */

    lst_status = LstLinkCurrentNext( tag_list );

  } /* while */

  XtMapWidget( actionRcW );


  return;

} /* createActionButtons */


/*----------------------------------------------------------------------*/

static Widget
  createInfoWindow( INFO_REC_REF     info_ref,
                    Widget           parent )
{

  /* Variables. */
  int        index;
  char       buffer[ 200 ];
  Arg        args[ 10 ];
  Cardinal   n;
  Dimension  height;
  Dimension  total_height;
  Widget     dataLocalW[ 13 ];
  Widget     doCmdBu[ 5 ];
  Widget     infoTl;
  Widget     menuCasc[ 2 ];
  Widget     menuFileBu[ 1 ];
  Widget     menuNavBu[ 2 ];
  Widget     pulldownMenu[ 2 ];
  Widget     tempW;
  Widget     workFo;
  XmString   xstr;

  static char  *pull_downs[] = { "pdown1", "pdown2" };

  static XIT_CASCADE_STRUCT menupane[] = {
    { "", "", "FilePane" },
    { "", "", "NavigatePane" },
  };

  static XIT_MENU_BUTTON_STRUCT file_casc[] = {
    { "", "", NULL, "ExitBu", True, False, False },
  };

  static XIT_MENU_BUTTON_STRUCT nav_casc[] = {
    { "", "", NULL, "TopBu",   True, False, False },
    { "", "", NULL, "BooksBu", True, False, False },
  };

  static XIT_TEXT_STRUCT help_text_def[] = {
    { "HelpTx", NULL, 5, False },
  };

  static XIT_PUSH_STRUCT do_cmd_def[] = {
    { "BackPb",       "", "", True, NULL },
    { "SearchPb",     "", "", True, NULL },
    { "TocPb",        "", "", True, NULL },
    { "PrintPb",      "", "", True, NULL },
    { "MasterBookPb", "", "", True, NULL },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   closeCB, NULL },
    { NULL, NULL,    NULL },
    { NULL, NULL,    NULL },
    { NULL, NULL,    NULL },
    { NULL, NULL,    NULL },
  };


  /* Code. */

  menupane[ 0 ].title    = msgDGetText( td, MINF_FILE_MENU );
  menupane[ 0 ].mnemonic = msgDGetText( td, MINF_FILE_MENU_ACC );
  menupane[ 1 ].title    = msgDGetText( td, MINF_NAV_MENU );
  menupane[ 1 ].mnemonic = msgDGetText( td, MINF_NAV_MENU_ACC );

  file_casc[ 0 ].title    = msgDGetText( td, MINF_EXIT_MENU );
  file_casc[ 0 ].mnemonic = msgDGetText( td, MINF_EXIT_MENU_ACC );

  nav_casc[ 0 ].title       = msgDGetText( td, MINF_TOP_MENU );
  nav_casc[ 0 ].mnemonic    = msgDGetText( td, MINF_TOP_MENU_ACC );
  nav_casc[ 1 ].title       = msgDGetText( td, MINF_BOOKS_MENU );
  nav_casc[ 1 ].mnemonic    = msgDGetText( td, MINF_BOOKS_MENU_ACC );

  do_cmd_def[ 0 ].title = msgDGetText( td, MINF_BACK_BUTTON );
  do_cmd_def[ 1 ].title = msgDGetText( td, MINF_SEARCH_BUTTON );
  do_cmd_def[ 2 ].title = msgDGetText( td, MINF_TOC_BUTTON );
  do_cmd_def[ 3 ].title = msgDGetText( td, MINF_MASTER_BOOK_BUTTON );
  do_cmd_def[ 4 ].title = msgDGetText( td, MINF_PRINT_BUTTON );

  action_buttons[ 0 ].label = msgDGetText( td, MINF_CLOSE_BUTTON );
  action_buttons[ 0 ].data  = info_ref;


  /* Create a separate window. */
  tempW = xitGetToplevelWidget( parent );

  infoTl = xitCreateToplevelDialog( tempW, "InfoTl", 
                                    1, 0,
                                    action_buttons, 
                                    XtNumber( action_buttons ) );

  n = 0;
  XtSetArg( args[ n ], XmNmappedWhenManaged, False ); n++;
  XtSetArg( args[ n ], XmNtitle, msgDGetText( td, MINF_HELP_TITLE ) ); n++;
  XtSetArg( args[ n ], XmNiconName, msgDGetText( td, MINF_HELP_TITLE ) ); n++;
  XtSetValues( infoTl, args, n );

  XtAddCallback( infoTl,    XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) info_ref );

  /* Close the window if this window is deleted. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( infoTl ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( infoTl,  &wm_delete_window, 1 );
    XmAddWMProtocolCallback( infoTl, wm_delete_window, 
                             (XtCallbackProc) closeCB, (XtPointer) info_ref );
  } /* block */


  /* Reference to the contents of the window. */
  workFo = XtNameToWidget( infoTl, "InfoTlBase.InfoTlFo" );


  /* Create the menubar and menu cascades. */
  menuBr = XmCreateMenuBar( workFo, "MenuBr", args, 0 );

  n = 0;
  for( index = 0; index < XtNumber( pull_downs ); index++ )
    pulldownMenu[ index ] = XmCreatePulldownMenu( menuBr,
                                                  pull_downs[ index ], 
                                                  NULL, n );

  for( index = 0; index < XtNumber( menuCasc ); index++ )
    menuCasc[ index ] = xitCreateCascadeButton( menuBr, 
                                                pulldownMenu[ index ], 
                                                &menupane[ index ] );


  /* Create the file menu. */
  for( index = 0; index < XtNumber( menuFileBu ); index++ )
    menuFileBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 0 ], 
                                                   &file_casc[ index ] );

  XtAddCallback( menuFileBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) closeCB, (XtPointer) info_ref );


  /* Create the navigate menu. */
  for( index = 0; index < XtNumber( menuNavBu ); index++ )
    menuNavBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 1 ], 
                                                  &nav_casc[ index ] );

  XtAddCallback( menuNavBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) topCB, (XtPointer) info_ref );
  XtAddCallback( menuNavBu[ 1 ], XmNactivateCallback, 
                 (XtCallbackProc) booksCB, (XtPointer) info_ref );


  /* Create the chapter title label. */
  sprintf( buffer, "%-50.50s", " " );

  chapterLa = xitCreateLabel( workFo, "ChapterLa", buffer, -1 );


  /* Paned window for text and topics. */
  n = 0;
  XtSetArg( args[ n ], XmNmarginWidth, 0 ); n++;
  infoPa = XmCreatePanedWindow( workFo, "InfoPa", args, n );


  /* First pane (text window). */
  n = 0;
  pane1Fo = XmCreateForm( infoPa, "Pane1Fo", args, n );

  /* Create window to hold the text (in a scrolled win). */
  helpTx = xitCreateTextScrolled( pane1Fo, &help_text_def[ 0 ] );

  n = 0;
  XtSetArg( args[ n ], XmNcursorPositionVisible, False ); n++;
  XtSetValues( helpTx, args, n );


  /* Second pane (action window). */
  n = 0;
  pane2Fo = XmCreateForm( infoPa, "Pane2Fo", args, n );


  /* Action buttons. */
  actionFr = XmCreateFrame( pane2Fo, "ActionFr", args, 0 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL  ); n++;
  XtSetArg( args[ n ], XmNpacking,     XmPACK_TIGHT ); n++;
  actionRc = XmCreateRowColumn( actionFr, "ActionRc", args, n );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmVERTICAL );  n++;
  XtSetArg( args[ n ], XmNheight, 40 ); n++;
  XtSetArg( args[ n ], XmNmappedWhenManaged, False ); n++;
  dummySp = XmCreateSeparator( actionRc, "DummySp", args, n );


  /* Create a command form. */
  n = 0;
  XtSetArg( args[ n ], XmNspacing, 10 ); n++;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
  commandRc = XmCreateRowColumn( pane2Fo, "CommandRc", args, n );

  for( index = 0; index < XtNumber( doCmdBu ); index++ ) {
    xstr = XmStringCreateLtoR( do_cmd_def[ index ].title, CS );

    n = 0;
    XtSetArg( args[ n ], XmNlabelString, xstr ); n++;

    doCmdBu[ index ] = XtCreateWidget( do_cmd_def[ index ].name, 
                                       xitRwPixButtonWidgetClass,
                                       commandRc, args, n );
    XmStringFree( xstr );

    /* Set the pixmap. */
    {
      int            height = 0;
      int            width = 0;
      unsigned char  *bits = NULL;
      Pixel          background;
      Pixel          foreground;
      Pixmap         pixmap;

      n = 0;
      XtSetArg( args[ n ], XmNforeground, &foreground ); n++;
      XtSetArg( args[ n ], XmNbackground, &background ); n++;
      XtGetValues( doCmdBu[ index ], args, n );

      switch( index ) {
        case 0:
          bits   = xitBackIcon_bits;
          width  = xitBackIcon_width;
          height = xitBackIcon_height;

          XtAddCallback( doCmdBu[ index ], XmNactivateCallback, 
                         (XtCallbackProc) doBackCB, (XtPointer) info_ref );
          break;
        case 1:
          bits   = xitSearchIcon_bits;
          width  = xitSearchIcon_width;
          height = xitSearchIcon_height;

          XtAddCallback( doCmdBu[ index ], XmNactivateCallback, 
                         (XtCallbackProc) doSearchCB, (XtPointer) info_ref );
          break;

        case 2:
          bits   = xitContentsIcon_bits;
          width  = xitContentsIcon_width;
          height = xitContentsIcon_height;

          XtAddCallback( doCmdBu[ index ], XmNactivateCallback, 
                         (XtCallbackProc) doContentsCB, (XtPointer) info_ref );
          break;

        case 3:
          bits   = xitMasterBookIcon_bits;
          width  = xitMasterBookIcon_width;
          height = xitMasterBookIcon_height;

          XtAddCallback( doCmdBu[ index ], XmNactivateCallback, 
                         (XtCallbackProc) doMasterBookCB,
                         (XtPointer) info_ref );
          break;

        case 4:
          bits   = xitPrinterIcon_bits;
          width  = xitPrinterIcon_width;
          height = xitPrinterIcon_height;

          XtAddCallback( doCmdBu[ index ], XmNactivateCallback, 
                         (XtCallbackProc) doPrintCB, (XtPointer) info_ref );
          break;
      }

      if( bits != NULL ) {
        pixmap = XCreatePixmapFromBitmapData(
                   XtDisplay( infoTl ),
                   RootWindowOfScreen( XtScreen( infoTl ) ),
                   (char *) bits, width, height,
                   foreground, background,
                   DefaultDepthOfScreen( XtScreen( infoTl ) ) );

        if( info_ref -> use_button_icons ) {
          n = 0;
          XtSetArg( args[ n ], XmNlabelPixmap, pixmap ); n++;
          XtSetValues( doCmdBu[ index ], args, n );
        }
      }
    } /* block */

  } /* loop */


  /* Third pane (topics). */
  n = 0;
  pane3Fo = XmCreateForm( infoPa, "Pane3Fo", args, n );

  /* Create the sections label. */
  sectionLa = xitCreateLabel( pane3Fo, "SectionLa", 
                              msgDGetText( td, MINF_SECTION_LABEL ), -1 );


  /* Create the subject list window. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,   XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,  XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight, 5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,  5 ); n++;
  sectionLi = XmCreateScrolledList( pane3Fo, "SectionLi", args, n );

  XtAddCallback( sectionLi, XmNsingleSelectionCallback, 
                 (XtCallbackProc) sectionListCB, (XtPointer) info_ref );
  XtAddCallback( sectionLi, XmNdefaultActionCallback, 
                 (XtCallbackProc) sectionListCB, (XtPointer) info_ref );


  /* Position the pane1 widgets. */
  xitAttachWidget( XtParent( helpTx ),
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL );

  /* Position the pane2 widgets. */
  xitAttachWidget( actionFr,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( commandRc,
                   XmATTACH_WIDGET, actionFr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );

  /* Position the pane3 widgets. */
  xitAttachWidget( sectionLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( sectionLi ),
                   XmATTACH_WIDGET, sectionLa, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,      XmATTACH_FORM, NULL );


  /* Position the widgets. */
  xitAttachWidget( menuBr,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( chapterLa,
                   XmATTACH_WIDGET, menuBr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( infoPa,
                   XmATTACH_WIDGET, chapterLa, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,      XmATTACH_FORM, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( chapterLa,             args, n );
  XtSetValues( XtParent( helpTx ),    args, n );
  XtSetValues( actionFr,              args, n );
  XtSetValues( actionRc,              args, n );
  XtSetValues( commandRc,             args, n );
  XtSetValues( sectionLa,             args, n );
  XtSetValues( XtParent( sectionLi ), args, n );


  /* Manage all the children. */
  XtManageChildren( doCmdBu,      XtNumber( doCmdBu ) );
  XtManageChildren( menuCasc,     XtNumber( menupane ) );
  XtManageChildren( menuFileBu,   XtNumber( menuFileBu ) );
  XtManageChildren( menuNavBu,    XtNumber( menuNavBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  XtManageChild( infoTl );

  if( ! info_ref -> use_action_buttons ) {
    XtUnmanageChild( actionFr );

    xitAttachWidget( commandRc,
                     XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                     XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  }

  XtRealizeWidget( infoTl );

  /* Set icon for this window. */
  {
    Pixel            background;
    Pixel            foreground;
    Pixmap           pixmap = XmUNSPECIFIED_PIXMAP;
    Window           dummy_win;
    GC               dummy_gc;
    XIT_ICON_WINDOW  icon_win;

    n = 0;
    XtSetArg( args[ n ], XmNbackground, &background ); n++;
    XtSetArg( args[ n ], XmNforeground, &foreground ); n++;
    XtGetValues( workFo, args, n );

    /* External pixmap? */
    if( strlen( info_ref -> icon_file ) > 0 )
      pixmap = XmGetPixmap( XtScreen( parent ), info_ref -> icon_file,
                            foreground, background );

    if( pixmap == XmUNSPECIFIED_PIXMAP )
      pixmap = XCreatePixmapFromBitmapData( 
                 XtDisplay( workFo ),
                 RootWindowOfScreen( XtScreen( workFo ) ),
                 (char *) xitHelpIcon_bits,
                 xitHelpIcon_width, xitHelpIcon_height,
                 foreground, background,
                 DefaultDepthOfScreen( XtScreen( workFo ) ) );

    icon_win.image                = pixmap;
    icon_win.icon_fg              = foreground;
    icon_win.icon_bg              = background;
    icon_win.report_expose_events = False;
    icon_win.x_offset             = 0;
    icon_win.y_offset             = 0;
    icon_win.label                = NULL;

    xitSetIconWindow( infoTl, &icon_win, &dummy_win, &dummy_gc );
  }


  /* We want to be able to change the pane 1/3 as we like. */
  n = 0;
  XtSetArg( args[ n ], XmNpaneMinimum, 100 ); n++;
  XtSetArg( args[ n ], XmNpaneMaximum, 1000 ); n++;
  XtSetValues( pane1Fo, args, n );
  XtSetValues( pane3Fo, args, n );

  /* Pane 2 should not change in size. */
  total_height = 3 * 5;

  if( info_ref -> use_action_buttons ) {
    n = 0;
    XtSetArg( args[ n ], XmNheight, &height ); n++;
    XtGetValues( actionFr, args, n );
    total_height = total_height + height;
  }

  n = 0;
  XtSetArg( args[ n ], XmNheight, &height ); n++;
  XtGetValues( commandRc, args, n );
  total_height = total_height + height;

  n = 0;
  XtSetArg( args[ n ], XmNpaneMinimum, total_height ); n++;
  XtSetArg( args[ n ], XmNpaneMaximum, total_height ); n++;
  XtSetValues( pane2Fo, args, n );


  /* Do we have a master book? */
  if( strlen( info_ref -> master_book ) == 0 )
    XtUnmanageChild( doCmdBu[ 3 ] );


  return( infoTl );

} /* createInfoWindow */


/*----------------------------------------------------------------------*/

static void
  displaySections( INFO_REC_REF  info_ref,
                   char          *chapter_id )
{

  /* Variables. */
  int             index;
  int             items;
  Arg             args[ 10 ];
  Cardinal        n;
  Widget          mainW;
  Widget          tempW;
  XmString        list_items[ MAX_SECTIONS ];
  XIT_IF_SECTION  section_rec;
  XIT_IF_STATUS   status;


  /* Code. */

  mainW = XtNameToWidget( info_ref -> infoW, "InfoTlBase.InfoTlFo" );

  strcpy( info_ref -> curr_chapter_id, chapter_id );


  /* Fetch and add the new subject items. */
  (void) xitIfResetFile( info_ref -> file_handle );

  status = xitIfGetNextSection( info_ref -> file_handle, chapter_id,
                                &section_rec );
  items = 0;

  while( status == XIT_IF_OK && items < MAX_SECTIONS ) {
    list_items[ items ] = XmStringCreate( section_rec.title, CS );
    items++;

    status = xitIfGetNextSection( info_ref -> file_handle, chapter_id,
                                  &section_rec );
  }


  /* Add the new items to the subject list. */
  tempW = XtNameToWidget( mainW, "InfoPa.Pane3Fo.SectionLiSW.SectionLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, items ); n++;
  XtSetValues( tempW, args, n );

  XmListDeselectAllItems( tempW );

  for( index = 0; index < items; index++ )
    XmStringFree( list_items[ index ] );

  XmListSetPos( tempW, 1 );


  return;

} /* displaySections */


/*----------------------------------------------------------------------*/

static void
  displaySectionSubject( INFO_REC_REF  info_ref,
                         char          *chapter_id,
                         char          *subject )
{

  /* Variables. */
  int       index;
  int       item_count;
  int       match = -1;
  char      *char_ref;
  char      *subject_list;
  char      *subject_list_ref;
  char      *subject_ref;
  Arg       args[ 3 ];
  Cardinal  n;
  Widget    mainW;
  Widget    tempW;
  XmString  *items;


  /* Code. */

  mainW = XtNameToWidget( info_ref -> infoW, "InfoTlBase.InfoTlFo" );


  /* Display the sections in the chapter. */
  displaySections( info_ref, chapter_id );


  /* Fetch the number of sections in the chapter. */
  tempW = XtNameToWidget( mainW, "InfoPa.Pane3Fo.SectionLiSW.SectionLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitemCount, &item_count ); n++;
  XtSetArg( args[ n ], XmNitems,     &items ); n++;
  XtGetValues( tempW, args, n );


  /* Strip leading and trailing spaces from the subject. */
  subject_ref = subject;
  while( isspace( *subject_ref ) )
    subject_ref++;

  char_ref = subject_ref + strlen( subject_ref ) - 1;
  while( isspace( *char_ref ) && char_ref > subject_ref )
    char_ref--;
  *(char_ref + 1) = '\0';


  /* Search all the sections in the chapter. */
  for( index = 0; index < item_count; index++ ) {

    /* Get the subject string. */
    subject_list = xitStringGetString( *(items + index), CS );

    /* Skip leading and trailing blanks. */
    subject_list_ref = subject_list;
    while( isspace( *subject_list_ref ) )
      subject_list_ref++;

    char_ref = subject_list_ref + strlen( subject_list_ref ) - 1;
    while( isspace( *char_ref ) && char_ref > subject_list_ref )
      char_ref--;
    *(char_ref + 1) = '\0';

    /* Check if we have a match. */
    match = strcmp( subject_ref, subject_list_ref );

    SysFree( subject_list );
       
    /* If we had a match, we have found our string. */
    if( match == 0 )
      break;

  } /* loop */


  if( match != 0 )
    return;


  /* Set position to the section in the list and select it. */
  tempW = XtNameToWidget( mainW, "InfoPa.Pane3Fo.SectionLiSW.SectionLi" );

  XmListSelectPos( tempW, index + 1, True );


  return;

} /* displaySectionSubject */


/*----------------------------------------------------------------------*/

static void
  displaySectionText( INFO_REC_REF        info_ref,
                      XIT_IF_SECTION_REF  section_ref )
{

  /* Variables. */
  int              segment_size = 10000;
  int              text_size;
  char             buffer[ 500 ];
  char             *char_ref;
  char             *text;
  Cursor           wait_cursor;
  Display          *display;
  Window           window;
  Widget           mainW;
  Widget           tempW;
  LST_DESC_TYPE    tag_list;
  XIT_IF_STATUS    status;


  /* Code. */

  mainW   = XtNameToWidget( info_ref -> infoW, "InfoTlBase.InfoTlFo" );
  display = XtDisplay( info_ref -> infoW );
  window  = XtWindow(  info_ref -> infoW );


  /* Fetch the tags in the text. */
  xitIfGetTags( info_ref -> file_handle, &tag_list );

  /* Add buttons for all the actions tags. */
  createActionButtons( info_ref, tag_list );

  xitIfFreeTagsInfo( tag_list );


  /* Display the current header for the text. */
  char_ref = section_ref -> title;
  while( isspace( *char_ref ) )
    char_ref++;

  tempW = XtNameToWidget( mainW, "ChapterLa" );
  xitStringSetLabel( tempW, char_ref );


  /* This might take time. */
  wait_cursor = XCreateFontCursor( display, XC_watch );
  XDefineCursor( display, window, wait_cursor );

  XFlush( display );


  /* Get the text for this item. */
  text      = SysMalloc( segment_size );
  text_size = segment_size;
  *text     = '\0';

  status = xitIfGetNextTextLine( info_ref -> file_handle,
                                 buffer, sizeof( buffer ) );

  while( status == XIT_IF_OK ) {

    /* Do we need to expand the buffer? */
    if( strlen( text ) + strlen( buffer ) + 2 > text_size ) {

      int  extend_size;

      extend_size = max( strlen( buffer ), segment_size ) + 10;
      text_size = text_size + extend_size;

      text = SysRealloc( text, text_size );

    } /* if */

    strcat( text, buffer );
    strcat( text, " \n" );

    status = xitIfGetNextTextLine( info_ref -> file_handle,
                                   buffer, sizeof( buffer ) );

  } /* while */


  /* Set the text. */
  tempW = XtNameToWidget( mainW, "InfoPa.Pane1Fo.HelpTxSW.HelpTx" );
  XmTextSetString( tempW, text );

  SysFree( text );  


  /* Normal cursor again. */
  XUndefineCursor( display, window );
  XFlush( display );


  return;

} /* displaySectionText */


/*----------------------------------------------------------------------*/

static void 
  doBackHistory( INFO_REC_REF  info_ref )
{

  /* Variables. */
  Boolean  ok;
  char     *book;
  char     *chapter_id;
  char     *subject;


  /* Code. */

  /* Fetch the previous history item. */
  ok = backHistory( info_ref,
                    &book, &chapter_id, &subject );
  if( ! ok )
    return;


  /* Switch books? */
  switchToBook( info_ref, book, False );


  /* Display the section. */
  info_ref -> save_history = False;

  displaySectionSubject( info_ref, chapter_id, subject );

  info_ref -> save_history = True;


  SysFree( book );
  SysFree( chapter_id );
  SysFree( subject );


  return;

} /* doBackHistory */


/*----------------------------------------------------------------------*/

static void 
  doBook( INFO_REC_REF  info_ref )
{

  /* Variables. */
  int              index;
  int              list;
  int              books_no[ 2 ];
  XIT_IF_BOOK_REC  book;
  LST_STATUS       lst_status;
  XIT_IB_BOOK_REC  *books_list[ 2 ];


  /* Code. */

  /* Initialize the book window? */
  if( info_ref -> book_handle == NULL )
    info_ref -> book_handle = xitIbInitialize(
                                info_ref -> infoW,
                                bookActionCB, info_ref );


  /* Create the list of system and user books to display. */
  for( list = 0; list < 2; list++ ) {

    books_no[ list ]   = LstLinkElements( info_ref -> books[ list ] );
    books_list[ list ] = NULL;

    if( books_no[ list ] > 0 ) {
      books_list[ list ] = (XIT_IB_BOOK_REC *)
        SysMalloc( books_no[ list ] * sizeof( XIT_IF_BOOK_REC ) );

      index = 0;

      lst_status = LstLinkCurrentFirst( info_ref -> books[ list ] );
      while( lst_status == LST_OK ) {
        (void) LstLinkGetCurrent( info_ref -> books[ list ], &book );

        strcpy( (books_list[ list ] + index) -> name,
                book.name );
        strcpy( (books_list[ list ] + index) -> description,
                book.description );

        index++;
        lst_status = LstLinkCurrentNext( info_ref -> books[ list ] );
      }
    } /* if */

  } /* loop */


  /* Display the book window. */
  xitIbSelectBook( info_ref -> book_handle,
                   books_list[ 0 ], books_no[ 0 ],
                   books_list[ 1 ], books_no[ 1 ] );


  /* Free allocated data. */
  if( books_list[ 0 ] != NULL )
    SysFree( books_list[ 0 ] );

  if( books_list[ 1 ] != NULL )
    SysFree( books_list[ 1 ] );


  return;

} /* doBook */


/*----------------------------------------------------------------------*/

static void 
  doMasterBook( INFO_REC_REF  info_ref )
{

  /* Variables. */
  int             items;
  char            book[ 50 ];
  char            chapter_id[ 50 ];
  XIT_IF_SECTION  section_rec;
  XIT_IF_STATUS   status;


  /* Code. */

  if( strlen( info_ref -> master_book ) == 0 )
    return;

  /* We need a book name and a chapter_id. */
  items = sscanf( info_ref -> master_book, "%[^:]:%s", book, chapter_id );
  if( items != 2 )
    return;


  /* Switch to new book. */
  switchToBook( info_ref, book, True );


  /* Goto the chapter id. */
  displaySections( info_ref, info_ref -> curr_book.first_chapter_id );

  status = xitIfGetChapter( info_ref -> file_handle,
                            info_ref -> curr_chapter_id, 1,
                            &section_rec );

  displaySectionText( info_ref, &section_rec );


  /* Save in the history book. */
  if( info_ref -> save_history )
    saveHistory( info_ref, 
                 info_ref -> curr_book.name,
                 section_rec.chapter_id,
                 section_rec.title );


  return;

} /* doMasterBook */


/*----------------------------------------------------------------------*/

static void 
  doPrint( INFO_REC_REF  info_ref )
{

  /* Variables. */
  int  section_no;


  /* Code. */

  /* Initialize the print window? */
  if( info_ref -> print_handle == NULL )
    info_ref -> print_handle = xitIpInitialize(
                                 info_ref -> infoW,
                                 info_ref -> system_pr_file,
                                 info_ref -> user_pr_file,
                                 info_ref -> print_script,
                                 printActionCB, info_ref );

  /* What section are we reading? */
  section_no = xitIfCurrentSection( info_ref -> file_handle );

  /* Display the print window. */
  xitIpDisplayPrinter( info_ref -> print_handle,
                       info_ref -> curr_book.name,
                       info_ref -> curr_chapter_id,
                       section_no,
                       info_ref -> curr_book.file );

  return;

} /* doPrint */


/*----------------------------------------------------------------------*/

static void 
  doSearch( INFO_REC_REF  info_ref )
{

  /* Code. */

  /* Initialize the search window? */
  if( info_ref -> search_handle == NULL )
    info_ref -> search_handle = xitIsInitialize( info_ref -> infoW,
                                                 info_ref -> books,
                                                 searchActionCB, info_ref );

  /* Display the search window. */
  xitIsDisplaySearch( info_ref -> search_handle,
                      info_ref -> curr_book.name );

  return;

} /* doSearch */


/*----------------------------------------------------------------------*/

static void 
  doToc( INFO_REC_REF  info_ref )
{

  /* Code. */

  /* Initialize the toc window? */
  if( info_ref -> toc_handle == NULL )
    info_ref -> toc_handle = xitItInitialize( info_ref -> infoW,
                                              tocActionCB, info_ref );

  /* Display the toc window. */
  xitItDisplayToc( info_ref -> toc_handle, info_ref -> curr_book.file );


  return;

} /* doToc */


/*----------------------------------------------------------------------*/

static
  char *expandFilename( char  *filename )
{

  /* Variables. */
  char  buffer[ PATH_MAX + 1 ];
  char  *filename_copy;
  char  *new_filename;
  char  *part_ref;


  /* Code. */

  /* Make a copy of the filename. */
  filename_copy = SysNewString( filename );

  buffer[ 0 ] = '\0';
  part_ref = strtok( filename_copy, "/" );

  while( part_ref != NULL ) {

    /* Is the part an environment variable? */
    if( *part_ref == '$' && getenv( part_ref + 1 ) != NULL )
      part_ref = getenv( part_ref + 1 );

    if( strlen( buffer ) > 0 || *filename == '/' )
      strcat( buffer, "/" );

    strcat( buffer, part_ref );

    part_ref = strtok( NULL, "/" );

  } /* while */

  SysFree( filename_copy );

  new_filename = SysNewString( buffer );


  return( new_filename );

} /* expandFilename */


/*----------------------------------------------------------------------*/

static Boolean
  jumpTo( INFO_REC_REF    info_ref,
          XIT_IF_SECTION  *section_rec )
{

  /* Variables. */
  XIT_IF_STATUS  status;


  /* Code. */

  /* Is this a link? */
  if( strlen( section_rec -> link ) > 0 ) {

    /* New book? */
    if( strlen( section_rec -> book ) > 0 )
      switchToBook( info_ref, section_rec -> book, True );

    /* Goto the chapter id. */
    displaySections( info_ref, section_rec -> link );

    /* Re-select the first header. */
    status = xitIfGetChapter( info_ref -> file_handle,
                              info_ref -> curr_chapter_id, 1,
                              section_rec );
  } /* if */


  /* Display the text for the selected section. */
  displaySectionText( info_ref, section_rec );


  /* Save in the history book. */
  if( info_ref -> save_history )
    saveHistory( info_ref, 
                 info_ref -> curr_book.name,
                 section_rec -> chapter_id,
                 section_rec -> title );


  return( True );

} /* jumpTo */


/*----------------------------------------------------------------------*/

static LST_DESC_TYPE
  parseBooks( char  *books )
{

  /* Variables. */
  int              char_read;
  int              items;
  char             buffer[ 100 + PATH_MAX ];
  char             *buffer_ref;
  char             *char_ref;
  LST_DESC_TYPE    book_list;
  LST_STATUS       lst_status;
  XIT_IF_BOOK_REC  book;


  /* Code. */

  /* Books are saved in a linked list. */
  book_list = LstLinkNew( sizeof( XIT_IF_BOOK_REC ), NULL );


  /* Start parsing the book information. */
  char_ref = books;

  char_read = strlen( char_ref );
  items = sscanf( char_ref, "%[^\n]%n", buffer, &char_read );
  while( items >= 1 ) {

    buffer_ref = buffer;
    while( isspace( *buffer_ref ) )
      buffer_ref++;

    /* Parse the book description. */
    items = sscanf( buffer_ref, "%[^#]#%[^#]#%[^#]#%[^#]",
                    book.name,
                    book.description,
                    book.first_chapter_id,
                    book.file );

    /* Insert the book definition in the list. */
    if( items != 4 )
      fprintf( stderr, "%s: Cannot parse %s\n",
               module_name, buffer );
    else {
      char*  file_name = SysExpandFilename( book.file );
      strcpy( book.file, file_name );
      SysFree( file_name );
      lst_status = LstLinkInsertLast( book_list, &book );
    }
    

    /* Skip the parsed token. */
    char_ref = char_ref + char_read;
    if( *char_ref == '\0' )
      break;

    /* Skip the \n at the end of the line. */
    char_ref++;
    if( *char_ref == '\0' )
      break;

    /* Next book. */
    char_read = strlen( char_ref );
    items = sscanf( char_ref, "%[^\n]%n", buffer, &char_read );

  } /* while */


  return( book_list );

} /* parseBooks */


/*----------------------------------------------------------------------*/

static void 
  saveHistory( INFO_REC_REF  info_ref,
               char          *book,
               char          *chapter_id,
               char          *subject )
{

  /* Variables. */
  int          index;
  HISTORY_REC  *from_history_ref;
  HISTORY_REC  *history_ref;
  HISTORY_REC  *to_history_ref;


  /* Code. */

  /* The same as before? */
  if( info_ref -> current_hist >= 0 ) {
    history_ref = &info_ref -> history[ info_ref -> current_hist ];

    if( strcmp( book,       history_ref -> book ) == 0 &&
        strcmp( chapter_id, history_ref -> chapter_id ) == 0 &&
        strcmp( subject,    history_ref -> subject ) == 0 )
      return;
  }


  /* History is full? */
  if( info_ref -> current_hist >= MAX_HISTORY - 1 ) {

    history_ref = &info_ref -> history[ 0 ];
    SysFree( history_ref -> book );
    SysFree( history_ref -> chapter_id );
    SysFree( history_ref -> subject );

    for( index = 1; index < MAX_HISTORY; index++ ) {
      from_history_ref = &info_ref -> history[ index ];
      to_history_ref   = &info_ref -> history[ index - 1 ];

      to_history_ref -> book       = from_history_ref -> book;
      to_history_ref -> chapter_id = from_history_ref -> chapter_id;
      to_history_ref -> subject    = from_history_ref -> subject;
    }

    history_ref = &info_ref -> history[ MAX_HISTORY - 1 ];
    SysFree( history_ref -> book );
    SysFree( history_ref -> chapter_id );
    SysFree( history_ref -> subject );

  } /* if */

  if( info_ref -> current_hist < MAX_HISTORY - 1 )
    info_ref -> current_hist++;


  /* Save Record. */
  history_ref = &info_ref -> history[ info_ref -> current_hist ];

  history_ref -> book       = SysNewString( book );
  history_ref -> chapter_id = SysNewString( chapter_id );
  history_ref -> subject    = SysNewString( subject );


  /* Can we go back in time? */
  canGoBack( info_ref );


  return;

} /* saveHistory */


/*----------------------------------------------------------------------*/

static void 
  scrollWindow( Widget  scrollBar,
                int     value )
{

  /* Variables. */
  int       sb_increment;
  int       sb_maximum;
  int       sb_minimum;
  int       sb_page_increment;
  int       sb_slider_size;
  int       sb_value;
  Arg       args[ 10 ];
  Cardinal  n;


  /* Code. */

  /* Information about scroll bar. */
  n = 0;
  XtSetArg( args[ n ], XmNminimum, &sb_minimum ); n++;
  XtSetArg( args[ n ], XmNmaximum, &sb_maximum ); n++;
  XtSetArg( args[ n ], XmNsliderSize, &sb_slider_size ); n++;
  XtSetArg( args[ n ], XmNincrement, &sb_increment ); n++;
  XtSetArg( args[ n ], XmNpageIncrement, &sb_page_increment ); n++;
  XtSetArg( args[ n ], XmNvalue, &sb_value ); n++;
  XtGetValues( scrollBar, args, n );

  if( value < sb_minimum )
    value = sb_minimum;
  if( value > (sb_maximum - sb_slider_size) )
    value = (sb_maximum - sb_slider_size);

  /* Set position for scroll bar. */
  XmScrollBarSetValues( scrollBar, value, sb_slider_size,
                        sb_increment, sb_page_increment, True );

  /* The scrolled window does not react on valueChanged callback. */
  {
    XmScrollBarCallbackStruct call_data;

    /* Create the call data, I hope it is OK to skip event and pixel. */
    call_data.value  = value;
    call_data.reason = XmCR_DRAG;

    /* The drag callback makes the scrolled window to react. */
    XtCallCallbacks( scrollBar, XmNdragCallback, (XtPointer) &call_data );

  } /* block */

  
  return;

} /* scrollWindow */


/*----------------------------------------------------------------------*/

static void
  startProcess( Widget  parent,
                char    *command )
{

  /* Variables. */
  Display  *display = NULL;
  Widget   tempW;


  /* Code. */

  if( parent != NULL )
    display = XtDisplay( parent );

  /* Start the process. */
  system( command );

  /* Tell the user what we are doing since this might take time. */
  if( parent != NULL ) {
    tempW = xitCreateInformationDialog(
              parent, "InformationDialog", 
              msgDGetText( td, MINF_INFORMATION_LABEL ),
              msgDGetText( td, MINF_START_PROGRAM ),
              NULL, NULL );

    XtAppAddTimeOut( XtWidgetToApplicationContext( parent ),
                     4 * 1000,
                     (XtTimerCallbackProc) XtDestroyWidget,
                     (XtPointer) tempW );

  } /* if */

  return;

} /* startProcess */


/*----------------------------------------------------------------------*/

static void 
  switchToBook( INFO_REC_REF  info_ref,
                char          *new_book,
                Boolean       also_if_current )
{

  /* Variables. */
  Boolean          found;
  int              index;
  char             buffer[ 200 ];
  Arg              args[ 10 ];
  Cardinal         n;
  XIT_IF_BOOK_REC  book;
  LST_STATUS       lst_status;


  /* Code. */

  /* Do we have this book already? */
  if( strlen( new_book ) == 0 )
    return;

  /* Are we already reading the book? */
  if( strcmp( new_book, info_ref -> curr_book.name ) == 0 && 
      ! also_if_current )
    return;


  found = False;

  /* Search the selected book. */
  for( index = 0; index < 2; index++ ) {

    /* Any books? */
    if( LstLinkElements( info_ref -> books[ index ] ) > 0 ) {

      lst_status = LstLinkCurrentFirst( info_ref -> books[ index ] );
      while( lst_status == LST_OK ) {

        lst_status = LstLinkGetCurrent( info_ref -> books[ index ], 
                                        &book );

        if( strcmp( new_book, book.name ) == 0 ) {
          found = True;
          goto done;
        }

        /* Next entry. */
        lst_status = LstLinkCurrentNext( info_ref -> books[ index ] );

      } /* while */

    } /* if */

  } /* loop */

  done:

  if( ! found ) {
    XBell( XtDisplay( info_ref -> infoW ), 100 );
    return;
  }


  /* Remove the current book. */
  xitIfDestroy( info_ref -> file_handle );


  /* Save the selected book as current book. */
  memcpy( &info_ref -> curr_book, &book, sizeof( XIT_IF_BOOK_REC ) );


  /* Remove any toc window (if exists). */
  if( info_ref -> toc_handle != NULL ) {
    xitItDestroy( info_ref -> toc_handle );

    info_ref -> toc_handle = NULL;
  }


  /* Open the new book. */
  info_ref -> file_handle = xitIfInitialize( info_ref -> curr_book.file );


  /* Set the window title. */
  sprintf( buffer, "%s  (%s)",
           msgDGetText( td, MINF_HELP_TITLE ), info_ref -> curr_book.name );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, buffer ); n++;
  XtSetValues( info_ref -> infoW, args, n );


  return;

} /* switchToBook */


/*----------------------------------------------------------------------*/

static void 
  activateCB( Widget             widget, 
              ACTIVATE_DATA_REC  *button_data_ref,
              XtPointer          call_data )
{

  /* Variables. */
  int             items;
  char            book[ 50 ];
  char            link[ 50 ];
  char            *cmd;
  Display         *display = NULL;
  XIT_IF_SECTION  section_rec;

  /* Code. */

  display = XtDisplay( button_data_ref -> info_ref -> infoW );

  /* Start external command? */
  if( button_data_ref -> tag_type == XIT_IF_TAG_ACTION_EXECUTE ) {
    cmd = (char *) SysMalloc( strlen( button_data_ref -> param3 ) + 50 );
    sprintf( cmd, "%s &", button_data_ref -> param3 );

    startProcess( button_data_ref -> info_ref -> infoW, cmd );
    SysFree( cmd );
  }

  /* Jump to book and chapter? */
  if( button_data_ref -> tag_type == XIT_IF_TAG_ACTION_JUMP ) {

    book[ 0 ] = '\0';
    link[ 0 ] = '\0';

    if( strchr( button_data_ref -> param3, ':' ) != NULL ) {
      items = sscanf( button_data_ref -> param3, "%[^:]:%s",
                      book, link );
      if( items != 2 )
          return;

    } else {
      items = sscanf( button_data_ref -> param3, "%s", link );
      if( items != 1 )
        return;
    }

    strcpy( section_rec.book, book );
    strcpy( section_rec.link, link );

    /* Jump to the book/chapter. */
    jumpTo( button_data_ref -> info_ref, &section_rec );

  } /* if */


  return;

} /* activateCB */


/*----------------------------------------------------------------------*/

static void 
  bookActionCB( XIT_IB_REASON  reason,
                void           *user_data,
                char           *book_name )
{

  /* Variables. */
  INFO_REC_REF    info_ref;
  XIT_IF_SECTION  section_rec;
  XIT_IF_STATUS   status;


  /* Code. */

  info_ref = (INFO_REC_REF) user_data;

  if( reason == XIT_IB_REASON_DESTROY ) {
    info_ref -> book_handle = NULL;
  }

  if( reason == XIT_IB_REASON_NEW_BOOK ) {
    switchToBook( info_ref, book_name, True );

    strcpy( info_ref -> curr_chapter_id, 
            info_ref -> curr_book.first_chapter_id );


    /* Display the sections within the chapter. */
    displaySections( info_ref, info_ref -> curr_chapter_id );


    /* Display the text for the first section. */
    status = xitIfGetChapter( info_ref -> file_handle,
                              info_ref -> curr_chapter_id, 1,
                              &section_rec );

    displaySectionText( info_ref, &section_rec );
  }


  return;

} /* bookActionCB */


/*----------------------------------------------------------------------*/

static void 
  booksCB( Widget        widget, 
           INFO_REC_REF  info_ref,
           XtPointer     call_data )
{

  /* Code. */

  doBook( info_ref );


  return;

} /* booksCB */


/*----------------------------------------------------------------------*/

static void 
  closeCB( Widget        widget, 
           INFO_REC_REF  info_ref,
           XtPointer     call_data )
{

  /* Code. */

  if( info_ref -> actionCB != NULL )
    (* info_ref -> actionCB)( XIT_IN_REASON_POPDOWN,
                              info_ref -> user_data );

  XtUnmapWidget( info_ref -> infoW );


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget        widget, 
             INFO_REC_REF  info_ref,
             XtPointer     call_data )
{

  /* Variables. */
  int          index;
  HISTORY_REC  *history_ref;


  /* Code. */

  /* Do we have a user action callback registered? */
  if( info_ref -> actionCB != NULL )
    (* info_ref -> actionCB)( XIT_IN_REASON_DESTROY, 
                              info_ref -> user_data );

  /* Books window? */
  if( info_ref -> book_handle != NULL )
    xitIbDestroy( info_ref -> book_handle );

  /* Printer window? */
  if( info_ref -> print_handle != NULL )
    xitIpDestroy( info_ref -> print_handle );

  /* Search window? */
  if( info_ref -> search_handle != NULL )
    xitIsDestroy( info_ref -> search_handle );

  /* Toc window? */
  if( info_ref -> toc_handle != NULL )
    xitItDestroy( info_ref -> toc_handle );


  /* Free allocated data. */
  for( index = 0; index <= info_ref -> current_hist; index++ ) {
    history_ref = &info_ref -> history[ index ];

    SysFree( history_ref -> book );
    SysFree( history_ref -> chapter_id );
    SysFree( history_ref -> subject );
  }

  SysFree( info_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  destroyActivateCB( Widget             widget, 
                     ACTIVATE_DATA_REC  *button_data_ref,
                     XtPointer          call_data )
{

  /* Code. */

  if( button_data_ref -> param1 != NULL )
    SysFree( button_data_ref -> param1 );

  if( button_data_ref -> param2 != NULL )
    SysFree( button_data_ref -> param2 );

  if( button_data_ref -> param3 != NULL )
    SysFree( button_data_ref -> param3 );

  if( button_data_ref -> param4 != NULL )
    SysFree( button_data_ref -> param4 );


  return;

} /* destroyActivateCB */


/*----------------------------------------------------------------------*/

static void 
  doBackCB( Widget        widget,
            INFO_REC_REF  info_ref,
            XtPointer     call_data )
{

  /* Code. */

  doBackHistory( info_ref );


  return;

} /* doBackCB */


/*----------------------------------------------------------------------*/

static void 
  doContentsCB( Widget        widget,
                INFO_REC_REF  info_ref,
                XtPointer     call_data )
{

  /* Code. */

  doToc( info_ref );


  return;

} /* doContentsCB */


/*----------------------------------------------------------------------*/

static void 
  doMasterBookCB( Widget        widget,
                  INFO_REC_REF  info_ref,
                  XtPointer     call_data )
{

  /* Code. */

  doMasterBook( info_ref );


  return;

} /* doMasterBookCB */


/*----------------------------------------------------------------------*/

static void 
  doPrintCB( Widget        widget,
             INFO_REC_REF  info_ref,
             XtPointer     call_data )
{

  /* Code. */

  doPrint( info_ref );


  return;

} /* doPrintCB */


/*----------------------------------------------------------------------*/

static void 
  doSearchCB( Widget        widget,
              INFO_REC_REF  info_ref,
              XtPointer     call_data )
{

  /* Code. */

  doSearch( info_ref );


  return;

} /* doSearchCB */


/*----------------------------------------------------------------------*/

static void 
  printActionCB( XIT_IP_REASON  reason,
                 void           *user_data )
{

  /* Variables. */
  INFO_REC_REF  info_ref;


  /* Code. */

  if( reason != XIT_IP_REASON_DESTROY )
    return;

  info_ref = (INFO_REC_REF) user_data;

  info_ref -> print_handle = NULL;


  return;

} /* printActionCB */


/*----------------------------------------------------------------------*/

static void 
  searchActionCB( XIT_IS_REASON  reason,
                  void           *user_data,
                  char           *book,
                  char           *chapter_id,
                  char           *title )
{

  /* Variables. */
  INFO_REC_REF  info_ref;


  /* Code. */

  info_ref = (INFO_REC_REF) user_data;

  if( reason == XIT_IS_REASON_DESTROY ) {
    info_ref -> search_handle = NULL;
  }

  if( reason == XIT_IS_REASON_NEW_SECTION ) {
    switchToBook( info_ref, book, False );

    displaySectionSubject( info_ref, chapter_id, title );
  }


  return;

} /* searchActionCB */


/*----------------------------------------------------------------------*/

static void 
  sectionListCB( Widget                widget, 
                 INFO_REC_REF          info_ref,
                 XmListCallbackStruct  *call_data )
{

  /* Variables. */
  XIT_IF_SECTION  section_rec;
  XIT_IF_STATUS   status;


  /* Code. */

  /* Position the counter on the correct header. */
  status = xitIfGetChapter( info_ref -> file_handle,
                            info_ref -> curr_chapter_id,
                            call_data -> item_position,
                            &section_rec );

  /* Jump to whatever. */
  (void) jumpTo( info_ref, &section_rec );


  return;

} /* sectionListCB */


/*----------------------------------------------------------------------*/

static void 
  tocActionCB( XIT_IT_REASON  reason,
               void           *user_data,
               char           *chapter_id,
               char           *title )
{

  /* Variables. */
  INFO_REC_REF  info_ref;


  /* Code. */

  info_ref = (INFO_REC_REF) user_data;

  if( reason == XIT_IT_REASON_DESTROY ) {
    info_ref -> toc_handle = NULL;
  }

  if( reason == XIT_IT_REASON_NEW_SECTION ) {
    displaySectionSubject( info_ref, chapter_id, title );
  }


  return;

} /* tocActionCB */


/*----------------------------------------------------------------------*/

static void 
  topCB( Widget        widget, 
         INFO_REC_REF  info_ref,
         XtPointer     call_data )
{

  /* Variables. */
  XIT_IF_SECTION  section_rec;
  XIT_IF_STATUS   status;


  /* Code. */

  /* Re-select the current book. */
  if( info_ref -> curr_book.name[ 0 ] == '\0' )
    return;

  switchToBook( info_ref, info_ref -> curr_book.name, True );


  /* Display the sections of the book. */
  displaySections( info_ref, info_ref -> curr_book.first_chapter_id );

  /* Re-select the first header. */
  status = xitIfGetChapter( info_ref -> file_handle,
                            info_ref -> curr_chapter_id, 1,
                            &section_rec );

  /* Display the text for the selected section. */
  displaySectionText( info_ref, &section_rec );


  return;

} /* topCB */
