/*----------------------------------------------------------------------------
--
--  Module:           xtmRemMain
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    The is a sort of alarm clock, you can define one or more alarms
--    and you will be reminded when at the time you defined. Alarms are
--    not saved when this program exits.
--
--    Syntax:
--      XDremind [flags]
--
--    Flags:
--      -usage     : Displays some help.
--      -help      : As above
--
--  Filename:         xtmRemMain.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-02-16
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmRemMain.c, Version: 1.1, Date: 95/02/18 16:09:36";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>

#include <Xm/Xm.h>

#include "System.h"
#include "Message.h"

#include "msgTopic.h"
#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCustBase.h"
#include "xtmHelp.h"
#include "xtmReminder.h"
#include "xtmResource.h"
#include "xitError.h"
#include "xitInfo.h"
#include "xitStickyMsg.h"
#include "xtmTools.h"

/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Name of program. */
#define PROGRAM_NAME   "xdremind"

/* Program class (also the name of the application defaults file). */
#define PROGRAM_CLASS  "XDiary"


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Basic data structure for the XDalarm appliaction. */
typedef struct {

  /* The X application context. */
  XtAppContext  context;

  /* Command line flags. */
  Boolean  startup_help;
  Boolean  version_help;

  /* Customization data. */
  XTM_GL_CUSTOM_DATA_REF  custom_data;

  /* Help system. */
  XIT_IN_HANDLE  info_handle;

} XTM_RD_BASE_DATA, *XTM_RD_BASE_DATA_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of program. */
static char  *program_name;

/* IDs for the help windows. */
static char  *reminder_window_id = "Reminder";


/* Name of text domain. */
static char  *text_domain      = "XDiary";
static char  *info_text_domain = "Topic";

/* Name of module. */
static char  *module_name = "xtmReminderMain";

/* Application data. */
static  XTM_RD_BASE_DATA  appl_data;

static  Widget  toplevel;

/* Program options. */
static XrmOptionDescRec options[] = {
  { "-geometry",     "RemindTl.geometry", XrmoptionSepArg, NULL },
  { "-h",            "*.StartupHelp",     XrmoptionNoArg,  "True" },
  { "-help",         "*.StartupHelp",     XrmoptionNoArg,  "True" },
  { "-iconic",       "RemindTl.iconic",   XrmoptionNoArg,  "True" },
  { "-lan",          "*.msgLanguage",     XrmoptionSepArg, NULL },
  { "-language",     "*.msgLanguage",     XrmoptionSepArg, NULL },
  { "-noFileLock",   "*.useFileLock",     XrmoptionNoArg,  "False" },
  { "-nofilelock",   "*.useFileLock",     XrmoptionNoArg,  "False" },
  { "-usage",        "*.StartupHelp",     XrmoptionNoArg,  "True" },
  { "-version",      "*.VersionHelp",     XrmoptionNoArg,  "True" },
};

/* Application resources. */
static XtResource  base_resources[] = {

  { "startupHelp", "StartupHelp", XtRBoolean, sizeof( Boolean ),
    XtOffset( XTM_RD_BASE_DATA_REF, startup_help ), 
    XtRString, "False" },

  { "versionHelp", "VersionHelp", XtRBoolean, sizeof( Boolean ),
    XtOffset( XTM_RD_BASE_DATA_REF, version_help ), 
    XtRString, "False" },

};


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  displayUsage();

static void
  exitCB( Widget  widget,
          void    *user_data,
          void    *call_data );

static void
  helpCB( Widget  widget,
          void    *user_data,
          int     help_index );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void 
  main( unsigned int argc, char *argv[] )
{

  /* Variables. */
  int                  index;
  char                 *char_ref;
  Arg                  args[ 10 ];
  Cardinal             n;
  Display              *display;
  TIM_STATUS_TYPE      time_status;
  XTM_CB_STATUS        custom_status;
  XTM_GL_CUSTOM_DATA   custom_data;
  XTM_RD_REMINDER_REF  reminder_ref;


  /* Code. */

  /* Fetch the name of the program. */
  program_name = PROGRAM_NAME;
  xitErSetApplicationName( PROGRAM_CLASS );


  /* Save the original command parameters. */
  custom_data.orig_argc = argc;
  custom_data.orig_argv = (char**) XtMalloc( argc * sizeof( char* ) );

  for( index = 0; index < argc; index++ )
    custom_data.orig_argv[ index ] = XtNewString( argv[ index ] );
  
  /* NLS enabled. */
  xtmToSetLocale( program_name );


  /* Initialization. */
  SysInitializeEnvironment();


/* Initialize toolkit and open display. */
  XtToolkitInitialize();

  appl_data.context = XtCreateApplicationContext();
  display = XtOpenDisplay( appl_data.context, NULL,
                           NULL, PROGRAM_CLASS,
                           options, XtNumber( options ),
#if XtSpecificationRelease < 5
                           (Cardinal *) &argc,
#else
                           (int *) &argc,
#endif
                           argv );

  if( display == NULL )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "main",
                  "Cannot open display, check your DISPLAY variable." );

  /* Resource mapping.*/
  xtmToInitializeResourceMapping( argc, argv, display );
  

  /* Create application shell. */
  n = 0;
  toplevel = XtAppCreateShell( NULL, PROGRAM_CLASS,
                               applicationShellWidgetClass,
                               display,
                               args, n );

  /* Get base application resources. */
  XtGetApplicationResources( toplevel, (XtPointer) &appl_data, 
                             base_resources, 
                             XtNumber( base_resources ), 
                             NULL, 0 );

  /* Get customize resources. */
  xtmRsFetchCustomResources( &custom_data, toplevel );

  
  /* A valid resource file? */
  if( ! custom_data.valid_resource_file )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "main",
                  "XDiary will only run with a correct X-Windows resource "
                  "file.\nPlease check the XDiary installation." );


  /* Initialize application data. */
  custom_data.cal_db_handle   = NULL;
  custom_data.group_db_handle = NULL;
  custom_data.archive_files   = NULL;
  custom_data.include_files   = NULL;

  appl_data.info_handle = NULL;
  appl_data.custom_data = &custom_data;


  /* Display current version? */
  if( appl_data.version_help ) {
    printf( "%s: Version: %s\n", program_name, VERSION_ID );
    exit( 0 );
  }

  /* Help requested? */
  if( appl_data.startup_help ) {
    displayUsage();
    exit( 0 );
  }


  /* Get customized data from file. */
  custom_status = xtmCbGetDataFromFile( appl_data.custom_data );

  if( custom_status == XTM_CB_WRONG_VERSION ) {
    char_ref = (char *) 
      SysMalloc( strlen( msgGetText( MXDI_CUST_WRONG_VERSION ) ) + 50 );

    sprintf( char_ref, msgGetText( MXDI_CUST_WRONG_VERSION ),
             xtmCbWhatVersion() );

    xitStDisplaySticky( toplevel,
                        char_ref, XmUNSPECIFIED_PIXMAP,
                        msgGetText( MXDI_OK_BUTTON ), NULL,
                        NULL, NULL, NULL,
                        NULL );
    SysFree( char_ref );
  }


  /* Initialize necessary text domains. */
  msgInitialize();
  msgInitCatalogue( text_domain, NULL, custom_data.msg_language,
                    msgXdiaryTexts );
  msgInitCatalogue( info_text_domain, NULL, custom_data.msg_language,
                    msgTopicTexts );

  /* Default catalogue Xdiary. */
  msgTextDomain( text_domain );


  /* Initialize the time formats. */
  time_status = TimInitializeFormat( custom_data.date_format,
                                     custom_data.time_format );
  if( time_status != TIM_OK )
    xitErMessage( toplevel, XIT_ER_ERROR, 
                  module_name, "main",
                  msgGetText( MXDI_ERRMSG_DATE_OR_TIME_FORMAT ) );


  /* Set colors and fonts in the resource database. */
  xtmRsFetchColors( &custom_data, toplevel );


  /* Initialize the help system. */
  appl_data.info_handle = xitInInitialize( toplevel,
                                           NULL, NULL );


  /* Initalize the reminder window. */
  reminder_ref = xtmRdInitializeReminder( toplevel,
                                          appl_data.context,
                                          exitCB, helpCB,
                                          (void *) &appl_data );

  /* Display the reminder window. */
  xtmRdDisplayReminder( reminder_ref );


  /* Session management. */
  xtmToSessionMgmt( xtmRdToplevelWindow( reminder_ref ),
                    custom_data.orig_argc, custom_data.orig_argv );


  /* Enter the event loop. */
  XtAppMainLoop( appl_data.context );

} /* main */


/*----------------------------------------------------------------------*/

static void
  displayUsage()
{

  printf( 
    "\n"
    "%s (%s): Defines and displays short-lived alarms.\n" 
    "\n"
    "Usage:\n"
    "  %s [flags]\n"
    "\n"
    "Flags:\n"
    "  -fmap large      : Use a large font.\n"
    "  -fmap medium     : Use a medium font.\n"
    "  -fmap small      : Use a small font.\n"
    "  -help            : Display this help.\n"
    "  -h               : See above.\n"
    "  -language <lan>  : Use the language <lan>.\n"
    "  -lan <lan>       : See above.\n"
    "  -noFileLock      : Don't use any file locking.\n"
    "  -palette gray    : Use color palette Gray.\n"
    "  -palette lila    : Use color palette Lila.\n"
    "  -palette motif   : Use color palette Motif.\n"
    "  -palette neon    : Use color palette Neon.\n"
    "  -palette nordic  : Use color palette Nordic.\n"
    "  -palette red     : Use color palette Red.\n"
    "  -palette sea     : Use color palette Sea.\n"
    "  -palette sky     : Use color palette Sky.\n"
    "  -palette wood    : Use color palette Wood.\n"
    "  -usage           : Display this help.\n"
    "  -version         : Display the current version.\n"
    "\n",
    program_name, VERSION_ID, program_name );

  return;

} /* displayUsage */


/*----------------------------------------------------------------------*/

static void
  exitCB( Widget  widget,
          void    *user_data,
          void    *call_data )
{

  /* Code. */

  exit( 0 );


} /* exitCB */


/*----------------------------------------------------------------------*/

static void
  helpCB( Widget  widget,
          void    *user_data,
          int     help_index )
{

  /* Variables. */
  XTM_RD_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = (XTM_RD_BASE_DATA_REF) user_data;

  xtmHlDisplayHelp( appl_data_ref -> info_handle,
                    help_index,
                    reminder_window_id, "" );

  return;

} /* helpCB */


