/*----------------------------------------------------------------------------
--
--  Module:           xtmAclUnix
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Manage access control lists for vanilla UNIX.
--
--  Filename:         xtmAclUnix.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-10-15
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmAclUnix.c, Version: 1.1, Date: 95/02/18 16:00:30";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"

#include "msgTopic.h"
#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmAccBase.h"
#include "xtmCalDb.h"
#include "xtmCustBase.h"
#include "xtmEditAclUx.h"
#include "xtmHelp.h"
#include "xtmIcons.h"
#include "xtmResource.h"
#include "xitError.h"
#include "xitInfo.h"
#include "xitStickyMsg.h"
#include "xitTools.h"
#include "xtmTools.h"

/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Name of program. */
#define PROGRAM_NAME   "xdaclunix"

/* Program class (also the name of the application defaults file). */
#define PROGRAM_CLASS  "XDiary"

/* Local widgets in the window. */
#define actionRc       dataLocalW[  0 ]
#define dbLocLa        dataLocalW[  1 ]
#define dbLocLi        dataLocalW[  2 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Basic data structure for the XDaclUnix appliaction. */
typedef struct {

  /* Command line flags. */
  Boolean  startup_help;
  Boolean  version_help;

  /* Any changes to the aACLs? */
  Boolean  acl_changed;

  /* Calendar selction window. */
  Widget  calSelectW;

  /* Toplevel widget. */
  Widget  toplevel;

  /* The X application context. */
  XtAppContext  context;

  /* Info reference. */
  XIT_IN_HANDLE  info_handle;

  /* ACL window. */
  XTM_AU_HANDLE  acl_handle;

  /* Customization data. */
  XTM_GL_CUSTOM_DATA_REF  custom_data;

} XTM_AL_BASE_DATA, *XTM_AL_BASE_DATA_REF;



/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of program. */
static char  *program_name;

/* Name of module. */
static char  *module_name = "xtmAclUnix";

/* Name of text domain. */
static char  *text_domain      = "XDiary";
static char  *info_text_domain = "Topic";

/* IDs for the help window. */
static char  *select_window_id   = "AclCalSelect";

/* Keyboard translations for newline in multi-line editor. */
XtTranslations  xtm_di_newline_trans;

/* Application data. */
static  XTM_AL_BASE_DATA  appl_data;


/* Key translations. */
static char newlineTrans[] = 
  "<Key>Return:    newline()";


/* Program options. */
static XrmOptionDescRec options[] = {
  { "-geometry",     "*.AclTl.geometry",  XrmoptionSepArg, NULL },
  { "-h",            "*.StartupHelp",     XrmoptionNoArg,  "True" },
  { "-help",         "*.StartupHelp",     XrmoptionNoArg,  "True" },
  { "-iconic",       "*.AclTl.iconic",    XrmoptionNoArg,  "True" },
  { "-lan",          "*.msgLanguage",     XrmoptionSepArg, NULL },
  { "-language",     "*.msgLanguage",     XrmoptionSepArg, NULL },
  { "-noFileLock",   "*.useFileLock",     XrmoptionNoArg,  "False" },
  { "-nofilelock",   "*.useFileLock",     XrmoptionNoArg,  "False" },
  { "-usage",        "*.StartupHelp",     XrmoptionNoArg,  "True" },
  { "-version",      "*.VersionHelp",     XrmoptionNoArg,  "True" },
};

/* Application resources. */
static XtResource  base_resources[] = {

  { "startupHelp", "StartupHelp", XtRBoolean, sizeof( Boolean ),
    XtOffset( XTM_AL_BASE_DATA_REF, startup_help ), 
    XtRString, "False" },

  { "versionHelp", "VersionHelp", XtRBoolean, sizeof( Boolean ),
    XtOffset( XTM_AL_BASE_DATA_REF, version_help ), 
    XtRString, "False" },

};


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  aclActionCB( XTM_AU_REASON  reason,
               void           *user_data );

static void 
  closeCB( Widget                widget,
           XTM_AL_BASE_DATA_REF  appl_data_ref,
           XtPointer             call_data );

static Widget
  createSelectWindow( Widget                parent,
                      XTM_AL_BASE_DATA_REF  appl_data_ref );

static void 
  defaultListCB( Widget                widget,
                 XTM_AL_BASE_DATA_REF  appl_data_ref,
                 XmListCallbackStruct  *call_data );

static void
  displayUsage();

static void 
  doExitCB( Widget     widget,
            XtPointer  client_data,
            XtPointer  call_data );

static void 
  editAclCB( Widget                widget,
             XTM_AL_BASE_DATA_REF  appl_data_ref,
             XtPointer             call_data );

static void 
  helpCB( Widget                widget,
          XTM_AL_BASE_DATA_REF  appl_data_ref,
          XtPointer             call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void 
  main( unsigned int argc, char *argv[] )
{

  /* Variables. */
  int                 index;
  char                *char_ref;
  Arg                 args[ 10 ];
  Cardinal            n;
  Display             *display;
  TIM_STATUS_TYPE     time_status;
  XTM_CB_STATUS       custom_status;
  XTM_GL_CUSTOM_DATA  custom_data;


  /* Code. */

  /* Fetch the name of the program. */
  program_name = PROGRAM_NAME;
  xitErSetApplicationName( PROGRAM_CLASS );

  
  /* Save the original command parameters. */
  custom_data.orig_argc = argc;
  custom_data.orig_argv = (char**) XtMalloc( argc * sizeof( char* ) );

  for( index = 0; index < argc; index++ )
    custom_data.orig_argv[ index ] = XtNewString( argv[ index ] );
  
  /* NLS enabled. */
  xtmToSetLocale( program_name );


  /* Initialization. */
  SysInitializeEnvironment();


  /* Initialize toolkit and open display. */
  XtToolkitInitialize();

  appl_data.context = XtCreateApplicationContext();
  display = XtOpenDisplay( appl_data.context, NULL,
                           NULL, PROGRAM_CLASS,
                           options, XtNumber( options ),
#if XtSpecificationRelease < 5
                           (Cardinal *) &argc,
#else
                           (int *) &argc,
#endif
                           argv );

  if( display == NULL )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "main",
                  "Cannot open display, check your DISPLAY variable." );

  /* Resource mapping.*/
  xtmToInitializeResourceMapping( argc, argv, display );
  
  /* Default translations */
  xtm_di_newline_trans   = XtParseTranslationTable( newlineTrans );


  /* Create application shell. */
  n = 0;
  appl_data.toplevel = XtAppCreateShell( NULL, PROGRAM_CLASS,
                                         applicationShellWidgetClass,
                                         display,
                                         args, n );

  /* Get base application resources. */
  XtGetApplicationResources( appl_data.toplevel, (XtPointer) &appl_data, 
                             base_resources, 
                             XtNumber( base_resources ), 
                             NULL, 0 );

  /* Get customize resources. */
  xtmRsFetchCustomResources( &custom_data, appl_data.toplevel );

  
  /* A valid resource file? */
  if( ! custom_data.valid_resource_file )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "main",
                  "XDiary will only run with a correct X-Windows resource "
                  "file.\nPlease check the XDiary installation." );


  /* Initialize application data. */
  custom_data.cal_db_handle   = NULL;
  custom_data.group_db_handle = NULL;
  custom_data.archive_files   = NULL;
  custom_data.include_files   = NULL;

  appl_data.custom_data = &custom_data;
  appl_data.acl_handle  = NULL;
  appl_data.info_handle = NULL;
  appl_data.acl_changed = False;


  /* Display current version? */
  if( appl_data.version_help ) {
    printf( "%s: Version: %s\n", program_name, VERSION_ID );
    exit( 0 );
  }

  /* Help requested? */
  if( appl_data.startup_help ) {
    displayUsage();
    exit( 0 );
  }


  /* Get customized data from file. */
  custom_status = xtmCbGetDataFromFile( appl_data.custom_data );

  if( custom_status == XTM_CB_WRONG_VERSION ) {
    char_ref = (char *) 
      SysMalloc( strlen( msgGetText( MXDI_CUST_WRONG_VERSION ) ) + 50 );

    sprintf( char_ref, msgGetText( MXDI_CUST_WRONG_VERSION ),
             xtmCbWhatVersion() );

    xitStDisplaySticky( appl_data.toplevel,
                        char_ref, XmUNSPECIFIED_PIXMAP,
                        msgGetText( MXDI_OK_BUTTON ), NULL,
                        NULL, NULL, NULL,
                        NULL );
    SysFree( char_ref );
  }


  /* Initialize necessary text domains. */
  msgInitialize();
  msgInitCatalogue( text_domain, NULL, custom_data.msg_language,
                    msgXdiaryTexts );
  msgInitCatalogue( info_text_domain, NULL, custom_data.msg_language,
                    msgTopicTexts );

  /* Default catalogue Xdiary. */
  msgTextDomain( text_domain );


  /* Initialize the time formats. */
  time_status = TimInitializeFormat( custom_data.date_format,
                                     custom_data.time_format );
  if( time_status != TIM_OK )
    xitErMessage( appl_data.toplevel, XIT_ER_ERROR, 
                  module_name, "main",
                  msgGetText( MXDI_ERRMSG_DATE_OR_TIME_FORMAT ) );


  /* Set colors and fonts in the resource database. */
  xtmRsFetchColors( &custom_data, appl_data.toplevel );


  /* Create the calendar select window. */
  appl_data.calSelectW = createSelectWindow( appl_data.toplevel, &appl_data );


  /* Session management. */
  xtmToSessionMgmt( appl_data.calSelectW,
                    custom_data.orig_argc, custom_data.orig_argv );


  /* Initialize the help system. */
  appl_data.info_handle = xitInInitialize( appl_data.toplevel,
                                           NULL, NULL );


  /* Enter the event loop. */
  XtAppMainLoop( appl_data.context );


} /* main */


/*----------------------------------------------------------------------*/

static Widget
  createSelectWindow( Widget                parent,
                      XTM_AL_BASE_DATA_REF  appl_data_ref )
{

  /* Variables. */
  int                     index;
  int                     index1;
  char                    buffer[ 100 ];
  char                    *char_ref;
  char                    *db_names;
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  aclTl;
  Widget                  actionBu[ 1 ];
  Widget                  dataLocalW[ 3 ];
  Widget                  toplevel;
  Widget                  workFo;
  XmString                list_items[ 500 ];
  XTM_GL_CUSTOM_DATA_REF  custom_data;

  static XIT_PUSH_STRUCT action_def[] = {
    { "EditAclPb", "", "", True, NULL },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   closeCB, NULL },
    { NULL, NULL,    NULL },
    { NULL, NULL,    NULL },
    { "",   helpCB,  NULL },
  };


  /* Code. */

  action_def[ 0 ].title = msgGetText( MXDI_ACL_EDIT );

  action_buttons[ 0 ].label = msgGetText( MXDI_CLOSE_BUTTON );
  action_buttons[ 0 ].data  = appl_data_ref;
  action_buttons[ 3 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 3 ].data  = appl_data_ref;


  custom_data = appl_data_ref -> custom_data;

  /* Create a toplevel dialog with buttons. */
  toplevel = xitGetToplevelWidget( parent );

  aclTl = xitCreateToplevelDialog( toplevel, "AclTl",
                                   1, 0,
                                   action_buttons, 
                                   XtNumber( action_buttons ) );

  char_ref = msgGetText( MXDI_UNIX_ACCESS_TITLE );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, char_ref ); n++;
  XtSetValues( aclTl, args, n );

  char_ref = msgGetText( MXDI_UNIX_ACCESS_IC_TITLE );

  n = 0;
  XtSetArg( args[ n ], XmNiconName, char_ref ); n++;
  XtSetValues( aclTl, args, n );


  /* Exit the application if this window is deleted. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( aclTl ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( aclTl, &wm_delete_window, 1 );
    XmAddWMProtocolCallback( aclTl, wm_delete_window, 
                             (XtCallbackProc) doExitCB, (XtPointer) NULL );
  } /* block */


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( aclTl, "AclTlBase.AclTlFo" );

  dbLocLa = xitCreateLabel( workFo, "DbLocLa", 
                            msgGetText( MXDI_DB_LOCATION_LABEL ), -1 );

  /* List with database location. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  dbLocLi = XmCreateScrolledList( workFo, "DbLocLi", args, n );

  XtAddCallback( dbLocLi, XmNdefaultActionCallback, 
                 (XtCallbackProc) editAclCB, (XtPointer) appl_data_ref );
  XtAddCallback( dbLocLi, XmNdefaultActionCallback,
                 (XtCallbackProc) defaultListCB, (XtPointer) appl_data_ref );



  /* Actions. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNspacing,      10 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 10 ); n++;
  actionRc = XmCreateRowColumn( workFo, "ActionRc", args, n );

  for( index = 0; index < XtNumber( actionBu ); index++ )
    actionBu[ index ] = xitCreatePushButton( actionRc, &action_def[ index ] );

  XtAddCallback( actionBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) editAclCB, (XtPointer) appl_data_ref );


  /* Put the elements together. */
  xitAttachWidget( dbLocLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( dbLocLi ),
                   XmATTACH_WIDGET, dbLocLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,    XmATTACH_NONE, NULL );
  xitAttachWidget( actionRc,
                   XmATTACH_WIDGET, XtParent( dbLocLi ), XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL,                  XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetValues( dbLocLa,             args, n );
  XtSetValues( XtParent( dbLocLi ), args, n );
  XtSetValues( actionRc,            args, n );


  /* Manage the widgets. */
  XtManageChildren(  actionBu,   XtNumber( actionBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set icon for this window. */
  xtmIcSetSimpleIcon( aclTl, workFo, XTM_IC_ICON_ACL );

  /* Set the size of the window. */
  xitSetSizeToplevelDialog( aclTl, True );


  /* Make the final attachments. */
  xitAttachWidget( actionRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( XtParent( dbLocLi ),
                   XmATTACH_WIDGET, dbLocLa, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL,    XmATTACH_WIDGET, actionRc );


  /* Display the database select window. */
  XtPopup( aclTl, XtGrabNone );


  /* Set values for the database locations. */
  (void) xtmCdFetchDbNames( custom_data -> cal_db_handle, &db_names );
  char_ref = db_names;
  index    = 0;

  do {

    int              char_read;
    char             db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
    XTM_CD_CAL_INFO  db_info;

    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    sscanf( char_ref, "%s%n", db_name, &char_read );
    char_ref = char_ref + char_read;


    /* Fetch information about the database. */
    (void) xtmCdFetchNamedDb( custom_data -> cal_db_handle, db_name,
                              &db_info, NULL );

    /* We cannot change included calendars. */
    if( flagIsSet( db_info.flags, XTM_CD_FLAG_IS_INCLUDED ) )
      continue;

    /* Only the owner can change ACL. */
    if( db_info.owner_uid != getuid() )
      continue;

    sprintf( buffer, "%-15.15s %s", 
             db_info.short_name, db_info.directory );

    list_items[ index ] = XmStringCreate( buffer, CS );
    index++;

  } while( True && index < XtNumber( list_items ) );

  if( db_names != NULL )
    SysFree( db_names );


  /* The list is always sorted. */
  xitSortStringList( list_items, index );

  /* Assign the database locations to the list. */
  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, index ); n++;
  XtSetValues( dbLocLi, args, n );

  /* Free allocated memory. */
  for( index1 = 0; index1 < index; index1++ )
    XmStringFree( list_items[ index1 ] );

  XmListSelectPos( dbLocLi, 1, False );


  return( aclTl );

} /* createSelectWindow */


/*----------------------------------------------------------------------*/

static void
  displayUsage()
{

  printf( 
    "\n"
    "%s (%s): XDiary access control for plain UNIX.\n" 
    "\n"
    "Allows you to define access control information for the calendars you\n"
    "are the owner of.\n"
    "\n"
    "Usage:\n"
    "  %s [flags]\n"
    "\n"
    "Flags:\n"
    "  -fmap large      : Use a large font.\n"
    "  -fmap medium     : Use a medium font.\n"
    "  -fmap small      : Use a small font.\n"
    "  -help            : Display this help.\n"
    "  -h               : See above.\n"
    "  -language <lan>  : Use the language <lan>.\n"
    "  -lan <lan>       : See above.\n"
    "  -noFileLock      : Don't use any file locking.\n"
    "  -os <osId>       : Use the version defined by <osId>.\n"
    "  -palette gray    : Use color palette Gray.\n"
    "  -palette lila    : Use color palette Lila.\n"
    "  -palette motif   : Use color palette Motif.\n"
    "  -palette neon    : Use color palette Neon.\n"
    "  -palette nordic  : Use color palette Nordic.\n"
    "  -palette red     : Use color palette Red.\n"
    "  -palette sea     : Use color palette Sea.\n"
    "  -palette sky     : Use color palette Sky.\n"
    "  -palette wood    : Use color palette Wood.\n"
    "  -usage           : Display this help.\n"
    "  -version         : Display the current version.\n"
    "\n",
    program_name, VERSION_ID, program_name );


  return;

} /* displayUsage */


/*----------------------------------------------------------------------*/

static void 
  aclActionCB( XTM_AU_REASON  reason,
               void           *user_data )
{

  /* Variables. */
  XTM_AL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = (XTM_AL_BASE_DATA_REF) user_data;


  /* ACL window destroyed? */
  if( reason == XTM_AU_REASON_DESTROY )
    appl_data_ref -> acl_handle = NULL;


  /* ACLs saved? */
  if( reason == XTM_AU_REASON_OK || reason == XTM_AU_REASON_APPLY )
    appl_data_ref -> acl_changed = True;
  

  return;

} /* aclActionCB */


/*----------------------------------------------------------------------*/

static void 
  closeCB( Widget                widget,
           XTM_AL_BASE_DATA_REF  appl_data_ref,
           XtPointer             call_data )
{

  /* Variables. */
  Widget  tempW;


  /* Code. */

  /* Any ACL changed? */
  if( appl_data_ref -> acl_changed ) {
    tempW = xitCreateInformationDialog( 
              appl_data_ref -> calSelectW, "InformationDialog", 
	      msgGetText( MXDI_INFORMATION_LABEL ),
              msgGetText( MXDI_CUST_HOW_TO_ACTIVATE ),
              doExitCB, NULL );
    return;
  }

  doExitCB( widget, NULL, NULL );


} /* closeCB */


/*----------------------------------------------------------------------*/

static void 
  defaultListCB( Widget                widget,
                 XTM_AL_BASE_DATA_REF  appl_data_ref,
                 XmListCallbackStruct  *call_data )
{

  /* Code. */

  /* This is called from a list default selection, select the object! */
  XmListSelectPos( widget, call_data -> item_position, False );

  editAclCB( widget, appl_data_ref, NULL );


  return;

} /* addFromListCB */


/*----------------------------------------------------------------------*/

static void 
  doExitCB( Widget     widget,
            XtPointer  client_data,
            XtPointer  call_data )
{

  /* Code. */

  exit( 0 );


} /* doExitCB */


/*----------------------------------------------------------------------*/

static void 
  editAclCB( Widget                widget,
             XTM_AL_BASE_DATA_REF  appl_data_ref,
             XtPointer             call_data )
{

  /* Variables. */
  int       selected_item_count;
  char      cal_location[ XTM_GL_MAX_LOCATION + 1 ];
  char      cal_name[ XTM_GL_MAX_CAL_NAME + 1 ];
  char      *char_ref;
  Arg       args[ 3 ];
  Cardinal  n;
  Widget    listW;
  XmString  *selected_items;


  /* Code. */

  listW = XtNameToWidget( appl_data_ref -> calSelectW,
                          "AclTlBase.AclTlFo.DbLocLiSW.DbLocLi" );

  /* Fetch selected items from the list. */
  n = 0;
  XtSetArg( args[ n ], XmNselectedItemCount, &selected_item_count ); n++;
  XtSetArg( args[ n ], XmNselectedItems, &selected_items ); n++;
  XtGetValues( listW, args, n );

  if( selected_item_count != 1 )
    return;  


  /* Extract the calendar and location. */
  char_ref = xitStringGetString( *selected_items, CS );

  sscanf( char_ref, "%s %s", cal_name, cal_location );
  SysFree( char_ref );


  /* Create the ACL window? */
  if( appl_data_ref -> acl_handle == NULL )
    appl_data_ref -> acl_handle = xtmAuInitialize( 
                                    appl_data_ref -> custom_data,
                                    appl_data_ref -> info_handle,
                                    appl_data_ref -> calSelectW,
                                    aclActionCB, (void *) appl_data_ref );

  /* Display the ACL window. */
  xtmAuEditAcl( appl_data_ref -> acl_handle,
                cal_name, cal_location );


  return;

} /* editAclCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget                widget,
          XTM_AL_BASE_DATA_REF  appl_data_ref,
          XtPointer             call_data )
{

  /* Code. */

  xtmHlDisplayHelp( appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    select_window_id, "" );

  return;

} /* selectHelpCB */


