/*----------------------------------------------------------------------------
--
--  Module:           xtmGroupDb
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Routines for the groups database.
--
--  Filename:         xtmGroupDb.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1993-02-06
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmGroupDb.c, Version: 1.1, Date: 95/02/18 15:52:23";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"

#include "xtmGroupDb.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* The group database. */
typedef struct {

  /* The group list. */
  LST_DESC_TYPE  group_list;

} GROUP_DB, *GROUP_DB_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmGroupDb";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  clearGroupEntry( void  *data_ref );

static LST_COMPARE
  searchGroupName( XTM_GD_ENTRY_REF  group_ref,
                   char              *group_name );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

Boolean
  xtmGdAddEntry( XTM_GD_HANDLE     group_handle,
                 char              *name,
                 XTM_GD_ENTRY_REF  group_ref )
{

  /* Variables. */
  char          *members;
  char          *old_members;
  GROUP_DB_REF  group_db_ref;
  LST_STATUS    lst_status;


  /* Code. */

  if( group_handle == NULL )
    return( False );

  group_db_ref = (GROUP_DB_REF) group_handle;


  /* If we already have an entry with this name, remove it. */
  (void) xtmGdDeleteEntry( group_handle, group_ref -> name );


  /* Save a copy. */
  strcpy( group_ref -> name, name );

  old_members = group_ref -> members;

  if( group_ref -> members != NULL ) {
    members = SysNewString( group_ref -> members );
    group_ref -> members = members;
  }


  /* Add the entry to the end of the list. */
  lst_status = LstLinkInsertLast( group_db_ref -> group_list, group_ref );

  group_ref -> members = old_members;

  if( lst_status != LST_OK )
    return( False );


  return( True );

} /* xtmGdAddEntry */


/*----------------------------------------------------------------------*/

Boolean
  xtmGdChangeEntry( XTM_GD_HANDLE     group_handle,
                    char              *name,
                    XTM_GD_ENTRY_REF  group_ref )
{

  /* Variables. */
  Boolean  status;


  /* Code. */

  if( group_handle == NULL )
    return( False );


  /* Delete the old entry. */
  status = xtmGdDeleteEntry( group_handle, name );


  /* Add the new entry. */
  status = xtmGdAddEntry( group_handle, name, group_ref );


  return( status );

} /* xtmGdChangeEntry */


/*----------------------------------------------------------------------*/

XTM_GD_HANDLE
  xtmGdCopyGroups( XTM_GD_HANDLE  group_handle )
{

  /* Variables. */
  GROUP_DB_REF      copy_group_db_ref;
  GROUP_DB_REF      orig_group_db_ref;
  LST_STATUS        lst_status;
  XTM_GD_ENTRY      group_info;
  XTM_GD_ENTRY_REF  group_ref;
  XTM_GD_HANDLE     copy_handle;


  /* Code. */

  if( group_handle == NULL )
    return( False );

  orig_group_db_ref = (GROUP_DB_REF) group_handle;


  /* The copy list. */
  copy_handle = xtmGdInitialize();

  copy_group_db_ref = (GROUP_DB_REF) copy_handle;


  /* Search all groups and make a copy. */
  lst_status = LstLinkCurrentFirst( orig_group_db_ref -> group_list );

  while( lst_status == LST_OK ) {

    group_ref = (XTM_GD_ENTRY_REF)
                  LstLinkGetCurrentRef( orig_group_db_ref -> group_list );

    memcpy( (void *) &group_info,
            (void *) group_ref,
            sizeof( XTM_GD_ENTRY ) );

    if( group_ref -> members != NULL )
      group_info.members = SysNewString( group_ref -> members );

    /* Add the entry to the end of the list. */
    (void) LstLinkInsertLast( copy_group_db_ref -> group_list, &group_info );

    lst_status = LstLinkCurrentNext( orig_group_db_ref -> group_list );

  } /* while */


  return( (XTM_GD_HANDLE) copy_group_db_ref );

} /* xtmGdCopyGroups */


/*----------------------------------------------------------------------*/

Boolean
  xtmGdDeleteEntry( XTM_GD_HANDLE  group_handle,
                    char           *group_name )
{

  /* Variables. */
  void          *group_ref;
  GROUP_DB_REF  group_db_ref;
  LST_STATUS    lst_status;


  /* Code. */

  if( group_handle == NULL )
    return( False );

  group_db_ref = (GROUP_DB_REF) group_handle;


  /* Search the entry in the group list. */
  lst_status = LstLinkSearchFirst( group_db_ref -> group_list,
                                   (void *) group_name,
                                   (EQUALS_FUNC_TYPE) searchGroupName );

  if( lst_status != LST_OK )
    return( False );


  /* Remove the group. */
  group_ref = LstLinkGetCurrentRef( group_db_ref -> group_list );

  clearGroupEntry( group_ref );
  (void) LstLinkDeleteCurrent( group_db_ref -> group_list );


  return( True );

} /* xtmGdDeleteEntry */


/*----------------------------------------------------------------------*/

Boolean
  xtmGdFetchGroupNames( XTM_GD_HANDLE  group_handle,
                        char           **group_names )
{

  /* Variables. */
  int               items;
  GROUP_DB_REF      group_db_ref;
  LST_STATUS        lst_status;
  XTM_GD_ENTRY_REF  group_ref;


  /* Code. */

  if( group_handle == NULL )
    return( False );

  group_db_ref = (GROUP_DB_REF) group_handle;


  items = LstLinkElements( group_db_ref -> group_list );

  if( items <= 0 ) {
    *group_names = SysNewString( " " );
    return( False );
  }

  /* We know the length of string to return. */
  *group_names = (char *) SysMalloc( items * (XTM_GD_MAX_GROUP_NAME + 1) + 5 );
  **group_names = '\0';


  /* Search all groups. */
  lst_status = LstLinkCurrentFirst( group_db_ref -> group_list );

  while( lst_status == LST_OK ) {

    group_ref = (XTM_GD_ENTRY_REF)
                  LstLinkGetCurrentRef( group_db_ref -> group_list );

    strcat( *group_names, group_ref -> name );
    strcat( *group_names, " " );

    lst_status = LstLinkCurrentNext( group_db_ref -> group_list );

  } /* while */


  return( True );

} /* xtmGdFetchGroupNames */


/*----------------------------------------------------------------------*/

Boolean
  xtmGdFetchNamedGroup( XTM_GD_HANDLE     group_handle,
                        char              *group_name,
                        XTM_GD_ENTRY_REF  return_group_ref )
{

  /* Variables. */
  GROUP_DB_REF      group_db_ref;
  LST_STATUS        lst_status;
  XTM_GD_ENTRY_REF  group_ref;


  /* Code. */

  if( group_handle == NULL )
    return( False );

  group_db_ref = (GROUP_DB_REF) group_handle;


  /* Search the entry in the group list. */
  lst_status = LstLinkSearchFirst( group_db_ref -> group_list,
                                   (void *) group_name,
                                   (EQUALS_FUNC_TYPE) searchGroupName );

  if( lst_status != LST_OK )
    return( False );


  /* Return a copy. */
  group_ref = (XTM_GD_ENTRY_REF)
                LstLinkGetCurrentRef( group_db_ref -> group_list );

  memcpy( (void *) return_group_ref,
          (void *) group_ref,
          sizeof( XTM_GD_ENTRY ) );

  if( group_ref -> members != NULL )
    return_group_ref -> members = SysNewString( group_ref -> members );


  return( True );

} /* xtmGdFetchNamedGroup */


/*----------------------------------------------------------------------*/

void
  xtmGdFree( XTM_GD_HANDLE  group_handle )
{

  /* Variables. */
  GROUP_DB_REF  group_db_ref;


  /* Code. */

  if( group_handle == NULL )
    return;

  group_db_ref = (GROUP_DB_REF) group_handle;


  /* Free the group list. */
  LstLinkClearDataAndList( group_db_ref -> group_list, clearGroupEntry );


  /* Free the record itself. */
  SysFree( group_db_ref );


  return;

} /* xtmGdFree */


/*----------------------------------------------------------------------*/

XTM_GD_HANDLE
  xtmGdInitialize()
{

  /* Variables. */
  GROUP_DB_REF  group_db_ref;


  /* Code. */

  /* Allocate space for a new group database. */
  group_db_ref = SysNew( GROUP_DB );

  if( group_db_ref == NULL )
    return( NULL );

  group_db_ref -> group_list = LstLinkNew( sizeof( XTM_GD_ENTRY ), NULL );


  return( (XTM_GD_HANDLE) group_db_ref );

} /* xtmGdInitialize */


/*----------------------------------------------------------------------*/

static void
  clearGroupEntry( void  *data_ref )
{

  /* Variables. */
  XTM_GD_ENTRY_REF  group_ref;


  /* Code. */

  group_ref = (XTM_GD_ENTRY_REF) data_ref;

  if( group_ref -> members != NULL ) {
    SysFree( group_ref -> members );

    group_ref -> members = NULL;
  }


  return;

} /* clearGroupEntry */


/*----------------------------------------------------------------------*/

static LST_COMPARE
  searchGroupName( XTM_GD_ENTRY_REF  group_ref,
                   char              *group_name )
{

  /* Code. */

  if( strcmp( group_ref -> name, group_name ) == 0 )
    return( LST_EQUAL );


  return( LST_NOT_EQUAL );

} /* searchGroupName */
