/*----------------------------------------------------------------------------
--
--  Module:           xtmDelEntry
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Delete an entry from the diary database.
--
--  Filename:         xtmDelEntry.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-04-10
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmDelEntry.c, Version: 1.1, Date: 95/02/18 15:52:11";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Xm.h>

#include "System.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmDbTools.h"
#include "xtmTools.h"
#include "xtmUpdate.h"
#include "xtmDbMisc.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmDelEntry.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the copy/move window. */
#define databaseFs    dataLocalW[  0 ]
#define databaseLa    dataLocalW[  1 ]
#define dbRc          dataLocalW[  2 ]
#define messageLa     dataLocalW[  3 ]
#define timeRc        dataLocalW[  4 ]
#define toDateFs      dataLocalW[  5 ]
#define toTimeFs      dataLocalW[  6 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Information about the entry to delete. */
typedef struct {

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Entry being moved/copied. */
  XTM_DB_ALL_ENTRY_REF  use_entry_ref;

  /* Our parent window. */
  Widget  parentW;

} ENTRY_INFO, *ENTRY_INFO_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmDeleteEntry";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  deleteEntryCB( Widget          widget,
                 ENTRY_INFO_REF  entry_ref,
                 XtPointer       call_data );

static void 
  destroyCB( Widget          widget,
             ENTRY_INFO_REF  entry_ref,
             XtPointer       call_data );


/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void 
  xtmDlDeleteEntry( XTM_GL_BASE_DATA_REF  appl_data_ref,
                    Widget                parent,
                    char                  *db_name,
                    UINT32                entry_id )
{

  /* Variables. */
  Boolean                 ok;
  Widget                  tempW;
  ENTRY_INFO_REF          entry_ref;
  XTM_DB_ALL_ENTRY_REF    use_entry_ref;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_STATUS           db_status;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  /* Fetch database information. */
  ok = xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle,
                          db_name,
                          &db_info, NULL );
  if( ! ok )
    return;

  /* We need write access. */
  if( ! flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_WRITE ) ) {
    xitErMessage( parent, XIT_ER_ERROR,
                  module_name, "xtmDlDeleteEntry",
                  msgGetText( MXDI_ERRMSG_NO_ACCESS_DB ) );

    return;
  }


  /* Create and initialize our private data. */
  entry_ref = SysNew( ENTRY_INFO );
  if( entry_ref == NULL )
    return;

  entry_ref -> appl_data_ref = appl_data_ref;
  entry_ref -> parentW       = parent;
  entry_ref -> use_entry_ref = SysNew( XTM_DB_ALL_ENTRY_DEF );

  entry_ref -> use_entry_ref -> all_text = NULL;


  /* Open the database and fetch the entry. */
  ok = xtmDmOpenDatabase( appl_data_ref, db_name, XTM_DB_FLAG_MODE_READ,
                          &database );
  if( ! ok )
    raise exception;


  /* Fetch the entry. */
  db_status = xtmDbFetchEntry( &database, entry_id, 
                               entry_ref -> use_entry_ref, 
                               &entry_ref -> use_entry_ref -> all_text );

  xtmDbCloseEntryDb( &database );


  if( db_status != XTM_DB_OK ) {
    xitErMessage( parent, XIT_ER_ERROR,
                  module_name, "xtmDlDeleteEntry",
                  msgGetText( MXDI_NO_ENTRY ) );

    raise exception;
  }

  use_entry_ref = entry_ref -> use_entry_ref;


  /* Ask for confirmation? */
  if( appl_data_ref -> custom_data -> confirm_actions ) {

    tempW = xitCreateQuestionDialog( 
              parent, "QuestionDialog",
              msgGetText( MXDI_DELETE_ENTRY_TITLE ),
              msgGetText( MXDI_DELETE_ENTRY_CONF ),
              deleteEntryCB, entry_ref, 
              NULL, NULL );

    XtAddCallback( tempW, XmNdestroyCallback, 
                   (XtCallbackProc) destroyCB, (XtPointer) entry_ref );

    return;
  }

  /* Remove the entry. */
  deleteEntryCB( NULL, entry_ref, NULL );

  /* Release the user data. */
  if( entry_ref -> use_entry_ref != NULL ) {
    SysFree( entry_ref -> use_entry_ref -> all_text );
    SysFree( entry_ref -> use_entry_ref );
  }

  SysFree( entry_ref );


  return;


  /* Exception handler. */
  exception:

    if( entry_ref -> use_entry_ref -> all_text != NULL )
      SysFree( entry_ref -> use_entry_ref -> all_text );

    SysFree( entry_ref -> use_entry_ref );
    SysFree( entry_ref );

    return;


} /* xtmDlDeleteEntry */


/*----------------------------------------------------------------------*/

static void 
  deleteEntryCB( Widget          widget,
                 ENTRY_INFO_REF  entry_ref,
                 XtPointer       call_data )
{

  /* Variables. */
  Boolean                 ok;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_STATUS           db_status;


  /* Code. */

  /* Open the database. */
  ok = xtmDmOpenDatabase( entry_ref -> appl_data_ref,
                          entry_ref -> use_entry_ref -> db_name, 
                          XTM_DB_FLAG_MODE_WRITE,
                          &database );
  if( ! ok )
    return;


  /* Delete the entry. */
  db_status = xtmDbDeleteEntry( &database, 
                                entry_ref -> use_entry_ref -> entry.id );

  xtmDbCloseEntryDb( &database );

  if( db_status != XTM_DB_OK ) {
    xitErMessage( entry_ref -> parentW, XIT_ER_ERROR,
                  module_name, "deleteEntryCB",
                  msgGetText( MXDI_ERRMSG_DELETE_ENTRY ) );

    return;
  }


  /* Update the daylist/calendar. */
  xtmUpDoUpdate( (XTM_UP_CALENDAR | XTM_UP_SCHEDULE | XTM_UP_PLANNER), NULL );


  return;

} /* deleteEntryCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget          widget,
             ENTRY_INFO_REF  entry_ref,
             XtPointer       call_data )
{

  /* Code. */

  /* Release the user data. */
  if( entry_ref -> use_entry_ref != NULL ) {
    SysFree( entry_ref -> use_entry_ref -> all_text );
    SysFree( entry_ref -> use_entry_ref );
  }

  SysFree( entry_ref );


  return;

} /* destroyCB */
