/*----------------------------------------------------------------------------
--
--  Module:           xtmAlarmControl
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Module to handle the separate running alarm processes. Each XDiary
--    can have a number of alarm processes running. In this module, the
--    user can check which processes are running, start and stop processes.
--
--  Filename:         xtmAlarmCtrl.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-07-03
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmAlarmCtrl.c, Version: 1.1, Date: 95/02/18 15:51:54";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/signal.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>
#include <X11/cursorfont.h>

#include <Xm/Xm.h>
#include <Xm/BulletinB.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"
#include "SigHandler.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmCustBase.h"
#include "xtmDbTools.h"
#include "xtmHelp.h"
#include "xtmTools.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmAlarmCtrl.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* File protections. */
#define  MODE_644  (S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)


/* Local widgets in the alarm control window. */
#define dbStatusLi         dataLocalW[  0 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmAlarmControl";

/* IDs for the help windows. */
static char  *alarm_window_id = "AlarmCtrl";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  alarmProcessExitCB( int   this_signal,
                      void  *user_data );

static void
  alarmProcessExitMsg( XTM_GL_BASE_DATA_REF  appl_data_ref );

static void 
  closeCB( Widget                widget,
           XTM_GL_BASE_DATA_REF  appl_data_ref,
           XtPointer             call_data );

static Boolean
  createPipe( XTM_GL_BASE_DATA_REF  appl_data_ref );

Widget
  createWindow( Widget                parent,
                XTM_GL_BASE_DATA_REF  appl_data_ref );

static void 
  dbStatusListCB( Widget                widget, 
                  XTM_GL_BASE_DATA_REF  appl_data_ref,
                  XmListCallbackStruct  *call_data );

static void 
  helpCB( Widget                widget,
          XTM_GL_BASE_DATA_REF  appl_data_ref,
          XtPointer             call_data );

static Boolean
  openCmdPipe( XTM_GL_BASE_DATA_REF  appl_data_ref );

static Boolean
  sendCmdPipe( XTM_GL_BASE_DATA_REF  appl_data_ref,
               Widget                controlW,
               char                  *command );

static void
  setActionButtons( XTM_GL_BASE_DATA_REF  appl_data_ref,
                    Boolean               start_prc,
                    Boolean               stop_prc );

static void
  setProcessData( XTM_GL_BASE_DATA_REF  appl_data_ref );

static void
  setProcessState( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   char                  *db_name,
                   Boolean               active );

static Boolean
  startAlarmProcess( XTM_GL_BASE_DATA_REF  appl_data_ref,
                     Widget                controlW );

static Boolean
  startCalendarAlarm( XTM_GL_BASE_DATA_REF  appl_data_ref,
                      Widget                controlW,
                      char                  *db_name );

static void
  startCalendarAlarmCB( Widget                widget,
                        XTM_GL_BASE_DATA_REF  appl_data_ref,
                        XtPointer             call_data );

static Boolean
  stopCalendarAlarm( XTM_GL_BASE_DATA_REF  appl_data_ref,
                     Widget                controlW,
                     char                  *db_name );

static void
  stopCalendarAlarmCB( Widget                widget,
                       XTM_GL_BASE_DATA_REF  appl_data_ref,
                       XtPointer             call_data );


/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void
  xtmAcAlarmProcessControl( Widget                parent,
                            XTM_GL_BASE_DATA_REF  appl_data_ref )
{

  /* Code. */

  /* Is this form already displayed? */
  if( appl_data_ref -> process_info -> mainFo == NULL )
    appl_data_ref -> process_info -> mainFo = createWindow( parent,
                                                            appl_data_ref );
  else
    XtManageChild( appl_data_ref -> process_info -> mainFo );


  /* Assign values to the widgets. */
  setProcessData(   appl_data_ref );
  setActionButtons( appl_data_ref, False, False );


  return;

} /* xtmAcAlarmProcessControl */


/*----------------------------------------------------------------------*/

Boolean
  xtmAcProcessRunning( XTM_GL_BASE_DATA_REF  appl_data_ref,
                       char                  *db_name )
{

  /* Variables. */
  int                   index;
  XTM_GL_PROC_INFO_REF  process_info;


  /* Code. */

  process_info = appl_data_ref -> process_info;

  /* Is the given process running? */
  for( index = 0; index < XTM_GL_MAX_ALARM_PRC; index++ ) {

    if( strcmp( db_name, process_info -> processes[ index ].db_name ) == 0 )
      return( True );

  } /* loop */


  return( False );

} /* xtmAcProcessRunning */


/*----------------------------------------------------------------------*/

Boolean
  xtmAcStartDefaultAlarmPrc( Widget                parent,
                             XTM_GL_BASE_DATA_REF  appl_data_ref )
{

  /* Variables. */
  Boolean                 ok;
  char                    buffer[ 200 ];
  char                    *char_ref;
  char                    *db_names;
  XTM_GL_CUSTOM_DATA_REF  custom_data;


  /* Code. */

  custom_data = appl_data_ref -> custom_data;


  /* Fetch all databases. */
  (void) xtmCdFetchDbNames( custom_data -> cal_db_handle, &db_names );
  char_ref = db_names;

  do {

    int              char_read;
    char             db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
    XTM_CD_CAL_INFO  db_info;

    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    sscanf( char_ref, "%s%n", db_name, &char_read );
    char_ref = char_ref + char_read;


    /* Fetch information about the database. */
    (void) xtmCdFetchNamedDb( custom_data -> cal_db_handle, db_name,
                              &db_info, NULL );


    /* Start an alarm process for this diary DB? */
    if( flagIsSet( db_info.flags, XTM_CD_FLAG_ALARM ) ) {

      /* If read access is not allowed, give the user a message. */
      if( ! flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_READ ) ) {

        sprintf( buffer, msgGetText( MXDI_CANNOT_START_ALARM_PROCESS ),
                 db_info.short_name );

        xitMsgWinDisplayMessage( appl_data_ref -> msg_win_ref, buffer, 0 );

      /* Start alarms for this database. */
      } else {

        ok = startCalendarAlarm( appl_data_ref, 
                                 parent, db_info.short_name );
        if( ! ok ) {
          SysFree( db_names );

          return( False );
        }

      } /* if */

    } /* if */

  } while( True );

  SysFree( db_names );


  return( True );

} /* xtmAcStartDefaultAlarmPrc */


/*----------------------------------------------------------------------*/

static void
  alarmProcessExitMsg( XTM_GL_BASE_DATA_REF  appl_data_ref )
{

  /* Code. */

  xitErMessage( appl_data_ref -> toplevel, XIT_ER_ERROR, 
                module_name, "alarmProcessExit",
                msgGetText( MXDI_ERRMSG_ALARM_PROCESS_EXIT ) );

  /* Popdown any window. */
  closeCB( NULL, appl_data_ref, NULL );


  return;

} /* alarmProcessExitMsg */


/*----------------------------------------------------------------------*/

static Boolean
  createPipe( XTM_GL_BASE_DATA_REF  appl_data_ref )
{

  /* Variables. */
  int   status;
  char  *fifo_name;


  /* Code. */

  /* Do we have a pipe for the alarm process communication? */
  if( appl_data_ref -> alarm_fifo_name != NULL )
    return( True );


  fifo_name = tmpnam( NULL );

  status = mkfifo( fifo_name, (mode_t) MODE_644 );
  if( status < 0 ) {
    xitMsgWinDisplayMessage( appl_data_ref -> msg_win_ref,
                             msgGetText( MXDI_CANNOT_OPEN_CMD_PIPE ), 0 );
    return( False );
  }

  appl_data_ref -> alarm_fifo_name = SysNewString( fifo_name );


  return( True );

} /* createPipe */


/*----------------------------------------------------------------------*/

Widget
  createWindow( Widget                parent,
                XTM_GL_BASE_DATA_REF  appl_data_ref )
{

  /* Variables. */
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    almProcFd;
  Widget    dataLocalW[ 1 ];
  Widget    workFo;

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", startCalendarAlarmCB, NULL },
    { "", stopCalendarAlarmCB,  NULL },
    { "", closeCB,              NULL },
    { "", helpCB,               NULL },
  };


  /* Code. */

  action_buttons[ 0 ].label = msgGetText( MXDI_ON_LABEL );
  action_buttons[ 0 ].data  = appl_data_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_OFF_LABEL );
  action_buttons[ 1 ].data  = appl_data_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_CLOSE_BUTTON );
  action_buttons[ 2 ].data  = appl_data_ref;
  action_buttons[ 3 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 3 ].data  = appl_data_ref;

  /* Create a form dialog with buttons. */
  almProcFd = xitCreateFormDialog( parent, "AlmProcFd",
                                   0, 0,
                                   action_buttons, 
                                   XtNumber( action_buttons ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_ALARM_PRC_TITLE ) ); n++;
  XtSetValues( XtParent( almProcFd ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
  XtSetValues( almProcFd, args, n );


  /* Use this form. */
  workFo = XtNameToWidget( almProcFd, "AlmProcFdFo" );


  /* List with databases and their status. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  dbStatusLi = XmCreateScrolledList( workFo, "DbStatusLi", args, n );

  XtAddCallback( dbStatusLi, XmNsingleSelectionCallback,
                 (XtCallbackProc) dbStatusListCB, (XtPointer) appl_data_ref );
  XtAddCallback( dbStatusLi, XmNdefaultActionCallback,
                 (XtCallbackProc) dbStatusListCB, (XtPointer) appl_data_ref );


  /* Put the Parts together. */
  xitAttachWidget( XtParent( dbStatusLi ),
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );

  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetValues( XtParent( dbStatusLi ),  args, n );

  /* Manage the widgets. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* Set the size of the window. */
  xitSetSizeFormDialog( almProcFd, True );

  /* Make the final attachments. */
  n = 0;
  XtSetArg( args[ n ], XmNrightAttachment,  XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
  XtSetValues( XtParent( dbStatusLi ), args, n );


  /* Make sure our children don't spoil our size. */
  n = 0;
  XtSetArg( args[ n ], XmNallowShellResize, False ); n++;
  XtSetValues( XtParent( almProcFd ), args, n );

  XtManageChild( almProcFd );


  return( almProcFd );

} /* createWindow */


/*----------------------------------------------------------------------*/

static Boolean
  openCmdPipe( XTM_GL_BASE_DATA_REF  appl_data_ref )
{

  /* Variables. */
  int  fd;


  /* Code. */

  if( appl_data_ref -> alarm_fifo_name == NULL )
    return( False );

  /* Do we have a pipe for the alarm process communication? */
  if( appl_data_ref -> alarm_fifo_ref != NULL )
    return( True );


  fd = open( appl_data_ref -> alarm_fifo_name, (O_WRONLY | O_NONBLOCK) );
  if( fd < 0 )
    return( False );


  /* Open the FIFO stream. */
  appl_data_ref -> alarm_fifo_ref = fdopen( fd, "w" );

  if( appl_data_ref -> alarm_fifo_ref == NULL ) {
    xitMsgWinDisplayMessage( appl_data_ref -> msg_win_ref,
                             msgGetText( MXDI_CANNOT_OPEN_CMD_PIPE ), 0 );
    return( False );
  }


  return( True );

} /* openCmdPipe */


/*----------------------------------------------------------------------*/

static Boolean
  sendCmdPipe( XTM_GL_BASE_DATA_REF  appl_data_ref,
               Widget                controlW,
               char                  *command )
{

  /* Variables. */
  Boolean  done = False;
  int      status;
  int      wait_time = 0;
  Cursor   wait_cursor;
  Display  *display = NULL;
  Window   window = None;


  /* Code. */

  if( appl_data_ref -> alarm_fifo_ref == NULL )
    return( False );

  if( controlW != NULL ) {
    display = XtDisplay( controlW );
    window  = XtWindow(  controlW );
  }


  /* Waiting cursor. */
  if( controlW != NULL ) {
    wait_cursor = XCreateFontCursor( display, XC_watch );
    XDefineCursor( display, window, wait_cursor );

    XFlush( display );
  }


  do {

    status = fprintf( appl_data_ref -> alarm_fifo_ref, "%s\n", command );
    fflush( appl_data_ref -> alarm_fifo_ref );

    if( status >= 0 ) {
      done = True;
      break;
    }

    sleep( 2 );
    wait_time = wait_time + 2;

  } while( wait_time < 90 );

  if( controlW != NULL )
    XUndefineCursor( display, window );


  if( ! done ) {
    xitMsgWinDisplayMessage( appl_data_ref -> msg_win_ref,
                             msgGetText( MXDI_NO_CMD_TO_ALARM_PROC ), 0 );
    return( False );
  }


  return( True );

} /* sendCmdPipe */


/*----------------------------------------------------------------------*/

static void
  setProcessData( XTM_GL_BASE_DATA_REF  appl_data_ref )
{

  /* Variables. */
  Boolean                 running;
  int                     index;
  int                     index1;
  char                    buffer[ 50 ];
  char                    *char_ref;
  char                    *db_names;
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  tempW;
  XmString                list_items[ 100 ];
  XTM_GL_CUSTOM_DATA_REF  custom_data;


  /* Code. */

  custom_data = appl_data_ref -> custom_data;


  /* Fetch all databases. */
  (void) xtmCdFetchDbNames( custom_data -> cal_db_handle, &db_names );
  char_ref = db_names;
  index    = 0;

  do {

    int              char_read;
    char             db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
    XTM_CD_CAL_INFO  db_info;

    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    sscanf( char_ref, "%s%n", db_name, &char_read );
    char_ref = char_ref + char_read;

    /* Fetch information about the database. */
    (void) xtmCdFetchNamedDb( custom_data -> cal_db_handle, db_name,
                              &db_info, NULL );

    /* If we don't have read access, don't display the database. */
    if( flagIsClear( db_info.operations, XTM_DB_FLAG_MODE_READ ) )
      continue;

    /* Do we have an alarm process running for this DB? */
    running = xtmAcProcessRunning( appl_data_ref, db_info.short_name );

    if( running ) {
      sprintf( buffer, "%-15.15s %s", 
               db_info.short_name, msgGetText( MXDI_ON_LABEL ) );
    } else {
      sprintf( buffer, "%-15.15s %s", 
               db_info.short_name, msgGetText( MXDI_OFF_LABEL ) );
    }

    list_items[ index ] = XmStringCreate( buffer, CS );
    index++;

    if( index >= 100 )
      break;

  } while( True );

  if( db_names != NULL )
    SysFree( db_names );


  /* The list is always sorted. */
  xitSortStringList( list_items, index );

  tempW = XtNameToWidget( appl_data_ref -> process_info -> mainFo,
                          "AlmProcFdFo.DbStatusLiSW.DbStatusLi" );

  /* Assign the database locations to the list. */
  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, index ); n++;
  XtSetValues( tempW, args, n );

  /* Free allocated memory. */
  for( index1 = 0; index1 < index; index1++ )
    XmStringFree( list_items[ index1 ] );

  XmListDeselectAllItems( tempW );


  return;

} /* setProcessData */


/*----------------------------------------------------------------------*/

static void
  setActionButtons( XTM_GL_BASE_DATA_REF  appl_data_ref,
                    Boolean               start_prc,
                    Boolean               stop_prc )
{

  /* Variables. */
  Widget  baseW;
  Widget  tempW;


  /* Code. */

  baseW = appl_data_ref -> process_info -> mainFo;

  /* Start process? */
  tempW = XtNameToWidget( baseW, "Bu1" );
  XtSetSensitive( tempW, start_prc );

  /* Stop process? */
  tempW = XtNameToWidget( baseW, "Bu2" );
  XtSetSensitive( tempW, stop_prc );


  return;

} /* setActionButtons */


/*----------------------------------------------------------------------*/

static void
  setProcessState( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   char                  *db_name,
                   Boolean               active )
{

  /* Variables. */
  int                   index;
  XTM_GL_PROC_INFO_REF  process_info;


  /* Code. */

  process_info = appl_data_ref -> process_info;

  /* Is the given process running? */
  for( index = 0; index < XTM_GL_MAX_ALARM_PRC; index++ ) {

    /* Process has been started? */
    if( active && 
        strlen( process_info -> processes[ index ].db_name ) == 0 ) {
      strcpy( process_info -> processes[ index ].db_name, db_name );

      return;
    }

    /* Process has been stopped? */
    if( ! active && 
        strcmp( db_name, process_info -> processes[ index ].db_name ) == 0 ) {
      process_info -> processes[ index ].db_name[ 0 ] = '\0';

      return;
    }

  } /* loop */


  return;

} /* setProcessState */


/*----------------------------------------------------------------------*/

static Boolean
  startAlarmProcess( XTM_GL_BASE_DATA_REF  appl_data_ref,
                     Widget                controlW )
{

  /* Variables. */
  Boolean                 ok;
  int                     index;
  int                     n;
  int                     pid;
  char                    **process_args;
  Display                 *display = NULL;
  Window                  window = None;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = appl_data_ref -> custom_data;

  /* Is the alarm process running? */
  if( appl_data_ref -> process_info -> alarm_pid != 0 )
    return( True );


  if( controlW != NULL ) {
    display = XtDisplay( controlW );
    window  = XtWindow(  controlW );
  }


  /* Do we have a FIFO pipe? */
  if( appl_data_ref -> alarm_fifo_name == NULL ) {
    ok = createPipe( appl_data_ref );

    if( ! ok )
      return( False );
  }
  

  /* Start the alarm process. */
  process_args = (char**) XtMalloc( (custom_data_ref -> orig_argc + 10) *
                                    sizeof( char* ) );

  n = 0;  
  process_args[ n ] = custom_data_ref -> alarm_script; n++;
  process_args[ n ] = "-cmdPipeId"; n++;
  process_args[ n ] = appl_data_ref -> alarm_fifo_name; n++;

  for( index = 1; index < custom_data_ref -> orig_argc; index++ ) {
    process_args[ n ] = custom_data_ref -> orig_argv[ index ]; n++;
  }
    
  process_args[ n ] = NULL;

  pid = xtmToStartProcess( NULL, False, NULL, process_args );

  XtFree( (char*) process_args );

  if( pid == 0 )
    return( False );


  /* Open the pipe for the alarm process. */
  if( appl_data_ref -> alarm_fifo_ref == NULL ) { 

    Boolean  done      = False;
    int      wait_time = 0;
    Cursor   wait_cursor;

    /* Waiting cursor. */
    if( controlW != NULL ) {
      wait_cursor = XCreateFontCursor( display, XC_watch );
      XDefineCursor( display, window, wait_cursor );

      XFlush( display );
    }

    do {

      ok = openCmdPipe( appl_data_ref );
      if( ok ) {
        done = True;
        break;
      }

      sleep( 2 );
      wait_time = wait_time + 2;

    } while( wait_time < 90 );

    if( controlW != NULL )
      XUndefineCursor( display, window );


    if( ! done ) {
      xitMsgWinDisplayMessage( appl_data_ref -> msg_win_ref,
                               msgGetText( MXDI_CANNOT_START_ALARM_PROCESS ),
                               0 );
      return( False );
    }

  } /* if */


  /* Callback when the alarm process dies. */
  appl_data_ref -> process_info -> alarm_pid = pid;

  (void) SigRegister( SIGCHLD, pid, alarmProcessExitCB, appl_data_ref );


  return( True );

} /* startAlarmProcess */


/*----------------------------------------------------------------------*/

static Boolean
  startCalendarAlarm( XTM_GL_BASE_DATA_REF  appl_data_ref,
                      Widget                controlW,
                      char                  *db_name )
{

  /* Variables. */
  Boolean          ok;
  char             buffer[ 500 ];
  XTM_CD_CAL_INFO  db_info;


  /* Code. */

  /* Make sure the alarm process is running. */
  ok = startAlarmProcess( appl_data_ref, controlW );
  if( ! ok )
    return( False );


  /* We need to send the directory to. */
  (void) xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle,
                            db_name,
                            &db_info, NULL );


  /* Try to send command to the alarm process. */
  sprintf( buffer, "start: %s %s", db_info.short_name, db_info.directory );

  ok = sendCmdPipe( appl_data_ref, controlW, buffer );
  if( ! ok )
    return( False );


  /* Start alarms for the given database. */
  setProcessState( appl_data_ref, db_info.short_name, True );


  return( True );

} /* startCalendarAlarm */


/*----------------------------------------------------------------------*/

static Boolean
  stopCalendarAlarm( XTM_GL_BASE_DATA_REF  appl_data_ref,
                     Widget                controlW,
                     char                  *db_name )
{

  /* Variables. */
  Boolean  ok;
  char     buffer[ 200 ];


  /* Code. */

  /* Make sure the alarm process is running. */
  ok = startAlarmProcess( appl_data_ref, controlW );
  if( ! ok )
    return( False );


  /* Try to send command to the alarm process. */
  sprintf( buffer, "stop: %s", db_name );

  ok = sendCmdPipe( appl_data_ref, controlW, buffer );
  if( ! ok )
    return( False );

  /* Start alarms for the given database. */
  setProcessState( appl_data_ref, db_name, False );


  return( True );

} /* stopCalendarAlarm */


/*----------------------------------------------------------------------*/

static void
  alarmProcessExitCB( int   this_signal,
                      void  *user_data )
{

  /* Variables. */
  int                   index;
  XTM_GL_BASE_DATA_REF  appl_data_ref;
  XTM_GL_PROC_INFO_REF  process_info;


  /* Code. */

  appl_data_ref = (XTM_GL_BASE_DATA_REF) user_data;


  /* No alarm process. */
  process_info = appl_data_ref -> process_info;

  for( index = 0; index < XTM_GL_MAX_ALARM_PRC; index++ )
    process_info -> processes[ index ].db_name[ 0 ] = '\0';

  /* Clean up. */
  (void) unlink( appl_data_ref -> alarm_fifo_name );
  (void) fclose( appl_data_ref -> alarm_fifo_ref );

  appl_data_ref -> alarm_fifo_name = NULL;
  appl_data_ref -> alarm_fifo_ref = NULL;
  appl_data_ref -> process_info -> alarm_pid = 0;

  /* Tell that we lost the alarm process. */
  (void) XtAppAddTimeOut( appl_data_ref -> context, 0L,
                          (XtTimerCallbackProc) alarmProcessExitMsg,
                          (XtPointer) appl_data_ref );


  return;

} /* alarmProcessExitCB */


/*----------------------------------------------------------------------*/

static void 
  closeCB( Widget                widget,
           XTM_GL_BASE_DATA_REF  appl_data_ref,
           XtPointer             call_data )
{

  /* Variables. */
  XTM_GL_PROC_INFO_REF  process_info;


  /* Code. */

  process_info = appl_data_ref -> process_info;

  if( process_info -> mainFo == NULL )
    return;

  XtUnmanageChild( process_info -> mainFo );


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void 
  dbStatusListCB( Widget                widget, 
                  XTM_GL_BASE_DATA_REF  appl_data_ref,
                  XmListCallbackStruct  *call_data )
{

  /* Variables. */
  char  db_name[ 20 ];
  char  status[ 20 ];
  char  *char_ref;


  /* Code. */

  /* Fetch the selected DB. */
  char_ref = xitStringGetString( call_data -> item, CS );

  (void) sscanf( char_ref, "%s %s", db_name, status );
  SysFree( char_ref );

  if( strcmp( status, msgGetText( MXDI_OFF_LABEL ) ) != 0 )
    setActionButtons( appl_data_ref, False, True );
  else
    setActionButtons( appl_data_ref, True,  False );


  return;

} /* dbStatusListCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget                widget,
          XTM_GL_BASE_DATA_REF  appl_data_ref,
          XtPointer             call_data )
{

  /* Code. */

  xtmHlDisplayHelp( appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    alarm_window_id, "" );

  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void
  startCalendarAlarmCB( Widget                widget,
                        XTM_GL_BASE_DATA_REF  appl_data_ref,
                        XtPointer             call_data )
{

  /* Variables. */
  Boolean                 running;
  int                     selected_item_count;
  char                    db_name[ 25 ];
  char                    *char_ref;
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  baseW;
  Widget                  tempW;
  XmString                *selected_items;
  XTM_GL_CUSTOM_DATA_REF  custom_data;


  /* Code. */

  custom_data = appl_data_ref -> custom_data;
  baseW       = XtNameToWidget( appl_data_ref -> process_info -> mainFo,
                                "AlmProcFdFo.DbStatusLiSW" );
  tempW       = XtNameToWidget( baseW, "DbStatusLi" );


  /* Fetch selected item from the list. */
  n = 0;
  XtSetArg( args[ n ], XmNselectedItemCount, &selected_item_count ); n++;
  XtSetArg( args[ n ], XmNselectedItems,     &selected_items ); n++;
  XtGetValues( tempW, args, n );

  if( selected_item_count != 1 )
    return;  


  /* Fetch the selected DB. */
  char_ref = xitStringGetString( *selected_items, CS );

  (void) sscanf( char_ref, "%s", db_name );
  SysFree( char_ref );


  /* The alarm process already running? */
  running = xtmAcProcessRunning( appl_data_ref, db_name );
  if( running )
    return;


  /* Start alarms for this calendar. */
  startCalendarAlarm( appl_data_ref, 
                      appl_data_ref -> process_info -> mainFo, db_name );


  /* Assign values to the widgets. */
  setProcessData(   appl_data_ref );
  setActionButtons( appl_data_ref, False, False );


  return;

} /* startCalendarAlarmCB */


/*----------------------------------------------------------------------*/

static void
  stopCalendarAlarmCB( Widget                widget,
                       XTM_GL_BASE_DATA_REF  appl_data_ref,
                       XtPointer             call_data )
{

  /* Variables. */
  Boolean   running;
  int       selected_item_count;
  char      db_name[ 25 ];
  char      *char_ref;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    baseW;
  Widget    tempW;
  XmString  *selected_items;


  /* Code. */

  baseW = XtNameToWidget( appl_data_ref -> process_info -> mainFo,
                          "AlmProcFdFo.DbStatusLiSW" );

  tempW = XtNameToWidget( baseW, "DbStatusLi" );


  /* Fetch selected item from the list. */
  n = 0;
  XtSetArg( args[ n ], XmNselectedItemCount, &selected_item_count ); n++;
  XtGetValues( tempW, args, n );

  if( selected_item_count != 1 )
    return;  

  n = 0;
  XtSetArg( args[ n ], XmNselectedItems, &selected_items ); n++;
  XtGetValues( tempW, args, n );

  /* Fetch the selected DB. */
  char_ref = xitStringGetString( *selected_items, CS );

  (void) sscanf( char_ref, "%s", db_name );
  SysFree( char_ref );


  /* Is the alarm process running (I hope so...). */
  running = xtmAcProcessRunning( appl_data_ref, db_name );
  if( ! running )
    return;


  /* Stop the alarms for this calendar. */
  stopCalendarAlarm( appl_data_ref, 
                     appl_data_ref -> process_info -> mainFo, db_name );


  /* Assign values to the widgets. */
  setProcessData(   appl_data_ref );
  setActionButtons( appl_data_ref, False, False );


  return;

} /* stopCalendarAlarmCB */


