/*----------------------------------------------------------------------------
--
--  Module:           xitInfoPrint
--
--  Project:          xit - X Internal Toolkit
--  System:           <>
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Print a section from the info file.
--
--  Filename:         xitInfoPrint.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-10-17
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitInfoPrint.c, Version: 1.1, Date: 95/02/18 15:10:37";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <limits.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>

#include "System.h"
#include "Message.h"

#include "msgTopic.h"
#include "xitError.h"
#include "xitInfoFile.h"
#include "xitTools.h"
#include "xitInfoPrint.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the print window. */
#define outputLa           dataLocalW[  0 ]
#define outputRb           dataLocalW[  1 ]
#define outputRc           dataLocalW[  2 ]
#define prNameLa           dataLocalW[  3 ]
#define prNameTx           dataLocalW[  4 ]
#define prSelectRc         dataLocalW[  5 ]
#define printerLi          dataLocalW[  6 ]
#define whatLa             dataLocalW[  7 ]
#define whatRb             dataLocalW[  8 ]
#define whatRc             dataLocalW[  9 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Print info about. */
typedef enum {
  INFO_SECTION,
  INFO_CHAPTER,
  INFO_BOOK
} INFO_TYPE;

/* Output to what? */
typedef enum {
  TO_PRINTER,
  TO_FILE
} OUTPUT_TYPE;


/* Record while printing entries. */
typedef struct {

  /* Section to use. */
  int  section_no;

  /* Book to use. */
  char  book[ 50 ];

  /* Chapter to use. */
  char  chapter_id[ 50 ];

  /* Selected printer. */
  char  pr_name[ 50 ];

  /* File where the book is. */
  char  *book_file;

  /* The printer command to use. */
  char  *print_script;

  /* The system printers. */
  char  *system_pr_file;

  /* The user printers. */
  char  *user_pr_file;

  /* The printer window. */
  Widget  printerW;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XIT_IP_ACTION_CB  actionCB;

} PRINTER_REC, *PRINTER_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xitInfoPrint";

/* Name of text domain. */
static char  *td = "Topic";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  closeCB( Widget           widget,
           PRINTER_REC_REF  printer_ref,
           XtPointer        call_data );

static Boolean
  createPrintFile( PRINTER_REC_REF  printer_ref,
                   INFO_TYPE        print_info,
                   char             *file_name );

static Widget
  createPrintWindow( PRINTER_REC_REF  printer_ref,
                     Widget           parent );

static void 
  destroyCB( Widget           widget,
             PRINTER_REC_REF  printer_ref,
             XtPointer        call_data );

static void
  loadPrinterList( PRINTER_REC_REF  printer_ref );

static void
  printCB( Widget           widget,
           PRINTER_REC_REF  printer_ref,
           XtPointer        call_data );

static void 
  printerSelectCB( Widget                widget,
                   PRINTER_REC_REF       printer_ref,
                   XmListCallbackStruct  *call_data );

static void
  startPrintJob( PRINTER_REC_REF  printer_ref );

static char
  *stripSpaces( char  *buffer_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XIT_IP_HANDLE
  xitIpInitialize( Widget            parent,
                   char              *system_pr_file,
                   char              *user_pr_file,
                   char              *print_script,
                   XIT_IP_ACTION_CB  actionCB,
                   void              *user_data )
{

  /* Variables. */
  PRINTER_REC_REF  printer_ref;


  /* Code. */

  /* Create and initialize our private data. */
  printer_ref = SysNew( PRINTER_REC );
  if( printer_ref == NULL )
    return( NULL );

  printer_ref -> system_pr_file = SysNewString( system_pr_file );
  printer_ref -> user_pr_file   = SysNewString( user_pr_file );
  printer_ref -> print_script   = SysNewString( print_script );
  printer_ref -> book_file      = NULL;
  printer_ref -> actionCB       = actionCB;
  printer_ref -> user_data      = user_data;



  /* Create the printer window. */
  printer_ref -> printerW = createPrintWindow( printer_ref, parent );

  if( printer_ref -> printerW == NULL ) {
    SysFree( printer_ref );

    return( NULL );
  }


  /* Take the printers and write them into the list. */
  loadPrinterList( printer_ref );


  return( (XIT_IP_HANDLE) printer_ref );

} /* xitIpInitialize */


/*----------------------------------------------------------------------*/

void
  xitIpDestroy( XIT_IP_HANDLE  printer_handle )
{

  /* Variables. */
  PRINTER_REC_REF  printer_ref;


  /* Code. */

  if( printer_handle == NULL )
    return;

  /* Our private data. */
  printer_ref = (PRINTER_REC_REF) printer_handle;


  /* Destroy the window. */
  XtDestroyWidget( printer_ref -> printerW );


  return;

} /* xitIpDestroy */


/*----------------------------------------------------------------------*/

void 
  xitIpDisplayPrinter( XIT_IP_HANDLE  printer_handle,
                       char           *book,
                       char           *chapter_id,
                       int            section_no,
                       char           *book_file )
{

  /* Variables. */
  Widget           mainW;
  PRINTER_REC_REF  printer_ref;


  /* Code. */

  printer_ref = (PRINTER_REC_REF) printer_handle;
  if( printer_ref == NULL )
    return;

  mainW = XtNameToWidget( printer_ref -> printerW, "PrintFdFo" );


  /* Save the 'what to print' data. */
  strcpy( printer_ref -> book, book );
  strcpy( printer_ref -> chapter_id, chapter_id );

  printer_ref -> section_no = section_no;

  if( printer_ref -> book_file != NULL )
    SysFree( printer_ref -> book_file );
  printer_ref -> book_file = SysNewString( book_file );


  /* Make sure the window is visible. */
  XtManageChild( printer_ref -> printerW );


  return;

} /* xitIpDisplayPrinter */


/*----------------------------------------------------------------------*/

static Boolean
  createPrintFile( PRINTER_REC_REF  printer_ref,
                   INFO_TYPE        print_info,
                   char             *file_name )
{

  /* Variables. */
  int            index;
  char           buffer[ 500 ];
  char           *char_ref;
  FILE           *file_ref;
  XIT_IF_HANDLE  info_handle;
  XIT_IF_SECTION section_rec;
  XIT_IF_STATUS  status;


  /* Code. */

  /* Initalize the info file. */
  info_handle = xitIfInitialize( printer_ref -> book_file );
  if( info_handle == NULL )
    return( False );

  file_ref = fopen( file_name, "w" );
  if( file_ref == NULL )
    return( False );


  /* Print the current section? */
  if( print_info == INFO_SECTION ) {

    status = xitIfGetChapter( info_handle,
                              printer_ref -> chapter_id,
                              printer_ref -> section_no,
                              &section_rec );

    if( status == XIT_IF_OK && section_rec.link[ 0 ] == '\0' ) {

      /* Print and underline the chapter title. */
      char_ref = stripSpaces( section_rec.title );

      fprintf( file_ref, "\n%s\n", char_ref );

      for( index = 0; index < strlen( char_ref ); index++ )
        fprintf( file_ref, "-" );
      fprintf( file_ref, "\n" );

      /* Read the text lines. */
      status = xitIfGetNextTextLine( info_handle,
                                     buffer, sizeof( buffer ) );
      while( status == XIT_IF_OK ) {
        fprintf( file_ref, "%s\n", buffer );

        status = xitIfGetNextTextLine( info_handle,
                                       buffer, sizeof( buffer ) );
      }

     } /* if */

  } /* if */


  /* Print the current chapter? */
  if( print_info == INFO_CHAPTER ) {

    status = xitIfGetChapter( info_handle,
                              printer_ref -> chapter_id, 1,
                              &section_rec );

    while( status == XIT_IF_OK ) {

      if( section_rec.link[ 0 ] == '\0' ) {

        /* Print and underline the chapter title. */
        char_ref = stripSpaces( section_rec.title );

        fprintf( file_ref, "\n%s\n", char_ref );

        for( index = 0; index < strlen( char_ref ); index++ )
          fprintf( file_ref, "-" );
        fprintf( file_ref, "\n" );

        /* Read the text lines. */
        status = xitIfGetNextTextLine( info_handle,
                                       buffer, sizeof( buffer ) );
        while( status == XIT_IF_OK ) {
          fprintf( file_ref, "%s\n", buffer );

          status = xitIfGetNextTextLine( info_handle,
                                         buffer, sizeof( buffer ) );
        }

      } /* if */

      status = xitIfGetNextSection( info_handle,
                                    printer_ref -> chapter_id, &section_rec );
    } /* while */

  } /* if */


  /* Print the current book? */
  if( print_info == INFO_BOOK ) {

    status = xitIfGetChapter( info_handle, "*", 1,
                              &section_rec );

    while( status == XIT_IF_OK ) {

      if( section_rec.link[ 0 ] == '\0' ) {

        /* Print and underline the chapter title. */
        char_ref = stripSpaces( section_rec.title );

        fprintf( file_ref, "\n%s\n", char_ref );

        for( index = 0; index < strlen( char_ref ); index++ )
          fprintf( file_ref, "-" );
        fprintf( file_ref, "\n" );

        /* Read the text lines. */
        status = xitIfGetNextTextLine( info_handle,
                                       buffer, sizeof( buffer ) );
        while( status == XIT_IF_OK ) {
          fprintf( file_ref, "%s\n", buffer );

          status = xitIfGetNextTextLine( info_handle,
                                         buffer, sizeof( buffer ) );
        }

      } /* if */

      status = xitIfGetNextSection( info_handle, "*", &section_rec );

    } /* while */

  } /* if */


  /* Clean up. */
  xitIfDestroy( info_handle );
  fclose( file_ref );


  return( True );

} /* createPrintFile */


/*----------------------------------------------------------------------*/

static Widget
  createPrintWindow( PRINTER_REC_REF  printer_ref,
                     Widget           parent )
{

  /* Variables. */
  Arg       args[ 10 ];
  Cardinal  n;
  char      *char_ref;
  Widget    dataLocalW[ 10 ];
  Widget    outputTb[ 2 ];
  Widget    whatTb[ 3 ];
  Widget    printFd;
  Widget    workFo;

  static XIT_TEXT_STRUCT text_field_def[] = {
    { "PrNameTx", NULL, 1, True },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   printCB,  NULL },
    { NULL, NULL,     NULL },
    { NULL, NULL,     NULL },
    { NULL, NULL,     NULL },
    { "",   closeCB,  NULL },
  };


  /* Code. */

  /* Get text for menues and buttons. */
  action_buttons[ 0 ].label = msgDGetText( td, MINF_DO_PRINT_BUTTON );
  action_buttons[ 0 ].data  = printer_ref;
  action_buttons[ 4 ].label = msgDGetText( td, MINF_CLOSE_BUTTON );
  action_buttons[ 4 ].data  = printer_ref;

  /* Create a form dialog with buttons. */
  printFd = xitCreateFormDialog( parent, "PrintFd",
                                 1, 0,
                                 action_buttons,
                                 XtNumber( action_buttons ) );

  XtAddCallback( printFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) printer_ref );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgDGetText( td, MINF_PRINT_TITLE ) ); n++;
  XtSetValues( XtParent( printFd ), args, n );

  /* Close the window if this window is deleted. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( printFd ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( XtParent( printFd ), &wm_delete_window, 1 );
    XmAddWMProtocolCallback( XtParent( printFd ), wm_delete_window, 
                             (XtCallbackProc) closeCB,
                             (XtPointer) printer_ref );
  } /* block */


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( printFd, "PrintFdFo" );


  /* Create the list of printers. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT      ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC        ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5               ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5               ); n++;
  printerLi = XmCreateScrolledList( workFo, "PrinterLi", args, n );

  XtAddCallback( printerLi, XmNsingleSelectionCallback,
                 (XtCallbackProc) printerSelectCB, (XtPointer) printer_ref );


  /* Create a print select form. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNpacking,     XmPACK_TIGHT ); n++;
  prSelectRc = XmCreateRowColumn( workFo, "PrSelectRc", args, n );

  /* Name of printer. */
  prNameLa = xitCreateLabel( prSelectRc, "PrNameLa",
                             msgDGetText( td, MINF_PRINTER_NAME ), -1 );

  prNameTx = xitCreateTextCols( prSelectRc, &text_field_def[ 0 ], 10 );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, 50 );  n++;
  XtSetValues( prNameTx, args, n );  


  /* Create the paper format form. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  whatRc = XmCreateRowColumn( workFo, "WhatRc", args, n );

  whatLa = xitCreateLabel( whatRc, "WhatLa",
                           msgDGetText( td, MINF_PRINT_WHAT_LABEL ), -1 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNradioBehavior, True ); n++;
  whatRb = XmCreateRowColumn( whatRc, "WhatRb", args, n );

  whatTb[ 0 ] = xitCreateToggleButton(
                  whatRb, "What1Tb", 
                  msgDGetText( td, MINF_PRINT_SECTION ), True );

  whatTb[ 1 ] = xitCreateToggleButton(
                  whatRb, "What2Tb", 
                  msgDGetText( td, MINF_PRINT_CHAPTER ), False );

  whatTb[ 2 ] = xitCreateToggleButton(
                  whatRb, "What3Tb", 
                  msgDGetText( td, MINF_PRINT_BOOK ), False );


  /* Create the 'output to' form. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  outputRc = XmCreateRowColumn( workFo, "OutputRc", args, n );

  outputLa = xitCreateLabel( outputRc, "OutputLa",
                             msgDGetText( td, MINF_OUTPUT_TO_LABEL ), -1 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNradioBehavior, True ); n++;
  outputRb = XmCreateRowColumn( outputRc, "OutputRb", args, n );

  outputTb[ 0 ] = xitCreateToggleButton( 
                    outputRb, "Output1Tb", 
                    msgDGetText( td, MINF_OUTPUT_TO_PRINTER ), True );

  outputTb[ 1 ] = xitCreateToggleButton( 
                    outputRb, "Output2Tb", 
                    msgDGetText( td, MINF_OUTPUT_TO_FILE ), False );


  /* Put the Parts together. */
  xitAttachWidget( XtParent( printerLi ),
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( prSelectRc, 
                   XmATTACH_WIDGET, XtParent( printerLi ), 
                   XmATTACH_FORM,   NULL,
                   XmATTACH_NONE,   NULL,
                   XmATTACH_NONE,   NULL );
  xitAttachWidget( whatRc, 
                   XmATTACH_WIDGET, prSelectRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,       XmATTACH_NONE, NULL );
  xitAttachWidget( outputRc, 
                   XmATTACH_WIDGET, whatRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( XtParent( printerLi ), args, n );
  XtSetValues( prSelectRc,            args, n );
  XtSetValues( whatRc,                args, n );
  XtSetValues( outputRc,              args, n );


  /* Manage all the children. */
  XtManageChildren( outputTb, XtNumber( outputTb ) );
  XtManageChildren( whatTb,   XtNumber( whatTb ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set the size of the window. */
  xitSetSizeFormDialog( printFd, True );


  /* Make the final attachments. */
  xitAttachWidget( outputRc, 
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( whatRc, 
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, outputRc );
  xitAttachWidget( prSelectRc, 
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, whatRc );
  xitAttachWidget( XtParent( printerLi ),
                   XmATTACH_FORM, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM, NULL, XmATTACH_WIDGET, prSelectRc );

  /* Default printer and printer type? */
  char_ref = getenv( "PRINTER" );

  if( char_ref != NULL )
    XmTextSetString( prNameTx, char_ref );


  return( printFd );

} /* createPrintWindow */


/*----------------------------------------------------------------------*/

static void
  loadPrinterList( PRINTER_REC_REF  printer_ref )
{

  /* Variables. */
  int       index;
  int       items = 0;
  int       parsed_items;
  int       source_index = 0;
  char      buffer[ 100 ];
  char      delimiter[ 10 ];
  char      printer_desc[ 100 ];
  char      printer_flags[ 50 ];
  char      printer_name[ 50 ];
  char      printer_type[ 50 ];
  char      *char_ref;
  char      *printer_source[ 10 ];
  Arg       args[ 10 ];
  Cardinal  n;
  FILE      *file_ref;
  XmString  list_items[ 100 ];
  Widget    mainW;
  Widget    prListW;


  /* Code. */

  mainW   = XtNameToWidget( printer_ref -> printerW, "PrintFdFo" );
  prListW = XtNameToWidget( mainW, "PrinterLiSW.PrinterLi" );

  /* Whre do we find our printer definitions? */
  printer_source[ 0 ] = printer_ref -> user_pr_file;
  printer_source[ 1 ] = printer_ref -> system_pr_file;
  printer_source[ 2 ] = NULL;


  /* Process all the printer sources. */
  while( printer_source[ source_index ] != NULL ) {

    /* Try to open the file, if we fail, try the next. */
    file_ref = fopen( printer_source[ source_index ], "r" );
    if( file_ref == NULL ) {
      source_index++;
      continue;
    }

    /* Read all the printer definitions and build the printer list. */
    while( fgets( buffer, sizeof( buffer ), file_ref ) != NULL ) {

      /* Skip all leading blanks. */
      char_ref = buffer;
      while( isspace( *char_ref ) && *char_ref != '\0' )
        char_ref++;

      /* Is this a comment? */
      if( *char_ref == '#' || *char_ref == '!' )
        continue;

      /* Parse the printer definition. */
      printer_desc[ 0 ]  = '\0';
      printer_flags[ 0 ] = '\0';
      printer_name[ 0 ]  = '\0';
      printer_type[ 0 ]  = '\0';

      parsed_items = sscanf( char_ref, "%[^$]%c%[^$]%c%[^$]%c%[^\n]",
                             printer_name,  delimiter,
                             printer_type,  delimiter,
                             printer_desc,  delimiter,
                             printer_flags );
      if( parsed_items < 5 )
        continue;

      if( printer_desc[ strlen( printer_desc ) -1 ] == '\n' )
        printer_desc[ strlen( printer_desc ) - 1 ] = '\0';

      /* Create the entry in the printer list. */
      sprintf( buffer, "%-15.15s  %-5.5s  %-40.40s",
               printer_name, printer_type, printer_desc );

      /* Add the printer definition to the list. */
      list_items[ items ] = XmStringCreate( buffer, CS );
      items++;

    } /* while */

    fclose( file_ref );
    source_index++;

  } /* while */

  if( items < 1 )
    return;


  /* Assign the printers to the list. */
  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, items ); n++;
  XtSetValues( prListW, args, n );

  /* Free the allocated strings. */
  for( index = 0; index < items; index++ )
    XmStringFree( list_items[ index ] );


  return;

} /* loadPrinterList */


/*----------------------------------------------------------------------*/

static void
  startPrintJob( PRINTER_REC_REF  printer_ref )
{

  /* Variables. */
  Boolean      ok;
  int          status;
  char         file_name[ PATH_MAX ];
  char         *char_ref;
  Widget       mainW;
  Widget       tempW;
  INFO_TYPE    print_info = INFO_SECTION;
  OUTPUT_TYPE  output_to = TO_PRINTER;


  /* Code. */

  mainW = XtNameToWidget( printer_ref -> printerW, "PrintFdFo" );


  /* The file where the data to print is written. */
  char_ref = getenv( "HOME" );
  if( char_ref != NULL )
    sprintf( file_name, "%s/XInfo.out", char_ref );
  else
    sprintf( file_name, "./XInfo.out" );


  /* What do we do? */
  tempW = XtNameToWidget( mainW, "WhatRc.WhatRb.What1Tb" );
  if( XmToggleButtonGetState( tempW ) )
    print_info = INFO_SECTION;

  tempW = XtNameToWidget( mainW, "WhatRc.WhatRb.What2Tb" );
  if( XmToggleButtonGetState( tempW ) )
    print_info = INFO_CHAPTER;

  tempW = XtNameToWidget( mainW, "WhatRc.WhatRb.What3Tb" );
  if( XmToggleButtonGetState( tempW ) )
    print_info = INFO_BOOK;


  /* Print to file/printer? */
  tempW = XtNameToWidget( mainW, "OutputRc.OutputRb.Output1Tb" );
  if( XmToggleButtonGetState( tempW ) )
    output_to = TO_PRINTER;

  tempW = XtNameToWidget( mainW, "OutputRc.OutputRb.Output2Tb" );
  if( XmToggleButtonGetState( tempW ) )
    output_to = TO_FILE;


  /* Create the file to print. */
  ok = createPrintFile( printer_ref, print_info, file_name );

  if( ! ok ) {
    xitErMessage( printer_ref -> printerW, XIT_ER_ERROR, 
                  module_name, "okCB",
                  msgDGetText( td, MINF_CANNOT_CREATE_PRINT_FILE ) );
    return;
  }


  /* Send to printer? */
  if( output_to == TO_PRINTER ) {

    Cursor   wait_cursor;
    Display  *display = XtDisplay( printer_ref -> printerW );
    Window   window   = XtWindow(  printer_ref -> printerW );

    /* This might take time. */
    wait_cursor = XCreateFontCursor( display, XC_watch );
    XDefineCursor( display, window, wait_cursor );

    XFlush( display );

    char_ref = (char *) SysMalloc( strlen( printer_ref -> print_script ) +
                                   strlen( printer_ref -> pr_name ) +
                                   strlen( file_name ) +
                                   10 );

    sprintf( char_ref, "%s %s %s",
             printer_ref -> print_script, printer_ref -> pr_name,
             file_name );

    status = system( char_ref );

    SysFree( char_ref );

    XUndefineCursor( display, window );
  }


  /* Notify the user about the result. */
  char_ref = (char *) SysMalloc( strlen( file_name ) + 100 );

  if( output_to == TO_PRINTER )
    sprintf( char_ref, msgDGetText( td, MINF_RESULT_TO_PRINTER ),
             printer_ref -> pr_name );
  else
    sprintf( char_ref, msgDGetText( td, MINF_RESULT_TO_FILE ),
             file_name );

  tempW = xitCreateInformationDialog( 
            printer_ref -> printerW, "InformationDialog",
            msgDGetText( td, MINF_INFORMATION_LABEL ),
            char_ref,
            NULL, NULL );

  SysFree( char_ref );


  return;

} /* startPrintJob */


/*----------------------------------------------------------------------*/

static char
  *stripSpaces( char  *buffer_ref )
{

  /* Variables. */
  char  *char_ref;
  char  *start_ref;


  /* Code. */

  start_ref = buffer_ref;
  while( isspace( *start_ref ) )
    start_ref++;

  if( *start_ref == '\0' )
    return( start_ref );

  char_ref = buffer_ref + strlen( buffer_ref ) - 1;
  while( isspace( *char_ref ) && char_ref > start_ref )
    char_ref--;
  *(char_ref + 1) = '\0';

  return( start_ref );

} /* stripSpaces */


/*----------------------------------------------------------------------*/

static void
  closeCB( Widget           widget,
           PRINTER_REC_REF  printer_ref,
           XtPointer        call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( printer_ref -> actionCB != NULL )
    (* printer_ref -> actionCB)( XIT_IP_REASON_POPDOWN, 
                                 printer_ref -> user_data );

  XtUnmanageChild( printer_ref -> printerW );


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget           widget,
             PRINTER_REC_REF  printer_ref,
             XtPointer        call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( printer_ref -> actionCB != NULL )
    (* printer_ref -> actionCB)( XIT_IP_REASON_DESTROY, 
                                 printer_ref -> user_data );

  /* Free allocated data. */
  if( printer_ref -> system_pr_file != NULL )
    SysFree( printer_ref -> system_pr_file );

  if( printer_ref -> user_pr_file != NULL )
    SysFree( printer_ref -> user_pr_file );

  if( printer_ref -> print_script != NULL )
    SysFree( printer_ref -> print_script );

  SysFree( printer_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void
  printCB( Widget           widget,
           PRINTER_REC_REF  printer_ref,
           XtPointer        call_data )
{

  /* Variables. */
  Boolean  need_printer = True;
  char     *char_ref;
  Widget   mainW;
  Widget   tempW;


  /* Code. */

  mainW = XtNameToWidget( printer_ref -> printerW, "PrintFdFo" );


  /* Do we need a printer name? */
  tempW = XtNameToWidget( mainW, "OutputRc.OutputRb.Output2Tb" );
  if( XmToggleButtonGetState( tempW ) )
    need_printer = False;


  /* Printer name. */
  tempW = XtNameToWidget( mainW, "PrSelectRc.PrNameTx" );

  char_ref = xitStringGetText( tempW );
  if( char_ref == NULL )
    return;

  strcpy( printer_ref -> pr_name, char_ref );
  SysFree( char_ref );

  char_ref = printer_ref -> pr_name;
  while( isspace( *char_ref ) )
    char_ref++;

  if( *char_ref == '\0' && need_printer ) {
    xitErMessage( printer_ref -> printerW, XIT_ER_ERROR, 
                  module_name, "okCB",
                  msgDGetText( td, MINF_NO_PRINTER_GIVEN ) );
    return;
  } else if( *char_ref == '\0' ) {
    strcpy( printer_ref -> pr_name, "x" );
  }

  /* Start the print job. */
  startPrintJob( printer_ref );


  return;

} /* printCB */


/*----------------------------------------------------------------------*/

static void 
  printerSelectCB( Widget                widget,
                   PRINTER_REC_REF       printer_ref,
                   XmListCallbackStruct  *call_data )
{

  /* Variables. */
  char    printer_name[ 20 ];
  char    *char_ref;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( printer_ref -> printerW, "PrintFdFo" );


  /* Fetch the selected printer. */
  char_ref = xitStringGetString( call_data -> item, CS );

  sscanf( char_ref, "%[^ ]", printer_name );
  SysFree( char_ref );


  /* Set the printer name. */
  tempW = XtNameToWidget( mainW, "PrSelectRc.PrNameTx" );
  XmTextSetString( tempW, printer_name );


  return;

} /* printerSelectCB */
