/*----------------------------------------------------------------------------
--
--  Module:           xitInfoFile.c
--
--  Project:          xit   - X Internal Toolkit
--  System:           <>
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Interface to manage a structured info file.
--
--  Filename:         xitInfoFile.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-01-11
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitInfoFile.c, Version: 1.1, Date: 95/02/18 15:10:36";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <ctype.h>
#include <limits.h>
#include <malloc.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include "System.h"
#include "LstLinked.h"

#include "xitInfoFile.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#define max( a, b )       (a < b ? b : a )


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Source for text file, main or include file. */
typedef enum {
  SOURCE_MAIN,
  SOURCE_INCLUDE
} TEXT_SOURCE;


/* Info file object. */
typedef struct {

  /* The current section we are reading. */
  int  curr_section;

  /* The info file. */
  FILE  *file_ref;

  /* The include (text) file. */
  FILE  *include_file_ref;

  /* What are we doing? */
  XIT_IF_STATE  read_state;

} INFO_FILE, *INFO_FILE_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xitInfoFile";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  deleteTagCB( void  *element );

static void
  expandEscape( char  *string_ref );

static char
  *stripSpaces( char  *string_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/
int
  xitIfCurrentSection( XIT_IF_HANDLE  info_file_handle )
{

  /* Variables. */
  INFO_FILE_REF  info_file_ref;


  /* Code. */

  info_file_ref = (INFO_FILE_REF) info_file_handle;
  if( info_file_ref == NULL )
    return( 0 );


  return( info_file_ref -> curr_section );

} /* xitIfCurrentSection */


/*----------------------------------------------------------------------*/

void 
  xitIfDestroy( XIT_IF_HANDLE  info_file_handle )
{

  /* Variables. */
  INFO_FILE_REF  info_file_ref;


  /* Code. */

  info_file_ref = (INFO_FILE_REF) info_file_handle;
  if( info_file_ref == NULL )
    return;


  if( info_file_ref -> file_ref != NULL )
    fclose( info_file_ref -> file_ref );

  if( info_file_ref -> include_file_ref != NULL )
    fclose( info_file_ref -> file_ref );

  SysFree( info_file_ref );


  return;

} /* xitIfDestroy */


/*----------------------------------------------------------------------*/

void
  xitIfFreeMatchInfo( XIT_IF_SECTION_REF *match_info,
                      int                match_info_size )
{

  /* Variables. */
  int  index;


  /* Code. */

  for( index = 0; index < match_info_size; index++ )
    SysFree( *(match_info + index) );


  return;

} /* xitIfFreeMatchInfo */


/*----------------------------------------------------------------------*/

void
  xitIfFreeTagsInfo( LST_DESC_TYPE  tag_list )
{

  /* Code. */

  LstLinkClearDataAndList( tag_list, deleteTagCB );


  return;

} /* xitIfFreeTagsInfo */


/*----------------------------------------------------------------------*/

XIT_IF_STATUS 
  xitIfGetChapter( XIT_IF_HANDLE       info_file_handle,
                   char                *chapter_id,
                   int                 section_number,
                   XIT_IF_SECTION_REF  section_ref )
{

  /* Variables. */
  int            index;
  INFO_FILE_REF  info_file_ref;
  XIT_IF_STATUS  status;


  /* Code. */

  info_file_ref = (INFO_FILE_REF) info_file_handle;
  if( info_file_ref == NULL )
    return( XIT_IF_ERROR );


  xitIfResetFile( info_file_handle );

  info_file_ref -> curr_section = 0;

  index = 1;
  status = xitIfGetNextSection( info_file_handle, chapter_id,
                                section_ref );

  /* All chapters? */
  if( strcmp( chapter_id, "*" ) == 0 )
    return( XIT_IF_OK );


  /* Search the file. */
  while( index != section_number && status == XIT_IF_OK ) {

    status = xitIfGetNextSection( info_file_handle, chapter_id,
                                  section_ref );
    index++;

  } /* while */

  /* Did we find it? */
  if( status != XIT_IF_OK )
    return( XIT_IF_ERROR );


  return( XIT_IF_OK );

} /* xitIfGetChapter */


/*----------------------------------------------------------------------*/

XIT_IF_STATUS 
  xitIfGetNextSection( XIT_IF_HANDLE       info_file_handle,
                       char                *chapter_id,
                       XIT_IF_SECTION_REF  section_ref )
{

  /* Variables. */
  Boolean        done = False;
  int            items;
  int            toc_level;
  long           file_pos;
  char           buffer[ 500 ];
  char           curr_chapter_id[ 50 ];
  char           book[ 50 ];
  char           link[ 50 ];
  char           title[ 200 ];
  char           *char_ref;
  INFO_FILE_REF  info_file_ref;


  /* Code. */

  info_file_ref = (INFO_FILE_REF) info_file_handle;
  if( info_file_ref == NULL )
    return( XIT_IF_ERROR );

  if( info_file_ref -> file_ref == NULL )
    return( XIT_IF_ERROR );

  if( info_file_ref -> read_state != XIT_IF_RESET &&
      info_file_ref -> read_state != XIT_IF_HEADER &&
      info_file_ref -> read_state != XIT_IF_TEXT )
    return( XIT_IF_ERROR );


  curr_chapter_id[ 0 ] = '\0';
  book[ 0 ]            = '\0';
  link[ 0 ]            = '\0';
  title[ 0 ]           = '\0';
 

  /* Search the file for the requested section. */
  while( ! done ) {

    char_ref = fgets( buffer, sizeof( buffer ), info_file_ref -> file_ref );
    if( char_ref == NULL )
      return( XIT_IF_ERROR );

    /* Is this a header item? */
    if( strlen( buffer ) < 2 || strncmp( buffer, "#H", 2 ) != 0 )
      continue;


    /* Read and parse the level, subject and header level. */
    toc_level = 1;

    items = sscanf( buffer, "#H <%[^>]> '%[^']' <%d>", 
                    curr_chapter_id, title, &toc_level );
    if( items < 2 )
      return( XIT_IF_ERROR );

    if( strcmp( chapter_id, "*" ) != 0 &&
        strlen( chapter_id ) > 0 && 
        strcmp( chapter_id, curr_chapter_id ) != 0 )
      continue;


    /* Save the current position. */
    file_pos = ftell( info_file_ref -> file_ref );

    /* Get the next line and look for a link. */
    char_ref = fgets( buffer, sizeof( buffer ), info_file_ref -> file_ref );

    if( strlen( buffer ) > 2 && strncmp( buffer, "#L", 2 ) == 0 ) {
      if( strchr( buffer, ':' ) != NULL ) {
        items = sscanf( buffer, "#L <%[^:]:%[^>]>", book, link );

        if( items != 2 )
          return( XIT_IF_ERROR );

      } else {
        items = sscanf( buffer, "#L <%[^>]>", link );

        if( items != 1 )
          return( XIT_IF_ERROR );
      }

    } else {
      fseek( info_file_ref -> file_ref, file_pos, 0 );
    }

    done = True;

  } /* while */

  /* Save the found data. */
  strcpy( section_ref -> chapter_id, curr_chapter_id );
  strcpy( section_ref -> title,      title );
  strcpy( section_ref -> book,       book );
  strcpy( section_ref -> link,       link );

  section_ref -> toc_level = toc_level;

  info_file_ref -> read_state   = XIT_IF_HEADER;
  info_file_ref -> curr_section++;


  return( XIT_IF_OK );

} /* xitIfGetNextSection */


/*----------------------------------------------------------------------*/

XIT_IF_STATUS 
  xitIfGetNextTextLine( XIT_IF_HANDLE  info_file_handle,
                        char           *buffer,
                        int            buffer_size )
{

  /* Variables. */
  Boolean        done;
  int            items;
  long           file_pos;
  char           dummy[ 200 ];
  char           filename[ PATH_MAX + 1 ];
  char           *char_ref;
  FILE           *file_ref;
  INFO_FILE_REF  info_file_ref;
  TEXT_SOURCE    text_source;


  /* Code. */

  info_file_ref = (INFO_FILE_REF) info_file_handle;
  if( info_file_ref == NULL )
    return( XIT_IF_ERROR );

  if( info_file_ref -> file_ref == NULL )
    return( XIT_IF_ERROR );


  /* Correct state? */
  if( info_file_ref -> read_state != XIT_IF_HEADER &&
      info_file_ref -> read_state != XIT_IF_TEXT )
    return( XIT_IF_ERROR );


  /* Are we reading for the main file or from an include file? */
  if( info_file_ref -> include_file_ref != NULL ) {
    file_ref    = info_file_ref -> include_file_ref;
    text_source = SOURCE_INCLUDE;
  } else {
    file_ref    = info_file_ref -> file_ref;
    text_source = SOURCE_MAIN;
  }


  /* Save the current position. */
  file_pos = ftell( file_ref );


  /* Read the text line, strip comments. */
  done = False;

  while( ! done ) {

    char_ref = fgets( buffer, buffer_size, file_ref );

    /* When end of file in an include file, continue in the main file. */
    if( char_ref == NULL && text_source == SOURCE_INCLUDE ) {

      fclose( info_file_ref -> include_file_ref );

      file_ref    = info_file_ref -> file_ref;
      file_pos    = ftell( file_ref );
      text_source = SOURCE_MAIN;

      /* Fetch the next text line from the main buffer. */
      continue;

    } else if( char_ref == NULL && text_source == SOURCE_MAIN ) {

      return( XIT_IF_ERROR );

    } /* if */


    /* Is the text line an include file? */
    if( strncmp( buffer, "#@", 2 ) == 0 && text_source == SOURCE_MAIN ) {

      items = sscanf( &buffer[ 2 ], "%[^']'%[^']'", dummy, filename );
      if( items != 2 ) {
        fprintf( stderr, "%s: No include file given\n", module_name );

        continue;
      }

      /* Try to open the include file. */      
      info_file_ref -> include_file_ref = fopen( filename, "r" );

      if( info_file_ref -> include_file_ref == NULL ) {
        fprintf( stderr, "%s: Cannot find include file %s\n",
                 module_name, filename );

        continue;
      }

      file_ref    = info_file_ref -> include_file_ref;
      file_pos    = ftell( file_ref );
      text_source = SOURCE_INCLUDE;

      continue;

    } /* if */


    /* Is this a header tag? */
    if( strncmp( buffer, "#H", 2 ) == 0 && text_source == SOURCE_MAIN ) {
      fseek( file_ref, file_pos, 0 );
      return( XIT_IF_ERROR );
    }


    /* If the line is not a tag, we have found our line. */
    if( buffer[ 0 ] != '#' )
      done = True;

  } /* while */

  *(buffer + strlen( buffer ) - 1) = '\0';

  info_file_ref -> read_state = XIT_IF_TEXT;


  return( XIT_IF_OK );

} /* xitIfGetNextTextLine */


/*----------------------------------------------------------------------*/

XIT_IF_STATUS 
  xitIfGetTags( XIT_IF_HANDLE  info_file_handle,
                LST_DESC_TYPE  *tag_list )
{

  /* Variables. */
  Boolean          save_tag;
  int              items;
  long             file_pos;
  char             buffer[ 500 ];
  char             dummy[ 200 ];
  char             par1[ 200 ];
  char             par2[ 200 ];
  char             par3[ 200 ];
  char             par4[ 200 ];
  char             *char_ref;
  INFO_FILE_REF    info_file_ref;
  XIT_IF_TAG_INFO  tag_info;


  /* Code. */

  *tag_list = NULL;

  info_file_ref = (INFO_FILE_REF) info_file_handle;
  if( info_file_ref == NULL )
    return( XIT_IF_ERROR );

  if( info_file_ref -> file_ref == NULL )
    return( XIT_IF_ERROR );

  if( info_file_ref -> read_state != XIT_IF_HEADER &&
      info_file_ref -> read_state != XIT_IF_TEXT )
    return( XIT_IF_ERROR );

  /* Save the current position. */
  file_pos = ftell( info_file_ref -> file_ref );


  /* Create a new tag list. */
  *tag_list = LstLinkNew( sizeof( XIT_IF_TAG_INFO ), NULL );


  /* Search until the next header for tags. */
  do {

    char_ref = fgets( buffer, sizeof( buffer ), info_file_ref -> file_ref );
    if( char_ref == NULL ) {
      fseek( info_file_ref -> file_ref, file_pos, 0 );
      return( XIT_IF_OK );
    }

    /* Is this a header item? */
    if( strlen( buffer ) >= 2 && strncmp( buffer, "#H", 2 ) == 0 ) {
      fseek( info_file_ref -> file_ref, file_pos, 0 );
      return( XIT_IF_OK );
    }

    /* Is this a tag? */
    if( strlen( buffer ) < 2 || buffer[ 0 ] != '#' )
      continue;

    par1[ 0 ] = '\0';
    par2[ 0 ] = '\0';
    par3[ 0 ] = '\0';
    par4[ 0 ] = '\0';

    save_tag = False;

    /* The different tags we know of. */
    switch( buffer[ 1 ] ) {

      /* Link. */
      case 'L':
        break;

      /* Start external action. */
      case 'A':
        items = sscanf( &buffer[ 2 ], "%[^']'%[^']'%[^']'%[^']'%[^']'%[^']'",
                        dummy, par1, dummy, par2, dummy, par3 );
        if( items == 6 ) {
          tag_info.tag_type = XIT_IF_TAG_ACTION_EXECUTE;

          expandEscape( par1 );
        }
        save_tag = True;
        break;

      /* Jump action. */
      case 'J':
        items = sscanf( &buffer[ 2 ], "%[^']'%[^']'%[^']'%[^']'%[^<]<%[^>]>",
                        dummy, par1, dummy, par2, dummy, par3 );
        if( items == 6 ) {
          tag_info.tag_type = XIT_IF_TAG_ACTION_JUMP;

          expandEscape( par1 );
        }
        save_tag = True;
        break;

      /* Include. */
      case '@':
        break;

    } /* switch */

    /* Save the tag? */
    if( save_tag ) {

      char_ref = stripSpaces( par1 );
      if( strlen( char_ref ) > 0 )
        tag_info.param1 = SysNewString( char_ref );
      else
        tag_info.param1 = NULL;

      char_ref = stripSpaces( par2 );
      if( strlen( char_ref ) > 0 )
        tag_info.param2 = SysNewString( char_ref );
      else
        tag_info.param2 = NULL;

      char_ref = stripSpaces( par3 );
      if( strlen( char_ref ) > 0 )
        tag_info.param3 = SysNewString( char_ref );
      else
        tag_info.param3 = NULL;

      char_ref = stripSpaces( par4 );
      if( strlen( char_ref ) > 0 )
        tag_info.param4 = SysNewString( char_ref );
      else
        tag_info.param4 = NULL;

      (void ) LstLinkInsertLast( *tag_list, &tag_info );

    } /* if */

  } while( True );


} /* xitIfGetTags */


/*----------------------------------------------------------------------*/

XIT_IF_STATUS
  xitIfGetToc( XIT_IF_HANDLE       info_file_handle,
               int                 max_toc_entries,
               XIT_IF_SECTION_REF  *toc_entries,
               int                 *no_toc_entries )
{

  /* Variables. */
  int                 entries = 0;
  XIT_IF_SECTION      section_rec;
  XIT_IF_SECTION_REF  toc_record_ref;
  XIT_IF_STATUS       status;


  /* Code. */

  if( info_file_handle == NULL )
    return( XIT_IF_ERROR );

  *no_toc_entries = 0;


  /* Start from scratch. */
  xitIfResetFile( info_file_handle );


  /* Fetch all the item headers in the help file. */
  status = xitIfGetNextSection( info_file_handle, "*", &section_rec );

  while( status == XIT_IF_OK ) {

    /* To many entries? */
    if( entries >= max_toc_entries ) {
      *no_toc_entries = entries;

      return( XIT_IF_OK );
    }

    /* Save the toc record (no link information). */
    if( strlen( section_rec.link ) == 0 ) {

      toc_record_ref = SysNew( XIT_IF_SECTION );

      strcpy( toc_record_ref -> chapter_id, section_rec.chapter_id );
      strcpy( toc_record_ref -> title,      section_rec.title );
      strcpy( toc_record_ref -> book,       section_rec.book ); 
      strcpy( toc_record_ref -> link,       section_rec.link ); 

      toc_record_ref -> toc_level = section_rec.toc_level;

      *(toc_entries + entries) = toc_record_ref;

      entries++;

    } /* if */

    status = xitIfGetNextSection( info_file_handle, "*", &section_rec );

  } /* while */

  *no_toc_entries = entries;


  return( XIT_IF_OK );

} /* xitIfGetToc */


/*----------------------------------------------------------------------*/

XIT_IF_HANDLE
  xitIfInitialize( char  *filename )
{

  /* Variables. */
  int            index;
  int            last_time;
  int            status;
  int            valid_version;
  char           *version_file;
  INFO_FILE_REF  info_file_ref;
  struct stat    file_info;


  /* Code. */

  /* Create a new info file object. */
  info_file_ref = SysNew( INFO_FILE );
  if( info_file_ref == NULL )
    return( NULL );


  /* Initialization. */
  info_file_ref -> include_file_ref = NULL;


  /* Try to open the topics file as is. */
  info_file_ref -> file_ref = fopen( filename, "r" );

  if( info_file_ref -> file_ref != NULL )
    return( (XIT_IF_HANDLE) info_file_ref );


  /* If the info file is not found, try with versions 1-5. */
  last_time     = 0;
  valid_version = 0;

  version_file = SysMalloc( strlen( filename ) + 10 );

  /* Find the latest version. */
  for( index = 1; index <= 5; index++ ) {
    sprintf( version_file, "%s_%d", filename, index );

    status = stat( version_file, &file_info );
    if( status == 0 && file_info.st_mtime > last_time ) {
      valid_version = index;
      last_time     = file_info.st_mtime;
    }
  } /* loop */


  /* Did we find a valid version? */
  if( valid_version == 0 ) {
    SysFree( version_file );
    SysFree( info_file_ref );

    return( NULL );
  }


  /* Open the info database file. */
  sprintf( version_file, "%s_%d", filename, valid_version );

  info_file_ref -> file_ref = fopen( version_file, "r" );

  if( info_file_ref -> file_ref == NULL ) {
    SysFree( version_file );
    SysFree( info_file_ref );

    return( NULL );
  }

  SysFree( version_file );


  return( (XIT_IF_HANDLE) info_file_ref );

} /* xitIfInitialize */


/*----------------------------------------------------------------------*/

XIT_IF_STATUS
  xitIfResetFile( XIT_IF_HANDLE  info_file_handle )
{

  /* Variables. */
  INFO_FILE_REF  info_file_ref;


  /* Code. */

  info_file_ref = (INFO_FILE_REF) info_file_handle;
  if( info_file_ref == NULL )
    return( XIT_IF_ERROR );


  if( info_file_ref -> file_ref == NULL )
    return( XIT_IF_ERROR );

  rewind( info_file_ref -> file_ref );


  /* If we have an include file open, close it. */
  if( info_file_ref -> include_file_ref != NULL )
    fclose( info_file_ref -> include_file_ref );

  info_file_ref -> read_state = XIT_IF_RESET;


  return( XIT_IF_OK );

} /* xitIfResetFile */


/*----------------------------------------------------------------------*/

XIT_IF_STATUS
  xitIfSearchString( XIT_IF_HANDLE          info_file_handle,
                     XIT_IF_SEARCH_PROFILE  *search,
                     int                    max_match_info,
                     XIT_IF_SECTION_REF     *match_info,
                     int                    *no_match_info )
{

  /* Variables. */
  Boolean             match;
  int                 chapter_size;
  int                 matches = 0;
  int                 segment_size = 10000;
  char                buffer[ 200 ];
  char                *chapter_text;
  char                *char_ref;
  XIT_IF_SECTION      section_rec;
  XIT_IF_SECTION_REF  match_record_ref;
  XIT_IF_STATUS       status;


  /* Code. */

  if( info_file_handle == NULL )
    return( XIT_IF_ERROR );


  *no_match_info = 0;


  /* Do we have anything to search for? */
  if( !(search -> headers || search -> text) )
    return( XIT_IF_OK );

  /* Case sensitive search? */
  if( ! search -> case_sensitive ) {
    char_ref = search -> string;

    while( *char_ref != '\0' ) {
      *char_ref = toupper( *char_ref );
      char_ref++;
    }
  }


  /* Search form the start. */
  xitIfResetFile( info_file_handle );

  /* Fetch all the item headers in the help file. */
  status = xitIfGetNextSection( info_file_handle, "", &section_rec );

  while( status == XIT_IF_OK ) {

    /* If text lines are following, read them into a buffer. */
    if( strlen( section_rec.link ) == 0 ) {

      chapter_text = SysMalloc( segment_size );
      chapter_size = segment_size;

      *chapter_text = '\0';
      match         = False;

      /* Search in headers? */
      if( ! match && search -> headers ) {

        strcpy( chapter_text, section_rec.title );

        /* Case sensitive search? */
        if( ! search -> case_sensitive ) {
          char_ref = chapter_text;

          while( *char_ref != '\0' ) {
            *char_ref = toupper( *char_ref );
            char_ref++;
          }
        }

        /* Now when we have the text string, search. */
        if( strstr( chapter_text, search -> string ) != NULL )
          match = True;

      } /* if */


      /* Search in text? */
      if( ! match && search -> text ) {

        status = xitIfGetNextTextLine( info_file_handle, 
                                       buffer, sizeof( buffer ) );
        while( status == XIT_IF_OK ) {

          /* Do we need to expand the buffer? */
          if( strlen( chapter_text ) + strlen( buffer ) + 2 > chapter_size ) {

            int  extend_size;

            extend_size  = max( strlen( buffer ), segment_size ) + 10;
            chapter_size = chapter_size + extend_size;

            chapter_text = SysRealloc( chapter_text, chapter_size );

          } /* if */

          /* Remove any trailing \n and add the buffer to the chapter text. */
          strtok( buffer, "\n" );

          strcat( chapter_text, buffer );
          strcat( chapter_text, " " );

          /* Fetch the next text line. */
          status = xitIfGetNextTextLine( info_file_handle, 
                                         buffer, sizeof( buffer ) );

        } /* while */

        /* Case sensitive search? */
        if( ! search -> case_sensitive ) {
          char_ref = chapter_text;

          while( *char_ref != '\0' ) {
            *char_ref = toupper( *char_ref );
            char_ref++;
          }
        }

        /* Now when we have the text string, search. */
        if( strstr( chapter_text, search -> string ) != NULL )
          match = True;

      } /* if */


      /* Relaese the memory occupied. */
      SysFree( chapter_text );

      /* If we had a match, save the item, header and link information. */
      if( match ) {

        /* To many matches? */
        if( matches >= max_match_info ) {
          *no_match_info = matches;

          return( XIT_IF_OK );
        }

        /* Save the match record. */
        match_record_ref = SysNew( XIT_IF_SECTION );

        strcpy( match_record_ref -> chapter_id, section_rec.chapter_id );
        strcpy( match_record_ref -> title,      section_rec.title );
        strcpy( match_record_ref -> book,       section_rec.book ); 
        strcpy( match_record_ref -> link,       section_rec.link ); 

        match_record_ref -> toc_level = section_rec.toc_level;

        *(match_info + matches) = match_record_ref;

        matches++;

      } /* if */

    } /* if */

    status = xitIfGetNextSection( info_file_handle, "", &section_rec );

  } /* while */

  *no_match_info = matches;


  return( XIT_IF_OK );

} /* xitIfSearchString */


/*----------------------------------------------------------------------*/

static void
  expandEscape( char  *string_ref )
{

  /* Variables. */
  char  *char_ref;
  char  *save_char_ref;


  /* Code. */

  char_ref      = string_ref;
  save_char_ref = string_ref;

  while( *char_ref != '\0' ) {

    if( *char_ref == '\\' ) {
      switch( *(char_ref + 1) ) {
        case 'n':
          *save_char_ref = '\n';
          break;
        case '\0':
          *save_char_ref = *char_ref;
          return;
        default:
          *save_char_ref = *(char_ref + 1 );
          break;
      }
      char_ref = char_ref + 2;

    } else {
      *save_char_ref = *char_ref;
      char_ref = char_ref + 1;
    }

    save_char_ref++;

  } /* while */

  *save_char_ref = '\0';


  return;

} /* expandEscape */


/*----------------------------------------------------------------------*/

static char
  *stripSpaces( char  *string_ref )
{

  /* Variables. */
  char  *char_ref;


  /* Code. */

  char_ref = string_ref;

  while( isspace( *char_ref ) )
    char_ref++;


  return( char_ref );

} /* stripSpaces */


/*----------------------------------------------------------------------*/

static void
  deleteTagCB( void  *element )
{

  /* Variables. */
  XIT_IF_TAG_INFO_REF  tag_ref;


  /* Code. */

  tag_ref = (XIT_IF_TAG_INFO_REF) element;

  if( tag_ref -> param1 != NULL )
    SysFree( tag_ref -> param1 );

  if( tag_ref -> param2 != NULL )
    SysFree( tag_ref -> param2 );

  if( tag_ref -> param3 != NULL )
    SysFree( tag_ref -> param3 );

  if( tag_ref -> param4 != NULL )
    SysFree( tag_ref -> param4 );


  return;

} /* deleteTagCB */
