/*----------------------------------------------------------------------------
--
--  Module:           xitClipboard
--
--  Project:          xit  - X Internal Toolkit
--  System:           xit  - X Internal Toolkit
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Cut, copy and paste to/from various clipboards. The supported
--    clipboards are XA_PRIMARY, XA_SECONDARY, XA_CLIPBOARD and
--    XA_CUT_BUFFER0.
--
--  Filename:         xitClipboard.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-05-05
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitClipboard.c, Version: 1.1, Date: 95/02/18 15:10:20";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <malloc.h>
#include <memory.h>
#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/Xatom.h>
#include <X11/StringDefs.h>

#include <Xm/Xm.h>
#include <Xm/RowColumn.h>

#include "System.h"

#include "xitTools.h"
#include "xitClipboard.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  actionCB( Widget              widget,
            XIT_CL_RECORD_REF   record,
            XmAnyCallbackStruct *call_data );

static void
  insertSelectionCB( Widget              widget,
                     XIT_CL_RECORD_REF   record,
                     Atom                *selection,
                     Atom                *type,
                     XtPointer           value,
                     unsigned long       *length,
                     int                 *format );


/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XIT_CL_RECORD_REF
  xitClInitialize( void )
{

  /* Variables. */
  XIT_CL_RECORD_REF   record;


  /* Code. */

  /* Allocate space for the record. */
  record = SysNew( XIT_CL_RECORD );
  if( record == NULL )
    return( NULL );

  /* Make sure the record is initalized to NULL. */
  memset( (char *)record, 0, sizeof( XIT_CL_RECORD ) );

  return( record );

} /* xitClInitialize */


/*----------------------------------------------------------------------*/

void
  xitClDestroy( XIT_CL_RECORD_REF  *record )
{

  /* Code. */

  if( record == NULL || *record == NULL )
    return;

  if( (*record) -> popup != NULL )
    XtDestroyWidget( (*record) -> popup );

  SysFree( *record );

  *record = NULL;

  return;

} /* xitClDestroy */


/*----------------------------------------------------------------------*/

void
  xitClCreateMenu( Widget             parent,
                   XIT_CL_RECORD_REF  record )
{

  /* Variables. */
  int     index;
  int     index1;
  Widget  nextMenu;
  Widget  tempWidget;

  static XIT_CASCADE_STRUCT selection_casc[] = {
    { "", "", "SelectionCasc" },
  };

  static XIT_MENU_BUTTON_STRUCT action_buttons[] = {
    { "", "", NULL, "ActionButtons", True, False, False },
  };


  /* Code. */

  record -> popup = XmCreatePopupMenu( parent, "ClPopup", NULL, 0 );

  /* Create the selections and attach the actions to each selection. */
  for( index = 0; index < XtNumber( record -> selection ); index++ ) {
    if( record -> selection_text[ index ] != NULL ) {

      /* Create the sub menu. */
      nextMenu = XmCreatePulldownMenu( record -> popup, "NextMenu", NULL, 0 );

      for( index1 = 0; index1 < XtNumber( record -> action_text ); index1++ ) {
        if( record -> action_text[ index1 ] != NULL ) {
          action_buttons[ 0 ].title = record -> action_text[ index1 ];

          tempWidget = xitCreateMenuPushButton( nextMenu, 
					        &action_buttons[ 0 ] );

          XtAddCallback( tempWidget, XmNactivateCallback,
                         (XtCallbackProc) actionCB, (XtPointer) record );
          XtManageChild( tempWidget );

          record -> selection[ index ].actionButton[ index1 ] = tempWidget;

          /* We cannot do everything yet. */
          if( index1 == XIT_CL_CUT || index1 == XIT_CL_COPY ) {
            Arg       args[ 3 ];
            Cardinal  n;

            n = 0;
            XtSetArg( args[ n ], XmNsensitive, False ); n++;
            XtSetValues( tempWidget, args, n );
          }

        } /* if */
      } /* loop */

      /* Create the selection cascade. */
      selection_casc[ 0 ].title = record -> selection_text[ index ];

      record -> selection[ index ].button = 
        xitCreateCascadeButton( record -> popup, nextMenu, 
                                &selection_casc[ 0 ] );

      XtManageChild( record -> selection[ index ].button );

      /* We cannot do everything yet. */
      if( index == XIT_CL_CLIPBOARD ) {
        Arg       args[ 3 ];
        Cardinal  n;

        n = 0;
        XtSetArg( args[ n ], XmNsensitive, False ); n++;
        XtSetValues( record -> selection[ index ].button, args, n );
      }

    } /* if */

  } /* loop */

  return;

} /* xitClCreateMenu */


/*----------------------------------------------------------------------*/

void
  xitClPostMenu( XIT_CL_RECORD_REF  record,
                 void               *client_data,
                 XEvent             *event )
{

  /* Code. */

  if( record == NULL || event == NULL )
    return;

  if( record -> popup == NULL )
    return;

  record -> client_data = client_data;

  XmMenuPosition( record -> popup, (XButtonPressedEvent *) event );
  XtManageChild(  record -> popup );
  
  return;

} /* xitClPostMenu */


/*----------------------------------------------------------------------*/

static void 
  actionCB( Widget              widget,
            XIT_CL_RECORD_REF   record,
            XmAnyCallbackStruct *call_data )
{

  /* Variables. */
  int     action;
  int     index;
  int     index1;
  int     selection;
  Atom    string_type = XA_STRING;


  /* Code. */

  /* Find the selection and action. */
  selection = XIT_CL_UNDEFINED;
  action    = XIT_CL_UNDEFINED;

  for( index = 0; index < XtNumber( record -> selection ); index++ ) {
    for( index1 = 0; 
         index1 < XtNumber( record -> selection[ 0 ].actionButton ); 
         index1++ ) {
      if( record -> selection[ index ].actionButton[ index1 ] == widget ) {
        selection = index;
        action    = index1;
        goto found;
      }
    }
  }

  found:
  if( selection == XIT_CL_UNDEFINED )
    return;

  /* Paste from clipboard? */
  if( action == XIT_CL_PASTE ) {
      switch( selection ) {
        case XIT_CL_PRIMARY:
          XtGetSelectionValue( record -> popup, XA_PRIMARY, string_type,
                               (XtSelectionCallbackProc) insertSelectionCB,
                               (XtPointer) record,
                               call_data -> event -> xbutton.time );
          break;

        case XIT_CL_SECONDARY:
          XtGetSelectionValue( record -> popup, XA_SECONDARY, string_type,
                               (XtSelectionCallbackProc) insertSelectionCB,
                               (XtPointer) record,
                               call_data -> event -> xbutton.time );
          break;

        case XIT_CL_CLIPBOARD:
          return;

        case XIT_CL_XBUFFER:
          {
            int   bytes;
            int   *bytes_ref = &bytes;
            char  *char_ref;
            Atom  x_buffer = XA_CUT_BUFFER0;

            char_ref = XFetchBuffer( XtDisplay( record -> popup ), 
                                     bytes_ref, 0 );
            if( bytes > 0 )
              insertSelectionCB( record -> popup, record, 
                                 &x_buffer, &string_type,
                                 (XtPointer) char_ref, 
                                 (unsigned long *) bytes_ref, 0 );
          }
          break;

        default:
          return;
      } /* switch */
  } /* if */

  /* Cut selection and paste to clipboard? */
  if( action == XIT_CL_CUT ) {
    return;
  } /* if */

  /* Copy selection and paste to clipboard? */
  if( action == XIT_CL_COPY ) {
    return;
  } /* if */

  return;

} /* actionCB */


/*----------------------------------------------------------------------*/

static void
  insertSelectionCB( Widget              widget,
                     XIT_CL_RECORD_REF   record,
                     Atom                *selection,
                     Atom                *type,
                     XtPointer           value,
                     unsigned long       *length,
                     int                 *format )
{

  /* Variables. */
  char        *buffer_ref;


  /* Code. */

  /* If we do not have an insert function to call, just return. */
  if( record -> insertCB == NULL )
    return;

  /* We can only deal with XA_STRING's with a length > 0. */
  if( *type != XA_STRING )
    return;

  if( *length <= 0 )
    return;

  /* Make sure a NULL is at the end of the bytes to copy. */
  buffer_ref = SysMalloc( *length + 1 );
  memcpy( buffer_ref, value, *length );
  *(buffer_ref + *length) = '\0';

  /* Let the client do the job. */
  (*record -> insertCB) ( buffer_ref, record -> client_data );

  SysFree( buffer_ref );

  return;

} /* insertSelectionCB */


