/*----------------------------------------------------------------------------
--
--  Module:           XmUbTimeBox
--
--  Project:          XmUb - Ulle's Motif widgets
--  System:           
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    This is the public header file for the widget.
--
--    The time box widget provides a user interface for selecting a date and/or
--    time and a date/time range. There is a pulldown menu that may be used 
--    to select the time and range.
--    The date and time fields have automatic completion so that only part of 
--    the string must be written. The date and time may also be selected 
--    from a popup window in the field (per default popped up at double-click)
--    which enables a totally mouse-oriented as well as a totally keyboard-
--    oriented interface.
--
--  Filename:         XmUbTimeB.h
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1993-09-12
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
/* SCCSID = @(#) Module: XmUbTimeB.h, Version: 1.1, Date: 95/02/18 15:10:15 */

#ifndef _XmUbTimeB_h
#define _XmUbTimeB_h

/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <time.h>
#include <X11/StringDefs.h>
#include <Xm/Xm.h>

/*----------------------------------------------------------------------------
--  Resources for the time box widget.
--  All XmManager resources apply. 
--  The following list describes resources added in this widget.
--
--  XmNactivateCallback              CSG
--      XmCCallback
--    Callback(s) that is called when the user pressed <Enter> in a text field
--    of the widget, a date or time is picked from the default dialogs,
--    a date is dropped on a date field or the user selects one of the
--    default actions from the menu.
--    See description of callbacks below for more information.
--    Type:    XtCallbackList
--    Default: NULL
--    
--  XmNcancelLabelString                 CSG
--      XmCCancelLabelString
--    The label used for "Cancel" buttons in created dialogs.
--    If null, the string "Cancel" will be used.
--    The string is copied internally. XtGetValues returns a copy.
--    Type:    XmString
--    Default: NULL
--
--  XmUbNtboxCodeDay                 CSG
--      XmUbCTboxCodeDay    
--    The characters that are used as a valid day abbreviation when scanning 
--    for date completion. A character matches if it is present in the string.
--    The user only has to type one character. The string is copied internally.
--    Max string length is 20.
--    The string is copied internally. A pointer to the internal structure is
--    returned by XtGetValues.
--    Type:    char*
--    Default: "dD"
--
--  XmUbNtboxCodeHour                CSG
--      XmUbCTboxCodeHour
--    The characters that are used as a valid hour abbreviation when scanning 
--    for time completion. A character matches if it is present in the string.
--    The user only has to type one character. The string is copied internally.
--    Max string length is 20.
--    The string is copied internally. A pointer to the internal structure is
--    returned by XtGetValues.
--    Type:    char*
--    Default: "hH"
--
--  XmUbNtboxCodeMinute              CSG
--      XmUbCTboxCodeMinute
--    The characters that are used as a valid minute abbreviation when 
--    scanning for time completion. A character matches if it is present in 
--    the string.
--    The user only has to type one character. The string is copied internally.
--    Max string length is 20.
--    The string is copied internally. A pointer to the internal structure is
--    returned by XtGetValues.
--    Type:    char*
--    Default: "mM"
--
--  XmUbNtboxCodeMonth               CSG
--      XmUbCTboxCodeMonth
--    The characters that are used as a valid month abbreviation when scanning 
--    for date completion. A character matches if it is present in the string.
--    The user only has to type one character. The string is copied internally.
--    Max string length is 20.
--    The string is copied internally. A pointer to the internal structure is
--    returned by XtGetValues.
--    Type:    char*
--    Default: "mM"
--
--  XmUbNtboxCodeWeek                CSG
--      XmUbCTboxCodeWeek
--    The characters that are used as a valid week abbreviation when scanning 
--    for date completion. A character matches if it is present in the string.
--    The user only has to type one character. The string is copied internally.
--    Max string length is 20.
--    The string is copied internally. A pointer to the internal structure is
--    returned by XtGetValues.
--    Type:    char*
--    Default: "wW"
--
--  XmUbNtboxCodeYear                CSG
--      XmUbCTboxCodeYear
--    The characters that are used as a valid year abbreviation when scanning 
--    for date completion. A character matches if it is present in the string.
--    The user only has to type one character. The string is copied internally.
--    Max string length is 20.
--    The string is copied internally. A pointer to the internal structure is
--    returned by XtGetValues.
--    Type:    char*
--    Default: "yY"
--
--  XmUbNtboxCompletionCallback      CSG
--      XmCCallback
--    Callback(s) that is called after an attempted completion of a date or
--    time field. The callback reason tells what happened.
--    See the callback description below for details.
--    Type:    XtCallbackList
--    Default: NULL
--
--  XmUbNtboxComponentTogether       CG
--      XmUbCTboxComponentTogether
--    If True, the two fields that form a time component (date and time) are
--    not split in the vertical mode. If False, they are.
--    Type:    Boolean
--    Default: True
--
--  XmUbNtboxDateSelectionTitle      CSG
--      XmUbCTboxDateSelectionTitle
--    The title of the date selection dialog. Applies only if the default 
--    dialog is used. If NULL, a default string will be used.
--    The string is copied internally. A pointer to the internal structure is
--    returned by XtGetValues.
--    Type:    String
--    Default: NULL
--
--  XmUbNtboxDestroyDialogs          CG
--      XmUbCTboxDestroyDialogs
--    If True, and the default dialogs are used, the dialogs are destoyed on
--    pop-down. If False, the default dialogs are created permanently the 
--    first time they are popped up. Note that the default dialogs are only
--    used if XmUbNtboxPopupSelectionCallback is NULL.
--    Type:    Boolean
--    Default: True
--
--  XmUbNtboxFieldSpacing            CSG
--      XmCSpacing
--    The spacing between the text widgets in the same range.
--    Type:    Dimension
--    Default: 2
--
--  XmUbNtboxFormat                  CG
--      XmUbCTboxFormat
--    The format of the time box. One of the widget layout formats defined 
--    in this file.
--      XmUbTB_FORMAT_D    Date
--      XmUbTB_FORMAT_T    Time
--      XmUbTB_FORMAT_DT   Date - Time
--      XmUbTB_FORMAT_DD   Date - Date
--      XmUbTB_FORMAT_DTDT Date - Time - Date - Time
--      XmUbTB_FORMAT_DTT  Date - Time - Time
--      XmUbTB_FORMAT_TT   Time - Time
--    When specifying in the resource file, skip XmUb.
--    Type:    int
--    Default: XmUbTB_FORMAT_DTDT
--
--  XmNmarginHeight                  CSG   
--      XmCMarginHeight
--    The margin used at the top and the bottom of the widget.
--    Type:    Dimension
--    Default: 0
--                        
--  XmNmarginWidth                   CSG    
--      XmCMarginWidth
--    The margin used at the left and right ends of the widget.
--    Type:    Dimension
--    Default: 0
--                        
--  XmUbNtboxMenuEnabled             CG
--      XmUbCTboxMenuEnabled
--    If True, the menu in front of the first field is enabled, if False, 
--    no menu is created.
--    Type:    Boolean
--    Default: True
--
--  XmUbNtboxMenuItems               CG
--      XmUbCTboxMenuItems
--    The contents of the menu (if enabled). This is an array of XmUbMenuItem
--    structs. The number of items must be given in XmUbNtboxNumItems.
--    If this resource is NULL, but the menu is enabled, a default menu with 
--    English labels is created.
--    Type:    XmUbMenuItemRef
--    Default: NULL
--                        
--  XmUbNtboxMenuLabel               CG
--      XmUbCTboxMenuLabel
--    The string to be used as label on the cascade button. 
--    See below for a description of the creation of that button.
--    Type:    XmString
--    Default: NULL
--
--  XmUbNtboxMenuPixmap              CG
--      XmUbCTboxMenuPixmap
--    The pixmap to be used on the cascade button. 
--    See below for a description of the creation of that button.
--    Type:    Pixmap
--    Default: XmUNSPECIFIED_PIXMAP
--
--  XmUbNtboxMenuPosition            CG
--      XmUbCTboxMenuPosition
--    The position of the menu cascade button. Valid values are
--    XmUbPOSITION_FIRST   The menu is placed left of the first field.
--    XmUbPOSITION_LAST    The menu is placed right of the last field.
--    Type:    unsigned char
--    Default: XmUbPOSITION_FIRST
--
--  XmUbNtboxMenuSpacing             CSG
--      XmCSpacing
--    The spacing between the menu and the first text field.
--    Type:    Dimension
--    Default: 4
--
--  XmUbNtboxMonthLabel              CSG
--      XmUbCTboxMonthLabel
--    The string used in the default month dialog for month scrolling.
--    If NULL, the default string "Month" will be used.
--    Type:    XmString
--    Default: NULL
--
--  XmUbNtboxNumItems                CG
--      XmUbCTboxNumItems
--    The number of items in the XmUbNtboxMenuItems array.
--    Type:    int
--    Default: 0
--
--  XmNokLabelString                 CSG
--      XmCOkLabelString
--    The label used for "Ok" buttons in created dialogs.
--    If null, the string "Ok" will be used.
--    The string is copied internally. XtGetValues returns a copy.
--    Type:    XmString
--    Default: NULL
--
--  XmNorientation                   CG
--      XmCOrientation
--    The orientation in which the children are laid out. XmHORIZONTAL makes 
--    the widget lay out the children left to right, XmVERTICAL lays them out
--    top to bottom.
--    Type:    unsigned char
--    Default: XmHORIZONTAL
--                        
--  XmUbNtboxPopupSelectionCallback      CSG
--      XmCCallback
--    Callback(s) that is called when the user double-clicks in a date or
--    time field. If no callback is set here, a default dialog will be created
--    and popped up at the double-click.
--    If the default dialog is used, the resource XmUbNtboxDestroyDialogs will 
--    be used to determine if the dialog will be created as it is popped 
--    down or if it will be permanent throughout the life of the time box
--    widget.
--    Type:    XtCallbackList
--    Default: NULL
--
--  XmUbNtboxRangeSpacing            CSG
--      XmCSpacing
--    The spacing between the text widgets in different ranges.
--    Type:    Dimension
--    Default: 10
--
--  XmUbNtboxTimeSelectionTitle      CSG
--      XmUbCTboxTimeSelectionTitle
--    The title of the time selection dialog. Applies only if the default
--    dialog is used.
--    If NULL, a default string will be used.
--    The string is copied internally. A pointer to the internal structure is
--    returned by XtGetValues.
--    Type:    String
--    Default: NULL
--
--  XmNvalueChangedCallback          CSG
--      XmCCallback
--    Callback(s) that is called when text was inserted into or deleted from
--    a child text widget. 
--    See description of callbacks below for more information.
--    Type:    XtCallbackList
--    Default: NULL
--    
--  XmUbNtboxWeekLabel              CSG
--      XmUbCTboxWeekLabel
--    The string used in the default month dialog as title of the week number
--    column. If NULL, the default string "W" will be used.
--    Type:    XmString
--    Default: NULL
--
--  XmUbNtboxYearLabel               CSG
--      XmUbCTboxYearLabel
--    The string used in the default month dialog for year scrolling.
--    If NULL, the default string "Year" will be used.
--    Type:    XmString
--    Default: NULL
--
----------------------------------------------------------------------------
--
--  Creation of the cascade button
--  ------------------------------
--
--  If the resource XmUbNtboxMenuLabel is non-NULL, a cascade button of the
--  type XmSTRING is created, with the given resource as labelString.
--  If XmUbNtboxMenuLabel is NULL and XmUbNtboxMenuPixmap is defined,
--  a cascade button of the type XmPIXMAP is created, with the given pixmap
--  as labelPixmap. 
--  If both resources are undefined, an XmSTRING type cascade button is 
--  created, with a default string as labelString (the default string is "-" )
--
----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#define XmUbNtboxCodeDay                 "tboxCodeDay"
#define XmUbNtboxCodeHour                "tboxCodeHour"
#define XmUbNtboxCodeMinute              "tboxCodeMinute"
#define XmUbNtboxCodeMonth               "tboxCodeMonth"
#define XmUbNtboxCodeWeek                "tboxCodeWeek"
#define XmUbNtboxCodeYear                "tboxCodeYear"
#define XmUbNtboxCompletionCallback      "tboxCompletionCallback"
#define XmUbNtboxDateSelectionTitle      "tboxDateSelectionTitle"
#define XmUbNtboxDestroyDialogs          "tboxDestroyDialogs"
#define XmUbNtboxFieldSpacing            "tboxFieldSpacing"
#define XmUbNtboxFormat                  "tboxFormat"
#define XmUbNtboxComponentTogether       "tboxComponentTogether"
#define XmUbNtboxMenuEnabled             "tboxMenuEnabled"
#define XmUbNtboxMenuItems               "tboxMenuItems"
#define XmUbNtboxMenuLabel               "tboxMenuLabel"
#define XmUbNtboxMenuPixmap              "tboxMenuPixmap"
#define XmUbNtboxMenuPosition            "tboxMenuPosition"
#define XmUbNtboxMenuSpacing             "tboxMenuSpacing"
#define XmUbNtboxMonthLabel              "tboxMonthLabel"
#define XmUbNtboxNumItems                "tboxNumItems"
#define XmUbNtboxPopupSelectionCallback  "tboxPopupSelectionCallback"
#define XmUbNtboxRangeSpacing            "tboxRangeSpacing"
#define XmUbNtboxTimeSelectionTitle      "tboxTimeSelectionTitle"
#define XmUbNtboxWeekLabel               "tboxWeekLabel"
#define XmUbNtboxYearLabel               "tboxYearLabel"

#define XmUbCTboxCodeDay                 "TboxCodeDay"
#define XmUbCTboxCodeHour                "TboxCodeHour"
#define XmUbCTboxCodeMinute              "TboxCodeMinute"
#define XmUbCTboxCodeMonth               "TboxCodeMonth"
#define XmUbCTboxCodeWeek                "TboxCodeWeek"
#define XmUbCTboxCodeYear                "TboxCodeYear"
#define XmUbCTboxComponentTogether       "TboxComponentTogether"
#define XmUbCTboxDateSelectionDialog     "TboxDateSelectionDialog"
#define XmUbCTboxDateSelectionTitle      "TboxDateSelectionTitle"
#define XmUbCTboxDestroyDialogs          "TboxDestroyDialogs"
#define XmUbCTboxFormat                  "TboxFormat"
#define XmUbCTboxMenuEnabled             "TboxMenuEnabled"
#define XmUbCTboxMenuItems               "TboxMenuItems"
#define XmUbCTboxMenuLabel               "TboxMenuLabel"
#define XmUbCTboxMenuPixmap              "TboxMenuPixmap"
#define XmUbCTboxMenuPosition            "TboxMenuPosition"
#define XmUbCTboxMonthLabel              "TboxMonthLabel"
#define XmUbCTboxNumItems                "TboxNumItems"
#define XmUbCTboxTimeSelectionDialog     "TboxTimeSelectionDialog"
#define XmUbCTboxTimeSelectionTitle      "TboxTimeSelectionTitle"
#define XmUbCTboxYearLabel               "TboxYearLabel"
#define XmUbCTboxWeekLabel               "TboxWeekLabel"


/* Callback reasons. */
#define XmUbCR_COMPLETION_SUCCESS    0
#define XmUbCR_COMPLETION_FAILED     1
#define XmUbCR_COMPLETION_EMPTY      2
#define XmUbCR_POPUP_DATE_SELECTION  3
#define XmUbCR_POPUP_TIME_SELECTION  4
#define XmUbCR_DATE_PICKED           5
#define XmUbCR_TIME_PICKED           6
#define XmUbCR_DATE_DROPPED          7
#define XmUbCR_DEFAULT_ACTION        8

/* Widget layout formats. */
/* D - date field, T - time field. */

/* Single entry formats. */
#define XmUbTB_FORMAT_D      0
#define XmUbTB_FORMAT_T      1
#define XmUbTB_FORMAT_DT     2

/* Range formats. */
#define XmUbTB_FORMAT_DD     3
#define XmUbTB_FORMAT_DTDT   4
#define XmUbTB_FORMAT_DTT    5
#define XmUbTB_FORMAT_TT     6

/* Menu positions. */
#define XmUbPOSITION_FIRST   0
#define XmUbPOSITION_LAST    1

/* Default menu item actions */
#define XmUbTB_NO_ACTION             ( (XtCallbackProc) 0 )
#define XmUbTB_ACTION_THIS_WEEK      ( (XtCallbackProc) 1 )
#define XmUbTB_ACTION_THIS_MONTH     ( (XtCallbackProc) 2 )
#define XmUbTB_ACTION_PLUS_ONE_HOUR  ( (XtCallbackProc) 3 )
#define XmUbTB_ACTION_PLUS_TWO_HOURS ( (XtCallbackProc) 4 )
#define XmUbTB_ACTION_PLUS_ONE_WEEK  ( (XtCallbackProc) 5 )
#define XmUbTB_ACTION_PLUS_TWO_WEEKS ( (XtCallbackProc) 6 )
#define XmUbTB_ACTION_PLUS_ONE_MONTH ( (XtCallbackProc) 7 )
#define XmUbTB_ACTION_TIME_NOW       ( (XtCallbackProc) 8 )
#define XmUbTB_ACTION_DATE_NOW       ( (XtCallbackProc) 9 )
#define XmUbTB_ACTION_NOW            ( (XtCallbackProc) 10 )
#define XmUbTB_LAST_PREDEF_ACTION    XmUbTB_ACTION_NOW

/* Id's for children retrieval. */
#define XmUbTB_CHILD_MENU_CASC     0 
#define XmUbTB_CHILD_MENU_PD       1 
#define XmUbTB_CHILD_MENU_BAR      2
#define XmUbTB_CHILD_START_DATE    3
#define XmUbTB_CHILD_START_TIME    4 
#define XmUbTB_CHILD_END_DATE      5
#define XmUbTB_CHILD_END_TIME      6

#define XmUbTB_LAST_CHILD  XmUbTB_CHILD_END_TIME

/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Status codes. */
/* When a status code of the type XmUbTimeBoxStatus is returned from a 
   function, it has the following meaning:

   TBOX_OK             Operation OK and out parameters are valid.
   TBOX_EMPTY          The desired field was empty.
   TBOX_NO_FIELD       The desired field is not present in the selected format.
   TBOX_CONV_ERROR     The text in the field could not be converted.
   TBOX_NO_COMPLETION  The text in the field could not be completed.
                       This code is only used internally.
*/

typedef enum { TBOX_OK, 
               TBOX_EMPTY, 
               TBOX_NO_FIELD, 
               TBOX_CONV_ERROR,
               TBOX_NO_COMPLETION } XmUbTimeBoxStatus;

/* The type XmUbMenuItem is used for the description of the menu items. */
/* The item field describes what type of widget that should be inserted.
   For pushbuttons, the fields proc and label should also be supplied.
   A 'proc' function is a callback to the pressed push button.
   It must perform all necessary changes to the internal fields in the 
   time box. 
   There are some default actions available. By using one of these actions,
   you don't have to supply your own function.
   Supply name and label if you don't want the default ones.

   The id of the time box widget is set in the user data resource of the
   pushbuttons, so don't fiddle around with it. */

typedef enum { XmUbITEM_PUSH_BUTTON, XmUbITEM_SEPARATOR } XmUbItem;

typedef struct {

  XmUbItem        item;
  String          name;
  XmString        label;
  XtCallbackProc  proc;
  XtPointer       client_data;  /* Only if user-supplied callback. */

} XmUbMenuItem, *XmUbMenuItemRef;


/* Callback structure. */

typedef struct {

  int             reason;
  XEvent          *event;
  int             child_index;
  Widget          child;
  XtCallbackProc  action;
  Boolean         range;
  Boolean         restart_completion;

} XmUbTimeBoxCallbackStruct;

/* 
-- CALLBACKS
--
-- XmNactivateCallback
--   Called when <Enter> is pressed in one of the text fields.
--  
--     reason              XmCR_ACTIVATE
--     event               The keyboard event.
--     child_index         The index of the child ( XmUbTB_CHILD_... )
--     child               The widget id for the child.
--
--   Also called when a date or time is picked from one of the default dialogs.
--
--     reason              XmUbCR_DATE_PICKED or XmUbCR_TIME_PICKED
--     event               NULL
--     child_index         The index of the child ( XmUbTB_CHILD_... )
--     child               The widget id for the child.
--     range               True if a range was set, False for a single field.
--
--   Also called when a date was dropped (from a month display widget).
--
--     reason              XmUbCR_DATE_DROPPED
--     event               NULL
--     child_index         The index of the child ( XmUbTB_CHILD_... )
--     child               The widget id for the child.
--     range               True if a range was transfered, False for a date.
--
--   Also called when a default action was selected from the menu.
--
--     reason              XmUbCR_DEFAULT_ACTION
--     event               NULL
--     action              The selected action.
--     child_index         0
--     child               NULL
--     range               True if a range was set, False for a single field.
--
-- XmNvalueChangedCallback
--   Called when in one of the text fields is changed.
--
--   reason              XmCR_VALUE_CHANGED
--   event               The keyboard event.
--   child_index         The index of the child ( XmUbTB_CHILD_... )
--   child               The widget id for the child.
--
-- XmUbNtboxCompletionCallback
--   Called when time or date completion has been attempted for a child text 
--   field. 
--
--   reason              One of the completion reasons.             
--   event               The losing focus callback event.
--   child_index         The index of the child ( XmUbTB_CHILD_... )
--   child               The widget id for the child.
--   restart_completion  False. Set this field to True to make the parsing
--                       start anew when the callback list is finished.
--
-- XmUbNtboxPopupSelectionCallback
--   Called when the user double-clicked in one of the text fields.
--
--   reason              XmUbCR_POPUP_DATE_SELECTION or 
--                       XmUbCR_POPUP_TIME_SELECTION.
--   event               NULL.
--   child_index         The index of the child ( XmUbTB_CHILD_... )
--   child               The widget id for the child.
--
*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Class record constants. */
extern WidgetClass xmUbTimeBoxWidgetClass;


typedef struct _XmUbTimeBoxClassRec *XmUbTimeBoxWidgetClass;
typedef struct _XmUbTimeBoxRec      *XmUbTimeBoxWidget;


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

Widget  
  XmUbCreateTimeBox( Widget    parent,
                     String    name,
                     ArgList   arglist,
                     Cardinal  argcount );

Widget 
  XmUbTimeBoxGetChild( Widget  widget,
                       int     child );

/* Gets a copy of the string entered in the different fields.
   The returned string must be freed by the caller, using XtFree(). */


XmUbTimeBoxStatus
  XmUbTimeBoxGetEndDateString( Widget  widget,
                               char    **str );

XmUbTimeBoxStatus
  XmUbTimeBoxGetEndTimeString( Widget  widget,
                               char    **str );

XmUbTimeBoxStatus
  XmUbTimeBoxGetStartDateString( Widget  widget,
                                 char    **str );

XmUbTimeBoxStatus
  XmUbTimeBoxGetStartTimeString( Widget  widget,
                                 char    **str );

/* Get the binary time entered in the different fields.
   For time fields, 1 Jan 1970 is used as the date, 
   for date fields, 00:00:00 is used as the time. */


XmUbTimeBoxStatus
  XmUbTimeBoxGetEndDate( Widget   widget,
                         time_t   *date );

XmUbTimeBoxStatus
  XmUbTimeBoxGetEndTime( Widget   widget,
                         time_t   *time );

XmUbTimeBoxStatus
  XmUbTimeBoxGetStartDate( Widget   widget,
                           time_t   *date );

XmUbTimeBoxStatus
  XmUbTimeBoxGetStartTime( Widget   widget,
                           time_t   *time );


/* Sets the time in the different fields. 
   Note that for date fieds, only the date component is used, and for
   time fields, only the time component is used. */

XmUbTimeBoxStatus
  XmUbTimeBoxSetEndDate( Widget  widget,
                         time_t  date );

XmUbTimeBoxStatus
  XmUbTimeBoxSetEndTime( Widget  widget,
                         time_t  time );

XmUbTimeBoxStatus
  XmUbTimeBoxSetStartDate( Widget  widget,
                           time_t  date );

XmUbTimeBoxStatus
  XmUbTimeBoxSetStartTime( Widget  widget,
                           time_t  time );

/* Sets the string in the different fields.
   The string is NOT converted to test for validity. */

XmUbTimeBoxStatus
  XmUbTimeBoxSetEndDateString( Widget  widget,
                               char    *str );

XmUbTimeBoxStatus
  XmUbTimeBoxSetEndTimeString( Widget  widget,
                               char    *str );

XmUbTimeBoxStatus
  XmUbTimeBoxSetStartDateString( Widget  widget,
                                 char    *str );

XmUbTimeBoxStatus
  XmUbTimeBoxSetStartTimeString( Widget  widget,
                                 char    *str );

/* Functions to pop up date and time dialogs. 
   Set the child_index to determine where the time/date is inserted after
   one has been selected. */

void
  XmUbTimeBoxPopupDateDialog( Widget  widget,
                              int     child_index );
void
  XmUbTimeBoxPopupTimeDialog( Widget  widget,
                              int     child_index );

#endif


