/*----------------------------------------------------------------------------
--
--  Module:           XmUbTimeBox
--
--  Project:          XmUb - Ulle's Motif widgets
--  System:           
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    This is the implementation of the widget.
--
--  Filename:         XmUbTimeB.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1993-09-12
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: XmUbTimeB.c, Version: 1.1, Date: 95/02/18 15:10:15";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <Xm/XmP.h>

#if XmVersion > 1001
#include <Xm/AtomMgr.h>
#include <Xm/DragDrop.h>
#endif

#include <Xm/CascadeB.h>
#include <Xm/Form.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/Separator.h>
#include <Xm/Text.h>

#include "XmUbMonthD.h"
#include "XmUbTimeSl.h"

/* Rogge's tools. */
#include "TimDate.h"

/* Private widget header file. */
#include "XmUbTimeBP.h"

/* Drag and drop. */
#if XmVersion > 1001
#include "XmUbDragDrop.h"
#endif

/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#ifdef MAX
#undef MAX
#endif

#ifdef MIN
#undef MIN
#endif

#define MAX( x, y )  ( ( x ) > ( y ) ? ( x ) : ( y ) )
#define MIN( x, y )  ( ( x ) < ( y ) ? ( x ) : ( y ) )

#define DATE_FIELD_COLUMNS      10
#define TIME_FIELD_COLUMNS       8
#define TIME_STRING_BUFFER_LEN  25
#define TEXT_BUFFER_LENGTH      20

/* Resource converters. */
#define XmRwidgetFormatType      "TimeBoxWidgetFormatType"
#define XmRmenuPositionType      "TimeBoxMenuPositionType"


/* Constants. */
#define MAX_NO_DEFAULT_MENU_ITEMS  15
#define MAX_PATTERN_LENGTH         30


/* General error codes. */
#define XmUbTB_CHILD_ERROR     -1

/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

typedef struct {

  Dimension  width;
  Dimension  height;
  Position   x;
  Position   y;

} KidDimensionRec;

/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Flag is set when the application context dependent things have been done. */
static Boolean  context_dependent_done = False;

static Atom protocol_atom;
static Atom delete_atom;

/* Internal Motif functions (how do we know about this....?) */
extern void
  _XmBackgroundColorDefault();
extern void
  _XmForegroundColorDefault();

/* Default scanning strings for date/time completion. */
static char default_day_marker[]    = "dD";
static char default_hour_marker[]   = "hH";
static char default_minute_marker[] = "mM";
static char default_month_marker[]  = "mM";
static char default_week_marker[]   = "wW";
static char default_year_marker[]   = "yY";

/* Default string for default action when completion failed. */
static char default_completion_failed_string[] = "??";

/* Default dialog titles. */
static char default_date_selection_title[] = "Select a date";
static char default_time_selection_title[] = "Select a time";

/* Default dialog labels. */
static char default_ok_string[]     = "Ok";
static char default_cancel_string[] = "Cancel";
static char default_year_string[]   = "Year";
static char default_month_string[]  = "Month";
static char default_week_string[]   = "W";

/* The time box resource list. */
static XtResource resources[] = {
  {
    XmNactivateCallback,
    XmCCallback,
    XmRCallback, sizeof( XtCallbackList ),
    XtOffset( XmUbTimeBoxWidget, tbox.activate_callback ),
    XmRCallback,
    NULL
  },
  {
    XmNvalueChangedCallback,
    XmCCallback,
    XmRCallback, sizeof( XtCallbackList ),
    XtOffset( XmUbTimeBoxWidget, tbox.value_changed_callback ),
    XmRCallback,
    NULL
  },
  {
    XmUbNtboxPopupSelectionCallback,
    XmCCallback,
    XmRCallback, sizeof( XtCallbackList ),
    XtOffset( XmUbTimeBoxWidget, tbox.popup_selection_callback ),
    XmRCallback,
    NULL
  },
  {
    XmNcancelLabelString,
    XmCCancelLabelString,
    XmRXmString, sizeof( XmString ),
    XtOffset( XmUbTimeBoxWidget, tbox.cancel_label ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmNmarginHeight,
    XmCMarginHeight,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbTimeBoxWidget, tbox.margin_height ),
    XtRImmediate,
    (XtPointer) 0
  },
  {
    XmNmarginWidth,
    XmCMarginWidth,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbTimeBoxWidget, tbox.margin_width ),
    XtRImmediate,
    (XtPointer) 0
  },
  {
    XmNokLabelString,
    XmCOkLabelString,
    XmRXmString, sizeof( XmString ),
    XtOffset( XmUbTimeBoxWidget, tbox.ok_label ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmNorientation,
    XmCOrientation,
    XmROrientation, sizeof( unsigned char ),
    XtOffset( XmUbTimeBoxWidget, tbox.orientation ),
    XtRImmediate,
    (XtPointer) XmHORIZONTAL
  },
  {
    XmUbNtboxCodeDay,
    XmUbCTboxCodeDay,
    XmRString, sizeof( char* ),
    XtOffset( XmUbTimeBoxWidget, tbox.day_marker ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNtboxCodeHour,
    XmUbCTboxCodeHour,
    XmRString, sizeof( char* ),
    XtOffset( XmUbTimeBoxWidget, tbox.hour_marker ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNtboxCodeMinute,
    XmUbCTboxCodeMinute,
    XmRString, sizeof( char* ),
    XtOffset( XmUbTimeBoxWidget, tbox.minute_marker ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNtboxCodeMonth,
    XmUbCTboxCodeMonth,
    XmRString, sizeof( char* ),
    XtOffset( XmUbTimeBoxWidget, tbox.month_marker ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNtboxCodeWeek,
    XmUbCTboxCodeWeek,
    XmRString, sizeof( char* ),
    XtOffset( XmUbTimeBoxWidget, tbox.week_marker ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNtboxCodeYear,
    XmUbCTboxCodeYear,
    XmRString, sizeof( char* ),
    XtOffset( XmUbTimeBoxWidget, tbox.year_marker ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNtboxCompletionCallback,
    XmCCallback,
    XmRCallback, sizeof( XtCallbackList ),
    XtOffset( XmUbTimeBoxWidget, tbox.completion_callback ),
    XmRCallback,
    NULL
  },
  {
    XmUbNtboxComponentTogether,
    XmUbCTboxComponentTogether,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbTimeBoxWidget, tbox.component_together ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNtboxDateSelectionTitle,
    XmUbCTboxDateSelectionTitle,
    XtRString, sizeof( String ),
    XtOffset( XmUbTimeBoxWidget, tbox.date_selection_title ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNtboxDestroyDialogs,
    XmUbCTboxDestroyDialogs,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbTimeBoxWidget, tbox.destroy_dialogs ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNtboxFieldSpacing,
    XmCSpacing,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbTimeBoxWidget, tbox.field_spacing ),
    XtRImmediate,
    (XtPointer) 2
  },
  {
    XmUbNtboxFormat,
    XmUbCTboxFormat, 
    XmRwidgetFormatType, sizeof( widgetFormatType ),
    XtOffset( XmUbTimeBoxWidget, tbox.widget_format ),
    XtRImmediate,
    (XtPointer) XmUbTB_FORMAT_DTDT
  },
  {
    XmUbNtboxMenuEnabled,
    XmUbCTboxMenuEnabled,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbTimeBoxWidget, tbox.menu_enabled ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNtboxMenuItems,
    XmUbCTboxMenuItems,
    XtRPointer, sizeof( XmUbMenuItemRef ),
    XtOffset( XmUbTimeBoxWidget, tbox.menu_items ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNtboxMenuLabel,
    XmUbCTboxMenuLabel,
    XmRXmString, sizeof( XmString ),
    XtOffset( XmUbTimeBoxWidget, tbox.menu_label ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNtboxMenuPixmap,
    XmUbCTboxMenuPixmap,
    XmRPrimForegroundPixmap, sizeof( Pixmap ),
    XtOffset( XmUbTimeBoxWidget, tbox.menu_pixmap ),
    XtRImmediate,
    (XtPointer) XmUNSPECIFIED_PIXMAP
  },
  {
    XmUbNtboxMenuPosition,
    XmUbCTboxMenuPosition,
    XmRmenuPositionType, sizeof( menuPositionType ),
    XtOffset( XmUbTimeBoxWidget, tbox.menu_position ),
    XtRImmediate,
    (XtPointer) XmUbPOSITION_FIRST
  },
  {
    XmUbNtboxMenuSpacing,
    XmCSpacing,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbTimeBoxWidget, tbox.menu_spacing ),
    XtRImmediate,
    (XtPointer) 4
  },
  {
    XmUbNtboxMonthLabel,
    XmUbCTboxMonthLabel,
    XmRXmString, sizeof( XmString ),
    XtOffset( XmUbTimeBoxWidget, tbox.month_label ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNtboxNumItems,
    XmUbCTboxNumItems,
    XmRInt, sizeof( int ),
    XtOffset( XmUbTimeBoxWidget, tbox.num_menu_items ),
    XtRImmediate,
    (XtPointer) 0
  },
  {
    XmUbNtboxRangeSpacing,
    XmCSpacing,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbTimeBoxWidget, tbox.range_spacing ),
    XtRImmediate,
    (XtPointer) 10
  },
  {
    XmUbNtboxTimeSelectionTitle,
    XmUbCTboxTimeSelectionTitle,
    XtRString, sizeof( String ),
    XtOffset( XmUbTimeBoxWidget, tbox.time_selection_title ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNtboxWeekLabel,
    XmUbCTboxWeekLabel,
    XmRXmString, sizeof( XmString ),
    XtOffset( XmUbTimeBoxWidget, tbox.week_label ),
    XtRImmediate,
    (XtPointer) NULL
  },
  {
    XmUbNtboxYearLabel,
    XmUbCTboxYearLabel,
    XmRXmString, sizeof( XmString ),
    XtOffset( XmUbTimeBoxWidget, tbox.year_label ),
    XtRImmediate,
    (XtPointer) NULL
  },

};  /* resources */

/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

/* Class methods. */
static void              
  ClassInitialize();

static void
  ChangeManaged( Widget  widget );

static void
  DeleteChild( Widget  widget );

static void 
  Destroy( Widget   widget );

static XtGeometryResult
  GeometryManager( Widget            widget,
                   XtWidgetGeometry  *request,
                   XtWidgetGeometry  *reply );

static void
  GetValuesHook( Widget    w,
                 ArgList   args,
                 Cardinal  *num_args );

static void
  Initialize( Widget     treq,
              Widget     tnew,
              ArgList    args,
              Cardinal   *num_args );

static void
  InsertChild( Widget  widget );

static void
  PopupDateSelection( Widget  widget );

static void
  PopupTimeSelection( Widget  widget );

static XtGeometryResult
  QueryGeometry( Widget             widget,
                 XtWidgetGeometry  *proposed,
                 XtWidgetGeometry  *answer );

static void
  Resize( Widget    widget );

static Boolean 
  SetValues( Widget     current,
             Widget     request,
             Widget     new,
             ArgList    args,
             Cardinal   *num_args );

/* Internal functions. */

static void
  ActivateCB( Widget               tw,
              XmUbTimeBoxWidget    tbox,
              XmAnyCallbackStruct  *call_data );

static void
  AddDateHour( XmUbTimeBoxWidget  tbox,
               int                hours );

static void
  AddDateMonth( XmUbTimeBoxWidget  tbox,
                int                months,
                int                sub_days );

static void
  AddDateWeek( XmUbTimeBoxWidget  tbox,
               int                weeks,
               int                sub_days );

static void
  CallDialogPopupCallback( XmUbTimeBoxWidget  tbox,
                           Widget             child,
                           int                reason );

/* If not in the child list, returns XmUbTB_CHILD_ERROR. */
static int
  ChildIndex( XmUbTimeBoxWidget  tbox,
              Widget             child );

static TIM_TIME_REF
  CombineDateAndTime( TIM_TIME_REF  date,
                      TIM_TIME_REF  time );

/* User must free returned string. */
static char
  *CompleteDateString( XmUbTimeBoxWidget  tbox,
                       char               *str );

static XmUbTimeBoxStatus
  CompleteEndDate( XmUbTimeBoxWidget  tbox );

/* True means time field has been updated. */
static Boolean
  CompleteEndDateKeyword( XmUbTimeBoxWidget  tbox,
                          char               *str );

static XmUbTimeBoxStatus
  CompleteEndTime( XmUbTimeBoxWidget  tbox );

/* True means time field has been updated. */
static Boolean
  CompleteEndTimeKeyword( XmUbTimeBoxWidget  tbox,
                          char               *str );

static XmUbTimeBoxStatus
  CompleteStartDate( XmUbTimeBoxWidget  tbox );

static XmUbTimeBoxStatus
  CompleteStartTime( XmUbTimeBoxWidget  tbox );

static char
  *CompleteTimeString( XmUbTimeBoxWidget  tbox,
                       char               *str );

static void
  ConvertToDateString( XmUbTimeBoxWidget  tbox,
                       time_t             time,
                       char               *buffer,
                       int                buf_len );

static void
  ConvertToMenuPosition( XrmValue   *args,
                         Cardinal   *num_args,
                         XrmValue   *from,
                         XrmValue   *to );

static void
  ConvertToTimeString( XmUbTimeBoxWidget  tbox,
                       time_t             time,
                       char               *buffer,
                       int                buf_len );

static void
  ConvertToWidgetFormat( XrmValue   *args,
                         Cardinal   *num_args,
                         XrmValue   *from,
                         XrmValue   *to );

static void
  CreateInternalWidgets( XmUbTimeBoxWidget  tbox );

static Widget
  CreateDateField( XmUbTimeBoxWidget  tbox,
                   char               *name );

static void
  CreateDateSelectionPopup( XmUbTimeBoxWidget  tbox );

static void
  CreateMenu( XmUbTimeBoxWidget  tbox,
              String             base_name );

static void
  CreateMenuItems( XmUbTimeBoxWidget  tbox,
                   Widget             parent,
                   XmUbMenuItemRef    items,
                   int                num_items );

static Widget
  CreateTimeField( XmUbTimeBoxWidget  tbox,
                   char               *name );

static void
  CreateTimeSelectionPopup( XmUbTimeBoxWidget  tbox );

static void
  DateCompletionCB( Widget                      tw,
                    XmUbTimeBoxWidget           tbox,
                    XmTextVerifyCallbackStruct  *call_data );

#if XmVersion > 1001
static void
  DateDropCB( Widget                    w,
              XtPointer                 client_data,
              XmDropProcCallbackStruct  *call_data );
#endif

static void
  DateSelectedInDialogCB( Widget                          md,
                          XmUbTimeBoxWidget               tbox,
                          XmUbMonthDisplayCallbackStruct  *call_data );

static void
  FillInStartOfMonth( XmUbTimeBoxWidget  tbox );

static void
  FillInStartOfWeek( XmUbTimeBoxWidget  tbox );

/* Fills in the width and height fields in the kids dimension recs. */
static void
  GetChildPrefSizes( XmUbTimeBoxWidget  tbox,
                     Widget             initiator,
                     XtWidgetGeometry   *request,
                     KidDimensionRec    sizes[] );

XmUbTimeBoxStatus
  GetChildString( XmUbTimeBoxWidget  tbox,
                  int                child, 
                  char               **str );

XmUbTimeBoxStatus
  GetStartDateTime( XmUbTimeBoxWidget  tbox,
                    Boolean            use_defaults,
                    TIM_TIME_REF       *date_time );

static void
  InitializeItem( int             index,
                  XmUbMenuItem    items[],
                  XmUbItem        item,
                  String          label,
                  XtCallbackProc  proc,
                  XtPointer       client_data );

static void
  InitializeDefaultMenu( XmUbTimeBoxWidget  tbox,
                         XmUbMenuItem       items[],
                         int                *num_items_created );

/* The sizes array  must have been processed by GetChildPrefSizes and
   PrepareLayout before DoLayout is called. */
static void
  DoLayout( XmUbTimeBoxWidget  tbox,
            Widget             initiator,
            XtWidgetGeometry   *request,
            KidDimensionRec    sizes[] );

static void 
  KidsPreferredGeometry( Widget            kid,
                         Widget            initiator,
                         XtWidgetGeometry  *request,
                         XtWidgetGeometry  *desired );

static void
  MenuItemActivatedCB( Widget               pb,
                       XtCallbackProc       action,
                       XmAnyCallbackStruct  *call_data );

Widget
  MonthDisplayId( XmUbTimeBoxWidget  tbox );

static void
  PopdownDateSelectionDialogCB( Widget               md,
                                XmUbTimeBoxWidget    tbox,
                                XmAnyCallbackStruct  *call_data );

static void
  PopdownTimeSelectionDialogCB( Widget               tw,
                                XmUbTimeBoxWidget    tbox,
                                XmAnyCallbackStruct  *call_data );

static void
  PopupDateSelectionAction( Widget    w,
                            XEvent    *event,
                            String    *params,
                            Cardinal  num_params );

static void
  PopupTimeSelectionAction( Widget    w,
                            XEvent    *event,
                            String    *params,
                            Cardinal  num_params );

/* Fills in the x and y fields in the kids dimension recs. */
/* The width and height fields must be set beforehand. */
static void
  PrepareLayout( XmUbTimeBoxWidget  tbox,
                 KidDimensionRec    sizes[],
                 Dimension          *own_width,
                 Dimension          *own_height );

/* ResizeIfNeeded fills in the sizes array. Does not have to be initialized. */
static Boolean
  ResizeIfNeeded( XmUbTimeBoxWidget  tbox,
                  KidDimensionRec    sizes[] );

static void
  SelectAllTextCB( Widget               tw,
                   XmUbTimeBoxWidget    tbox, 
                   XmAnyCallbackStruct  *call_data );

XmUbTimeBoxStatus
  SetChildString( XmUbTimeBoxWidget  tbox,
                  int                child, 
                  char               *str );

static void
  TimeCompletionCB( Widget                      tw,
                    XmUbTimeBoxWidget           tbox,
                    XmTextVerifyCallbackStruct  *call_data );

/* If TimeCompletionFailed returns True, the problem has been repaired and
   the calling callback should restart, otherwise it should return. */
static Boolean
  TimeCompletionCallback( Widget               tw,
			  int                  child_index,
			  XmUbTimeBoxWidget    tbox,
			  XmAnyCallbackStruct  *call_data,
			  XmUbTimeBoxStatus    reason );

static void
  TimeSelectedInDialogCB( Widget               tw,
                          XmUbTimeBoxWidget    tbox,
                          XmAnyCallbackStruct  *call_data );

#if XmVersion > 1001
static void
  TransferDateField( Widget         w,
                     XtPointer      closure,
                     Atom           *seltype,
                     Atom           *type,
                     XtPointer      value,
                     unsigned long  *length,
                     int            format );
#endif

Widget
  TimeSliderId( XmUbTimeBoxWidget  tbox );

static void
  ValueChangedCB( Widget               tw,
                  XmUbTimeBoxWidget    tbox,
                  XmAnyCallbackStruct  *call_data );

static void
  WarningNoResourceChange( XmUbTimeBoxWidget  tbox,
                           String             resource );

/*----------------------------------------------------------------------------
--  The action definitions need the declared functions.
----------------------------------------------------------------------------*/

/* Time field widget translation list. */
static char timeTextTranslationTable[] =
  "<Btn1Up>(2):     popup-time-selection() \n\
   <Btn1Down>:      grab-focus()";

/* Date field widget translation list. */
static char dateTextTranslationTable[] =
  "<Btn1Up>(2):     popup-date-selection() \n\
   <Btn1Down>:      grab-focus()";

static XtTranslations  dateTextTranslations;
static XtTranslations  timeTextTranslations;

/* Action list. */
static XtActionsRec actionsList[] =
{
  { "popup-time-selection", (XtActionProc) PopupTimeSelectionAction },
  { "popup-date-selection", (XtActionProc) PopupDateSelectionAction },
};


/*----------------------------------------------------------------------------
--  Initialization of the class record.
----------------------------------------------------------------------------*/

/* This initialization has to be done after the methods have been declared. */
XmUbTimeBoxClassRec xmUbTimeBoxClassRec = {

  { /* Core class fields. */
    /* superclass */                    (WidgetClass) &xmManagerClassRec,
    /* class_name */                    "XmUbTimeBox",
    /* widget_size */                   sizeof( XmUbTimeBoxRec ),
    /* class_initialize */              ClassInitialize,
    /* class_part_initialize */         NULL,
    /* class_inited */                  False,
    /* initialize */                    Initialize,
    /* initialize_hook */               NULL,
    /* realize */                       XtInheritRealize, 
    /* actions */                       actionsList,
    /* num_actions */                   XtNumber( actionsList ),
    /* resources */                     resources,
    /* num_resources */                 XtNumber( resources ),
    /* xrm_class */                     NULLQUARK,
    /* compress_motion */               True,
    /* compress_exposure */             True,
    /* compress_enterleave */           True,
    /* visible_interest */              False,
    /* destroy */                       Destroy,
    /* resize */                        Resize,
    /* expose */                        NULL,
    /* set_values */                    SetValues,
    /* set_values_hook */               NULL,
    /* set_values_almost */             XtInheritSetValuesAlmost,
    /* get_values_hook */               GetValuesHook,
    /* accept_focus */                  NULL,
    /* version */                       XtVersion,
    /* callback_private */              NULL,
    /* tm_table */                      NULL,
    /* query_geometry */                QueryGeometry,
    /* display_accelerator */           XtInheritDisplayAccelerator,
    /* extension */                     NULL
  },
  { /* Composite class part. */
    /* geometry_manager */              GeometryManager,
    /* change_managed */                ChangeManaged,
    /* insert_child */                  InsertChild,
    /* delete_child */                  DeleteChild,
    /* extension */                     NULL
  },
  { /* Constraint class fields. */
    /* subresources */                  NULL,
    /* subresource_count */             0,
    /* constraint_size */               sizeof( XmUbTimeBoxConstraintsRec ),
    /* initialize */                    NULL,
    /* destroy */                       NULL,
    /* set_values */                    NULL,
    /* extension */                     NULL
  },
  { /* XmManager class part. */
    /* translations */                  NULL,
    /* get_resources */                 NULL,
    /* num_get_resources */             0,
    /* get_constraint_resources */      NULL,
    /* num_get_constraint_resources */  0,
    /* extension */                     NULL
  },
  { /* Time box class part. */
    /* popup_time_selection */          PopupTimeSelection,
    /* popup_date_selection */          PopupDateSelection,
    /* extension */                     NULL
  },

}; 



/* Class record pointer. */
WidgetClass 
  xmUbTimeBoxWidgetClass = (WidgetClass) &xmUbTimeBoxClassRec;



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

static void
  ActivateCB( Widget               tw,
              XmUbTimeBoxWidget    tbox,
              XmAnyCallbackStruct  *call_data )
{
  /* Variables. */
  XmUbTimeBoxCallbackStruct  cb;

  /* Code. */

  if( tbox -> tbox.activate_callback == NULL )
    return;

  /* Set up callback structure. */

  cb.reason      = XmCR_ACTIVATE;
  cb.event       = call_data -> event;
  cb.child_index = ChildIndex( tbox, tw );
  cb.child       = tw;

  XtCallCallbackList( (Widget) tbox, tbox -> tbox.activate_callback,
                      (XtPointer) &cb );


  return;

} /* ActivateCB */


/*----------------------------------------------------------------------*/

static void
  AddDateHour( XmUbTimeBoxWidget  tbox,
               int                hours )
{
  /* Variables. */
  TIM_TIME_REF  curr_time;

  /* Code. */

  /* Get start time. */
  /* Must get OK back since we defaults are enabled. */
  (void) GetStartDateTime( tbox, True, &curr_time );

  /* Add the specified number of hours. */
  TimAddHours( &curr_time, hours );

  /* Format the time and display it. */
  (void) XmUbTimeBoxSetEndDate( (Widget) tbox, curr_time );
  (void) XmUbTimeBoxSetEndTime( (Widget) tbox, curr_time );


  return;

} /* AddDateHour */


/*----------------------------------------------------------------------*/

static void
  AddDateMonth( XmUbTimeBoxWidget  tbox,
                int                months,
                int                sub_days )
{
  /* Variables. */
  TIM_TIME_REF  curr_time;

  /* Code. */

  /* Get start time. */
  /* Must get OK back since we defaults are enabled. */
  (void) GetStartDateTime( tbox, True, &curr_time );

  /* Add the specified number of weeks. */
  TimAddMonths( &curr_time, months );

  /* Subtract days if specified. */
  if( sub_days > 0 )
    TimAddDays( &curr_time, - ( sub_days ) );

  /* Format the time and display it. */
  (void) XmUbTimeBoxSetEndDate( (Widget) tbox, curr_time );
  (void) XmUbTimeBoxSetEndTime( (Widget) tbox, curr_time );


  return;

} /* AddDateMonth */


/*----------------------------------------------------------------------*/

static void
  AddDateWeek( XmUbTimeBoxWidget  tbox,
               int                weeks,
               int                sub_days )
{
  /* Variables. */
  TIM_TIME_REF  curr_time;

  /* Code. */

  /* Get start time. */
  /* Must get OK back since we defaults are enabled. */
  (void) GetStartDateTime( tbox, True, &curr_time );

  /* Add the specified number of weeks. */
  TimAddDays( &curr_time, ( 7 * weeks - sub_days ) );

  /* Format the time and display it. */
  (void) XmUbTimeBoxSetEndDate( (Widget) tbox, curr_time );
  (void) XmUbTimeBoxSetEndTime( (Widget) tbox, curr_time );


  return;

} /* AddDateWeek */


/*----------------------------------------------------------------------*/

static void
  CallDialogPopupCallback( XmUbTimeBoxWidget  tbox,
                           Widget             child,
                           int                reason )
{
  /* Variables. */
  XmUbTimeBoxCallbackStruct  cb;

  /* Code. */

  if( tbox -> tbox.popup_selection_callback == NULL )
    return;

  /* Set up callback structure. */

  cb.reason      = reason;
  cb.event       = NULL;
  cb.child_index = ChildIndex( tbox, child );
  cb.child       = child;

  XtCallCallbackList( (Widget) tbox, 
                      tbox -> tbox.popup_selection_callback,
                      (XtPointer) &cb );


  return;

} /* CallDialogPopupCallback */


/*----------------------------------------------------------------------*/

static void
  ChangeManaged( Widget  widget )
{
  /* Variables. */
  Boolean            layout_done;
  KidDimensionRec    kids_sizes[ NO_INTERNAL_CHILDREN ];
  XmUbTimeBoxWidget  tbox;


  /* Code. */

  tbox = (XmUbTimeBoxWidget) widget;

  /* ResizeIfNeeded checks the size of all children and resizes the widget. */
  layout_done = ResizeIfNeeded( tbox, kids_sizes );

  /* Do the layout. */
  if( !layout_done )
    DoLayout( tbox, NULL, NULL, kids_sizes );


  return;

} /* ChangeManaged */


/*----------------------------------------------------------------------*/

static int
  ChildIndex( XmUbTimeBoxWidget  tbox,
              Widget             child )
{
  /* Variables. */
  int  index;

  /* Code. */

  for( index = 0; index < NO_INTERNAL_CHILDREN; index ++ ){

    if( tbox -> tbox.internal_children[ index ] == child )
      return index;

  } /* for */

 
  /* Specified child not found. */
  return XmUbTB_CHILD_ERROR;

} /* ChildIndex */


/*----------------------------------------------------------------------*/

static void              
  ClassInitialize()
{

  /* Variables. */
  TIM_STATUS_TYPE  status;

  /* Code. */

  /* Register type converters. */
  XtAddConverter( XmRString, XmRwidgetFormatType, 
                  ConvertToWidgetFormat, NULL, 0 );
  XtAddConverter( XmRString, XmRmenuPositionType, 
                  ConvertToMenuPosition, NULL, 0 );


  /* Initialize class-wide flag. */
  context_dependent_done = False;

  /* Parse translations tables. */
  dateTextTranslations = XtParseTranslationTable( 
    (String) dateTextTranslationTable );
  timeTextTranslations = XtParseTranslationTable( 
    (String) timeTextTranslationTable );


  return;

} /* ClassInitialize */


/*----------------------------------------------------------------------*/

static TIM_TIME_REF
  CombineDateAndTime( TIM_TIME_REF  date,
                      TIM_TIME_REF  time )
{
  /* Variables. */
  TIM_TIME_REF  combined_time;

  /* Code. */

  combined_time = TimMakeTime( TimIndexOfYear( date ),
                               TimIndexOfMonth( date ),
                               TimIndexOfDay( date ),
                               TimHour( time ),
                               TimMinute( time ),
                               TimSecond( time ) );

  return( combined_time );

} /* CombineDateAndTime */


/*----------------------------------------------------------------------*/

static char
  *CompleteDateString( XmUbTimeBoxWidget  tbox,
                       char               *str )
{
  /* Variables. */
  char             buffer[ TIME_STRING_BUFFER_LEN ];
  TIM_TIME_REF     conv_time;
  char             *return_str;
  TIM_STATUS_TYPE  status;

  /* Code. */

  /* For empty string, we have no completion. */
  if( ( str == NULL ) || ( strlen( str ) == 0 ) )
    return( NULL );

  /* Initialize. */
  buffer[ 0 ] = '\0';

  /* Try to parse the string with time completion. */
  status = TimMakeDateFromString( &conv_time, str );

  if( status != TIM_OK )
    return( NULL );

  /* Convert the time back to a string. */
  ConvertToDateString( tbox, conv_time, buffer, TIME_STRING_BUFFER_LEN - 1 );

  if( strlen( buffer ) == 0 )
    return( NULL );

  return_str = XtMalloc( strlen( buffer ) + 1 );
  strcpy( return_str, buffer );


  return( return_str );

} /* CompleteDateString */


/*----------------------------------------------------------------------*/

static XmUbTimeBoxStatus
  CompleteEndDate( XmUbTimeBoxWidget  tbox )
{
  /* Variables. */
  char               *completed_str;
  Boolean            done;
  XmUbTimeBoxStatus  status;
  char               *str;

  /* Code. */

  /* Extract the date string. */
  status = XmUbTimeBoxGetEndDateString( (Widget) tbox, &str );
  if( status != TBOX_OK )
    return( status );

  /* Try to complete as keyword. */
  done = CompleteEndDateKeyword( tbox, str );
  if( !done ){

    /* If not successful, try numeric completion. */
    completed_str = CompleteDateString( tbox, str );

    if( completed_str == NULL )
      status = TBOX_NO_COMPLETION;

    else {
      status = XmUbTimeBoxSetEndDateString( (Widget) tbox, completed_str );
      XtFree( completed_str );
    }

  }

  /* Cleanup. */
  if( str != NULL )
    XtFree( str );


  return( status );

} /* CompleteEndDate */


/*----------------------------------------------------------------------*/

static Boolean
  CompleteEndDateKeyword( XmUbTimeBoxWidget  tbox,
                          char               *str )
{
#define COMPLETE_END_DATE_NOOP        0
#define COMPLETE_END_DATE_ADD_WEEKS   1
#define COMPLETE_END_DATE_ADD_MONTHS  2
#define COMPLETE_END_DATE_ADD_DAYS    3
#define COMPLETE_END_DATE_ADD_YEARS   4

  /* Variables. */
  TIM_TIME_REF       conv_time;
  int                days, months, weeks, years;
  char               keyword[ 20 ];
  int                matches;
  char               pattern[ MAX_PATTERN_LENGTH ];
  Boolean            ok;
  int                operation = COMPLETE_END_DATE_NOOP;
  XmUbTimeBoxStatus  status;

  /* Code. */

  /* For empty string, we have no completion. */
  if( ( str == NULL ) || ( strlen( str ) == 0 ) )
    return( False );

  /* Try different completion schemes. */
  sprintf( pattern, "+%s[%s]", "%d%", tbox -> tbox.week_marker );
  matches = sscanf( str, pattern, &weeks, keyword );
  if( matches == 2 ){

    if( weeks < 0 )
      return( False );

    operation = COMPLETE_END_DATE_ADD_WEEKS;
  }

  if( operation == COMPLETE_END_DATE_NOOP ){
    sprintf( pattern, "+%s[%s]", "%d%", tbox -> tbox.month_marker );
    matches = sscanf( str, pattern, &months, keyword );
    if( matches == 2 ){

      if( months < 0 )
        return( False );

      operation = COMPLETE_END_DATE_ADD_MONTHS;
    }
  }

  if( operation == COMPLETE_END_DATE_NOOP ){
    sprintf( pattern, "+%s[%s]", "%d%", tbox -> tbox.day_marker );
    matches = sscanf( str, pattern, &days, keyword );
    if( matches == 2 ){

      if( days < 0 )
        return( False );

      operation = COMPLETE_END_DATE_ADD_DAYS;
    }
  }

  if( operation == COMPLETE_END_DATE_NOOP ){
    sprintf( pattern, "+%s[%s]", "%d%", tbox -> tbox.year_marker );
    matches = sscanf( str, pattern, &years, keyword );
    if( matches == 2 ){

      if( days < 0 )
        return( False );

      operation = COMPLETE_END_DATE_ADD_YEARS;
    }
  }

  /* No keyword means add days. */
  if( operation == COMPLETE_END_DATE_NOOP ){
    matches = sscanf( str, "+%d", &days );
    if( matches == 1 ){

      if( days < 0 )
        return( False );

      operation = COMPLETE_END_DATE_ADD_DAYS;
    }
  }

  if( operation == COMPLETE_END_DATE_NOOP )
    return( False );


  /* We must have a date to start from. */
  status = XmUbTimeBoxGetStartDate( (Widget) tbox, &conv_time );
  if( status != TBOX_OK )
    return( False );

  /* We have a valid start date. Complete the desired operation. */
  switch( operation ){

    case COMPLETE_END_DATE_ADD_WEEKS:
      TimAddDays( &conv_time, 7 * weeks );
      break;

    case COMPLETE_END_DATE_ADD_MONTHS:
      TimAddMonths( &conv_time, months );
      break;

    case COMPLETE_END_DATE_ADD_DAYS:
      TimAddDays( &conv_time, days );
      break;

    case COMPLETE_END_DATE_ADD_YEARS:
      TimAddYears( &conv_time, years );
      break;

    default:
      return( NULL );
  }


  /* Set the new time, both date and time. */
  (void) XmUbTimeBoxSetEndDate( (Widget) tbox, conv_time );
  (void) XmUbTimeBoxSetEndTime( (Widget) tbox, conv_time );


  return( True );

#undef COMPLETE_END_DATE_NOOP        
#undef COMPLETE_END_DATE_ADD_WEEKS   
#undef COMPLETE_END_DATE_ADD_MONTHS  
#undef COMPLETE_END_DATE_ADD_DAYS    
#undef COMPLETE_END_DATE_ADD_YEARS 
} /* CompleteEndDateKeyword */


/*----------------------------------------------------------------------*/

static XmUbTimeBoxStatus
  CompleteEndTime( XmUbTimeBoxWidget  tbox )
{
  /* Variables. */
  char               *completed_str;
  Boolean            done;
  char               *str;
  XmUbTimeBoxStatus  status;

  /* Code. */

  /* Extract the time string. */
  status = XmUbTimeBoxGetEndTimeString( (Widget) tbox, &str );
  if( status != TBOX_OK )
    return( status );

  /* Try to complete as keyword. */
  done = CompleteEndTimeKeyword( tbox, str );

  if( !done ){

    /* If not successful, try numeric completion. */
    completed_str = CompleteTimeString( tbox, str );

    if( completed_str == NULL )
      status = TBOX_NO_COMPLETION;

    else {
      status = XmUbTimeBoxSetEndTimeString( (Widget) tbox, completed_str );
      XtFree( completed_str );
    }

  }

  /* Cleanup. */
  if( str != NULL )
    XtFree( str );


  return( status );

} /* CompleteEndTime */


/*----------------------------------------------------------------------*/

static Boolean
  CompleteEndTimeKeyword( XmUbTimeBoxWidget  tbox,
                          char               *str )
{
#define COMPLETE_END_TIME_NOOP         0
#define COMPLETE_END_TIME_ADD_HOURS    1
#define COMPLETE_END_TIME_ADD_MINUTES  2

  /* Variables. */
  TIM_TIME_REF  conv_time;
  int           hours, minutes, secs;
  char          keyword[ 20 ];
  int           matches;
  Boolean       ok;
  int           operation = COMPLETE_END_TIME_NOOP;
  char          pattern[ MAX_PATTERN_LENGTH ];
  XmUbTimeBoxStatus  status;

  /* Code. */

  /* For empty string, we have no completion. */
  if( ( str == NULL ) || ( strlen( str ) == 0 ) )
    return( False );

  /* Try different completion schemes. */
  sprintf( pattern, "+%s[%s]", "%d%", tbox -> tbox.hour_marker );
  matches = sscanf( str, pattern, &hours, keyword );
  if( matches == 2 ){

    if( hours < 0 )
      return( False );

    operation = COMPLETE_END_TIME_ADD_HOURS;
  }

  if( operation == COMPLETE_END_TIME_NOOP ){
    sprintf( pattern, "+%s[%s]", "%d%", tbox -> tbox.minute_marker );
    matches = sscanf( str, pattern, &minutes, keyword );
    if( matches == 2 ){

      if( minutes < 0 )
        return( False );

      operation = COMPLETE_END_TIME_ADD_MINUTES;
    }
  }

  /* No keyword means add hours. */
  if( operation == COMPLETE_END_TIME_NOOP ){
    matches = sscanf( str, "+%d", &hours );
    if( matches == 1 ){

      if( hours < 0 )
        return( False );

      operation = COMPLETE_END_TIME_ADD_HOURS;
    }
  }

  if( operation == COMPLETE_END_TIME_NOOP )
    return( False );


  /* We must have a time to start from. */
  status = XmUbTimeBoxGetStartTime( (Widget) tbox, &conv_time );
  if( status != TBOX_OK )
    return( False );

  /* We have a valid start time. Complete the desired operation. */
  switch( operation ){

    case COMPLETE_END_TIME_ADD_HOURS:
      TimAddHours( &conv_time, hours );
      break;

    case COMPLETE_END_TIME_ADD_MINUTES:
      TimAddMinutes( &conv_time, minutes );
      break;

    default:
      return( False );
  }

  /* Set the new time, both date and time. */
  (void) XmUbTimeBoxSetEndTime( (Widget) tbox, conv_time );

  
  return( True );

#undef COMPLETE_END_TIME_NOOP         
#undef COMPLETE_END_TIME_ADD_HOURS    
#undef COMPLETE_END_TIME_ADD_MINUTES  
} /* CompleteEndTimeKeyword */


/*----------------------------------------------------------------------*/

static XmUbTimeBoxStatus
  CompleteStartDate( XmUbTimeBoxWidget  tbox )
{
  /* Variables. */
  char               *completed_str;
  XmUbTimeBoxStatus  status;
  char               *str;

  /* Code. */

  /* Extract the date string. */
  status = XmUbTimeBoxGetStartDateString( (Widget) tbox, &str );
  if( status != TBOX_OK )
    return( status );

  completed_str = CompleteDateString( tbox, str );

  if( completed_str == NULL )
    status = TBOX_NO_COMPLETION;

  else {
    status = XmUbTimeBoxSetStartDateString( (Widget) tbox, completed_str );
    XtFree( completed_str );
  }

  /* Cleanup. */
  if( str != NULL )
    XtFree( str );


  return( status );

} /* CompleteStartDate */


/*----------------------------------------------------------------------*/

static XmUbTimeBoxStatus
  CompleteStartTime( XmUbTimeBoxWidget  tbox )
{
  /* Variables. */
  char               *completed_str;
  XmUbTimeBoxStatus  status;
  char               *str;

  /* Code. */

  /* Extract the time string. */
  status = XmUbTimeBoxGetStartTimeString( (Widget) tbox, &str );
  if( status != TBOX_OK )
    return( status );

  completed_str = CompleteTimeString( tbox, str );

  if( completed_str == NULL )
    status = TBOX_NO_COMPLETION;

  else {
    status = XmUbTimeBoxSetStartTimeString( (Widget) tbox, completed_str );
    XtFree( completed_str );
  }

  /* Cleanup. */
  if( str != NULL )
    XtFree( str );


  return( status );

} /* CompleteStartTime */


/*----------------------------------------------------------------------*/

static char
  *CompleteTimeString( XmUbTimeBoxWidget  tbox,
                       char               *str )
{
  /* Variables. */
  char             buffer[ TIME_STRING_BUFFER_LEN ];
  TIM_TIME_REF     conv_time;
  int              hours;
  int              matches;
  char             *return_str;
  TIM_STATUS_TYPE  status;

  /* Code. */

  /* For empty string, we have no completion. */
  if( ( str == NULL ) || ( strlen( str ) == 0 ) )
    return( NULL );

  /* Initialize. */
  buffer[ 0 ] = '\0';

  /* Try to parse the string with time completion. */
  status = TimMakeTimeFromString( &conv_time, str );

  if( status != TIM_OK ){

    /* Allow expansion of hours. Not yet in time package. */
    matches = sscanf( str, "%d", &hours );
    if( matches == 1 )
      conv_time = TimMakeTime( TIM_NOW, TIM_NOW, TIM_NOW, hours, 0, 0 );
    else
      return( NULL );
  }

  /* Convert the time back to a string. */
  ConvertToTimeString( tbox, conv_time, buffer, TIME_STRING_BUFFER_LEN - 1 );

  if( strlen( buffer ) == 0 )
    return( NULL );

  return_str = XtMalloc( strlen( buffer ) + 1 );
  strcpy( return_str, buffer );


  return( return_str );

} /* CompleteTimeString */


/*----------------------------------------------------------------------*/

static void
  ConvertToDateString( XmUbTimeBoxWidget  tbox,
                       time_t             time,
                       char               *buffer,
                       int                buf_len )
{

  /* Code. */

  TimFormatDate( time, buffer, buf_len );


  return;

} /* ConvertToDateString */


/*----------------------------------------------------------------------*/

static void
  ConvertToMenuPosition( XrmValue   *args,
                         Cardinal   *num_args,
                         XrmValue   *from,
                         XrmValue   *to )
{

  /* Variables. */
  static menuPositionType  conv_menu_position;


  /* Code. */

  if( *num_args != 0 )
    XtWarningMsg( "wrongParameters", "ConvertToMenuPosition", 
      "XtToolkitError", "Conversion needs no extra arguments", 
      (String *) NULL, (Cardinal *) NULL );

  if( strcmp( (char *) from -> addr, "POSITION_FIRST" ) == 0 )
    conv_menu_position = XmUbPOSITION_FIRST;

  else if( strcmp( (char *) from -> addr, "POSITION_LAST" ) == 0 )
    conv_menu_position = XmUbPOSITION_LAST;

  else {
    XtWarningMsg( "wrongParameters", "ConvertToMenuPosition", 
      "XtToolkitError", "Cannot convert to type menuPositionType",
      (String *) NULL, (Cardinal *) 0 );
    conv_menu_position = XmUbPOSITION_FIRST;
  }

  ( *to ).size = sizeof( menuPositionType );
  ( *to ).addr = (XtPointer) &conv_menu_position;


  return;

} /* ConvertToMenuPosition */


/*----------------------------------------------------------------------*/

static void
  ConvertToTimeString( XmUbTimeBoxWidget  tbox,
                       time_t             time,
                       char               *buffer,
                       int                buf_len )
{

  /* Code. */

  TimFormatTime( time, buffer, buf_len );


  return;

} /* ConvertToTimeString */


/*----------------------------------------------------------------------*/

static void
  ConvertToWidgetFormat( XrmValue   *args,
                         Cardinal   *num_args,
                         XrmValue   *from,
                         XrmValue   *to )
{

  /* Variables. */
  static widgetFormatType  conv_widget_format;


  /* Code. */

  if( *num_args != 0 )
    XtWarningMsg( "wrongParameters", "ConvertToWidgetFormat", 
      "XtToolkitError", "Conversion needs no extra arguments", 
      (String *) NULL, (Cardinal *) NULL );

  if( strcmp( (char *) from -> addr, "TB_FORMAT_D" ) == 0 )
    conv_widget_format = XmUbTB_FORMAT_D;

  else if( strcmp( (char *) from -> addr, "TB_FORMAT_T" ) == 0 )
    conv_widget_format = XmUbTB_FORMAT_T;

  else if( strcmp( (char *) from -> addr, "TB_FORMAT_DT" ) == 0 )
    conv_widget_format = XmUbTB_FORMAT_DT;

  else if( strcmp( (char *) from -> addr, "TB_FORMAT_DD" ) == 0 )
    conv_widget_format = XmUbTB_FORMAT_DD;

  else if( strcmp( (char *) from -> addr, "TB_FORMAT_DTDT" ) == 0 )
    conv_widget_format = XmUbTB_FORMAT_DTDT;

  else if( strcmp( (char *) from -> addr, "TB_FORMAT_DTT" ) == 0 )
    conv_widget_format = XmUbTB_FORMAT_DTT;

  else if( strcmp( (char *) from -> addr, "TB_FORMAT_TT" ) == 0 )
    conv_widget_format = XmUbTB_FORMAT_TT;


  else {
    XtWarningMsg( "wrongParameters", "ConvertToWidgetFormat", 
      "XtToolkitError", "Cannot convert to type widgetFormatType",
      (String *) NULL, (Cardinal *) 0 );
    conv_widget_format = XmUbTB_FORMAT_DTDT;
  }

  ( *to ).size = sizeof( widgetFormatType );
  ( *to ).addr = (XtPointer) &conv_widget_format;


  return;

} /* ConvertToWidgetFormat */


/*----------------------------------------------------------------------*/

static void
  CreateInternalWidgets( XmUbTimeBoxWidget  tbox )
{

  /* Variables. */
  char    *name;
  String  wname;

  /* Code. */

  /* Get the name of the "parent" widget. */
  wname = XtName( tbox );

  name = XtMalloc( strlen( wname ) + 3 );

  /* Create the menu. */
  if( tbox -> tbox.menu_enabled )
    CreateMenu( tbox, wname );

  /* First date field. */
  if( ( tbox -> tbox.widget_format == XmUbTB_FORMAT_D )    ||
      ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DT )   ||
      ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DD )   ||
      ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DTDT ) ||
      ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DTT ) ){

    sprintf( name, "%sFd", wname );

    tbox -> tbox.internal_children[ XmUbTB_CHILD_START_DATE ] = 
      CreateDateField( tbox, name );

    XtManageChild( tbox -> tbox.internal_children[ XmUbTB_CHILD_START_DATE ] );
  }

  /* First time field. */
  if( ( tbox -> tbox.widget_format == XmUbTB_FORMAT_T )    ||
      ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DT )   ||
      ( tbox -> tbox.widget_format == XmUbTB_FORMAT_TT )   ||
      ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DTDT ) ||
      ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DTT ) ){

    sprintf( name, "%sFt", wname );

    tbox -> tbox.internal_children[ XmUbTB_CHILD_START_TIME ] = 
      CreateTimeField( tbox, name );

    XtManageChild( tbox -> tbox.internal_children[ XmUbTB_CHILD_START_TIME ] );
  }


  /* Second date field. */
  if( ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DD )   ||
      ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DTDT ) ) {

    sprintf( name, "%sSd", wname );

    tbox -> tbox.internal_children[ XmUbTB_CHILD_END_DATE ] = 
      CreateDateField( tbox, name );

    XtManageChild( tbox -> tbox.internal_children[ XmUbTB_CHILD_END_DATE ] );
  }

  /* Second time field. */
  if( ( tbox -> tbox.widget_format == XmUbTB_FORMAT_TT )   ||
      ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DTDT ) ||
      ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DTT ) ){

    sprintf( name, "%sSt", wname );

    tbox -> tbox.internal_children[ XmUbTB_CHILD_END_TIME ] = 
      CreateTimeField( tbox, name );

    XtManageChild( tbox -> tbox.internal_children[ XmUbTB_CHILD_END_TIME ] );
  }

  /* Cleanup. */
  XtFree( name );



  return;

} /* CreateInternalWidgets */


/*----------------------------------------------------------------------*/

static Widget
  CreateDateField( XmUbTimeBoxWidget  tbox,
                   char               *name )
{

  /* Variables. */
  Arg           args[ 5 ];
  short         columns;
  Atom          compound_text_atom;
  Atom          date_transfer_atom;
  Cardinal      n;
  Atom          targets[ 2 ];
  char          text_buffer[ TEXT_BUFFER_LENGTH ];
  TIM_TIME_REF  time;
  Widget        w;

  /* Code. */

  /* Find out how long the longest text in the time field is. */
  /* Assume that Dec. 31 yields the longest text. */
  time = TimMakeTime( 1970, 12, 31, 23, 59, 59 );
  TimFormatDate( time, text_buffer, TEXT_BUFFER_LENGTH - 1 );
  columns = (short) ( strlen( text_buffer ) + 1 );

  n = 0;
  XtSetArg( args[ n ], XmNeditMode, XmSINGLE_LINE_EDIT ); n++;
  XtSetArg( args[ n ], XmNcolumns, columns ); n++;
  XtSetArg( args[ n ], XmNtraversalOn, True ); n++;

  w = XmCreateText( (Widget) tbox, name, args, n );

  /* We want to be able to pop up our own date selection dialog. */
  XtOverrideTranslations( w, dateTextTranslations ); 

  /* We want the completion callback when losing focus. */
  XtAddCallback( w, XmNlosingFocusCallback, 
                 (XtCallbackProc) DateCompletionCB, (XtPointer) tbox );

  /* Select all text when gaining focus. */
  XtAddCallback( w, XmNfocusCallback, 
                 (XtCallbackProc) SelectAllTextCB, (XtPointer) tbox );

  XtAddCallback( w, XmNactivateCallback,
                 (XtCallbackProc) ActivateCB, (XtPointer) tbox );

  XtAddCallback( w, XmNvalueChangedCallback,
                 (XtCallbackProc) ValueChangedCB, (XtPointer) tbox );

#if XmVersion > 1001
  /* We want to enable date transfer in drag and drop format. */
  compound_text_atom = XmInternAtom( XtDisplay( w ), "COMPOUND_TEXT", False );
  date_transfer_atom = XmInternAtom( XtDisplay( w ), XmUbDATE_TRANSFER, 
                                     False );

  targets[ 0 ] = date_transfer_atom;
  targets[ 1 ] = compound_text_atom;
  
  n = 0;
  XtSetArg( args[ n ], XmNimportTargets, targets ); n++;
  XtSetArg( args[ n ], XmNnumImportTargets, XtNumber( targets ) ); n++;
  XtSetArg( args[ n ], XmNdropProc, DateDropCB ); n++;

  XmDropSiteUpdate( w, args, n );

#endif

  return( w );

} /* CreateDateField */


/*----------------------------------------------------------------------*/

static void
  CreateDateSelectionPopup( XmUbTimeBoxWidget  tbox )
{
  /* Variables. */
  Arg       args[ 8 ];
  Widget    month_display;
  Cardinal  n;
  char      *name_buffer;
  Widget    shell;
  String    wname;
  XmString  xm, mxm, wxm;

  /* Code. */

  /* Reset flag that stop child creation temporarily. */
  tbox -> tbox.internal_widgets_created = False;

  /* Get the name of the "parent" widget. */
  wname = XtName( tbox );

  name_buffer = XtMalloc( strlen( wname ) + 8 );

  /* The dialog itself. */
  sprintf( name_buffer, "%sDtDi", wname );
  /* We don't need any OK or Cancel buttons, so create a normal form dialog. */

  n = 0;
  tbox -> tbox.date_selection_popup = XmCreateFormDialog(
    (Widget) tbox, name_buffer, args, n );

  /* Set resources on the shell. */
  n = 0;
  XtSetArg( args[ n ], XmNdeleteResponse, XmDO_NOTHING ); n++;
  if( tbox -> tbox.date_selection_title != NULL ){
    XtSetArg( args[ n ], XmNtitle, tbox -> tbox.date_selection_title ); n++;
  } else {
    XtSetArg( args[ n ], XmNtitle, default_date_selection_title ); n++;
  }

  XtSetValues( XtParent( tbox -> tbox.date_selection_popup ), args, n );

  /* We must catch close events on the shell and prevent it from being
     destroyed. */
  XmAddProtocolCallback( XtParent( tbox -> tbox.date_selection_popup ),
                         protocol_atom, delete_atom,
                         (XtCallbackProc) PopdownDateSelectionDialogCB,
                         (XtPointer) tbox );

  sprintf( name_buffer, "%sMd", wname );
  n = 0;
  XtSetArg( args[ n ], XmNmarginHeight, 10 ); n++;
  XtSetArg( args[ n ], XmNmarginWidth, 5 ); n++;

  if( tbox -> tbox.year_label == NULL )
    xm = XmStringCreateLtoR( default_year_string, XmSTRING_DEFAULT_CHARSET );
  else
    xm = tbox -> tbox.year_label;
  XtSetArg( args[ n ], XmUbNmdiYearLabel, xm ); n++;

  if( tbox -> tbox.month_label == NULL )
    mxm = XmStringCreateLtoR( default_month_string, XmSTRING_DEFAULT_CHARSET );
  else
    mxm = tbox -> tbox.month_label;
  XtSetArg( args[ n ], XmUbNmdiMonthLabel, mxm ); n++;

  if( tbox -> tbox.week_label == NULL )
    wxm = XmStringCreateLtoR( default_week_string, XmSTRING_DEFAULT_CHARSET );
  else
    wxm = tbox -> tbox.week_label;
  XtSetArg( args[ n ], XmUbNmdiWeekLabel, wxm ); n++;


  month_display = XmUbCreateMonthDisplay( 
    tbox -> tbox.date_selection_popup, name_buffer, args, n );

  if( tbox -> tbox.year_label == NULL )
    XmStringFree( xm );
  if( tbox -> tbox.month_label == NULL )
    XmStringFree( mxm );
  if( tbox -> tbox.week_label == NULL )
    XmStringFree( wxm );

  /* DateSelectedInDialogCB must also pop the dialog down. If we add both
     here it is not guaranteed that the dialog exists when our selection
     callback is called. */
  XtAddCallback( month_display, XmUbNmdiDateSelectedCallback,
                 (XtCallbackProc) DateSelectedInDialogCB, (XtPointer) tbox );

  n = 0;
  XtSetArg( args[ n ], XmNtopAttachment, XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
  XtSetValues( month_display, args, n );

  /* For own callbacks to take effect. */
  XmUbMonthDisplaySetMonth( month_display, 0, 0 );

  XtManageChild( month_display );

  /* Cleanup. */
  XtFree( name_buffer );

  /* Back to stopping child widgets. */
  tbox -> tbox.internal_widgets_created = True;


  return;

} /* CreateDateSelectionPopup */


/*----------------------------------------------------------------------*/

static void
  CreateMenu( XmUbTimeBoxWidget  tbox,
              String             base_name )
{

  /* Variables. */
  Arg       args[ 8 ];
  Cardinal  n;
  char      *name;


  /* Code. */

  /* Allocate room enough for the possible default names. */
  name = XtMalloc( strlen( base_name ) + 10 );

  /* Must create a menu bar, since a cascade button will only accept such a
     parent. */
  sprintf( name, "%sBr", base_name );
  n = 0;
  XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;
  XtSetArg( args[ n ], XmNmarginWidth, 0 ); n++;
  XtSetArg( args[ n ], XmNspacing, 0 ); n++;
  XtSetArg( args[ n ], XmNtraversalOn, True ); n++;
  tbox -> tbox.internal_children[ XmUbTB_CHILD_MENU_BAR ] =
    XmCreateMenuBar( (Widget) tbox, name, args, n );

  /* Pulldown menu. */
  sprintf( name, "%sPd", base_name );
  n = 0;
  tbox -> tbox.internal_children[ XmUbTB_CHILD_MENU_PD ] =
    XmCreatePulldownMenu( 
      tbox -> tbox.internal_children[ XmUbTB_CHILD_MENU_BAR ], name, args, n );

  /* The cascade button. */
  sprintf( name, "%sCa", base_name );
  n = 0;

  if( tbox -> tbox.menu_label != NULL ){
    XtSetArg( args[ n ], XmNlabelString, tbox -> tbox.menu_label ); n++;
    XtSetArg( args[ n ], XmNlabelType, XmSTRING ); n++;

  } else if( tbox -> tbox.menu_pixmap != XmUNSPECIFIED_PIXMAP ){
    XtSetArg( args[ n ], XmNlabelPixmap, tbox -> tbox.menu_pixmap ); n++;
    XtSetArg( args[ n ], XmNlabelType, XmPIXMAP ); n++;

  } else {
    tbox -> tbox.menu_label = 
      XmStringCreateLtoR( "-", XmSTRING_DEFAULT_CHARSET );

    XtSetArg( args[ n ], XmNlabelString, tbox -> tbox.menu_label ); n++;
    XtSetArg( args[ n ], XmNlabelType, XmSTRING ); n++;
    XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;

  }

  XtSetArg( args[ n ], XmNalignment, XmALIGNMENT_CENTER ); n++;
  XtSetArg( args[ n ], XmNsubMenuId, 
            tbox -> tbox.internal_children[ XmUbTB_CHILD_MENU_PD ] ); n++;
  XtSetArg( args[ n ], XmNshadowThickness, 0 ); n++;
  XtSetArg( args[ n ], XmNtraversalOn, True ); n++;

  tbox -> tbox.internal_children[ XmUbTB_CHILD_MENU_CASC ] =
    XmCreateCascadeButton( 
      tbox -> tbox.internal_children[ XmUbTB_CHILD_MENU_BAR ], name, args, n );
  
  /* The menu items. */ 
  if( tbox -> tbox.menu_items == NULL ){

    XmUbMenuItem  default_menu_items[ MAX_NO_DEFAULT_MENU_ITEMS ];
    int           num_created;

    /* Initialize the array. */
    InitializeDefaultMenu( tbox, default_menu_items, &num_created );

    CreateMenuItems( 
      tbox, tbox -> tbox.internal_children[ XmUbTB_CHILD_MENU_PD ],
      default_menu_items, num_created );

  } else
    CreateMenuItems( 
      tbox, tbox -> tbox.internal_children[ XmUbTB_CHILD_MENU_PD ],
      tbox -> tbox.menu_items, tbox -> tbox.num_menu_items );


  XtManageChild( tbox -> tbox.internal_children[ XmUbTB_CHILD_MENU_CASC ] );
  XtManageChild( tbox -> tbox.internal_children[ XmUbTB_CHILD_MENU_BAR ] );

  XtFree( name );


  return;

} /* CreateMenu */


/*----------------------------------------------------------------------*/

static void
  CreateMenuItems( XmUbTimeBoxWidget  tbox,
                   Widget             parent,
                   XmUbMenuItemRef    items,
                   int                num_items )
{
  /* Variables. */
  Arg       args[ 5 ];
  char      *default_name;
  int       index;
  Cardinal  n;
  Widget    w;

  /* Code. */

  default_name = XtMalloc( strlen( XtName( parent ) ) + 3 );
  sprintf( default_name, "%sMI", XtName( parent ) );

  for( index = 0; index < num_items; index ++ ){

    switch( items[ index ].item ){
      case XmUbITEM_SEPARATOR:
        n = 0;
        XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
        w = XmCreateSeparator( parent, 
          ( items[ index ].name == NULL ? default_name : items[ index ].name ),
          args, n );

        break;

      case XmUbITEM_PUSH_BUTTON:
        /* Set the function to be called as user data. */
        n = 0;
        XtSetArg( args[ n ], XmNlabelString, items[ index ].label ); n++;
        XtSetArg( args[ n ], XmNuserData,    (XtPointer) tbox ); n++;
        w = XmCreatePushButton( parent,
          ( items[ index ].name == NULL ? default_name : items[ index ].name ),
          args, n );

        /* For predefined actions, call common routine. 
           For other actions, register as callbacks directly. */
        if( ( items[ index ].proc != XmUbTB_NO_ACTION ) &&
            ( (int) items[ index ].proc <= (int) XmUbTB_LAST_PREDEF_ACTION ) )
          XtAddCallback( w, XmNactivateCallback, 
                         (XtCallbackProc) MenuItemActivatedCB, 
                         (XtPointer) items[ index ].proc );

        else if( items[ index ].proc != XmUbTB_NO_ACTION )
          XtAddCallback( w, XmNactivateCallback, 
                         items[ index ].proc, items[ index ].client_data );

        break;


      default:
        /* Insert error message here !!!!!!! */
        return;

    }

    XtManageChild( w );

  } /* for */


  return;

} /* CreateMenuItems */


/*----------------------------------------------------------------------*/

static Widget
  CreateTimeField( XmUbTimeBoxWidget  tbox,
                   char               *name )
{

  /* Variables. */
  Arg           args[ 5 ];
  short         columns;
  int           length;
  Cardinal      n;
  char          text_buffer[ TEXT_BUFFER_LENGTH ];
  TIM_TIME_REF  time;
  Widget        w;

  /* Code. */

  /* Find out how long the longest text in the time field is. */
  /* Assume that 23:59 yields the longest text. */
  time = TimMakeTime( 1970, 1, 1, 23, 59, 59 );
  TimFormatTime( time, text_buffer, TEXT_BUFFER_LENGTH - 1 );
  columns = (short) ( strlen( text_buffer ) + 1 );

  n = 0;
  XtSetArg( args[ n ], XmNeditMode, XmSINGLE_LINE_EDIT ); n++;
  XtSetArg( args[ n ], XmNcolumns, columns ); n++;
  XtSetArg( args[ n ], XmNtraversalOn, True ); n++;

  w = XmCreateText( (Widget) tbox, name, args, n );

  /* We want to be able to pop up our own time selection dialog. */
  XtOverrideTranslations( w, timeTextTranslations ); 
/*  XtAugmentTranslations( w, timeTextTranslations ); */

  /* We want the completion callback when losing focus. */
  XtAddCallback( w, XmNlosingFocusCallback, 
                 (XtCallbackProc) TimeCompletionCB, (XtPointer) tbox );

  /* Select all text when gaining focus. */
  XtAddCallback( w, XmNfocusCallback, 
                 (XtCallbackProc) SelectAllTextCB, (XtPointer) tbox );

  XtAddCallback( w, XmNactivateCallback,
                 (XtCallbackProc) ActivateCB, (XtPointer) tbox );

  XtAddCallback( w, XmNvalueChangedCallback,
                 (XtCallbackProc) ValueChangedCB, (XtPointer) tbox );


  return( w );

} /* CreateTimeField */


/*----------------------------------------------------------------------*/

static void
  CreateTimeSelectionPopup( XmUbTimeBoxWidget  tbox )
{
  /* Variables. */
  Arg        args[ 8 ];
  Widget     cancel;
  Widget     form;
  Dimension  height, max_height;
  Cardinal   n;
  char       *name_buffer;
  Widget     ok;
  Widget     sep;
  Widget     time_slider;
  Dimension  width, max_width;
  String     wname;

  /* Code. */

  /* Reset flag that stop child creation temporarily. */
  tbox -> tbox.internal_widgets_created = False;

  /* Get the name of the "parent" widget. */
  wname = XtName( tbox );

  name_buffer = XtMalloc( strlen( wname ) + 6 );

  /* The dialog itself. */
  sprintf( name_buffer, "%sTDi", wname );

  n = 0;
  form = XmCreateFormDialog( (Widget) tbox, name_buffer, args, n );

  tbox -> tbox.time_selection_popup = form;

  /* In the form, we want a time slider, a separator and two buttons. */
  sprintf( name_buffer, "%sTs", wname );
  n = 0;
  XtSetArg( args[ n ], XmNmarginHeight, 10 ); n++;
  XtSetArg( args[ n ], XmNmarginWidth, 5 ); n++;
  XtSetArg( args[ n ], XmNwidth, 300 ); n++;

  time_slider = XmUbCreateTimeSlider( form, name_buffer, args, n );
  
  sprintf( name_buffer, "%sTSep", wname );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  sep = XmCreateSeparator( form, name_buffer, args, n );

  sprintf( name_buffer, "%sTok", wname );
  n = 0;
  XtSetArg( args[ n ], XmNlabelString, tbox -> tbox.ok_label ); n++;
  ok = XmCreatePushButton( form, name_buffer, args, n );

  sprintf( name_buffer, "%sTcan", wname );
  n = 0;
  XtSetArg( args[ n ], XmNlabelString, tbox -> tbox.cancel_label ); n++;
  cancel = XmCreatePushButton( form, name_buffer, args, n );

  /* Attach the components. */
  n = 0;
  XtSetArg( args[ n ], XmNtopAttachment, XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_WIDGET ); n++;
  XtSetArg( args[ n ], XmNbottomWidget, sep ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 3 ); n++;
  XtSetValues( time_slider, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_WIDGET ); n++;
  XtSetArg( args[ n ], XmNbottomWidget, ok ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 3 ); n++;
  XtSetValues( sep, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNleftOffset, 6 ); n++;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 3 ); n++;
  XtSetValues( ok, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNrightOffset, 6 ); n++;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 3 ); n++;
  XtSetValues( cancel, args, n );


  /* Set resources on the shell. */
  n = 0;
  XtSetArg( args[ n ], XmNdeleteResponse, XmDO_NOTHING ); n++;
  if( tbox -> tbox.time_selection_title != NULL ){
    XtSetArg( args[ n ], XmNtitle, tbox -> tbox.time_selection_title ); n++;
  } else {
    XtSetArg( args[ n ], XmNtitle, default_time_selection_title ); n++;
  }

  XtSetValues( XtParent( form ), args, n );

  /* We must catch close events on the shell and prevent it from being
     destroyed. */
  XmAddProtocolCallback( XtParent( form ), protocol_atom, delete_atom,
                         (XtCallbackProc) PopdownTimeSelectionDialogCB,
                         (XtPointer) tbox );

  /* Callbacks. */
  XtAddCallback( time_slider, XmNactivateCallback,
                 (XtCallbackProc) TimeSelectedInDialogCB, (XtPointer) tbox );
  XtAddCallback( ok, XmNactivateCallback,
                 (XtCallbackProc) TimeSelectedInDialogCB, (XtPointer) tbox );
  XtAddCallback( cancel, XmNactivateCallback,
                 (XtCallbackProc) PopdownTimeSelectionDialogCB, 
                 (XtPointer) tbox );


  /* Manage the internal widget (but not the form!) */
  XtManageChild( time_slider );
  XtManageChild( sep );
  XtManageChild( ok );
  XtManageChild( cancel );

  /* Set both buttons the same size. */
  n = 0;
  XtSetArg( args[ n ], XmNwidth, &width ); n++;
  XtSetArg( args[ n ], XmNheight, &height ); n++;
  XtGetValues( ok, args, n );

  max_width  = width;
  max_height = height;

  XtGetValues( cancel, args, n );

  if( width > max_width )
    max_width = width;
  if( height > max_height )
     max_height = height;
  
  n = 0;
  XtSetArg( args[ n ], XmNwidth, max_width ); n++;
  XtSetArg( args[ n ], XmNheight, max_height ); n++;
  XtSetValues( ok, args, n );
  XtSetValues( cancel, args, n );

  /* Cleanup. */
  XtFree( name_buffer );

  /* Back to stopping child widgets. */
  tbox -> tbox.internal_widgets_created = True;


  return;

} /* CreateTimeSelectionPopup */


/*----------------------------------------------------------------------*/

static void
  DateCompletionCB( Widget                      tw,
                    XmUbTimeBoxWidget           tbox,
                    XmTextVerifyCallbackStruct  *call_data )
{
  /* Variables. */
  int                child_index;
  Boolean            restart;
  XmUbTimeBoxStatus  status;

  /* Code. */

DateCompletionCB_START:

  /* Find child index for this field. */
  child_index = ChildIndex( tbox, tw );  

  /* Completion is different depending on which field we are dealing with. */
  switch( child_index ){

    case XmUbTB_CHILD_START_DATE:
      status = CompleteStartDate( tbox );
      break;


    case XmUbTB_CHILD_END_DATE:
      status = CompleteEndDate( tbox );
      break;

    default:
      /* This should not happen. */
      printf( "DateCompletionCB ERROR, called for wrong child.\n" );
      return;

  } /* switch */

  /* Issue the callback(s). */
  restart = TimeCompletionCallback( tw, child_index, tbox, 
				  (XmAnyCallbackStruct *) call_data, 
				  status );
  if( restart )
    goto DateCompletionCB_START;


  return;

} /* DateCompletionCB */


/*----------------------------------------------------------------------*/

#if XmVersion > 1001

static void
  DateDropCB( Widget                    w,
              XtPointer                 client_data,
              XmDropProcCallbackStruct  *drop_data )
{
  /* Variables. */
  Arg                     args[ 5 ];
  Atom                    compound_text_atom;
  Atom                    date_transfer_atom;
  Atom                    *export_list;
  int                     index;
  XmTextPosition          insert_pos;
  Cardinal                n;
  Cardinal                num_exports;
  int                     num_transfers = 0;
  XmDropTransferEntryRec  transfer_entries[ 2 ];
  XmDropTransferEntry     transfer_list;

  /* Code. */

  compound_text_atom = XmInternAtom( XtDisplay( w ), "COMPOUND_TEXT", False );
  date_transfer_atom = XmInternAtom( XtDisplay( w ), XmUbDATE_TRANSFER, 
                                     False );

  /* Set the transfer resources. */
  n = 0;

  /* Cancel the drop if the action is not a drop or if user wants a link. */
  if( ( drop_data -> dropAction != XmDROP ) ||
      ( drop_data -> operation  == XmDROP_LINK ) ){
    XtSetArg( args[ n ], XmNtransferStatus, XmTRANSFER_FAILURE ); n++;

  } else {
    /* Go on with the drop. Establish the transfer list and start transfer. */

    /* Get the exported targets for the drag context and select the
       desired one. */
    n = 0;
    XtSetArg( args[ n ], XmNexportTargets, &export_list ); n++;
    XtSetArg( args[ n ], XmNnumExportTargets, &num_exports ); n++;
    XtGetValues( drop_data -> dragContext, args, n );

    /* Our preferred format is date transfer. See if we can find it. */
    for( index = 0; index < num_exports; index ++ ){

      if( export_list[ index ] == date_transfer_atom ){
        transfer_entries[ 0 ].target      = date_transfer_atom;
        transfer_entries[ 0 ].client_data = (XtPointer) w;
        num_transfers = 1;

        break;
      }
    }

    /* Alternative format is compound text. */
    if( num_transfers == 0 ){

      for( index = 0; index < num_exports; index ++ ){

        if( export_list[ index ] == compound_text_atom ){
          transfer_entries[ 0 ].target      = compound_text_atom;
          transfer_entries[ 0 ].client_data = (XtPointer) w;
          num_transfers = 1;
 
          /* The text should be inserted at the drop position. */
          insert_pos = XmTextXYToPos( w, drop_data -> x, drop_data -> y );
          XmTextSetInsertionPosition( w, insert_pos );

          break;
        }
      }
    }

    transfer_list = transfer_entries;

    XtSetArg( args[ n ], XmNdropTransfers, transfer_list ); n++;
    XtSetArg( args[ n ], XmNnumDropTransfers, num_transfers ); n++;
    XtSetArg( args[ n ], XmNtransferProc, TransferDateField ); n++;
  }

  /* Start the transfer or cancel. */
  XmDropTransferStart( drop_data -> dragContext, args, n );

  
  return;

} /* DateDropCB */

#endif

/*----------------------------------------------------------------------*/

static void
  DeleteChild( Widget  widget )
{

  /* Variables. */
  int                index;
  XmUbTimeBoxWidget  tbox;

  /* Code. */

  tbox = (XmUbTimeBoxWidget) XtParent( widget );

  /* Clear the internal reference. */
  for( index = 0; index < NO_INTERNAL_CHILDREN; index ++ ){
    if( tbox -> tbox.internal_children[ index ] == widget ){
      tbox -> tbox.internal_children[ index ] = NULL;
      break;
    }
  }

  /* Perform the actual operation */
  (* ( (CompositeWidgetClass) (xmUbTimeBoxWidgetClass ->
     core_class.superclass) ) -> composite_class.delete_child ) ( widget );


  return;

} /* DeleteChild */


/*----------------------------------------------------------------------*/

static void 
  Destroy( Widget   widget )
{
  /* Variables. */
  XmUbTimeBoxWidget  tbox;

  /* Code. */

  tbox = (XmUbTimeBoxWidget) widget;

  /* Free the copied strings. */
  if( tbox -> tbox.year_marker != NULL )
    XtFree( tbox -> tbox.year_marker );
  if( tbox -> tbox.month_marker != NULL )
    XtFree( tbox -> tbox.month_marker );
  if( tbox -> tbox.week_marker != NULL )
    XtFree( tbox -> tbox.week_marker );
  if( tbox -> tbox.day_marker != NULL )
    XtFree( tbox -> tbox.day_marker );
  if( tbox -> tbox.hour_marker != NULL )
    XtFree( tbox -> tbox.hour_marker );
  if( tbox -> tbox.minute_marker != NULL )
    XtFree( tbox -> tbox.minute_marker );

  if( tbox -> tbox.time_selection_title != NULL )
    XtFree( tbox -> tbox.time_selection_title ); 
  if( tbox -> tbox.date_selection_title != NULL )
    XtFree( tbox -> tbox.date_selection_title ); 

  /* Free XmStrings. */
  if( tbox -> tbox.ok_label != NULL )
    XmStringFree( tbox -> tbox.ok_label );
  if( tbox -> tbox.cancel_label != NULL )
    XmStringFree( tbox -> tbox.cancel_label );
  if( tbox -> tbox.week_label != NULL )
    XmStringFree( tbox -> tbox.week_label );
  if( tbox -> tbox.month_label != NULL )
    XmStringFree( tbox -> tbox.month_label );
  if( tbox -> tbox.year_label != NULL )
    XmStringFree( tbox -> tbox.year_label );

  /* Remove callbacks. */
  XtRemoveAllCallbacks( widget, XmNactivateCallback );
  XtRemoveAllCallbacks( widget, XmNvalueChangedCallback );
  XtRemoveAllCallbacks( widget, XmUbNtboxCompletionCallback );


  return;

} /* Destroy */


/*----------------------------------------------------------------------*/

static void
  DoLayout( XmUbTimeBoxWidget  tbox,
            Widget             initiator,
            XtWidgetGeometry   *request,
            KidDimensionRec    sizes[] )
{
  /* Variables. */
  XtWidgetGeometry  desired;
  int               index;
  Widget            kid;
  Position          kid_x, kid_y;
  Dimension         spacing = 0;
  Position          x_pos;
  Position          y_pos;

  /* Code. */

  /* All positions and dimensions are in the sizes array. */
  if( tbox -> tbox.menu_enabled && 
      ( tbox -> tbox.internal_children[ XmUbTB_CHILD_MENU_BAR ] != NULL ) ){

    index = XmUbTB_CHILD_MENU_BAR;
    kid   = tbox -> tbox.internal_children[ index ];

    XtConfigureWidget( kid, sizes[ index ].x, sizes[ index ].y,
      sizes[ index ].width, sizes[ index ].height, kid -> core.border_width );
  }

  /* Loop for the text widgets. */
  for( index = XmUbTB_CHILD_START_DATE; index < NO_INTERNAL_CHILDREN; 
       index ++ ){

    kid = tbox -> tbox.internal_children[ index ];

    if( ( kid != NULL ) && XtIsManaged( kid ) )
      XtConfigureWidget( kid, sizes[ index ].x, sizes[ index ].y,
        sizes[ index ].width, sizes[ index ].height, 
        kid -> core.border_width );
  }


  return;

} /* DoLayout */


/*----------------------------------------------------------------------*/

static void
  DateSelectedInDialogCB( Widget                          md,
                          XmUbTimeBoxWidget               tbox,
                          XmUbMonthDisplayCallbackStruct  *call_data )
{
  /* Variables. */
  char                       buffer[ TIME_STRING_BUFFER_LEN ];
  XmUbTimeBoxCallbackStruct  cb;
  Boolean                    known_field = True;
  TIM_TIME_REF               time;

  /* Code. */

  /* Get the selected date. */
  time = TimMakeTime( call_data -> selected_year, call_data -> selected_month,
                      call_data -> selected_day, 0, 0, 0 );

  /* Create the string. */
  TimFormatDate( time, buffer, TIME_STRING_BUFFER_LEN - 1 );

  if( call_data -> range ){

    /* The first date is the start date. */
    (void) XmUbTimeBoxSetStartDateString( (Widget) tbox, buffer );

    /* Set the end date too. */
    time = TimMakeTime( call_data -> end_year, call_data -> end_month,
                        call_data -> end_day, 0, 0, 0 );
    TimFormatDate( time, buffer, TIME_STRING_BUFFER_LEN - 1 );

    (void) XmUbTimeBoxSetEndDateString( (Widget) tbox, buffer );

  } else {

    switch( tbox -> tbox.field_for_date_selection ){

      case XmUbTB_CHILD_START_DATE:
        (void) XmUbTimeBoxSetStartDateString( (Widget) tbox, buffer );
        break;

      case XmUbTB_CHILD_END_DATE:
        (void) XmUbTimeBoxSetEndDateString( (Widget) tbox, buffer );
        break;

      default:
        known_field = False;
        break;
    }
  }

  /* We have to call the callback to pop down the dialog here, since it 
     might destroy the widget. */
  PopdownDateSelectionDialogCB( md, tbox, (XmAnyCallbackStruct *) call_data );

  /* Call activate callback. */
  if( known_field && ( tbox -> tbox.activate_callback != NULL ) ){

    /* Set up callback structure. */

    cb.reason      = XmUbCR_DATE_PICKED;
    cb.event       = NULL;
    cb.child_index = tbox -> tbox.field_for_date_selection;
    cb.child       = tbox -> tbox.internal_children[ cb.child_index ];
    cb.range       = call_data -> range;

    XtCallCallbackList( (Widget) tbox, tbox -> tbox.activate_callback,
                        (XtPointer) &cb );
  }


  return;

} /* DateSelectedInDialogCB */


/*----------------------------------------------------------------------*/

static void
  FillInStartOfMonth( XmUbTimeBoxWidget  tbox )
{
  /* Variables. */
  TIM_TIME_REF  now;
  TIM_TIME_REF  tloc;
  int           month, year;

  /* Code. */

  /* Get the start date of the week and display it in the text field. */
  now  = TimMakeTimeNow();
  tloc = TimLocalTime( now );

  year  = TimIndexOfYear( tloc );
  month = TimIndexOfMonth( tloc );

  /* Get start of week. */
  tloc = TimMakeTime( year, month, 1, 0, 0, 0 );

  (void) XmUbTimeBoxSetStartDate( (Widget) tbox, tloc );
  (void) XmUbTimeBoxSetStartTime( (Widget) tbox, tloc );


  return;

} /* FillInStartOfMonth */


/*----------------------------------------------------------------------*/

static void
  FillInStartOfWeek( XmUbTimeBoxWidget  tbox )
{
  /* Variables. */
  TIM_TIME_REF     now;
  TIM_STATUS_TYPE  status;
  TIM_TIME_REF     tloc;
  int              week, year;

  /* Code. */

  /* Get the start date of the week and display it in the text field. */
  now  = TimMakeTimeNow();
  tloc = TimLocalTime( now );

  year = TimIndexOfYear( tloc );
  week = TimIndexOfWeek( tloc );

  /* Get start of week. */
  status = TimMakeTimeFromWeek( year, week, &tloc );
  if( status != TIM_OK )
    return;

  (void) XmUbTimeBoxSetStartDate( (Widget) tbox, tloc );
  (void) XmUbTimeBoxSetStartTime( (Widget) tbox, tloc );


  return;

} /* FillInStartOfWeek */


/*----------------------------------------------------------------------*/

static XtGeometryResult
  GeometryManager( Widget            widget,
                   XtWidgetGeometry  *request,
                   XtWidgetGeometry  *reply )
{

  XtWidgetGeometry   own_request;
  Dimension          old_width, old_height;
  Dimension          pref_height;
  Dimension          pref_width;
  KidDimensionRec    kids_sizes[ NO_INTERNAL_CHILDREN ];
  XtGeometryResult   result;
  XmUbTimeBoxWidget  tbox;

  /* Code. */

  tbox = (XmUbTimeBoxWidget) XtParent( widget );

  /* Find out how big the widget would be if the resize were allowed. */
  GetChildPrefSizes( tbox, NULL, request, kids_sizes );
  PrepareLayout( tbox, kids_sizes, &pref_width, &pref_height );

  /* If no change in dimensions, allow the request. */
  if( ( pref_width == tbox -> core.width ) && 
      ( pref_height == tbox -> core.height )){
    DoLayout( tbox, widget, request, kids_sizes );
    return XtGeometryYes;
  }

  /* We must ask our parent to resize us. */
  own_request.request_mode = CWWidth | CWHeight;
  own_request.width  = pref_width;
  own_request.height = pref_height;

  /* Save dimensions. */
  old_width  = tbox -> core.width;
  old_height = tbox -> core.height;

  tbox -> tbox.resize_called = False;

  /* We are not interested in any compromise geometry. */
  result = XtMakeGeometryRequest( (Widget) tbox, &own_request, NULL );

  /* Reset to old dimensions if request not granted. */
  if( result != XtGeometryYes ){
    tbox -> core.width  = old_width;
    tbox -> core.height = old_height;

  } else {
    if( !tbox -> tbox.resize_called )
      Resize( (Widget) tbox );
  }

  /* Always grant child's request. */
  return XtGeometryYes;

} /* GeometryManager */


/*----------------------------------------------------------------------*/

static void
  GetChildPrefSizes( XmUbTimeBoxWidget  tbox,
                     Widget             initiator,
                     XtWidgetGeometry   *request,
                     KidDimensionRec    sizes[] )
{
  /* Variables. */
  XtWidgetGeometry  desired;
  int               index;
  Widget            kid;

  /* Code. */

  /* Initialize. */
  for( index = 0; index < NO_INTERNAL_CHILDREN; index ++ ){
    sizes[ index ].width  = 0;
    sizes[ index ].height = 0;
    sizes[ index ].x      = 0;
    sizes[ index ].y      = 0;
  }

  /* Get the preferred sizes for the children. */
  for( index = 0; index < NO_INTERNAL_CHILDREN; index ++ ){

    /* The pulldown menu should not be considered. */
    if( ( index == XmUbTB_CHILD_MENU_PD ) ||
        ( index == XmUbTB_CHILD_MENU_CASC ) )
      continue;

    kid = tbox -> tbox.internal_children[ index ];

    if( ( kid != NULL ) && XtIsManaged( kid ) ){

      KidsPreferredGeometry( kid, initiator, request, &desired );

      sizes[ index ].width  = desired.width;
      sizes[ index ].height = desired.height;

    }

  } /* for */

  
  return;

} /* GetChildPrefSizes */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  GetChildString( XmUbTimeBoxWidget  tbox,
                  int                child, 
                  char               **str )
{
  /* Variables. */
  Widget  tw;

  /* Code. */

  tw = tbox -> tbox.internal_children[ child ];
  if( tw == NULL )
    return( TBOX_NO_FIELD );

  *str = XmTextGetString( tw );

  /* For an empty string, we return a status value. */
  if( strlen( *str ) == 0 ){
    XtFree( *str );
    return( TBOX_EMPTY );
  }

  return( TBOX_OK );

} /* GetChildString */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  GetStartDateTime( XmUbTimeBoxWidget  tbox,
                    Boolean            use_defaults,
                    TIM_TIME_REF       *date_time )
{
  /* Variables. */
  TIM_TIME_REF       date;
  XmUbTimeBoxStatus  status;
  TIM_TIME_REF       time;

  /* Code. */

  status = XmUbTimeBoxGetStartDate( (Widget) tbox, &date );
  if( status != TBOX_OK ){
    if( use_defaults )
      date = TimMakeTime( 1970, 1, 1, 0, 0, 0 );
    else
      return( status );
  }

  status = XmUbTimeBoxGetStartTime( (Widget) tbox, &time );
  if( status != TBOX_OK ){
    if( use_defaults )
      time = TimMakeTime( 1970, 1, 1, 0, 0, 0 );
    else
      return( status );
  }

  /* Construct combined time. */
  *date_time = CombineDateAndTime( date, time );  


  return( TBOX_OK );

} /* GetStartDateTime */


/*----------------------------------------------------------------------*/

static void
  GetValuesHook( Widget    w,
                 ArgList   args,
                 Cardinal  *num_args )
{
  /* Variables. */
  int                index;
  XmUbTimeBoxWidget  tbox;

  /* Code. */

  tbox = (XmUbTimeBoxWidget) w;

  /* Copy the XmStrings. */
  for( index = 0; index < *num_args; index ++ ){

    if( strcmp( args[ index ].name, XmNcancelLabelString ) == 0 ){
      * ( XmString *) ( args[ index ].value ) = 
        XmStringCopy( tbox -> tbox.cancel_label );

    } else if( strcmp( args[ index ].name, XmNokLabelString ) == 0 ){
      * ( XmString *) ( args[ index ].value ) = 
        XmStringCopy( tbox -> tbox.ok_label );

    } else if( strcmp( args[ index ].name, XmUbNtboxMonthLabel ) == 0 ){
      * ( XmString *) ( args[ index ].value ) = 
        XmStringCopy( tbox -> tbox.month_label );

    } else if( strcmp( args[ index ].name, XmUbNtboxWeekLabel ) == 0 ){
      * ( XmString *) ( args[ index ].value ) = 
        XmStringCopy( tbox -> tbox.week_label );

    } else if( strcmp( args[ index ].name, XmUbNtboxYearLabel ) == 0 ){
      * ( XmString *) ( args[ index ].value ) = 
        XmStringCopy( tbox -> tbox.year_label );

    } 
  }


  return;

} /* GetValuesHook */


/*----------------------------------------------------------------------*/

static void
  Initialize( Widget     treq,
              Widget     tnew,
              ArgList    args,
              Cardinal   *num_args )
{

  /* Variables. */
  int                index;
  KidDimensionRec    kids_sizes[ NO_INTERNAL_CHILDREN ];
  XmUbTimeBoxWidget  new;
  char               *newstr;
  Dimension          pref_width;
  Dimension          pref_height;
  char               *str;
  XmString           xm;

  /* Code. */

  new = (XmUbTimeBoxWidget) tnew;

  /* Initialize private fields. */
  for( index = 0; index < NO_INTERNAL_CHILDREN; index ++ )
    new -> tbox.internal_children[ index ] = NULL;

  new -> tbox.time_selection_popup = NULL;
  new -> tbox.date_selection_popup = NULL;

  new -> tbox.field_for_time_selection = XmUbTB_CHILD_START_TIME;
  new -> tbox.field_for_date_selection = XmUbTB_CHILD_START_DATE;

  /* Start size if none supplied. */
  if( new -> core.width == 0 )
    new -> core.width = 100;
  if( new -> core.height == 0 )
    new -> core.height = 100;

  /* The atoms must be set up here, since we need a display varaible for it. */
  if( ! context_dependent_done ){
    protocol_atom = XmInternAtom( XtDisplay( tnew ), "WM_PROTOCOLS", False );
    delete_atom   = XmInternAtom( XtDisplay( tnew ), 
      "WM_DELETE_WINDOW", False );

    context_dependent_done = True;
  }

  /* Copy XmStrings and/or use defaults. */
  if( new -> tbox.ok_label == NULL )
    xm = XmStringCreateLtoR( default_ok_string, XmSTRING_DEFAULT_CHARSET );
  else
    xm = XmStringCopy( new -> tbox.ok_label );
  new -> tbox.ok_label = xm;

  if( new -> tbox.cancel_label == NULL )
    xm = XmStringCreateLtoR( default_cancel_string, XmSTRING_DEFAULT_CHARSET );
  else
    xm = XmStringCopy( new -> tbox.cancel_label );
  new -> tbox.cancel_label = xm;

  if( new -> tbox.week_label != NULL ){
    xm = XmStringCopy( new -> tbox.week_label );
    new -> tbox.week_label = xm;
  }
  if( new -> tbox.month_label != NULL ){
    xm = XmStringCopy( new -> tbox.month_label );
    new -> tbox.month_label = xm;
  }
  if( new -> tbox.year_label != NULL ){
    xm = XmStringCopy( new -> tbox.year_label );
    new -> tbox.year_label = xm;
  }

  /* Copy title strings. */
  if( new -> tbox.time_selection_title != NULL ){
    str = XtMalloc( strlen( new -> tbox.time_selection_title ) + 1 );
    strcpy( str, new -> tbox.time_selection_title );
    new -> tbox.time_selection_title = str;
  }
  if( new -> tbox.date_selection_title != NULL ){
    str = XtMalloc( strlen( new -> tbox.date_selection_title ) + 1 );
    strcpy( str, new -> tbox.date_selection_title );
    new -> tbox.date_selection_title = str;
  }


  /* Copy strings for scanning. */
#define SetMarker( field, default )   \
{ \
  if( field == NULL ){ \
    newstr = XtMalloc( strlen( default ) ); \
    strcpy( newstr, default ); \
    field = newstr; \
  } else { \
    newstr = XtMalloc( strlen( field ) ); \
    strcpy( newstr, field ); \
    field = newstr; \
  } \
}

  SetMarker( new -> tbox.day_marker,    default_day_marker );
  SetMarker( new -> tbox.hour_marker,   default_hour_marker );
  SetMarker( new -> tbox.minute_marker, default_minute_marker );
  SetMarker( new -> tbox.month_marker,  default_month_marker );
  SetMarker( new -> tbox.week_marker,   default_week_marker );
  SetMarker( new -> tbox.year_marker,   default_year_marker );

#undef SetMarker


  /* Create the internal widgets. */
  new -> tbox.internal_widgets_created = False;

  CreateInternalWidgets( new );

  new -> tbox.internal_widgets_created = True;


  /* Set the desired dimensions of the widget. */
  GetChildPrefSizes( new, NULL, NULL, kids_sizes );
  PrepareLayout( new, kids_sizes, &pref_width, &pref_height );

  new -> core.width  = pref_width;
  new -> core.height = pref_height;

  DoLayout( new, NULL, NULL, kids_sizes );


  return;

} /* Initialize */


/*----------------------------------------------------------------------*/

static void
  InitializeItem( int             index,
                  XmUbMenuItem    items[],
                  XmUbItem        item,
                  String          label,
                  XtCallbackProc  proc,
                  XtPointer       client_data )
{
  /* Variables. */

  /* Code. */

  items[ index ].item = item;

  items[ index ].name = XtMalloc( 5 );
  if( item == XmUbITEM_PUSH_BUTTON )
    sprintf( items[ index ].name, "Bu%d", index );
  else
    sprintf( items[ index ].name, "Sep%d", index );

  items[ index ].label = XmStringCreateLtoR( label, XmSTRING_DEFAULT_CHARSET );

  items[ index ].proc        = proc;
  items[ index ].client_data = client_data;


  return;

} /* InitializeItem */


/*----------------------------------------------------------------------*/

static void
  InitializeDefaultMenu( XmUbTimeBoxWidget  tbox,
                         XmUbMenuItem       items[],
                         int                *num_items_created )
{
  /* Variables. */
  int   n = 0;

  /* Code. */

  /* The items in the menu depend on the widget format. */

  /* The first field applies to all formats. */
  InitializeItem( n++, items, XmUbITEM_PUSH_BUTTON, 
                  "Now", XmUbTB_ACTION_NOW, NULL );

  /* These apply only to ranges with two date fields. */
  if( ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DD ) ||
      ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DTDT ) ){

    InitializeItem( n++, items, XmUbITEM_SEPARATOR,
                    "", XmUbTB_NO_ACTION, NULL );
    InitializeItem( n++, items, XmUbITEM_PUSH_BUTTON, 
                    "This week", XmUbTB_ACTION_THIS_WEEK, NULL );
    InitializeItem( n++, items, XmUbITEM_PUSH_BUTTON, 
                    "This month", XmUbTB_ACTION_THIS_MONTH, NULL );
   }

  /* The following apply to range fields that include the time. */
  if( ( tbox -> tbox.widget_format == XmUbTB_FORMAT_TT )  ||
      ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DTT ) ||
      ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DTDT ) ){

    InitializeItem( n++, items, XmUbITEM_SEPARATOR, 
                    "", XmUbTB_NO_ACTION, NULL );
    InitializeItem( n++, items, XmUbITEM_PUSH_BUTTON,
                    "+ One Hour", XmUbTB_ACTION_PLUS_ONE_HOUR, NULL );
    InitializeItem( n++, items, XmUbITEM_PUSH_BUTTON,
                    "+ Two Hours", XmUbTB_ACTION_PLUS_TWO_HOURS, NULL );
  }

  /* These apply to ranges with date fields. */
  if( ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DD ) ||
      ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DTDT ) ){

    InitializeItem( n++, items, XmUbITEM_SEPARATOR, 
                    "", XmUbTB_NO_ACTION, NULL );
    InitializeItem( n++, items, XmUbITEM_PUSH_BUTTON, 
                    "+ One Week", XmUbTB_ACTION_PLUS_ONE_WEEK, NULL );
    InitializeItem( n++, items, XmUbITEM_PUSH_BUTTON, 
                    "+ Two Weeks", XmUbTB_ACTION_PLUS_TWO_WEEKS, NULL );
    InitializeItem( n++, items, XmUbITEM_PUSH_BUTTON, 
                    "+ One Month", XmUbTB_ACTION_PLUS_ONE_MONTH, NULL );
  }


  *num_items_created = n;

  return;

} /* InitializeDefaultMenu */


/*----------------------------------------------------------------------*/

static void
  InsertChild( Widget  widget )
{

  /* Variables. */
  Cardinal           num_params;
  String             params[ 1 ];
  XmUbTimeBoxWidget  tbox;

  /* Code. */

  tbox = (XmUbTimeBoxWidget) XtParent( widget );

  /* We do not allow the application to create children. */
  if( tbox -> tbox.internal_widgets_created ){

    params[ 0 ] = XtClass( tbox ) -> core_class.class_name;
    num_params  = 1;
    XtAppErrorMsg( XtWidgetToApplicationContext( widget ),
                   "childError", "number", "WidgetError",
                   "Applications cannot add children to %s widgets.",
                   params, &num_params );
  }

  /* This is an internal child. Adding it is OK. */
  (* ( (CompositeWidgetClass) (xmUbTimeBoxWidgetClass ->
     core_class.superclass) ) -> composite_class.insert_child ) ( widget );


  return;

} /* InsertChild */


/*----------------------------------------------------------------------*/

static void 
  KidsPreferredGeometry( Widget            kid,
                         Widget            initiator,
                         XtWidgetGeometry  *request,
                         XtWidgetGeometry  *desired )
{

  /* Code. */

  if( ( kid == initiator ) && ( request != NULL ) ){
    /* The initiator should not be queried. */
    if( request -> request_mode & CWWidth )
      desired -> width = request -> width;
    else
      desired -> width = (Dimension) kid -> core.width;

    if( request -> request_mode & CWHeight )
      desired -> height = request -> height;
    else
      desired -> height = (Dimension) kid -> core.height;

  } else
    (void) XtQueryGeometry( kid, NULL, desired );


  return;

} /* KidsPreferredGeometry */


/*----------------------------------------------------------------------*/

static void
  MenuItemActivatedCB( Widget               pb,
                       XtCallbackProc       action,
                       XmAnyCallbackStruct  *call_data )
{
  /* Variables. */
  Arg                        args[ 5 ];
  XmUbTimeBoxCallbackStruct  cb;
  Cardinal                   n;
  Boolean                    range = True;
  XmUbTimeBoxWidget          tbox;

  /* Code. */

  /* The time box widget id is in user data. */
  n = 0;
  XtSetArg( args[ n ], XmNuserData, &tbox ); n++;
  XtGetValues( pb, args, n );

  /* What to do depends on the selected action. */
  switch( (int) action ){

    case (int) XmUbTB_ACTION_THIS_WEEK:
      FillInStartOfWeek( tbox );
      AddDateWeek( tbox, 1, 1 );
      break;

    case (int) XmUbTB_ACTION_THIS_MONTH:
      FillInStartOfMonth( tbox );
      AddDateMonth( tbox, 1, 1 );
      break;

    case (int) XmUbTB_ACTION_PLUS_ONE_HOUR:
      AddDateHour( tbox, 1 );
      break;

    case (int) XmUbTB_ACTION_PLUS_TWO_HOURS:
      AddDateHour( tbox, 2 );
      break;

    case (int) XmUbTB_ACTION_PLUS_ONE_WEEK:
      AddDateWeek( tbox, 1, 0 );
      break;

    case (int) XmUbTB_ACTION_PLUS_TWO_WEEKS:
      AddDateWeek( tbox, 2, 0 );
      break;

    case (int) XmUbTB_ACTION_PLUS_ONE_MONTH:
      AddDateMonth( tbox, 1, 0 );
      break;

    case (int) XmUbTB_ACTION_TIME_NOW:
      (void) XmUbTimeBoxSetStartTime( (Widget) tbox, 
                                      TimLocalTime( TimMakeTimeNow() ) );
      range = False;
      break;

    case (int) XmUbTB_ACTION_DATE_NOW:
      (void) XmUbTimeBoxSetStartDate( (Widget) tbox, 
                                      TimLocalTime( TimMakeTimeNow() ) );
      range = False;
      break;

    case (int) XmUbTB_ACTION_NOW:
      (void) XmUbTimeBoxSetStartDate( (Widget) tbox, 
                                      TimLocalTime( TimMakeTimeNow() ) );
      (void) XmUbTimeBoxSetStartTime( (Widget) tbox, 
                                      TimLocalTime( TimMakeTimeNow() ) );
      range = False;

      break;

    default:
      return;

  } /* switch */

  /* Set the keyboard focus to the most recently inserted text. */
  switch( (int) action ){

    case (int) XmUbTB_ACTION_TIME_NOW:
      /* Set keyboard focus to the start time field. */
      if( tbox -> tbox.internal_children[ XmUbTB_CHILD_START_TIME ] != NULL )
        XmProcessTraversal( 
          tbox -> tbox.internal_children[ XmUbTB_CHILD_START_TIME ], 
          XmTRAVERSE_CURRENT );

      break;

    case (int) XmUbTB_ACTION_DATE_NOW:
      /* Set keyboard focus to the start date field. */
      if( tbox -> tbox.internal_children[ XmUbTB_CHILD_START_DATE ] != NULL )
        XmProcessTraversal( 
          tbox -> tbox.internal_children[ XmUbTB_CHILD_START_DATE ], 
          XmTRAVERSE_CURRENT );

      break;

    case (int) XmUbTB_ACTION_NOW:
      /* Set keyboard focus to the start time or date field. */
      if( tbox -> tbox.internal_children[ XmUbTB_CHILD_START_TIME ] != NULL )
        XmProcessTraversal( 
          tbox -> tbox.internal_children[ XmUbTB_CHILD_START_TIME ], 
          XmTRAVERSE_CURRENT );
      else if( 
           tbox -> tbox.internal_children[ XmUbTB_CHILD_START_DATE ] != NULL )
        XmProcessTraversal( 
          tbox -> tbox.internal_children[ XmUbTB_CHILD_START_DATE ], 
          XmTRAVERSE_CURRENT );

      break;

    default:
      /* Set keyboard focus to the end time or date field. */
      if( tbox -> tbox.internal_children[ XmUbTB_CHILD_END_TIME ] != NULL )
        XmProcessTraversal( 
          tbox -> tbox.internal_children[ XmUbTB_CHILD_END_TIME ], 
          XmTRAVERSE_CURRENT );
      else if( 
           tbox -> tbox.internal_children[ XmUbTB_CHILD_END_DATE ] != NULL )
        XmProcessTraversal( 
          tbox -> tbox.internal_children[ XmUbTB_CHILD_END_DATE ], 
          XmTRAVERSE_CURRENT );

      break;

  } /* switch */

  /* A default action was selected. Invoke activate callbacks. */
  if( tbox -> tbox.activate_callback != NULL ){

    cb.reason      = XmUbCR_DEFAULT_ACTION;
    cb.event       = NULL;
    cb.child_index = 0;
    cb.child       = NULL;
    cb.action      = action;
    cb.range       = range;

    XtCallCallbackList( (Widget) tbox, tbox -> tbox.activate_callback,
                        (XtPointer) &cb );

  }



  return;

} /* MenuItemActivatedCB */


/*----------------------------------------------------------------------*/

Widget
  MonthDisplayId( XmUbTimeBoxWidget  tbox )
{
  /* Variables. */
  char    *name_buffer;
  Widget  md;
  String  wname;

  /* Code. */

  wname = XtName( tbox );

  /* Get the id of the month display widget. */
  name_buffer = XtMalloc( strlen( wname ) + 8 );
  sprintf( name_buffer, "%sMd", wname, wname );

  md = XtNameToWidget( tbox -> tbox.date_selection_popup, name_buffer );

  XtFree( name_buffer );


  return( md );

} /* MonthDisplayId */


/*----------------------------------------------------------------------*/

static void
  Resize( Widget    widget )
{

  XmUbTimeBoxWidget  tbox;

  /* Code. */

  tbox = (XmUbTimeBoxWidget) widget;

  tbox -> tbox.resize_called = True;

  /* The sizes or positions of the children do not depend on the 
     time box widget size, so we don't need to recalculate anything. */


  return;

} /* Resize */


/*----------------------------------------------------------------------*/

static XtGeometryResult
  QueryGeometry( Widget             widget,
                 XtWidgetGeometry  *proposed,
                 XtWidgetGeometry  *answer )
{

  KidDimensionRec    kids_sizes[ NO_INTERNAL_CHILDREN ];
  Dimension          pref_height;
  Dimension          pref_width;
  XmUbTimeBoxWidget  tbox;

  /* Code. */


  tbox = (XmUbTimeBoxWidget) widget;

  /* Get dimensions that we *really* want. */
  GetChildPrefSizes( tbox, NULL, NULL, kids_sizes );
  PrepareLayout( tbox, kids_sizes, &pref_width, &pref_height );

  answer -> request_mode = CWWidth | CWHeight;
  answer -> width  = pref_width;
  answer -> height = pref_height;


  if( proposed == NULL ){
    /* This is a query for the requested geometry. */

    if(( answer -> height == (int) tbox -> core.height ) &&
       ( answer -> width  == (int) tbox -> core.width ))
      return XtGeometryNo;
    else
      return XtGeometryAlmost;
  }


  /* The parent supplied a geometry suggestion. */
  if(( answer -> height == proposed -> height ) &&
     ( answer -> width  == proposed -> width ))
    return XtGeometryYes;

  if( ( proposed -> height <= 1 ) ||
      ( proposed -> width  <= 1 ) )
    /* That's too small ! */
    return XtGeometryNo;


  /* Only a compromise left. */
  return XtGeometryAlmost;

} /* QueryGeometry */


/*----------------------------------------------------------------------*/

static void
  PopdownDateSelectionDialogCB( Widget               md,
                                XmUbTimeBoxWidget    tbox,
                                XmAnyCallbackStruct  *call_data )
{
  /* Code. */

  if( tbox -> tbox.destroy_dialogs ){
    XtDestroyWidget( XtParent( tbox -> tbox.date_selection_popup ) );
    tbox -> tbox.date_selection_popup = NULL;

  } else
    XtUnmanageChild( tbox -> tbox.date_selection_popup );


  return;

} /* PopdownDateSelectionDialogCB */


/*----------------------------------------------------------------------*/

static void
  PopdownTimeSelectionDialogCB( Widget               tw,
                                XmUbTimeBoxWidget    tbox,
                                XmAnyCallbackStruct  *call_data )
{
  /* Code. */

  if( tbox -> tbox.destroy_dialogs ){
    XtDestroyWidget( XtParent( tbox -> tbox.time_selection_popup ) );
    tbox -> tbox.time_selection_popup = NULL;

  } else
    XtUnmanageChild( tbox -> tbox.time_selection_popup );


  return;

} /* PopdownTimeSelectionDialogCB */


/*----------------------------------------------------------------------*/

static void
  PopupDateSelection( Widget  widget )
{
  /* Variables. */
  XmUbTimeBoxStatus  status;
  XmUbTimeBoxWidget  tbox;
  time_t             time;

  /* Code. */

  /* We are called with the child. */
  tbox = (XmUbTimeBoxWidget) XtParent( widget );

  /* If user callback, call it. */
  if( tbox -> tbox.popup_selection_callback != NULL ){
    CallDialogPopupCallback( tbox, widget, XmUbCR_POPUP_DATE_SELECTION );

    return;
  }

  /* No callback is set and we should use our own dialog. */
  if( tbox -> tbox.date_selection_popup == NULL )
    CreateDateSelectionPopup( tbox );

  /* Set the index of the child that popped up the dialog. */
  tbox -> tbox.field_for_date_selection = ChildIndex( tbox, widget );

  /* Pop up the dialog. */

  /* Set the current month first. */
  switch( tbox -> tbox.field_for_date_selection ){
    case XmUbTB_CHILD_START_DATE: 
      status = XmUbTimeBoxGetStartDate( (Widget) tbox, &time );
      break;

    case XmUbTB_CHILD_END_DATE:
      status = XmUbTimeBoxGetEndDate( (Widget) tbox, &time );

      /* If no date is entered, try with the starting date as default. */
      if( status != TBOX_OK )
        status = XmUbTimeBoxGetStartDate( (Widget) tbox, &time );
      break;

    default:
      /* Can only be popped up from a date field. */
      return;
  }

  /* If no default date, use today. */
  if( status != TBOX_OK )
    time = TimLocalTime( TimMakeTimeNow() );

  XmUbMonthDisplaySetMonth( MonthDisplayId( tbox ), TimIndexOfYear( time ),
                            TimIndexOfMonth( time ) );

  XtManageChild( tbox -> tbox.date_selection_popup );

  /* Set the focus to the current date. */
  XmUbMonthDisplaySetFocusToDay( MonthDisplayId( tbox ), 
                                 TimIndexOfDay( time ) );


  return;

} /* PopupDateSelection */


/*----------------------------------------------------------------------*/

static void
  PopupDateSelectionAction( Widget    w,
                            XEvent    *event,
                            String    *params,
                            Cardinal  num_params )
{
  /* Variables. */
  Widget  parent;

  /* Code. */

  parent = XtParent( w );

  (* ( (XmUbTimeBoxClassRec *) ( XtClass( parent ) ) ) -> 
     timebox_class.popup_date_selection ) ( w );

  
  return;

} /* PopupDateSelectionAction */


/*----------------------------------------------------------------------*/

static void
  PopupTimeSelection( Widget  widget )
{
  /* Variables. */
  XmUbTimeBoxStatus  status;
  XmUbTimeBoxWidget  tbox;
  Widget             text_field;
  time_t             time;

  /* Code. */

  /* We are called with the child. */
  tbox = (XmUbTimeBoxWidget) XtParent( widget );

  /* If user callback, call it. */
  if( tbox -> tbox.popup_selection_callback != NULL ){
    CallDialogPopupCallback( tbox, widget, XmUbCR_POPUP_TIME_SELECTION );

    return;
  }

  /* No callback is set and we should use our own dialog. */
  if( tbox -> tbox.time_selection_popup == NULL )
    CreateTimeSelectionPopup( tbox );

  /* Set the index of the child that popped up the dialog. */
  tbox -> tbox.field_for_time_selection = ChildIndex( tbox, widget );


  /* Set the start time first. */
  switch( tbox -> tbox.field_for_time_selection ){
    case XmUbTB_CHILD_START_TIME: 
      status = XmUbTimeBoxGetStartTime( (Widget) tbox, &time );
      break;

    case XmUbTB_CHILD_END_TIME:
      status = XmUbTimeBoxGetEndTime( (Widget) tbox, &time );
      break;

    default:
      /* Can only be popped up from a time field. */
      return;

  }

  /* If we could get no default, use now. */
  if( status != TBOX_OK )
    time = TimLocalTime( TimMakeTimeNow() );

  XmUbTimeSliderSetTime( TimeSliderId( tbox ), time );

  XtManageChild( tbox -> tbox.time_selection_popup );

  /* Set keyboard focus to the text field. */
  text_field = XmUbTimeSliderGetChild( TimeSliderId( tbox ), 
                                       XmUbTS_CHILD_TEXT_FIELD );
  if( text_field != NULL )
    XmProcessTraversal( text_field, XmTRAVERSE_CURRENT );


  return;

} /* PopupTimeSelection */


/*----------------------------------------------------------------------*/

static void
  PopupTimeSelectionAction( Widget    w,
                            XEvent    *event,
                            String    *params,
                            Cardinal  num_params )
{
  /* Variables. */
  Widget  parent;

  /* Code. */

  parent = XtParent( w );

  (* ( (XmUbTimeBoxClassRec *) ( XtClass( parent ) ) ) -> 
     timebox_class.popup_time_selection ) ( w );

  
  return;

} /* PopupTimeSelectionAction */


/*----------------------------------------------------------------------*/

static void
  PrepareLayout( XmUbTimeBoxWidget  tbox,
                 KidDimensionRec    sizes[],
                 Dimension          *own_width,
                 Dimension          *own_height )
{
  /* Variables. */
  int        first_field;
  int        index_of_first_field = 0;
  int        index_of_last_field = 0;
  int        last_field;
  int        index;
  Dimension  max_widget_height;
  Dimension  max_widget_width;
  Dimension  menu_width_offset;
  Position   pos_offset;
  Dimension  spacing;
  Position   text_field_start;
  int        use_field;

  /* Code. */

  if( tbox -> tbox.menu_enabled && 
      ( tbox -> tbox.menu_position == XmUbPOSITION_FIRST ) ){
    /* The menu should be placed to the left of the first field. */
    sizes[ XmUbTB_CHILD_MENU_BAR ].x = (Position) tbox -> tbox.margin_width;

    text_field_start = sizes[ XmUbTB_CHILD_MENU_BAR ].x + 
      (Position) sizes[ XmUbTB_CHILD_MENU_BAR ].width + 
      (Position) tbox -> tbox.menu_spacing;

  } else {
    /* Menu last or not at all. */
    text_field_start = (Position) tbox -> tbox.margin_width;
  }

  /* Position the text widgets. */

  switch( tbox -> tbox.widget_format ){
    case XmUbTB_FORMAT_D:
    case XmUbTB_FORMAT_T:
      if( tbox -> tbox.widget_format == XmUbTB_FORMAT_D )
        index = XmUbTB_CHILD_START_DATE;
      else
        index = XmUbTB_CHILD_START_TIME;

      sizes[ index ].x = text_field_start;
      sizes[ index ].y = (Position) tbox -> tbox.margin_height;

      *own_width  = (Dimension) sizes[ index ].x + sizes[ index ].width +
                    2 * tbox -> tbox.margin_width;
      *own_height = MAX( sizes[ XmUbTB_CHILD_MENU_BAR ].height,
                         sizes[ index ].height ) +
                    2 * tbox -> tbox.margin_height;

      index_of_first_field = index;
      index_of_last_field  = index;

      break;

    case XmUbTB_FORMAT_DT:
    case XmUbTB_FORMAT_DD:
    case XmUbTB_FORMAT_TT:

      if( tbox -> tbox.widget_format == XmUbTB_FORMAT_DT ){
        first_field = XmUbTB_CHILD_START_DATE;
        last_field  = XmUbTB_CHILD_START_TIME;
        spacing     = tbox -> tbox.field_spacing;

      } else if( tbox -> tbox.widget_format == XmUbTB_FORMAT_DD ){
        first_field = XmUbTB_CHILD_START_DATE;
        last_field  = XmUbTB_CHILD_END_DATE;
        spacing     = tbox -> tbox.range_spacing;

      } else {
        first_field = XmUbTB_CHILD_START_TIME;
        last_field  = XmUbTB_CHILD_END_TIME;
        spacing     = tbox -> tbox.range_spacing;
      }

      sizes[ first_field ].x = text_field_start;
      sizes[ first_field ].y = (Position) tbox -> tbox.margin_height;

      if( ( tbox -> tbox.orientation == XmHORIZONTAL ) ||
          ( ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DT ) &&
            tbox -> tbox.component_together ) ){
        /* Text fields should be ordered horizontally. */
        sizes[ last_field ].x = sizes[ first_field ].x + 
          (Position) sizes[ first_field ].width + (Position) spacing;

        sizes[ last_field ].y = sizes[ first_field ].y;

        *own_width = (Dimension) sizes[ last_field ].x +
          sizes[ last_field ].width + tbox -> tbox.margin_width;

        *own_height = MAX( MAX( sizes[ first_field ].height,
                                sizes[ last_field ].height ),
                           sizes[ XmUbTB_CHILD_MENU_BAR ].height ) +
                      2 * tbox -> tbox.margin_height;

      } else {
        /* Text fields should be ordered vertically. */

        sizes[ last_field ].x = sizes[ first_field ].x;

        sizes[ last_field ].y = sizes[ first_field ].y + 
          (Position) sizes[ first_field ].height + (Position) spacing;

        *own_width = (Dimension) sizes[ first_field ].x +
          MAX( sizes[ first_field ].width, sizes[ last_field ].width ) + 
          tbox -> tbox.margin_width;

        *own_height = (Dimension) sizes[ last_field ].y +
                      sizes[ last_field ].height + tbox -> tbox.margin_height;
      }

      index_of_first_field = first_field;
      index_of_last_field  = last_field;

      break;

    case XmUbTB_FORMAT_DTDT:
      sizes[ XmUbTB_CHILD_START_DATE ].x = text_field_start;
      sizes[ XmUbTB_CHILD_START_DATE ].y = 
        (Position) tbox -> tbox.margin_height;

      if( tbox -> tbox.orientation == XmHORIZONTAL ){
        /* Horizontal orientation. */

        sizes[ XmUbTB_CHILD_START_TIME ].x = 
          sizes[ XmUbTB_CHILD_START_DATE ].x +
          (Position) sizes[ XmUbTB_CHILD_START_DATE ].width + 
          (Position) tbox -> tbox.field_spacing;
        sizes[ XmUbTB_CHILD_START_TIME ].y = 
          sizes[ XmUbTB_CHILD_START_DATE ].y;
        
        sizes[ XmUbTB_CHILD_END_DATE ].x = 
          sizes[ XmUbTB_CHILD_START_TIME ].x +
          (Position) sizes[ XmUbTB_CHILD_START_TIME ].width + 
          (Position) tbox -> tbox.range_spacing;
        sizes[ XmUbTB_CHILD_END_DATE ].y = 
          sizes[ XmUbTB_CHILD_START_DATE ].y;
        
        sizes[ XmUbTB_CHILD_END_TIME ].x = 
          sizes[ XmUbTB_CHILD_END_DATE ].x +
          (Position) sizes[ XmUbTB_CHILD_END_DATE ].width + 
          (Position) tbox -> tbox.field_spacing;
        sizes[ XmUbTB_CHILD_END_TIME ].y = 
          sizes[ XmUbTB_CHILD_START_DATE ].y;
      
        max_widget_height = sizes[ XmUbTB_CHILD_MENU_BAR ].height;
        for( index = XmUbTB_CHILD_START_DATE; index < NO_INTERNAL_CHILDREN;
             index ++ ){
          if( sizes[ index ].height > max_widget_height )
            max_widget_height = sizes[ index ].height;
        }

        *own_width = (Dimension) sizes[ XmUbTB_CHILD_END_TIME ].y +
                     sizes[ XmUbTB_CHILD_END_TIME ].width +
                     tbox -> tbox.margin_width;
        *own_height = max_widget_height + 2 * tbox -> tbox.margin_height;

      } else if( tbox -> tbox.component_together ){
        /* Place the text widgets in two rows. */

        sizes[ XmUbTB_CHILD_START_TIME ].x = 
          sizes[ XmUbTB_CHILD_START_DATE ].x + 
          (Position) sizes[ XmUbTB_CHILD_START_DATE ].width +
          (Position) tbox -> tbox.field_spacing;
        sizes[ XmUbTB_CHILD_START_TIME ].y = 
          sizes[ XmUbTB_CHILD_START_DATE ].y;

        sizes[ XmUbTB_CHILD_END_DATE ].x = sizes[ XmUbTB_CHILD_START_DATE ].x;
        sizes[ XmUbTB_CHILD_END_DATE ].y = sizes[ XmUbTB_CHILD_START_DATE ].y +
          (Position) sizes[ XmUbTB_CHILD_START_DATE ].height +
          (Position) tbox -> tbox.range_spacing;

        sizes[ XmUbTB_CHILD_END_TIME ].x = sizes[ XmUbTB_CHILD_END_DATE ].x +
          (Position) sizes[ XmUbTB_CHILD_END_DATE ].width + 
          (Position) tbox -> tbox.field_spacing;
        sizes[ XmUbTB_CHILD_END_TIME ].y = sizes[ XmUbTB_CHILD_END_DATE ].y;

        *own_width = MAX( (Dimension) sizes[ XmUbTB_CHILD_START_TIME ].x +
                          sizes[ XmUbTB_CHILD_START_TIME ].width,
                          (Dimension) sizes[ XmUbTB_CHILD_END_TIME ].x +
                          sizes[ XmUbTB_CHILD_END_TIME ].width ) +
                     tbox -> tbox.margin_width;
        *own_height = (Dimension) sizes[ XmUbTB_CHILD_END_TIME ].y +
                      MAX( sizes[ XmUbTB_CHILD_END_TIME ].height, 
                           sizes[ XmUbTB_CHILD_END_DATE ].height ) +
                      tbox -> tbox.margin_height;

      } else {
        /* Place all text fields under eachother. */
       
        sizes[ XmUbTB_CHILD_START_TIME ].x = 
          sizes[ XmUbTB_CHILD_START_DATE ].x;
        sizes[ XmUbTB_CHILD_START_TIME ].y = 
          sizes[ XmUbTB_CHILD_START_DATE ].y +
          (Position) sizes[ XmUbTB_CHILD_START_DATE ].height +
          (Position) tbox -> tbox.field_spacing;

        sizes[ XmUbTB_CHILD_END_DATE ].x = 
          sizes[ XmUbTB_CHILD_START_DATE ].x;
        sizes[ XmUbTB_CHILD_END_DATE ].y = 
          sizes[ XmUbTB_CHILD_START_TIME ].y +
          (Position) sizes[ XmUbTB_CHILD_START_TIME ].height +
          (Position) tbox -> tbox.range_spacing;

        sizes[ XmUbTB_CHILD_END_TIME ].x = 
          sizes[ XmUbTB_CHILD_START_DATE ].x;
        sizes[ XmUbTB_CHILD_END_TIME ].y = 
          sizes[ XmUbTB_CHILD_END_DATE ].y +
          (Position) sizes[ XmUbTB_CHILD_END_DATE ].height +
          (Position) tbox -> tbox.field_spacing;

        max_widget_width = 0;
        for( index = XmUbTB_CHILD_START_DATE; index < NO_INTERNAL_CHILDREN;
             index ++ ){
          if( sizes[ index ].width > max_widget_width )
            max_widget_width = sizes[ index ].width;
        }

        *own_width = (Dimension) sizes[ XmUbTB_CHILD_START_DATE ].x +
                     max_widget_width + tbox -> tbox.margin_width;
        *own_height = (Dimension) sizes[ XmUbTB_CHILD_END_TIME ].y +
                      sizes[ XmUbTB_CHILD_END_TIME ].height +
                      tbox -> tbox.margin_height;
      }
      
      index_of_first_field = XmUbTB_CHILD_START_DATE;
      index_of_last_field  = XmUbTB_CHILD_END_TIME;

      break;

    case XmUbTB_FORMAT_DTT:
      sizes[ XmUbTB_CHILD_START_DATE ].x = text_field_start;
      sizes[ XmUbTB_CHILD_START_DATE ].y = 
        (Position) tbox -> tbox.margin_height;

      if( tbox -> tbox.orientation == XmHORIZONTAL ){
        /* Lay out the text fields horizontally. */
        sizes[ XmUbTB_CHILD_START_TIME ].x = 
          sizes[ XmUbTB_CHILD_START_DATE ].x + 
          (Position) sizes[ XmUbTB_CHILD_START_DATE ].width + 
          (Position) tbox -> tbox.field_spacing;
        sizes[ XmUbTB_CHILD_START_TIME ].y = 
          sizes[ XmUbTB_CHILD_START_DATE ].y;

        sizes[ XmUbTB_CHILD_END_TIME ].x = sizes[ XmUbTB_CHILD_START_TIME ].x +
          (Position) sizes[ XmUbTB_CHILD_START_TIME ].width +
          (Position) tbox -> tbox.range_spacing;
        sizes[ XmUbTB_CHILD_END_TIME ].y = 
          sizes[ XmUbTB_CHILD_START_DATE ].y;

        max_widget_height = sizes[ XmUbTB_CHILD_MENU_BAR ].height;
        for( index = XmUbTB_CHILD_START_DATE; index < NO_INTERNAL_CHILDREN;
             index ++ ){
          if( sizes[ index ].height > max_widget_height )
            max_widget_height = sizes[ index ].height;
        }

        *own_width = (Dimension) sizes[ XmUbTB_CHILD_END_TIME ].x +
          sizes[ XmUbTB_CHILD_END_TIME ].width + tbox -> tbox.margin_width;
        *own_height = max_widget_height + 2 * tbox -> tbox.margin_height;

      } else if( tbox -> tbox.component_together ){
        /* Place the text widgets in two rows. */

        sizes[ XmUbTB_CHILD_START_TIME ].x = 
          sizes[ XmUbTB_CHILD_START_DATE ].x + 
          (Position) sizes[ XmUbTB_CHILD_START_DATE ].width +
          (Position) tbox -> tbox.field_spacing;
        sizes[ XmUbTB_CHILD_START_TIME ].y = 
          sizes[ XmUbTB_CHILD_START_DATE ].y;

        sizes[ XmUbTB_CHILD_END_TIME ].x = sizes[ XmUbTB_CHILD_START_TIME ].x;
        sizes[ XmUbTB_CHILD_END_TIME ].y = sizes[ XmUbTB_CHILD_START_TIME ].y +
          (Position) sizes[ XmUbTB_CHILD_START_TIME ].height +
          (Position) tbox -> tbox.range_spacing;

        *own_width = MAX( (Dimension) sizes[ XmUbTB_CHILD_START_TIME ].x +
                          sizes[ XmUbTB_CHILD_START_TIME ].width,
                          (Dimension) sizes[ XmUbTB_CHILD_END_TIME ].x +
                          sizes[ XmUbTB_CHILD_END_TIME ].width ) +
                     tbox -> tbox.margin_width;

        *own_height = (Dimension) sizes[ XmUbTB_CHILD_END_TIME ].y +
           sizes[ XmUbTB_CHILD_END_TIME ].height +
           tbox -> tbox.margin_height;

      } else {
        /* Lay out all text widgets vertically. */

        sizes[ XmUbTB_CHILD_START_TIME ].x = 
          sizes[ XmUbTB_CHILD_START_DATE ].x;
        sizes[ XmUbTB_CHILD_START_TIME ].y = 
          sizes[ XmUbTB_CHILD_START_DATE ].y + 
          (Position) sizes[ XmUbTB_CHILD_START_DATE ].height +
          (Position) tbox -> tbox.field_spacing;

        sizes[ XmUbTB_CHILD_END_TIME ].x = 
          sizes[ XmUbTB_CHILD_START_DATE ].x;
        sizes[ XmUbTB_CHILD_END_TIME ].y = 
          sizes[ XmUbTB_CHILD_START_TIME ].y + 
          (Position) sizes[ XmUbTB_CHILD_START_TIME ].height +
          (Position) tbox -> tbox.range_spacing;

        max_widget_width = 0;
        for( index = XmUbTB_CHILD_START_DATE; index < NO_INTERNAL_CHILDREN;
             index ++ ){
          if( sizes[ index ].width > max_widget_width )
            max_widget_width = sizes[ index ].width;
        }

        *own_width = (Dimension) sizes[ XmUbTB_CHILD_START_DATE ].x +
                     max_widget_width + tbox -> tbox.margin_width;
        *own_height = (Dimension) sizes[ XmUbTB_CHILD_END_TIME ].y +
                      sizes[ XmUbTB_CHILD_END_TIME ].height +
                      tbox -> tbox.margin_height;
      }

      index_of_first_field = XmUbTB_CHILD_START_DATE;
      index_of_last_field  = XmUbTB_CHILD_END_TIME;

      break;
  }

  /* If the menu button should be placed last, this is the time to do it. */
  if( tbox -> tbox.menu_enabled ){

    if( tbox -> tbox.menu_position == XmUbPOSITION_FIRST ){
      /* The menu has already been positioned horizontally. */
      use_field = index_of_first_field;

    } else {
      /* Position the menu after the last field. */
      use_field = index_of_last_field;

      sizes[ XmUbTB_CHILD_MENU_BAR ].x = sizes[ use_field ].x +
        (Position) ( sizes[ use_field ].width + tbox -> tbox.menu_spacing );

      *own_width += ( sizes[ XmUbTB_CHILD_MENU_BAR ].width +
                      tbox -> tbox.menu_spacing );
    }

    /* Center the menu vertically with the adjacent field. */

    if( sizes[ use_field ].height > sizes[ XmUbTB_CHILD_MENU_BAR ].height )
      pos_offset = (Position) ( sizes[ use_field ].height -
        sizes[ XmUbTB_CHILD_MENU_BAR ].height ) / 2;
    else
      pos_offset = - (Position) ( sizes[ XmUbTB_CHILD_MENU_BAR ].height -
        sizes[ use_field ].height ) / 2;

    sizes[ XmUbTB_CHILD_MENU_BAR ].y = sizes[ use_field ].y + pos_offset;

  }

  /* Modify own height if menu button last. */
  if( tbox -> tbox.menu_enabled && 
      ( tbox -> tbox.menu_position == XmUbPOSITION_LAST ) ){

      *own_height = MAX( ( (Dimension) sizes[ XmUbTB_CHILD_MENU_BAR ].y +
                           sizes[ XmUbTB_CHILD_MENU_BAR ].height ),
                         ( (Dimension) sizes[ use_field ].y +
                           sizes[ use_field ].height ) ) +
                    tbox -> tbox.margin_height;
  }


  return;

} /* PrepareLayout */


/*----------------------------------------------------------------------*/

static Boolean
  ResizeIfNeeded( XmUbTimeBoxWidget  tbox,
                  KidDimensionRec    sizes[] )
{

  /* Variables. */
  Boolean           layout_done;
  Dimension         pref_height;
  Dimension         pref_width;
  XtWidgetGeometry  request;
  XtGeometryResult  result;

  /* Code. */

  /* Initialize. */
  layout_done = False;

  /* Get the preferred dimensions of the time box widget. */
  GetChildPrefSizes( tbox, NULL, NULL, sizes );
  PrepareLayout( tbox, sizes, &pref_width, &pref_height );

  /* If we want the same dimensions, no resizing is needed. */
  if(( pref_width  == tbox -> core.width ) &&
     ( pref_height == tbox -> core.height ))
    return False;

  /* Dimensions are different. Try to resize. */
  request.request_mode = CWWidth | CWHeight;

  request.width  = pref_width;
  request.height = pref_height;

  tbox -> tbox.resize_called = False;

  do {

    result = XtMakeGeometryRequest( (Widget) tbox, &request, &request );

  } while( result == XtGeometryAlmost );

  if( result == XtGeometryNo )
    return False;

  /* Resize done. Core fields have already been updated. */
  return( tbox -> tbox.resize_called );

} /* ResizeIfNeeded */


/*----------------------------------------------------------------------*/

static void
  SelectAllTextCB( Widget               tw,
                   XmUbTimeBoxWidget    tbox, 
                   XmAnyCallbackStruct  *call_data )
{
  /* Variables. */
  XmTextPosition  last;

  /* Code. */

  /* Select all text in the text widget. */

  last = XmTextGetLastPosition( tw );

  if( last > 0 )
    XmTextSetSelection( tw, 0, last, CurrentTime );

  /* Set the insertion position to the start of the widget. */
  XmTextSetInsertionPosition( tw, 0 );


  return;

} /* SelectAllTextCB */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  SetChildString( XmUbTimeBoxWidget  tbox,
                  int                child, 
                  char               *str )
{
  /* Variables. */
  Widget  tw;

  /* Code. */

  tw = tbox -> tbox.internal_children[ child ];
  if( tw == NULL )
    return( TBOX_NO_FIELD );

  XmTextSetString( tw, str );


  return( TBOX_OK );

} /* SetChildString */


/*----------------------------------------------------------------------*/

static Boolean 
  SetValues( Widget     current,
             Widget     request,
             Widget     new,
             ArgList    args,
             Cardinal   *num_args )
{
#ifdef Differs
#undef Differs
#endif

#define Differs( field )  ( curW -> field != newW -> field )

  /* Variables. */
  XmUbTimeBoxWidget  curW;
  int                index;
  Arg                local_args[ 3 ];
  Cardinal           n;
  char               *name_buffer;
  XmUbTimeBoxWidget  newW;
  char               *newstr;
  Boolean            redisplay = False;
  char               *str;
  Boolean            visual_changed = False;
  Widget             w;
  char               *wname;
  XmString           xm;

  /* Code. */

  curW = (XmUbTimeBoxWidget) current;
  newW = (XmUbTimeBoxWidget) new;

  /* Margins and spacings. */
  if( Differs( tbox.field_spacing ) ||
      Differs( tbox.menu_spacing )  ||
      Differs( tbox.range_spacing ) ||
      Differs( tbox.margin_width )  ||
      Differs( tbox.margin_height ) )
    visual_changed = True;

  /* These changes should be propagated to the child widgets. */
  if( Differs( core.background_pixel ) ||
      Differs( manager.foreground ) ){

    n = 0;

    if( Differs( core.background_pixel ) ){
      XtSetArg( local_args[ n ], XmNbackground, 
                                 newW -> core.background_pixel ); n++;
    }
    if( Differs( manager.foreground ) ){
      XtSetArg( local_args[ n ], XmNforeground, 
                                 newW -> manager.foreground ); n++;
    }

    /* Set the change for all child widgets. */
    for( index = 0; index < NO_INTERNAL_CHILDREN; index ++ )
      XtSetValues( newW -> tbox.internal_children[ index ], local_args, n );

    redisplay = True;
  }

  /* Changes affecting the time selection dialog. */

  if( Differs( tbox.ok_label ) ){

    if( curW -> tbox.ok_label != NULL )
      XmStringFree( curW -> tbox.ok_label );

    /* Copy the XmStrings and set new resources for dialog. */
    if( newW -> tbox.ok_label != NULL ){
      xm = XmStringCopy( newW -> tbox.ok_label );
      newW -> tbox.ok_label = xm;
    }

    if( newW -> tbox.time_selection_popup != NULL ){
      /* We need the Ok button. */
      wname = XtName( newW );
      name_buffer = XtMalloc( strlen( wname ) + 6 );
      sprintf( name_buffer, "%sTok", wname );

      w = XtNameToWidget( newW -> tbox.time_selection_popup, name_buffer );

      if( w != NULL ){
        n = 0;
        XtSetArg( local_args[ n ], XmNlabelString, 
                                   newW -> tbox.ok_label ); n++;
        XtSetValues( w, local_args, n );
      } else
        printf( "Error in XtNameToWidget, ok button is NULL.\n" );
    }
  }

  if( Differs( tbox.cancel_label ) ){

    if( curW -> tbox.cancel_label != NULL )
      XmStringFree( curW -> tbox.cancel_label );

    /* Copy the XmStrings and set new resources for dialog. */
    if( newW -> tbox.cancel_label != NULL ){
      xm = XmStringCopy( newW -> tbox.cancel_label );
      newW -> tbox.cancel_label = xm;
    }

    if( newW -> tbox.time_selection_popup != NULL ){
      /* We need the Ok button. */
      wname = XtName( newW );
      name_buffer = XtMalloc( strlen( wname ) + 6 );
      sprintf( name_buffer, "%sTcan", wname );

      w = XtNameToWidget( newW -> tbox.time_selection_popup, name_buffer );

      if( w != NULL ){
        n = 0;
        XtSetArg( local_args[ n ], XmNlabelString, 
                                   newW -> tbox.cancel_label ); n++;
        XtSetValues( w, local_args, n );
      } else
        printf( "Error in XtNameToWidget, cancel button is NULL.\n" );
    }
  }

  /* Changes affecting the time selection dialog shell. */
  if( Differs( tbox.time_selection_title ) ){
    
    if( curW -> tbox.time_selection_title != NULL )
      XtFree( curW -> tbox.time_selection_title );

    if( newW -> tbox.time_selection_title != NULL ){
      str = XtMalloc( strlen( newW -> tbox.time_selection_title ) + 1 );
      strcpy( str, newW -> tbox.time_selection_title );
      newW -> tbox.time_selection_title = str;
    }

    if( newW -> tbox.time_selection_popup != NULL ){
      n = 0;
      XtSetArg( local_args[ n ], XmNtitle, 
                                 newW -> tbox.time_selection_title ); n++;
      XtSetValues( XtParent( newW -> tbox.time_selection_popup ), 
                   local_args, n );
    }
  }

  /* Changes affecting the date selection dialog shell. */
  if( Differs( tbox.date_selection_title ) ){
    
    if( curW -> tbox.date_selection_title != NULL )
      XtFree( curW -> tbox.date_selection_title );

    if( newW -> tbox.date_selection_title != NULL ){
      str = XtMalloc( strlen( newW -> tbox.date_selection_title ) + 1 );
      strcpy( str, newW -> tbox.date_selection_title );
      newW -> tbox.date_selection_title = str;
    }

    if( newW -> tbox.date_selection_popup != NULL ){
      n = 0;
      XtSetArg( local_args[ n ], XmNtitle, 
                                 newW -> tbox.date_selection_title ); n++;
      XtSetValues( XtParent( newW -> tbox.date_selection_popup ), 
                   local_args, n );
    }
  }


  /* Changes affecting the month display dialog. */
  if( Differs( tbox.week_label )  ||
      Differs( tbox.month_label ) ||
      Differs( tbox.year_label ) ){

    n = 0;

    if( Differs( tbox.week_label ) ){

      if( curW -> tbox.week_label != NULL )
        XmStringFree( curW -> tbox.week_label );

      if( newW -> tbox.week_label != NULL ){
        xm = XmStringCopy( newW -> tbox.week_label );
        newW -> tbox.week_label = xm;
      }

      XtSetArg( local_args[ n ], XmUbNmdiWeekLabel, 
                                 newW -> tbox.week_label ); n++;
    }    

    if( Differs( tbox.month_label ) ){

      if( curW -> tbox.month_label != NULL )
        XmStringFree( curW -> tbox.week_label );

      if( newW -> tbox.month_label != NULL ){
        xm = XmStringCopy( newW -> tbox.month_label );
        newW -> tbox.month_label = xm;
      }

      XtSetArg( local_args[ n ], XmUbNmdiMonthLabel, 
                                 newW -> tbox.month_label ); n++;
    }    

    if( Differs( tbox.year_label ) ){

      if( curW -> tbox.year_label != NULL )
        XmStringFree( curW -> tbox.week_label );

      if( newW -> tbox.year_label != NULL ){
        xm = XmStringCopy( newW -> tbox.year_label );
        newW -> tbox.year_label = xm;
      }

      XtSetArg( local_args[ n ], XmUbNmdiYearLabel, 
                                 newW -> tbox.year_label ); n++;
    }    

    if( newW -> tbox.date_selection_popup != NULL )
      XtSetValues( newW -> tbox.date_selection_popup, local_args, n );
  }



  /* These resources may only be set at creation. */
  if( Differs( tbox.component_together ) ){
    WarningNoResourceChange( newW, "XmUbNtboxComponentTogether" );
    newW -> tbox.component_together = curW -> tbox.component_together;
  }

  if( Differs( tbox.destroy_dialogs ) ){
    WarningNoResourceChange( newW, "XmUbNtboxDestroyDialogs" );
    newW -> tbox.destroy_dialogs = curW -> tbox.destroy_dialogs;
  }

  if( Differs( tbox.menu_enabled ) ){
    WarningNoResourceChange( newW, "XmUbNtboxMenuEnabled" );
    newW -> tbox.menu_enabled = curW -> tbox.menu_enabled;
  }

  if( Differs( tbox.menu_items ) ){
    WarningNoResourceChange( newW, "XmUbNtboxMenuItems" );
    newW -> tbox.menu_items = curW -> tbox.menu_items;
  }

  if( Differs( tbox.menu_label ) ){
    WarningNoResourceChange( newW, "XmUbNtboxMenuLabel" );
    newW -> tbox.menu_label = curW -> tbox.menu_label;
  }

  if( Differs( tbox.menu_pixmap ) ){
    WarningNoResourceChange( newW, "XmUbNtboxMenuPixmap" );
    newW -> tbox.menu_pixmap = curW -> tbox.menu_pixmap;
  }

  if( Differs( tbox.menu_position ) ){
    WarningNoResourceChange( newW, "XmUbNtboxMenuPosition" );
    newW -> tbox.menu_position = curW -> tbox.menu_position;
  }

  if( Differs( tbox.num_menu_items ) ){
    WarningNoResourceChange( newW, "XmUbNtboxNumMenuItems" );
    newW -> tbox.num_menu_items = curW -> tbox.num_menu_items;
  }

  if( Differs( tbox.orientation ) ){
    WarningNoResourceChange( newW, "XmNorientation" );
    newW -> tbox.orientation = curW -> tbox.orientation;
  }

  if( Differs( tbox.widget_format ) ){
    WarningNoResourceChange( newW, "XmUbNtboxFormat" );
    newW -> tbox.widget_format = curW -> tbox.widget_format;
  }

  /* Copy strings for scanning. */
#define SetMarker( field, default )   \
{ \
  if( curW -> field != newW -> field ){ \
    if( curW -> field != NULL ) \
      XtFree( curW -> field ); \
    if( newW -> field == NULL ){ \
      newstr = XtMalloc( strlen( default ) ); \
      strcpy( newstr, default ); \
      newW -> field = newstr; \
    } else { \
      newstr = XtMalloc( strlen( newW -> field ) ); \
      strcpy( newstr, newW -> field ); \
      newW -> field = newstr; \
    } \
  } \
}

  SetMarker( tbox.day_marker,    default_day_marker );
  SetMarker( tbox.hour_marker,   default_hour_marker );
  SetMarker( tbox.minute_marker, default_minute_marker );
  SetMarker( tbox.month_marker,  default_month_marker );
  SetMarker( tbox.week_marker,   default_week_marker );
  SetMarker( tbox.year_marker,   default_year_marker );

#undef SetMarker

  if( visual_changed ){

    KidDimensionRec  kids_sizes[ NO_INTERNAL_CHILDREN ];
    Boolean          resized;

    /* Code. */

    resized = ResizeIfNeeded( newW, kids_sizes );
    if( resized )
      DoLayout( newW, NULL, NULL, kids_sizes );

  }

  return( redisplay || visual_changed );

#undef Differs

} /* SetValues */


/*----------------------------------------------------------------------*/

static void
  TimeCompletionCB( Widget                      tw,
                    XmUbTimeBoxWidget           tbox,
                    XmTextVerifyCallbackStruct  *call_data )
{
  /* Variables. */
  int                child_index;
  Boolean            restart;
  XmUbTimeBoxStatus  status;

  /* Code. */

TimeCompletionCB_START:

  /* Find child index for this field. */
  child_index = ChildIndex( tbox, tw );  

  /* Completion is different depending on which field we are dealing with. */
  switch( child_index ){

    case XmUbTB_CHILD_START_TIME:
      status = CompleteStartTime( tbox );
      break;


    case XmUbTB_CHILD_END_TIME:
      status = CompleteEndTime( tbox );
      break;

    default:
      /* This should not happen. */
      printf( "TimeCompletionCB ERROR, called for wrong child.\n" );
      return;

  } /* switch */

  /* If there was an error, take care of it. May lead to restart. */
  restart = TimeCompletionCallback( tw, child_index, tbox, 
				    (XmAnyCallbackStruct *) call_data,
				    status );
  if( restart )
    goto TimeCompletionCB_START;


  return;

} /* TimeCompletionCB */


/*----------------------------------------------------------------------*/

static Boolean
  TimeCompletionCallback( Widget               tw,
			  int                  child_index,
			  XmUbTimeBoxWidget    tbox,
			  XmAnyCallbackStruct  *call_data,
			  XmUbTimeBoxStatus    reason )
{
  /* Variables. */
  XmUbTimeBoxCallbackStruct  callback_struct;
  char                       *str;

  /* Code. */

  /* Initialize. Needed also if no callback. */
  callback_struct.child_index        = child_index;
  callback_struct.child              = tw; 
  callback_struct.restart_completion = False;

  if( tbox -> tbox.completion_callback != NULL ){

    /* Common fields. */
    if( call_data == NULL )
      callback_struct.event = NULL;
    else
      callback_struct.event = call_data -> event;

    switch( reason ){

      case TBOX_OK:
        callback_struct.reason = XmUbCR_COMPLETION_SUCCESS;
        break;

      case TBOX_EMPTY:
        callback_struct.reason = XmUbCR_COMPLETION_EMPTY;
        break;

      case TBOX_NO_COMPLETION:
        callback_struct.reason = XmUbCR_COMPLETION_FAILED;
        break;
    }

    XtCallCallbackList( (Widget) tbox, 
                        tbox -> tbox.completion_callback, 
                        (XtPointer) &callback_struct );

  } else {

    /* No callback was defined. */

    /* If completion failed, we should indicate this. */
    if( reason == TBOX_NO_COMPLETION ){
      /* Don't flicker the string if it is already set. */
      str = XmTextGetString( tw );

      if( strcmp( str, default_completion_failed_string ) != 0 )
        XmTextSetString( tw, default_completion_failed_string );

      XtFree( str );
    }
  }


  return( callback_struct.restart_completion );

} /* TimeCompletionCallback */


/*----------------------------------------------------------------------*/

static void
  TimeSelectedInDialogCB( Widget               tw,
                          XmUbTimeBoxWidget    tbox,
                          XmAnyCallbackStruct  *call_data )
{
  /* Variables. */
  XmUbTimeBoxCallbackStruct  cb;
  Boolean                    known_field = True;
  Widget                     slider;
  char                       *str;
  TIM_TIME_REF               time;

  /* Code. */

  /* Get the id of the slider widget. */
  slider = TimeSliderId( tbox );

  if( slider == NULL )
    printf( "Error in XtNameToWidget call, slider is NULL.\n" );

  /* Get the selected time. */
  str = XmUbTimeSliderGetTimeString( (Widget) slider );

  /* Set the time in the box. */
  switch( tbox -> tbox.field_for_time_selection ){
    case XmUbTB_CHILD_START_TIME: 
     XmUbTimeBoxSetStartTimeString( (Widget) tbox, str );
     break;

    case XmUbTB_CHILD_END_TIME:
      XmUbTimeBoxSetEndTimeString( (Widget) tbox, str );
      break;

    default:
      /* Not a recognized field. */
      known_field = False;
      break;
  }

  /* Get rid of the dialog. */
  PopdownTimeSelectionDialogCB( tw, tbox, (XmAnyCallbackStruct *) call_data );


  /* Cleanup. */
  XtFree( str );

  /* Call activate callback. */
  if( known_field && ( tbox -> tbox.activate_callback != NULL ) ){

    /* Set up callback structure. */

    cb.reason      = XmUbCR_TIME_PICKED;
    cb.event       = NULL;
    cb.child_index = tbox -> tbox.field_for_time_selection;
    cb.child       = tbox -> tbox.internal_children[ cb.child_index ];

    XtCallCallbackList( (Widget) tbox, tbox -> tbox.activate_callback,
                        (XtPointer) &cb );
  }


  return;

} /* TimeSelectedInDialogCB */


/*----------------------------------------------------------------------*/

Widget
  TimeSliderId( XmUbTimeBoxWidget  tbox )
{
  /* Variables. */
  char    *name_buffer;
  Widget  slider;
  String  wname;

  /* Code. */

  wname = XtName( tbox );

  /* Get the id of the slider widget. */
  name_buffer = XtMalloc( strlen( wname ) * 2 + 8 );
  sprintf( name_buffer, "%sTs", wname );

  slider = XtNameToWidget( tbox -> tbox.time_selection_popup, name_buffer );

  XtFree( name_buffer );


  return( slider );

} /* TimeSliderId */


/*----------------------------------------------------------------------*/

#if XmVersion > 1001

static void
  TransferDateField( Widget         w,
                     XtPointer      closure,
                     Atom           *seltype,
                     Atom           *type,
                     XtPointer      value,
                     unsigned long  *length,
                     int            format )
{
  /* Variables. */
  Arg                        args[ 2 ];
  char                       buffer[ TIME_STRING_BUFFER_LEN ];
  XmUbTimeBoxCallbackStruct  cb;
  Atom                       compound_text_atom;
  Atom                       date_transfer_atom;
  Cardinal                   n;
  Boolean                    ok;
  XmTextPosition             pos;
  Boolean                    range_transfer = False;
  char                       *str;
  XmUbTimeBoxWidget          tbox;
  TIM_TIME_REF               time;
  XmUbDateTransferRef        transfer_data;
  Widget                     tw;
  XmString                   xm;

  /* Code. */

  compound_text_atom = XmInternAtom( XtDisplay( w ), "COMPOUND_TEXT", False );
  date_transfer_atom = XmInternAtom( XtDisplay( w ), XmUbDATE_TRANSFER, 
                                     False );

  /* The text widget is passed in the parameter closure. */
  tw   = (Widget) closure;
  tbox = (XmUbTimeBoxWidget) XtParent( tw );

  if( *type == date_transfer_atom ){
    /* We are being passed a structure with date information. 
       Construct a string with the date in the local format. */

    /* Convert the first date, regardless of the type. */
    transfer_data = (XmUbDateTransferRef) value;

    time = TimMakeTime( transfer_data -> year, transfer_data -> month,
                        transfer_data -> day, 0, 0, 0 );
    ConvertToDateString( tbox, time, buffer, TIME_STRING_BUFFER_LEN - 1 );

    if( transfer_data -> type == XmUbTRANSFER_DATE ){

      /* Replace the text with the new date. */
      XmTextSetString( tw, buffer );

    } else {

      /* This is a range transfer. Set the start date. */
      (void) XmUbTimeBoxSetStartDateString( (Widget) tbox, buffer );

      /* Convert end of range. */
      if( ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DD ) ||
          ( tbox -> tbox.widget_format == XmUbTB_FORMAT_DTDT ) ){

        time = TimMakeTime( transfer_data -> end_year, 
                            transfer_data -> end_month,
                            transfer_data -> end_day, 0, 0, 0 );
        ConvertToDateString( tbox, time, buffer, TIME_STRING_BUFFER_LEN - 1 );

        (void) XmUbTimeBoxSetEndDateString( (Widget) tbox, buffer );

        range_transfer = True;

      }
    }


    /* Call the activate callback. */
    if( tbox -> tbox.activate_callback != NULL ){

      cb.reason      = XmUbCR_DATE_DROPPED;
      cb.event       = NULL;
      cb.child_index = ChildIndex( tbox, tw );
      cb.child       = tw;
      cb.range       = range_transfer;

      XtCallCallbackList( (Widget) tbox, tbox -> tbox.activate_callback,
                          (XtPointer) &cb );
    }


  } else if( *type == compound_text_atom ){
    /* We are getting compound text. Need to convert it to normal string. */
    xm = XmCvtCTToXmString( value );
    ok = XmStringGetLtoR( xm, XmSTRING_DEFAULT_CHARSET, &str );

    /* Insert the string at the current position. */
    if( ok ){

      pos = XmTextGetInsertionPosition( tw );
      XmTextInsert( tw, pos, str );

      XtFree( str );
    }
    
    XmStringFree( xm );
  }


  return;

} /* TransferDateField */

#endif

/*----------------------------------------------------------------------*/

static void
  ValueChangedCB( Widget               tw,
                  XmUbTimeBoxWidget    tbox,
                  XmAnyCallbackStruct  *call_data )
{
  /* Variables. */
  XmUbTimeBoxCallbackStruct  cb;

  /* Code. */

  if( tbox -> tbox.value_changed_callback == NULL )
    return;

  /* Set up callback structure. */

  cb.reason      = XmCR_VALUE_CHANGED;
  cb.event       = call_data -> event;
  cb.child_index = ChildIndex( tbox, tw );
  cb.child       = tw;

  XtCallCallbackList( (Widget) tbox, 
                      tbox -> tbox.value_changed_callback,
                      (XtPointer) &cb );


  return;

} /* ValueChangedCB */


/*----------------------------------------------------------------------*/

static void
  WarningNoResourceChange( XmUbTimeBoxWidget  tbox,
                           String             resource )
{
  /* Variables. */
  Cardinal  num_params;
  String    params[ 2 ];

  /* Code. */

  params[ 0 ] = resource;
  params[ 1 ] = XtClass( tbox ) -> core_class.class_name;
  num_params  = 2;
  XtAppWarningMsg( XtWidgetToApplicationContext( (Widget) tbox ),
		   "resourceError", "setValues", "WidgetError",
		   "Resource %s may not be changed in %s widgets.",
		   params, &num_params );

  return;

} /* WarningNoResourceChange */


/*----------------------------------------------------------------------*/

Widget  
  XmUbCreateTimeBox( Widget    parent,
                     String    name,
                     ArgList   arglist,
                     Cardinal  argcount )
{

  /* Code. */

  return XtCreateWidget( name, xmUbTimeBoxWidgetClass, 
                         parent, arglist, argcount );

} /* XmUbCreateTimeBox */


/*----------------------------------------------------------------------*/

Widget 
  XmUbTimeBoxGetChild( Widget  widget,
                       int     child )
{

  /* Variables. */
  XmUbTimeBoxWidget  tbox;

  /* Code. */

  tbox = (XmUbTimeBoxWidget) widget;


  return( tbox -> tbox.internal_children[ child ] );
  

} /* XmUbTimeBoxGetChild */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  XmUbTimeBoxGetEndDateString( Widget  widget,
                               char    **str )
{
  /* Variables. */
  XmUbTimeBoxStatus  status;

  /* Code. */

  status = GetChildString( (XmUbTimeBoxWidget) widget, 
                           XmUbTB_CHILD_END_DATE, str );


  return( status );

} /* XmUbTimeBoxGetEndDateString */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  XmUbTimeBoxGetEndTimeString( Widget  widget,
                               char    **str )
{
  /* Variables. */
  XmUbTimeBoxStatus  status;

  /* Code. */

  status = GetChildString( (XmUbTimeBoxWidget) widget, 
                           XmUbTB_CHILD_END_TIME, str );


  return( status );

} /* XmUbTimeBoxGetEndTimeString */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  XmUbTimeBoxGetStartDateString( Widget  widget,
                                 char    **str )
{
  /* Variables. */
  XmUbTimeBoxStatus  status;

  /* Code. */

  status = GetChildString( (XmUbTimeBoxWidget) widget, 
                           XmUbTB_CHILD_START_DATE, str );


  return( status );

} /* XmUbTimeBoxGetStartDateString */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  XmUbTimeBoxGetStartTimeString( Widget  widget,
                                 char    **str )
{
  /* Variables. */
  XmUbTimeBoxStatus  status;

  /* Code. */

  status = GetChildString( (XmUbTimeBoxWidget) widget, 
                           XmUbTB_CHILD_START_TIME, str );


  return( status );

} /* XmUbTimeBoxGetStartTimeString */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  XmUbTimeBoxGetEndDate( Widget   widget,
                         time_t   *date )
{
  /* Variables. */
  char               *datestr;
  XmUbTimeBoxStatus  status;
  TIM_STATUS_TYPE    tim_status;
  XmUbTimeBoxWidget  tbox;
  TIM_TIME_REF       tmp_date;

  /* Code. */

  status = XmUbTimeBoxGetEndDateString( widget, &datestr );

  if( status != TBOX_OK )
    return( status );

  /* Try to convert to date. */
  tim_status = TimMakeDateFromString( &tmp_date, datestr );

  if( tim_status == TIM_OK )
    *date = tmp_date;
  else
    status = TBOX_CONV_ERROR;

  /* Cleanup. */
  XtFree( datestr );


  return( status ); 

} /* XmUbTimeBoxGetEndDate */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  XmUbTimeBoxGetStartDate( Widget   widget,
                           time_t   *date )
{
  /* Variables. */
  char               *datestr;
  XmUbTimeBoxStatus  status;
  TIM_STATUS_TYPE    tim_status;
  XmUbTimeBoxWidget  tbox;
  TIM_TIME_REF       tmp_date;

  /* Code. */

  status = XmUbTimeBoxGetStartDateString( widget, &datestr );

  if( status != TBOX_OK )
    return( status );

  /* Try to convert to date. */
  tim_status = TimMakeDateFromString( &tmp_date, datestr );

  if( tim_status == TIM_OK )
    *date = tmp_date;
  else
    status = TBOX_CONV_ERROR;

  /* Cleanup. */
  XtFree( datestr );


  return( status ); 

} /* XmUbTimeBoxGetStartDate */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  XmUbTimeBoxGetEndTime( Widget   widget,
                         time_t   *time )
{
  /* Variables. */
  char               *timestr;
  XmUbTimeBoxStatus  status;
  TIM_STATUS_TYPE    tim_status;
  XmUbTimeBoxWidget  tbox;
  TIM_TIME_REF       tmp_time;

  /* Code. */

  status = XmUbTimeBoxGetEndTimeString( widget, &timestr );

  if( status != TBOX_OK )
    return( status );

  /* Try to convert to date. */
  tim_status = TimMakeTimeFromString( &tmp_time, timestr );

  if( tim_status == TIM_OK )
    *time = tmp_time;
  else
    status = TBOX_CONV_ERROR;

  /* Cleanup. */
  XtFree( timestr );


  return( status ); 

} /* XmUbTimeBoxGetEndTime */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  XmUbTimeBoxGetStartTime( Widget   widget,
                           time_t   *time )
{
  /* Variables. */
  char               *timestr;
  XmUbTimeBoxStatus  status;
  TIM_STATUS_TYPE    tim_status;
  XmUbTimeBoxWidget  tbox;
  TIM_TIME_REF       tmp_time;

  /* Code. */

  status = XmUbTimeBoxGetStartTimeString( widget, &timestr );

  if( status != TBOX_OK )
    return( status );

  /* Try to convert to date. */
  tim_status = TimMakeTimeFromString( &tmp_time, timestr );

  if( tim_status == TIM_OK )
    *time = tmp_time;
  else
    status = TBOX_CONV_ERROR;

  /* Cleanup. */
  XtFree( timestr );


  return( status ); 

} /* XmUbTimeBoxGetStartTime */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  XmUbTimeBoxSetEndDate( Widget  widget,
                         time_t  date )
{
  /* Variables. */
  char               buffer[ TIME_STRING_BUFFER_LEN ];
  XmUbTimeBoxWidget  tbox;

  /* Code. */

  tbox = (XmUbTimeBoxWidget) widget;

  /* Convert the date to a string. */
  ConvertToDateString( tbox, date, buffer, TIME_STRING_BUFFER_LEN - 1 );
    
  /* Set the string. */
  return( SetChildString( tbox, XmUbTB_CHILD_END_DATE, buffer ) );

} /* XmUbTimeBoxSetEndDate */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  XmUbTimeBoxSetEndTime( Widget  widget,
                         time_t  time )
{
  /* Variables. */
  char               buffer[ TIME_STRING_BUFFER_LEN ];
  XmUbTimeBoxWidget  tbox;

  /* Code. */

  tbox = (XmUbTimeBoxWidget) widget;

  /* Convert the date to a string. */
  ConvertToTimeString( tbox, time, buffer, TIME_STRING_BUFFER_LEN - 1 );
    
  /* Set the string. */
  return( SetChildString( tbox, XmUbTB_CHILD_END_TIME, buffer ) );

} /* XmUbTimeBoxSetEndTime */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  XmUbTimeBoxSetStartDate( Widget  widget,
                           time_t  date )
{
  /* Variables. */
  char               buffer[ TIME_STRING_BUFFER_LEN ];
  XmUbTimeBoxWidget  tbox;

  /* Code. */

  tbox = (XmUbTimeBoxWidget) widget;

  /* Convert the date to a string. */
  ConvertToDateString( tbox, date, buffer, TIME_STRING_BUFFER_LEN - 1 );
    
  /* Set the string. */
  return( SetChildString( tbox, XmUbTB_CHILD_START_DATE, buffer ) );

} /* XmUbTimeBoxSetStartDate */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  XmUbTimeBoxSetStartTime( Widget  widget,
                           time_t  time )
{
  /* Variables. */
  char               buffer[ TIME_STRING_BUFFER_LEN ];
  XmUbTimeBoxWidget  tbox;

  /* Code. */

  tbox = (XmUbTimeBoxWidget) widget;

  /* Convert the date to a string. */
  ConvertToTimeString( tbox, time, buffer, TIME_STRING_BUFFER_LEN - 1 );
    
  /* Set the string. */
  return( SetChildString( tbox, XmUbTB_CHILD_START_TIME, buffer ) );

} /* XmUbTimeBoxSetStartTime */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  XmUbTimeBoxSetEndDateString( Widget  widget,
                               char    *str )
{
  /* Code. */

  return( SetChildString( (XmUbTimeBoxWidget) widget, 
                          XmUbTB_CHILD_END_DATE, str ) );

} /* XmUbTimeBoxSetEndDateString */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  XmUbTimeBoxSetEndTimeString( Widget  widget,
                               char    *str )
{
  /* Code. */

  return( SetChildString( (XmUbTimeBoxWidget) widget, 
                          XmUbTB_CHILD_END_TIME, str ) );

} /* XmUbTimeBoxSetEndTimeString */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  XmUbTimeBoxSetStartDateString( Widget  widget,
                                 char    *str )
{
  /* Code. */

  return( SetChildString( (XmUbTimeBoxWidget) widget, 
                          XmUbTB_CHILD_START_DATE, str ) );

} /* XmUbTimeBoxSetStartDateString */


/*----------------------------------------------------------------------*/

XmUbTimeBoxStatus
  XmUbTimeBoxSetStartTimeString( Widget  widget,
                                 char    *str )
{
  /* Code. */

  return( SetChildString( (XmUbTimeBoxWidget) widget, 
                          XmUbTB_CHILD_START_TIME, str ) );

} /* XmUbTimeBoxSetStartTimeString */


/*----------------------------------------------------------------------*/

void
  XmUbTimeBoxPopupDateDialog( Widget  widget,
                              int     child_index )
{
  /* Variables. */
  Widget  child;

  /* Code. */

  child = XmUbTimeBoxGetChild( widget, child_index );

  if( child != NULL )
    PopupDateSelection( child );


  return;

} /* XmUbTimeBoxPopupDateDialog */


/*----------------------------------------------------------------------*/

void
  XmUbTimeBoxPopupTimeDialog( Widget  widget,
                              int     child_index )
{
  /* Variables. */
  Widget  child;

  /* Code. */

  child = XmUbTimeBoxGetChild( widget, child_index );

  if( child != NULL )
    PopupTimeSelection( child );


  return;

} /* XmUbTimeBoxPopupTimeDialog */
