/*----------------------------------------------------------------------------
--
--  Module:           XmUbMonthDisplay
--
--  Project:          XmUb - Ulle's Motif widgets
--  System:           
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    This is the public header file for the widget.
--
--    The month display widget displays the name and year of a month and
--    the dates according to the day in the week. Optionally, arrow buttons
--    for stepping forwards and backwards month-wise or year-wise are
--    included.
--    The user may select a date by clicking on it. When that happens, a 
--    callback is invoked.
--
--  Filename:         XmUbMonthD.h
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1993-09-26
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
/* SCCSID = @(#) Module: XmUbMonthD.h, Version: 1.1, Date: 95/02/18 15:10:08 */

#ifndef _XmUbMonthD_h
#define _XmUbMonthD_h

/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <time.h>
#include <X11/StringDefs.h>
#include <Xm/Xm.h>

/*----------------------------------------------------------------------------
--  Resources for the month display widget.
--  All XmManager resources apply. 
--  The following list describes resources added in this widget.
--
--  XmUbNmdiArrowSpacing              CSG
--      XmCSpacing
--    The vertical spacing between the date display and the arrow buttons 
--    (if any).
--    Type:    Dimension
--    Default: 2
--                        
--  XmUbNmdiCurrentMonth              CSG
--      XmUbCMdiCurrentMonth
--    The number of the month that is currently being displayed. 
--    An integer in the range 1 .. 12.
--    Type:    int
--    Default: dynamic (the current month)
--                        
--  XmUbNmdiCurrentYear               CSG
--      XmUbCMdiCurrentYear
--    The number of the year that is currently being displayed. 
--    Type:    int
--    Default: dynamic (the current year)
--                        
--  XmUbNmdiDateSelectedCallback      CSG
--      XmCCallback
--    Callback(s) that is called when a date was selected.
--    Callback reason: XmUbCR_DATE_SELECTED.
--    Type:    XtCallbackList
--    Default: NULL
--
--  XmUbNmdiDayChars                  CG
--      XmUbCMdiDayChars
--    The max number of characters to show in the day name. If 0, no 
--    abbreviation of the string returned by strftime( "%a" ) is done.
--    Type:    int
--    Default: 2
--                        
--  XmUbNmdiEnableSeparator           CG
--      XmUbCMdiEnableSeparator
--    If True, a separator is created betweeen the row column and the arrow
--    labels, but only if at least one arrow label is enabled.
--    If False, no separator is created.
--    Type:    Boolean
--    Default: True
--
--  XmUbNmdiFillOutWeek               CSG
--      XmUbCMdiFillOutWeek
--    If True, the week display is filled out with days from the preceding and 
--    following months. If False, these positions are left empty.
--    Type:    Boolean
--    Default: True
--                        
--  XmUbNmdiLabelSpacing              CSG
--      XmCSpacing
--    The vertical spacing between the date display and the title.
--    Type:    Dimension
--    Default: 2
--                        
--  XmNmarginHeight                   CSG
--      XmCMarginHeight
--    The margin used at the top and the bottom of the widget.
--    Type:    Dimension
--    Default: 0
--                        
--  XmNmarginWidth                    CSG
--      XmCMarginWidth
--    The margin used at the left and right ends of the widget.
--    Type:    Dimension
--    Default: 0
--                        
--  XmUbNmdiMonthArrows               CG
--      XmUbCMdiMonthArrows
--    If True, arrow buttons for month scrolling are created below the month 
--    display.
--    Type:    Boolean
--    Default: True
--                        
--  XmUbNmdiMonthLabel                CSG
--      XmUbCMdiMonthLabel
--    The text to be displayed in the month scrolling label, if enabled. 
--    If none is specified, a default string will be used.
--    The string is copied internally. A copy is returned by XtGetValues.
--    Type:    XmString
--    Default: NULL
--                        
--  XmUbNmdiNewMonthCallback          CSG
--      XmCCallback
--    Callback(s) that is called when a new month is displayed.
--    Callback reason: XmUbCR_NEW_MONTH.
--    Type:    XtCallbackList
--    Default: NULL
--                       
--  XmUbNmdiOtherMonthForeground      CSG
--      XmCForeground
--    The color in which the dates of the adjoining months are displayed. 
--    Only applicable if XmUbNmdiFillOutWeek is True.
--    Type:    Pixel
--    Default: the default foreground color
--                        
--  XmUbNmdiRecomputeHeight           CSG
--      XmUbCMdiRecomputeHeight
--    If True, the height of the widget is dynamically calculated. If False, 
--    the user set height is kept. The resource is forced to False if the user 
--    sets a height.
--    Type:    Boolean
--    Default: True
--                        
--  XmUbNmdiRecomputeWidth            CSG
--      XmUbCMdiRecomputeWidth
--    If True, the width of the widget is dynamically calculated. If False, 
--    the user set width is kept. The resource is forced to False if the user 
--    sets a width.
--    Type:    Boolean
--    Default: True
--                        
--  XmUbNmdiSaturdayForeground        CSG
--      XmCForeground
--    The foreground color used for Saturdays.
--    Type:    Pixel
--    Default: the default foreground color
--
--  XmUbNmdiSundayForeground          CSG
--      XmCForeground
--    The foreground color used for Sundays.
--    Type:    Pixel
--    Default: the default foreground color
--
--  XmUbNmdiTitleAlignment            CSG
--      XmCAlignment
--    The alignment of the month title.
--    Valid values are XmALIGNMENT_BEGINNING, XmALIGNMENT_CENTER and 
--    XmALIGNMENT_END.
--    Type:    unsigned char
--    Default: XmALIGNMENT_CENTER
--                        
--  XmUbNmdiTitleCallback             CSG
--      XmCCallback
--    Callback(s) that is called when the title bar is pressed.
--    If the resource is True, the current month and year will be set 
--    before this callback is invoked.
--    Callback reason: XmUbCR_TITLE_PRESSED.
--    Type:    XtCallbackList
--    Default: NULL 
--
--  XmUbNmdiTitleShadows              CG
--      XmUbCMdiTitleShadows
--    If True, the title bar pushbutton will have a shadow thickness of 1.
--    If False, no shadows will be drawn. 
--    Set this resource to False if there is no action on the title bar.
--    Type:    Boolean
--    Default: True
--                        
--  XmUbNmdiTodayForeground           CSG
--      XmCForeground
--    The foreground color used for the current date.
--    Type:    Pixel
--    Default: the default foreground color
--
--  XmUbNmdiUpdateDayCallback         CSG
--      XmCCallback
--    Callback(s) that is called before a new date is displayed.
--    Callback reason: XmUbCR_UPDATE_DAY
--    Type:    XtCallbackList
--    Default: NULL 
--
--  XmUbNmdiUseDefaultTitleCallback   CSG
--      XmUbCMdiUseDefaultTitleCallback
--    If True, the current month and year are set when the user presses the
--    title bar. If False, nothing is done.
--    The user's XmUbNmdiTitleCallback is called anyway, in the case of True
--    after the month and year have been set.
--    Type:    Boolean
--    Default: True
--                        
--  XmUbNmdiWeekdayForeground         CSG
--      XmCForeground
--    The color in which the weekdays of the current month are displayed. 
--    Type:    Pixel
--    Default: the default foreground color
--                        
--  XmUbNmdiWeekLabel                 CG
--      XmUbCMdiWeekLabel
--    The label displayed on the same level as the day name, in the week
--    number column. Only used if XmUbNmdiWeekNumbers is True.
--    If NULL, a default string is used.
--    The string is copied internally. A copy is returned by XtGetValues.
--    Type:    XmString
--    Default: NULL
--
--  XmUbNmdiWeekNumberForeground      CSG
--      XmCForeground
--    The color in which to display the week numbers.
--    Only used if XmUbNmdiWeekNumbers is True.
--    Type:    Pixel
--    Default: the default foreground color
--
--  XmUbNmdiWeekNumbers               CG
--      XmUbCMdiWeekNumbers
--    If True, week numbers are displayed to the left of the day numbers.
--    If False, no week numbers are displayed.
--    Type:    Boolean
--    Default: True
--
--  XmUbNmdiYearArrows                CG
--      XmUbCMdiYearArrows
--    If True, arrow buttons for year scrolling are created below the month 
--    display.
--    Type:    Boolean
--    Default: True
--                         
--  XmUbNmdiYearLabel                 CSG
--      XmUbCMdiYearLabel
--    The text to be displayed in the year scrolling label, if enabled. 
--    If none is specified, a default string will be used.
--    The string is copied internally. A copy is returned by XtGetValues.
--    Type:    XmString
--    Default: NULL
--                        
----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#define XmUbNmdiArrowSpacing              "mdiArrowSpacing"
#define XmUbNmdiCurrentMonth              "mdiCurrentMonth"
#define XmUbNmdiCurrentYear               "mdiCurrentYear"
#define XmUbNmdiDateSelectedCallback      "mdiDateSelectedCallback"
#define XmUbNmdiDayChars                  "mdiDayChars"
#define XmUbNmdiEnableSeparator           "mdiEnableSeparator"
#define XmUbNmdiFillOutWeek               "mdiFillOutWeek"
#define XmUbNmdiLabelSpacing              "mdiLabelSpacing"
#define XmUbNmdiMonthArrows               "mdiMonthArrows"
#define XmUbNmdiMonthLabel                "mdiMonthLabel"
#define XmUbNmdiNewMonthCallback          "mdiNewMonthCallback"
#define XmUbNmdiOtherMonthForeground      "mdiOtherMonthForeground"
#define XmUbNmdiRecomputeHeight           "mdiRecomputeHeight"
#define XmUbNmdiRecomputeWidth            "mdiRecomputeWidth"
#define XmUbNmdiSaturdayForeground        "mdiSaturdayForeground"
#define XmUbNmdiSundayForeground          "mdiSundayForeground"
#define XmUbNmdiTitleAlignment            "mdiTitleAlignment"
#define XmUbNmdiTitleCallback             "mdiTitleCallback"
#define XmUbNmdiTitleShadows              "mdiTitleShadows"
#define XmUbNmdiTodayForeground           "mdiTodayForeground"
#define XmUbNmdiUpdateDayCallback         "mdiUpdateDayCallback"
#define XmUbNmdiUseDefaultTitleCallback   "mdiUseDefaultTitleCallback"
#define XmUbNmdiWeekdayForeground         "mdiWeekdayForeground"
#define XmUbNmdiWeekLabel                 "mdiWeekLabel"
#define XmUbNmdiWeekNumberForeground      "mdiWeekNumberForeground"
#define XmUbNmdiWeekNumbers               "mdiWeekNumbers"
#define XmUbNmdiYearArrows                "mdiYearArrows"
#define XmUbNmdiYearLabel                 "mdiYearLabel"

#define XmUbCMdiCurrentMonth              "MdiCurrentMonth"
#define XmUbCMdiCurrentYear               "MdiCurrentYear"
#define XmUbCMdiDayChars                  "MdiDayChars"
#define XmUbCMdiFillOutWeek               "MdiFillOutWeek"
#define XmUbCMdiEnableSeparator           "MdiEnableSeparator"
#define XmUbCMdiMonthArrows               "MdiMonthArrows"
#define XmUbCMdiMonthLabel                "MdiMonthLabel"
#define XmUbCMdiRecomputeHeight           "MdiRecomputeHeight"
#define XmUbCMdiRecomputeWidth            "MdiRecomputeWidth"
#define XmUbCMdiTitleShadows              "MdiTitleShadows"
#define XmUbCMdiUseDefaultTitleCallback   "MdiUseDefaultTitleCallback"
#define XmUbCMdiWeekLabel                 "MdiWeekLabel"
#define XmUbCMdiWeekNumbers               "MdiWeekNumbers"
#define XmUbCMdiYearArrows                "MdiYearArrows"
#define XmUbCMdiYearLabel                 "MdiYearLabel"


/* Id's for children retrieval. */
#define XmUbMD_CHILD_TITLE          0
#define XmUbMD_CHILD_ROW_COLUMN     1
#define XmUbMD_CHILD_YEAR_ARROW     2
#define XmUbMD_CHILD_MONTH_ARROW    3
#define XmUbMD_CHILD_WEEK_RC        4
#define XmUbMD_CHILD_SEPARATOR      5

#define XmUbMD_FIRST_CHILD  XmUbMD_CHILD_TITLE
#define XmUbMD_LAST_CHILD   XmUbMD_CHILD_SEPARATOR

/* Callback reasons. */
#define XmUbCR_DATE_SELECTED   0
#define XmUbCR_NEW_MONTH       1
#define XmUbCR_TITLE_PRESSED   2
#define XmUbCR_UPDATE_DAY      3
#define XmUbCR_UPDATE_WEEK     4

/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

typedef enum { XmUbCURRENT_MONTH, 
               XmUbPREVIOUS_MONTH, 
               XmUbNEXT_MONTH } XmUbWhichMonth;

/* Callback structure. */

typedef struct {

  int      reason;
  XEvent   *event;
  Widget   child;
  Boolean  range;

  int      selected_day;
  int      selected_month;
  int      selected_year;

  int      end_day;
  int      end_month;
  int      end_year;
  
} XmUbMonthDisplayCallbackStruct;


typedef struct {

  int             reason;
  XEvent          *event;
  int             day;
  int             month;
  int             year;
  int             weekday;
  int             day_num_in_year;
  int             week_number;
  Pixel           foreground;
  Pixel           background;
  Pixel           frame;
  Boolean         use_shadows;
  Boolean         mapped;
  Pixel           default_foreground;
  Pixel           default_background;
  Pixel           default_frame;
  Boolean         default_mapped;
  XmUbWhichMonth  which_month;
  int             loop_index;
  Boolean         today;
  
} XmUbMdiUpdateDayCallbackStruct;

/* 
-- CALLBACKS
--
-- XmUbNmdiDateSelectedCallback
--   Called when a date or week button is pressed.
--   
--   reason              XmUbCR_DATE_SELECTED
--   event               The button press callback event.
--   child               The widget id for the button that was pressed.
--   range               True if a range was selected, False if a date.
--                       If True, the end_... fields are valid.
--   selected_day        The day number that was selected. (1..31)
--   selected_month      The month that was selected. (1..12)
--                       May be different from the current month.
--   selected_year       The year that was selected. (1970..)
--                       May be different from the current year.
--   end_day             The end day number if a range was selected.
--   end_month           The end month number if a range was selected.
--   end_year            The end year if a range was selected.
--
-- XmUbNmdiNewMonthCallback
--   Called when a new month is displayed.
--   Enables the user to set fore- and background colors and frames, as well
--   as setting the keyboard focus for the new month.
--   To enable correct setup at start, this callback is also invoked when
--   the widget is exposed for the first time.
--
--   reason              XmUbCR_NEW_MONTH
--   event               NULL
--   selected_month      The new month. (1..12)
--   selected_year       The new year. (1970..)
--
--    selected_year and selected_month are always equal to the current year
--    and the current month.
--
-- XmUbNmdiTitleCallback
--   Called when the title bar is pressed.
--   See also resource XmUbNmdiUseDefaultTitleCallback.
--
--   reason              XmUbCR_TITLE_PRESSED
--   event               The button press event for the title button.
--   selected_month      The current month. (1..12)
--   selected_year       The current year. (1970..)
--
-- XmUbNmdiUpdateDayCallback
--   Called when a new date is about to be displayed.
--   If this callback is set, the widget does not set the colors when a new
--   date is displayed. Instead, the current colors are extracted and put 
--   in the callback structure. The user may then change these colors in the 
--   callback function. The processed values are then set in the widget after
--   the callback has returned.
--   If no callback is set, the widget displays the dates according to its
--   resources.
--   
--   The callback structure is of the type XmUbMdiUpdateDayCallbackStruct.
--
--   reason              XmUbCR_UPDATE_DAY
--   event               NULL
--   day                 The day that is about to be displayed.
--   month               The current month. (1..12)
--   year                The current year. (1970..)
--   weekday             The day of the week ( 1 = Monday,... 7 = Sunday)
--   day_num_in_year     The day number in the year.
--   week_number         The number of the current week.
--   foreground          The current foreground color.
--   background          The current background color.
--   frame               The current frame color.
--   use_shadows         Always set to False. If user sets this field to 
--                       True, the default shadow colors are used, overriding
--                       the frame color.
--   mapped              True if the date is mapped, False if not.
--   default_foreground  The new foreground color according to resources.
--   default_background  The new background color according to resources.
--   default_frame       The new frame color according to resources.
--   default_mapped      The mapped status according to resources.
--   which_month         If the date is in the current month, the one before
--                       or the one after.
--   loop_index          The loop index of the date.
--   today               True if the day in question is "today".
--
--  The callback is also called when week numbers are updated.
--  The following fields are then valid.
--
--   reason              XmUbCR_UPDATE_WEEK
--   event               NULL
--   day                 The day of the first day in the week.
--   month               The month of the first day in the week.
--   year                The year of the first day in the week.
--   week_number         The number of the current week.
--   foreground          The current foreground color.
--   background          The current background color.
--   frame               The current frame color.
--   use_shadows         Always set to False. If user sets this field to 
--                       True, the default shadow colors are used, overriding
--                       the frame color.
--   mapped              True if the button is mapped, False if not.
--   default_foreground  The new foreground color according to resources.
--   default_background  The new background color according to resources.
--   default_frame       The new frame color according to resources.
--   default_mapped      The mapped status according to resources.
--   loop_index          The loop index of the date.
--
*/

/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Class record constants. */
extern WidgetClass xmUbMonthDisplayWidgetClass;


typedef struct _XmUbMonthDisplayClassRec *XmUbMonthDisplayWidgetClass;
typedef struct _XmUbMonthDisplayRec      *XmUbMonthDisplayWidget;


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

Widget  
  XmUbCreateMonthDisplay( Widget    parent,
                          String    name,
                          ArgList   arglist,
                          Cardinal  argcount );

Widget 
  XmUbMonthDisplayGetChild( Widget  widget,
                            int     child );

/* Sets the keyboard focus to the day_number in the current month. */
void
  XmUbMonthDisplaySetFocusToDay( Widget  widget,
                                 int     day_number );

void
  XmUbMonthDisplaySetDateBackground( Widget  widget,
                                     int     day_number,
                                     Pixel   bg );

void
  XmUbMonthDisplaySetDateForeground( Widget  widget,
                                     int     day_number,
                                     Pixel   fg );

void
  XmUbMonthDisplaySetDateFrame( Widget  widget,
                                int     day_number,
                                Pixel   color );

/* Year and month may be set to zero. In that case, their values are not
   changed, but a redisplay is triggered anyway. */
void
  XmUbMonthDisplaySetMonth( Widget  widget,
                            int     year,
                            int     month );

#endif


