/*----------------------------------------------------------------------------
--
--  Module:           xtmRestore
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Restore a backup of the XDiary database. The backup must follow
--    the format used by the XDdump program.
--
--  Filename:         xtmRestore.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-10-16
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmRestore.c, Version: 1.1, Date: 95/02/18 16:10:59";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <time.h>

#include <X11/Intrinsic.h>

#include "System.h"
#include "TimDate.h"

#include "xtmGlobal.h"
#include "xtmDbTools.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Name of program. */
#define PROGRAM_NAME   "xdrestore"

/* Start of backup entry. */
#define ENTRY_START  "@!XDiaryDump"

#define max( a, b )  (a < b ? b : a )


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

typedef struct {
  FILE  *file_ref;
} BACKUP_FILE_DEF, *BACKUP_FILE_REF;



/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of program. */
static char  *program_name;

/* Name of module. */
static char  *module_name = "xtmRestoreDiary";

static Boolean  log = False;

/* Default alarm times. */
static int 
  def_alarm_time[] = { -10, -5, 0, 0, 0 };

static int 
  def_alarm_valid[] = { 1, 1, 1, 0, 0 };



/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  closeBackupFile( BACKUP_FILE_REF  backup_ref );

static void
  closeDatabase( XTM_DB_ENTRY_DATABASES  *database_ref );

static void
  displayUsage();

static Boolean
  fetchEntry( BACKUP_FILE_REF       backup_ref,
              XTM_DB_ALL_ENTRY_REF  all_entry_ref,
              char                  **entry_text,
              char                  *entry_id,
              int                   size_entry_id );

static BACKUP_FILE_REF
  initializeBackupFile( char  *file_name );

static int
  noCaseStrcmp( char  *buffer1,
                char  *buffer2 );

static XTM_DB_ENTRY_DATABASES
  *openDatabase( char  *directory );

static Boolean
  saveEntry( char                    *directory,
             XTM_DB_ENTRY_DATABASES  *database_ref,
             UINT32                  operations,
             XTM_DB_ALL_ENTRY_REF    all_entry_ref,
             char                    *entry_text );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

main( int argc, char *argv[] )
{

  /* Variables. */
  Boolean                 do_file_lock = True;
  Boolean                 ok;
  int                     arg_ref;
  UINT32                  operations;
  char                    backup_file[ 250 ];
  char                    diary_directory[ 250 ];
  char                    entry_id[ 256 ];
  char                    *entry_text;
  BACKUP_FILE_REF         backup_ref;
  XTM_DB_ALL_ENTRY_DEF    entry;
  XTM_DB_ENTRY_DATABASES  *database_ref;


  /* Code. */

  /* Fetch the name of the program. */
  program_name = PROGRAM_NAME;


  /* Check arguments. */
  if( argc < 3 ) {
    displayUsage();
    exit( 1 );
  }

  
  /* Initialization. */
  SysInitializeEnvironment();

  
  /* Fetch qualifiers. */
  arg_ref = 1;
  while( arg_ref < argc && argv[ arg_ref ][ 0 ] == '-' ) {

    if( noCaseStrcmp( argv[ arg_ref ], "-log" ) == 0 ) {
      log = True;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-help" ) == 0 ) {
      displayUsage();
      exit( 0 );

    } else if( noCaseStrcmp( argv[ arg_ref ], "-h" ) == 0 ) {
      displayUsage();
      exit( 0 );

    } else if( noCaseStrcmp( argv[ arg_ref ], "-nofilelock" ) == 0 ) {
      do_file_lock = False;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-usage" ) == 0 ) {
      displayUsage();
      exit( 0 );

    } else if( noCaseStrcmp( argv[ arg_ref ], "-version" ) == 0 ) {
      printf( "%s: Version: %s\n", program_name, VERSION_ID );
      exit( 0 );

    } /* if */

    arg_ref++;

  } /* while */


  /* Fetch the required parameters. */
  strcpy( backup_file,     argv[ arg_ref++ ] );
  strcpy( diary_directory, argv[ arg_ref++ ] );


  /* Use file locking? */
  xtmDbUseFileLock( do_file_lock );


  /* Initialize the backup file. */
  backup_ref = initializeBackupFile( backup_file );
  if( backup_ref == NULL ) {
    fprintf( stderr, "%s: Cannot read the backup file %s.\n",
             program_name, backup_file );
    exit( 1 );
  }


  /* Initialize the database (create if not exist). */
  database_ref = openDatabase( diary_directory );
  if( database_ref == NULL ) {
    fprintf( stderr, "%s: Cannot open the diary database %s.\n",
             program_name, diary_directory );
    exit( 1 );
  }


  /* What operations can we do on the database? */
  xtmDbCheckDbOperations( diary_directory, True, &operations );
  if( ! flagIsSet( operations, XTM_DB_FLAG_MODE_WRITE ) ) {
    fprintf( stderr, 
             "%s: You do not have write privileges to the\n"
             "diary database in directory %s.\n",
             program_name, diary_directory );

    closeBackupFile( backup_ref );
    closeDatabase(   database_ref );

    exit( 1 );
  } /* if */


  /* Fetch all entries in the backup file and save them in the database. */
  ok = fetchEntry( backup_ref, 
                   &entry, &entry_text,
                   entry_id, sizeof( entry_id ) );
  while( ok ) {

    /* Save the entry in the diary database. */
    ok = saveEntry( diary_directory, 
                    database_ref, operations,
                    &entry, entry_text );

    /* Free memory. */
    if( entry_text != NULL )
      SysFree( entry_text );

    /* Log the transaction? */
    if( log ) {
      printf( "Saved entry: %s.\n", entry_id );
    }

    /* Fetch the next entry. */
    ok = fetchEntry( backup_ref, 
                     &entry, &entry_text,
                     entry_id, sizeof( entry_id ) );
  } /* while */


  /* Close the database and backup file. */
  closeBackupFile( backup_ref );
  closeDatabase(   database_ref );

  exit( 0 );

} /* main */


/*----------------------------------------------------------------------*/

static void
  closeBackupFile( BACKUP_FILE_REF  backup_ref )
{

  /* Code. */

  if( backup_ref -> file_ref != NULL )
    return;

  fclose( backup_ref -> file_ref );

  SysFree( backup_ref );

  return;

} /* closeBackupFile */


/*----------------------------------------------------------------------*/

static void
  closeDatabase( XTM_DB_ENTRY_DATABASES  *database_ref )
{

  /* Code. */

  xtmDbCloseEntryDb( database_ref );

  SysFree( database_ref );

  return;

} /* closeDatabase */


/*----------------------------------------------------------------------*/

static void
  displayUsage()
{

  printf( 
    "\n"
    "%s (%s): Restore XDiary entries to database.\n" 
    "\n"
    "Usage:\n"
    "  %s [flags] backupFile diaryDir\n"
    "\n"
    "where:\n"
    "  backupFile : File containing backup entries.\n"
    "  diaryDir   : Directory containing the diary database.\n"
    "\n"
    "Flags:\n"
    "  -help        : Display this help.\n"
    "  -h           : See above.\n"
    "  -log         : Turn on logging.\n"
    "  -noFileLock  : Don't use any file locking.\n"
    "  -usage       : Print this message.\n"
    "  -version     : Display the current version.\n"
    "\n",
    program_name, VERSION_ID, program_name );

  return;

} /* displayUsage */


/*----------------------------------------------------------------------*/

static BACKUP_FILE_REF
  initializeBackupFile( char  *file_name )
{

  /* Variables. */
  BACKUP_FILE_REF  backup_ref;
  FILE             *file_ref;


  /* Code. */

  /* Open the file, if we fail, just return. */
  file_ref = fopen( file_name, "r" );
  if( file_ref == NULL )
    return( NULL );

  /* Allocate a new backup record. */
  backup_ref = SysNew( BACKUP_FILE_DEF );
  if( backup_ref == NULL )
    return( NULL );

  backup_ref -> file_ref = file_ref;

  return( backup_ref );

} /* initializeBackupFile */


/*----------------------------------------------------------------------*/

static Boolean
  fetchEntry( BACKUP_FILE_REF       backup_ref,
              XTM_DB_ALL_ENTRY_REF  all_entry_ref,
              char                  **entry_text,
              char                  *entry_id,
              int                   size_entry_id )
{

  /* Variables. */
  int                     index;
  int                     integer1;
  int                     integer2;
  int                     items;
  int                     segment_size = 500;
  int                     text_buffer_size = 0;
  UINT32                  flags;
  char                    buffer[ 1024 ];
  char                    buffer1[ 256 ];
  char                    buffer2[ 256 ];
  char                    *char_ref;
  TIM_STATUS_TYPE         time_status;
  TIM_TIME_REF            time_data;
  XTM_DB_ENTRY_REF        entry_ref       = &all_entry_ref -> entry;
  XTM_DB_STAND_ENTRY_REF  stand_entry_ref = &all_entry_ref -> stand_entry;


  /* Code. */

  *entry_text = NULL;
  *entry_id   = '\0';

  if( backup_ref == NULL )
    return( False );

  /* Skip all lines until an entry start tag. */
  do {
    char_ref = fgets( buffer, sizeof( buffer ), backup_ref -> file_ref );
    if( char_ref == NULL )
      return( False );

    if( strncmp( buffer, ENTRY_START, strlen( ENTRY_START ) ) == 0 )
      break;
  } while( 1 );


  /* Allocate space for the appointment text. */
  *entry_text      = (char *) SysMalloc( segment_size );
  **entry_text     = '\0';
  text_buffer_size = segment_size;

  /* Record initialization. */
  entry_ref -> time_stamp     = 0;
  entry_ref -> date_stamp     = 0;
  entry_ref -> duration       = 0;
  entry_ref -> id             = 0;

  entry_ref -> last_update    = (UINT32) TimMakeTimeNow();
  entry_ref -> flags          = 0;
  entry_ref -> tag[ 0 ]       = '\0';
  entry_ref -> text[ 0 ]      = '\0';
  entry_ref -> alarm_melody   = 1;
  entry_ref -> day_list_lines = 1;
  entry_ref -> alarm_lines    = 1;
  entry_ref -> fg_color_index = 0;
  entry_ref -> bg_color_index = 0;
  entry_ref -> entry_type     = (UINT8) XTM_DB_DAY_ENTRY;
  entry_ref -> entry_category = (UINT8) XTM_DB_ENTRY_LIST;

  for( index = 0; index < 5; index++ ) {
    entry_ref -> alarm_offset[ index ] = (INT16) def_alarm_time[  index ];
    entry_ref -> alarm_valid[  index ] = (UINT8) def_alarm_valid[ index ];
  }

  stand_entry_ref -> id             = 0;
  stand_entry_ref -> flags          = 0;
  stand_entry_ref -> from           = 0;
  stand_entry_ref -> to             = 0;
  stand_entry_ref -> every_n        = 0;
  stand_entry_ref -> valid_every    = 0;
  stand_entry_ref -> skip_week[ 0 ] = 0;
  stand_entry_ref -> skip_week[ 1 ] = 0;

  for( index = 0; index < 7; index++ )
    stand_entry_ref -> valid_days[ index ] = False;


  /* Start reading the entry data, stop at the end marker ($$). */
  do {
    char_ref = fgets( buffer, sizeof( buffer ), backup_ref -> file_ref );
    if( char_ref == NULL )
      return( False );


    /* End of record? */
    if( strncmp( buffer, "$$", 2 ) == 0 ) {
      strncpy( entry_ref -> text, *entry_text, XTM_DB_RECORD_TEXT_LEN );
      entry_ref -> text[ XTM_DB_RECORD_TEXT_LEN ] = '\0';

      return( True );
    }


    /* Entry date, time and duration. */
    if( strncmp( buffer, "Ti", 2 ) == 0 ) {

      items = sscanf( &buffer[ 2 ], "%s %s %d",
                      buffer1, buffer2, &integer1 );
      if( items != 3 ) {
        fprintf( stderr, "%s: Don't know line: %s.\n", program_name, buffer );
        return( False );
      }

      /* Date. */
      time_status = TimMakeDateFromIsoString( &time_data, buffer1 );
      if( time_status != TIM_OK ) {
        fprintf( stderr, "%s: Date error: %s.\n", program_name, buffer1 );
        return( False );
      }

      entry_ref -> date_stamp = (UINT32) time_data;

      /* Time. */
      time_status = TimMakeTimeFromIsoString( &time_data, buffer2 );
      if( time_status != TIM_OK ) {
        fprintf( stderr, "%s: Time error: %s.\n", program_name, buffer1 );
        return( False );
      }
      entry_ref -> time_stamp = (UINT32) time_data;

      /* Duration. */
      if( integer1 < 0 || integer1 > 24 * 60 ) {
        fprintf( stderr, "%s: Duration error: %d.\n", program_name, integer1 );
        return( False );
      }
      entry_ref -> duration = (UINT16) integer1;

      /* This is part of the information to return. */
      if( strlen( buffer1 ) + strlen( buffer2 ) + 3 < size_entry_id ) {
        strcat( entry_id, buffer1 );
        strcat( entry_id, " " );
        strcat( entry_id, buffer2 );
        strcat( entry_id, " " );
      }

    } /* if */


    /* Flags. */
    if( strncmp( buffer, "Fl", 2 ) == 0 ) {

      items = sscanf( &buffer[ 2 ], "%s", buffer1 );
      if( items != 1 ) {
        fprintf( stderr, "%s: Don't know line: %s.\n", program_name, buffer );
        return( False );
      }

      flags = 0;

      if( strchr( buffer1, 'A' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_ALARM );

      if( strchr( buffer1, 'a' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_HIDE_IN_ALARM );

      if( strchr( buffer1, 'b' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_KEEP_IN_BG );

      if( strchr( buffer1, 'c' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_HIDE_IN_CALENDAR );

      if( strchr( buffer1, 'D' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_NOTE_DONE );

      if( strchr( buffer1, 'd' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_HIDE_IN_DAY_VIEW );

      if( strchr( buffer1, 'f' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_KEEP_IN_FG );

      if( strchr( buffer1, 'G' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_ONLY_OWNER_DELETE );

      if( strchr( buffer1, 'g' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_ONLY_OWNER_CHANGE );

      if( strchr( buffer1, 'I' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_IMPORTANT );
 
      if( strchr( buffer1, 'P' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_PRIVATE );

      if( strchr( buffer1, 'p' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_HIDE_IN_PRINT );

      if( strchr( buffer1, 's' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_HIDE_IN_SUMMARY );

      if( strchr( buffer1, 'u' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_HIDE_IN_DUMP );

      flagSet( entry_ref -> flags, flags );

      if( strchr( buffer1, 'S' ) != NULL )
        entry_ref -> entry_category = XTM_DB_REP_ENTRY_LIST;

      if( strchr( buffer1, 'T' ) != NULL )
        entry_ref -> entry_category = XTM_DB_STICKY_LIST;

      if( strchr( buffer1, 'N' ) != NULL )
        entry_ref -> entry_type = XTM_DB_DAY_NOTE;

    } /* if */


    /* Alarm flags. */
    if( strncmp( buffer, "Af", 2 ) == 0 ) {

      items = sscanf( &buffer[ 2 ], "%s", buffer1 );
      if( items != 1 ) {
        fprintf( stderr, "%s: Don't know line: %s.\n", program_name, buffer );
        return( False );
      }

      flags = 0;

      if( strchr( buffer1, 'a' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_ACTION_SCRIPT );

      if( strchr( buffer1, 't' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_ACTION_TEXT );

      if( strchr( buffer1, '1' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_ACTION_ALARM1 );

      if( strchr( buffer1, '2' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_ACTION_ALARM2 );

      if( strchr( buffer1, '3' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_ACTION_ALARM3 );

      if( strchr( buffer1, '4' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_ACTION_ALARM4 );

      if( strchr( buffer1, '5' ) != NULL )
        flagSet( flags, XTM_DB_FLAG_ACTION_ALARM5 );

      flagSet( entry_ref -> flags, flags );

    } /* if */


    /* Standing entry valid from. */
    if( strncmp( buffer, "Sf", 2 ) == 0 ) {
      items = sscanf( &buffer[ 2 ], "%s", buffer1 );
      if( items != 1 ) {
        fprintf( stderr, "%s: Don't know line: %s.\n", program_name, buffer );
        return( False );
      }

      time_status = TimMakeDateFromIsoString( &time_data, buffer1 );
      if( time_status != TIM_OK ) {
        fprintf( stderr, "%s: Date error: %s.\n", program_name, buffer1 );
        return( False );
      }
      stand_entry_ref -> from = (UINT32) time_data;
    }


    /* Standing entry valid to. */
    if( strncmp( buffer, "St", 2 ) == 0 ) {
      items = sscanf( &buffer[ 2 ], "%s", buffer1 );
      if( items != 1 ) {
        fprintf( stderr, "%s: Don't know line: %s.\n", program_name, buffer );
        return( False );
      }

      time_status = TimMakeDateFromIsoString( &time_data, buffer1 );
      if( time_status != TIM_OK ) {
        fprintf( stderr, "%s: Date error: %s.\n", program_name, buffer1 );
        return( False );
      }
      stand_entry_ref -> to = (UINT32) time_data;
    }


    /* Standing entry day in month. */
    if( strncmp( buffer, "Sr", 2 ) == 0 ) {
      items = sscanf( &buffer[ 2 ], "%s", buffer1 );
      if( items != 1 ) {
        fprintf( stderr, "%s: Don't know line: %s.\n", program_name, buffer );
        return( False );
      }

      flagSet( stand_entry_ref -> flags, XTM_DB_FLAG_SE_DAY_IN_MONTH );

      if( buffer1[ 0 ] == 'b' )
        flagSet( stand_entry_ref -> flags, XTM_DB_FLAG_SE_2ND );
      else if( buffer1[ 0 ] == 'c' )
        flagSet( stand_entry_ref -> flags, XTM_DB_FLAG_SE_3RD );
      else if( buffer1[ 0 ] == 'd' )
        flagSet( stand_entry_ref -> flags, XTM_DB_FLAG_SE_1ST_LAST );
      else if( buffer1[ 0 ] == 'e' )
        flagSet( stand_entry_ref -> flags, XTM_DB_FLAG_SE_2ND_LAST );
      else if( buffer1[ 0 ] == 'f' )
        flagSet( stand_entry_ref -> flags, XTM_DB_FLAG_SE_3RD_LAST );
      else
        flagSet( stand_entry_ref -> flags, XTM_DB_FLAG_SE_1ST );
    }


    /* Standing entry valid every. */
    if( strncmp( buffer, "Sn", 2 ) == 0 ) {
      items = sscanf( &buffer[ 2 ], "%d %s", &integer1, buffer1 );
      if( items != 2 ) {
        fprintf( stderr, "%s: Don't know line: %s.\n", program_name, buffer );
        return( False );
      }

      stand_entry_ref -> every_n = (UINT16) integer1;

      if( buffer1[ 0 ] ==  'D' )
        stand_entry_ref -> valid_every = XTM_DB_VALID_DAY;
      else if( buffer1[ 0 ] ==  'W' )
        stand_entry_ref -> valid_every = XTM_DB_VALID_WEEK;
      else if( buffer1[ 0 ] ==  'M' )
        stand_entry_ref -> valid_every = XTM_DB_VALID_MONTH;
      else if( buffer1[ 0 ] ==  'L' )
        stand_entry_ref -> valid_every = XTM_DB_VALID_MONTH_LAST;
      else if( buffer1[ 0 ] ==  'Y' )
        stand_entry_ref -> valid_every = XTM_DB_VALID_YEAR;
      else {
        fprintf( stderr, "%s: Don't know line: %s.\n", program_name, buffer );
        return( False );
      }
    } /* if */


    /* Standing entry valid flags. */
    if( strncmp( buffer, "Sd", 2 ) == 0 ) {

      int  valid[ 7 ];

      items = sscanf( &buffer[ 2 ], "%d %d %d %d %d %d %d",
                      &valid[ 0 ], &valid[ 1 ],
                      &valid[ 2 ], &valid[ 3 ],
                      &valid[ 4 ], &valid[ 5 ],
                      &valid[ 6 ] );
      if( items != 7 ) {
        fprintf( stderr, "%s: Don't know line: %s.\n", program_name, buffer );
        return( False );
      }

      stand_entry_ref -> valid_days[ 0 ] = (UINT8) valid[ 0 ];
      stand_entry_ref -> valid_days[ 1 ] = (UINT8) valid[ 1 ];
      stand_entry_ref -> valid_days[ 2 ] = (UINT8) valid[ 2 ];
      stand_entry_ref -> valid_days[ 3 ] = (UINT8) valid[ 3 ];
      stand_entry_ref -> valid_days[ 4 ] = (UINT8) valid[ 4 ];
      stand_entry_ref -> valid_days[ 5 ] = (UINT8) valid[ 5 ];
      stand_entry_ref -> valid_days[ 6 ] = (UINT8) valid[ 6 ];

    } /* if */


    /* Skip weeks for standing entries? */
    if( strncmp( buffer, "Sw", 2 ) == 0 ) {

      int     char_read;
      int     index;
      UINT32  skip_flag;
      char    *char_ref;

      char_ref = &buffer[ 3 ];

      do {
        while( isspace( *char_ref ) )
          char_ref++;

        if( *char_ref == '\0' )
          break;

        char_read = strlen( char_ref );
        sscanf( char_ref, "%d%n", &index, &char_read );
        char_ref = char_ref + char_read;

        skip_flag = (1 << (index % 30));
        if( index > 30 )
          flagSet( stand_entry_ref -> skip_week[ 0 ], skip_flag );
        else
          flagSet( stand_entry_ref -> skip_week[ 1 ], skip_flag );

      } while( True );

    } /* if */


    /* Action on workday. */
    if( strncmp( buffer, "Sy", 2 ) == 0 ) {
      items = sscanf( &buffer[ 2 ], "%d %s", &integer1, buffer1 );
      if( items != 2 ) {
        fprintf( stderr, "%s: Don't know line: %s.\n", program_name, buffer );
        return( False );
      }

      if( buffer1[ 0 ] == 's' )
        flagSet( stand_entry_ref -> flags, XTM_DB_FLAG_SE_NWDAY_SKIP );
      else if( buffer1[ 0 ] == 'n' )
        flagSet( stand_entry_ref -> flags, XTM_DB_FLAG_SE_NWDAY_NEXT );
      else if( buffer1[ 0 ] == 'p' )
        flagSet( stand_entry_ref -> flags, XTM_DB_FLAG_SE_NWDAY_PREV );

    } /* if */


    /* Line with alarm deltas? */
    if( strncmp( buffer, "Al", 2 ) == 0 ) {

      int  offset[ 5 ];
      int  valid[ 5 ];

      items = sscanf( &buffer[ 2 ], "%d %d %d %d %d %d %d %d %d %d",
                      &valid[ 0 ],  &offset[ 0 ], 
                      &valid[ 1 ],  &offset[ 1 ],
                      &valid[ 2 ],  &offset[ 2 ], 
                      &valid[ 3 ],  &offset[ 3 ],
                      &valid[ 4 ],  &offset[ 4 ] );

       entry_ref -> alarm_offset[ 0 ] = (UINT16) offset[ 0 ];
       entry_ref -> alarm_offset[ 1 ] = (UINT16) offset[ 1 ];
       entry_ref -> alarm_offset[ 2 ] = (UINT16) offset[ 2 ];
       entry_ref -> alarm_offset[ 3 ] = (UINT16) offset[ 3 ];
       entry_ref -> alarm_offset[ 4 ] = (UINT16) offset[ 4 ];

       entry_ref -> alarm_valid[ 0 ] = (UINT8) valid[ 0 ];
       entry_ref -> alarm_valid[ 1 ] = (UINT8) valid[ 1 ];
       entry_ref -> alarm_valid[ 2 ] = (UINT8) valid[ 2 ];
       entry_ref -> alarm_valid[ 3 ] = (UINT8) valid[ 3 ];
       entry_ref -> alarm_valid[ 4 ] = (UINT8) valid[ 4 ];

    } /* if */


    /* Background and foreground colors. */
    if( strncmp( buffer, "Cl", 2 ) == 0 ) {

      items = sscanf( &buffer[ 2 ], "%d %d", &integer1, &integer2 );

      entry_ref -> bg_color_index = (UINT8) integer1;
      entry_ref -> fg_color_index = (UINT8) integer2;

    } /* if */


    /* Alarm tune to use. */
    if( strncmp( buffer, "Tu", 2 ) == 0 ) {

      items = sscanf( &buffer[ 2 ], "%d", &integer1 );
      if( items != 1 ) {
        fprintf( stderr, "%s: Don't know line: %s.\n", program_name, buffer );
        return( False );
      }

      if( integer1 < 1 || integer1 > 5 ) {
        fprintf( stderr, "%s: Invalid alarm tune: %d.\n" ,
                 program_name, integer1 );
        return( False );
      }

      entry_ref -> alarm_melody = (UINT8) integer1;

    } /* if */


    /* Number of lines in the day view and day list? */
    if( strncmp( buffer, "Li", 2 ) == 0 ) {

      items = sscanf( &buffer[ 2 ], "%d %d", &integer1, &integer2 );
      if( items != 2 ) {
        fprintf( stderr, "%s: Don't know line: %s.\n", program_name, buffer );
        return( False );
      }

      if( integer1 < 0 || integer2 < 0 ) {
        fprintf( stderr, "%s: Invalid line count: %d %d.\n",
                 program_name, integer1, integer2 );
        return( False );
      }

      entry_ref -> day_list_lines = (UINT8) integer1;
      entry_ref -> alarm_lines    = (UINT8) integer2;

    } /* if */


    /* Entry tag. */
    if( strncmp( buffer, "Tg", 2 ) == 0 ) {

      items = sscanf( &buffer[ 2 ], "%s", buffer1 );
      if( items == 1 ) {
        strncpy( entry_ref -> tag, buffer1, XTM_DB_TAG_LEN );
        entry_ref -> tag[ XTM_DB_TAG_LEN ] = '\0';
      }

    } /* if */


    /* Text line? */
    if( strncmp( buffer, "<<", 2 ) == 0 ) {

      /* Do we need to expand the buffer? */
      if( strlen( *entry_text ) + strlen( buffer ) + 2 > text_buffer_size ) {

        int  extend_size;

        extend_size = max( strlen( buffer ), segment_size ) + 10;
        text_buffer_size = text_buffer_size + extend_size;

        *entry_text = SysRealloc( *entry_text, text_buffer_size );

      } /* if */

      strcat( *entry_text, &buffer[ 2 ] );

    } /* if */

  } while( True );


} /* fetchEntry */


/*----------------------------------------------------------------------*/

static int
  noCaseStrcmp( char  *buffer1,
                char  *buffer2 )
{

  /* Variables. */
  char  *char_ref1;
  char  *char_ref2;


  /* Code. */

  if( strlen( buffer1 ) != strlen( buffer2 ) )
    return( strcmp( buffer1, buffer2 ) );

  char_ref1 = buffer1;
  char_ref2 = buffer2;

  while( *char_ref1 != '\0' ) {
    if( tolower( *char_ref1 ) < tolower( *char_ref2 ) )
      return( -1 );
    else if( tolower( *char_ref1 ) > tolower( *char_ref2 ) )
      return( 1 );

    char_ref1++;
    char_ref2++;
  }

  return( 0 );

} /* noCaseStrcmp */


/*----------------------------------------------------------------------*/

static XTM_DB_ENTRY_DATABASES
  *openDatabase( char  *directory )
{

  /* Variables. */
  XTM_DB_ENTRY_DATABASES  *database_ref;
  XTM_DB_OPEN_REQUEST     open_request;
  XTM_DB_STATUS           db_status;


  /* Code. */

  /* Open the entry database. */
  database_ref = SysNew( XTM_DB_ENTRY_DATABASES );

  open_request.name       = "";
  open_request.directory  = directory;
  open_request.operations = XTM_DB_FLAG_MODE_WRITE;
  open_request.database   = XTM_DB_ALL_ENTRY_DB;

  db_status = xtmDbOpenEntryDb( &open_request, database_ref );
  if( db_status != XTM_DB_OK ) {
    SysFree( database_ref );
    return( NULL );
  }

  return( database_ref );

} /* openDatabase */


/*----------------------------------------------------------------------*/

static Boolean
  saveEntry( char                    *directory,
             XTM_DB_ENTRY_DATABASES  *database_ref,
             UINT32                  operations,
             XTM_DB_ALL_ENTRY_REF    all_entry_ref,
             char                    *entry_text )
{

  /* Variables. */
  UINT32             id;
  XTM_DB_ENTRY_REF   entry_ref = &all_entry_ref -> entry;
  XTM_DB_ID_REQUEST  id_request;
  XTM_DB_STATUS      db_status;


  /* Code. */

  /* Only uers who are privileged can save private entries. */
  if(   flagIsSet( entry_ref -> flags, XTM_DB_FLAG_PRIVATE ) &&
      ! flagIsSet( operations, XTM_DB_FLAG_MODE_PRIV ) ) {
    if( log ) {
      fprintf( stderr, "%s: You are not allowed to save private entries.\n",
               program_name );
      fprintf( stderr, "The private flag for the entry is cleared.\n" );
    }

    flagClear( entry_ref -> flags, XTM_DB_FLAG_PRIVATE );
  }

  /* Get a new identifier. */
  id_request.directory  = directory;
  id_request.operations = 0;
  id_request.lock_file  = False;

  db_status = xtmDbGenerateId( &id_request, &id );
  if( db_status != XTM_DB_OK ) {
    fprintf( stderr, "%s: Cannot generate appointment ID.\n", 
             program_name );
    return( False );
  }

  /* Save the identifier. */
  all_entry_ref -> entry.id       = (UINT32) id;
  all_entry_ref -> stand_entry.id = (UINT32) id;

  /* Save the entry in the database. */
  db_status = xtmDbInsertEntry( database_ref,
                                all_entry_ref, entry_text );

  if( db_status != XTM_DB_OK ) {
    fprintf( stderr, "%s: Cannot insert appointment into database.\n",
             program_name );
    return( False );
  }

  return( True );

} /* saveEntry */
