/*----------------------------------------------------------------------------
--
--  Module:           TimDate
--
--  Project:          Tools - General C objects.
--  System:           Tim - Time and date manipulation.
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Interface file for module TimDate.
--
--  Filename:         TimDate.h
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1990-11-30
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
/* SCCSID = @(#) Module: TimDate.h, Version: 1.1, Date: 95/02/18 14:32:32 */


/* Only include this module once. */
#ifndef define_TimDate_h
#  define define_TimDate_h


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <time.h>


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#define TIM_NOW  -1


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

typedef time_t  TIM_TIME_REF;

typedef enum {
  TIM_OK,
  TIM_ERROR,
  TIM_YES,
  TIM_NO
} TIM_STATUS_TYPE;

typedef enum {
  TIM_KEEP,
  TIM_NOON,
  TIM_MIDNIGHT
} TIM_DAY_TYPE;

typedef enum {
  TIM_START_MONTH,
  TIM_END_MONTH
} TIM_MONTH_TYPE;

typedef struct {
  int  weeks;
  int  days;
  int  hours;
  int  minutes;
  int  seconds;
} TIM_DELTA_TYPE;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

void
  TimAddDays( TIM_TIME_REF *time, int days );

void
  TimAddHours( TIM_TIME_REF *time, int hours );

void
  TimAddMinutes( TIM_TIME_REF *time, int minutes );

void 
  TimAddMonths( TIM_TIME_REF *time, int months );

void 
  TimAddSeconds( TIM_TIME_REF *time, int seconds );

TIM_TIME_REF
  TimAddTime( TIM_TIME_REF  time,
              TIM_TIME_REF  to_time );

void
  TimAddYears( TIM_TIME_REF *time, int years );

int
  TimDaysInMonth( TIM_TIME_REF  time );

int
  TimDaysInYear( TIM_TIME_REF  time );

TIM_STATUS_TYPE
  TimDelta( TIM_TIME_REF    time1,
            TIM_TIME_REF    time2,
            TIM_DELTA_TYPE  *delta );

void
  TimFormatDate( TIM_TIME_REF  time,
                 char          *buffer,
                 int           buffer_size );

void
  TimFormatIsoDate( TIM_TIME_REF  time,
                    char          *buffer,
                    int           buffer_size );

void
  TimFormatFullDate( TIM_TIME_REF  time,
                     char          *buffer,
                     int           buffer_size );

void
  TimFormatFullIsoDate( TIM_TIME_REF  time,
                        char          *buffer,
                        int           buffer_size );

void
  TimFormatTime( TIM_TIME_REF  time,
                 char          *buffer,
                 int           buffer_size );

void
  TimFormatIsoTime( TIM_TIME_REF  time,
                    char          *buffer,
                    int           buffer_size );

void
  TimFormatStrTime( TIM_TIME_REF  time,
                    char          *format,
                    char          *buffer,
                    int           buffer_size );

int
  TimHour( TIM_TIME_REF  time );

int 
  TimIndexOfDay( TIM_TIME_REF  time );

int 
  TimIndexOfDayInIsoWeek( TIM_TIME_REF  time );

int 
  TimIndexOfDayInWeek( TIM_TIME_REF  time );

int
  TimIndexOfDayInYear( TIM_TIME_REF time );

int 
  TimIndexOfFirstDayInWeek();

int 
  TimIndexOfMonth( TIM_TIME_REF  time );

int
  TimIndexOfIsoWeek( TIM_TIME_REF  time );

int 
  TimIndexOfWeek( TIM_TIME_REF  time );

int 
  TimIndexOfYear( TIM_TIME_REF  time );

TIM_STATUS_TYPE
  TimInitializeFormat( char  *date_format_str,
                       char  *time_format_str );

TIM_STATUS_TYPE
  TimIsLeapYear( TIM_TIME_REF time );

TIM_STATUS_TYPE
  TimIsSameDate( TIM_TIME_REF time1,
                 TIM_TIME_REF time2 );

TIM_TIME_REF
  TimLocalTime( TIM_TIME_REF  time );

TIM_TIME_REF
  TimMakeTime( int  year,
               int  month,
               int  day,
               int  hour,
               int  minute,
               int  second );

TIM_STATUS_TYPE
  TimMakeDateFromString( TIM_TIME_REF  *time,
                         char          *string );

TIM_STATUS_TYPE
  TimMakeDateFromIsoString( TIM_TIME_REF  *time,
                            char          *string );

TIM_STATUS_TYPE
  TimMakeTimeFromString( TIM_TIME_REF  *time,
                         char          *string );

TIM_STATUS_TYPE
  TimMakeTimeFromIsoString( TIM_TIME_REF  *time,
                            char          *string );

TIM_TIME_REF
  TimMakeTimeNow();

TIM_STATUS_TYPE  
  TimMakeTimeFromWeek( int           year,
                       int           week_number,
                       TIM_TIME_REF  *time );

int
  TimMinute( TIM_TIME_REF  time );

void
  TimNextDay( TIM_TIME_REF  *time, 
              TIM_DAY_TYPE  time_of_day );

void
  TimNextMonth( TIM_TIME_REF    *time, 
                TIM_MONTH_TYPE  day_in_month );

void
  TimPreviousDay( TIM_TIME_REF  *time, 
                  TIM_DAY_TYPE  time_of_day );

void
  TimPreviousMonth( TIM_TIME_REF    *time, 
                    TIM_MONTH_TYPE  day_in_month );

int
  TimSecond( TIM_TIME_REF  time );

void
  TimSetUnixWeekNo( Boolean  use_unix_week_no );

TIM_STATUS_TYPE  
  TimTimeInSecondsRange( TIM_TIME_REF  master,
                         TIM_TIME_REF  check,
                         int           seconds_range );

char
  *TimWhatDateFormat();

char
  *TimWhatTimeFormat();

#endif
