/*----------------------------------------------------------------------------
--
--  Module:           TimDate
--
--  Project:          Tools - General C objects.
--  System:           Tim - Time and date manipulation.
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Manage time and date.
--
--    Date format:
--      Order:            YMD, DMY or MDY
--      Separator:        One character, 0 is no separator
--      Day leading 0:    0 is no, 1 is yes
--      Month leading 0:  0 is no, 1 is yes
--      Century:          0 is no, 1 is yes
--      1st weekday:      0 is Sunday, 1 is Monday
--
--    Time format:
--      12 or 24 hour:    12 or 24
--      12 hour suffix:   Two characters, two blanks is no suffix
--      24 hour suffix:   Two characters, two blanks is no suffix
--      Separator:        One character, 0 is no separator
--      Hour leading 0:   0 is no, 1 is yes
--
--  Filename:         TimDate.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1990-11-30
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: TimDate.c, Version: 1.1, Date: 95/02/18 14:32:31";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <memory.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>

#include "System.h"
#include "TimDate.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Order to use for year, month and day. */
typedef enum {
  DATE_MMDDYY, DATE_DDMMYY, DATE_YYMMDD
} DATE_ORDER;

/* Format used for dates. */
typedef struct {
  DATE_ORDER  order;
  char        separator[ 2 ];
  Boolean     day_leading_zero;
  Boolean     month_leading_zero;
  Boolean     include_century;
  char        ymd_format[ 50 ];
  char        md_format[ 50 ];
  char        d_format[ 50 ];
  int         week_1st_day;
} DATE_FORMAT;

/* Format used for times. */
typedef struct {
  Boolean  hour_12;
  char     hour_12_suffix[ 5 ];
  char     hour_12_suffix_separator[ 2 ];
  char     hour_24_suffix[ 5 ];
  char     hour_24_suffix_separator[ 2 ];
  char     separator[ 2 ];
  Boolean  hour_leading_zero;
  char     hm_format[ 50 ];
  char     m_format[ 50 ];
} TIME_FORMAT;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Date and time format. */
static char  date_format_string[ 50 ] = { 
  'Y', 'M', 'D', '-', '1', '1', '1', '1', '\0' };
static char  time_format_string[ 50 ] = { 
  '2', '4', ' ', ' ', ' ', ' ', ':', '0', '\0' };

static DATE_FORMAT  user_date_format = {
  DATE_YYMMDD, { '-', '\0' }, True, True, True,
  { '%','d','%','*','[','^','0','-','9',']','%','d','%','*','[','^','0','-',
    '9',']','%','d','\0' },
  { '%','d','%','*','[','^','0','-','9',']','%','d','\0' },
  { '%','d','\0' },
  1
};

static DATE_FORMAT  iso_date_format = {
  DATE_YYMMDD, { '-', '\0' }, True, True, True,
  { '%','d','%','*','[','^','0','-','9',']','%','d','%','*','[','^','0','-',
    '9',']','%','d','\0' },
  { '%','d','%','*','[','^','0','-','9',']','%','d','\0' },
  { '%','d','\0' },
  1
};

static TIME_FORMAT  user_time_format = {
  False,
  { '\0' }, { ' ', '\0' },
  { '\0' }, { ' ', '\0' },
  { ':', '\0' }, False,
  { '%','d','%','*','[','^','0','-','9',']','%','d', '\0' },
  { '%','d','\0' }
};

static TIME_FORMAT  iso_time_format = {
  False,
  { '\0' }, { ' ', '\0' },
  { '\0' }, { ' ', '\0' },
  { ':', '\0' }, False,
  { '%','d','%','*','[','^','0','-','9',']','%','d', '\0' },
  { '%','d','\0' }
};

/* Number of days in the month. */
static int days_in_month_leap_year[] = {
  31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

static int days_in_month[] = {
  31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

/* Offset of the days within the year (0 - 365). */
static int day_offset[] = {
  0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 };

static int day_offset_leap_year[] = {
  0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335 };

/* Week number offsets 1970-2070 (not nice but it works). */
static UINT8 mon_week_offsets[] = {
   0, 1, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1,
   0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0,
   1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 0,
   0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 0,
   0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1,
   1,
};

static UINT8 sun_week_offsets[] = {
   1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 0,
   0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 1,
   1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 0,
   0, 0, 1, 1, 0, 0, 0, 1, 1, 0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 0,
   0, 0, 1, 1, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1,
   1,
};

/* Use the standard UNIX week numbers. */
static  unix_week_no = False;


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static struct tm
  *convertToLocalStruct( TIM_TIME_REF  time,
                         struct tm     *container );

static struct tm
  *convertToStruct( TIM_TIME_REF  time,
                    struct tm     *container );

static int
  dayNumberInYear( int year,
                   int month,
                   int day );

static int
  daysInYear( int  year );

static void
  formatDate( DATE_FORMAT   *date_format,
              TIM_TIME_REF  time,
              char          *buffer,
              int           buffer_size );

static void
  formatFullDate( DATE_FORMAT   *date_format,
                  TIM_TIME_REF  time,
                  char          *buffer,
                  int           buffer_size );

static void
  formatTime( TIME_FORMAT   *time_format,
              TIM_TIME_REF  time,
              char          *buffer,
              int           buffer_size );

static TIM_STATUS_TYPE  
  isTimeOk( int  year,
            int  month,
            int  day,
            int  hour,
            int  minute,
            int  second );

static TIM_STATUS_TYPE
  makeDateFromString( DATE_FORMAT   *date_format,
                      TIM_TIME_REF  *time,
                      char          *string );

static TIM_STATUS_TYPE
  makeTimeFromString( TIME_FORMAT   *time_format,
                      TIM_TIME_REF  *time,
                      char          *string );

static int
  noCaseStrcmp( char  *buffer1,
                char  *buffer2 );

static time_t
  secondsSince1970( struct tm *time_ref );


/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void 
  TimAddDays( TIM_TIME_REF *time, int days )
{

  /* Code. */

  *time = *time + (days * 24 * 60 * 60);

} /* TimAddDays */


/*----------------------------------------------------------------------*/

void 
  TimAddHours( TIM_TIME_REF *time, int hours )
{

  /* Code. */

  *time = *time + (hours * 60 * 60);


  return;

} /* TimAddHours */


/*----------------------------------------------------------------------*/

void 
  TimAddMinutes( TIM_TIME_REF *time, int minutes )
{

  /* Code. */

  *time = *time + (minutes * 60);


  return;

} /* TimAddMinutes */


/*----------------------------------------------------------------------*/

void 
  TimAddMonths( TIM_TIME_REF *time, int months )
{

  /* Variables. */
  int        add_months;
  int        add_year;
  time_t     new_time;
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToStruct( *time, &time_struct_obj );

  add_year   = abs( months ) / 12;
  add_months = abs( months ) % 12;

  if( months < 0 ) {
    time_struct -> tm_year = time_struct -> tm_year - add_year;
    time_struct -> tm_mon  = time_struct -> tm_mon  - add_months;
  } else {
    time_struct -> tm_year = time_struct -> tm_year + add_year;
    time_struct -> tm_mon  = time_struct -> tm_mon  + add_months;
  }

  if( time_struct -> tm_mon > 11 ) {
    time_struct -> tm_year = time_struct -> tm_year + 1;
    time_struct -> tm_mon  = time_struct -> tm_mon  - 12;
  }    

  if( time_struct -> tm_mon < 0 ) {
    time_struct -> tm_year = time_struct -> tm_year - 1;
    time_struct -> tm_mon  = time_struct -> tm_mon  + 12;
  }    

  time_struct -> tm_yday  = dayNumberInYear( 
                              time_struct -> tm_year,
                              time_struct -> tm_mon + 1,
                              time_struct -> tm_mday ) - 1;

  new_time = (TIM_TIME_REF) secondsSince1970( time_struct );

  *time = new_time;


  return;

} /* TimAddMonths */


/*----------------------------------------------------------------------*/

void 
  TimAddSeconds( TIM_TIME_REF *time, int seconds )
{

  /* Code. */

  *time = *time + seconds;


  return;

} /* TimAddSeconds */


/*----------------------------------------------------------------------*/

TIM_TIME_REF
  TimAddTime( TIM_TIME_REF  time,
              TIM_TIME_REF  to_time )
{

  /* Code. */

  return( time + to_time );

} /* TimAdd */


/*----------------------------------------------------------------------*/

void 
  TimAddYears( TIM_TIME_REF *time, int years )
{

  /* Variables. */
  time_t     new_time;
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToStruct( *time, &time_struct_obj );

  time_struct -> tm_year = time_struct -> tm_year + years;
  time_struct -> tm_yday  = dayNumberInYear( 
                              time_struct -> tm_year,
                              time_struct -> tm_mon + 1,
                              time_struct -> tm_mday ) - 1;

  new_time = (TIM_TIME_REF) secondsSince1970( time_struct );

  if( new_time >= 0 )
    *time = new_time;


  return;

} /* TimAddYears */


/*----------------------------------------------------------------------*/

int 
  TimDaysInMonth( TIM_TIME_REF  time )
{

  /* Variables. */
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToStruct( time, &time_struct_obj );

  if( TimIsLeapYear( time ) == TIM_YES )
    return( days_in_month_leap_year[ time_struct -> tm_mon ] );


  return( days_in_month[ time_struct -> tm_mon ] );

} /* TimDaysInMonth */


/*----------------------------------------------------------------------*/

int 
  TimDaysInYear( TIM_TIME_REF  time )
{

  /* Code. */

  if( TimIsLeapYear( time ) == TIM_YES )
    return( 366 );


  return( 365 );

} /* TimDaysInYear */


/*----------------------------------------------------------------------*/

TIM_STATUS_TYPE
  TimDelta( TIM_TIME_REF    time1,
            TIM_TIME_REF    time2,
            TIM_DELTA_TYPE  *delta )
{

  /* Variables. */
  TIM_TIME_REF  temp_time;


  /* Code. */

  if( time1 > time2 ) {
    temp_time = time1;
    time1     = time2;
    time2     = temp_time;
  }

  delta -> seconds = ((int) time2) - ((int) time1);  
  delta -> minutes = delta -> seconds / 60;
  delta -> hours   = delta -> seconds / (60 * 60);
  delta -> days    = delta -> seconds / (60 * 60 * 24);
  delta -> weeks   = delta -> seconds / (60 * 60 * 24 * 7);


  return( TIM_OK );

} /* TimDelta */


/*----------------------------------------------------------------------*/

void
  TimFormatDate( TIM_TIME_REF  time,
                 char          *buffer,
                 int           buffer_size )
{

  /* Code. */

  formatDate( &user_date_format, time, buffer, buffer_size );


  return;

} /* TimFormatDate */


/*----------------------------------------------------------------------*/

void
  TimFormatIsoDate( TIM_TIME_REF  time,
                    char          *buffer,
                    int           buffer_size )
{

  /* Code. */

  formatDate( &iso_date_format, time, buffer, buffer_size );


  return;

} /* TimFormatIsoDate */


/*----------------------------------------------------------------------*/

void
  TimFormatFullDate( TIM_TIME_REF  time,
                     char          *buffer,
                     int           buffer_size )
{

  /* Code. */

  formatFullDate( &user_date_format, time, buffer, buffer_size );


  return;

} /* TimFormatFullDate */


/*----------------------------------------------------------------------*/

void
  TimFormatFullIsoDate( TIM_TIME_REF  time,
                        char          *buffer,
                        int           buffer_size )
{

  /* Code. */

  formatFullDate( &iso_date_format, time, buffer, buffer_size );


  return;

} /* TimFormatFullIsoDate */


/*----------------------------------------------------------------------*/

void
  TimFormatStrTime( TIM_TIME_REF  time,
                    char          *format,
                    char          *buffer,
                    int           buffer_size )
{

  /* Variables. */
  int        size;
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToStruct( time, &time_struct_obj );

  size = strftime( buffer, buffer_size, format, time_struct );

  if( size == 0 )
    *buffer = '\0';


  return;

} /* TimFormatStrTime */


/*----------------------------------------------------------------------*/

void
  TimFormatTime( TIM_TIME_REF  time,
                 char          *buffer,
                 int           buffer_size )
{

  /* Code. */

  formatTime( &user_time_format, time, buffer, buffer_size );


  return;

} /* TimFormatTime */


/*----------------------------------------------------------------------*/

void
  TimFormatIsoTime( TIM_TIME_REF  time,
                    char          *buffer,
                    int           buffer_size )
{

  /* Code. */

  formatTime( &iso_time_format, time, buffer, buffer_size );


  return;

} /* TimFormatIsoTime */


/*----------------------------------------------------------------------*/

int 
  TimHour( TIM_TIME_REF  time )
{

  /* Variables. */
  struct tm *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToStruct( time, &time_struct_obj );


  return( time_struct -> tm_hour );

} /* TimHour */


/*----------------------------------------------------------------------*/

int 
  TimIndexOfDay( TIM_TIME_REF  time )
{

  /* Variables. */
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToStruct( time, &time_struct_obj );


  return( time_struct -> tm_mday );

} /* TimIndexOfDay */


/*----------------------------------------------------------------------*/

int 
  TimIndexOfDayInIsoWeek( TIM_TIME_REF  time )
{

  /* Variables. */
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToStruct( time, &time_struct_obj );

  if( time_struct -> tm_wday == 0 )
    return( time_struct -> tm_wday + 7 );


  return( time_struct -> tm_wday );

} /* TimIndexOfDayInIsoWeek */


/*----------------------------------------------------------------------*/

int 
  TimIndexOfDayInWeek( TIM_TIME_REF  time )
{

  /* Variables. */
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToStruct( time, &time_struct_obj );

  switch( user_date_format.week_1st_day ) {

    case 0:
      return( time_struct -> tm_wday + 1 );

    case 1:
    default:
      if( time_struct -> tm_wday == 0 )
        return( time_struct -> tm_wday + 7 );
      else
        return( time_struct -> tm_wday );

    } /* switch */

} /* TimIndexOfDayInWeek */


/*----------------------------------------------------------------------*/

int 
  TimIndexOfFirstDayInWeek()
{

  /* Code. */


  return( user_date_format.week_1st_day );

} /* TimIndexOfFirstDayInWeek */


/*----------------------------------------------------------------------*/

int 
  TimIndexOfDayInYear( TIM_TIME_REF time )
{

  /* Variables. */
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToStruct( time, &time_struct_obj );


  return( time_struct -> tm_yday + 1 );

} /* TimIndexOfDayInYear */


/*----------------------------------------------------------------------*/

int 
  TimIndexOfMonth( TIM_TIME_REF  time )
{

  /* Variables. */
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToStruct( time, &time_struct_obj );


  return( time_struct -> tm_mon + 1 );

} /* TimIndexOfMonth */


/*----------------------------------------------------------------------*/

int
  TimIndexOfIsoWeek( TIM_TIME_REF  time )
{

  /* Variables. */
  int        week_no;
  int        year;
  char       buffer[ 10 ];
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToStruct( time, &time_struct_obj );

  strftime( buffer, sizeof( buffer ), "%W", time_struct );

  week_no = atoi( buffer ) + 1;
  year    = time_struct -> tm_year + 1900;

  if( week_no > 1 && year >= 1970 && year <= 2070 && ! unix_week_no )
    week_no = week_no - (int) mon_week_offsets[ year - 1970 ];


  return( week_no );

} /* TimIndexOfIsoWeek */


/*----------------------------------------------------------------------*/

int
  TimIndexOfWeek( TIM_TIME_REF  time )
{

  /* Variables. */
  int        week_no;
  int        year;
  char       buffer[ 10 ];
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToStruct( time, &time_struct_obj );

  switch( user_date_format.week_1st_day ) {
    case 0:
      strftime( buffer, sizeof( buffer ), "%U", time_struct );
      break;

    case 1:
    default:
      strftime( buffer, sizeof( buffer ), "%W", time_struct );
      break;
    }

  week_no = atoi( buffer ) + 1;
  year    = time_struct -> tm_year + 1900;

  switch( user_date_format.week_1st_day ) {
    case 0:
      if( week_no > 1 && year >= 1970 && year <= 2070 && ! unix_week_no )
        week_no = week_no - (int) sun_week_offsets[ year - 1970 ];
      break;

    case 1:
    default:
      if( week_no > 1 && year >= 1970 && year <= 2070 && ! unix_week_no )
        week_no = week_no - (int) mon_week_offsets[ year - 1970 ];
      break;
  }


  return( week_no );

} /* TimIndexOfWeek */


/*----------------------------------------------------------------------*/

int 
  TimIndexOfYear( TIM_TIME_REF  time )
{

  /* Variables. */
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToStruct( time, &time_struct_obj );


  return( time_struct -> tm_year + 1900 );

} /* TimIndexOfYear */


/*----------------------------------------------------------------------*/

TIM_STATUS_TYPE
  TimInitializeFormat( char  *date_format_str,
                       char  *time_format_str )
{

  /* Variables. */
  char         day_format[ 50 ];
  char         hour_format[ 50 ];
  char         minute_format[ 50 ];
  char         month_format[ 50 ];
  char         sep_format[ 50 ];
  char         suffix_format[ 50 ];
  char         year_format[ 50 ];
  DATE_FORMAT  new_date_format;
  TIME_FORMAT  new_time_format;


  /* Code. */

  /* Correct length of the format strings? */
  if( strlen( date_format_str ) != 8 )
    return( TIM_ERROR );

  if( strlen( time_format_str ) != 8 )
    return( TIM_ERROR );


  /* Date format. */

  /* Date order. */
  if( strncmp( &date_format_str[ 0 ], "MDY", 3 ) == 0 )
    new_date_format.order = DATE_MMDDYY;
  else if( strncmp( &date_format_str[ 0 ], "DMY", 3 ) == 0 )
    new_date_format.order = DATE_DDMMYY;
  else if( strncmp( &date_format_str[ 0 ], "YMD", 3 ) == 0 )
    new_date_format.order = DATE_YYMMDD;
  else
    return( TIM_ERROR );

  /* Separator. */
  if( date_format_str[ 3 ] == '0' )
    strcpy( new_date_format.separator, "" );
  else if( ! isdigit( date_format_str[ 3 ] ) ) {
    new_date_format.separator[ 0 ] = date_format_str[ 3 ];
    new_date_format.separator[ 1 ] = '\0';
  } else
    return( TIM_ERROR );

  /* Day leading zero. */
  if( date_format_str[ 4 ] == '1' )
    new_date_format.day_leading_zero = True;
  else
    new_date_format.day_leading_zero = False;

  /* Month leading zero. */
  if( date_format_str[ 5 ] == '1' )
    new_date_format.month_leading_zero = True;
  else
    new_date_format.month_leading_zero = False;

  /* Include century. */
  if( date_format_str[ 6 ] == '1' )
    new_date_format.include_century = True;
  else
    new_date_format.include_century = False;

  /* First day in week. */
  if( date_format_str[ 7 ] == '0' )
    new_date_format.week_1st_day = 0;
  else
    new_date_format.week_1st_day = 1;


  /* Time format. */

  /* 12/24 hour time. */
  if( strncmp( &time_format_str[ 0 ], "12", 2 ) == 0 )
    new_time_format.hour_12 = True;
  else if( strncmp( &time_format_str[ 0 ], "24", 2 ) == 0 )
    new_time_format.hour_12 = False;
  else
    return( TIM_ERROR );

  /* 12 hour suffix. */
  if( strncmp( &time_format_str[ 2 ], "  ", 2 ) == 0 ) {
    if( new_time_format.hour_12 ) {
      strncpy( new_time_format.hour_12_suffix, "am", 2 );
      new_time_format.hour_12_suffix[ 2 ] = '\0';
      strcpy( new_time_format.hour_12_suffix_separator, " " );
    } else {
      new_time_format.hour_12_suffix[ 0 ] = '\0';
      strcpy( new_time_format.hour_12_suffix_separator, "" );
    }
  } else {
    strncpy( new_time_format.hour_12_suffix, &time_format_str[ 2 ], 2 );
    new_time_format.hour_12_suffix[ 2 ] = '\0';
    strcpy( new_time_format.hour_12_suffix_separator, " " );
  }

  /* 24 hour suffix. */
  if( strncmp( &time_format_str[ 4 ], "  ", 2 ) == 0 ) {
    if( new_time_format.hour_12 ) {
      strncpy( new_time_format.hour_24_suffix, "pm", 2 );
      new_time_format.hour_24_suffix[ 2 ] = '\0';
      strcpy( new_time_format.hour_24_suffix_separator, " " );
    } else {
      new_time_format.hour_24_suffix[ 0 ] = '\0';
      strcpy( new_time_format.hour_24_suffix_separator, "" );
    }
  } else {
    strncpy( new_time_format.hour_24_suffix, &time_format_str[ 4 ], 2 );
    new_time_format.hour_24_suffix[ 2 ] = '\0';
    strcpy( new_time_format.hour_24_suffix_separator, " " );
  }

  /* Separator. */
  if( time_format_str[ 6 ] == '0' ) {
    strcpy( new_time_format.separator, "" );
  } else if( ! isdigit( time_format_str[ 6 ] ) ) {
    new_time_format.separator[ 0 ] = time_format_str[ 6 ];
    new_time_format.separator[ 1 ] = '\0';
  } else
    return( TIM_ERROR );

  /* Hour leading zero. */
  if( time_format_str[ 7 ] == '1' )
    new_time_format.hour_leading_zero = True;
  else
    new_time_format.hour_leading_zero = False;


  /* Store the result. */
  memcpy( (void *) &user_date_format, (void *) &new_date_format,
           sizeof( DATE_FORMAT ) );

  memcpy( (void *) &user_time_format, (void *) &new_time_format,
           sizeof( TIME_FORMAT ) );


  /* Read dates. */

  /* Year format? */
  if( user_date_format.separator[ 0 ] == '\0' ) {
    if( user_date_format.include_century )
      strcpy( year_format, "%4d" );
    else
      strcpy( year_format, "%2d" );
  } else
    strcpy( year_format, "%d" );

  /* Month format? */
  if( user_date_format.separator[ 0 ] == '\0' ) {
    if( user_date_format.month_leading_zero )
      strcpy( month_format, "%2d" );
    else
      strcpy( month_format, "%1d" );
  } else
    strcpy( month_format, "%d" );

  /* Day format? */
  if( user_date_format.separator[ 0 ] == '\0' ) {
    if( user_date_format.day_leading_zero )
      strcpy( day_format, "%2d" );
    else
      strcpy( day_format, "%1d" );
  } else
    strcpy( day_format, "%d" );

  /* Separator format? */
  if( user_date_format.separator[ 0 ] != '\0' )
    strcpy( sep_format, "%*[^0-9]" );
  else
    sep_format[ 0 ] = '\0';

  /* MMDDYY format? */
  if( user_date_format.order == DATE_MMDDYY ) {
    sprintf( user_date_format.ymd_format, "%s%s%s%s%s",
             month_format, sep_format, day_format, sep_format, year_format );
    sprintf( user_date_format.md_format, "%s%s%s",
             month_format, sep_format, day_format );
    sprintf( user_date_format.d_format, "%s",
             day_format );
  }

  /* DDMMYY format? */
  if( user_date_format.order == DATE_DDMMYY ) {
    sprintf( user_date_format.ymd_format, "%s%s%s%s%s",
             day_format, sep_format, month_format, sep_format, year_format );
    sprintf( user_date_format.md_format,  "%s%s%s",
             day_format, sep_format, month_format );
    sprintf( user_date_format.d_format,   "%s",
             day_format );
  }

  /* YYMMDD format? */
  if( user_date_format.order == DATE_YYMMDD ) {
    sprintf( user_date_format.ymd_format, "%s%s%s%s%s",
             year_format, sep_format, month_format, sep_format, day_format );
    sprintf( user_date_format.md_format, "%s%s%s",
             month_format, sep_format, day_format );
    sprintf( user_date_format.d_format, "%s",
             day_format );
  }


  /* Read times. */

  /* Hour format? */
  if( user_time_format.separator[ 0 ] == '\0' ) {
    if( user_time_format.hour_leading_zero )
      strcpy( hour_format, "%2d" );
    else
      strcpy( hour_format, "%1d" );
  } else
    strcpy( hour_format, "%d" );

  /* Minute format? */
  strcpy( minute_format, "%2d" );

  /* Separator format? */
  if( user_time_format.separator[ 0 ] != '\0' )
    strcpy( sep_format, "%*[^0-9]" );
  else
    sep_format[ 0 ] = '\0';

  /* Suffix format. */
  if( user_time_format.hour_12 )
    strcpy( suffix_format, "%s" );
  else
    suffix_format[ 0 ] = '\0';

  /* Time formats. */
  sprintf( user_time_format.hm_format, "%s%s%s%s",
           hour_format, sep_format, minute_format, suffix_format );
  sprintf( user_time_format.m_format, "%s%s",
           minute_format, suffix_format );


  /* Save the date and time formats. */
  strcpy( date_format_string, date_format_str );
  strcpy( time_format_string, time_format_str );


  return( TIM_OK );

} /* TimInitializeFormat */


/*----------------------------------------------------------------------*/

TIM_STATUS_TYPE
  TimIsLeapYear( TIM_TIME_REF time )
{

  /* Variables. */
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToStruct( time, &time_struct_obj );

  if( daysInYear( time_struct -> tm_year ) == 366 )
    return( TIM_YES );


  return( TIM_NO );

} /* TimIsLeapYear */


/*----------------------------------------------------------------------*/

TIM_STATUS_TYPE
  TimIsSameDate( TIM_TIME_REF time1,
                 TIM_TIME_REF time2 )
{

  /* Code. */

  if( TimIndexOfYear(  time1 ) == TimIndexOfYear(  time2 ) &&
      TimIndexOfMonth( time1 ) == TimIndexOfMonth( time2 ) &&
      TimIndexOfDay(   time1 ) == TimIndexOfDay(   time2 ) )
    return( TIM_YES );


  return( TIM_NO );

} /* TimIsSameDate */


/*----------------------------------------------------------------------*/

TIM_TIME_REF
  TimLocalTime( TIM_TIME_REF  time )
{

  /* Variables. */
  time_t     new_time;
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToLocalStruct( time, &time_struct_obj );

  new_time = secondsSince1970( time_struct );


  return( (TIM_TIME_REF) new_time );  

} /* TimLocalTime */


/*----------------------------------------------------------------------*/

TIM_TIME_REF
  TimMakeTime( int  year,
               int  month,
               int  day,
               int  hour,
               int  minute,
               int  second )
{

  /* Variables. */
  time_t     new_time;
  time_t     now;
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  now = time( NULL );

  time_struct = convertToStruct( now, &time_struct_obj );

  if( year != TIM_NOW )
    time_struct -> tm_year = year - 1900;

  if( month != TIM_NOW )
    time_struct -> tm_mon = month - 1;

  if( day != TIM_NOW )
    time_struct -> tm_mday = day;

  if( hour != TIM_NOW )
    time_struct -> tm_hour = hour;

  if( minute != TIM_NOW )
    time_struct -> tm_min = minute;

  if( second != TIM_NOW )
    time_struct -> tm_sec = second;

  time_struct -> tm_yday  = dayNumberInYear( 
                              time_struct -> tm_year,
                              time_struct -> tm_mon + 1,
                              time_struct -> tm_mday ) - 1;

  new_time = (TIM_TIME_REF) secondsSince1970( time_struct );


  return( new_time );

} /* TimMakeTime */


/*----------------------------------------------------------------------*/

TIM_STATUS_TYPE
  TimMakeDateFromString( TIM_TIME_REF  *time,
                         char          *string )
{

  /* Variables. */
  TIM_STATUS_TYPE  status;


  /* Code. */

  status = makeDateFromString( &user_date_format, time, string );


  return( status );

} /* TimMakeDateFromString */


/*----------------------------------------------------------------------*/

TIM_STATUS_TYPE
  TimMakeDateFromIsoString( TIM_TIME_REF  *time,
                            char          *string )
{

  /* Variables. */
  TIM_STATUS_TYPE  status;


  /* Code. */

  status = makeDateFromString( &iso_date_format, time, string );


  return( status );

} /* TimMakeDateFromIsoString */


/*----------------------------------------------------------------------*/

TIM_STATUS_TYPE
  TimMakeTimeFromString( TIM_TIME_REF  *time,
                         char          *string )
{

  /* Variables. */
  TIM_STATUS_TYPE  status;


  /* Code. */

  status = makeTimeFromString( &user_time_format, time, string );


  return( status );

} /* TimMakeTimeFromString */


/*----------------------------------------------------------------------*/

TIM_STATUS_TYPE
  TimMakeTimeFromIsoString( TIM_TIME_REF  *time,
                            char          *string )
{

  /* Variables. */
  TIM_STATUS_TYPE  status;


  /* Code. */

  status = makeTimeFromString( &iso_time_format, time, string );


  return( status );

} /* TimMakeTimeFromIsoString */


/*----------------------------------------------------------------------*/

TIM_TIME_REF 
  TimMakeTimeNow()
{

  /* Code. */

  return( (TIM_TIME_REF) time( NULL ) );

} /* TimMakeTimeNow */


/*----------------------------------------------------------------------*/

TIM_STATUS_TYPE  
  TimMakeTimeFromWeek( int           year,
                       int           week_number,
                       TIM_TIME_REF  *time )
{

  /* Variables. */
  TIM_TIME_REF  new_time;


  /* Code. */

  new_time = TimMakeTime( year, 1, 1, 0, 0, 0 );

  TimAddDays( &new_time, (week_number - 2) * 7 );

  while( TimIndexOfWeek( new_time ) != week_number )
    TimAddDays( &new_time, 1 );

  *time = new_time;


  return( TIM_OK );

} /* TimMakeTimeFromWeek */


/*----------------------------------------------------------------------*/

int 
  TimMinute( TIM_TIME_REF  time )
{

  /* Variables. */
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToStruct( time, &time_struct_obj );


  return( time_struct -> tm_min );

} /* TimMinute */


/*----------------------------------------------------------------------*/

void 
  TimNextDay( TIM_TIME_REF  *time, 
              TIM_DAY_TYPE  time_of_day )
{

  /* Variables. */
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  TimAddDays( time, 1 );

  time_struct = convertToStruct( *time, &time_struct_obj );

  /* Set to a specific time of the day. */
  switch( time_of_day ) {
    case TIM_NOON:
      time_struct -> tm_hour = 12;
      time_struct -> tm_min  = 0;
      time_struct -> tm_sec  = 0;  

      *time = (TIM_TIME_REF) secondsSince1970( time_struct );
      break;

    case TIM_MIDNIGHT:
      time_struct -> tm_hour = 0;
      time_struct -> tm_min  = 0;
      time_struct -> tm_sec  = 0;  

      *time = (TIM_TIME_REF) secondsSince1970( time_struct );
      break;

    default:
      break;

  } /* switch */


  return;

} /* TimNextDay */


/*----------------------------------------------------------------------*/

void 
  TimNextMonth( TIM_TIME_REF    *time, 
                TIM_MONTH_TYPE  day_in_month )
{

  /* Variables. */
  TIM_TIME_REF  new_time;


  /* Code. */

  /* The end of this month. */
  new_time = TimMakeTime( TimIndexOfYear(  *time ), 
                          TimIndexOfMonth( *time ),
                          TimDaysInMonth(  *time ), 
                          0, 0, 0 );

  /* Add days so that we move to the next month. */
  TimAddDays( &new_time, 1 );

  /* Which part of the month? */
  switch( day_in_month ) {

    case TIM_START_MONTH:
      break;

    case TIM_END_MONTH:
      new_time = TimMakeTime( TimIndexOfYear(  new_time ),
                              TimIndexOfMonth( new_time ),
                              TimDaysInMonth(  new_time ),
                              0, 0, 0 );
      break;

  } /* switch */

  *time = new_time;


  return;

} /* TimNextMonth */


/*----------------------------------------------------------------------*/

void 
  TimPreviousDay( TIM_TIME_REF  *time, 
                  TIM_DAY_TYPE  time_of_day )
{

  /* Variables. */
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  TimAddDays( time, -1 );

  time_struct = convertToStruct( *time, &time_struct_obj );

  /* Set to a specific time of the day. */
  switch( time_of_day ) {
    case TIM_NOON:
      time_struct -> tm_hour = 12;
      time_struct -> tm_min  = 0;
      time_struct -> tm_sec  = 0;  

      *time = (TIM_TIME_REF) secondsSince1970( time_struct );
      break;

    case TIM_MIDNIGHT:
      time_struct -> tm_hour = 0;
      time_struct -> tm_min  = 0;
      time_struct -> tm_sec  = 0;  

      *time = (TIM_TIME_REF) secondsSince1970( time_struct );
      break;

    default:
      break;

  } /* switch */


  return;

} /* TimPreviousDay */


/*----------------------------------------------------------------------*/

void
  TimPreviousMonth( TIM_TIME_REF    *time, 
                    TIM_MONTH_TYPE  day_in_month )
{

  /* Variables. */
  TIM_TIME_REF  new_time;


  /* Code. */

  /* The first in this month. */
  new_time = TimMakeTime( TimIndexOfYear(  *time ), 
                          TimIndexOfMonth( *time ),
                          1, 0, 0, 0 );

  /* Add days so that we move to the end of next month. */
  TimAddDays( &new_time, -1 );

  /* Which part of the month? */
  switch( day_in_month ) {

    case TIM_START_MONTH:
      new_time = TimMakeTime( TimIndexOfYear(  new_time ),
                              TimIndexOfMonth( new_time ),
                              1, 0, 0, 0 );
      break;

    case TIM_END_MONTH:
      break;

  } /* switch */

  *time = new_time;


  return;

} /* TimPreviousMonth */


/*----------------------------------------------------------------------*/

int
  TimSecond( TIM_TIME_REF  time )
{

  /* Variables. */
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  time_struct = convertToStruct( time, &time_struct_obj );


  return( time_struct -> tm_sec );

} /* TimSecond */


/*----------------------------------------------------------------------*/

void
  TimSetUnixWeekNo( Boolean  use_unix_week_no )
{

  /* Code. */

  unix_week_no = use_unix_week_no;


  return;

} /* TimSetUnixWeekNo */


/*----------------------------------------------------------------------*/

TIM_STATUS_TYPE  
  TimTimeInSecondsRange( TIM_TIME_REF  master,
                         TIM_TIME_REF  check,
                         int           seconds_range )
{

  /* Code. */
  if( abs( master - check ) <= (seconds_range / 2) )
    return( TIM_YES );


  return( TIM_NO );

} /* TimTimeInSecondsRange */


/*----------------------------------------------------------------------*/

char
  *TimWhatDateFormat()
{

  /* Code. */

  return( date_format_string );

} /* TimWhatDateFormat */


/*----------------------------------------------------------------------*/

char
  *TimWhatTimeFormat()
{

  /* Code. */

  return( time_format_string );

} /* TimWhatTimeFormat */


/*----------------------------------------------------------------------*/

static struct tm
  *convertToLocalStruct( TIM_TIME_REF  time,
                         struct tm     *container )
{

  /* Variables. */
  struct tm  *ref;


  /* Code. */

  ref = localtime( (time_t *) &time );

  memcpy( container, ref, sizeof( struct tm ) );


  return( container );

} /* convertToLocalStruct */


/*----------------------------------------------------------------------*/

static struct tm
  *convertToStruct( TIM_TIME_REF  time,
                    struct tm     *container )
{

  /* Variables. */
  struct tm  *ref;


  /* Code. */

  ref = gmtime( (time_t *) &time );

  memcpy( container, ref, sizeof( struct tm ) );


  return( container );

} /* convertToStruct */


/*----------------------------------------------------------------------*/

static int
  dayNumberInYear( int year,
                   int month,
                   int day )
{

  /* Code. */

  if( daysInYear( year ) == 366 )
    return( day_offset_leap_year[ month - 1 ] + day );


  return( day_offset[ month - 1 ] + day );

} /* dayNumberInYear */


/*----------------------------------------------------------------------*/

static int
  daysInYear( int  year )
{

  /* Code. */

  if( year % 4 == 0 )
    return( 366 );


  return( 365 );

} /* daysInYear */


/*----------------------------------------------------------------------*/

static void
  formatDate( DATE_FORMAT   *date_format,
              TIM_TIME_REF  time,
              char          *buffer,
              int           buffer_size )
{

  /* Variables. */
  char       day[ 10 ];
  char       month[ 10 ];
  char       year[ 10 ];
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  if( buffer_size < 11 ) {
    *buffer = '\0';
    return;
  }

  time_struct = convertToStruct( time, &time_struct_obj );

  /* Year. */
  if( date_format -> include_century )
    (void) strftime( year, sizeof( year ), "%Y", time_struct );
  else
    (void) strftime( year, sizeof( year ), "%y", time_struct );

  /* Month. */
  (void) strftime( month, sizeof( month ), "%m", time_struct );
  if( ! date_format -> month_leading_zero && month[ 0 ] == '0' ) {
    month[ 0 ] = month[ 1 ];
    month[ 1 ] = '\0';
  }

  /* Day. */
  (void) strftime( day, sizeof( day ), "%d", time_struct );
  if( ! date_format -> day_leading_zero && day[ 0 ] == '0' ) {
    day[ 0 ] = day[ 1 ];
    day[ 1 ] = '\0';
  }

  /* Build the date. */
  if( date_format -> order == DATE_MMDDYY )
    sprintf( buffer, "%s%s%s%s%s",
             month, date_format -> separator,
             day, date_format -> separator,
             year );
  else if( date_format -> order == DATE_DDMMYY )
    sprintf( buffer, "%s%s%s%s%s",
             day, date_format -> separator,
             month, date_format -> separator,
             year );
  else
    sprintf( buffer, "%s%s%s%s%s",
             year, date_format -> separator,
             month, date_format -> separator,
             day );


  return;

} /* formatDate */


/*----------------------------------------------------------------------*/

static void
  formatFullDate( DATE_FORMAT   *date_format,
                  TIM_TIME_REF  time,
                  char          *buffer,
                  int           buffer_size )
{

  /* Variables. */
  char       day[ 10 ];
  char       month[ 10 ];
  char       year[ 10 ];
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  if( buffer_size < 30 ) {
    *buffer = '\0';
    return;
  }

  time_struct = convertToStruct( time, &time_struct_obj );

  /* Year. */
  (void) strftime( year, sizeof( year ), "%Y", time_struct );

  /* Month. */
  (void) strftime( month, sizeof( month ), "%B", time_struct );

  /* Day. */
  (void) strftime( day, sizeof( day ), "%d", time_struct );
  if( day[ 0 ] == '0' ) {
    day[ 0 ] = day[ 1 ];
    day[ 1 ] = '\0';
  }

  /* Build the date. */
  if( date_format -> order == DATE_MMDDYY )
    sprintf( buffer, "%s %s, %s", month, day, year );
  else if( date_format -> order == DATE_DDMMYY )
    sprintf( buffer, "%s. %s %s", day, month, year );
  else
    sprintf( buffer, "%s. %s %s", day, month, year );


  return;

} /* formatFullDate */


/*----------------------------------------------------------------------*/

static void
  formatTime( TIME_FORMAT   *time_format,
              TIM_TIME_REF  time,
              char          *buffer,
              int           buffer_size )
{

  /* Variables. */
  char       hour[ 10 ];
  char       minute[ 10 ];
  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  if( buffer_size < 9 ) {
    *buffer = '\0';
    return;
  }

  time_struct = convertToStruct( time, &time_struct_obj );

  /* Hour. */
  if( time_format -> hour_12 )
    (void) strftime( hour, sizeof( hour ), "%I", time_struct );
  else
    (void) strftime( hour, sizeof( hour ), "%H", time_struct );

  if( ! time_format -> hour_leading_zero && hour[ 0 ] == '0' ) {
    hour[ 0 ] = hour[ 1 ];
    hour[ 1 ] = '\0';
  }

  /* Minute. */
  (void) strftime( minute, sizeof( minute ), "%M", time_struct );

  /* Build the time. */
  if( TimHour( time ) > 11 )
    sprintf( buffer, "%s%s%s%s%s",
             hour, time_format -> separator,
             minute, time_format -> hour_24_suffix_separator,
             time_format -> hour_24_suffix );
  else
    sprintf( buffer, "%s%s%s%s%s",
             hour, time_format -> separator,
             minute, time_format -> hour_12_suffix_separator,
             time_format -> hour_12_suffix );


  return;

} /* formatTime */


/*----------------------------------------------------------------------*/

static TIM_STATUS_TYPE  
  isTimeOk( int  year,
            int  month,
            int  day,
            int  hour,
            int  minute,
            int  second )
{

  /* Variables. */
  TIM_TIME_REF  time;


  /* Code. */

  /* The easy check. */
  if( year < 1900 )
    year = year + 1900;

  if( year   < 1970 || year   > 2100 ||
      month  < 1    || month  > 12 ||
      day    < 1    || day    > 31 ||
      hour   < 0    || hour   > 23 ||
      minute < 0    || minute > 59 ||
      second < 0    || second > 59 )
    return( TIM_ERROR );

  time = TimMakeTime( year, month, 1, 0, 0, 0 );

  if( day > TimDaysInMonth( time ) )
    return( TIM_ERROR );


  return( TIM_OK );

} /* isTimeOk */


/*----------------------------------------------------------------------*/

static TIM_STATUS_TYPE
  makeDateFromString( DATE_FORMAT   *date_format,
                      TIM_TIME_REF  *time,
                      char          *string )
{

  /* Variables. */
  Boolean  has_match = False;
  int      day = 1;
  int      matches;
  int      month = 1;
  int      par1;
  int      par2;
  int      par3;
  int      year = 1970;

  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  *time = TimLocalTime( TimMakeTimeNow() );

  time_struct = convertToStruct( *time, &time_struct_obj );

  time_struct -> tm_hour = 0;
  time_struct -> tm_min  = 0;
  time_struct -> tm_sec  = 0;


  /* Three parameters? */
  if( ! has_match ) {
    matches = sscanf( string, date_format -> ymd_format,
                      &par1, &par2, &par3 );
    if( matches == 3 )
      has_match = True;
  }

  /* Two parameters? */
  if( ! has_match ) {
    matches = sscanf( string, date_format -> md_format,
                      &par1, &par2 );
    if( matches == 2 )
      has_match = True;
  }

  /* One parameters? */
  if( ! has_match ) {
    matches = sscanf( string, date_format -> d_format,
                      &par1 );
    if( matches == 1 )
      has_match = True;
  }

  /* Any matches? */
  if( ! has_match )
    return( TIM_ERROR );

  /* MMDDYY format. */
  if( date_format -> order == DATE_MMDDYY ) {
    if( matches == 3 ) {
      year  = par3;
      month = par1;
      day   = par2;
    } else if( matches == 2 ) {
      year  = time_struct -> tm_year;
      month = par1;
      day   = par2;
    } else {
      year  = time_struct -> tm_year;
      month = time_struct -> tm_mon + 1;
      day   = par1;
    }
  }

  /* DDMMYY format. */
  if( date_format -> order == DATE_DDMMYY ) {
    if( matches == 3 ) {
      year  = par3;
      month = par2;
      day   = par1;
    } else if( matches == 2 ) {
      year  = time_struct -> tm_year;
      month = par2;
      day   = par1;
    } else {
      year  = time_struct -> tm_year;
      month = time_struct -> tm_mon + 1;
      day   = par1;
    }
  }

  /* YYMMDD format. */
  if( date_format -> order == DATE_YYMMDD ) {
    if( matches == 3 ) {
      year  = par1;
      month = par2;
      day   = par3;
    } else if( matches == 2 ) {
      year  = time_struct -> tm_year;
      month = par1;
      day   = par2;
    } else {
      year  = time_struct -> tm_year;
      month = time_struct -> tm_mon + 1;
      day   = par1;
    }
  }


  /* Check the time. */
  if( isTimeOk( year, month, day, 0, 0, 0 ) != TIM_OK )
    return( TIM_ERROR );

  /* Build the time. */
  if( year < 1000 )
    time_struct -> tm_year  = year;
  else
    time_struct -> tm_year  = year - 1900;

  time_struct -> tm_mon  = month - 1;
  time_struct -> tm_mday = day;
  time_struct -> tm_yday = dayNumberInYear( year, month, day ) - 1;

  time_struct -> tm_hour = 0;
  time_struct -> tm_min  = 0;
  time_struct -> tm_sec  = 0;

  *time = secondsSince1970( time_struct );


  return( TIM_OK );

} /* makeDateFromString */


/*----------------------------------------------------------------------*/

static TIM_STATUS_TYPE
  makeTimeFromString( TIME_FORMAT   *time_format,
                      TIM_TIME_REF  *time,
                      char          *string )
{

  /* Variables. */
  Boolean  before_12 = False;
  Boolean  has_match = False;
  int      hour;
  int      matches;
  int      minute;
  int      par1;
  int      par2;
  char     suffix[ 50 ];

  struct tm  *time_struct;
  struct tm  time_struct_obj;


  /* Code. */

  *time = TimLocalTime( TimMakeTimeNow() );

  time_struct = convertToStruct( *time, &time_struct_obj );

  hour   = time_struct -> tm_hour;
  minute = time_struct -> tm_min;


  /* Two parameters? */
  if( ! has_match ) {
    if( time_format -> hour_12 ) {
      matches = sscanf( string, time_format -> hm_format,
                        &par1, &par2, suffix );
      if( matches == 3 )
        has_match = True;
    } else {
      matches = sscanf( string, time_format -> hm_format,
                        &par1, &par2 );
      if( matches == 2 )
        has_match = True;
    }
  }

  /* One parameter? */
  if( ! has_match ) {
    if( time_format -> hour_12 ) {
      matches = sscanf( string, time_format -> hm_format,
                        &par1, suffix );
      if( matches == 2 )
        has_match = True;
    } else {
      matches = sscanf( string, time_format -> hm_format,
                        &par1 );
      if( matches == 1 )
        has_match = True;
    }
  }

  /* Any matches? */
  if( ! has_match )
    return( TIM_ERROR );

  /* 12-hour suffix? */
  if( time_format -> hour_12 && matches == 3 ) {
    if( noCaseStrcmp( suffix, time_format -> hour_12_suffix ) == 0 )
      before_12 = True;
    else if( noCaseStrcmp( suffix, time_format -> hour_24_suffix ) == 0 )
      before_12 = False;
    else
      return( TIM_ERROR );
  }


  /* Build the time. */
  if( time_format -> hour_12 ) {
    if( matches == 3 ) {
      hour   = par1;
      minute = par2;

      /* 12 AM = 0:00, 12 PM = 12:00. */
      if( ! before_12 && hour != 12 )
        hour = hour + 12;
      else if( before_12 && hour == 12 )
        hour = 0;

    } else {
      minute = par1;
    }
  }

  if( ! time_format -> hour_12 ) {
    if( matches == 2 ) {
      hour   = par1;
      minute = par2;
    } else {
      minute = par1;
    }
  }


  /* Check the time. */
  if( isTimeOk( 1970, 1, 1, hour, minute, 0 ) != TIM_OK )
    return( TIM_ERROR );

  time_struct -> tm_year = 70;
  time_struct -> tm_mon  = 0;
  time_struct -> tm_mday = 1;
  time_struct -> tm_yday = 0;

  time_struct -> tm_hour = hour;
  time_struct -> tm_min  = minute;
  time_struct -> tm_sec  = 0;

  *time = secondsSince1970( time_struct );


  return( TIM_OK );

} /* makeTimeFromString */


/*----------------------------------------------------------------------*/

static int
  noCaseStrcmp( char  *buffer1,
                char  *buffer2 )
{

  /* Variables. */
  char  *char_ref1;
  char  *char_ref2;


  /* Code. */

  if( strlen( buffer1 ) != strlen( buffer2 ) )
    return( strcmp( buffer1, buffer2 ) );

  char_ref1 = buffer1;
  char_ref2 = buffer2;

  while( *char_ref1 != '\0' ) {
    if( tolower( *char_ref1 ) < tolower( *char_ref2 ) )
      return( -1 );
    else if( tolower( *char_ref1 ) > tolower( *char_ref2 ) )
      return( 1 );

    char_ref1++;
    char_ref2++;
  }


  return( 0 );

} /* noCaseStrcmp */


/*----------------------------------------------------------------------*/

static time_t
  secondsSince1970( struct tm *time_ref )
{

  /* Variables. */
  int     index;
  time_t  seconds = 0;


  /* Code. */

  if( time_ref -> tm_year < 70 )
    return( 0 );

  for( index = 70 + 1900; index < time_ref -> tm_year + 1900; index++ )
    seconds = seconds + (daysInYear( index ) * 86400);

  seconds = seconds + time_ref -> tm_yday * 86400;
  seconds = seconds + time_ref -> tm_hour * 3600;
  seconds = seconds + time_ref -> tm_min  * 60;
  seconds = seconds + time_ref -> tm_sec;


  return( seconds );

} /* secondsSince1970 */
