/*----------------------------------------------------------------------------
--
--  Module:           DirScan
--
--  Project:          Tools - General C objects.
--  System:           <>
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    This module provides a way to scan a directory tree.
--
--  Filename:         DirScan.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1993-01-12
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: DirScan.c, Version: 1.2, Date: 95/02/18 14:33:54";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <limits.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>

#include "DirScan.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Exception handler (raise exception). */
#define raise  goto


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record with internal data. */
typedef struct {

  /* Abort? */
  Boolean  abort;

  /* Error count. */
  int  err_count;

  /* Max length for a directory. */
  int  max_dir_length;

  /* Flags. */
  UINT32  flags;

  /* User action routines. */
  DIR_SCAN_ACTION_CB  actionCB;
  void                *user_data;

} DIR_SCAN_REC, *DIR_SCAN_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static char
  *getCurrentDir( DIR_SCAN_REC_REF  dir_scan_ref );

static void
  traverseDirectoryTree( DIR_SCAN_REC_REF  dir_scan_ref,
                         char              *top_directory );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

DIR_SCAN_HANDLE
  DirScanInitialize( DIR_SCAN_ACTION_CB  actionCB,
                     void                *user_data )
{

  /* Variables. */
  DIR_SCAN_REC_REF  dir_scan_ref;


  /* Code. */

  dir_scan_ref = SysNew( DIR_SCAN_REC );
  if( dir_scan_ref == NULL )
    return( NULL );

  dir_scan_ref -> max_dir_length = PATH_MAX;
  dir_scan_ref -> actionCB       = actionCB;
  dir_scan_ref -> user_data      = user_data;


  return( (DIR_SCAN_HANDLE) dir_scan_ref );

} /* DirScanInitialize */


/*----------------------------------------------------------------------*/

void
  DirScanDestroy( DIR_SCAN_HANDLE  dir_scan_handle )
{

  /* Variables. */
  DIR_SCAN_REC_REF  dir_scan_ref;


  /* Code. */

  dir_scan_ref = (DIR_SCAN_REC_REF) dir_scan_handle;

  SysFree( dir_scan_ref );


  return;

} /* DirScanDestroy */


/*----------------------------------------------------------------------*/

int
  DirScanTraverseTree( DIR_SCAN_HANDLE  dir_scan_handle,
                       char             *top_directory,
                       UINT32           flags )
{

  /* Variables. */
  DIR_SCAN_REC_REF  dir_scan_ref;


  /* Code. */

  dir_scan_ref = (DIR_SCAN_REC_REF) dir_scan_handle;

  dir_scan_ref -> abort     = False;
  dir_scan_ref -> err_count = 0;
  dir_scan_ref -> flags     = flags;

  /* Start the traverse. */
  traverseDirectoryTree( dir_scan_ref, top_directory );


  return( dir_scan_ref -> err_count );

} /* DirScanTraverseTree */


/*----------------------------------------------------------------------*/

static void
  traverseDirectoryTree( DIR_SCAN_REC_REF  dir_scan_ref,
                         char              *top_directory )
{

  /* Variables. */
  Boolean          call_function;
  Boolean          continue_processing;
  Boolean          descend_into_dir;
  int              bytes;
  int              status;
  UINT32           status_flags;
  long             dir_location;
  char             file_buffer[ PATH_MAX + 1 ];
  char             link_path[ PATH_MAX + 2 ];
  char             *dir_path = NULL;
  char             *file_name;
  char             *orig_dir_path = NULL;
  struct dirent*   current_dirent;
  struct stat      file_info;
  DIR*             current_dir = NULL;
  DIR_SCAN_INFO    dir_info;


  /* Code. */

  /* Save path to current directory. */
  orig_dir_path = getCurrentDir( dir_scan_ref );
  if( orig_dir_path == NULL )
    raise exception;

  /* It is easier to access files if we are positioned in the directory. */
  status = chdir( top_directory );
  if( status == -1 )
    raise exception;

  dir_path = getCurrentDir( dir_scan_ref );
  if( dir_path == NULL )
    raise exception;


  /* We need to open the directory to be able to scan it. */
  current_dir = opendir( dir_path );
  if( current_dir == NULL )
    raise exception;

  while( (current_dirent = readdir( current_dir )) != NULL &&
         ! dir_scan_ref -> abort ) {

    status_flags = 0;
    file_name = current_dirent -> d_name;

    do {

      status = lstat( file_name, &file_info );
      if( status != 0 ) {
        dir_scan_ref -> err_count++;
        continue_processing = True;

        continue;
      }
  
      /* Check if function should be called for the file. */
      continue_processing = False;

      call_function = False;

      if( S_ISREG( file_info.st_mode ) ) {
        if( flagIsSet( dir_scan_ref -> flags, DIR_SCAN_CALL_FOR_FILES ) )
          call_function = True;

      } else if( S_ISDIR( file_info.st_mode ) ) {
        if( flagIsSet( dir_scan_ref -> flags, DIR_SCAN_CALL_FOR_SUBDIRS ) )
          call_function = True;

      } else if( S_ISLNK( file_info.st_mode ) ) {
        if( flagIsSet( dir_scan_ref -> flags, DIR_SCAN_CALL_FOR_SOFT_LINKS ) )
          call_function = True;

      } else if( S_ISBLK( file_info.st_mode ) ||
                 S_ISCHR( file_info.st_mode ) ) {
        if( flagIsSet( dir_scan_ref -> flags, DIR_SCAN_CALL_FOR_DRIVERS ) )
          call_function = True;

      } else if( S_ISSOK( file_info.st_mode ) ) {
        if( flagIsSet( dir_scan_ref -> flags, DIR_SCAN_CALL_FOR_SOCKETS ) )
          call_function = True;

      } else if( S_ISFIFO( file_info.st_mode ) ) {
        if( flagIsSet( dir_scan_ref -> flags, DIR_SCAN_CALL_FOR_PIPES ) )
          call_function = True;
      }


      /* Call user action? */
      if( call_function ) {
        dir_info.path      = dir_path;
        dir_info.name      = file_name;
        dir_info.file_info = &file_info;

        if( dir_scan_ref -> actionCB != NULL )
          status_flags = 
            (* dir_scan_ref -> actionCB)( DIR_SCAN_REASON_FILE_INFO,
                                          &dir_info,
                                          dir_scan_ref -> user_data );
          if( flagIsSet( status_flags, DIR_SCAN_ABORT ) )
            dir_scan_ref -> abort = True;
      }


      /* Actions for directories. */
      if( S_ISDIR( file_info.st_mode ) ) {

        descend_into_dir = False;

        /* Block out current and parent dirs. */
        if( strcmp( file_name, "." )  != 0 &&
            strcmp( file_name, ".." ) != 0 ) {

          /* Mounted-over directory? */
#ifdef NotWorking
          if( flagIsSet( file_info.st_flag, FS_MOUNT ) ) {
            if( flagIsSet( dir_scan_ref -> flags,
                           DIR_SCAN_DESCEND_IN_MOUNTED ) )
              descend_into_dir = True;

          } else {
#endif
            if( flagIsSet( dir_scan_ref -> flags,
                           DIR_SCAN_DESCEND_IN_SUBDIRS ) )
              descend_into_dir = True;
#ifdef NotWorking
          }
#endif

        } /* if */

        /* Descend in directory? */
        if( descend_into_dir && 
            flagIsClear( status_flags, DIR_SCAN_DO_NOT_DESCEND ) &&
            ! dir_scan_ref -> abort ) {

          /* Close the current directory first. */
          dir_location = telldir( current_dir );
          closedir( current_dir );

          current_dir = NULL;

          /* Perform same calculation for subdirectory. */
          strcpy( file_buffer, file_name );
          file_name = file_buffer;

          traverseDirectoryTree( dir_scan_ref, file_name );

          /* Restore to status before call. */
          current_dir = opendir( dir_path );
          seekdir( current_dir, dir_location );

        } /* if */

      /* Action for symbolic links. */
      } else if( S_ISLNK( file_info.st_mode ) &&
                 flagIsSet( dir_scan_ref -> flags,
                            DIR_SCAN_FOLLOW_SOFT_LINKS ) &&
                 ! dir_scan_ref -> abort ) {

        bytes = readlink( file_name, link_path, PATH_MAX );

        if( bytes > 0 && bytes <= PATH_MAX ) {
          link_path[ bytes ] = '\0';

          strcpy( file_buffer, link_path );
          file_name = file_buffer;
          continue_processing = True;

        } else {
          dir_scan_ref -> err_count++;
          continue_processing = False;
        }

      } /* if */

    } while( continue_processing );

  } /* while */ 


  /* Close current directory. */
  if( current_dir != NULL )
    closedir( current_dir );

  if( dir_path != NULL )
    SysFree( dir_path );

  status = chdir( orig_dir_path );
  if( orig_dir_path != NULL )
    SysFree( orig_dir_path );


  return;


  /* Exceptions. */
  exception:
    if( dir_path != NULL )
      SysFree( dir_path );

    if( orig_dir_path != NULL )
      SysFree( orig_dir_path );

    if( current_dir != NULL );
      closedir( current_dir );

    dir_scan_ref -> err_count++;

    return;

} /* traverseDirectoryTree */


/*----------------------------------------------------------------------*/

static char
  *getCurrentDir( DIR_SCAN_REC_REF  dir_scan_ref )
{

  /* Variables. */
  char  *buffer;
  char  *dir_buffer;


  /* Code. */

  dir_buffer  = SysMalloc( dir_scan_ref -> max_dir_length + 1 );
  *dir_buffer = '\0';

  do {

    buffer = (char *) getcwd( dir_buffer,
                              (size_t) dir_scan_ref -> max_dir_length );

    if( buffer != NULL )
      return( dir_buffer );

    if( errno != ERANGE )
      return( NULL );

    SysFree( dir_buffer );

    dir_scan_ref -> max_dir_length = dir_scan_ref -> max_dir_length + 30;

    dir_buffer  = SysMalloc( dir_scan_ref -> max_dir_length + 1 );
    *dir_buffer = '\0';

  } while( True );


} /* getCurrentDir */
