/*----------------------------------------------------------------------------
--
--  Module:           xtmOpenView
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Open a single day list view.
--
--  Filename:         xtmOpenView.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-06-15
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmOpenView.c, Version: 1.1, Date: 95/02/18 15:52:36";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmDbTools.h"
#include "xtmHelp.h"
#include "xtmLocate.h"
#include "xtmPickDiary.h"
#include "xtmTools.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmOpenView.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* How many calendars can we have? */
#define MAX_CALENDARS     200


/* Local widgets in the database select window. */
#define calActionRc       dataLocalW[  0 ]
#define calLocLa          dataLocalW[  1 ]
#define calLocTx          dataLocalW[  2 ]
#define calNameLa         dataLocalW[  3 ]
#define calNameTx         dataLocalW[  4 ]
#define dbLocLa           dataLocalW[  5 ]
#define dbLocLi           dataLocalW[  6 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Our own user data. */
typedef struct {

  /* The window is application modal? */
  Boolean  appl_modal;

  /* The selected calendar. */
  char  selected_cal[ XTM_GL_MAX_CAL_NAME + 1 ];

  /* The open view window. */
  Widget  openW;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Pick calendars. */
  XTM_PD_HANDLE  pick_handle;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XTM_OV_ACTION_CB  actionCB;

} OPEN_REC, *OPEN_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmOpenView";

/* IDs for the help windows. */
static char  *open_window_id = "OpenView";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  applyCB( Widget        widget,
           OPEN_REC_REF  open_ref,
           XtPointer     call_data );

static void 
  calSelectedCB( Widget                widget, 
                 OPEN_REC_REF          open_ref,
                 XmListCallbackStruct  *call_data );

static void
  cancelCB( Widget        widget,
            OPEN_REC_REF  open_ref,
            XtPointer     call_data );

Widget
  createOpenViewWindow( OPEN_REC_REF  open_ref,
                        Widget        parent );

static void 
  destroyCB( Widget        widget,
             OPEN_REC_REF  open_ref,
             XtPointer     call_data );

static Boolean
  fetchSelectedCal( OPEN_REC_REF  open_ref );

static void
  guessLocationCB( Widget        widget,
                   OPEN_REC_REF  open_ref,
                   XtPointer     call_data );

static void 
  helpCB( Widget        widget,
          OPEN_REC_REF  open_ref,
          XtPointer     call_data );

static void
  okCB( Widget        widget,
        OPEN_REC_REF  open_ref,
        XtPointer     call_data );

static void
  pickCalCB( Widget        widget,
             OPEN_REC_REF  open_ref,
             XtPointer     call_data );

static void
  pickCalApplyCB( XTM_PD_REASON    reason,
                  XTM_CD_CAL_INFO  *db_ref,
                  void             *client_data );

void
  setCalendarList( OPEN_REC_REF  open_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_OV_HANDLE
  xtmOvInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                parent,
                   Boolean               appl_modal,
                   XTM_OV_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  OPEN_REC_REF  open_ref;


  /* Code. */

  /* Create and initialize our private data. */
  open_ref = SysNew( OPEN_REC );
  if( open_ref == NULL )
    return( NULL );

  open_ref -> appl_data_ref     = appl_data_ref;
  open_ref -> appl_modal        = appl_modal;
  open_ref -> pick_handle       = NULL;
  open_ref -> actionCB          = actionCB;
  open_ref -> user_data         = user_data;
  open_ref -> selected_cal[ 0 ] = '\0';


  /* Create the open view window. */
  open_ref -> openW = createOpenViewWindow( open_ref, parent );

  if( open_ref -> openW == NULL ) {
    SysFree( open_ref );

    return( NULL );
  }


  return( (XTM_OV_HANDLE) open_ref );

} /* xtmOvInitialize */


/*----------------------------------------------------------------------*/

void
  xtmOvDestroy( XTM_OV_HANDLE  open_handle )
{

  /* Variables. */
  OPEN_REC_REF  open_ref;


  /* Code. */

  if( open_handle == NULL )
    return;

  /* Our private data. */
  open_ref = (OPEN_REC_REF) open_handle;


  /* Destroy the window. */
  XtDestroyWidget( open_ref -> openW );


  return;

} /* xtmOvDestroy */


/*----------------------------------------------------------------------*/

void 
  xtmOvOpenView( XTM_OV_HANDLE  open_handle )
{

  /* Variables. */
  OPEN_REC_REF  open_ref;


  /* Code. */

  if( open_handle == NULL )
    return;

  /* Our private data. */
  open_ref = (OPEN_REC_REF) open_handle;


  /* Set the list of calendars we can use. */
  setCalendarList( open_ref );


  /* Make sure the window is visible. */
  XtManageChild( open_ref -> openW );


  return;

} /* xtmOvOpenView */


/*----------------------------------------------------------------------*/

Widget
  createOpenViewWindow( OPEN_REC_REF  open_ref,
                        Widget        parent )
{

  /* Variables. */
  int       index;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    calActionBu[ 2 ];
  Widget    openFd;
  Widget    workFo;
  Widget    dataLocalW[ 7 ];

  static XIT_PUSH_STRUCT cal_action_def[] = {
    { "PickPb",  "", "", True, NULL },
    { "GuessPb", "", "", True, NULL },
  };

  static XIT_TEXT_STRUCT text_buffer[] = {
    { "CalNameTx", NULL, 1, True },
    { "CalLocTx",  NULL, 1, True },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", okCB,     NULL },
    { "", applyCB,  NULL },
    { "", cancelCB, NULL },
    { "", helpCB,   NULL },
  };


  /* Code. */

  /* Set message strings. */
  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = open_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_APPLY_BUTTON );
  action_buttons[ 1 ].data  = open_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 2 ].data  = open_ref;
  action_buttons[ 3 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 3 ].data  = open_ref;

  cal_action_def[ 0 ].title = msgGetText( MXDI_PICK_CALENDAR );
  cal_action_def[ 1 ].title = msgGetText( MXDI_GUESS_LOCATION_BUTTON );


  /* Create a form dialog with buttons. */
  openFd = xitCreateFormDialog( parent, "OpenFd",
                                1, 0,
                                action_buttons,
                                XtNumber( action_buttons ) );

  XtAddCallback( openFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) open_ref );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, " " ); n++;
  XtSetValues( XtParent( openFd ), args, n );

  if( open_ref -> appl_modal ) {
    n = 0;
    XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
    XtSetValues( openFd, args, n );
  }


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( openFd, "OpenFdFo" );


  /* What should the user do? */
  dbLocLa = xitCreateLabel( workFo, "DbLocLa", 
                            msgGetText( MXDI_SELECT_CALENDAR ), -1 );


  /* List with calendars. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy,         XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy,        XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight,       5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth,        5 ); n++;
  dbLocLi = XmCreateScrolledList( workFo, "DbLocLi", args, n );

  XtAddCallback( dbLocLi, XmNsingleSelectionCallback, 
                 (XtCallbackProc) calSelectedCB, (XtPointer) open_ref );
  XtAddCallback( dbLocLi, XmNdefaultActionCallback, 
                 (XtCallbackProc) calSelectedCB, (XtPointer) open_ref );


  /* Name of calendar. */
  calNameLa = xitCreateLabel( workFo, "CalNameLa", 
                              msgGetText( MXDI_DB_NAME_LABEL ), -1 );

  calNameTx = xitCreateTextCols( workFo, 
                                 &text_buffer[ 0 ], XTM_GL_MAX_CAL_NAME );


  /* Location for the calendar. */
  calLocLa = xitCreateLabel( workFo, "CalLocLa", 
                             msgGetText( MXDI_DB_DIR_LABEL ), -1 );

  calLocTx = xitCreateTextCols( workFo, &text_buffer[ 1 ], 35 );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, XTM_GL_MAX_CAL_NAME ); n++;
  XtSetValues( calNameTx, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, XTM_GL_MAX_LOCATION + 1 ); n++;
  XtSetValues( calLocTx, args, n );


  /* Action (select). */
  n = 0;
  XtSetArg( args[ n ], XmNorientation,  XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNspacing,      10 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight, 10 ); n++;
  calActionRc = XmCreateRowColumn( workFo, "CalActionRc", args, n );

  for( index = 0; index < XtNumber( cal_action_def ); index++ )
    calActionBu[ index ] = xitCreatePushButton( calActionRc,
                                                &cal_action_def[ index ] );

  XtAddCallback( calActionBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) pickCalCB, (XtPointer) open_ref );
  XtAddCallback( calActionBu[ 1 ], XmNactivateCallback, 
                 (XtCallbackProc) guessLocationCB, (XtPointer)  open_ref );


  /* Put the elements together. */
  xitAttachWidget( dbLocLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( dbLocLi ),
                   XmATTACH_WIDGET, dbLocLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,    XmATTACH_NONE, NULL );
  xitAttachWidget( calNameLa,
                   XmATTACH_WIDGET, XtParent( dbLocLi ),
                   XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( calNameTx,
                   XmATTACH_WIDGET, calNameLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( calLocLa,
                   XmATTACH_WIDGET, calNameTx, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( calLocTx,
                   XmATTACH_WIDGET, calLocLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );
  xitAttachWidget( calActionRc,
                   XmATTACH_WIDGET, calLocTx, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetValues( dbLocLa,             args, n );
  XtSetValues( XtParent( dbLocLi ), args, n );
  XtSetValues( calNameLa,           args, n );
  XtSetValues( calNameTx,           args, n );
  XtSetValues( calLocLa,            args, n );
  XtSetValues( calLocTx,            args, n );
  XtSetValues( calActionRc,         args, n );


  /* Manage the widgets. */
  xitManageChildren( calActionBu, XtNumber( calActionBu ) );
  xitManageChildren( dataLocalW,  XtNumber( dataLocalW ) );

  /* Set the size of the window. */
  xitSetSizeFormDialog( openFd, True );


  /* Make sure our children don't spoil our size. */
  n = 0;
  XtSetArg( args[ n ], XmNallowShellResize, False ); n++;
  XtSetValues( XtParent( openFd ), args, n );


  /* Make the final attachments. */
  xitAttachWidget( calActionRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( calLocTx,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, calActionRc );
  xitAttachWidget( calLocLa,
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, calLocTx );
  xitAttachWidget( calNameTx,
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, calLocLa );
  xitAttachWidget( calNameLa,
                   XmATTACH_NONE, NULL, XmATTACH_FORM,   NULL,
                   XmATTACH_NONE, NULL, XmATTACH_WIDGET, calNameTx );
  xitAttachWidget( XtParent( dbLocLi ),
                   XmATTACH_WIDGET, dbLocLa, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL,    XmATTACH_WIDGET, calNameLa );


  return( openFd );

} /* createOpenViewWindow */


/*----------------------------------------------------------------------*/

static Boolean
  fetchSelectedCal( OPEN_REC_REF  open_ref )
{

  /* Variables. */
  Boolean                 ok;
  int                     items;
  UINT32                  operations;
  char                    cal_name[ XTM_GL_MAX_CAL_NAME + 1 ];
  char                    location[ XTM_GL_MAX_LOCATION + 1 ];
  char                    *char_ref;
  Widget                  mainW;
  Widget                  tempW;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  custom_data_ref = open_ref -> appl_data_ref -> custom_data;
  mainW           = XtNameToWidget( open_ref -> openW, "OpenFdFo" );

  open_ref -> selected_cal[ 0 ] = '\0';


  /* Fetch the calendar name. */
  tempW    = XtNameToWidget( mainW, "CalNameTx" );
  char_ref = xitStringGetText( tempW );

  items = sscanf( char_ref, "%s", cal_name );
  SysFree( char_ref );

  if( items != 1 ) {
    xitErMessage( open_ref -> openW, XIT_ER_INFO,
                  module_name, "fetchSelectedCal",
                  msgGetText( MXDI_MISSING_NAME_OR_LOC ) );

    return( False );
  }

  /* Fetch the calendar location. */
  tempW = XtNameToWidget( mainW, "CalLocTx" );
  char_ref = xitStringGetText( tempW );

  location[ 0 ] = '\0';
  items = sscanf( char_ref, "%s", location );
  SysFree( char_ref );



  /* Fetch information about the calendar. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, cal_name,
                          &db_info, NULL );
  if( ok ) {
    strcpy( open_ref -> selected_cal, db_info.short_name );

    return( True );
  }


  /* We did not find the calendar in our calendar DB. Any location? */
  if( strlen( location ) == 0 ) {
    xitErMessage( open_ref -> openW, XIT_ER_INFO,
                  module_name, "fetchSelectedCal",
                  msgGetText( MXDI_NO_CAL_NO_LOC ) );

    return( False );
  }


  /* We need at least read access. */
  xtmDbCheckDbOperations( location, True, &operations );

  if( flagIsClear( operations, XTM_DB_FLAG_MODE_READ ) ) {
    xitErMessage( open_ref -> openW, XIT_ER_ERROR,
                  module_name, "fetchSelectedCal",
                  msgGetText( MXDI_CUSTOM_DIR_NOT_EXIST ) );

    return( False );
  }


  /* Add the calendar to the DB and the list. */
  xtmCdAddDatabase( custom_data_ref -> cal_db_handle,
                    cal_name, location, 0 );

  setCalendarList( open_ref );
  strcpy( open_ref -> selected_cal, cal_name );


  return( True );

} /* fetchSelectedCal */


/*----------------------------------------------------------------------*/

void
  setCalendarList( OPEN_REC_REF  open_ref )
{

  /* Variables. */
  int                     index;
  int                     index1;
  UINT32                  operations;
  char                    buffer[ 100 ];
  char                    flags_buffer[ 20 ];
  char                    *char_ref;
  char                    *db_names;
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  mainW;
  Widget                  tempW;
  XmString                list_items[ MAX_CALENDARS ];
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = open_ref -> appl_data_ref -> custom_data;
  mainW           = XtNameToWidget( open_ref -> openW, "OpenFdFo" );


  /* Set values for the database locations. */
  (void) xtmCdFetchDbNames( custom_data_ref -> cal_db_handle, &db_names );
  char_ref = db_names;
  index    = 0;

  do {

    int              char_read;
    char             db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
    XTM_CD_CAL_INFO  db_info;

    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    sscanf( char_ref, "%s%n", db_name, &char_read );
    char_ref = char_ref + char_read;


    /* Fetch information about the database. */
    (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, db_name,
                              &db_info, NULL );

    /* Database operations possible. */
    xtmDbCheckDbOperations( db_info.directory, True, &operations );


    /* Only calendars we can read are shown. */
    if( flagIsClear( operations, XTM_DB_FLAG_MODE_READ ) )
      continue;


    /* Fetch the flags values. */
    flags_buffer[ 0 ] = '\0';

    if( flagIsSet( operations, XTM_DB_FLAG_MODE_READ ) )
      strcat( flags_buffer, "r" );
    else
      strcat( flags_buffer, "-" );

    if( flagIsSet( operations, XTM_DB_FLAG_MODE_WRITE ) )
      strcat( flags_buffer, "w" );
    else
      strcat( flags_buffer, "-" );

    strcat( flags_buffer, " " );


    sprintf( buffer, "%-15.15s %-5.5s %s", 
	     db_info.short_name, flags_buffer,
	     db_info.directory );

    list_items[ index ] = XmStringCreate( buffer, CS );
    index++;

    if( index >= MAX_CALENDARS )
      break;

  } while( True );

  if( db_names != NULL )
    SysFree( db_names );


  /* The list is always sorted. */
  xitSortStringList( list_items, index );


  tempW = XtNameToWidget( mainW, "DbLocLiSW.DbLocLi" );

  /* Assign the database locations to the list. */
  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, index ); n++;
  XtSetArg( args[ n ], XmNselectedItemCount, 0 ); n++;
  XtSetValues( tempW, args, n );

  /* Free allocated memory. */
  for( index1 = 0; index1 < index; index1++ )
    XmStringFree( list_items[ index1 ] );


  return;

} /* setCalendarList */


/*----------------------------------------------------------------------*/

static void
  applyCB( Widget        widget,
           OPEN_REC_REF  open_ref,
           XtPointer     call_data )
{

  /* Variables. */
  Boolean                 ok;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  custom_data_ref = open_ref -> appl_data_ref -> custom_data;

  /* Fetch the selected database. */
  ok = fetchSelectedCal( open_ref );
  if( ! ok )
    return;


  /* Fetch the selected calendar. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                          open_ref -> selected_cal,
                          &db_info, NULL );
  if( ! ok )
    return;


  /* Call callback function? */
  if( open_ref -> actionCB != NULL )
    (* open_ref -> actionCB )( XTM_OV_REASON_APPLY, &db_info,
                               open_ref -> user_data );


  return;

} /* applyCB */


/*----------------------------------------------------------------------*/

static void 
  calSelectedCB( Widget                widget, 
                 OPEN_REC_REF          open_ref,
                 XmListCallbackStruct  *call_data )
{

  /* Variables. */
  Boolean                 ok;
  char                    db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
  char                    *char_ref;
  Widget                  mainW;
  Widget                  tempW;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  custom_data_ref = open_ref -> appl_data_ref -> custom_data;
  mainW           = XtNameToWidget( open_ref -> openW, "OpenFdFo" );

  open_ref -> selected_cal[ 0 ] = '\0';


  /* Fetch the selected calendar line. */
  char_ref = xitStringGetString( call_data -> item, CS );

  sscanf( char_ref, "%s", db_name );
  SysFree( char_ref );


  /* Fetch the selected calendar. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                          db_name,
                          &db_info, NULL );
  if( ! ok )
    return;

  strcpy( open_ref -> selected_cal, db_info.short_name );


  /* Selected name and location. */
  tempW = XtNameToWidget( mainW, "CalNameTx" );
  XmTextSetString( tempW, db_info.short_name );

  tempW = XtNameToWidget( mainW, "CalLocTx" );
  XmTextSetString( tempW, db_info.directory );


  /* Default action is to open the day view. */  
  if( call_data -> reason == XmCR_DEFAULT_ACTION )
    okCB( widget, open_ref, NULL );


  return;

} /* calSelectedCB */


/*----------------------------------------------------------------------*/

static void
  cancelCB( Widget        widget,
            OPEN_REC_REF  open_ref,
            XtPointer     call_data )
{

  /* Code. */

  XtUnmanageChild( open_ref -> openW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget        widget,
             OPEN_REC_REF  open_ref,
             XtPointer     call_data )
{

  /* Code. */

  /* Call callback function? */
  if( open_ref -> actionCB != NULL )
    (* open_ref -> actionCB )( XTM_OV_REASON_DESTROY, NULL,
                               open_ref -> user_data );

  /* Release the user data. */
  if( open_ref -> pick_handle != NULL )
    xtmPdDestroy( open_ref -> pick_handle );

  SysFree( open_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void
  guessLocationCB( Widget        widget,
                   OPEN_REC_REF  open_ref,
                   XtPointer     call_data )
{

  /* Variables. */
  Boolean                 ok;
  int                     items;
  char                    cal_name[ XTM_GL_MAX_CAL_NAME + 1 ];
  char                    location[ XTM_GL_MAX_LOCATION + 1 ];
  char                    *char_ref;
  Widget                  mainW;
  Widget                  tempW;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  custom_data_ref = open_ref -> appl_data_ref -> custom_data;
  mainW           = XtNameToWidget( open_ref -> openW, "OpenFdFo" );


  /* Fetch the calendar name. */
  tempW    = XtNameToWidget( mainW, "CalNameTx" );
  char_ref = xitStringGetText( tempW );

  items = sscanf( char_ref, "%s", cal_name );
  SysFree( char_ref );

  if( items != 1 ) {
    xitErMessage( open_ref -> openW, XIT_ER_INFO,
                  module_name, "guessLocationCB",
                  msgGetText( MXDI_MISSING_CALENDAR_NAME ) );

    return;
  }


  /* Do we know of the calendar already? */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, cal_name,
                          &db_info, NULL );

  if( ok && flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_READ ) ) {
    tempW = XtNameToWidget( mainW, "CalLocTx" );
    XmTextSetString( tempW, db_info.directory );

    return;
  }


  /* Try to locate the calendar. */
  ok = xtmLcLocateCalendar( open_ref -> openW, cal_name,
                            custom_data_ref -> find_location_script,
                            location, sizeof( location ) );

  if( ! ok || strlen( location ) == 0 ) {
    xitErMessage( open_ref -> openW, XIT_ER_INFO,
                  module_name, "guessLocationCB",
                  msgGetText( MXDI_CANNOT_LOCATE_CALENDAR ) );

    return;
  }

  tempW = XtNameToWidget( mainW, "CalLocTx" );
  XmTextSetString( tempW, location );


  return;

} /* guessLocationCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget        widget,
          OPEN_REC_REF  open_ref,
          XtPointer     call_data )
{

  /* Code. */

  xtmHlDisplayHelp( open_ref -> appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    open_window_id, "" );

  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void
  okCB( Widget        widget,
        OPEN_REC_REF  open_ref,
        XtPointer     call_data )
{

  /* Variables. */
  Boolean                 ok;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  custom_data_ref = open_ref -> appl_data_ref -> custom_data;

  /* Fetch the selected database. */
  ok = fetchSelectedCal( open_ref );
  if( ! ok )
    return;


  /* Fetch the selected calendar. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                          open_ref -> selected_cal,
                          &db_info, NULL );
  if( ! ok )
    return;


  /* Call callback function? */
  if( open_ref -> actionCB != NULL )
    (* open_ref -> actionCB )( XTM_OV_REASON_APPLY, &db_info,
                               open_ref -> user_data );

  XtUnmanageChild( open_ref -> openW );


  return;

} /* okCB */


/*----------------------------------------------------------------------*/

static void
  pickCalCB( Widget        widget,
             OPEN_REC_REF  open_ref,
             XtPointer     call_data )
{

  /* Variables. */
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = open_ref -> appl_data_ref -> custom_data;

  /* Initialize? */
  if( open_ref -> pick_handle == NULL )
    open_ref -> pick_handle = xtmPdInitialize( 
                                open_ref -> openW,
                                True,
                                open_ref -> appl_data_ref -> info_handle,
                                pickCalApplyCB,
                                (void *) open_ref );

  /* Diaplay the window. */
  xtmPdDisplayPickWindow( open_ref -> pick_handle,
                          custom_data_ref -> db_selection_dir,
                          "*_cal" );

  return;

} /* pickCalCB */


/*----------------------------------------------------------------------*/

static void
  pickCalApplyCB( XTM_PD_REASON    reason,
                  XTM_CD_CAL_INFO  *db_ref,
                  void             *client_data )
{

  /* Variables. */
  Widget        mainW;
  Widget        tempW;
  OPEN_REC_REF  open_ref;


  /* Code. */

  if( reason != XTM_PD_REASON_OK )
    return;

  open_ref = (OPEN_REC_REF) client_data;
  mainW    = XtNameToWidget( open_ref -> openW, "OpenFdFo" );

  tempW = XtNameToWidget( mainW, "CalNameTx" );
  XmTextSetString( tempW, db_ref -> short_name );

  tempW = XtNameToWidget( mainW, "CalLocTx" );
  XmTextSetString( tempW, db_ref -> directory );
  

  return;

} /* pickCalApplyCB */
