/*----------------------------------------------------------------------------
--
--  Module:           xtmNavCal
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Create a navigation calendar.
--
--  Filename:         xtmNavCal.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-04-24
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmNavCal.c, Version: 1.1, Date: 95/02/18 15:52:35";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/Form.h>

#include "System.h"
#include "Message.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmBaseCal.h"
#include "xtmCalDb.h"
#include "xtmCustBase.h"
#include "xtmDateSel.h"
#include "xtmDbTools.h"
#include "xtmDbMisc.h"
#include "xtmUpdate.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmNavCal.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record for summary. */
typedef struct {

  /* Modal dialog (normally not). */
  Boolean  appl_modal;

  /* Display calendar for a sepecific calendar database. */
  Boolean  has_cal_db;

  /* Id to use for update functions. */
  UINT32  update_id;

  /* This name of the calendar database. */
  char  cal_name[ XTM_GL_MAX_CAL_NAME + 1 ];

  /* The navigator window. */
  Widget  navCalW;

  /* Date for the calendar. */
  TIM_TIME_REF  cal_date;

  /* Calendar. */
  XTM_BC_HANDLE  cal_handle;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XTM_NC_ACTION_CB  actionCB;

} NAV_CAL_REC, *NAV_CAL_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmNavCal";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  calendarActionCB( XTM_BC_REASON        reason,
                    TIM_TIME_REF         time,
                    XmAnyCallbackStruct  *call_data,
                    void                 *user_data );

static void 
  calendarEntriesCB( XTM_BC_REASON  reason,
                     void           *user_data,
                     TIM_TIME_REF   start_date,
                     int            days,
                     Boolean        *entries );

static Widget
  createNavCalWindow( NAV_CAL_REC_REF  nav_cal_ref,
                      Widget           parent );

static void 
  destroyCB( Widget           widget,
             NAV_CAL_REC_REF  nav_cal_ref,
             XtPointer        call_data );

static void
  updateCalendar( NAV_CAL_REC_REF  nav_cal_ref );

static void
  updateCB( UINT32  flags,
            void    *user_data,
            void    *update_user_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_NC_HANDLE
  xtmNcInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                parent,
                   char                  *cal_name,
                   Boolean               appl_modal,
                   XTM_NC_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  Boolean                 ok;
  int                     index;
  NAV_CAL_REC_REF         nav_cal_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  custom_data_ref = appl_data_ref -> custom_data;

  /* Fetch calendar information. */
  if( cal_name != NULL ) {
    ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle,
                            cal_name,
                            &db_info, NULL );
    if( ! ok )
      return( False );

    /* Minium permissions are read. */
    if( ! flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_READ ) )
      return( NULL );
  }


  /* Create the private record? */
  nav_cal_ref = SysNew( NAV_CAL_REC );
  if( nav_cal_ref == NULL )
    return( NULL );

  nav_cal_ref -> appl_data_ref = appl_data_ref;
  nav_cal_ref -> cal_date      = 0;
  nav_cal_ref -> update_id     = 0;
  nav_cal_ref -> cal_handle    = NULL;
  nav_cal_ref -> navCalW       = NULL;
  nav_cal_ref -> appl_modal    = appl_modal;
  nav_cal_ref -> actionCB      = actionCB;
  nav_cal_ref -> user_data     = user_data;

  if( cal_name != NULL ) {
    strcpy( nav_cal_ref -> cal_name, cal_name );
    nav_cal_ref -> has_cal_db = True;
  } else {
    nav_cal_ref -> has_cal_db = False;
  }

  /* Create the navigate window. */
  nav_cal_ref -> navCalW = createNavCalWindow( nav_cal_ref, parent );

  if( nav_cal_ref -> navCalW == NULL ) {
    SysFree( nav_cal_ref );
    return( NULL );
  }


  return( (XTM_NC_HANDLE) nav_cal_ref );

} /* xtmNcInitialize */


/*----------------------------------------------------------------------*/

void
  xtmNcDestroy( XTM_NC_HANDLE  nav_cal_handle )
{

  /* Variables. */
  NAV_CAL_REC_REF  nav_cal_ref;


  /* Code. */

  if( nav_cal_handle == NULL )
    return;

  /* Our private data. */
  nav_cal_ref = (NAV_CAL_REC_REF) nav_cal_handle;


  /* Destroy the window. */
  XtDestroyWidget( nav_cal_ref -> navCalW );


  return;

} /* xtmNcDestroy */


/*----------------------------------------------------------------------*/

void 
  xtmNcDisplayNavCal( XTM_NC_HANDLE  nav_cal_handle,
                      TIM_TIME_REF   at_date )
{

  /* Variables. */
  UINT32           flags;
  NAV_CAL_REC_REF  nav_cal_ref;


  /* Code. */

  nav_cal_ref = (NAV_CAL_REC_REF) nav_cal_handle;


  /* When? */
  nav_cal_ref -> cal_date = TimMakeTime( TimIndexOfYear(  at_date ),
                                         TimIndexOfMonth( at_date ),
                                         1, 0, 0, 0 );

  /* Register for updates? */
  if( nav_cal_ref -> update_id == 0 ) {
    flags = (XTM_UP_MINUTE_TICK | 
             XTM_UP_CALENDAR | 
             XTM_UP_NAV_CALENDAR);

    nav_cal_ref -> update_id = xtmUpRegister( flags, updateCB,
                                              (void *) nav_cal_ref );
  }


  /* Update calendar time. */
  updateCB( (XTM_UP_MINUTE_TICK | XTM_UP_NAV_CALENDAR), 
            (void *) nav_cal_ref, NULL );


  /* Display the calendar. */
  XtManageChild( nav_cal_ref -> navCalW );


  /* Make sure the navigation window is visable. */
  XRaiseWindow( XtDisplay( nav_cal_ref -> navCalW ), 
                XtWindow(  nav_cal_ref -> navCalW ) );

  XtMapWidget( XtParent( nav_cal_ref -> navCalW ) );


  return;

} /* xtmNcDisplayNavCal */


/*----------------------------------------------------------------------*/

static Widget
  createNavCalWindow( NAV_CAL_REC_REF  nav_cal_ref,
                      Widget           parent )
{

  /* Variables. */
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  baseCalW;
  Widget                  navCalFd;
  Widget                  workFo;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = nav_cal_ref -> appl_data_ref -> custom_data;


  /* Create a separate window. */
  navCalFd = xitCreateFormDialog( parent, "NavCalFd", 0, 0, NULL, 0 );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_NAV_CAL_TITLE ) ); n++;
  XtSetValues( XtParent( navCalFd ), args, n );

  XtAddCallback( navCalFd,  XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) nav_cal_ref );

  if( nav_cal_ref -> appl_modal ) {
    n = 0;
    XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
    XtSetValues( navCalFd, args, n );
  }


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( navCalFd, "NavCalFdFo" );

  /* Create the base calendar. */
  if( nav_cal_ref -> cal_handle == NULL ) {

    XTM_BC_CAL_STYLE  style;

    if( custom_data_ref -> cal_is_fancy )
      style = XTM_BC_STYLE_NOTEBOOK;
    else
      style = XTM_BC_STYLE_PLAIN;

    nav_cal_ref -> cal_handle =
      xtmBcInitialize( nav_cal_ref -> appl_data_ref, workFo,
                       style, calendarActionCB, calendarEntriesCB,
                       (void *) nav_cal_ref );

    if( nav_cal_ref -> cal_handle == NULL )
      return( NULL );
  }


  /* We already have the basic calendar. */
  baseCalW = xtmBcGetWidget( nav_cal_ref -> cal_handle );

  /* Put the parts together. */
  xitAttachWidget( baseCalW,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( baseCalW, args, n );

  /* Manage the calendar. */
  xtmBcViewCalendar( nav_cal_ref -> cal_handle );


  /* Set the size of the window. */
  xitSetSizeFormDialog( navCalFd, True );


  return( navCalFd );

} /* createNavCalWindow */


/*----------------------------------------------------------------------*/

static void
  updateCalendar( NAV_CAL_REC_REF  nav_cal_ref )
{

  /* Code. */

  xtmBcDisplayMonth( nav_cal_ref -> cal_handle,
                     nav_cal_ref -> cal_date );


  return;  

} /* updateCalendar */


/*----------------------------------------------------------------------*/

static void 
  calendarActionCB( XTM_BC_REASON        reason,
                    TIM_TIME_REF         time,
                    XmAnyCallbackStruct  *call_data,
                    void                 *user_data )
{

  /* Variables. */                    
  NAV_CAL_REC_REF  nav_cal_ref;


  /* Code. */

  nav_cal_ref = (NAV_CAL_REC_REF) user_data;

  if( reason == XTM_BC_REASON_DATE_SELECTED ) {
    if( nav_cal_ref -> actionCB != NULL )
      (* nav_cal_ref -> actionCB)( XTM_NC_REASON_DATE_SELECT,
                                   (void *) time,
                                   nav_cal_ref -> user_data );

    return;
  } /* if */


  return;

} /* calendarActionCB */


/*----------------------------------------------------------------------*/

static void 
  calendarEntriesCB( XTM_BC_REASON  reason,
                     void           *user_data,
                     TIM_TIME_REF   start_date,
                     int            days,
                     Boolean        *entries )
{

  /* Variables. */                    
  NAV_CAL_REC_REF         nav_cal_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  nav_cal_ref = (NAV_CAL_REC_REF) user_data;

  custom_data_ref = nav_cal_ref -> appl_data_ref -> custom_data;

  /* If we don't have a calendar, just return. */
  if( ! nav_cal_ref -> has_cal_db )
    return;

  /* Fetch the entries defined. */
  xtmDmEntriesDefined( nav_cal_ref -> appl_data_ref,
                       nav_cal_ref -> cal_name,
                       start_date, days, True,
                       custom_data_ref -> show_stand_in_cal,
                       entries );


  return;

} /* calendarEntriesCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget           widget,
             NAV_CAL_REC_REF  nav_cal_ref,
             XtPointer        call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( nav_cal_ref -> actionCB != NULL )
    (* nav_cal_ref -> actionCB)( XTM_NC_REASON_DESTROY,
                                 NULL,
                                 nav_cal_ref -> user_data );

  /* De-register updates. */
  if( nav_cal_ref -> update_id != 0 )
    xtmUpRemove( nav_cal_ref -> update_id );

  nav_cal_ref -> update_id = 0;

  /* Free calendar. */
  if( nav_cal_ref -> cal_handle ) {
    xtmBcDestroy( nav_cal_ref -> cal_handle );
    nav_cal_ref -> cal_handle = NULL;
  }

  /* Release the user data. */
  SysFree( nav_cal_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void
  updateCB( UINT32  flags,
            void    *user_data,
            void    *update_user_data )
{

  /* Variables. */
  NAV_CAL_REC_REF  nav_cal_ref;
  TIM_TIME_REF     now;


  /* Code. */

  nav_cal_ref = (NAV_CAL_REC_REF) user_data;


  /* Update calendar? */
  if( flagIsSet( flags, (XTM_UP_CALENDAR | XTM_UP_NAV_CALENDAR) ) )
    updateCalendar( nav_cal_ref );


  /* Every minute tick. */
  if( flagIsSet( flags, XTM_UP_MINUTE_TICK ) ) {

    now = TimLocalTime( TimMakeTimeNow() );

    /* Update the calendar time. */
    xtmBcDisplayTime( nav_cal_ref -> cal_handle );

    /* Do we have another date? */
    if( flagIsSet( flags, XTM_UP_NEW_DAY ) )
      xtmUpDoUpdate( XTM_UP_CALENDAR, (void *) now );

  } /* if */


  return;

} /* updateCB */
