/*----------------------------------------------------------------------------
--
--  Module:           xtmMsgBase
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Basic XDiary message routines.
--
--  Filename:         xtmMsgBase.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-04-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmMsgBase.c, Version: 1.1, Date: 95/02/18 15:52:32";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <pwd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>

#include "System.h"
#include "DirList.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmCustBase.h"
#include "xtmDbTools.h"
#include "xtmFormat.h"
#include "xtmGroupDb.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmMsgBase.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#define max( a, b )  (a < b ? b : a )

/* Alloc memory size in bytes. */
#define SEGMENT_SIZE          1000


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmMsgBase";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static Boolean
  sendInternalMessage( XTM_GL_BASE_DATA_REF  appl_data_ref,
                       Widget                mainW,
                       char                  *user_ref,
                       char                  *directory,
                       TIM_TIME_REF          entry_date,
                       TIM_TIME_REF          entry_time,
                       int                   duration,
                       char                  *message,
                       char                  *text );

static Boolean
  sendMailMessage( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                mainW,
                   char                  *mail_address,
                   TIM_TIME_REF          entry_date,
                   TIM_TIME_REF          entry_time,
                   int                   duration,
                   char                  *message,
                   char                  *text );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

Boolean
  xtmMbSendMessage( XTM_GL_BASE_DATA_REF  appl_data_ref,
                    Widget                mainW,
                    int                   msg_to_count,
                    XmStringTable         msg_to,
                    Boolean               only_mail,
                    Boolean               mail_fail,
                    TIM_TIME_REF          entry_date,
                    TIM_TIME_REF          entry_time,
                    int                   duration,
                    char                  *message,
                    char                  *text )
{

  /* Type definitions. */
  typedef enum { MESSAGE, MAIL } MESSAGE_TYPE;
  typedef enum { NONE, CALENDAR, GROUP } RECIPIENT_TYPE;

  /* Variables. */
  Boolean         ok;
  int             alloc_size;
  int             index;
  char            recipient_buffer[ 250 ];
  char            *char_ref;
  char            *recipients;
  char            *recipient_ref;
  MESSAGE_TYPE    message_type;
  RECIPIENT_TYPE  recipient_type;
  TIM_TIME_REF    default_date;
  TIM_TIME_REF    default_time;
  XTM_GD_ENTRY    group_info;


  /* Code. */

  default_time = TimMakeTime( 1970, 1, 1, 12, 0, 0 );
  default_date = TimLocalTime( TimMakeTimeNow() );
  default_date = TimMakeTime(  TimIndexOfYear(  default_date ),
                               TimIndexOfMonth( default_date ),
                               TimIndexOfDay(   default_date ),
                               0, 0, 0 );

  if( entry_date <= 0 )
    entry_date = default_date;

  if( entry_time <= 0 )
    entry_time = default_time;


  /* Buffer for recipients. */
  recipients  = (char *) SysMalloc( SEGMENT_SIZE );
  alloc_size  = SEGMENT_SIZE;
  *recipients = '\0';


  /* Expand databases, groups and mail addresses. */
  index = 0;

  while( index < msg_to_count ) {

    char_ref = xitStringGetString( *(msg_to + index), CS );

    sscanf( char_ref, "%s", recipient_buffer );
    SysFree( char_ref );

    recipient_ref = recipient_buffer;


    /* Find the type of recipient. */
    recipient_type = CALENDAR;

    /* A group? */
    ok = xtmGdFetchNamedGroup( appl_data_ref -> custom_data -> group_db_handle,
                               recipient_ref, &group_info );

    if( ok ) {
      recipient_type = GROUP;

      if( group_info.members != NULL )
        recipient_ref = group_info.members;
      else
        recipient_type = NONE;
    }


    /* Save the recipient(s). */
    if( recipient_type != NONE ) {

      int  append_size;
      int  extend_size;

      append_size = strlen( recipient_ref ) + 5;

      /* Any space left? */
      if( strlen( recipients ) + append_size >= alloc_size ) {
        extend_size = max( append_size, SEGMENT_SIZE );

        alloc_size = alloc_size + extend_size + 5;
        recipients = (char *) SysRealloc( recipients, alloc_size );
      }

      strcat( recipients, recipient_ref );
      strcat( recipients, " " );

      if( recipient_type == GROUP )
        SysFree( recipient_ref );

    } /* if */

    index++;

  } /* while */


  /* Start to send messages/mails. */
  recipient_ref = recipients;

  do {

    int              char_read;
    char             location[ XTM_GL_MAX_DIR + 1 ];
    char             *directory_ref;
    char             *mail_address;
    XTM_CD_CAL_INFO  db_info;

    while( isspace( *recipient_ref ) )
      recipient_ref++;

    if( *recipient_ref == '\0' )
      break;

    char_read = strlen( recipient_ref );
    sscanf( recipient_ref, "%s%n", recipient_buffer, &char_read );
    recipient_ref = recipient_ref + char_read;


    /* Where do we send the message? */
    mail_address  = recipient_buffer;
    directory_ref = NULL;

    /* Fetch user diary DB directory. */
    ok = xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle, 
                            recipient_buffer,
                            &db_info, NULL );

    /* A 'real' calendar. */
    if( ok ) {
      directory_ref = db_info.directory;

      if( strlen( db_info.mail_address ) > 0 )
          mail_address = db_info.mail_address;

    /* Try to locate the calendar. */
    } else {
      ok = xtmLcLocateCalendar( 
             mainW, recipient_buffer,
             appl_data_ref -> custom_data -> find_location_script,
             location, sizeof( location ) );

      if( ok )
        directory_ref = location;

    } /* if */


    /* What kind of message do we send? */
    if( only_mail )
      message_type = MAIL;
    else
      message_type = MESSAGE;


    /* Try internal message. */
    if( message_type == MESSAGE ) {

      ok = sendInternalMessage( appl_data_ref, mainW,
                                recipient_buffer, directory_ref,
                                entry_date, entry_time, duration,
                                message, text );

      /* If we failed, try normal mail message? */
      if( ! ok && mail_fail )
        message_type = MAIL;

    } /* if */
    

    /* Try mail message. */
    if( message_type == MAIL )
      ok = sendMailMessage( appl_data_ref, mainW,
                            mail_address,
                            entry_date, entry_time, duration,
                            message, text );

  } while( True );


  /* Free occupied memory. */
  SysFree( recipients );


  return( True );

} /* xtmMbSendMessage */


/*----------------------------------------------------------------------*/

static Boolean
  sendInternalMessage( XTM_GL_BASE_DATA_REF  appl_data_ref,
                       Widget                mainW,
                       char                  *user_ref,
                       char                  *directory,
                       TIM_TIME_REF          entry_date,
                       TIM_TIME_REF          entry_time,
                       int                   duration,
                       char                  *message,
                       char                  *text )
{

  /* Variables. */
  UINT32             operations;
  XTM_DB_MESSAGE_DEF msg;
  XTM_DB_STATUS      db_status;
  struct passwd      *password_ref;


  /* Code. */

  if( directory == NULL || strlen( directory ) == 0 )
    return( False );

  /* Fetch the database permissions. */
  xtmDbCheckDbOperations( directory, True, &operations );
  if( ! flagIsSet( operations, XTM_DB_FLAG_MODE_MSG ) )
    return( False );


  /* Construct the message record. */
  msg.version        = XTM_DB_MESSAGE_VERSION;
  msg.flags          = 0;
  msg.date           = (UINT32) entry_date;
  msg.time           = (UINT32) entry_time;
  msg.duration       = (UINT16) duration;
  msg.message_length = (UINT16) strlen( message );
  msg.text_length    = (UINT16) strlen( text );


  /* Name of sender. */
  password_ref = getpwuid( getuid() );
  if( password_ref != NULL )
    strncpy( msg.from, password_ref -> pw_name, 20 );
  else
    strcpy( msg.from, "---" );
  msg.from[ 20 ] = '\0';


  /* Send the message. */
  db_status = xtmDbInsertMessage( directory, &msg, message, text );
  if( db_status != XTM_DB_OK )
    return( False );


  return( True);

} /* sendInternalMessage */


/*----------------------------------------------------------------------*/

static Boolean
  sendMailMessage( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                mainW,
                   char                  *mail_address,
                   TIM_TIME_REF          entry_date,
                   TIM_TIME_REF          entry_time,
                   int                   duration,
                   char                  *message,
                   char                  *text )
{

  /* Variables. */
  char  buffer1[ 41 ];
  char  buffer2[ 41 ];
  char  command[ 100 ];
  char  duration_buffer[ 20 ];
  char  time_buffer[ 40 ];
  char  *char_ref;
  FILE  *file_ref;


  /* Code. */

  /* The subject is the first 40 characters in the first line. */
  strncpy( buffer1, message, 40 );
  buffer1[ 40 ] = '\0';

  char_ref = strchr( buffer1, '\n' );
  if( char_ref != NULL )
    *char_ref = '\0';

  if( strlen( buffer1 ) == 0 )
    strcpy( buffer1, "---" );

  /* Construct command. */
  sprintf( command, "mail -s '%s' %s", buffer1, mail_address );


  /* Make the entry time field. */
  xtmFoFormatDate( entry_date, buffer1, sizeof( buffer1 ) );
  xtmFoFormatTime( entry_time, buffer2, sizeof( buffer2 ) );

  sprintf( time_buffer, "%s %s", buffer1, buffer2 );


  /* Make the duration field. */
  sprintf( duration_buffer, "%2d:%02d", duration / 60, duration % 60 );


  /* Open pipe to send mail. */
  file_ref = (FILE *) popen( command, "w" );
  if( file_ref == NULL )
    return( False );


  /* Send the mail. */
  if( entry_date == 0 )
    fprintf( file_ref, "%s\n\n%s", message, text );
  else
    fprintf( file_ref, msgGetText( MXDI_ENTRY_MAIL_MESSAGE ),
             message, time_buffer, duration_buffer, text );


  /* That's it. */
#ifdef XD_HAS_NO_WAITPID
  fclose( file_ref );
#else
  pclose( file_ref );
#endif

  return( True );

} /* sendMailMessage */




