/*----------------------------------------------------------------------------
--
--  Module:           xtmMsgAnswer
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Answer an XDiary message.
--
--  Filename:         xtmMsgAnswer.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-04-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmMsgAnswer.c, Version: 1.1, Date: 95/02/18 15:52:31";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>

#include "System.h"
#include "DirList.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmCustBase.h"
#include "xtmDbTools.h"
#include "xtmFormat.h"
#include "xtmMsgBase.h"
#include "xtmTools.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmMsgAnswer.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the message answer window. */
#define answTx            dataLocalW[  0 ]
#define flagsRc           dataLocalW[  1 ]
#define mailFailTb        dataLocalW[  2 ]
#define msgLa             dataLocalW[  3 ]
#define onlyMailTb        dataLocalW[  4 ]
#define toLa              dataLocalW[  5 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/* Record describing one instance of the message select window. */
typedef struct {

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Answer this message id. */
  UINT32  msg_id;

  /* Message recipient. */
  char  recipient[ 51 ];

  /* Answer window. */
  Widget  answerW;

  /* Parent window. */
  Widget  parentW;

} ANSWER_REC, *ANSWER_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmMsgAnswer";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  closeCB( Widget          widget,
           ANSWER_REC_REF  answer_ref,
           XtPointer       call_data );

static Widget 
  createAnswerWindow( Widget          parent,
                      ANSWER_REC_REF  answer_ref );

static void 
  destroyCB( Widget          widget,
             ANSWER_REC_REF  answer_ref,
             XtPointer       call_data );

static void 
  sendCB( Widget          widget,
          ANSWER_REC_REF  answer_ref,
          XtPointer       call_data );

static void
  setAnswerWindowValues( ANSWER_REC_REF  answer_ref,
                         UINT32          msg_id );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_MA_HANDLE
  xtmMaInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                parent )
{

  /* Variables. */
  ANSWER_REC_REF  answer_ref;


  /* Code. */

  /* Create and initialize our private data. */
  answer_ref = SysNew( ANSWER_REC );
  if( answer_ref == NULL )
    return( NULL );

  answer_ref -> appl_data_ref  = appl_data_ref;
  answer_ref -> msg_id         = 0;
  answer_ref -> answerW        = NULL;
  answer_ref -> parentW        = parent;
  answer_ref -> recipient[ 0 ] = '\0';


  /* Create the message answer window. */
  answer_ref -> answerW = createAnswerWindow( parent, answer_ref );
  if( answer_ref -> answerW == NULL ) {
    SysFree( answer_ref );

    return( NULL );
  }


  return( (XTM_MA_HANDLE) answer_ref );

} /* xtmMaInitialize */


/*----------------------------------------------------------------------*/

void
  xtmMaDestroy( XTM_MA_HANDLE  answer_handle )
{

  /* Variables. */
  ANSWER_REC_REF  answer_ref;


  /* Code. */

  if( answer_handle == NULL )
    return;

  /* Our private data. */
  answer_ref = (ANSWER_REC_REF) answer_handle;


  /* Get rid of all windows. */
  if( answer_ref -> answerW != NULL )
    XtDestroyWidget( answer_ref -> answerW );


  return;

} /* xtmMaDestroy */


/*---------------------------------------------------------------------------*/

void
  xtmMaAnswerMessage( XTM_MA_HANDLE  answer_handle,
                      UINT32         msg_id )
{

  /* Variables. */
  ANSWER_REC_REF  answer_ref;


  /* Code. */

  /* Our private data. */
  answer_ref = (ANSWER_REC_REF) answer_handle;

  answer_ref -> msg_id = msg_id;


  /* Make sure the message window is visable. */
  XtManageChild( answer_ref -> answerW );


  /* Set the values in the message answer window. */
  setAnswerWindowValues( answer_ref, msg_id );


  return;

} /* xtmMrAnswerMessage */


/*----------------------------------------------------------------------*/

static Widget 
  createAnswerWindow( Widget          parent,
                      ANSWER_REC_REF  answer_ref )
{

  /* Variables. */
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dataLocalW[ 6 ];
  Widget    msgAnswFd;
  Widget    workFo;

  static XIT_TEXT_STRUCT text_buffer_def[] = {
    { "AnswTx", NULL, 5, True },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   sendCB,  NULL },
    { NULL, NULL,    NULL },
    { "",   closeCB, NULL },
  };


  /* Code. */

  action_buttons[ 0 ].label = msgGetText( MXDI_SEND_BUTTON );
  action_buttons[ 0 ].data  = answer_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_CLOSE_BUTTON );
  action_buttons[ 2 ].data  = answer_ref;


  /* Create a form dialog with buttons. */
  msgAnswFd = xitCreateFormDialog( parent, "MsgAnswFd",
                                   1, 0,
                                   action_buttons,
                                   XtNumber( action_buttons ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_ANSWER_MSG_TITLE ) ); n++;
  XtSetValues( XtParent( msgAnswFd ), args, n );

  XtAddCallback( msgAnswFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) answer_ref );


  /* Use this form as working form. */
  workFo = XtNameToWidget( msgAnswFd, "MsgAnswFdFo" );


  /* To user. */
  toLa = xitCreateLabel( workFo, "ToLa", " ", -1 );


  /* Message text. */
  msgLa = xitCreateLabel( workFo, "MsgLa", 
                          msgGetText( MXDI_REMARK_LABEL ), -1 );

  answTx = xitCreateTextScrolled( workFo, &text_buffer_def[ 0 ] );

  XtOverrideTranslations( answTx, xtm_di_newline_trans );


  /* Create the flags form. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  flagsRc = XmCreateRowColumn( workFo, "FlagsRc", args, n );

  onlyMailTb = xitCreateToggleButton( 
                 flagsRc, "OnlyMailTb", 
                 msgGetText( MXDI_ONLY_MAIL_LABEL ), False );

  mailFailTb = xitCreateToggleButton( 
                 flagsRc, "MailFailTb", 
                 msgGetText( MXDI_MAIL_IF_FAIL_LABEL ), True );


  /* Place the main elements together. */
  xitAttachWidget( toLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( msgLa,
                   XmATTACH_WIDGET, toLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( answTx ),
                   XmATTACH_WIDGET, msgLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,  XmATTACH_NONE, NULL );
  xitAttachWidget( flagsRc,
                   XmATTACH_WIDGET, XtParent( answTx ),
                   XmATTACH_FORM,   NULL,
                   XmATTACH_NONE,   NULL, XmATTACH_NONE, NULL );

  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( toLa,               args, n );
  XtSetValues( msgLa,              args, n );
  XtSetValues( XtParent( answTx ), args, n );
  XtSetValues( flagsRc,            args, n );

  /* Manage all the children. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* Set the initial sizes. */
  xitSetSizeFormDialog( msgAnswFd, True );


  /* Make the final attachments. */
  xitAttachWidget( flagsRc,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_FORM, NULL );
  xitAttachWidget( XtParent( answTx ),
                   XmATTACH_WIDGET, msgLa, XmATTACH_FORM,   NULL,
                   XmATTACH_FORM,   NULL,  XmATTACH_WIDGET, flagsRc );


  return( msgAnswFd );

} /* createAnswerWindow */


/*----------------------------------------------------------------------*/

static void
  setAnswerWindowValues( ANSWER_REC_REF  answer_ref,
                         UINT32          msg_id )
{

  /* Variables. */
  char                    *message;
  char                    *text;
  char                    buffer[ 100 ];
  Widget                  mainW;
  Widget                  tempW;
  XTM_DB_MESSAGE_DEF      msg_info;
  XTM_DB_STATUS           db_status;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         default_db;


  /* Code. */

  appl_data_ref   = answer_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( answer_ref -> answerW, "MsgAnswFdFo" );


  /* Erase any old message. */
  tempW = XtNameToWidget( mainW, "AnswTxSW.AnswTx" );
  XmTextSetString( tempW, "" );


  strcpy( answer_ref -> recipient, "" );


  /* All messages are saved in the default database. */
  (void) xtmCdFetchDefaultDb( custom_data_ref -> cal_db_handle,
                              &default_db, NULL );


  /* Fetch the message. */
  db_status = xtmDbFetchMessage( default_db.directory, 
                                 msg_id,
                                 &msg_info, &message, &text );
  if( db_status != XTM_DB_OK )
    return;

  strcpy( answer_ref -> recipient, msg_info.from );


  /* Set name of recipient. */
  sprintf( buffer, "%s %s", 
           msgGetText( MXDI_TO_USER_LABEL ), answer_ref -> recipient );

  tempW = XtNameToWidget( mainW, "ToLa" );
  xitStringSetLabel( tempW, buffer );


  /* Free allocated data. */
  if( message != NULL )
    SysFree( message );

  if( text != NULL )
    SysFree( text );


  return;

} /* setAnswerWindowValues */


/*----------------------------------------------------------------------*/

static void 
  closeCB( Widget          widget,
           ANSWER_REC_REF  answer_ref,
           XtPointer       call_data )
{

  /* Code. */

  XtUnmanageChild( answer_ref -> answerW );


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget          widget,
             ANSWER_REC_REF  answer_ref,
             XtPointer       call_data )
{

  /* Code. */

  /* Release the user data. */
  SysFree( answer_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  sendCB( Widget          widget,
          ANSWER_REC_REF  answer_ref,
          XtPointer       call_data )
{

  /* Variables. */
  Boolean   mail_fail;
  Boolean   ok;
  Boolean   only_mail;
  char      *message;
  Widget    mainW;
  Widget    tempW;
  XmString  recipients[ 1 ];


  /* Code. */

  mainW = XtNameToWidget( answer_ref -> answerW, "MsgAnswFdFo" );


  /* Fetch the recipient (including mail address). */
  recipients[ 0 ] = XmStringCreate( answer_ref -> recipient, CS );


  /* Flags. */
  tempW = XtNameToWidget( mainW, "FlagsRc.OnlyMailTb" );
  only_mail = XmToggleButtonGetState( tempW );

  tempW = XtNameToWidget( mainW, "FlagsRc.MailFailTb" );
  mail_fail = XmToggleButtonGetState( tempW );


  /* Message. */
  tempW = XtNameToWidget( mainW, "AnswTxSW.AnswTx" );
  message = xitStringGetText( tempW );


  /* Send the message. */
  ok = xtmMbSendMessage( answer_ref -> appl_data_ref, 
                         answer_ref -> answerW,
                         1, recipients,
                         only_mail, mail_fail,
                         0, 0, 0, 
                         message, "---" );

  /* Free occupied memory. */
  XmStringFree( recipients[ 0 ] );
  SysFree( message );


  /* Close the window. */
  closeCB( widget, answer_ref, call_data );


  /* Tell we are done. */
  (void) xitCreateInformationDialog( 
           answer_ref -> parentW, "InformationDialog",
           msgGetText( MXDI_INFO_MESSAGE_LABEL ),
           msgGetText( MXDI_MESSAGE_SENT ),
           NULL, NULL );


  return;

} /* sendCB */
