/*----------------------------------------------------------------------------
--
--  Module:           xtmEditEntry
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Create and edit a singe time entry.
--
--  Filename:         xtmEditEntry.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1990-12-18
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmEditEntry.c, Version: 1.1, Date: 95/02/18 15:52:15";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/signal.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/BulletinB.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/RowColumn.h>
#include <Xm/ScrolledW.h>
#include <Xm/SeparatoG.h>
#include <Xm/Text.h>
#include <Xm/ToggleB.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"
#include "SigHandler.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmCustBase.h"
#include "xtmDbMisc.h"
#include "xtmDbTools.h"
#include "xtmEditAlarm.h"
#include "xtmEditPref.h"
#include "xtmEditSkip.h"
#include "xtmEditStand.h"
#include "xtmFields.h"
#include "xtmFormat.h"
#include "xtmHelp.h"
#include "xtmIcons.h"
#include "xtmSchedMain.h"
#include "xtmTools.h"
#include "xtmUpdate.h"
#include "xitColorSel.h"
#include "xitError.h"
#include "xitFieldSel.h"
#include "xitTools.h"
#include "xitTune.h"
#include "XmUbTimeB.h"
#include "xtmEditEntry.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Editors in the cache? */
#define  MAX_CACHE_ENTRIES   3

/* File protection. */
#define  MODE_600   (S_IRUSR | S_IWUSR)

/* Action on entry/repeated entry part? */
#define  SET_ALL_ENTRY       (1<<0)
#define  SET_ENTRY           (1<<1)
#define  SET_REPEATED_ENTRY  (1<<2)


/* Local widgets in the entry edit window. */
#define actionRc            dataLocalW[  0 ]
#define alarmPb             dataLocalW[  1 ]
#define dataEntFo           dataLocalW[  2 ]
#define databaseFs          dataLocalW[  3 ]
#define databaseLa          dataLocalW[  4 ]
#define dateEntTb           dataLocalW[  5 ]
#define dateEntLa           dataLocalW[  6 ]
#define dayViewPb           dataLocalW[  7 ]
#define dummy1Pb            dataLocalW[  8 ]
#define durEntFs            dataLocalW[  9 ]
#define durEntLa            dataLocalW[ 10 ]
#define editorPb            dataLocalW[ 11 ]
#define entryDateLa         dataLocalW[ 12 ]
#define entryTagFs          dataLocalW[ 13 ]
#define entryTagLa          dataLocalW[ 14 ]
#define menuBr              dataLocalW[ 15 ]
#define miscFo              dataLocalW[ 16 ]
#define miscRc              dataLocalW[ 17 ]
#define prefPb              dataLocalW[ 18 ]
#define selAlarmBu          dataLocalW[ 19 ]
#define selFo               dataLocalW[ 20 ]
#define selFr               dataLocalW[ 21 ]
#define selImpBu            dataLocalW[ 22 ]
#define selNoteDoneBu       dataLocalW[ 23 ]
#define selPrivateBu        dataLocalW[ 24 ]
#define standPb             dataLocalW[ 25 ]
#define startEntTb          dataLocalW[ 26 ]
#define startEntLa          dataLocalW[ 27 ]
#define textEntLa           dataLocalW[ 28 ]
#define textEntTx           dataLocalW[ 29 ]
#define timeEntFr           dataLocalW[ 30 ]
#define timeEntRc           dataLocalW[ 31 ]
#define typeAppTb           dataLocalW[ 32 ]
#define typeNoteTb          dataLocalW[ 33 ]
#define typeRb              dataLocalW[ 34 ]
#define typeStandAppTb      dataLocalW[ 35 ]
#define typeStandNoteTb     dataLocalW[ 36 ]
#define typeStickyNoteTb    dataLocalW[ 37 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Times for an entry. */
typedef struct {
  TIM_TIME_REF  date;
  TIM_TIME_REF  start_time;
  TIM_TIME_REF  end_time;
  int           duration;
} ENTRY_TIMES_REC, *ENTRY_TIMES_REF;

/* Record describing one instance of the appointment editor. */
typedef struct {

  /* New entry to edit? */
  Boolean  new_entry;

  /* Did we see the standing window? */
  Boolean  standing_seen;

  /* Default duration to use. */
  int  default_duration;

  /* Limits used as suggestions. */
  int  earliest_hour;
  int  latest_hour;
  int  min_delta;

  /* Process id for the editor. */
  int  editor_pid;

  /* Signal id to use. */
  int  signal_id;

  /* Temp file name. */
  char  *temp_file;

  /* Editor window. */
  Widget  editorW;

  /* File selection window. */
  Widget  fileSelectW;

  /* Default date and time to use. */
  TIM_TIME_REF  default_date;
  TIM_TIME_REF  default_time;

  /* Entry being edited. */
  XTM_DB_ALL_ENTRY_REF  edit_entry_ref;

  /* The type of entry we are editing just now. */
  XTM_DB_ALL_ENTRY_TYPES  entry_type;

  /* The alarm window. */
  XTM_EA_HANDLE  alarm_handle;

  /* The preferences window. */
  XTM_EP_HANDLE  pref_handle;

  /* The standing window. */
  XTM_ES_HANDLE  stand_handle;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Callback and user data to inform our creator of specific actions. */
  XTM_ED_ACTION_CB  actionCB;
  void              *user_data;

} EDITOR_REC, *EDITOR_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmEditEntry";

/* IDs for the help windows. */
static char  *edit_window_id = "EntryEdit";

/* Cache entries. */
static Boolean        init_cache = True;
static EDITOR_REC_REF  cache_entries[ MAX_CACHE_ENTRIES ];


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  alarmCB( Widget          widget,
           EDITOR_REC_REF  editor_ref,
           XtPointer       call_data );

static void 
  alarmActionCB( XTM_EA_REASON          reason,
                 void                   *user_data,
                 XTM_EA_GET_ALARM_DATA  *values );

static void 
  applyCB( Widget          widget,
           EDITOR_REC_REF  editor_ref,
           XtPointer       call_data );

static void 
  cancelCB( Widget          widget,
            EDITOR_REC_REF  editor_ref,
            XtPointer       call_data );

static Widget 
  createEditWindow( Widget          parent,
                    EDITOR_REC_REF  editor_ref );

static void 
  dayViewCB( Widget          widget,
             EDITOR_REC_REF  editor_ref,
             XtPointer       call_data );

static void 
  dbChangedCB( Widget          widget,
               EDITOR_REC_REF  editor_ref,
               XtPointer       call_data );

static void 
  destroyCB( Widget          widget,
             EDITOR_REC_REF  editor_ref,
             XtPointer       call_data );

static void 
  editorCB( Widget          widget,
            EDITOR_REC_REF  editor_ref,
            XtPointer       call_data );

static void 
  entryTypeCB( Widget                        widget,
               EDITOR_REC_REF                editor_ref,
               XmToggleButtonCallbackStruct  *call_data );

static Boolean
  fetchEntryTimes( EDITOR_REC_REF   editor_ref,
                   Widget           parentW,
                   ENTRY_TIMES_REF  entry_times );

static Boolean
  fetchEntryValues( EDITOR_REC_REF        editor_ref,
                    XTM_DB_ALL_ENTRY_REF  entry_ref,
                    char                  **entry_text );

static void 
  helpCB( Widget                     widget,
          EDITOR_REC_REF             editor_ref,
          XmRowColumnCallbackStruct  *call_data );

static void 
  includeFileCB( Widget          widget,
                 EDITOR_REC_REF  editor_ref,
                 XtPointer       call_data );

static void 
  includeFileActionCB( XIT_FILE_SEL_REASON  reason,
                       char                 *file_name,
                       void                 *user_data );

static void 
  insertEditedFile( EDITOR_REC_REF  editor_ref );

static void 
  okCB( Widget          widget,
        EDITOR_REC_REF  editor_ref,
        XtPointer       call_data );

static void 
  preferencesCB( Widget          widget,
                 EDITOR_REC_REF  editor_ref,
                 XtPointer       call_data );

static void 
  preferencesActionCB( XTM_EP_REASON         reason,
                       void                  *user_data,
                       XTM_EP_GET_PREF_DATA  *values );

static Boolean
  readEntryToEdit( EDITOR_REC_REF  editor_ref,
                   char            *db_name,
                   UINT32          entry_id,
                   Widget          parent );

static void
  removeEditFile( EDITOR_REC_REF  editor_ref );

static Boolean
  saveEntry( EDITOR_REC_REF  editor_ref );


static void
  setEntryValues( EDITOR_REC_REF        editor_ref,
                  XTM_DB_ALL_ENTRY_REF  entry_ref );

static XTM_DB_STATUS
  setEntryDefault( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   XTM_DB_ALL_ENTRY_REF  entry_ref,
                   UINT32                set_parts,
                   TIM_TIME_REF          entry_date,
                   TIM_TIME_REF          entry_time,
                   int                   entry_duration );

static void 
  standingCB( Widget          widget,
              EDITOR_REC_REF  editor_ref,
              XtPointer       call_data );

static void 
  standingActionCB( XTM_ES_REASON          reason,
                    void                   *user_data,
                    XTM_ES_GET_STAND_DATA  *values );

static void
  timeFieldChangedCB( Widget          widget,
                      EDITOR_REC_REF  editor_ref,
                      XtPointer       call_data );

static void 
  updateCanDo( EDITOR_REC_REF editor_ref );

static void 
  userEditorDoneCB( int   signal,
                    void  *user_data );

static Boolean
  writeEditedEntry( XTM_GL_BASE_DATA_REF  appl_data_ref,
                    Widget                parent,
                    XTM_DB_ALL_ENTRY_REF  old_entry_ref,
                    XTM_DB_ALL_ENTRY_REF  new_entry_ref );

static void 
  writeEntryDate( EDITOR_REC_REF editor_ref );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_ED_HANDLE
  xtmEdInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                parent,
                   int                   default_duration,
                   int                   earliest_hour,
                   int                   latest_hour,
                   int                   min_delta,
                   XTM_ED_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  int             index;
  EDITOR_REC_REF  editor_ref;


  /* Code. */

  /* Initialize the cache? */
  if( init_cache ) {
    for( index = 0; index < MAX_CACHE_ENTRIES; index++ )
      cache_entries[ index ] = NULL;

    init_cache = False;
  }


  /* Anything in the cache? */
  editor_ref = NULL;

  for( index = 0; index < MAX_CACHE_ENTRIES; index++ ) {
    if( cache_entries[ index ] != NULL ) {
      editor_ref = cache_entries[ index ];
      cache_entries[ index ] = NULL;

      break;
    }
  }


  /* Create the editor record? */
  if( editor_ref == NULL ) {

    editor_ref = SysNew( EDITOR_REC );
    if( editor_ref == NULL )
      return( NULL );

    editor_ref -> editorW = NULL;

  } /* if */


  editor_ref -> appl_data_ref      = appl_data_ref;
  editor_ref -> edit_entry_ref     = NULL;
  editor_ref -> default_duration   = default_duration;
  editor_ref -> earliest_hour      = earliest_hour;
  editor_ref -> latest_hour        = latest_hour;
  editor_ref -> min_delta          = min_delta;
  editor_ref -> actionCB           = actionCB;
  editor_ref -> user_data          = user_data;
  editor_ref -> temp_file          = NULL;
  editor_ref -> fileSelectW        = NULL;
  editor_ref -> alarm_handle       = NULL;
  editor_ref -> pref_handle        = NULL;
  editor_ref -> stand_handle       = NULL;
  editor_ref -> editor_pid         = 0;
  editor_ref -> signal_id          = 0;


  /* Create the base editor window. */
  if( editor_ref -> editorW == NULL ) {

    editor_ref -> editorW = createEditWindow( parent, editor_ref );
    if( editor_ref -> editorW == NULL ) {
      SysFree( editor_ref );

      return( NULL );
    }

  } /* if */


  return( (XTM_ED_HANDLE) editor_ref );

} /* xtmEdInitialize */


/*----------------------------------------------------------------------*/

void
  xtmEdEmptyCache()
{

  /* Variables. */
  int  index;


  /* Code. */

  if( ! init_cache )
    return;

  for( index = 0; index < MAX_CACHE_ENTRIES; index++ ) {
    if( cache_entries[ index ] != NULL )
      XtDestroyWidget( cache_entries[ index ] -> editorW );

    cache_entries[ index ] = NULL;
  }


  return;

} /* xtmEdEmptyCache */


/*----------------------------------------------------------------------*/

void
  xtmEdDestroy( XTM_ED_HANDLE  editor_handle )
{

  /* Variables. */
  EDITOR_REC_REF  editor_ref;


  /* Code. */

  if( editor_handle == NULL )
    return;

  /* Our private data. */
  editor_ref = (EDITOR_REC_REF) editor_handle;

  /* Get rid of all windows. */
  if( editor_ref -> editorW != NULL )
    cancelCB( NULL, editor_ref, NULL );


  return;

} /* xtmEdDestroy */


/*----------------------------------------------------------------------*/

void
  xtmEdEditEntry( XTM_ED_HANDLE  editor_handle,
                  char           *db_name,
                  UINT32         entry_id,
                  TIM_TIME_REF   use_date,
                  TIM_TIME_REF   use_time )
{

  /* Variables. */
  Boolean         ok;
  Widget          mainW;
  Widget          tempW;
  EDITOR_REC_REF  editor_ref;


  /* Code. */

  /* Our private data. */
  editor_ref = (EDITOR_REC_REF) editor_handle;

  editor_ref -> default_date = use_date;
  editor_ref -> default_time = use_time;

  mainW = XtNameToWidget( editor_ref -> editorW, 
                          "ApEditorTlBase.ApEditorTlFo" );


  /* Read the entry. */
  ok = readEntryToEdit( editor_ref, 
                        db_name, entry_id, editor_ref -> editorW );


  /* Make sure the editor window is visable. */
  XtPopup( editor_ref -> editorW, XtGrabNone );

  XRaiseWindow( XtDisplay( editor_ref -> editorW ), 
                XtWindow(  editor_ref -> editorW ) );

  XtMapWidget( editor_ref -> editorW );


  /* Set values for our windows. */
  setEntryValues( editor_ref, editor_ref -> edit_entry_ref );


  /* Start at the text field. */
  tempW = XtNameToWidget( mainW, "TextEntTxSW.TextEntTx" );
  xitGrabFocus( tempW );


  return;

} /* xtmEdEditEntry */


/*----------------------------------------------------------------------*/

void
  xtmEdSetValues( XTM_ED_HANDLE  editor_handle,
                  UINT32         flags,
                  TIM_TIME_REF   use_date,
                  TIM_TIME_REF   use_time,
                  UINT32         duration,
                  char           *text )
{

  /* Variables. */
  char                  buffer[ 50 ];
  Widget                baseW;
  Widget                mainW;
  Widget                tempW;
  EDITOR_REC_REF        editor_ref;
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  /* Our private data. */
  editor_ref    = (EDITOR_REC_REF) editor_handle;
  appl_data_ref = editor_ref -> appl_data_ref;

  mainW = XtNameToWidget( editor_ref -> editorW, 
                          "ApEditorTlBase.ApEditorTlFo" );


  /* Entry is a note. */
  if( flagIsSet( flags, XTM_ED_SET_IS_NOTE ) ) {
    baseW = XtNameToWidget( mainW, "TimeEntFr.DataEntFo.TypeRb" );
    tempW = XtNameToWidget( baseW, "TypeNoteTb" );

    XmToggleButtonSetState( tempW, True, True );

    baseW = XtNameToWidget( mainW, "SelFr.SelFo" );
    tempW = XtNameToWidget( baseW, "SelAlarmBu" );

    XmToggleButtonSetState( tempW, False, False );
  }


  /* Date for the entry. */
  if( flagIsSet( flags, XTM_ED_SET_DATE ) ) {
    baseW = XtNameToWidget( mainW, "TimeEntFr.DataEntFo.TimeEntRc" );
    tempW = XtNameToWidget( baseW, "DateEntTb" );

    XmUbTimeBoxSetStartDate( tempW, use_date );

    if( editor_ref -> edit_entry_ref != NULL )
      editor_ref -> edit_entry_ref -> stand_entry.from = use_date;
  }


  /* Entry start. */
  if( flagIsSet( flags, XTM_ED_SET_TIME ) ) {
    baseW = XtNameToWidget( mainW, "TimeEntFr.DataEntFo.TimeEntRc" );
    tempW = XtNameToWidget( baseW, "StartEntTb" );

    XmUbTimeBoxSetStartTime( tempW, use_time );
  }


  /* Duration. */
  if( flagIsSet( flags, XTM_ED_SET_DURATION ) ) {
    buffer[ 0 ] = '\0';

    if( duration > 0 )
      sprintf( buffer, "%d:%02d", duration / 60, duration % 60 );

    baseW = XtNameToWidget( mainW, "TimeEntFr.DataEntFo.TimeEntRc" );
    tempW = XtNameToWidget( baseW, "DurEntFs" );

    xitFieldSelectSetCurrent( tempW, buffer, False );
  }


  /* The text entry. */
  if( flagIsSet( flags, XTM_ED_SET_TEXT ) ) {
    tempW = XtNameToWidget( mainW, "TextEntTxSW.TextEntTx" );

    if( text == NULL )
      XmTextSetString( tempW, "" );
    else
      XmTextSetString( tempW, text );
  }


  return;

} /* xtmEdSetValues */


/*----------------------------------------------------------------------*/

static Widget 
  createEditWindow( Widget          parent,
                    EDITOR_REC_REF  editor_ref )
{

  /* Variables. */
  int                     index;
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  dataLocalW[ 38 ];
  Widget                  editorTl;
  Widget                  menuCasc[ 2 ];
  Widget                  menuHelpBu[ 6 ];
  Widget                  menuFileBu[ 3 ];
  Widget                  pulldownMenu[ 2 ];
  Widget                  tempW;
  Widget                  workFo;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  static char  def_durations[] = 
    "0:10 0:15 0:30 0:45 1:00 1:30 2:00 4:00 6:00";

  static char  *pull_downs[] = { "pdown1", "pdown2" };

  static XIT_PUSH_STRUCT action_button_def[] = {
    { "AlarmPb",   "", "", True, NULL },
    { "PrefPb",    "", "", True, NULL },
    { "StandPb",   "", "", True, NULL },
    { "EditorPb",  "", "", True, NULL },
    { "DummyPb",   "", "", True, NULL },
    { "DayViewPb", "", "", True, NULL },
  };

  static XIT_TEXT_STRUCT text_buffer_def[] = {
    { "TextEntTx", NULL, 5, True },
  };

  static XIT_CASCADE_STRUCT menupane[] = {
    { "", "", "FilePane" },
    { "", "", "HelpPane" },
  };

  static XIT_MENU_BUTTON_STRUCT file_casc[] = {
    { "",      "",  NULL, "IncludeBu", True,  False, False },
    { XIT_SEP, " ", NULL, "",          False, False, False },
    { "",      "",  NULL, "CloseBu",   True,  False, False },
  };

  static XIT_MENU_BUTTON_STRUCT help_casc[] = {
    { "", "", NULL, "ContextBu", True, False, False },
    { "", "", NULL, "WindowsBu", True, False, False },
    { "", "", NULL, "KeysBu",    True, False, False },
    { "", "", NULL, "IndexBu",   True, False, False },
    { "", "", NULL, "HelpBu",    True, False, False },
    { "", "", NULL, "VersionBu", True, False, False },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   okCB,     NULL },
    { NULL, NULL,     NULL },
    { "",   applyCB,  NULL },
    { NULL, NULL,     NULL },
    { "",   cancelCB, NULL },
  };


  /* Code. */

  appl_data_ref   = editor_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  /* Text items. */
  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = editor_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_APPLY_BUTTON );
  action_buttons[ 2 ].data  = editor_ref;
  action_buttons[ 4 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 4 ].data  = editor_ref;


  menupane[ 0 ].title    = msgGetText( MXDI_FILE_MENU );
  menupane[ 0 ].mnemonic = msgGetText( MXDI_FILE_MENU_ACC );
  menupane[ 1 ].title    = msgGetText( MXDI_HELP_MENU );
  menupane[ 1 ].mnemonic = msgGetText( MXDI_HELP_MENU_ACC );

  file_casc[ 0 ].title    = msgGetText( MXDI_INCL_FILE_MENU );
  file_casc[ 0 ].mnemonic = msgGetText( MXDI_INCL_FILE_MENU_ACC );
  file_casc[ 2 ].title    = msgGetText( MXDI_CLOSE_MENU );
  file_casc[ 2 ].mnemonic = msgGetText( MXDI_CLOSE_MENU_ACC );

  help_casc[ 0 ].title    = msgGetText( MXDI_HELP_CONTEXT );
  help_casc[ 0 ].mnemonic = msgGetText( MXDI_HELP_CONTEXT_ACC );
  help_casc[ 1 ].title    = msgGetText( MXDI_HELP_WINDOWS );
  help_casc[ 1 ].mnemonic = msgGetText( MXDI_HELP_WINDOWS_ACC );
  help_casc[ 2 ].title    = msgGetText( MXDI_HELP_KEYS );
  help_casc[ 2 ].mnemonic = msgGetText( MXDI_HELP_KEYS_ACC );
  help_casc[ 3 ].title    = msgGetText( MXDI_HELP_INDEX );
  help_casc[ 3 ].mnemonic = msgGetText( MXDI_HELP_INDEX_ACC );
  help_casc[ 4 ].title    = msgGetText( MXDI_HELP_HELP );
  help_casc[ 4 ].mnemonic = msgGetText( MXDI_HELP_HELP_ACC );
  help_casc[ 5 ].title    = msgGetText( MXDI_HELP_VERSION );
  help_casc[ 5 ].mnemonic = msgGetText( MXDI_HELP_VERSION_ACC );

  action_button_def[ 0 ].title = msgGetText( MXDI_ALARM_ACTION );
  action_button_def[ 1 ].title = msgGetText( MXDI_PREF_ACTION );
  action_button_def[ 2 ].title = msgGetText( MXDI_STAND_ACTION );
  action_button_def[ 3 ].title = msgGetText( MXDI_MY_EDITOR_ACTION );
  action_button_def[ 5 ].title = msgGetText( MXDI_DISPLAY_DAY_VIEW );


  /* Create a separate window. */
  tempW = xitGetToplevelWidget( parent );

  editorTl = xitCreateToplevelDialog( tempW, "ApEditorTl", 
                                      1, 0,
                                      action_buttons, 
                                      XtNumber( action_buttons ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle,    msgGetText( MXDI_ENTRY_TITLE ) ); n++;
  XtSetArg( args[ n ], XmNiconName, msgGetText( MXDI_ENTRY_IC_TITLE ) ); n++;
  XtSetValues( editorTl, args, n );

  XtAddCallback( editorTl,  XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) editor_ref );

  /* Close the window if this window is deleted. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( editorTl ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( editorTl, &wm_delete_window, 1 );
    XmAddWMProtocolCallback( editorTl, wm_delete_window, 
                             (XtCallbackProc) cancelCB,
                             (XtPointer) editor_ref );
  } /* block */


  /* Reference to the contents of the window. */
  workFo = XtNameToWidget( editorTl, "ApEditorTlBase.ApEditorTlFo" );


  /* Create the menubar and menu cascades. */
  menuBr = XmCreateMenuBar( workFo, "MenuBr", args, 0 );

  n = 0;
  for( index = 0; index < XtNumber( pulldownMenu ); index++ )
    pulldownMenu[ index ] = XmCreatePulldownMenu( menuBr, 
                                                  pull_downs[ index ], 
                                                  NULL, n );

  for( index = 0; index < XtNumber( menuCasc ); index++ )
    menuCasc[ index ] = xitCreateCascadeButton( menuBr, 
                                                pulldownMenu[ index ], 
                                                &menupane[ index ] );

  /* The help button should be placed to the right. */
  index = XtNumber( menuCasc ) - 1;
  n     = 0;
  XtSetArg( args[ n ], XmNmenuHelpWidget, menuCasc[ index ] ); n++;
  XtSetValues( menuBr, args, n );


  /* Create the file menu. */
  for( index = 0; index < XtNumber( menuFileBu ); index++ )
    menuFileBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 0 ],
                                                   &file_casc[ index ] );
  XtAddCallback( menuFileBu[ 0 ], XmNactivateCallback, 
                 (XtCallbackProc) includeFileCB, (XtPointer) editor_ref );
  XtAddCallback( menuFileBu[ 2 ], XmNactivateCallback, 
                 (XtCallbackProc) cancelCB, (XtPointer) editor_ref );


  /* Create the help menu. */
  XtAddCallback( pulldownMenu[ 1 ], XmNentryCallback, 
                 (XtCallbackProc) helpCB, (XtPointer) editor_ref );

  for( index = 0; index < XtNumber( menuHelpBu ); index++ ) {
    menuHelpBu[ index ] = xitCreateMenuPushButton( pulldownMenu[ 1 ], 
                                                   &help_casc[ index ] );

    XtAddCallback( menuHelpBu[ index ], XmNactivateCallback, 
                   (XtCallbackProc) helpCB, (XtPointer) index );
  }

  /* We can't do context sensitive help. */
  XtSetSensitive( menuHelpBu[ 0 ], False );


  /* Header with entry date. */
  entryDateLa = xitCreateLabel( workFo, "EntryDateLa",
                                " \n ", XmALIGNMENT_CENTER );


  /* Create the time entry form. */
  timeEntFr = XmCreateFrame( workFo,    "TimeEntFr", args, 0 );
  dataEntFo = XmCreateForm(  timeEntFr, "DataEntFo", args, 0 );


  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL  ); n++;
  XtSetArg( args[ n ], XmNpacking,     XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNnumColumns,  3             ); n++;
  timeEntRc = XmCreateRowColumn( dataEntFo, "TimeEntRc", args, n );


  /* Entry date. */
  dateEntLa = xitCreateLabel( timeEntRc, "DateEntLa", 
                              msgGetText( MXDI_DATE_LABEL ), -1 );

  dateEntTb = xtmFlCreateDateField( timeEntRc, "DateEntTb" );

  XtAddCallback( dateEntTb, XmNvalueChangedCallback, 
                 (XtCallbackProc) timeFieldChangedCB,
                 (XtPointer) editor_ref );


  /* Start time. */
  startEntLa = xitCreateLabel( timeEntRc, "StartEntLa", 
                               msgGetText( MXDI_ENTRY_START_LABEL ), -1 );

  startEntTb = xtmFlCreateTimeField( timeEntRc, "StartEntTb" );

  XtAddCallback( startEntTb, XmNvalueChangedCallback, 
                 (XtCallbackProc) timeFieldChangedCB,
                 (XtPointer) editor_ref );



  /* Duration. */
  durEntLa = xitCreateLabel( timeEntRc, "DurEntLa",
                             msgGetText( MXDI_ENTRY_DURATION_LABEL ), -1 );

  durEntFs = xtmFlCreateSelField( timeEntRc, "DurEntFs",
                                  6, def_durations,
                                  ' ', True,
                                  NULL, NULL );

  tempW = xitFieldSelectGetChild( durEntFs, xitFIELD_SELECT_TEXT_FIELD );

  XtAddCallback( tempW, XmNvalueChangedCallback, 
                 (XtCallbackProc) timeFieldChangedCB,
                 (XtPointer) editor_ref );


  /* Entry types. */
  n = 0;
  XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNradioBehavior, True ); n++;
  typeRb = XmCreateRowColumn( dataEntFo, "TypeRb", args, n );

  typeAppTb = xitCreateToggleButton( 
                typeRb, "TypeAppTb", 
                msgGetText( MXDI_TYPE_APPOINT ), False );

  XtAddCallback( typeAppTb, XmNvalueChangedCallback, 
                 (XtCallbackProc) entryTypeCB, (XtPointer) editor_ref );


  typeStandAppTb = xitCreateToggleButton( 
                     typeRb, "TypeStandAppTb", 
                     msgGetText( MXDI_TYPE_STAND_APPOINT ), False );

  XtAddCallback( typeStandAppTb, XmNvalueChangedCallback, 
                 (XtCallbackProc) entryTypeCB, (XtPointer) editor_ref );


  typeNoteTb = xitCreateToggleButton( 
                 typeRb, "TypeNoteTb", 
                 msgGetText( MXDI_TYPE_NOTE ), False );

  XtAddCallback( typeNoteTb, XmNvalueChangedCallback, 
                 (XtCallbackProc) entryTypeCB, (XtPointer) editor_ref );


  typeStandNoteTb = xitCreateToggleButton( 
                      typeRb, "TypeStandNoteTb", 
                      msgGetText( MXDI_TYPE_STAND_NOTE ), False );

  XtAddCallback( typeStandNoteTb, XmNvalueChangedCallback, 
                 (XtCallbackProc) entryTypeCB, (XtPointer) editor_ref );


  typeStickyNoteTb = xitCreateToggleButton( 
                       typeRb, "TypeStickyNoteTb", 
                       msgGetText( MXDI_TYPE_STICKY_NOTE ), False );

  XtAddCallback( typeStickyNoteTb, XmNvalueChangedCallback, 
                 (XtCallbackProc) entryTypeCB, (XtPointer) editor_ref );


  /* Place the elements together. */
  xitAttachWidget( timeEntRc,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( typeRb,
                   XmATTACH_FORM, NULL, XmATTACH_WIDGET, timeEntRc,
                   XmATTACH_NONE, NULL, XmATTACH_NONE,   NULL );


  /* Create the selection form. */
  selFr = XmCreateFrame(     workFo, "SelFr", args, 0 );
  selFo = XmCreateRowColumn( selFr,  "SelFo", args, 0 );

  selAlarmBu     = xitCreateToggleButton( 
                     selFo, "SelAlarmBu", 
                     msgGetText( MXDI_ALARM_BUTTON ), False );

  selImpBu       = xitCreateToggleButton( 
                     selFo, "SelImpBu",
                     msgGetText( MXDI_IMPORTANT_BUTTON ), False );

  selNoteDoneBu  = xitCreateToggleButton( 
                     selFo, "SelNoteDoneBu",
                     msgGetText( MXDI_DONE_BUTTON ), False );

  selPrivateBu   = xitCreateToggleButton( 
                     selFo, "SelPrivateBu",
                     msgGetText( MXDI_PRIVATE_BUTTON ), False );



  /* Actions to take. */
  n = 0;
  XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNnumColumns, 2 ); n++;
  XtSetArg( args[ n ], XmNspacing, 10 ); n++;
  actionRc = XmCreateRowColumn( workFo, "ActionRc", args, n );

  alarmPb   = xitCreatePushButton( actionRc, &action_button_def[ 0 ] );
  prefPb    = xitCreatePushButton( actionRc, &action_button_def[ 1 ] );
  standPb   = xitCreatePushButton( actionRc, &action_button_def[ 2 ] );
  editorPb  = xitCreatePushButton( actionRc, &action_button_def[ 3 ] );
  dummy1Pb  = xitCreatePushButton( actionRc, &action_button_def[ 4 ] );
  dayViewPb = xitCreatePushButton( actionRc, &action_button_def[ 5 ] );

  XtAddCallback( alarmPb, XmNactivateCallback,
                 (XtCallbackProc) alarmCB, (XtPointer) editor_ref );
  XtAddCallback( prefPb, XmNactivateCallback,
                 (XtCallbackProc) preferencesCB, (XtPointer) editor_ref );
  XtAddCallback( standPb, XmNactivateCallback,
                 (XtCallbackProc) standingCB, (XtPointer) editor_ref );
  XtAddCallback( editorPb, XmNactivateCallback,
                 (XtCallbackProc) editorCB, (XtPointer) editor_ref );
  XtAddCallback( dayViewPb, XmNactivateCallback,
                 (XtCallbackProc) dayViewCB, (XtPointer) editor_ref );


  /* Create form fot entry tag and database selection. */
  miscFo = XmCreateForm( workFo, "MiscFo", args, n );


  /* Entry tag. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  miscRc = XmCreateRowColumn( miscFo, "MiscRc",  args, n );

  entryTagLa = xitCreateLabel( miscRc, "EntryTagLa", 
                               msgGetText( MXDI_ENTRY_TAG_LABEL ), -1 );


  /* Entry tag selections. */
  entryTagFs = xtmFlCreateTagsField( miscRc, "EntryTagFs",
                                     4,
                                     custom_data_ref -> entry_tags_menu,
                                     custom_data_ref -> tags_db,
                                     NULL, NULL );

  tempW = xitFieldSelectGetChild( entryTagFs, xitFIELD_SELECT_TEXT_FIELD );


  /* Database selection. */
  databaseLa = xitCreateLabel( miscRc, "DatabaseLa", 
                               msgGetText( MXDI_SAVE_IN_DB_LABEL ), -1 );


  /* Database selections. */
  databaseFs = xtmFlCreateDbField( miscRc, "DatabaseFs",
                                   custom_data_ref -> cal_db_handle,
                                   True,
                                   (XtCallbackProc) dbChangedCB, 
                                   (void *) editor_ref );


  /* Entry text label. */
  if( custom_data_ref -> text_marker_pos > 0 &&
      custom_data_ref -> text_marker_pos < 150 ) {

    char  buffer1[ 200 ];
    char  format[ 20 ];

    sprintf( format, "%%-%d.%ds", 
             custom_data_ref -> text_marker_pos -1,
             custom_data_ref -> text_marker_pos -1 );

    sprintf( buffer1, format, msgGetText( MXDI_TEXT_LABEL ) );
    strcat( buffer1, "." );

    textEntLa = xitCreateLabel( workFo, "TextEntLa", buffer1, -1 );

  } else {
    textEntLa = xitCreateLabel( workFo, "TextEntLa", 
                                msgGetText( MXDI_TEXT_LABEL ), -1 );
  }

  /* Create the text input form. */
  textEntTx = xitCreateTextScrolled( workFo, &text_buffer_def[ 0 ] );

  XtOverrideTranslations( textEntTx, xtm_di_newline_trans );


  /* Place the various elements together. */
  xitAttachWidget( menuBr,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( entryDateLa,
                   XmATTACH_WIDGET, menuBr, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( timeEntFr,
                   XmATTACH_WIDGET, entryDateLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,        XmATTACH_NONE, NULL );
  if( custom_data_ref -> read_only_mode ) {
    xitAttachWidget( textEntLa,
                     XmATTACH_WIDGET, timeEntFr, XmATTACH_FORM, NULL,
                     XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  } else {
    xitAttachWidget( selFr,
                     XmATTACH_WIDGET, timeEntFr, XmATTACH_FORM, NULL,
                     XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
    xitAttachWidget( actionRc,
                     XmATTACH_WIDGET, timeEntFr, XmATTACH_WIDGET, selFr,
                     XmATTACH_NONE,   NULL,      XmATTACH_NONE,   NULL );
    xitAttachWidget( miscFo,
                     XmATTACH_WIDGET, selFr, XmATTACH_FORM, NULL,
                     XmATTACH_NONE,   NULL,  XmATTACH_NONE, NULL );
    xitAttachWidget( textEntLa,
                     XmATTACH_WIDGET, miscFo, XmATTACH_FORM, NULL,
                     XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  }

  xitAttachWidget( XtParent( textEntTx ),
                   XmATTACH_WIDGET, textEntLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( entryDateLa,           args, n );
  XtSetValues( timeEntRc,             args, n );
  XtSetValues( typeRb,                args, n );
  XtSetValues( selFr,                 args, n );
  XtSetValues( timeEntFr,             args, n );
  XtSetValues( actionRc,              args, n );
  XtSetValues( miscRc,                args, n );
  XtSetValues( textEntLa,             args, n );
  XtSetValues( XtParent( textEntTx ), args, n );


  /* Manage all the children. */
  XtManageChildren( menuCasc,   XtNumber( menuCasc ) );
  XtManageChildren( menuHelpBu, XtNumber( menuHelpBu ) );
  XtManageChildren( menuFileBu, XtNumber( menuFileBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set icon for this window. */
  xtmIcSetSimpleIcon( editorTl, workFo, XTM_IC_ICON_EDIT_ENTRY );

  /* Set the size of the window. */
  xitSetSizeToplevelDialog( editorTl, True );


  /* Make the final attachments. */
  n = 0;
  XtSetArg( args[ n ], XmNrightAttachment,  XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
  XtSetValues( XtParent( textEntTx ), args, n );

  if( custom_data_ref -> read_only_mode ) {
    XtUnmanageChild( selFr );
    XtUnmanageChild( actionRc );
    XtUnmanageChild( miscFo );

    XtSetSensitive( timeEntFr,             False );
    XtSetSensitive( XtParent( textEntTx ), False );
  } else {
    XtUnmapWidget( dummy1Pb );
  }


  return( editorTl );

} /* createEditWindow */


/*----------------------------------------------------------------------*/

static Boolean
  fetchEntryTimes( EDITOR_REC_REF   editor_ref,
                   Widget           parentW,
                   ENTRY_TIMES_REF  entry_times )
{

  /* Variables. */
  Boolean                 is_appointment = False;
  Boolean                 ok;
  char                    *char_ref;
  Widget                  baseW;
  Widget                  mainW;
  Widget                  tempW;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = editor_ref -> appl_data_ref -> custom_data;


  /* Editor window. */
  mainW = XtNameToWidget( editor_ref -> editorW, 
                          "ApEditorTlBase.ApEditorTlFo" );

  /* Default values. */
  entry_times -> date       = 0;
  entry_times -> start_time = 0;
  entry_times -> end_time   = 0;
  entry_times -> duration   = 0;


  baseW = XtNameToWidget( mainW, "TimeEntFr.DataEntFo.TypeRb" );

  /* Kind of entry. */
  tempW = XtNameToWidget( baseW, "TypeAppTb" );
  if( XmToggleButtonGetState( tempW ) )
    is_appointment = True;

  tempW = XtNameToWidget( baseW, "TypeStandAppTb" );
  if( XmToggleButtonGetState( tempW ) )
    is_appointment = True;


  /* Fetch the entry times. */
  baseW = XtNameToWidget( mainW, "TimeEntFr.DataEntFo.TimeEntRc" );


  /* Date. */
  tempW = XtNameToWidget( baseW, "DateEntTb" );

  ok = xtmFoFetchDate( parentW, tempW, XTM_FO_START_DATE, False,
                       &entry_times -> date );
  if( ! ok )
    return( False );


  /* Start time. */
  if( is_appointment ) {
    tempW = XtNameToWidget( baseW, "StartEntTb" );

    ok = xtmFoFetchTime( parentW, tempW, XTM_FO_START_TIME, False,
                         &entry_times -> start_time );
    if( ! ok )
      return( False );
  }


  /* Duration. */
  if( is_appointment ) {

    tempW = XtNameToWidget( baseW, "DurEntFs" );
    xitFieldSelectGetCurrent( tempW, &char_ref );

    ok = xtmFoFetchMinuteTime( char_ref, True, parentW, 23 * 60,
                               &entry_times -> duration );

    SysFree( char_ref );
    if( ! ok )
      return( False );

    if( entry_times -> duration <= 0 ) {
      entry_times -> duration = 0;
    } else {
      entry_times -> end_time = entry_times -> start_time;
      TimAddMinutes( &entry_times -> end_time, entry_times -> duration );
    }

  } /* if */


  return( True );

} /* fetchEntryTimes */


/*----------------------------------------------------------------------*/

static Boolean
  fetchEntryValues( EDITOR_REC_REF        editor_ref,
                    XTM_DB_ALL_ENTRY_REF  entry_ref,
                    char                  **entry_text )
{

  /* Variables. */
  Boolean               ok;
  char                  *char_ref;
  Widget                baseW;
  Widget                mainW;
  Widget                tempW;
  ENTRY_TIMES_REC       entry_times;
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = editor_ref -> appl_data_ref;


  /* Editor window. */
  mainW = XtNameToWidget( editor_ref -> editorW, 
                          "ApEditorTlBase.ApEditorTlFo" );

  /* Default values. */
  entry_ref -> entry.entry_category = XTM_DB_ENTRY_LIST;
  entry_ref -> entry.entry_type     = XTM_DB_DAY_ENTRY;


  baseW = XtNameToWidget( mainW, "TimeEntFr.DataEntFo.TypeRb" );

  /* Standing appointment. */
  tempW = XtNameToWidget( baseW, "TypeStandAppTb" );
  if( XmToggleButtonGetState( tempW ) )
    entry_ref -> entry.entry_category = XTM_DB_REP_ENTRY_LIST;

  /* Note. */
  tempW = XtNameToWidget( baseW, "TypeNoteTb" );
  if( XmToggleButtonGetState( tempW ) )  
    entry_ref -> entry.entry_type = XTM_DB_DAY_NOTE;

  /* Standing note. */
  tempW = XtNameToWidget( baseW, "TypeStandNoteTb" );
  if( XmToggleButtonGetState( tempW ) ) {
    entry_ref -> entry.entry_type     = XTM_DB_DAY_NOTE;
    entry_ref -> entry.entry_category = XTM_DB_REP_ENTRY_LIST;
  }

  /* Sticky note. */
  tempW = XtNameToWidget( baseW, "TypeStickyNoteTb" );
  if( XmToggleButtonGetState( tempW ) ) {
    entry_ref -> entry.entry_type     = XTM_DB_DAY_NOTE;
    entry_ref -> entry.entry_category = XTM_DB_STICKY_LIST;
  }


  /* Fetch the entry times. */
  ok = fetchEntryTimes( editor_ref, editor_ref -> editorW, &entry_times );
  if( ! ok )
    return( False );

  entry_ref -> entry.date_stamp = (UINT32) entry_times.date;
  entry_ref -> entry.time_stamp = (UINT32) entry_times.start_time;
  entry_ref -> entry.duration   = (UINT16) entry_times.duration;


  /* Entry flags. */
  baseW = XtNameToWidget( mainW, "SelFr.SelFo" );

  /* Alarm selection. */
  tempW = XtNameToWidget( baseW, "SelAlarmBu" );
  if( XmToggleButtonGetState( tempW ) )  
    flagSet( entry_ref -> entry.flags, XTM_DB_FLAG_ALARM );
  else
    flagClear( entry_ref -> entry.flags, XTM_DB_FLAG_ALARM );


  /* Importance. */
  tempW = XtNameToWidget( baseW, "SelImpBu" );
  if( XmToggleButtonGetState( tempW ) )  
    flagSet( entry_ref -> entry.flags, XTM_DB_FLAG_IMPORTANT );
  else
    flagClear( entry_ref -> entry.flags, XTM_DB_FLAG_IMPORTANT );


  /* Note done. */
  tempW = XtNameToWidget( baseW, "SelNoteDoneBu" );
  if( XmToggleButtonGetState( tempW ) )
    flagSet( entry_ref -> entry.flags, XTM_DB_FLAG_NOTE_DONE );
  else
    flagClear( entry_ref -> entry.flags, XTM_DB_FLAG_NOTE_DONE );


  /* Private. */
  tempW = XtNameToWidget( baseW, "SelPrivateBu" );
  if( XmToggleButtonGetState( tempW ) )  
    flagSet( entry_ref -> entry.flags, XTM_DB_FLAG_PRIVATE );
  else
    flagClear( entry_ref -> entry.flags, XTM_DB_FLAG_PRIVATE );


  /* Entry tag and database. */
  baseW = XtNameToWidget( mainW, "MiscFo.MiscRc" );

  tempW = XtNameToWidget( baseW, "EntryTagFs" );
  xitFieldSelectGetCurrent( tempW, &char_ref );

  if( char_ref != NULL ) {
    strncpy( entry_ref -> entry.tag, char_ref, XTM_DB_TAG_LEN );
    entry_ref -> entry.tag[ XTM_DB_TAG_LEN ] = '\0';

    SysFree( char_ref );
  }

 
  /* Calendar. */
  tempW = XtNameToWidget( baseW, "DatabaseFs" );
  xitFieldSelectGetCurrent( tempW, &char_ref );

  if( char_ref == NULL )
    return( False );

  if( *char_ref == '\0' ) {
    xitErMessage( editor_ref -> editorW, XIT_ER_ERROR,
                  module_name, "fetchBaseFromWindow",
                  msgGetText( MXDI_NO_DB_SELECTED ) );

    SysFree( char_ref );
    return( False );
  }

  strcpy(  entry_ref -> db_name, char_ref );
  SysFree( char_ref );


  /* Text for entry. */
  tempW = XtNameToWidget( mainW, "TextEntTxSW.TextEntTx" );
  *entry_text = xitStringGetText( tempW );

  char_ref = *entry_text;
  while( isspace( *char_ref ) )
    char_ref++;

  /* Entries without text are not saved. */
  if( *char_ref == '\0' ) {
    xitErMessage( editor_ref -> editorW, XIT_ER_ERROR,
                  module_name, "fetchBaseFromWindow",
                  msgGetText( MXDI_ENTRY_HAS_NO_TEXT ) );

    SysFree( *entry_text );
    return( False );
  }

  strncpy( entry_ref -> entry.text, *entry_text, 100 );
  entry_ref -> entry.text[ 100 ] = '\0';


  return( True );

} /* fetchEntryValues */


/*----------------------------------------------------------------------*/

static Boolean
  readEntryToEdit( EDITOR_REC_REF  editor_ref,
                   char            *db_name,
                   UINT32          entry_id,
                   Widget          parent )
{

  /* Variables. */
  Boolean                 ok;
  XTM_DB_ALL_ENTRY_REF    entry_ref;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_STATUS           db_status;


  /* Code. */

  entry_ref = editor_ref -> edit_entry_ref;

  /* If we are editing an entry, free the data. */
  if( entry_ref != NULL ) {
    SysFree( entry_ref -> all_text );
    SysFree( entry_ref );
  }

  /* Allocate a record to hold an entry. */
  editor_ref -> edit_entry_ref = SysNew( XTM_DB_ALL_ENTRY_DEF );
  entry_ref = editor_ref -> edit_entry_ref;

  editor_ref -> new_entry     = True;
  editor_ref -> standing_seen = False;


  /* Default values for the entry. */
  setEntryDefault( editor_ref -> appl_data_ref, 
                   entry_ref,
                   (SET_ALL_ENTRY | SET_ENTRY | SET_REPEATED_ENTRY),
                   editor_ref -> default_date, 
                   editor_ref -> default_time,
                   editor_ref -> default_duration );

  strcpy( entry_ref -> db_name, db_name );

  /* Enpty entry? */
  if( entry_id == 0 )
    return( True );


  /* Open the database. */
  ok = xtmDmOpenDatabase( editor_ref -> appl_data_ref,
                          db_name, XTM_DB_FLAG_MODE_READ,
                          &database );
  if( ! ok )
    return( False );


  /* Fetch the entry. */
  db_status = xtmDbFetchEntry( &database, entry_id, 
                               entry_ref, &entry_ref -> all_text );

  xtmDbCloseEntryDb( &database );


  /* If the entry could not be fetched, it is perhaps removed? */
  if( db_status != XTM_DB_OK ) {
    xitErMessage( parent, XIT_ER_ERROR,
                  module_name, "fetchEntryToEdit",
                  msgGetText( MXDI_NO_ENTRY ) );

    return( False );
  }

  /* If the entry is not repeated, set the repeated part. */
  if( entry_ref -> entry.entry_category == XTM_DB_ENTRY_LIST )
    setEntryDefault( editor_ref -> appl_data_ref, 
                     entry_ref, SET_REPEATED_ENTRY,
                     editor_ref -> default_date, 
                     editor_ref -> default_time,
                     editor_ref -> default_duration );

  editor_ref -> new_entry = False;


  return( True );

} /* readEntryToEdit */


/*----------------------------------------------------------------------*/

static void
  removeEditFile( EDITOR_REC_REF  editor_ref )
{

  /* Variables. */
  int     pid;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  /* Kill the editor? */
  if( editor_ref -> editor_pid > 0 ) {
    pid = editor_ref -> editor_pid;

    editor_ref -> editor_pid = 0;
    (void) kill( pid, SIGKILL );
  }

  /* Free the file name. */
  if( editor_ref -> temp_file != NULL ) {
    (void) unlink( editor_ref -> temp_file );

    SysFree( editor_ref -> temp_file );
    editor_ref -> temp_file = NULL;
  }

  /* Reset the editor or change the text field. */
  if( editor_ref -> editorW != NULL ) {
    mainW = XtNameToWidget( editor_ref -> editorW, 
                            "ApEditorTlBase.ApEditorTlFo" );

    tempW = XtNameToWidget( mainW, "ActionRc.EditorPb" );
    XtSetSensitive( tempW, True );

    tempW = XtNameToWidget( mainW, "TextEntTxSW.TextEntTx" );
    XtSetSensitive( tempW, True );
  }


  return;

} /* removeEditFile */


/*----------------------------------------------------------------------*/

static Boolean
  saveEntry( EDITOR_REC_REF  editor_ref )
{

  /* Variables. */
  Boolean               ok;
  XTM_DB_ALL_ENTRY_REF  new_entry_ref;
  XTM_DB_ALL_ENTRY_REF  old_entry_ref;


  /* Code. */

  new_entry_ref = SysNew( XTM_DB_ALL_ENTRY_DEF );
  old_entry_ref = editor_ref -> edit_entry_ref;

  new_entry_ref -> all_text = NULL;


  /* The new entry is a copy of the old (with the exception of all_text). */
  memcpy( new_entry_ref, old_entry_ref, sizeof( XTM_DB_ALL_ENTRY_DEF ) );


  /* Fetch data entered in windows. */
  ok = fetchEntryValues( editor_ref, 
                         new_entry_ref, &new_entry_ref -> all_text );
  if( ! ok )
    raise exception;


  /* Make sure the user sees the standing window. */
  if( editor_ref -> appl_data_ref -> custom_data -> prompt_repeat_win ) {

    if( new_entry_ref -> entry.entry_category == XTM_DB_REP_ENTRY_LIST ||
        new_entry_ref -> entry.entry_category == XTM_DB_STICKY_LIST ) {

      if( editor_ref -> new_entry && ! editor_ref -> standing_seen ) {
        standingCB( editor_ref -> editorW, editor_ref, NULL );

        raise exception;
      }
    }

  } /* if */


  /* Save the entry. */
  ok = writeEditedEntry( editor_ref -> appl_data_ref,
                         editor_ref -> editorW,
                         old_entry_ref, new_entry_ref );
  if( ! ok )
    raise exception;


  /* The new entry now becomes the old entry (including all_text). */
  if( editor_ref -> edit_entry_ref != NULL ) {
    SysFree( editor_ref -> edit_entry_ref -> all_text );
    SysFree( editor_ref -> edit_entry_ref );
  }

  editor_ref -> edit_entry_ref = new_entry_ref;

  editor_ref -> edit_entry_ref -> entry.id = 0;
  editor_ref -> edit_entry_ref -> stand_entry.id = 0;

  return( True );


  /* Exception handler. */
  exception:

    if( new_entry_ref -> all_text != NULL )
      SysFree( new_entry_ref -> all_text );

    SysFree( new_entry_ref );

    return( False );

} /* saveEntry */


/*----------------------------------------------------------------------*/

static void
  setEntryValues( EDITOR_REC_REF        editor_ref,
                  XTM_DB_ALL_ENTRY_REF  entry_ref )
{

  /* Variables. */
  Boolean                 ok;
  UINT32                  can_do;
  char                    buffer[ 50 ];
  Arg                     args[ 5 ];
  Cardinal                n;
  Widget                  baseW;
  Widget                  mainW;
  Widget                  tempW;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  appl_data_ref   = editor_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  /* Do we have a valid window? */
  if( editor_ref -> editorW == NULL )
    return;

  mainW = XtNameToWidget( editor_ref -> editorW, 
                          "ApEditorTlBase.ApEditorTlFo" );


  /* Fetch info regarding the database for the entry. */
  ok = xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle,
                          entry_ref -> db_name,
                          &db_info, NULL );
  if( ! ok )
    return;


  /* Make sure our children don't spoil our size. */
  n = 0;
  XtSetArg( args[ n ], XmNallowShellResize, False ); n++;
  XtSetValues( editor_ref -> editorW, args, n );


  /* Date for the entry. */
  baseW = XtNameToWidget( mainW, "TimeEntFr.DataEntFo.TimeEntRc" );
  tempW = XtNameToWidget( baseW, "DateEntTb" );

  XmUbTimeBoxSetStartDate( tempW, editor_ref -> default_date );


  /* Entry start. */
  baseW = XtNameToWidget( mainW, "TimeEntFr.DataEntFo.TimeEntRc" );
  tempW = XtNameToWidget( baseW, "StartEntTb" );

  if( entry_ref -> entry.entry_type == XTM_DB_DAY_ENTRY )
    XmUbTimeBoxSetStartTime( tempW, entry_ref -> entry.time_stamp );
  else
    XmUbTimeBoxSetStartTimeString( tempW, "" );


  /* Duration. */
  buffer[ 0 ] = '\0';

  switch( entry_ref -> entry.entry_type ) {
    case XTM_DB_DAY_ENTRY:
      if( entry_ref -> entry.duration > 0 )
        sprintf( buffer, "%d:%02d", 
                 (int) entry_ref -> entry.duration / 60,
                 (int) entry_ref -> entry.duration % 60 );
      break;
  }

  baseW = XtNameToWidget( mainW, "TimeEntFr.DataEntFo.TimeEntRc" );
  tempW = XtNameToWidget( baseW, "DurEntFs" );

  xitFieldSelectSetCurrent( tempW, buffer, False );


  /* Type of entry. */
  baseW = XtNameToWidget( mainW, "TimeEntFr.DataEntFo.TypeRb" );

  /* No buttons set. */
  tempW = XtNameToWidget( baseW, "TypeAppTb" );
  XmToggleButtonSetState( tempW, False, False );

  tempW = XtNameToWidget( baseW, "TypeStandAppTb" );
  XmToggleButtonSetState( tempW, False, False );

  tempW = XtNameToWidget( baseW, "TypeNoteTb" );
  XmToggleButtonSetState( tempW, False, False );

  tempW = XtNameToWidget( baseW, "TypeStandNoteTb" );
  XmToggleButtonSetState( tempW, False, False );

  tempW = XtNameToWidget( baseW, "TypeStickyNoteTb" );
  XmToggleButtonSetState( tempW, False, False );


  /* Appointment? */
  if( entry_ref -> entry.entry_type == XTM_DB_DAY_ENTRY ) {

    if( entry_ref -> entry.entry_category == XTM_DB_REP_ENTRY_LIST ) {
      tempW = XtNameToWidget( baseW, "TypeStandAppTb" );
      editor_ref -> entry_type = XTM_DB_TYPE_STAND_APPOINTMENT;

    } else {
      tempW = XtNameToWidget( baseW, "TypeAppTb" );
      editor_ref -> entry_type = XTM_DB_TYPE_APPOINTMENT;
    }

    XmToggleButtonSetState( tempW, True, False );
  }

  /* Note? */
  if( entry_ref -> entry.entry_type == XTM_DB_DAY_NOTE ) {

    if( entry_ref -> entry.entry_category == XTM_DB_REP_ENTRY_LIST ) {
      tempW = XtNameToWidget( baseW, "TypeStandNoteTb" );
      editor_ref -> entry_type = XTM_DB_TYPE_STAND_NOTE;

    } else if( entry_ref -> entry.entry_category == XTM_DB_STICKY_LIST ) {
      tempW = XtNameToWidget( baseW, "TypeStickyNoteTb" );
      editor_ref -> entry_type = XTM_DB_TYPE_STICKY_NOTE;

    } else {
      tempW = XtNameToWidget( baseW, "TypeNoteTb" );
      editor_ref -> entry_type = XTM_DB_TYPE_NOTE;
    }

    XmToggleButtonSetState( tempW, True, False );
  }


  /* Alarm selection. */
  baseW = XtNameToWidget( mainW, "SelFr.SelFo" );

  tempW = XtNameToWidget( baseW, "SelAlarmBu" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_ALARM ) &&
      entry_ref -> entry.entry_type != XTM_DB_DAY_NOTE )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );


  /* Importance. */
  tempW = XtNameToWidget( baseW, "SelImpBu" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_IMPORTANT ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );


  /* Note done. */
  tempW = XtNameToWidget( baseW, "SelNoteDoneBu" );

  if( entry_ref -> entry.entry_type == XTM_DB_DAY_NOTE &&
      flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_NOTE_DONE ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );


  /* Private. */
  tempW = XtNameToWidget( baseW, "SelPrivateBu" );

  if( flagIsSet( entry_ref -> entry.flags, XTM_DB_FLAG_PRIVATE ) )
    XmToggleButtonSetState( tempW, True, False );
  else
    XmToggleButtonSetState( tempW, False, False );


  /* Show/hide valid days? */
  baseW = XtNameToWidget( mainW, "ActionRc" );
  tempW = XtNameToWidget( baseW, "StandPb" );

  if( entry_ref -> entry.entry_category == XTM_DB_REP_ENTRY_LIST ||
      entry_ref -> entry.entry_category == XTM_DB_STICKY_LIST )
    XtSetSensitive( tempW, True );
  else
    XtSetSensitive( tempW, False );


  /* Entry tag and database name. */
  baseW = XtNameToWidget( mainW, "MiscFo.MiscRc" );


  /* Entry tag. */
  tempW = XtNameToWidget( baseW, "EntryTagFs" );

  if( entry_ref -> entry.tag[ 0 ] != '\0' )
    xitFieldSelectSetCurrent( tempW, entry_ref -> entry.tag, False );
  else
    xitFieldSelectSetCurrent( tempW, "", False );


  /* Database name. */
  tempW = XtNameToWidget( baseW, "DatabaseFs" );

  xitFieldSelectSetCurrent( tempW, entry_ref -> db_name, False );


  /* The text entry. */
  tempW = XtNameToWidget( mainW, "TextEntTxSW.TextEntTx" );

  if( editor_ref -> new_entry ) {
    XmTextSetString( tempW, custom_data_ref -> default_entry_text );
  } else {
    if( entry_ref -> all_text == NULL )
      XmTextSetString( tempW, entry_ref -> entry.text );
    else
      XmTextSetString( tempW, entry_ref -> all_text );
  }


  /* Entry date. */
  writeEntryDate( editor_ref );


  /* What can we do? */
  updateCanDo( editor_ref );


  /* Can we change the private flag? */
  baseW = XtNameToWidget( mainW, "SelFr.SelFo" );
  tempW = XtNameToWidget( baseW, "SelPrivateBu" );

  if( flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_PRIV ) )
    XtSetSensitive( tempW, True );
  else
    XtSetSensitive( tempW, False );


  /* Can we save the entries or not? */
  xtmDbGetEntryPermissions( db_info.operations,
                            entry_ref -> entry.owner,
                            entry_ref -> entry.flags,
                            &can_do );

  baseW = XtNameToWidget( editor_ref -> editorW, "ApEditorTlBase" );

  if( flagIsSet( can_do, XTM_DB_PROT_CHANGE ) &&
      ! appl_data_ref -> custom_data -> read_only_mode ) {

    tempW = XtNameToWidget( baseW, "Bu1" );
    XtMapWidget( tempW );

    tempW = XtNameToWidget( baseW, "Bu3" );
    XtMapWidget( tempW );

  } else {

    tempW = XtNameToWidget( baseW, "Bu1" );
    XtUnmapWidget( tempW );

    tempW = XtNameToWidget( baseW, "Bu3" );
    XtUnmapWidget( tempW );

  } /* if */


  return;

} /* setEntryValues */


/*----------------------------------------------------------------------*/

static XTM_DB_STATUS
  setEntryDefault( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   XTM_DB_ALL_ENTRY_REF  entry_ref,
                   UINT32                set_parts,
                   TIM_TIME_REF          entry_date,
                   TIM_TIME_REF          entry_time,
                   int                   entry_duration )
{

  /* Variables. */
  int                     index;
  UINT32                  date_stamp;
  UINT32                  default_flags;
  UINT32                  time_stamp;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = appl_data_ref -> custom_data;

  time_stamp = (UINT32) TimMakeTime( 1970, 1, 1, 
                                     TimHour( entry_time ),
                                     TimMinute( entry_time ),
                                     0 );

  date_stamp = (UINT32) TimMakeTime( TimIndexOfYear( entry_date ),
                                     TimIndexOfMonth( entry_date ),
                                     TimIndexOfDay( entry_date ),
                                     0, 0, 0 );

  /* All entry data. */
  if( flagIsSet( set_parts, SET_ALL_ENTRY ) ) {
    entry_ref -> db_name[ 0 ] = '\0';
    entry_ref -> all_text     = NULL;
  }

  /* Base entry data. */
  if( flagIsSet( set_parts, SET_ENTRY ) ) {

    default_flags = 0;

    if( appl_data_ref -> custom_data -> alarm_for_new_app )
      flagSet( default_flags, XTM_DB_FLAG_ALARM );
             
    entry_ref -> entry.id              = 0;
    entry_ref -> entry.time_stamp      = time_stamp;
    entry_ref -> entry.date_stamp      = date_stamp;
    entry_ref -> entry.last_update     = 0;
    entry_ref -> entry.owner           = getuid();
    entry_ref -> entry.entry_type      = XTM_DB_DAY_ENTRY;
    entry_ref -> entry.entry_category  = XTM_DB_ENTRY_LIST;
    entry_ref -> entry.duration        = entry_duration;
    entry_ref -> entry.flags           = default_flags;
    entry_ref -> entry.fg_color_index  = 0;
    entry_ref -> entry.bg_color_index  = 0;
    entry_ref -> entry.text[ 0 ]       = '\0';

    strncpy( entry_ref -> entry.tag, 
             appl_data_ref -> custom_data -> default_entry_tag, 
             XTM_DB_TAG_LEN );
    entry_ref -> entry.tag[ XTM_DB_TAG_LEN ] = '\0';

    entry_ref -> entry.alarm_melody    = 
      appl_data_ref -> custom_data -> default_tune;;
    entry_ref -> entry.day_list_lines  = 
      appl_data_ref -> custom_data -> entry_show_lines;
    entry_ref -> entry.alarm_lines     = 
      appl_data_ref -> custom_data -> alarm_show_lines;

    for( index = 0; index < 5; index++ ) {
      entry_ref -> entry.alarm_offset[ index ] = 
        appl_data_ref -> custom_data -> alarm_offset[ index ];
      entry_ref -> entry.alarm_valid[ index ]  =
        appl_data_ref -> custom_data -> alarm_valid[ index ];
    }

  } /* if */

  /* Repeated entry data. */
  if( flagIsSet( set_parts, SET_REPEATED_ENTRY ) ) {

    entry_ref -> stand_entry.id             = 0;
    entry_ref -> stand_entry.from           = date_stamp;
    entry_ref -> stand_entry.to             = 0;
    entry_ref -> stand_entry.flags          = 0;
    entry_ref -> stand_entry.every_n        = 0;
    entry_ref -> stand_entry.valid_every    = 0;
    entry_ref -> stand_entry.skip_week[ 0 ] = 0;
    entry_ref -> stand_entry.skip_week[ 1 ] = 0;

    /* We must provide the day valid. */
    for( index = 0; index < 7; index++ )
      entry_ref -> stand_entry.valid_days[ index ] = 0;

  } /* if */


  return( XTM_DB_OK );

} /* setEntryDefault */


/*----------------------------------------------------------------------*/

static void 
  updateCanDo( EDITOR_REC_REF editor_ref )
{

  /* Variables. */
  Boolean  alarms   = True;
  Boolean  standing = False;
  Widget   mainW;
  Widget   tempW;


  /* Code. */

  mainW = XtNameToWidget( editor_ref -> editorW, 
                          "ApEditorTlBase.ApEditorTlFo" );


  if( editor_ref -> entry_type == XTM_DB_TYPE_STAND_APPOINTMENT )
    standing = True;

  if( editor_ref -> entry_type == XTM_DB_TYPE_NOTE )
    alarms = False;

  if( editor_ref -> entry_type == XTM_DB_TYPE_STAND_NOTE ) {
    standing = True;
    alarms   = False;
  }

  if( editor_ref -> entry_type == XTM_DB_TYPE_STICKY_NOTE ) {
    standing = True;
    alarms   = False;
  }

  /* Alarm and standing buttons sensitive/not sensitive? */
  tempW = XtNameToWidget( mainW, "ActionRc.AlarmPb" );
  XtSetSensitive( tempW, alarms );

  tempW = XtNameToWidget( mainW, "ActionRc.StandPb" );
  XtSetSensitive( tempW, standing );


  return;

} /* updateCanDo */


/*----------------------------------------------------------------------*/

static Boolean
  writeEditedEntry( XTM_GL_BASE_DATA_REF  appl_data_ref,
                    Widget                parent,
                    XTM_DB_ALL_ENTRY_REF  old_entry_ref,
                    XTM_DB_ALL_ENTRY_REF  new_entry_ref )
{

  /* Variables. */
  Boolean                 ok;
  Boolean                 remove_old_entry = False;
  Boolean                 use_same_db = False;
  UINT32                  id;
  char                    *new_db;
  char                    *old_db;
  XTM_DB_ID_REQUEST       id_request;
  XTM_DB_STATUS           db_status;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_ENTRY_DATABASES  *database_ref = NULL;


  /* Code. */

  old_db = old_entry_ref -> db_name;
  new_db = new_entry_ref -> db_name;

  if( strcmp( old_db, new_db ) == 0 )
    use_same_db = True;


  /* Are we changing an existing entry? */
  if( old_entry_ref -> entry.id > 0 )
    remove_old_entry = True;


  /* Remove the old entry? */
  if( remove_old_entry ) {

    database_ref = &database;

    /* Open the database. */
    ok = xtmDmOpenDatabase( appl_data_ref, old_db, XTM_DB_FLAG_MODE_WRITE,
                            database_ref );
    if( ! ok )
      return( False );

    (void) xtmDbDeleteEntry( database_ref, old_entry_ref -> entry.id );

    /* Let the database stay open? */
    if( ! use_same_db ) {
      xtmDbCloseEntryDb( database_ref );
      database_ref = NULL;
    }

  } /* if */


  /* Save the new entry. */

  /* Do we need to open the database. */
  if( database_ref == NULL ) {

    database_ref = &database;

    /* Open the database. */
    ok = xtmDmOpenDatabase( appl_data_ref, new_db, XTM_DB_FLAG_MODE_WRITE,
                            database_ref );
    if( ! ok )
      return( False );

  } /* if */


  /* Generate new entry id. */
  id_request.directory  = database_ref -> database_dir;
  id_request.operations = 0;
  id_request.lock_file  = False;

  db_status = xtmDbGenerateId( &id_request, &id );
  if( db_status != XTM_DB_OK )
      xitErMessage( NULL, XIT_ER_FATAL, 
                    module_name, "writeEditedEntry",
                    msgGetText( MXDI_ERRMSG_GENERATE_ID ) );

  new_entry_ref -> entry.id       = id;
  new_entry_ref -> stand_entry.id = id;

  /* Write the entry. */
  db_status = xtmDbInsertEntry( database_ref,
                                new_entry_ref, new_entry_ref -> all_text );
  if( db_status != XTM_DB_OK )
      xitErMessage( NULL, XIT_ER_FATAL, 
                    module_name, "writeEditedEntry",
                    msgGetText( MXDI_ERRMSG_INSERT_ENTRY ) );

  /* Close the database. */
  xtmDbCloseEntryDb( database_ref );


  return( True );

} /* writeEditedEntry */


/*----------------------------------------------------------------------*/

static void 
  writeEntryDate( EDITOR_REC_REF editor_ref )
{

  /* Variables. */
  Boolean          ok;
  char             buffer[ 50 ];
  char             time_buffer[ 50 ];
  Widget           mainW;
  Widget           tempW;
  ENTRY_TIMES_REC  entry_times;


  /* Code. */

  mainW = XtNameToWidget( editor_ref -> editorW, 
                          "ApEditorTlBase.ApEditorTlFo" );

  /* Blank out the current entry date. */
  tempW = XtNameToWidget( mainW, "EntryDateLa" );
  xitStringSetLabel( tempW, " \n " );


  /* Fetch the date, start time and duration. */
  ok = fetchEntryTimes( editor_ref, NULL, &entry_times );
  if( ! ok )
    return;


  time_buffer[ 0 ] = '\0';

  /* Build the entry date */
  if( entry_times.date > 0 ) {
    xtmFoFormatDate( entry_times.date, buffer, sizeof( buffer ) );

    strcat( time_buffer, buffer );
    strcat( time_buffer, ", " );

    TimFormatStrTime( entry_times.date, "%A", buffer, sizeof( buffer ) );
    strcat( time_buffer, buffer );
  }

  strcat( time_buffer, "\n" );

  /* Build the start time. */
  if( entry_times.start_time > 0 ) {
    xtmFoFormatTime( entry_times.start_time, buffer, sizeof( buffer ) );

    strcat( time_buffer, buffer );
  }

  /* Build the end time. */
  if( entry_times.end_time > 0 ) {
    xtmFoFormatTime( entry_times.end_time, buffer, sizeof( buffer ) );

    strcat( time_buffer, " - " );
    strcat( time_buffer, buffer );
  }

  /* Set the new time. */
  tempW = XtNameToWidget( mainW, "EntryDateLa" );
  xitStringSetLabel( tempW, time_buffer );


  return;

} /* writeEntryDate */


/*----------------------------------------------------------------------*/

static void 
  alarmCB( Widget          widget,
           EDITOR_REC_REF  editor_ref,
           XtPointer       call_data )
{

  /* Variables. */
  int                    index;
  XTM_DB_ALL_ENTRY_REF   edit_entry_ref;
  XTM_EA_SET_ALARM_DATA  values;


  /* Code. */

  edit_entry_ref = editor_ref -> edit_entry_ref;

  /* If the window is not created, do it. */
  if( editor_ref -> alarm_handle == NULL )
    editor_ref -> alarm_handle = xtmEaInitialize( editor_ref -> appl_data_ref,
                                                  editor_ref -> editorW,
                                                  alarmActionCB,
                                                  (void *) editor_ref );

  /* Fill the window with data and display it. */
  for( index = 0; index < XTM_EA_MAX_ALARMS; index++ ) {
    values.alarm_valid[ index ] = 
      (Boolean) edit_entry_ref -> entry.alarm_valid[ index ];

    values.alarm_offset[ index ] = 
      (int) edit_entry_ref -> entry.alarm_offset[ index ];
  }

  values.alarm_melody = (int) edit_entry_ref -> entry.alarm_melody;
  values.alarm_action = edit_entry_ref -> entry.flags;
  values.what_action  = edit_entry_ref -> entry.flags;

  xtmEaEditAlarms( editor_ref -> alarm_handle,
                   &values );


  return;

} /* alarmCB */


/*----------------------------------------------------------------------*/

static void 
  alarmActionCB( XTM_EA_REASON          reason,
                 void                   *user_data,
                 XTM_EA_GET_ALARM_DATA  *values )
{

  /* Variables. */
  int                    index;
  INT16                  min_offset;
  XTM_DB_ALL_ENTRY_REF   edit_entry_ref;
  EDITOR_REC_REF         editor_ref;


  /* Code. */

  if( user_data == NULL )
    return;

  /* Our private data. */
  editor_ref = (EDITOR_REC_REF) user_data;

  edit_entry_ref = editor_ref -> edit_entry_ref;


  /* Is the window destroyed? */
  if( reason == XTM_EA_REASON_DESTROY ) {
    editor_ref -> alarm_handle = NULL;

    return;
  }


  /* OK confirmation? */
  if( reason == XTM_EA_REASON_OK ) {

    for( index = 0; index < XTM_EA_MAX_ALARMS; index++ ) {
      edit_entry_ref -> entry.alarm_valid[ index ] =
        (UINT8) values -> alarm_valid[ index ];

      min_offset = (INT16) values -> alarm_offset[ index ];
      edit_entry_ref -> entry.alarm_offset[ index ] = min_offset;
    }

    edit_entry_ref -> entry.alarm_melody = (UINT8) values -> alarm_melody;

    flagClear( edit_entry_ref -> entry.flags,
               (XTM_DB_FLAG_ACTION_ALARM1 | XTM_DB_FLAG_ACTION_ALARM2 |
                XTM_DB_FLAG_ACTION_ALARM3 | XTM_DB_FLAG_ACTION_ALARM4 |
                XTM_DB_FLAG_ACTION_ALARM5) );

    flagClear( edit_entry_ref -> entry.flags,
               (XTM_DB_FLAG_ACTION_SCRIPT | XTM_DB_FLAG_ACTION_TEXT ) );

    flagSet( edit_entry_ref -> entry.flags, values -> alarm_action );
    flagSet( edit_entry_ref -> entry.flags, values -> what_action );

  } /* if */


  return;

} /* alarmActionCB */


/*----------------------------------------------------------------------*/

static void 
  applyCB( Widget          widget,
           EDITOR_REC_REF  editor_ref,
           XtPointer       call_data )
{

  /* Variables. */
  Boolean  ok;
  Widget   tempW;


  /* Code. */

  /* Save the entry. */
  ok = saveEntry( editor_ref );
  if( ! ok )
    return;

  /* Do we have a user action callback registered? */
  if( editor_ref -> actionCB != NULL )
    (* editor_ref -> actionCB)( XTM_ED_REASON_UPDATE, 
                                editor_ref -> user_data );

  /* Make sure the user knows the entry is saved. */
  tempW = xitCreateInformationDialog( 
            editor_ref -> editorW, "InformationDialog", 
	    msgGetText( MXDI_INFORMATION_LABEL ),
            msgGetText( MXDI_ENTRY_IS_SAVED ),
            NULL, NULL );

  XmUpdateDisplay( editor_ref -> editorW );

  /* Update the day list/calendar. */
  xtmUpDoUpdate( (XTM_UP_CALENDAR | XTM_UP_SCHEDULE | XTM_UP_PLANNER), NULL );


  return;

} /* applyCB */


/*----------------------------------------------------------------------*/

static void 
  cancelCB( Widget          widget,
            EDITOR_REC_REF  editor_ref,
            XtPointer       call_data )
{

  /* Variables. */
  int  index;


  /* Code. */

  removeEditFile( editor_ref );

  if( editor_ref -> actionCB != NULL )
    (* editor_ref -> actionCB)( XTM_ED_REASON_DESTROY,
                                editor_ref -> user_data );

  /* Keep the window in cache? */
  for( index = 0; index < MAX_CACHE_ENTRIES; index++ ) {
    if( cache_entries[ index ] == NULL ) {
      cache_entries[ index ] = editor_ref;

      destroyCB( NULL, editor_ref, NULL );
      XtPopdown( editor_ref -> editorW );

      return;
    }
  }

  /* Window not kept in cache, really remove it. */
  XtDestroyWidget( editor_ref -> editorW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void 
  dayViewCB( Widget          widget,
             EDITOR_REC_REF  editor_ref,
             XtPointer       call_data )
{

  /* Variables. */
  Boolean                 ok;
  char                    *char_ref;
  Widget                  mainW;
  Widget                  tempW;
  TIM_TIME_REF            entry_date;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;
  XTM_SM_HANDLE           sched_handle;


  /* Code. */

  appl_data_ref   = editor_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;


  mainW = XtNameToWidget( editor_ref -> editorW, 
                          "ApEditorTlBase.ApEditorTlFo" );

  /* Fetch the date we want to use. */
  tempW = XtNameToWidget( mainW, "TimeEntFr.DataEntFo.TimeEntRc.DateEntTb" );

  ok = xtmFoFetchDate( mainW, tempW, XTM_FO_START_DATE, False, &entry_date );
  if( ! ok )
    return;


  /* Display the day view for this date. */
  ok = xtmSmViewLastUsed( entry_date );
  if( ok )
    return;


  /* No previous day view found. */
  (void) xtmCdFetchDefaultDb( custom_data_ref -> cal_db_handle,
                              &db_info, NULL );

  sched_handle = xtmSmInitialize( appl_data_ref,
                                  appl_data_ref -> toplevel,
                                  db_info.short_name, NULL,
                                  NULL, NULL );

  /* Give an error message if the database could not be opened. */
  if( sched_handle == NULL ) {
    xitErMessage( NULL, XIT_ER_ERROR,
                  module_name, "dayViewCB",
                  msgGetText( MXDI_ERRMSG_NO_ACCESS_DB ) );
    return;
  }

  /* Display the schedule. */
  xtmSmViewSchedule( sched_handle, entry_date, False );
  

  return;

} /* dayViewCB */


/*----------------------------------------------------------------------*/

static void 
  dbChangedCB( Widget          widget,
               EDITOR_REC_REF  editor_ref,
               XtPointer       call_data )
{

  /* Variables. */
  Boolean                 ok;
  char                    *db_name;
  Widget                  baseW;
  Widget                  mainW;
  Widget                  tempW;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  custom_data_ref = editor_ref -> appl_data_ref -> custom_data;

  mainW = XtNameToWidget( editor_ref -> editorW, 
                          "ApEditorTlBase.ApEditorTlFo" );

  baseW = XtNameToWidget( mainW, "MiscFo.MiscRc" );


  /* Fetch the name of the new database (we have write access?). */
  tempW = XtNameToWidget( baseW, "DatabaseFs" );
  xitFieldSelectGetCurrent( tempW, &db_name );


  /* Database info. */
  ok = xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, db_name,
                          &db_info, NULL );
  SysFree( db_name );

  if( ! ok )
    return;

  if( ! flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_WRITE ) )
    return;


  /* Can we save as private entry. */
  baseW = XtNameToWidget( mainW, "SelFr.SelFo" );
  tempW = XtNameToWidget( baseW, "SelPrivateBu" );

  if( ! flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_PRIV ) ) {
    XmToggleButtonSetState( tempW, False, False );
    XtSetSensitive( tempW, False );
  } else {
    XtSetSensitive( tempW, True );
  }


  /* We can now save the entry. */
  baseW = XtNameToWidget( editor_ref -> editorW, "ApEditorTlBase" );

  tempW = XtNameToWidget( baseW, "Bu1" );
  XtMapWidget( tempW );

  tempW = XtNameToWidget( baseW, "Bu3" );
  XtMapWidget( tempW );


  return;

} /* dbChangedCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget          widget,
             EDITOR_REC_REF  editor_ref,
             XtPointer       call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( editor_ref -> actionCB != NULL )
    (* editor_ref -> actionCB)( XTM_ED_REASON_DESTROY,
                                editor_ref -> user_data );

  /* Remove any editor? */
  removeEditFile( editor_ref );

  /* File selection. */
  if( editor_ref -> fileSelectW != NULL )
    XtDestroyWidget( editor_ref -> fileSelectW );

  /* Release the user data. */
  if( editor_ref -> edit_entry_ref != NULL ) {
    SysFree( editor_ref -> edit_entry_ref -> all_text );
    SysFree( editor_ref -> edit_entry_ref );
  }


  /* Release the user data (only if not cached). */
  if( widget != NULL )
    SysFree( editor_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  editorCB( Widget          widget,
            EDITOR_REC_REF  editor_ref,
            XtPointer       call_data )
{

  /* Variables. */
  int                     file_ref;
  char                    *char_ref;
  char                    *command;
  char                    *entry_text;
  char                    *process_args[ 10 ];
  Widget                  mainW;
  Widget                  tempW;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = editor_ref -> appl_data_ref -> custom_data;

  mainW = XtNameToWidget( editor_ref -> editorW, 
                          "ApEditorTlBase.ApEditorTlFo" );

  /* Fetch the text we are editing. */
  tempW = XtNameToWidget( mainW, "TextEntTxSW.TextEntTx" );
  entry_text = xitStringGetText( tempW );


  /* Save the text in a working file. */
  char_ref = tmpnam( NULL );
  editor_ref -> temp_file = SysNewString( char_ref );

  file_ref = creat( editor_ref -> temp_file, (mode_t) MODE_600 );
  if( file_ref < 0 ) {
    xitErMessage( editor_ref -> editorW, XIT_ER_ERROR,
                  module_name, "editorCB",
                  msgGetText( MXDI_CANNOT_WRITE_TMP_FILE ) );

    SysFree( entry_text );
    return;
  }

  (void) lseek( file_ref, 0, SEEK_SET );
  (void) write( file_ref, entry_text, strlen( entry_text ) );
  close( file_ref );

  SysFree( entry_text );


  /* Start the editor with the temporary file. */
  command = (char *) SysMalloc( strlen( custom_data_ref -> user_editor ) +
                                strlen( editor_ref -> temp_file ) + 20 );
  sprintf( command, custom_data_ref -> user_editor, 
           editor_ref -> temp_file );

  process_args[ 0 ] = "/bin/sh";
  process_args[ 1 ] = "-c";
  process_args[ 2 ] = command;
  process_args[ 3 ] = NULL;

  editor_ref -> editor_pid = xtmToStartProcess(
                               editor_ref -> editorW, True,
                               msgGetText( MXDI_STARTING_PROCESS_MSG ),
                               process_args );
  SysFree( command );

  if( editor_ref -> editor_pid <= 0 )
    raise exception;

  /* We want to know when the editor exits. */
  editor_ref -> signal_id = SigRegister( SIGCHLD, editor_ref -> editor_pid, 
                                         userEditorDoneCB, editor_ref );


  /* Don't call the editor or change the text field. */
  tempW = XtNameToWidget( mainW, "ActionRc.EditorPb" );
  XtSetSensitive( tempW, False );

  tempW = XtNameToWidget( mainW, "TextEntTxSW.TextEntTx" );
  XtSetSensitive( tempW, False );


  return;


  /* Exception handler. */
  exception:

    /* We don't need the temporary file. */
    removeEditFile( editor_ref );

    return;

} /* editorCB */


/*----------------------------------------------------------------------*/

static void 
  entryTypeCB( Widget                        widget,
               EDITOR_REC_REF                editor_ref,
               XmToggleButtonCallbackStruct  *call_data )
{

  /* Variables. */
  Boolean  clear_time = False;
  Widget   baseW;
  Widget   mainW;
  Widget   tempW;


  /* Code. */

  /* Only react when buttons are selected. */
  if( ! call_data -> set )
    return;


  mainW = XtNameToWidget( editor_ref -> editorW, 
                          "ApEditorTlBase.ApEditorTlFo" );

  baseW = XtNameToWidget( mainW, "TimeEntFr.DataEntFo.TypeRb" );


  editor_ref -> entry_type = XTM_DB_TYPE_APPOINTMENT;
  
  tempW = XtNameToWidget( baseW, "TypeStandAppTb" );
  if( widget == tempW ) {
    editor_ref -> entry_type = XTM_DB_TYPE_STAND_APPOINTMENT;
  }

  tempW = XtNameToWidget( baseW, "TypeNoteTb" );
  if( widget == tempW ) {
    clear_time = True;
    editor_ref -> entry_type = XTM_DB_TYPE_NOTE;
  }

  tempW = XtNameToWidget( baseW, "TypeStandNoteTb" );
  if( widget == tempW ) {
    clear_time = True;
    editor_ref -> entry_type = XTM_DB_TYPE_STAND_NOTE;
  }

  tempW = XtNameToWidget( baseW, "TypeStickyNoteTb" );
  if( widget == tempW ) {
    clear_time = True;
    editor_ref -> entry_type = XTM_DB_TYPE_STICKY_NOTE;
  }

  /* Clear the time fields? */
  if( clear_time ) {
    baseW = XtNameToWidget( mainW, "TimeEntFr.DataEntFo.TimeEntRc" );

    tempW = XtNameToWidget( baseW, "StartEntTb" );
    XmUbTimeBoxSetStartTimeString( tempW, "" );

    tempW = XtNameToWidget( baseW, "DurEntFs" );
    xitFieldSelectSetCurrent( tempW, "", False );
  }

  /* What can we do? */
  updateCanDo( editor_ref );


  return;

} /* entryTypeCB */


/*----------------------------------------------------------------------*/

static void 
  includeFileCB( Widget          widget,
                 EDITOR_REC_REF  editor_ref,
                 XtPointer       call_data )
{

  /* Variables. */
  XIT_FILE_SEL_LABELS  labels;


  /* Code. */

  /* Do we need to create the dialog? */
  if( editor_ref -> fileSelectW == NULL ) {

    labels.dir_list_label       = msgGetText( MXDI_DIRECTORY_LABEL );
    labels.file_list_label      = msgGetText( MXDI_FILES_LABEL );
    labels.filter_label         = msgGetText( MXDI_FILTER_ACTION_LABEL );
    labels.selection_label      = msgGetText( MXDI_SELECTION_LABEL );
    labels.cancel_button        = msgGetText( MXDI_CANCEL_BUTTON );
    labels.filter_button        = msgGetText( MXDI_FILTER_ACTION_LABEL );
    labels.ok_button            = msgGetText( MXDI_OK_BUTTON );
    labels.no_file_sel_msg      = msgGetText( MXDI_NO_FILE_SELECTED );
    labels.cannot_read_file_msg = msgGetText( MXDI_CANNOT_READ_FILE );

    /* The file selection dialog. */
    editor_ref -> fileSelectW = 
      xitCreateFileSelect( editor_ref -> editorW, "FileSelect",
                           0, "", &labels,
                           includeFileActionCB, (void *) editor_ref );

  } /* if */

  /* Display the dialog. */
  xitFileSelectDisplay( editor_ref -> fileSelectW );


  return;

} /* includeFileCB */


/*----------------------------------------------------------------------*/

static void 
  includeFileActionCB( XIT_FILE_SEL_REASON  reason,
                       char                 *file_name,
                       void                 *user_data )
{

  /* Variables. */
  Widget          mainW;
  Widget          tempW;
  EDITOR_REC_REF  editor_ref;


  /* Code. */

  editor_ref = (EDITOR_REC_REF) user_data;

  mainW = XtNameToWidget( editor_ref -> editorW, 
                          "ApEditorTlBase.ApEditorTlFo" );

  /* Only OK reasons. */
  if( reason != XIT_FILE_SELECT_OK )
    return;

  tempW = XtNameToWidget( mainW, "TextEntTxSW.TextEntTx" );
  xitStringSetTextFromFile( tempW, file_name );


  return;

} /* includeFileActionCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget                     widget,
          EDITOR_REC_REF             editor_ref,
          XmRowColumnCallbackStruct  *call_data )
{

  /* Code. */

  xtmHlDisplayHelp( editor_ref -> appl_data_ref -> info_handle,
                    (int) call_data -> data,
                    edit_window_id, "" );

  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void 
  insertEditedFile( EDITOR_REC_REF  editor_ref )
{

  /* Variables. */
  int          file_ref;
  int          status;
  char         *char_ref;
  Widget       mainW;
  Widget       tempW;
  struct stat  file_info;


  /* Code. */

  mainW = XtNameToWidget( editor_ref -> editorW, 
                          "ApEditorTlBase.ApEditorTlFo" );

  /* Make sure the window is visible. */
  XRaiseWindow( XtDisplay( editor_ref -> editorW ), 
                XtWindow(  editor_ref -> editorW ) );

  XtMapWidget( editor_ref -> editorW );


  /* Any temporary file? */
  if( editor_ref -> temp_file == NULL )
    return;


  /* We can edit the text field again. */
  tempW = XtNameToWidget( mainW, "ActionRc.EditorPb" );
  XtSetSensitive( tempW, True );

  tempW = XtNameToWidget( mainW, "TextEntTxSW.TextEntTx" );
  XtSetSensitive( tempW, True );


  /* Information about the file. */
  status = stat( editor_ref -> temp_file, &file_info );
  if( status != 0 )
    raise exception;


  /* Go and get the file. */
  file_ref = open( editor_ref -> temp_file, O_RDONLY );
  if( file_ref < 0 ) {
    xitErMessage( editor_ref -> editorW, XIT_ER_ERROR,
                  module_name, "editorCB",
                  msgGetText( MXDI_CANNOT_READ_TMP_FILE ) );

    raise exception;
  }

  char_ref = (char *) SysMalloc( file_info.st_size + 50 );
  (void) read( file_ref, char_ref, file_info.st_size );

  close( file_ref );

  *(char_ref + file_info.st_size) = '\0';

  /* Save the text. */
  tempW = XtNameToWidget( mainW, "TextEntTxSW.TextEntTx" );
  XmTextSetString( tempW, char_ref );

  SysFree( char_ref );


  /* We don't need the temporary file. */
  removeEditFile( editor_ref );

  return;


  /* Exception handler. */
  exception:

    /* We don't need the temporary file. */
    removeEditFile( editor_ref );

    return;

} /* insertEditedFile */


/*----------------------------------------------------------------------*/

static void 
  okCB( Widget          widget,
        EDITOR_REC_REF  editor_ref,
        XtPointer       call_data )
{

  /* Variables. */
  Boolean  ok;


  /* Code. */

  /* Save the entry. */
  ok = saveEntry( editor_ref );
  if( ! ok )
    return;

  /* Do we have a user action callback registered? */
  if( editor_ref -> actionCB != NULL )
    (* editor_ref -> actionCB)( XTM_ED_REASON_UPDATE,
                                editor_ref -> user_data );


  /* Procedd as when cancel. */
  cancelCB( widget, editor_ref, NULL );

  XmUpdateDisplay( editor_ref -> editorW );


  /* Update the day list/calendar. */
  xtmUpDoUpdate( (XTM_UP_CALENDAR | XTM_UP_SCHEDULE | XTM_UP_PLANNER), NULL );


  return;

} /* okCB */


/*----------------------------------------------------------------------*/

static void 
  preferencesCB( Widget          widget,
                 EDITOR_REC_REF  editor_ref,
                 XtPointer       call_data )
{

  /* Variables. */
  XTM_DB_ALL_ENTRY_REF  edit_entry_ref;
  XTM_EP_SET_PREF_DATA  values;


  /* Code. */

  edit_entry_ref = editor_ref -> edit_entry_ref;

  /* If the window is not created, do it. */
  if( editor_ref -> pref_handle == NULL )
    editor_ref -> pref_handle = xtmEpInitialize( editor_ref -> appl_data_ref,
                                                 editor_ref -> editorW,
                                                 preferencesActionCB,
                                                 (void *) editor_ref );

  /* Fill the window with data and display it. */
  values.alarm_lines    = (int) edit_entry_ref -> entry.alarm_lines;
  values.day_list_lines = (int) edit_entry_ref -> entry.day_list_lines;
  values.bg_color_index = (int) edit_entry_ref -> entry.bg_color_index;
  values.fg_color_index = (int) edit_entry_ref -> entry.fg_color_index;
  values.flags          = edit_entry_ref -> entry.flags;
  values.entry_type     = editor_ref -> entry_type;

  xtmEpEditPreferences( editor_ref -> pref_handle,
                        &values );


  return;

} /* preferencesCB */


/*----------------------------------------------------------------------*/

static void 
  preferencesActionCB( XTM_EP_REASON         reason,
                       void                  *user_data,
                       XTM_EP_GET_PREF_DATA  *values )
{

  /* Variables. */
  XTM_DB_ALL_ENTRY_REF  edit_entry_ref;
  EDITOR_REC_REF        editor_ref;


  /* Code. */

  if( user_data == NULL )
    return;

  /* Our private data. */
  editor_ref = (EDITOR_REC_REF) user_data;

  edit_entry_ref = editor_ref -> edit_entry_ref;


  /* Is the window destroyed? */
  if( reason == XTM_EP_REASON_DESTROY ) {
    editor_ref -> pref_handle = NULL;

    return;
  }


  /* OK confirmation? */
  if( reason == XTM_EP_REASON_OK ) {

    edit_entry_ref -> entry.alarm_lines    = (UINT8) values -> alarm_lines;
    edit_entry_ref -> entry.day_list_lines = (UINT8) values -> day_list_lines;
    edit_entry_ref -> entry.bg_color_index = (UINT8) values -> bg_color_index;
    edit_entry_ref -> entry.fg_color_index = (UINT8) values -> fg_color_index;

    flagClear( edit_entry_ref -> entry.flags,
               (XTM_DB_FLAG_HIDE_IN_CALENDAR | 
                XTM_DB_FLAG_HIDE_IN_DAY_VIEW |
                XTM_DB_FLAG_HIDE_IN_SUMMARY |
                XTM_DB_FLAG_HIDE_IN_PRINT |
                XTM_DB_FLAG_HIDE_IN_DUMP |
                XTM_DB_FLAG_HIDE_IN_ALARM |
                XTM_DB_FLAG_ONLY_OWNER_DELETE |
                XTM_DB_FLAG_ONLY_OWNER_CHANGE |
                XTM_DB_FLAG_KEEP_IN_BG |
                XTM_DB_FLAG_KEEP_IN_FG) );

    flagSet( edit_entry_ref -> entry.flags, values -> flags );

  } /* if */


  return;

} /* preferencesActionCB */


/*----------------------------------------------------------------------*/

static void 
  standingCB( Widget          widget,
              EDITOR_REC_REF  editor_ref,
              XtPointer       call_data )
{

  /* Variables. */
  int                    index;
  XTM_DB_ALL_ENTRY_REF   edit_entry_ref;
  XTM_ES_SET_STAND_DATA  values;


  /* Code. */

  editor_ref -> standing_seen = True;

  edit_entry_ref = editor_ref -> edit_entry_ref;


  /* If the window is not created, do it. */
  if( editor_ref -> stand_handle == NULL )
    editor_ref -> stand_handle = xtmEsInitialize( editor_ref -> appl_data_ref,
                                                  editor_ref -> editorW,
                                                  standingActionCB,
                                                  (void *) editor_ref );

  /* Fill the window with data and display it. */
  for( index = 0; index < 7; index++ )
    values.valid_days[ index ] = 
      (Boolean) edit_entry_ref -> stand_entry.valid_days[ index ];

  values.every_n     = (int) edit_entry_ref -> stand_entry.every_n;
  values.valid_every = (int) edit_entry_ref -> stand_entry.valid_every;

  values.from = (TIM_TIME_REF) edit_entry_ref -> stand_entry.from;
  values.to   = (TIM_TIME_REF) edit_entry_ref -> stand_entry.to;

  values.flags          = edit_entry_ref -> stand_entry.flags;
  values.skip_week[ 0 ] = edit_entry_ref -> stand_entry.skip_week[ 0 ];
  values.skip_week[ 1 ] = edit_entry_ref -> stand_entry.skip_week[ 1 ];

  values.default_date = editor_ref -> default_date;
  values.entry_type   = editor_ref -> entry_type;

  xtmEsEditStanding( editor_ref -> stand_handle, &values );


  return;

} /* standingCB */


/*----------------------------------------------------------------------*/

static void 
  standingActionCB( XTM_ES_REASON          reason,
                    void                   *user_data,
                    XTM_ES_GET_STAND_DATA  *values )
{

  /* Variables. */
  int                  index;
  XTM_DB_ALL_ENTRY_REF edit_entry_ref;
  EDITOR_REC_REF       editor_ref;


  /* Code. */

  if( user_data == NULL )
    return;

  /* Our private data. */
  editor_ref = (EDITOR_REC_REF) user_data;

  edit_entry_ref = editor_ref -> edit_entry_ref;


  /* Is the window destroyed? */
  if( reason == XTM_ES_REASON_DESTROY ) {
    editor_ref -> stand_handle = NULL;

    return;
  }


  /* OK confirmation? */
  if( reason == XTM_ES_REASON_OK ) {

    for( index = 0; index < 7; index++ )
      edit_entry_ref -> stand_entry.valid_days[ index ] =
        (UINT8) values -> valid_days[ index ];

    edit_entry_ref -> stand_entry.every_n     = (UINT16) values -> every_n;
    edit_entry_ref -> stand_entry.valid_every = (UINT8)  values -> valid_every;

    edit_entry_ref -> stand_entry.from = (UINT32) values -> from;
    edit_entry_ref -> stand_entry.to   = (UINT32) values -> to;

    edit_entry_ref -> stand_entry.flags          = values -> flags;
    edit_entry_ref -> stand_entry.skip_week[ 0 ] = values -> skip_week[ 0 ];
    edit_entry_ref -> stand_entry.skip_week[ 1 ] = values -> skip_week[ 1 ];

  } /* if */


  return;

} /* standingActionCB */


/*----------------------------------------------------------------------*/

static void
  timeFieldChangedCB( Widget          widget,
                      EDITOR_REC_REF  editor_ref,
                      XtPointer       call_data )
{

  /* Code. */

  writeEntryDate( editor_ref );


  return;

} /* timeFieldChangedCB */


/*----------------------------------------------------------------------*/

static void 
  userEditorDoneCB( int   signal,
                    void  *user_data )
{

  /* Variables. */
  EDITOR_REC_REF  editor_ref;


  /* Code. */

  editor_ref = (EDITOR_REC_REF) user_data;

  /* No signal notifications any more. */
  SigRemove( editor_ref -> signal_id );

  /* Start a 'NULL' timer to do the update. */
  if( editor_ref -> editor_pid != 0 )
    XtAppAddTimeOut( editor_ref -> appl_data_ref -> context, 0, 
                     (XtTimerCallbackProc) insertEditedFile,
                     (XtPointer) editor_ref );

  editor_ref -> editor_pid = 0;


  return;

} /* userEditorDoneCB */
