/*----------------------------------------------------------------------------
--
--  Module:           xtmDbMisc
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    High-level interface to the diary database. Provides functions that
--    work on databses with include lists.
--
--  Filename:         xtmDbMisc.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-01-09
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmDbMisc.c, Version: 1.1, Date: 95/02/18 15:52:09";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <malloc.h>
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>

#include <X11/Intrinsic.h>
#include <X11/cursorfont.h>

#include <Xm/Xm.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmAccBase.h"
#include "xtmCalDb.h"
#include "xtmCustBase.h"
#include "xtmDbTools.h"
#include "xitError.h"
#include "xitTools.h"
#include "xtmDbMisc.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Priorities. */
#define  PRIO_TAG             (1<<0)
#define  PRIO_NOTE_NOT_DONE   (1<<1)
#define  PRIO_IMPORTANT       (1<<2)


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmDbMisc";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  checkEntries( XTM_GL_BASE_DATA_REF  appl_data_ref,
                char                  *db_name,
                TIM_TIME_REF          date,
                int                   days,
                Boolean               show_standing,
                Boolean               entry_defined[] );

static void
  deleteEntryCB( void  *element );

static LST_COMPARE
  entrySortFunc( XTM_DB_ALL_ENTRY_REF  list_entry_ref,
                 XTM_DB_ALL_ENTRY_REF  search_entry_ref );

static XTM_DB_STATUS
  fetchFilteredEntriesDay( XTM_GL_BASE_DATA_REF  appl_data_ref,
                           char                  *db_name,
                           UINT32                flags,
                           XtAppContext          context,
                           Widget                workW,
                           TIM_TIME_REF          starts,
                           TIM_TIME_REF          ends,
                           XTM_DM_FILTER_REC     *filter_ref,
                           LST_DESC_TYPE         entries );

static Boolean
  searchString( char     *search_string,
                Boolean  case_sens1,
                char     *in_string,
                Boolean  case_sens2 );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void
  xtmDmDeleteEntriesList( LST_DESC_TYPE entries )
{

  /* Code. */

  LstLinkClearDataAndList( entries, deleteEntryCB );


  return;

} /* xtmDmDeleteEntriesList */


/*----------------------------------------------------------------------*/

Boolean
  xtmDmDeleteEntry( Widget                  parentW,
                    XTM_DB_ENTRY_DATABASES  *database,
                    UINT32                  entry_id )
{

  /* Variables. */
  XTM_DB_STATUS  db_status;


  /* Code. */

  /* Delete the entry. */
  db_status = xtmDbDeleteEntry( database, entry_id );

  if( db_status != XTM_DB_OK ) {
    xitErMessage( parentW, XIT_ER_ERROR,
                  module_name, "xtmDmDeleteEntry",
                  msgGetText( MXDI_ERRMSG_DELETE_ENTRY ) );

    return( False );
  }


  return( True );

} /* xtmDmDeleteEntry */


/*----------------------------------------------------------------------*/

Boolean
  xtmDmFetchEntry( Widget                  parentW,
                   XTM_DB_ENTRY_DATABASES  *database,
                   UINT32                  entry_id,
                   XTM_DB_ALL_ENTRY_REF    entry_ref,
                   char                    **entry_text )
{

  /* Variables. */
  XTM_DB_STATUS  db_status;


  /* Code. */

  /* Fetch the entry. */
  db_status = xtmDbFetchEntry( database,  entry_id, 
                               entry_ref, entry_text );

  if( db_status != XTM_DB_OK ) {
    xitErMessage( parentW, XIT_ER_ERROR,
                  module_name, "xtmDmFetchEntry",
                  msgGetText( MXDI_NO_ENTRY ) );

    return( False );
  }


  return( True );

} /* xtmDmFetchEntry */


/*----------------------------------------------------------------------*/

XTM_DB_STATUS
  xtmDmFetchFilteredEntriesRange( XTM_GL_BASE_DATA_REF  appl_data_ref,
                                  char                  *db_name,
                                  XtAppContext          context,
                                  Widget                parent,
                                  TIM_TIME_REF          starts,
                                  TIM_TIME_REF          ends,
                                  UINT32                flags,
                                  XTM_DM_FILTER_REC     *filter_ref,
                                  LST_DESC_TYPE         *entries )
{

  /* Variables. */
  Boolean                 ok;
  int                     index;
  char                    buffer[ 100 ];
  char                    *char_ref;
  Cursor                  wait_cursor;
  Widget                  workW = NULL;
  TIM_DELTA_TYPE          delta;
  XTM_DB_STATUS           db_status;
  XTM_GL_CUSTOM_DATA_REF  custom_data;
  XTM_CD_CAL_INFO         db_incl_info;
  XTM_CD_CAL_INFO         db_info;
  XTM_CD_INCL_CALS        db_incl;


  /* Code. */

  custom_data = appl_data_ref -> custom_data;
  *entries    = NULL;

  ok = xtmCdFetchNamedDb( custom_data -> cal_db_handle, db_name,
                          &db_info, &db_incl );
  if( ! ok )
    return( XTM_DB_ERROR );


  *entries = LstLinkNew( sizeof( XTM_DB_ALL_ENTRY_DEF ), NULL );

  /* Initialize search strings. */
  if( flagIsClear( filter_ref -> flags, XTM_DM_FLAG_TEXT_CASE_SENS ) ) {
    char_ref = filter_ref -> text_string;

    while( *char_ref != '\0' ) {
      *char_ref = toupper( *char_ref );
      char_ref++;
    }
  }

  if( flagIsClear( filter_ref -> flags, XTM_DM_FLAG_TAG_CASE_SENS ) ) {
    char_ref = filter_ref -> tag_string;

    while( *char_ref != '\0' ) {
      *char_ref = toupper( *char_ref );
      char_ref++;
    }
  }


  /* Display a working form? */
  TimDelta( starts, ends, &delta );

  if( parent != NULL ) {
    if( delta.days > 31 ) {
      workW = xitCreateBusyDialog(
                parent, "WorkingDialog",
                msgGetText( MXDI_WORKING_TITLE ),
                msgGetText( MXDI_WORKING_MESSAGE ) );
    } else {
      wait_cursor = XCreateFontCursor( XtDisplay( parent ), XC_watch );
      XDefineCursor( XtDisplay( parent ), XtWindow( parent ), wait_cursor );

      XFlush( XtDisplay( parent ) );
    }
  }

  /* Process the main database. */
  db_status = fetchFilteredEntriesDay( appl_data_ref, 
                                       db_name, flags,
                                       context, workW,
                                       starts, ends, 
                                       filter_ref, *entries );
  if( db_status != XTM_DB_OK )
    raise exception;

  /* Process the included databases. */
  if( db_status == XTM_DB_OK &&
      db_incl.no > 0 && 
      flagIsClear( flags, XTM_DB_FETCH_SKIP_INCLUDE ) ) {

    flagSet( flags, XTM_DB_FETCH_INCLUDE );

    for( index = 0; index < db_incl.no; index++ ) {

      /* Search the database info. */
      ok = xtmCdFetchNamedDb( custom_data -> cal_db_handle, 
                              db_incl.db[ index ].name,
                              &db_incl_info, NULL );
      if( ! ok ) {
        sprintf( buffer, 
                 msgGetText( MXDI_ERRMSG_FETCH_NAMED_DB ),
                 db_incl_info.directory );

        xitErMessage( parent, XIT_ER_ERROR, 
                      module_name, "xtmDmFetchCompleteDay",
                      buffer );
        continue;
      }

      /* We must have at least read access. */
      if( ! flagIsSet( db_incl_info.operations, XTM_DB_FLAG_MODE_READ ) ) {
        sprintf( buffer, 
                 msgGetText( MXDI_ERRMSG_NO_ACCESS_NAMED_DB ),
                 db_incl_info.directory );

        xitErMessage( NULL, XIT_ER_ERROR, 
                      module_name, "xtmDmFetchCompleteDay",
                      buffer );
        continue;
      }

      /* Fetch entries in the include database. */    
      db_status = fetchFilteredEntriesDay( appl_data_ref,
                                           db_incl.db[ index ].name,
                                           flags, 
                                           context, workW,
                                           starts, ends, 
                                           filter_ref, *entries );
      if( db_status != XTM_DB_OK )
        raise exception;

    } /* loop */

  } /* if */


  if( parent != NULL ) {
    if( workW != NULL )
      xitBusyDialogRemove( workW );
    else
      XUndefineCursor( XtDisplay( parent ), XtWindow( parent ) );
  }


  return( XTM_DB_OK );


  /* Exception handler. */
  exception:
    if( workW != NULL )
      xitBusyDialogRemove( workW );

    return( db_status );

} /* xtmDmFetchFilteredEntriesRange */


/*----------------------------------------------------------------------*/

Boolean
  xtmDmInsertEntry( Widget                  parentW,
                    XTM_DB_ENTRY_DATABASES  *database,
                    UINT32                  entry_id,
                    XTM_DB_ALL_ENTRY_REF    entry_ref,
                    char                    *entry_text )
{

  /* Variables. */
  UINT32             id;
  XTM_DB_ID_REQUEST  id_request;
  XTM_DB_STATUS      db_status;


  /* Code. */

  /* Generate a new id for the entry. */
  id_request.directory  = database -> database_dir;
  id_request.operations = 0;
  id_request.lock_file  = False;

  db_status = xtmDbGenerateId( &id_request, &id );
  if( db_status != XTM_DB_OK )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "xtmDmInsertEntry",
                  msgGetText( MXDI_ERRMSG_GENERATE_ID ) );

  /* New data for entry to save. */
  entry_ref -> entry.id       = id;
  entry_ref -> stand_entry.id = id;

  /* Save the new entry. */
  db_status = xtmDbInsertEntry( database, 
                                entry_ref, entry_text );

  if( db_status != XTM_DB_OK )
    xitErMessage( NULL, XIT_ER_FATAL, 
                  module_name, "xtmDmInsertEntry",
                  msgGetText( MXDI_ERRMSG_INSERT_ENTRY ) );


  return( True );

} /* xtmDmInsertEntry */


/*----------------------------------------------------------------------*/

void
  xtmDmEntriesDefined( XTM_GL_BASE_DATA_REF  appl_data_ref,
                       char                  *db_name,
                       TIM_TIME_REF          date,
                       int                   days,
                       Boolean               include_db,
                       Boolean               show_standing,
                       Boolean               entry_defined[] )
{

  /* Variables. */
  Boolean                 ok;
  int                     index;
  XTM_GL_CUSTOM_DATA_REF  custom_data;
  XTM_CD_CAL_INFO         db_incl_info;
  XTM_CD_CAL_INFO         db_info;
  XTM_CD_INCL_CALS        db_incl;


  /* Code. */

  custom_data = appl_data_ref -> custom_data;

  for( index = 0; index < days; index++ )
    entry_defined[ index ] = False;


  /* Check the main database. */
  checkEntries( appl_data_ref, db_name,
                date, days, show_standing, entry_defined );


  /* Fetch the calendar information. */
  (void) xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle,
                            db_name,
                            &db_info, &db_incl );


  /* Check the included databases? */
  if( db_incl.no == 0 || ! include_db )
    return;

  for( index = 0; index < db_incl.no; index++ ) {
    ok = xtmCdFetchNamedDb( custom_data -> cal_db_handle, 
                            db_incl.db[ index ].name,
                            &db_incl_info, NULL );

    if( ! ok || flagIsSet( db_incl_info.flags, XTM_CD_FLAG_HIDE_IN_CAL ) )
      continue;

    checkEntries( appl_data_ref,
                  db_incl.db[ index ].name,
                  date, days, show_standing, entry_defined );
  } /* loop */


  return;

} /* xtmDmEntriesDefined */


/*----------------------------------------------------------------------*/

Boolean
  xtmDmOpenDatabase( XTM_GL_BASE_DATA_REF    appl_data_ref,
                     char                    *db_name,
                     UINT32                  access_mode,
                     XTM_DB_ENTRY_DATABASES  *database_ref )
{

  /* Variables. */
  Boolean              ok;
  char                 buffer[ 200 ];
  XTM_DB_OPEN_REQUEST  open_request;
  XTM_DB_STATUS        db_status;
  XTM_CD_CAL_INFO      db_info;


  /* Code. */

  /* Fetch database information. */
  ok = xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle,
                          db_name,
                          &db_info, NULL );
  if( ! ok )
    return( False );


  /* Do we have the necessary access? */
  if( ! flagIsSet( db_info.operations, access_mode ) ) {
    sprintf( buffer, 
             msgGetText( MXDI_ERRMSG_NO_ACCESS_NAMED_DB ), 
             db_info.short_name );

    xitMsgWinDisplayMessage( appl_data_ref -> msg_win_ref, buffer, 0 );
    return( False );
  }    


  /* Open the database to write the new entry. */
  open_request.name       = db_info.short_name;
  open_request.directory  = db_info.directory;
  open_request.operations = access_mode;
  open_request.database   = XTM_DB_ALL_ENTRY_DB;

  db_status = xtmDbOpenEntryDb( &open_request, database_ref );

  if( db_status != XTM_DB_OK ) {

    if( db_status == XTM_DB_LOCKED ) {
      sprintf( buffer, "%s: %s", 
               db_info.short_name, msgGetText( MXDI_ERRMSG_DB_LOCKED ) );

      xitMsgWinDisplayMessage( appl_data_ref -> msg_win_ref, buffer, 0 );
    } else {
      sprintf( buffer, "%s: %s", 
               db_info.short_name, msgGetText( MXDI_ERRMSG_CANNOT_OPEN_DB ) );

      xitMsgWinDisplayMessage( appl_data_ref -> msg_win_ref, buffer, 0 );
    }

    return( False );
  }


  return( True );

} /* xtmDmOpenDatabase */


/*----------------------------------------------------------------------*/

static void
  checkEntries( XTM_GL_BASE_DATA_REF  appl_data_ref,
                char                  *db_name,
                TIM_TIME_REF          date,
                int                   days,
                Boolean               show_standing,
                Boolean               entry_defined[] )
{

  /* Variables. */
  Boolean                 ok;
  int                     index;
  LST_DESC_TYPE           stand_entries[ 2 ];
  TIM_TIME_REF            curr_date;
  XTM_DB_ENTRY_DATABASES  database;


  /* Code. */

  /* Open the database for this calendar. */
  ok = xtmDmOpenDatabase( appl_data_ref,
                          db_name, XTM_DB_FLAG_MODE_READ,
                          &database );
  if( ! ok )
    return;

  /* Fetch all standing entries? */
  stand_entries[ 0 ] = NULL;
  stand_entries[ 1 ] = NULL;

  if( show_standing )
    (void) xtmDbFetchStandEntries( &database,
                                   &stand_entries[ 0 ], 
                                   &stand_entries[ 1 ] );

  /* Start the 1st of the month. */
  curr_date = date;

  /* Check all days. */
  for( index = 0; index < days; index++ ) {

    if( ! entry_defined[ index ] )
      entry_defined[ index ] = xtmDbIsEntryDefined( &database, 
                                                    stand_entries, curr_date );
    TimAddDays( &curr_date, 1 );

  } /* loop */


  /* Free the standing entries. */
  LstLinkClear( stand_entries[ 0 ] );
  LstLinkClear( stand_entries[ 1 ] );

  /* Close the database again. */
  xtmDbCloseEntryDb( &database );


  return;

} /* checkEntries */		     


/*----------------------------------------------------------------------*/

static void
  deleteEntryCB( void  *element )
{

  /* Variables. */
  XTM_DB_ALL_ENTRY_DEF  *entry_ref;


  /* Code. */

  entry_ref = (XTM_DB_ALL_ENTRY_DEF *) element;

  if( entry_ref -> all_text != NULL )
    SysFree( entry_ref -> all_text );


  return;

} /* deleteEntryCB */


/*----------------------------------------------------------------------*/

static LST_COMPARE
  entrySortFunc( XTM_DB_ALL_ENTRY_REF  list_entry_ref,
                 XTM_DB_ALL_ENTRY_REF  search_entry_ref )
{

  /* Variables. */
  UINT32  list_entry_prio   = 0;
  UINT32  search_entry_prio = 0;
  char    *list_tag;
  char    *search_tag;


  /* Code. */

  /* Correct day? */
  if( search_entry_ref -> entry.date_stamp >
      list_entry_ref   -> entry.date_stamp )
    return( LST_NOT_EQUAL );

  if( search_entry_ref -> entry.date_stamp <
      list_entry_ref   -> entry.date_stamp )
    return( LST_EQUAL );



  /* Sort notes? */
  if( search_entry_ref -> entry.entry_type == XTM_DB_DAY_NOTE ) {

    if( list_entry_ref -> entry.entry_type != XTM_DB_DAY_NOTE )
      return( LST_EQUAL );

    /* Ranking for entry to insert. */
    if( flagIsSet( search_entry_ref -> entry.flags, XTM_DB_FLAG_IMPORTANT ) )
      flagSet( search_entry_prio, PRIO_IMPORTANT );

    if( flagIsClear( search_entry_ref -> entry.flags, XTM_DB_FLAG_NOTE_DONE ) )
      flagSet( search_entry_prio, PRIO_NOTE_NOT_DONE );

    /* Ranking for entry in list. */
    if( flagIsSet( list_entry_ref -> entry.flags, XTM_DB_FLAG_IMPORTANT ) )
      flagSet( list_entry_prio, PRIO_IMPORTANT );

    if( flagIsClear( list_entry_ref -> entry.flags, XTM_DB_FLAG_NOTE_DONE ) )
      flagSet( list_entry_prio, PRIO_NOTE_NOT_DONE );

    /* Tag priority. */
    if( search_entry_ref -> entry.tag[ 0 ] == '\0' )
      search_tag = "xxxx";
    else
      search_tag = search_entry_ref -> entry.tag;

    if( list_entry_ref -> entry.tag[ 0 ] == '\0' )
      list_tag = "xxxx";
    else
      list_tag = list_entry_ref -> entry.tag;

    if( strcmp( search_tag, list_tag ) < 0 )
      flagSet( search_entry_prio, PRIO_TAG );
    else
      flagSet( list_entry_prio, PRIO_TAG );


    /* Insert the entry here? */
    if( search_entry_prio > list_entry_prio )
      return( LST_EQUAL );


  /* Sort appointments? */
  } else if( search_entry_ref -> entry.entry_type == XTM_DB_DAY_ENTRY ) {

    if( search_entry_ref -> entry.time_stamp < 
        list_entry_ref -> entry.time_stamp )
      return( LST_EQUAL );

  } /* if */


  return( LST_NOT_EQUAL );

} /* entrySortFunc */


/*----------------------------------------------------------------------*/

static XTM_DB_STATUS
  fetchFilteredEntriesDay( XTM_GL_BASE_DATA_REF  appl_data_ref,
                           char                  *db_name,
                           UINT32                flags,
                           XtAppContext          context,
                           Widget                workW,
                           TIM_TIME_REF          starts,
                           TIM_TIME_REF          ends,
                           XTM_DM_FILTER_REC     *filter_ref,
                           LST_DESC_TYPE         entries )
{

  /* Variables. */
  Boolean                 case_sens;
  Boolean                 match;
  Boolean                 ok;
  int                     index;
  UINT32                  filter;
  char                    *char_ref;
  TIM_TIME_REF            current_date;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_STATUS           db_status;
  LST_DESC_TYPE           list_ref[ 2 ];
  LST_STATUS              lst_status;


  /* Code. */

  /* Open the database for this calendar. */
  ok = xtmDmOpenDatabase( appl_data_ref,
                          db_name, XTM_DB_FLAG_MODE_READ,
                          &database );
  if( ! ok )
    return( XTM_DB_ERROR );


  filter       = filter_ref -> flags;
  current_date = starts;

  /* Process all days. */
  while( current_date <= ends ) {

    /* Fetch entries this day. */
    flagSet( flags, (XTM_DB_FETCH_STANDING | 
                     XTM_DB_FETCH_STICKY | 
                     XTM_DB_FETCH_ALL_TEXT ) );

    db_status = xtmDbFetchEntriesInDay( &database, current_date, flags,
                                        &list_ref[ 1 ], &list_ref[ 0 ] );

    if( db_status != XTM_DB_OK ) {
      xtmDbCloseEntryDb( &database );

      return( XTM_DB_ERROR );
    }

    /* Process the notes and entries. */
    for( index = 0; index < 2; index++ ) {

      /* Any elements in the list? */
      if( LstLinkElements( list_ref[ index ] ) > 0 ) {

        /* Process all the elements. */
        lst_status = LstLinkCurrentFirst( list_ref[ index ] );
        while( lst_status == LST_OK ) {

          XTM_DB_ENTRY_REF        entry_ref;
          XTM_DB_STAND_ENTRY_REF  stand_entry_ref;

          lst_status = LstLinkGetCurrent( list_ref[ index ], &entry_record );

          entry_ref       = &entry_record.entry;
          stand_entry_ref = &entry_record.stand_entry;

          match = False;

          if( flagIsSet( filter, XTM_DM_FLAG_ALL ) )
            match = True;

          /* No other flags? */
          if( flagIsSet( filter, ~ XTM_DM_FLAG_ALL ) ) {

            /* Pick type of entry. */
            if( flagIsSet( filter, XTM_DM_FLAG_APPOINTMENTS ) && 
                entry_ref -> entry_type == XTM_DB_DAY_ENTRY )
              match = True;

            if( flagIsSet( filter, XTM_DM_FLAG_NOTES ) &&
                entry_ref -> entry_type == XTM_DB_DAY_NOTE )
              match = True;


            /* Pick out all specialities. */
            if( flagIsSet( filter, XTM_DM_FLAG_PRIVATE ) &&
                flagIsClear( entry_ref -> flags, XTM_DB_FLAG_PRIVATE ) )
              match = False;

            if( flagIsSet( filter, XTM_DM_FLAG_IMPORTANT ) &&
                flagIsClear( entry_ref -> flags, XTM_DB_FLAG_IMPORTANT ) )
              match = False;

            if( flagIsSet( filter, XTM_DM_FLAG_ALARMS ) &&
                flagIsClear( entry_ref -> flags, XTM_DB_FLAG_ALARM ) )
              match = False;

            if( flagIsSet( filter, XTM_DM_FLAG_REPEATED ) &&
                entry_ref -> entry_category != XTM_DB_REP_ENTRY_LIST )
              match = False;

            if( flagIsSet( filter, XTM_DM_FLAG_STICKY ) &&
                entry_ref -> entry_category != XTM_DB_STICKY_LIST )
              match = False;

            if( entry_ref -> entry_type == XTM_DB_DAY_NOTE ) {
              if( flagIsSet(   filter, XTM_DM_FLAG_DONE ) &&
                  flagIsClear( filter, XTM_DM_FLAG_NOT_DONE ) &&
                  flagIsClear( entry_ref -> flags, XTM_DB_FLAG_NOTE_DONE ) )
                match = False;

              if( flagIsSet(   filter, XTM_DM_FLAG_NOT_DONE ) &&
                  flagIsClear( filter, XTM_DM_FLAG_DONE ) &&
                  flagIsSet( entry_ref -> flags, XTM_DB_FLAG_NOTE_DONE ) )
                match = False;
            }


            /* Search in the entry text? */
            if( match && flagIsSet( filter, XTM_DM_FLAG_SEARCH_TEXT ) ) {
              case_sens = flagIsSet( filter, XTM_DM_FLAG_TEXT_CASE_SENS );

              if( entry_record.all_text != NULL )
                match = searchString( filter_ref -> text_string, True,
                                      entry_record.all_text, case_sens );
              else
                match = searchString( filter_ref -> text_string, True,
                                      entry_ref -> text, case_sens );
            }

            /* Search the tag? */
            if( match && flagIsSet( filter, XTM_DM_FLAG_SEARCH_TAG ) ) {
              case_sens = flagIsSet( filter, XTM_DM_FLAG_TAG_CASE_SENS );

              if( entry_ref -> tag[ 0 ] != '\0' )
                match = searchString( entry_ref -> tag, case_sens,
                                      filter_ref -> tag_string, True );
              else
                match = False;
            }

          } /* if */


          /* If there is a match, save the entry in the match list. */
          if( match ) {

            /* The original will be freed when we delete list. */
            if( entry_record.all_text != NULL ) {
              char_ref = SysNewString( entry_record.all_text );

              entry_record.all_text = char_ref;
	    }

            /* Notes are inserted first in the day. */
            if( entry_record.entry.entry_type == XTM_DB_DAY_NOTE )
              entry_record.entry.time_stamp = 0;

            entry_record.entry.date_stamp = current_date;

            /* Insert the entry in the appointments list. */
            lst_status = LstLinkSearchFirst(
                           entries,
                           &entry_record,
                           (EQUALS_FUNC_TYPE) entrySortFunc );

            if( lst_status == LST_OK )
              lst_status = LstLinkInsertCurrent( entries,
                                                 &entry_record );
            else
              lst_status = LstLinkInsertLast( entries,
	  				      &entry_record );

          } /* if */


          /* Next record. */
          lst_status = LstLinkCurrentNext( list_ref[ index ] );

        } /* while */
    
      } /* if */

      /* Delete the database list. */
      xtmDmDeleteEntriesList( list_ref[ index ] );

    } /* loop */


    /* Did the user press the cancel button? */
    if( workW != NULL ) {
      if( xitBusyDialogCancelled( workW, context ) ) {
        xtmDbCloseEntryDb( &database );

        return( XTM_DB_ERROR );
      }
    }

    /* The next day. */
    TimAddDays( &current_date, 1 );

  } /* while */

  xtmDbCloseEntryDb( &database );


  return( XTM_DB_OK );

} /* fetchFilteredEntriesDay */


/*----------------------------------------------------------------------*/

static Boolean
  searchString( char     *search_string,
                Boolean  case_sens1,
                char     *in_string,
                Boolean  case_sens2 )
{

  /* Variables. */
  char  *char_ref;
  char  *new_in_string;
  char  *new_search_string;

  /* Code. */

  new_search_string = SysNewString( search_string );
  new_in_string     = SysNewString( in_string );


  /* Convert to uppercase if not case sensitive. */
  if( ! case_sens1 ) {
    char_ref = new_search_string;

    while( *char_ref != '\0' ) {
      *char_ref = toupper( *char_ref );
      char_ref++;
    }
  }

  if( ! case_sens2 ) {
    char_ref = new_in_string;

    while( *char_ref != '\0' ) {
      *char_ref = toupper( *char_ref );
      char_ref++;
    }
  }

  /* Search the string. */
  char_ref = strstr( new_in_string, new_search_string );

  SysFree( new_search_string );
  SysFree( new_in_string );

  if( char_ref == NULL )
    return( False );


  return( True );

} /* searchString */
