/*----------------------------------------------------------------------------
--
--  Module:           xtmBaseCal
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Display a basic calendar.
--
--  Filename:         xtmBaseCal.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1993-07-01
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmBaseCal.c, Version: 1.1, Date: 95/02/18 15:51:56";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/ArrowB.h>
#include <Xm/Form.h>
#include <Xm/RowColumn.h>

#include "System.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmDbTools.h"
#include "xtmIcons.h"
#include "xitError.h"
#include "xitTools.h"
#include "XmUbMonthD.h"
#include "XmUbNoteBk.h"

#include "xtmBaseCal.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* How many year tabs do we have? */
#define  NO_OF_YEAR_TABS      6

/* Different style for buttons in calenadr. */
#define  STYLE_LABEL_MARK     1
#define  STYLE_COLOR_SHADOW   2
#define  STYLE_MOTIF_SHADOW   3

/* Local widgets in the calendar window. */
#define nextYearAb          notebookW[  0 ]
#define prevYearAb          notebookW[  1 ]

#define calFo               dataLocalW[  0 ]
#define calMd               dataLocalW[  1 ]
#define todayDateBu         dataLocalW[  2 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

typedef enum { YEAR_PREV, YEAR_NEXT } YEAR_DIRECTION;

/* Data for the calendar. */
typedef struct {
  Boolean                entry_defined[ 6 * 7 ];
  Boolean                first_time;
  int                    curr_year;
  int                    curr_month;
  int                    first_year_tab;
  int                    last_year_tab;
  Widget                 calendarW;
  Widget                 workAreaW;
  TIM_TIME_REF           this_date;
  XTM_BC_CAL_STYLE       style;
  XTM_GL_BASE_DATA_REF   appl_data_ref;
  XTM_BC_ACTION_CB       actionCB;
  XTM_BC_HAS_ENTRIES_CB  hasEntriesCB;
  void                   *user_data;
} CAL_REC, *CAL_REC_REF;


/* Data for year scroll buttons. */
typedef struct {
  YEAR_DIRECTION  direction;
  CAL_REC_REF     cal_ref;
} SCROLL_YEAR_REC, *SCROLL_YEAR_REF;


/* Data for year select buttons. */
typedef struct {
  int          index;
  int          year;
  CAL_REC_REF  cal_ref;
} YEAR_REC, *YEAR_REF;


/* Data for month select buttons. */
typedef struct {
  int          index;
  int          month;
  CAL_REC_REF  cal_ref;
} MONTH_REC, *MONTH_REF;

/* Data for date buttons. */
typedef struct {
  TIM_TIME_REF  date;
  CAL_REC_REF   cal_ref;
} DATE_REC, *DATE_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmBaseCalendar";

/* Pixmaps to use for binding. */
static Pixmap  top_pixmap       = None;
static Pixmap  left_top_pixmap  = None;
static Pixmap  right_top_pixmap = None;


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static Widget
  createCalendarWindow( CAL_REC_REF  cal_ref,
                        Widget       parentW );

static void 
  dateSelectCB( Widget                          widget,
                CAL_REC_REF                     cal_ref,
                XmUbMonthDisplayCallbackStruct  *call_data );

static void 
  destroyTabCB( Widget     widget,
                XtPointer  user_data_ref,
                XtPointer  call_data );

static void
  displayMonth( CAL_REC_REF   cal_ref,
                TIM_TIME_REF  cal_date );

static void
  displayTimeLabel( CAL_REC_REF  cal_ref );

static void 
  messageCB( Widget       widget,
             CAL_REC_REF  cal_ref,
             XtPointer    call_data );

static void 
  monthSelectCB( Widget     widget,
                 MONTH_REF  month_ref,
                 XtPointer  call_data );

static void 
  scrollYearCB( Widget           widget,
                SCROLL_YEAR_REF  scroll_ref,
                XtPointer        call_data );

static void
  setYearTabs( CAL_REC_REF  cal_ref,
               int          start_on_year );

static void 
  todayCB( Widget       widget,
           CAL_REC_REF  cal_ref,
           XtPointer    call_data );

static void 
  updateMonthCB( Widget                          widget,
                 CAL_REC_REF                     cal_ref,
                 XmUbMdiUpdateDayCallbackStruct  *call_data );

static void 
  yearSelectCB( Widget     widget,
                YEAR_REF   year_ref,
                XtPointer  call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_BC_HANDLE
  xtmBcInitialize( XTM_GL_BASE_DATA_REF   appl_data_ref,
                   Widget                 parentW,
                   XTM_BC_CAL_STYLE       style,
                   XTM_BC_ACTION_CB       actionCB,
                   XTM_BC_HAS_ENTRIES_CB  hasEntriesCB,
                   void                   *user_data )
{

  /* Variables. */
  CAL_REC_REF  cal_ref;


  /* Code. */

  cal_ref = SysNew( CAL_REC );
  if( cal_ref == NULL )
    return( NULL );

  cal_ref -> appl_data_ref = appl_data_ref;
  cal_ref -> curr_year     = 1981;
  cal_ref -> curr_month    = 1;
  cal_ref -> first_time    = True;
  cal_ref -> style         = style;
  cal_ref -> calendarW     = NULL;
  cal_ref -> actionCB      = actionCB;
  cal_ref -> hasEntriesCB  = hasEntriesCB;
  cal_ref -> user_data     = user_data;


  /* Calendar window. */
  cal_ref -> calendarW = createCalendarWindow( cal_ref, parentW );

  if( cal_ref -> calendarW == NULL ) {
    SysFree( cal_ref );
    return( NULL );
  }


  return( cal_ref );

} /* xtmBcInitialize */


/*----------------------------------------------------------------------*/

void
  xtmBcDestroy( XTM_BC_HANDLE  cal_handle )
{

  /* Variables. */
  CAL_REC_REF  cal_ref;


  /* Code. */

  if( cal_handle == NULL )
    return;

  /* Our private data. */
  cal_ref = (CAL_REC_REF) cal_handle;

  /* Do we have a user action callback registered? */
  if( cal_ref -> actionCB != NULL )
    (* cal_ref -> actionCB)( XTM_BC_REASON_DESTROY,
                             (TIM_TIME_REF) 0, NULL,
                             cal_ref -> user_data );

  /* Destroy the widget. */
  XtDestroyWidget( cal_ref -> calendarW );

  /* Free allocated data. */
  SysFree( cal_ref );


  return;

} /* xtmBcDestroy */


/*----------------------------------------------------------------------*/

void
  xtmBcDisplayMonth( XTM_BC_HANDLE  cal_handle,
                     TIM_TIME_REF   cal_date )
{

  /* Variables. */
  CAL_REC_REF  cal_ref;


  /* Code. */

  if( cal_handle == NULL )
    return;

  /* Our private data. */
  cal_ref = (CAL_REC_REF) cal_handle;

  /* Display the month. */
  displayMonth( cal_ref, cal_date );


  return;

} /* xtmBcDisplayMonth */


/*----------------------------------------------------------------------*/

void
  xtmBcDisplayTime( XTM_BC_HANDLE  cal_handle )
{

  /* Variables. */
  CAL_REC_REF  cal_ref;


  /* Code. */

  if( cal_handle == NULL )
    return;

  /* Our private data. */
  cal_ref = (CAL_REC_REF) cal_handle;

  /* Display the current time. */
  displayTimeLabel( cal_ref );


  return;

} /* xtmBcDisplayTime */


/*----------------------------------------------------------------------*/

Widget
  xtmBcGetWidget( XTM_BC_HANDLE  cal_handle )
{

  /* Variables. */
  CAL_REC_REF  cal_ref;


  /* Code. */

  if( cal_handle == NULL )
    return( NULL );

  /* Our private data. */
  cal_ref = (CAL_REC_REF) cal_handle;


  return( cal_ref -> calendarW );

} /* xtmBcGetWidget */


/*----------------------------------------------------------------------*/

void
  xtmBcSignalMessage( XTM_BC_HANDLE  cal_handle,
                      Boolean        has_message )
{

  /* Variables. */
  Arg                     args[ 5 ];
  Cardinal                n;
  Widget                  msgW;
  CAL_REC_REF             cal_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  static Pixmap  msg_pixmap = None;


  /* Code. */

  if( cal_handle == NULL )
    return;

  /* Our private data. */
  cal_ref = (CAL_REC_REF) cal_handle;

  custom_data_ref = cal_ref -> appl_data_ref -> custom_data;


  /* Fetch the pixmap to display. */
  msgW = XtNameToWidget( cal_ref -> calendarW, "CalFo.MessageBu" );

  if( msg_pixmap == None ) {
    msg_pixmap = xtmIcFetchSimplePixmap( msgW, XTM_IC_ICON_MESSAGE, False );

    n = 0;
    XtSetArg( args[ n ], XmNlabelPixmap, msg_pixmap ); n++;
    XtSetValues( msgW, args, n );
  }


  /* Display the message icon? */
  if( has_message ) {
    XtManageChild( msgW );
    if( XtIsRealized( msgW ) )
      XRaiseWindow( XtDisplay( msgW ), XtWindow( msgW ) );
  } else {
    XtUnmanageChild( msgW );
  }


  if( has_message )
    XBell( XtDisplay( msgW ), 100 );


  return;

} /* xtmBcSignalMessage */


/*----------------------------------------------------------------------*/

void
  xtmBcViewCalendar( XTM_BC_HANDLE  cal_handle )
{

  /* Variables. */
  CAL_REC_REF  cal_ref;


  /* Code. */

  if( cal_handle == NULL )
    return;

  /* Our private data. */
  cal_ref = (CAL_REC_REF) cal_handle;

  /* Manage the calendar window. */
  XtManageChild( cal_ref -> calendarW );

  cal_ref -> first_time = True;


  return;

} /* xtmBcViewCalendar */


/*----------------------------------------------------------------------*/

static Widget
  createCalendarWindow( CAL_REC_REF  cal_ref,
                        Widget       parentW )
{

  /* Variables. */
  int                     index;
  char                    buffer[ 20 ];
  Arg                     args[ 20 ];
  Cardinal                n;
  Widget                  calendarW;
  Widget                  dataLocalW[ 3 ];
  Widget                  messageBu;
  Widget                  month_tab[ 12 ];
  Widget                  notebookW[ 2 ];
  Widget                  plainW[ 2 ];
  Widget                  tempW;
  Widget                  year_tab[ NO_OF_YEAR_TABS ];
  XmString                xstr;
  XmString                xstr1;
  XmString                xstr2;
  MONTH_REF               tab_month_ref;
  SCROLL_YEAR_REF         scroll_year_ref;
  TIM_TIME_REF            curr_date;
  XTM_GL_CUSTOM_DATA_REF  custom_data;
  YEAR_REF                tab_year_ref;

  static XIT_ARROW_STRUCT scroll_year_button[] = {
    { "PrevYearAr", True, XmARROW_UP,   NULL },
    { "NextYearAr", True, XmARROW_DOWN,  NULL },
  };

  static XIT_PUSH_STRUCT button_def[] = {
    { "TodayDateBu", "Date\nTime", "", True, NULL },
    { "MessageBu",   " ",          "", True, NULL },
  };

  static XIT_PUSH_STRUCT tab_button_def[] = {
    { "", "99", "", True, NULL },
    { "", "9",  "", True, NULL },
  };


  /* Code. */

  custom_data = cal_ref -> appl_data_ref -> custom_data;


  /* The calendar is kept within a note book? */
  if( cal_ref -> style == XTM_BC_STYLE_NOTEBOOK ) {

    n = 0;
    calendarW = XmUbCreateNoteBook( parentW, "Calendar", args, n );

    /* Binding pixmaps? */
    if( top_pixmap == None )
      top_pixmap = xtmIcFetchSimplePixmap( 
                     calendarW, XTM_IC_ICON_CAL_TOP, False );

    if( left_top_pixmap == None )
      left_top_pixmap = xtmIcFetchSimplePixmap(
                          calendarW, XTM_IC_ICON_CAL_LEFT_TOP, False );

    if( right_top_pixmap == None )
      right_top_pixmap = xtmIcFetchSimplePixmap(
                           calendarW, XTM_IC_ICON_CAL_RIGHT_TOP, False );

    n = 0;
    XtSetArg( args[ n ], XmUbNbindingType, XmUbBINDING_PIXMAP ); n++;
    XtSetArg( args[ n ], XmUbNbindingPixmapCenter, top_pixmap ); n++;
    XtSetArg( args[ n ], XmUbNbindingPixmapStart, left_top_pixmap ); n++;
    XtSetArg( args[ n ], XmUbNbindingPixmapEnd, right_top_pixmap ); n++;
    XtSetValues( calendarW, args, n );


    /* Year tabs. */
    for( index = 0; index < XtNumber( year_tab ); index ++ ) {

      tab_year_ref = SysNew( YEAR_REC );
      if( tab_year_ref == NULL )
        return( NULL );

      tab_year_ref -> index   = index;
      tab_year_ref -> year    = 0;
      tab_year_ref -> cal_ref = cal_ref;

      sprintf( buffer, "YearTab%dPb", index + 1 );

      tab_button_def[ 0 ].name = buffer;
      year_tab[ index ] = xitCreatePushButton( calendarW,
                                               &tab_button_def[ 0 ] );

      n = 0;
      XtSetArg( args[ n ], XmUbNchildType, XmUbTAB ); n++;
      XtSetArg( args[ n ], XmUbNposition, XmUbRIGHT ); n++;
      XtSetArg( args[ n ], XmNuserData, (XtPointer) tab_year_ref ); n++;
      XtSetArg( args[ n ], XmNshadowThickness, 1 ); n++;
      XtSetArg( args[ n ], XmNrecomputeSize, False ); n++;
      XtSetValues( year_tab[ index ], args, n );

      xitStringSetLabel( year_tab[ index ], " " );

      XtAddCallback( year_tab[ index ], XmNactivateCallback,
                     (XtCallbackProc) yearSelectCB, (XtPointer) tab_year_ref );
      XtAddCallback( year_tab[ index ], XmNdestroyCallback,
                     (XtCallbackProc) destroyTabCB, (XtPointer) tab_year_ref );

    } /* loop */


    /* Scroll to previous years. */
    scroll_year_ref = SysNew( SCROLL_YEAR_REC );
    if( scroll_year_ref == NULL )
      return( NULL );

    scroll_year_ref -> direction = YEAR_PREV;
    scroll_year_ref -> cal_ref   = cal_ref;

    prevYearAb = xitCreateArrowPushButton( calendarW,
                                           &scroll_year_button[ 0 ] );

    n = 0;
    XtSetArg( args[ n ], XmUbNchildType, XmUbSCROLL_BUTTON ); n++;
    XtSetArg( args[ n ], XmUbNscrollDirection, XmUbSCROLL_BACK ); n++;
    XtSetArg( args[ n ], XmUbNposition, XmUbRIGHT ); n++;
    XtSetArg( args[ n ], XmNshadowThickness, 1 ); n++;
    XtSetArg( args[ n ], XmNuserData, (XtPointer) scroll_year_ref ); n++;
    XtSetValues( prevYearAb, args, n );

    XtAddCallback( prevYearAb, XmNactivateCallback,
                   (XtCallbackProc) scrollYearCB,
                   (XtPointer) scroll_year_ref );
    XtAddCallback( prevYearAb, XmNdestroyCallback,
                   (XtCallbackProc) destroyTabCB,
                   (XtPointer) scroll_year_ref );


    /* Scroll to next years. */
    scroll_year_ref = SysNew( SCROLL_YEAR_REC );
    if( scroll_year_ref == NULL )
      return( NULL );

    scroll_year_ref -> direction = YEAR_NEXT;
    scroll_year_ref -> cal_ref   = cal_ref;

    nextYearAb = xitCreateArrowPushButton( calendarW,
                                           &scroll_year_button[ 1 ] );

    n = 0;
    XtSetArg( args[ n ], XmUbNchildType, XmUbSCROLL_BUTTON ); n++;
    XtSetArg( args[ n ], XmUbNscrollDirection, XmUbSCROLL_FORWARD ); n++;
    XtSetArg( args[ n ], XmUbNposition, XmUbRIGHT ); n++;
    XtSetArg( args[ n ], XmNshadowThickness, 1 ); n++;
    XtSetArg( args[ n ], XmNuserData, (XtPointer) scroll_year_ref ); n++;
    XtSetValues( nextYearAb, args, n );

    XtAddCallback( nextYearAb, XmNactivateCallback,
                   (XtCallbackProc) scrollYearCB,
                   (XtPointer) scroll_year_ref );
    XtAddCallback( nextYearAb, XmNdestroyCallback,
                   (XtCallbackProc) destroyTabCB,
                   (XtPointer) scroll_year_ref );


    /* Month tabs. */
    curr_date = TimMakeTime( 1970, 1, 1, 0, 0, 0 );

    for( index = 0; index < XtNumber( month_tab ); index ++ ) {

      char  month_name[ 10 ];

      tab_month_ref = SysNew( MONTH_REC );
      if( tab_month_ref == NULL )
        return( NULL );

      tab_month_ref -> index   = index;
      tab_month_ref -> month   = index + 1;
      tab_month_ref -> cal_ref = cal_ref;

      sprintf( buffer, "MonthTab%dPb", index + 1 );

      tab_button_def[ 1 ].name = buffer;
      month_tab[ index ] = xitCreatePushButton( calendarW,
                                                &tab_button_def[ 1 ] );

      n = 0;
      XtSetArg( args[ n ], XmUbNchildType, XmUbTAB ); n++;
      XtSetArg( args[ n ], XmUbNposition, XmUbBOTTOM ); n++;
      XtSetArg( args[ n ], XmNuserData, (XtPointer) tab_month_ref ); n++;
      XtSetArg( args[ n ], XmNshadowThickness, 1 ); n++;
      XtSetArg( args[ n ], XmNrecomputeSize, False ); n++;
      XtSetValues( month_tab[ index ], args, n );

      XtAddCallback( month_tab[ index ], XmNactivateCallback,
                     (XtCallbackProc) monthSelectCB,
                     (XtPointer) tab_month_ref );
      XtAddCallback( month_tab[ index ], XmNdestroyCallback,
                     (XtCallbackProc) destroyTabCB,
                     (XtPointer) tab_month_ref );

      /* Month name (one character). */
      TimFormatStrTime( curr_date, "%b", buffer,  sizeof( buffer ) );
      month_name[ 0 ] = buffer[ 0 ];
      month_name[ 1 ] = '\0';

      xitStringSetLabel( month_tab[ index ], month_name );
      TimAddMonths( &curr_date, 1 );

    } /* loop */


    /* Create form to hold calendar. */
    n = 0;
    calFo = XmCreateForm( calendarW, "CalFo", args, n );
    cal_ref -> workAreaW = calFo;

    n = 0;
    XtSetArg( args[ n ], XmUbNworkArea, calFo ); n++;
    XtSetValues( calendarW, args, n );

  } /* if */


  /* The calendar is kept within a form? */
  if( cal_ref -> style == XTM_BC_STYLE_PLAIN ) {

    n = 0;
    calendarW = XmCreateForm( parentW, "Calendar", args, n );

    /* Create form to hold calendar. */
    n = 0;
    calFo = XmCreateForm( calendarW, "CalFo", args, n );
    cal_ref -> workAreaW = calFo;

  } /* if */


  /* Create date button. */
  todayDateBu = xitCreatePushButton(  calFo, &button_def[ 0 ] );
  XtAddCallback( todayDateBu, XmNactivateCallback,
                 (XtCallbackProc) todayCB, (XtPointer) cal_ref );

  n = 0;
  XtSetArg( args[ n ], XmNshadowThickness, 0 ); n++;
  XtSetValues( todayDateBu, args, n );


  /* Create the calendar. */
  xstr  = XmStringCreateLtoR( msgGetText( MXDI_WEEK_NUMBER_LABEL ), CS );
  xstr1 = XmStringCreateLtoR( msgGetText( MXDI_MONTH_LABEL ), CS );
  xstr2 = XmStringCreateLtoR( msgGetText( MXDI_YEAR_LABEL ), CS );

  n = 0;
  XtSetArg( args[ n ], XmUbNmdiTitleAlignment, XmALIGNMENT_BEGINNING ); n++;
  XtSetArg( args[ n ], XmUbNmdiWeekLabel, xstr ); n++;
  XtSetArg( args[ n ], XmUbNmdiUseDefaultTitleCallback, False ); n++;
  XtSetArg( args[ n ], XmUbNmdiTitleShadows, False ); n++;

  if( ! custom_data -> show_week_numbers ) {
    XtSetArg( args[ n ], XmUbNmdiWeekNumbers, False ); n++;
  }

  if( cal_ref -> style == XTM_BC_STYLE_NOTEBOOK ) {
    XtSetArg( args[ n ], XmUbNmdiMonthArrows, False ); n++;
    XtSetArg( args[ n ], XmUbNmdiMonthLabel, xstr1 ); n++;
    XtSetArg( args[ n ], XmUbNmdiYearArrows, False ); n++;
    XtSetArg( args[ n ], XmUbNmdiYearLabel, xstr2 ); n++;
  }
  calMd = XmUbCreateMonthDisplay( calFo, "CalMd", args, n );

  XmStringFree( xstr );
  XmStringFree( xstr1 );
  XmStringFree( xstr2 );

  XtAddCallback( calMd, XmUbNmdiUpdateDayCallback,
                 (XtCallbackProc) updateMonthCB, (XtPointer) cal_ref );
  XtAddCallback( calMd, XmUbNmdiDateSelectedCallback,
                 (XtCallbackProc) dateSelectCB, (XtPointer) cal_ref );


  /* Create the message indicator. */
  messageBu = xitCreatePushButton( calFo, &button_def[ 1 ] );

  XtAddCallback( messageBu, XmNactivateCallback,
                 (XtCallbackProc) messageCB, (XtPointer) cal_ref );

  n = 0;
  XtSetArg( args[ n ], XmNtraversalOn, False ); n++;
  XtSetArg( args[ n ], XmNlabelType, XmPIXMAP ); n++;
  XtSetArg( args[ n ], XmNforeground, custom_data -> entry_exist_fg ); n++;
  XtSetValues( messageBu, args, n );


  /* Position the widgets. */
  xitAttachWidget( todayDateBu,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( messageBu,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL,
                   XmATTACH_FORM, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( calMd,
                   XmATTACH_WIDGET, todayDateBu, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,        XmATTACH_NONE, NULL );


  /* Manage the style widgets. */
  if( cal_ref -> style == XTM_BC_STYLE_NOTEBOOK ) {
    xitManageChildren( notebookW, XtNumber( notebookW ) );
    xitManageChildren( month_tab, XtNumber( month_tab ) );
    xitManageChildren( year_tab,  XtNumber( year_tab ) );
  }

  /* Manage the common widgets. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Grab focus. */
  if( cal_ref -> style == XTM_BC_STYLE_PLAIN ) {
    tempW = XmUbMonthDisplayGetChild( calMd, XmUbMD_CHILD_MONTH_ARROW );

    xitSetFocus( calendarW, tempW );
  }


  return( calendarW );

} /* createCalendarWindow */


/*----------------------------------------------------------------------*/

static void
  displayMonth( CAL_REC_REF   cal_ref,
                TIM_TIME_REF  cal_date )
{

  /* Variables. */
  char                    buffer[ 50 ];
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  tempW;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = cal_ref -> appl_data_ref -> custom_data;

  /* Keep the year and month. */
  cal_ref -> curr_year  = TimIndexOfYear(  cal_date );
  cal_ref -> curr_month = TimIndexOfMonth( cal_date );


  /* Display the month. */
  tempW = XtNameToWidget( cal_ref -> workAreaW, "CalMd" );

  n = 0;
  XtSetArg( args[ n ], XmUbNmdiFillOutWeek,
            custom_data_ref -> cal_month_extend ); n++;
  XtSetValues( tempW, args, n );

  XmUbMonthDisplaySetMonth( tempW, 
                            cal_ref -> curr_year, cal_ref -> curr_month );


  /* Set the current time. */
  displayTimeLabel( cal_ref );


  /* If this is the first time, display the year labels. */
  if( cal_ref -> first_time && cal_ref -> style == XTM_BC_STYLE_NOTEBOOK ) {
    if( cal_ref -> curr_year > 1971 )
      setYearTabs( cal_ref, cal_ref -> curr_year - 1 );
    else
      setYearTabs( cal_ref, cal_ref -> curr_year );
  }

  cal_ref -> first_time = False;


  /* Make sure the year tabs are displayed correctly. */
  if( cal_ref -> style == XTM_BC_STYLE_NOTEBOOK ) {

    if( cal_ref -> curr_year >= cal_ref -> first_year_tab &&
        cal_ref -> curr_year <= cal_ref -> last_year_tab )
      sprintf( buffer, "YearTab%dPb",
               cal_ref -> curr_year - cal_ref -> first_year_tab + 1 );
    else
      sprintf( buffer, "YearTab%dPb", 1 );

    tempW = XtNameToWidget( cal_ref -> calendarW, buffer );
    XmUbNoteBookSelectedTab( cal_ref -> calendarW, tempW, XmUbRIGHT );


    /* Make sure the month tabs are displayed correctly. */
    sprintf( buffer, "MonthTab%dPb", cal_ref -> curr_month );
    tempW = XtNameToWidget( cal_ref -> calendarW, buffer );

    XmUbNoteBookSelectedTab( cal_ref -> calendarW, tempW, XmUbBOTTOM );

  } /* if */


  /* Anyone interested? */
  if( cal_ref -> actionCB != NULL )
    (* cal_ref -> actionCB)( XTM_BC_REASON_MONTH_DISPLAYED,
                             cal_date, NULL,
                             cal_ref -> user_data );


  return;

} /* displayMonth */


/*----------------------------------------------------------------------*/

static void
  displayTimeLabel( CAL_REC_REF  cal_ref )
{

  /* Variables. */
  char                    buffer[ 40 ];
  Widget                  todayDateW;
  XTM_GL_CUSTOM_DATA_REF  custom_data;


  /* Code. */

  custom_data = cal_ref -> appl_data_ref -> custom_data;


  /* Get the widget. */
  todayDateW = XtNameToWidget( cal_ref -> workAreaW, "TodayDateBu" );

  xtmFoFormatFullTime( TimLocalTime( TimMakeTimeNow() ),
                       buffer, sizeof( buffer ) );

  /* Set the today date label. */
  xitStringSetLabel( todayDateW, buffer );


  return;

} /* displayTimeLabel */


/*----------------------------------------------------------------------*/

static void
  setYearTabs( CAL_REC_REF  cal_ref,
               int          start_on_year )
{

  /* Variables. */
  int        full_year;
  int        index;
  int        year;
  char       buffer[ 50 ];
  Arg        args[ 10 ];
  Cardinal   n;
  Widget     tempW;
  XtPointer  user_data_ref;
  YEAR_REF   tab_year_ref;


  /* Code. */

  cal_ref -> first_year_tab = start_on_year;
  cal_ref -> last_year_tab  = start_on_year + NO_OF_YEAR_TABS - 1;

  /* Only disply the last two digits. */
  full_year = start_on_year;
  year      = start_on_year;

  while( year > 99 )
    year = year - 100;


  /* Set the correct year on the tabs. */
  for( index = 0; index < NO_OF_YEAR_TABS; index++ ) {

    sprintf( buffer, "YearTab%dPb", index + 1 );
    tempW = XtNameToWidget( cal_ref -> calendarW, buffer );

    n = 0;
    XtSetArg( args[ n ], XmNuserData, &user_data_ref ); n++;
    XtGetValues( tempW, args, n );

    tab_year_ref = (YEAR_REF) user_data_ref;

    /* Set the year label */
    sprintf( buffer, "%2.2d", year );
    xitStringSetLabel( tempW, buffer );

    /* Save the year for later. */
    tab_year_ref -> year = full_year;

    full_year++;
    year++;
    if( year > 99 )
      year = year - 100;

  } /* loop */


  return;

} /* setYearTabs */


/*----------------------------------------------------------------------*/

static void 
  dateSelectCB( Widget                          widget,
                CAL_REC_REF                     cal_ref,
                XmUbMonthDisplayCallbackStruct  *call_data )
{

  /* Variables. */
  TIM_TIME_REF  selected_date;


  /* Code. */

  if( call_data -> reason != XmUbCR_DATE_SELECTED )
    return;

  selected_date = TimMakeTime( call_data -> selected_year,
                               call_data -> selected_month,
                               call_data -> selected_day, 0, 0, 0 );

  if( cal_ref -> actionCB != NULL )
    (* cal_ref -> actionCB)( XTM_BC_REASON_DATE_SELECTED,
                             selected_date, 
                             (XmAnyCallbackStruct *) call_data,
                             cal_ref -> user_data );


  return;

} /* dateSelectCB */


/*----------------------------------------------------------------------*/

static void 
  destroyTabCB( Widget     widget,
                XtPointer  user_data_ref,
                XtPointer  call_data )
{

  /* Code. */

  SysFree( user_data_ref );


  return;

} /* destroyTabCB */


/*----------------------------------------------------------------------*/

static void 
  messageCB( Widget       widget,
             CAL_REC_REF  cal_ref,
             XtPointer    call_data )
{

  /* Code. */

  if( cal_ref -> actionCB != NULL )
    (* cal_ref -> actionCB)( XTM_BC_REASON_MSG_SELECTED,
                             (TIM_TIME_REF) 0, NULL,
                             cal_ref -> user_data );


  return;

} /* messageCB */


/*----------------------------------------------------------------------*/

static void 
  monthSelectCB( Widget     widget,
                 MONTH_REF  month_ref,
                 XtPointer  call_data )
{

  /* Variables. */
  TIM_TIME_REF  curr_date;


  /* Code. */

  /* Display this month. */
  curr_date = TimMakeTime( month_ref -> cal_ref -> curr_year,
                           month_ref -> month, 1, 0, 0, 0 );

  displayMonth( month_ref -> cal_ref, curr_date );


  return;

} /* monthSelectCB */


/*----------------------------------------------------------------------*/

static void 
  scrollYearCB( Widget           widget,
                SCROLL_YEAR_REF  scroll_ref,
                XtPointer        call_data )
{

  /* Variables. */
  int           to_year;
  TIM_TIME_REF  curr_date;


  /* Code. */

  /* In which direction yo we move? */
  if( scroll_ref -> direction == YEAR_PREV ) {
    to_year = scroll_ref -> cal_ref -> first_year_tab - NO_OF_YEAR_TABS + 1;
    if( to_year < 1971 )
      to_year = 1971;
  } else {
    to_year = scroll_ref -> cal_ref -> last_year_tab;
    if( to_year + NO_OF_YEAR_TABS > 2030 )
      to_year = 2030 - NO_OF_YEAR_TABS;
  }


  /* Set the new tabs. */
  setYearTabs( scroll_ref -> cal_ref, to_year );


  /* Display this month. */
  curr_date = TimMakeTime( to_year,
                           scroll_ref -> cal_ref -> curr_month,
                           1, 0, 0, 0 );

  displayMonth( scroll_ref -> cal_ref, curr_date );


  return;

} /* scrollYearCB */


/*----------------------------------------------------------------------*/

static void 
  todayCB( Widget       widget,
           CAL_REC_REF  cal_ref,
           XtPointer    call_data )
{

  /* Variables. */
  int           year;
  TIM_TIME_REF  curr_date;
  TIM_TIME_REF  now;


  /* Code. */

  /* Display today. */
  now       = TimLocalTime( TimMakeTimeNow() );
  curr_date = TimMakeTime( TimIndexOfYear( now ),
                           TimIndexOfMonth( now ), 1, 0, 0, 0 );

  year = TimIndexOfYear( now );

  if( cal_ref -> style == XTM_BC_STYLE_NOTEBOOK ) {
    if( year > 1971 )
      setYearTabs( cal_ref, year - 1 );
    else
      setYearTabs( cal_ref, year );
  }

  displayMonth( cal_ref, curr_date );


  return;

} /* todayCB */


/*----------------------------------------------------------------------*/

static void 
  updateMonthCB( Widget                          widget,
                 CAL_REC_REF                     cal_ref,
                 XmUbMdiUpdateDayCallbackStruct  *call_data )
{

  /* Variables. */
  Boolean                 is_holiday;
  int                     index;
  Pixel                   color;
  XTM_GL_CUSTOM_DATA_REF  custom_data;


  /* Code. */

  custom_data = cal_ref -> appl_data_ref -> custom_data;

  /* Update week numbers. */
  if( call_data -> reason == XmUbCR_UPDATE_WEEK ) {

    call_data -> foreground = custom_data -> week_number_fg;
    call_data -> mapped = call_data -> default_mapped;

  } /* if */


  /* Update dates. */
  if( call_data -> reason == XmUbCR_UPDATE_DAY ) {

    /* Fetch entries defined? */
    if( call_data -> loop_index == 0 ) {
      cal_ref -> this_date = TimMakeTime( call_data -> year, 
                                          call_data -> month,
                                          call_data -> day, 0, 0, 0 );

      for( index = 0; index < 6 * 7; index++ )
        cal_ref -> entry_defined[ index ] = False;

      if( cal_ref -> hasEntriesCB != NULL )
        (* cal_ref -> hasEntriesCB)( XTM_BC_REASON_ENTRIES_DEFINED,
                                     cal_ref -> user_data,
                                     cal_ref -> this_date, 6 * 7,
                                     cal_ref -> entry_defined );
    } else {
      TimAddDays( &cal_ref -> this_date, 1 );

    } /* if */


    /* Holidays. */
    is_holiday = xtmHoIsHoliday( cal_ref -> this_date );


    /* Find the foreground color. */
    color = custom_data -> week_number_fg;

    if( custom_data -> cal_button_style == 1 &&
        cal_ref -> entry_defined[ call_data -> loop_index ] )
      color = custom_data -> entry_exist_fg;

    else if( is_holiday )
      color = custom_data -> holiday_fg;

    else if( call_data -> which_month != XmUbCURRENT_MONTH )
      color = custom_data -> week_number_fg;

    else if( call_data -> weekday == 1 )
      color = custom_data -> monday_fg;

    else if( call_data -> weekday == 2 )
      color = custom_data -> tuesday_fg;

    else if( call_data -> weekday == 3 )
      color = custom_data -> wednesday_fg;

    else if( call_data -> weekday == 4 )
      color = custom_data -> thursday_fg;

    else if( call_data -> weekday == 5 )
      color = custom_data -> friday_fg;

    else if( call_data -> weekday == 6 )
      color = custom_data -> saturday_fg;

    else if( call_data -> weekday == 7 )
      color = custom_data -> sunday_fg;

    call_data -> foreground = color;


    /* Find the background color. */
    if( call_data -> today )
      color = custom_data -> today_bg;
    else
      color = custom_data -> date_mark_bg;

    call_data -> background = color;


    /* Find the shadow. */
    if( cal_ref -> entry_defined[ call_data -> loop_index ] ) {
      if( custom_data -> cal_button_style == 2 )
        color = custom_data -> entry_exist_fg;
      else
        color = custom_data -> date_mark_bg;

      if( custom_data -> cal_button_style == 3 )
        call_data -> use_shadows = True;
      else
        call_data -> use_shadows = False;
    } else {
      color = custom_data -> date_mark_bg;
      call_data -> use_shadows = False;
    }

    call_data -> frame = color;


    /* Extended month. */
    call_data -> mapped = call_data -> default_mapped;

   } /* if */


  return;

} /* updateMonthCB */


/*----------------------------------------------------------------------*/

static void 
  yearSelectCB( Widget     widget,
                YEAR_REF   year_ref,
                XtPointer  call_data )
{

  /* Variables. */
  TIM_TIME_REF  curr_date;


  /* Code. */

  /* Display this month. */
  curr_date = TimMakeTime( year_ref -> year,
                           year_ref -> cal_ref -> curr_month,
                           1, 0, 0, 0 );

  displayMonth( year_ref -> cal_ref, curr_date );


  return;

} /* yearSelectCB */
