/*----------------------------------------------------------------------------
--
--  Module:           xmubCustUi
--
--  Project:          <>
--  System:           <>
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    This module provides user interface functions to the functionality in 
--    xmubCustUtilities.
--
--  Contents:
--    <>
--
--  Filename:         xmubCustUi.c
--
--  Authors:          bre
--  Creation date:    1994-09-06
--
--  Modifications:
--
--  Who            When
--  What
--  -----------------------------
--  <>             <>
--  <>
--
--
--  Copyright 1994 by Union Bank of Switzerland.
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies.
--  Union Bank of Switzerland makes no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
--
------------------------------------------------------------------------------
--
-- SCCSID = "@(#) Module: %M%, Version: %R%.%L%, Date: %E% %U%"
--
----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>

#include <Xm/Xm.h>
#include <Xm/Protocols.h>
#include <Xm/AtomMgr.h>

#include <Xm/Frame.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/List.h>
#include <Xm/PushB.h>
#include <Xm/Separator.h>

#ifdef VMS
#include ctype
#include "xmubCustUtil.h"
#include "xmubCustUi.h"
#include "xmubUtils.h"

#else
#include <xmub/xmubCustUtil.h>
#include <xmub/xmubCustUi.h>
#include <xmub/xmubUtils.h>

#endif

/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#define BUFFER_LENGTH     200

#if XmVersion < 1002
#define XmFONTLIST_DEFAULT_TAG  XmSTRING_DEFAULT_CHARSET

#define XmStringCreateLocalized( str ) \
  XmStringCreateLtoR( str, XmSTRING_DEFAULT_CHARSET )

#endif

/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

typedef struct {

  Widget                   toplevel;
  Widget                   list;
  Widget                   dialog;
  xmubResourceMappingInfo  *info;
  String                   orig_palette;
  Boolean                  update_shadows;
  String                   orig_fontmap;

} CustDiCallbackInfo;

typedef struct {
  Pixel  foreground;
  Pixel  background;
} ColorDefs;


typedef struct {
  XmFontList  font_list;
} FontDefs;

typedef struct {

  char            *name;  /* NULL generates empty one. */
  XmString        label;  /* If NULL, not set. */
  XtCallbackProc  callback;
  XtPointer       callback_data;

} ActionAreaItem;

/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/


XtResource  color_resource_list[] = {

  { XmNforeground, XmCForeground, XmRPixel, sizeof( Pixel ), 
    XtOffsetOf( ColorDefs, foreground ), XmRString, XtDefaultForeground },

  { XmNbackground, XmCBackground, XmRPixel, sizeof( Pixel ), 
    XtOffsetOf( ColorDefs, background ), XmRString, XtDefaultBackground },
};

XtResource  font_resource_list[] = {
  { XmNfontList, XmCFontList, XmRFontList, sizeof( XmFontList ), 
    XtOffsetOf( FontDefs, font_list ), XmRString, XtDefaultFont },
};

/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/


static void
  ApplyColorResourcesCB( xmubResourceMappingInfo  *info,
                         CustDiCallbackInfo       *cb_info,
                         XtPointer                dummy );

static void 
  ApplyFont( Widget              widget,
             CustDiCallbackInfo  *cb_info );

static void 
  ApplyFontCB( Widget                      widget,
               CustDiCallbackInfo          *cb_info,
               XmPushButtonCallbackStruct  *call_data );

static void
  ApplyFontResourcesCB( xmubResourceMappingInfo  *info,
                        CustDiCallbackInfo       *cb_info,
                        XtPointer                dummy );

static void 
  ApplyPalette( Widget              widget,
                CustDiCallbackInfo  *cb_info );

static void 
  ApplyPaletteCB( Widget                      widget,
                  CustDiCallbackInfo          *cb_info,
                  XmPushButtonCallbackStruct  *call_data );

static void 
  CancelFontDialogCB( Widget                      widget,
                      CustDiCallbackInfo          *cb_info,
                      XmPushButtonCallbackStruct  *call_data );

static void 
  CancelPaletteDialogCB( Widget                      widget,
                         CustDiCallbackInfo          *cb_info,
                         XmPushButtonCallbackStruct  *call_data );

static void
  ChangeWidgetColorsCB( Widget     widget,
                        XtPointer  dummy,
                        XtPointer  update_shadows );

static void
  ChangeWidgetFontsCB( Widget     widget,
                       XtPointer  dummy1,
                       XtPointer  dummy2 );

static Widget 
  CreateActionArea( Widget          parent,
                    char            *name,
                    ActionAreaItem  *items,
                    int             num_items,
                    int             tightness,
                    int             default_index );

static Widget 
  CreateFormDialog( Widget          parent,
                    char            *name,
                    XtCallbackProc  close_callback,
                    XtPointer       close_callback_data,
                    ActionAreaItem  *items,
                    int             num_items,
                    int             tightness,
                    int             default_index );

static void 
  FontOkCB( Widget                      widget,
            CustDiCallbackInfo          *cb_info,
            XmPushButtonCallbackStruct  *call_data );

static void 
  PaletteOkCB( Widget                      widget,
               CustDiCallbackInfo          *cb_info,
               XmPushButtonCallbackStruct  *call_data );

static void 
  PopdownCustDialog( Widget               widget,
                     CustDiCallbackInfo  *cb_info );

static void 
  RemoveCustDialog( Widget              widget,
                    CustDiCallbackInfo  *cb_info );


/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

static void
  ApplyColorResourcesCB( xmubResourceMappingInfo  *info,
                         CustDiCallbackInfo       *cb_info,
                         XtPointer                dummy )
{

  xmubApplyColorResources( cb_info -> toplevel, cb_info -> update_shadows );

}


/*----------------------------------------------------------------------*/

static void 
  ApplyFont( Widget              widget,
             CustDiCallbackInfo  *cb_info )
{

  Arg       args[ 5 ];
  String    current_fontmap;
  Cardinal  n;
  int       num_selected;
  Boolean   ok;
  char      *str;
  XmString  *selected_xms;

  /* Code. */

  /* Get the selected font map. */

  n = 0;
  XtSetArg( args[ n ], XmNselectedItems,     &selected_xms ); n++;
  XtSetArg( args[ n ], XmNselectedItemCount, &num_selected ); n++;
  XtGetValues( cb_info -> list, args, n );

  if( num_selected > 0 ){

    ok = XmStringGetLtoR( selected_xms[ 0 ], XmFONTLIST_DEFAULT_TAG, &str );

    if( ok ){

      current_fontmap = xmubCurrentFontmap( XtDisplay( widget ) );

      if( ( current_fontmap == NULL ) ||
          ( strcmp( current_fontmap, str ) != 0 ) ){

        xmubResUpdateResourceSet(
          cb_info -> info, FONTMAP_RESOURCE_NAME, FONTMAP_CLASS_NAME, str,
          (xmubResSetChangeCallback) ApplyFontResourcesCB,
          (XtPointer) cb_info );
      }

      XtFree( str );

      if( current_fontmap != NULL )
        XtFree( (char *) current_fontmap );

    }
  }


}


/*----------------------------------------------------------------------*/

static void 
  ApplyFontCB( Widget                      widget,
               CustDiCallbackInfo          *cb_info,
               XmPushButtonCallbackStruct  *call_data )
{

  ApplyFont( widget, cb_info );

}


/*----------------------------------------------------------------------*/

static void
  ApplyFontResourcesCB( xmubResourceMappingInfo  *info,
                        CustDiCallbackInfo       *cb_info,
                        XtPointer                dummy )
{

  xmubApplyFontResources( cb_info -> toplevel );

}


/*----------------------------------------------------------------------*/

static void 
  ApplyPalette( Widget              widget,
                CustDiCallbackInfo  *cb_info )
{

  Arg       args[ 5 ];
  String    current_palette;
  Cardinal  n;
  int       num_selected;
  Boolean   ok;
  char      *palstr;
  XmString  *selected_xms;

  /* Code. */

  /* Get the selected palette. */

  n = 0;
  XtSetArg( args[ n ], XmNselectedItems,     &selected_xms ); n++;
  XtSetArg( args[ n ], XmNselectedItemCount, &num_selected ); n++;
  XtGetValues( cb_info -> list, args, n );

  if( num_selected > 0 ){

    ok = XmStringGetLtoR( selected_xms[ 0 ], XmFONTLIST_DEFAULT_TAG, &palstr );

    if( ok ){

      current_palette = xmubCurrentPalette( XtDisplay( widget ) );

      if( ( current_palette == NULL ) ||
          ( strcmp( current_palette, palstr ) != 0 ) ){

        xmubResUpdateResourceSet( 
          cb_info -> info, PALETTE_RESOURCE_NAME, PALETTE_CLASS_NAME, palstr,
          (xmubResSetChangeCallback) ApplyColorResourcesCB, 
          (XtPointer) cb_info );
      }

      XtFree( palstr );

      if( current_palette != NULL )
        XtFree( (char *) current_palette );

    }
  }


}


/*----------------------------------------------------------------------*/

static void 
  ApplyPaletteCB( Widget                      widget,
                  CustDiCallbackInfo          *cb_info,
                  XmPushButtonCallbackStruct  *call_data )
{

  ApplyPalette( widget, cb_info );

}


/*----------------------------------------------------------------------*/

static void 
  CancelFontDialogCB( Widget                      widget,
                      CustDiCallbackInfo          *cb_info,
                      XmPushButtonCallbackStruct  *call_data )
{

  Boolean  change_back;
  String   current_fontmap;

  /* Code. */

  /* For aesthetic reasons, pop the dialog down first. */
  PopdownCustDialog( widget, cb_info );


  current_fontmap = xmubCurrentFontmap( XtDisplay( widget ) );

  /* We are canceling the selection. Revert to original fontmap. */
  change_back = True;
  if( current_fontmap == cb_info -> orig_fontmap )
    change_back = False;
  else {
    if( ( current_fontmap         != NULL ) &&
	( cb_info -> orig_fontmap != NULL ) &&
	( strcmp( current_fontmap, cb_info -> orig_fontmap ) == 0 ) )
      change_back = False;
  }

  if( change_back )
    /* We are using a different fontmap now. */
    xmubResUpdateResourceSet(
      cb_info -> info, FONTMAP_RESOURCE_NAME, FONTMAP_CLASS_NAME,
      ( cb_info -> orig_fontmap == NULL ? "" : cb_info -> orig_fontmap ),
      (xmubResSetChangeCallback) ApplyFontResourcesCB, (XtPointer) cb_info );


  RemoveCustDialog( widget, cb_info );

}


/*----------------------------------------------------------------------*/

static void 
  CancelPaletteDialogCB( Widget                      widget,
                         CustDiCallbackInfo          *cb_info,
                         XmPushButtonCallbackStruct  *call_data )
{

  Boolean  change_back;
  String   current_palette;

  /* Code. */

  /* For aesthetic reasons, pop the dialog down first. */
  PopdownCustDialog( widget, cb_info );

  current_palette = xmubCurrentPalette( XtDisplay( widget ) );

  /* We are canceling the selection. Revert to original palette. */
  change_back = True;
  if( current_palette == cb_info -> orig_palette )
    change_back = False;
  else {
    if( ( current_palette         != NULL ) &&
        ( cb_info -> orig_palette != NULL ) &&
        ( strcmp( current_palette, cb_info -> orig_palette ) == 0 ) )
      change_back = False;
  }


  if( change_back )
    /* We are using a different palette now. */
    xmubResUpdateResourceSet( 
      cb_info -> info, PALETTE_RESOURCE_NAME, PALETTE_CLASS_NAME,
      ( cb_info -> orig_palette == NULL ? "" : cb_info -> orig_palette ),
      (xmubResSetChangeCallback) ApplyColorResourcesCB, 
      (XtPointer) cb_info );

  RemoveCustDialog( widget, cb_info );

}


/*----------------------------------------------------------------------*/

static void
  ChangeWidgetColorsCB( Widget     widget,
                        XtPointer  dummy,
                        XtPointer  update_shadows )
{

  Arg        args[ 8 ];
  Pixel      bottom_shadow;
  ColorDefs  color_defs;
  Colormap   colormap;
  Cardinal   n;
  Pixel      top_shadow;

  /* Code. */

  /* We don't do this for gadgets, since they have no colors of their own. */
  if( ! XtIsWidget( widget ) )
    return;

  /* Impossible to deallocate old color resources.
     Eat the colormap cell waste. */

  n = 0;  
  XtSetArg( args[ n ], XmNcolormap, &colormap ); n++;
  XtGetValues( widget, args, n );

  /* Get the resources for the current widget. */
  XtGetSubresources( widget, &color_defs, NULL, NULL, color_resource_list, 
                     XtNumber( color_resource_list ), NULL, 0 );

  /* Get the shadow colors as well. */
  if( update_shadows )
    XmGetColors( XtScreenOfObject( widget ), colormap, color_defs.background,
                 NULL, &top_shadow, &bottom_shadow, NULL );

  n = 0;  
  XtSetArg( args[ n ], XmNforeground,        color_defs.foreground ); n++;
  XtSetArg( args[ n ], XmNbackground,        color_defs.background ); n++;

  if( update_shadows ){
    XtSetArg( args[ n ], XmNtopShadowColor,    top_shadow ); n++;
    XtSetArg( args[ n ], XmNbottomShadowColor, bottom_shadow ); n++;
  }

  XtSetValues( widget, args, n );

}


/*----------------------------------------------------------------------*/

static void
  ChangeWidgetFontsCB( Widget     widget,
                       XtPointer  dummy1,
                       XtPointer  dummy2 )
{


  Arg       args[ 5 ];
  FontDefs  font_defs;
  Cardinal  n;

  /* Code. */

  /* Get the resources for the current widget. */
  XtGetSubresources( widget, &font_defs, NULL, NULL, font_resource_list, 
                     XtNumber( font_resource_list ), NULL, 0 );

  n = 0;  
  XtSetArg( args[ n ], XmNfontList, font_defs.font_list ); n++;
  XtSetValues( widget, args, n );


}

/*----------------------------------------------------------------------*/

static Widget 
  CreateActionArea( Widget          parent,
                    char            *name,
                    ActionAreaItem  *items,
                    int             num_items,
                    int             tightness,
                    int             default_index )
{

  Arg        args[ 20 ];
  int        bottom_offset;
  Widget     *created_buttons;
  Widget     default_button = NULL;
  Dimension  default_shadow;
  Widget     form;
  int        fraction_base;
  int        index;
  Cardinal   n;
  Widget     pb;
  int        top_offset;

  /* Code. */

  created_buttons = (Widget *) XtMalloc( sizeof( Widget ) * num_items );

  /* Guard against zero fraction base. */
  fraction_base = tightness * num_items - 1;
  if( fraction_base < 1 )
    fraction_base = 100;

  n = 0;
  XtSetArg( args[ n ], XmNfractionBase, fraction_base ); n++;
  form = XmCreateForm( parent, name, args, n );

  /* Create the buttons. */
  for( index = 0; index < num_items; index ++ ){
    
    n = 0;
    XtSetArg( args[ n ], XmNtopAttachment,    XmATTACH_FORM ); n++;
    XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++; 

    if( index == 0 ){
      XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_FORM ); n++;
      XtSetArg( args[ n ], XmNleftOffset,     5 ); n++;
    } else {
      XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_POSITION ); n++;
      XtSetArg( args[ n ], XmNleftPosition,   tightness * index ); n++;
    }

    if( index == ( num_items - 1 ) ){
      XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_FORM ); n++;
      XtSetArg( args[ n ], XmNrightOffset,     5 ); n++;
    } else {
      XtSetArg( args[ n ], XmNrightAttachment, XmATTACH_POSITION ); n++;
      XtSetArg( args[ n ], XmNrightPosition,   
                           tightness * index + ( tightness - 1 ) ); n++;
    }

    if( items[ index ].label != NULL ){
      XtSetArg( args[ n ], XmNlabelString, items[ index ].label ); n++;
    }

    if( index == default_index ){
      XtSetArg( args[ n ], XmNshowAsDefault, True ); n++;
      XtSetArg( args[ n ], XmNtopOffset,     5 ); n++;
      XtSetArg( args[ n ], XmNbottomOffset,  5 ); n++;

    } else {
      XtSetArg( args[ n ], XmNtopOffset,     10 ); n++;
      XtSetArg( args[ n ], XmNbottomOffset,  10 ); n++;
    }


    if( items[ index ].name != NULL )
      pb = XmCreatePushButton( form, items[ index ].name, args, n );
    else {
      XtSetArg( args[ n ], XmNmappedWhenManaged, False ); n++;
      pb = XmCreatePushButton( form, "dummyPb", args, n );
    }

    if( items[ index ].callback != NULL )
      XtAddCallback( pb, XmNactivateCallback, 
                     items[ index ].callback,items[ index ].callback_data );

    XtManageChild( pb );

    if( index == default_index )
      default_button = pb;

    created_buttons[ index ] = pb;
  }


  XtFree( (char *) created_buttons );


  return( form );

}

/*----------------------------------------------------------------------*/

static Widget 
  CreateFormDialog( Widget          parent,
                    char            *name,
                    XtCallbackProc  close_callback,
                    XtPointer       close_callback_data,
                    ActionAreaItem  *items,
                    int             num_items,
                    int             tightness,
                    int             default_index )
{

  Arg       args[ 10 ];
  Widget    action_area;
  Atom      delete_window;
  Widget    dialog_form;
  Widget    form;
  Cardinal  n;
  char      *name_buffer;
  Widget    sep;

  /* Code. */

  name_buffer = (char *) XtMalloc( strlen( name ) + 3 );

  /* The main dialog. */
  sprintf( name_buffer, "%sDf", name );

  n = 0;
  dialog_form = XmCreateFormDialog( parent, name_buffer, args, n );

  /* Set the close window callback. */
  if( close_callback != NULL ){
    delete_window = XmInternAtom( XtDisplay( dialog_form ), 
                                  "WM_DELETE_WINDOW", False );

    XmAddWMProtocolCallback( XtParent( dialog_form ), delete_window,
                             close_callback, close_callback_data );
  }

  /* The main form. */
  n = 0;
  form = XmCreateForm( dialog_form, name, args, n );

  /* Separator. */
  sprintf( name_buffer, "%sSp", name );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  sep = XmCreateSeparator( dialog_form, name_buffer, args, n );

  /* Action area. */
  sprintf( name_buffer, "%sAa", name );

  action_area = CreateActionArea( dialog_form, name_buffer, items,
                                  num_items, tightness, default_index );

  /* Fix attachments. */
  n = 0;
  XtSetArg( args[ n ], XmNleftAttachment,   XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNrightAttachment,  XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
  XtSetValues( action_area, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNleftAttachment,   XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNrightAttachment,  XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_WIDGET ); n++;
  XtSetArg( args[ n ], XmNbottomWidget,     action_area ); n++;
  XtSetValues( sep, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNtopAttachment,    XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNleftAttachment,   XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNrightAttachment,  XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_WIDGET ); n++;
  XtSetArg( args[ n ], XmNbottomWidget,     sep ); n++;
  XtSetValues( form, args, n );

  /* Manage the widgets. */
  XtManageChild( action_area );
  XtManageChild( sep );
  XtManageChild( form );


  /* Free up allocated space. */
  XtFree( (char *) name_buffer );

  return( form );

}

/*----------------------------------------------------------------------*/

static void 
  FontOkCB( Widget                      widget,
            CustDiCallbackInfo          *cb_info,
            XmPushButtonCallbackStruct  *call_data )
{

  RemoveCustDialog( widget, cb_info );

}


/*----------------------------------------------------------------------*/

static void 
  PaletteOkCB( Widget                      widget,
               CustDiCallbackInfo          *cb_info,
               XmPushButtonCallbackStruct  *call_data )
{

  ApplyPalette( widget, cb_info );

  RemoveCustDialog( widget, cb_info );

}



/*----------------------------------------------------------------------*/

static void 
  PopdownCustDialog( Widget               widget,
                     CustDiCallbackInfo  *cb_info )
{

  XtUnmanageChild( cb_info -> dialog );

}


/*----------------------------------------------------------------------*/

static void 
  RemoveCustDialog( Widget               widget,
                    CustDiCallbackInfo  *cb_info )
{

  /* Remove the dialog. */
  XtDestroyWidget( cb_info -> dialog );

  if( cb_info -> orig_palette != NULL )
    XtFree( (char *) cb_info -> orig_palette );
  if( cb_info -> orig_fontmap != NULL )
    XtFree( (char *) cb_info -> orig_fontmap );

  XtFree( (char *) cb_info );


  return;

}


/*----------------------------------------------------------------------*/

void
  xmubApplyColorResources( Widget   toplevel,
                           Boolean  update_shadows )
{

  xmubIterateTree( toplevel, (XtCallbackProc) ChangeWidgetColorsCB, 
                             (XtPointer) update_shadows );

}


/*----------------------------------------------------------------------*/

void
  xmubApplyFontResources( Widget  toplevel )
{

  xmubIterateTree( toplevel, (XtCallbackProc) ChangeWidgetFontsCB, NULL );

}


/*----------------------------------------------------------------------*/

String
  xmubCurrentFontmap( Display  *display )
{

  Bool      ok;
  String    app_class;
  String    name;
  String    fontmap;
  char      res_class[ BUFFER_LENGTH ];
  char      res_name[ BUFFER_LENGTH ];
  char      *str_type_return;
  XrmValue  value_return;

  /* Code. */

  XtGetApplicationNameAndClass( display, &name, &app_class );
  sprintf( res_name, "%s.%s", name, FONTMAP_RESOURCE_NAME );
  sprintf( res_class, "%s.%s", app_class, FONTMAP_CLASS_NAME );

  ok = XrmGetResource( XtDatabase( display ), res_name, res_class, 
                       &str_type_return, &value_return );
  if( ok )
    fontmap = XtNewString( (String) value_return.addr );
  else
    fontmap = NULL;


  return( fontmap );

}


/*----------------------------------------------------------------------*/

String
  xmubCurrentPalette( Display  *display )
{

  Bool      ok;
  String    app_class;
  String    name;
  String    palette;
  char      res_class[ BUFFER_LENGTH ];
  char      res_name[ BUFFER_LENGTH ];
  char      *str_type_return;
  XrmValue  value_return;

  /* Code. */

  XtGetApplicationNameAndClass( display, &name, &app_class );
  sprintf( res_name, "%s.%s", name, PALETTE_RESOURCE_NAME );
  sprintf( res_class, "%s.%s", app_class, PALETTE_CLASS_NAME );

  ok = XrmGetResource( XtDatabase( display ), res_name, res_class, 
                       &str_type_return, &value_return );
  if( ok )
    palette = XtNewString( (String) value_return.addr );
  else
    palette = NULL;


  return( palette );

}


/*----------------------------------------------------------------------*/

void
  xmubPopupFontDialog( Widget                   toplevel,
                       xmubResourceMappingInfo  *info )
{

  Arg                 args[ 10 ];
  CustDiCallbackInfo  *cb_info;
  Widget              form;
  int                 index;
  Widget              lb;
  int                 length;
  Widget              list;
  Cardinal            n;
  int                 num_visible;
  Widget              fntframe;
  char                *str;
  String              *str_table;
  XmString            xm;
  XmString            *xm_table;

  ActionAreaItem  button_items[] = {
    { "okBu",  NULL, (XtCallbackProc) FontOkCB,           NULL },
    { "canBu", NULL, (XtCallbackProc) CancelFontDialogCB, NULL },
  };

  /* Code. */

  /* Create list of available fonts. */
  str_table = xmubResListClasses( info, "UserFont", True, info -> mapping_db );
  
  /* Get the number of entries in the table. */
  for( length = 0; str_table[ length ] != NULL; length ++ );

  if( length == 0 )
    return;

  xm_table = (XmString *) XtMalloc( length * sizeof( XmString ) );

  for( index = 0; index < length; index ++ )
    xm_table[ index ] = XmStringCreateLocalized( str_table[ index ] );

  /* Create callback info. */
  cb_info = (CustDiCallbackInfo *) XtMalloc( sizeof( CustDiCallbackInfo ) );

  for( index = 0; index < XtNumber( button_items ); index ++ )
    button_items[ index ].callback_data = (XtPointer) cb_info;

  /* Create the dialog box for changing the stuff. */
  form = CreateFormDialog( toplevel, "fntDi", 
                           (XtCallbackProc) CancelFontDialogCB, 
                           (XtPointer) cb_info,
                           button_items, XtNumber( button_items ), 5, 0 );


  /* We want a scrolled list with the fonts in the work area. */
  n = 0;
  fntframe = XmCreateFrame( form, "fntFr", args, n );

#if XmVersion > 1001

  n = 0;
  XtSetArg( args[ n ], XmNchildType,   XmFRAME_TITLE_CHILD ); n++;
  lb = XmCreateLabel( fntframe, "fntLb", args, n );

#endif

  n = 0;
  XtSetArg( args[ n ], XmNitems,           xm_table ); n++;
  XtSetArg( args[ n ], XmNitemCount,       length ); n++;
#if XmVersion > 1001
  XtSetArg( args[ n ], XmNchildType,       XmFRAME_WORKAREA_CHILD ); n++;
#endif
  XtSetArg( args[ n ], XmNselectionPolicy, XmSINGLE_SELECT ); n++;
  list = XmCreateScrolledList( fntframe, "fntLi", args, n );

  /* Check how many items are visible. */
  n = 0;
  XtSetArg( args[ n ], XmNvisibleItemCount, &num_visible ); n++;
  XtGetValues( list, args, n );

  if( num_visible > length ){
    n = 0;
    XtSetArg( args[ n ], XmNvisibleItemCount, length ); n++;
    XtSetValues( list, args, n );
  }


  /* We should display on selection. */
  XtAddCallback( list, XmNsingleSelectionCallback,
		 (XtCallbackProc) ApplyFontCB, (XtPointer) cb_info );
  
#if XmVersion > 1001
  XtManageChild( lb );
#endif
  XtManageChild( list );
  XtManageChild( fntframe );

  /* Attach the widgets. */
  n = 0;
  XtSetArg( args[ n ], XmNtopAttachment,    XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNtopOffset,        5 ); n++;
  XtSetArg( args[ n ], XmNleftAttachment,   XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNleftOffset,       5 ); n++;
  XtSetArg( args[ n ], XmNrightAttachment,  XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNrightOffset,      5 ); n++;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNbottomOffset,     5 ); n++;
  XtSetValues( fntframe, args, n );

  /* Fill in the callback information. */
  cb_info -> toplevel        = toplevel;
  cb_info -> list            = list;
  cb_info -> dialog          = XtParent( form );
  cb_info -> info            = info;
  cb_info -> orig_palette    = NULL;
  cb_info -> orig_fontmap    = xmubCurrentFontmap( XtDisplay( toplevel ) );


  /* If a fontmap is used, preselect it in the list. */
  if( cb_info -> orig_fontmap != NULL ){

    Boolean  found = False;

    for( index = 0; str_table[ index ] != NULL; index ++ ){
      if( strcmp( cb_info -> orig_fontmap, str_table[ index ] ) == 0 ){
        found = True;
        break;
      }
    }

    if( found ){
      XmListSelectPos( list, index + 1, False );
      XmListSetPos( list, index + 1 );
    }
  }


  /* Pop it up. */
  XtManageChild( XtParent( form ) );


  /* Clean up. */
  for( index = 0; str_table[ index ] != NULL; index ++ )
    XmStringFree( xm_table[ index ] );
  for( index = 0; str_table[ index ] != NULL; index ++ )
    XtFree( (char *) str_table[ index ] );

  XtFree( (char *) xm_table );
  XtFree( (char *) str_table );


  /* Free the button labels. */
  for( index = 0; index < XtNumber( button_items ); index ++ ){
    if( button_items[ index ].label != NULL )
      XmStringFree( button_items[ index ].label );
  }

}


/*----------------------------------------------------------------------*/

void
  xmubPopupPaletteDialog( Widget                   toplevel,
                          Boolean                  update_shadows,
                          xmubResourceMappingInfo  *info )
{

  Arg                 args[ 10 ];
  CustDiCallbackInfo  *cb_info;
  Widget              form;
  int                 index;
  Widget              lb;
  int                 length;
  Widget              list;
  Cardinal            n;
  Widget              palframe;
  char                *str;
  String              *str_table;
  XmString            *xm_table;

  ActionAreaItem  button_items[] = {
    { "okBu",  NULL, (XtCallbackProc) PaletteOkCB,           NULL },
    { "canBu", NULL, (XtCallbackProc) CancelPaletteDialogCB, NULL }, 
  };

  /* Code. */


  /* Create list of available palettes. */
  str_table = xmubResListClasses( info, "PaletteBg", True,
                                  info -> mapping_db );
  
  /* Get the number of entries in the table. */
  for( length = 0; str_table[ length ] != NULL; length ++ );

  if( length == 0 )
    return;

  xm_table = (XmString *) XtMalloc( length * sizeof( XmString ) );

  for( index = 0; index < length; index ++ )
    xm_table[ index ] = XmStringCreateLocalized( str_table[ index ] );

  /* Set the button labels. */

  /* Create callback info. */
  cb_info = (CustDiCallbackInfo *) XtMalloc( sizeof( CustDiCallbackInfo ) );

  for( index = 0; index < XtNumber( button_items ); index ++ )
    button_items[ index ].callback_data = (XtPointer) cb_info;

  /* Create the dialog box for changing the stuff. */
  form = CreateFormDialog( toplevel, "palDi", 
                           (XtCallbackProc) CancelPaletteDialogCB, 
                           (XtPointer) cb_info,
                           button_items, XtNumber( button_items ), 5, 0 );

  /* We want a scrolled list with the palettes in the work area. */
  n = 0;
  palframe = XmCreateFrame( form, "palFr", args, n );

#if XmVersion > 1001
  n = 0;
  XtSetArg( args[ n ], XmNchildType,   XmFRAME_TITLE_CHILD ); n++;
  lb = XmCreateLabel( palframe, "palLb", args, n );
#endif

  n = 0;
  XtSetArg( args[ n ], XmNitems,           xm_table ); n++;
  XtSetArg( args[ n ], XmNitemCount,       length ); n++;
#if XmVersion > 1001
  XtSetArg( args[ n ], XmNchildType,       XmFRAME_WORKAREA_CHILD ); n++;
#endif
  XtSetArg( args[ n ], XmNselectionPolicy, XmSINGLE_SELECT ); n++;

  list = XmCreateScrolledList( palframe, "palLi", args, n );

  /* We should display on selection. */
  XtAddCallback( list, XmNsingleSelectionCallback,
		 (XtCallbackProc) ApplyPaletteCB, (XtPointer) cb_info );
  
#if XmVersion > 1001
  XtManageChild( lb );
#endif
  XtManageChild( list );
  XtManageChild( palframe );

  /* Attach the widgets. */
  n = 0;
  XtSetArg( args[ n ], XmNtopAttachment,    XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNtopOffset,        5 ); n++;
  XtSetArg( args[ n ], XmNleftAttachment,   XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNleftOffset,       5 ); n++;
  XtSetArg( args[ n ], XmNrightAttachment,  XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNrightOffset,      5 ); n++;
  XtSetArg( args[ n ], XmNbottomAttachment, XmATTACH_FORM ); n++;
  XtSetArg( args[ n ], XmNbottomOffset,     5 ); n++;
  XtSetValues( palframe, args, n );

  /* Fill in the callback information. */
  cb_info -> toplevel       = toplevel;
  cb_info -> list           = list;
  cb_info -> dialog         = XtParent( form );
  cb_info -> info           = info;
  cb_info -> orig_palette   = xmubCurrentPalette( XtDisplay( toplevel ) );
  cb_info -> update_shadows = update_shadows;
  cb_info -> orig_fontmap   = NULL;


  /* If a palette is used, preselect it in the list. */
  if( cb_info -> orig_palette != NULL ){

    Boolean  found = False;

    for( index = 0; str_table[ index ] != NULL; index ++ ){
      if( strcmp( cb_info -> orig_palette, str_table[ index ] ) == 0 ){
        found = True;
        break;
      }
    }

    if( found ){
      XmListSelectPos( list, index + 1, False );
      XmListSetPos( list, index + 1 );
    }
  }


  /* Pop it up. */
  XtManageChild( XtParent( form ) );


  /* Clean up. */
  for( index = 0; str_table[ index ] != NULL; index ++ )
    XmStringFree( xm_table[ index ] );
  for( index = 0; str_table[ index ] != NULL; index ++ )
    XtFree( (char *) str_table[ index ] );

  XtFree( (char *) xm_table );
  XtFree( (char *) str_table );


  /* Free the button labels. */
  for( index = 0; index < XtNumber( button_items ); index ++ ){
    if( button_items[ index ].label != NULL )
      XmStringFree( button_items[ index ].label );
  }

}


/*----------------------------------------------------------------------*/

void
  xmubUpdateFontmap( char                     *fontmap,
                     xmubResourceMappingInfo  *info )
{

  xmubResUpdateResourceSet( info, FONTMAP_RESOURCE_NAME, FONTMAP_CLASS_NAME,
                            fontmap, NULL, NULL );

}


/*----------------------------------------------------------------------*/

void
  xmubUpdatePalette( char                     *palette,
                     xmubResourceMappingInfo  *info )
{

  xmubResUpdateResourceSet( info, PALETTE_RESOURCE_NAME, PALETTE_CLASS_NAME,
                            palette, NULL, NULL );

}


