%{
#include <stdio.h>
#include <ctype.h>
#include <unistd.h>
#include <assert.h>

#include "tamil.h"

/*
The following defines are not needed for flex version 2.4.7 or later
since we use -P option of flex to set the prefix.

#define yyFlexLexer          tamil_yyFlexLexer
#define yy_create_buffer     tamil_yy_create_buffer
#define yy_delete_buffer     tamil_yy_delete_buffer
#define yy_flex_debug        tamil_yy_flex_debug
#define yy_init_buffer       tamil_yy_init_buffer
#define yy_load_buffer_state tamil_yy_load_buffer_state
#define yy_switch_to_buffer  tamil_yy_switch_to_buffer
#define yyin                 tamil_yyin
#define yyleng               tamil_yyleng
#define yylex                tamil_yylex
#define yyout                tamil_yyout
#define yyrestart            tamil_yyrestart
#define yytext               tamil_yytext
#define yywrap               tamil_yywrap

*/

/* provide a default yywrap */
int tamil_yywrap () { return 1; }

static void  echo_char          PROTO((int c));
static void  tamil_char2mdu     PROTO((FILE* fp, int c));
static int   tamil_read         PROTO((void* buf, size_t size));
static char* map_keys           PROTO((int p, int q));
static void  print_octal_string PROTO((FILE* ofp,
				       unsigned char* s, unsigned char* p));
static int   is_ps_special      PROTO((unsigned char c));


#undef YY_INPUT
#define YY_INPUT(buf,result,max_size) \
   if ( (result = tamil_read((char*)buf, max_size)) < 0 )\
       YY_FATAL_ERROR( "tamil_read() in flex scanner failed" );

#define ECHOCHAR(x) echo_char (x)
#define OUTPUT_CHAR(x) fputc ((char)(x), yyout)

#define FLUSH_CHAR -2
#define FLUSH_WORD -1

%}

%x TAMIL IITS

WS1		[ \t\f]
OPT_WS1         [ \t\f]?

BEGTAMIL        \\bt
ENDTAMIL        \\et
BEGIITS1        (\*)+BEGIN[ \t]?TEXT[^\n]*\n
BEGIITS2        \\BT

%%

{BEGTAMIL}                 BEGIN(TAMIL);
^{BEGIITS1}                ECHO; BEGIN(IITS);
{BEGIITS2}                 BEGIN(IITS);

\x80|\x81|\x82|\x83|\x84|\x85|\x86|\x87       ECHO;
\x88|\x89|\x8a|\x8b|\x8c|\x8d|\x8e|\x8f       ECHO;
\x90|\x91|\x92|\x93|\x94|\x95|\x96|\x97       ECHO;
\x98|\x99|\x9a|\x9b|\x9c|\x9d|\x9e|\x9f       ECHO;
\xa0|\xa1|\xa2|\xa3|\xa4|\xa5|\xa6|\xa7       ECHO;
\xa8|\xa9|\xaa|\xab|\xac|\xad|\xae|\xaf       ECHO;
\xb0|\xb1|\xb2|\xb3|\xb4|\xb5|\xb6|\xb7       ECHO;
\xb8|\xb9|\xba|\xbb|\xbc|\xbd|\xbe|\xbf       ECHO;
\xc0|\xc1|\xc2|\xc3|\xc4|\xc5|\xc6|\xc7       ECHO;
\xc8|\xc9|\xca|\xcb|\xcc|\xcd|\xce|\xcf       ECHO;
\xd0|\xd1|\xd2|\xd3|\xd4|\xd5|\xd6|\xd7       ECHO;
\xd8|\xd9|\xda|\xdb|\xdc|\xdd|\xde|\xdf       ECHO;
\xe0|\xe1|\xe2|\xe3|\xe4|\xe5|\xe6|\xe7       ECHO;
\xe8|\xe9|\xea|\xeb|\xec|\xed|\xee|\xef       ECHO;
\xf0|\xf1|\xf2|\xf3|\xf4|\xf5|\xf6|\xf7       ECHO;
\xf8|\xf9|\xfa|\xfb|\xfc|\xfd|\xfe|\xff       ECHO;

.|\n                       ECHO;

<TAMIL,IITS>A|aa           ECHOCHAR (XK_tamil_A);
<TAMIL,IITS>I|ee|ii        ECHOCHAR (XK_tamil_I);
<TAMIL,IITS>U|oo|uu        ECHOCHAR (XK_tamil_U);
<TAMIL,IITS>E              ECHOCHAR (XK_tamil_E);
<TAMIL,IITS>ai             ECHOCHAR (XK_tamil_ai);
<TAMIL>ay/[^aeiouAEIOU]    ECHOCHAR (XK_tamil_ai);
<TAMIL,IITS>O              ECHOCHAR (XK_tamil_O);
<TAMIL,IITS>au|ow|ou       ECHOCHAR (XK_tamil_au);
          
<TAMIL,IITS>a              ECHOCHAR (XK_tamil_a);
<TAMIL,IITS>i              ECHOCHAR (XK_tamil_i);
<TAMIL,IITS>u              ECHOCHAR (XK_tamil_u);
<TAMIL,IITS>e              ECHOCHAR (XK_tamil_e);
<TAMIL,IITS>o              ECHOCHAR (XK_tamil_o);

<TAMIL>ah|q                ECHOCHAR (FLUSH_CHAR); OUTPUT_CHAR (XK_tamil_aghu);
<IITS>H                    ECHOCHAR (FLUSH_CHAR); OUTPUT_CHAR (XK_tamil_aghu);

<TAMIL,IITS>"+A"           {
                             ECHOCHAR (FLUSH_CHAR);
			     OUTPUT_CHAR (XK_tamil_thunai_A);
			   }
<TAMIL,IITS>"+e"           {
                             ECHOCHAR (FLUSH_CHAR);
			     OUTPUT_CHAR (XK_tamil_thunai_e);
			   }
<TAMIL,IITS>"+E"           {
                             ECHOCHAR (FLUSH_CHAR);
			     OUTPUT_CHAR (XK_tamil_thunai_E);
			   }
<TAMIL,IITS>"+ai"          {
                             ECHOCHAR (FLUSH_CHAR);
			     OUTPUT_CHAR (XK_tamil_thunai_ai);
			   }

<TAMIL>[kKgG]              ECHOCHAR (XK_tamil_k);
<IITS>k                    ECHOCHAR (XK_tamil_k);
<TAMIL>[cC]h?|s            ECHOCHAR (XK_tamil_s);
<IITS>c                    ECHOCHAR (XK_tamil_s);
<TAMIL>[tTdD]h             ECHOCHAR (XK_tamil_th);
<IITS>t                    ECHOCHAR (XK_tamil_th);
<TAMIL>[tTdD]              ECHOCHAR (XK_tamil_t);
<IITS>T                    ECHOCHAR (XK_tamil_t);
<TAMIL>[bBpP]              ECHOCHAR (XK_tamil_p);
<IITS>p                    ECHOCHAR (XK_tamil_p);
<TAMIL>[fF]                {
                             ECHOCHAR (FLUSH_CHAR);
			     OUTPUT_CHAR (XK_tamil_aghu);
			     ECHOCHAR (XK_tamil_p);
			   }
<TAMIL,IITS>R              ECHOCHAR (XK_tamil_R);

<TAMIL>N[G^]               ECHOCHAR (XK_tamil_ng);
<IITS>g                    ECHOCHAR (XK_tamil_ng);
<TAMIL>ny|NY               ECHOCHAR (XK_tamil_gn);
<IITS>n^|jn                ECHOCHAR (XK_tamil_gn);
<TAMIL,IITS>N              ECHOCHAR (XK_tamil_N);
<TAMIL>n^                  ECHOCHAR (XK_tamil_nn);
<IITS>n                    ECHOCHAR (XK_tamil_nn);
<TAMIL,IITS>[mM]           ECHOCHAR (XK_tamil_m);
<TAMIL>n                   ECHOCHAR (XK_tamil_n);
<IITS>n[2_]                ECHOCHAR (XK_tamil_n);

<TAMIL,IITS>[yY]           ECHOCHAR (XK_tamil_y);
<TAMIL,IITS>r              ECHOCHAR (XK_tamil_r);
<TAMIL,IITS>l              ECHOCHAR (XK_tamil_l);
<TAMIL,IITS>[vVwW]         ECHOCHAR (XK_tamil_v);
<TAMIL>[zZ]h?              ECHOCHAR (XK_tamil_zh);
<IITS>z                    ECHOCHAR (XK_tamil_zh);
<TAMIL,IITS>L              ECHOCHAR (XK_tamil_L);

<TAMIL>[hH]                {
                             ECHOCHAR (FLUSH_CHAR);
			     OUTPUT_CHAR (XK_tamil_aghu);
			     ECHOCHAR (XK_tamil_k);
			   }
<IITS>h                    {
                             ECHOCHAR (FLUSH_CHAR);
			     OUTPUT_CHAR (XK_tamil_aghu);
			     ECHOCHAR (XK_tamil_k);
			   }
<TAMIL>[jJ]|[sS]h|S|x      ECHOCHAR (XK_tamil_s);
<IITS>j|S|s|kS             ECHOCHAR (XK_tamil_s);
<IITS>SRI                  {
                             ECHOCHAR (FLUSH_CHAR);
			     OUTPUT_CHAR (XK_tamil_si);
			     OUTPUT_CHAR (XK_tamil_ri);
			   }
<TAMIL>ng?[gk]             ECHOCHAR (XK_tamil_ng); ECHOCHAR (XK_tamil_k);
<TAMIL>n([cC]h?|s|j)       ECHOCHAR (XK_tamil_gn); ECHOCHAR (XK_tamil_s);
<TAMIL>n[tTdD]h            ECHOCHAR (XK_tamil_nn); ECHOCHAR (XK_tamil_th);
<TAMIL>n[tTdD]             ECHOCHAR (XK_tamil_N); ECHOCHAR (XK_tamil_t);
<TAMIL>nr                  ECHOCHAR (XK_tamil_n); ECHOCHAR (XK_tamil_R);
<TAMIL>[tT][rR]            ECHOCHAR (XK_tamil_R); ECHOCHAR (XK_tamil_R);

<TAMIL>_n                  ECHOCHAR (FLUSH_CHAR); ECHOCHAR (XK_tamil_n);
<TAMIL>_[tTdD]h            ECHOCHAR (FLUSH_CHAR); ECHOCHAR (XK_tamil_th);
<TAMIL>_[tTdD]             ECHOCHAR (FLUSH_CHAR); ECHOCHAR (XK_tamil_t);


<TAMIL>{OPT_WS1}_{WS1}     ECHOCHAR (FLUSH_WORD); ECHO;
<TAMIL>_                   ECHOCHAR (FLUSH_CHAR);
<TAMIL>\\.*para            ECHOCHAR (FLUSH_WORD); OUTPUT_CHAR ('\n');
<TAMIL,IITS>~[^~]*~        {
                             ECHOCHAR (FLUSH_WORD);
			     fwrite (yytext + 1,
				     yyleng - 2, sizeof (unsigned char), 
				     yyout);
			   }
<TAMIL,IITS>\\~            ECHOCHAR (FLUSH_WORD); OUTPUT_CHAR ('~');
<TAMIL,IITS>.|\n           ECHOCHAR (FLUSH_WORD); ECHO;

<TAMIL,IITS>{ENDTAMIL}     ECHOCHAR (FLUSH_WORD); BEGIN(0);
<TAMIL,IITS><<EOF>>        {
                             ECHOCHAR (FLUSH_WORD);
			     BEGIN (0);
			     yyterminate ();
			   }
%%

#define IS_UYIR(x)  (XK_tamil_a <= (x) && (x) <= XK_tamil_O \
		     || (x) == XK_tamil_au)

static void
#ifdef __STDC__
echo_char (int c)
#else
echo_char (c)
   int c;
#endif /* __STDC__ */
{
  static int prev_char = FLUSH_WORD; /* FLUSH_WORD == space
                                        FLUSH_CHAR == end of char */

  if (IS_UYIR (c))
    {
      if (prev_char >= 0)
        {
         switch (c)
         {
         case XK_tamil_a:
           OUTPUT_CHAR (prev_char + 1);
           break;
         case XK_tamil_A:
           OUTPUT_CHAR (prev_char + 1);
           OUTPUT_CHAR (XK_tamil_thunai_A);
           break;
         case XK_tamil_i:
           OUTPUT_CHAR (prev_char + 2);
           break;
         case XK_tamil_I:
           OUTPUT_CHAR (prev_char + 3);
           break;
         case XK_tamil_u:
           OUTPUT_CHAR (prev_char + 4);
           break;
         case XK_tamil_U:
           OUTPUT_CHAR (prev_char + 5);
           break;
         case XK_tamil_au:
         case XK_tamil_o:
         case XK_tamil_e:
           OUTPUT_CHAR (XK_tamil_thunai_e);
           OUTPUT_CHAR (prev_char + 1);
           break;
         case XK_tamil_O:
         case XK_tamil_E:
           OUTPUT_CHAR (XK_tamil_thunai_E);
           OUTPUT_CHAR (prev_char + 1);
           break;
         case XK_tamil_ai:
           OUTPUT_CHAR (XK_tamil_thunai_ai);
           OUTPUT_CHAR (prev_char + 1);
           break;
         default:
           fprintf (stderr, "%d is not uyir\n", c);
         }
         if (c == XK_tamil_o || c == XK_tamil_O)
           OUTPUT_CHAR (XK_tamil_thunai_A);
         else if (c == XK_tamil_au)
           OUTPUT_CHAR (XK_tamil_La);
         prev_char = FLUSH_CHAR;
        }
        else
        {
         if (c == XK_tamil_au)
           {
             OUTPUT_CHAR (XK_tamil_o);
             OUTPUT_CHAR (XK_tamil_La);
           }
         else if (prev_char == FLUSH_CHAR && c == XK_tamil_i)
	   OUTPUT_CHAR (XK_tamil_y);
	 else
           OUTPUT_CHAR (c);
	 prev_char = FLUSH_CHAR;
        }
      return;
    }
    else
    {
      /* consonant */
      switch (c)
      {
      case FLUSH_WORD:
      case FLUSH_CHAR:
	if (prev_char >= 0)
	  {
	    OUTPUT_CHAR (prev_char);
	  }
	prev_char = c;
	return;
      case XK_tamil_n:
      case XK_tamil_N:
        if (prev_char == FLUSH_WORD)
          c = XK_tamil_nn;
      default:
	break;
      }
    }
  if (prev_char >= 0)
     {
       OUTPUT_CHAR (prev_char);
     }
  prev_char = c;
  return;
}

typedef struct {
  unsigned char* buf;
  int   len;
  int   curpos;
} madurai_buffer;

static madurai_buffer* mdu_obuf;

static int
#ifdef __STDC__
tamil_read (void* buf, size_t sz)
#else
tamil_read (buf, sz)
     void* buf;
     size_t sz;
#endif /* __STDC__ */
{
  int size;

  if (mdu_obuf == 0)
    {
      return read (fileno (yyin), buf, sz);
    }
  
  size = mdu_obuf->len - mdu_obuf->curpos;
  if (size > sz)
      size = sz;

  strncpy (buf, mdu_obuf->buf + mdu_obuf->curpos, size);
  mdu_obuf->curpos += size;
  return size;
}

int
#ifdef __STDC__
buf_madurai2tamil (unsigned char* buf, int sz, FILE* op)
#else
buf_madurai2tamil (buf, sz, op)
     unsigned char* buf;
     int   sz;
     FILE* op;
#endif /* __STDC__ */
{
  madurai_buffer mb;

  mb.buf = buf;
  mb.len = sz;
  mb.curpos = 0;
  mdu_obuf = &mb;

  yyout = op ? op: stdout;

  if (yy_current_buffer)
    {
      yyin = stdin;
      YY_NEW_FILE;
    }
  yylex ();

  mdu_obuf = 0;
  if (mb.len != mb.curpos)
    return -1;
  return 0;
}

void
#ifdef __STDC__
fn_madurai2tamil (char* ifile, char* ofile)
#else
fn_madurai2tamil (ifile, ofile)
  char* ifile;
  char* ofile;
#endif /* __STDC__ */
{
  FILE* ip = 0;
  FILE* op = 0;

  if (ifile && (ip = fopen (ifile, "r")) == 0)
    {
      fprintf (stderr, "could not open input file %s. using stdin instead.\n",
	       ifile);
    }

  if (ofile && (op = fopen (ofile, "w")) == 0)
    {
      fprintf (stderr, "could not open output file %s. using stdout instead.\n", ofile);
    }

  fp_madurai2tamil (ip, op);

  if (ip)
    fclose (ip);
  if (op)
    fclose (op);
}

void
#ifdef __STDC__
fp_madurai2tamil (FILE* ip, FILE* op)
#else
fp_madurai2tamil (ip, op)
     FILE* ip;
     FILE* op;
#endif /* __STDC__ */
{
  yyin = (ip) ? ip: stdin;
  yyout = (op) ? op: stdout;

  if (yy_current_buffer)
    {
      YY_NEW_FILE;
    }
  yylex ();
}

static char madurai [][4] = {
  "a", "A", "i", "I", "u", "U", "e", "E", "ai", "o", "O", "q",
  "k", "ka", "ki", "kI", "ku", "kU",
  "NG", "NGa", "NGi", "NGI", "NGu", "NGU",
  "s", "sa", "si", "sI", "su", "sU",
  "ny", "nya", "nyi", "nyI", "nyu", "nyU",
  "_t", "_ta", "_ti", "_tI", "_tu", "_tU",
  "_N", "_Na", "_Ni", "_NI", "_Nu", "_NU",
  "th", "tha", "thi", "thI", "thu", "thU",
  "n^", "n^a", "n^i", "n^I", "n^u", "n^U",
  "p", "pa", "pi", "pI", "pu", "pU",
  "m", "ma", "mi", "mI", "mu", "mU",
  "_y", "_ya", "_yi", "_yI", "_yu", "_yU",
  "r", "ra", "ri", "rI", "ru", "rU",
  "l", "la", "li", "lI", "lu", "lU",
  "v", "va", "vi", "vI", "vu", "vU",
  "zh", "zha", "zhi", "zhI", "zhu", "zhU",
  "L", "La", "Li", "LI", "Lu", "LU",
  "R", "Ra", "Ri", "RI", "Ru", "RU",
  "_n", "_na", "_ni", "_nI", "_nu", "_nU",
  "", "", "", "", "+ai", "+e", "+E", "+A"
};

#define IS_THUNAI(x) (XK_tamil_thunai_ai <= (x) && (x) <= XK_tamil_thunai_A)

static void
#ifdef __STDC__
tamil_char2mdu (FILE* op, int c)
#else
tamil_char2mdu (op, c)
  FILE* op;
  int   c;
#endif /* __STDC__ */
{
  static int curchar = 0;
  static int pthunai = 0;
  static int lthunai = 0;
  
  if (c == -1)
    {
      /* send the previous chars */
      if (curchar)
        {
	  if (lthunai == XK_tamil_thunai_A)
            {
              fprintf (op, "%s", madurai [curchar - 128 - 1]);
	      if (pthunai == 0)
		fprintf (op, "A");
	      else if (pthunai == XK_tamil_thunai_e)
                fprintf (op, "o");
              else
                fprintf (op, "O");
              pthunai = curchar = lthunai = 0;
	    }
          else if (lthunai == XK_tamil_La)
            {
              if (pthunai == XK_tamil_thunai_e)
                {
                  fprintf (op, "%sow", madurai [curchar - 128 - 1]);
		  pthunai = curchar = lthunai = 0;
                }
              else if (curchar == XK_tamil_o)
		{
		  fprintf (op, "ow");
		  pthunai = curchar = lthunai = 0;
		}
	      else
                {
                  lthunai = 0;
                  tamil_char2mdu (op, -1);
		  curchar = XK_tamil_La;
		}
            }
          else if (lthunai > 0)
            {
              int ch = lthunai;
              lthunai = 0;
              tamil_char2mdu (op, ch);
            }
	  else if (curchar == XK_tamil_o)
	    {
	      fprintf (op, "o");
	      pthunai = lthunai = curchar = 0;
	    }
          else
            {
              /* lthunai == 0 */
              fprintf (op, "%s", madurai [curchar - 128 - 1]);
              if (pthunai == XK_tamil_thunai_e)
                fprintf (op, "e");
              else if (pthunai == XK_tamil_thunai_E)
                fprintf (op, "E");
              else if (pthunai == XK_tamil_thunai_ai)
                fprintf (op, "ai");
	      else
		fprintf (op, "a");
              pthunai = curchar = lthunai = 0;
            }
        }
      else if (pthunai)
      {
	  switch (pthunai) {
	  case XK_tamil_thunai_A:
	      fprintf (op, "+A");
	      break;
	  case XK_tamil_thunai_e:
	      fprintf (op, "+e");
	      break;
	  case XK_tamil_thunai_E:
	      fprintf (op, "+E");
	      break;
	  case XK_tamil_thunai_ai:
	      fprintf (op, "+ai");
	      break;
	  default:
	      break;
	  }
	  pthunai = curchar = lthunai = 0;
      }
    }             
  else if (lthunai)
    {
      tamil_char2mdu (op, -1);
      tamil_char2mdu (op, c);
    }
  else if (curchar == 0)
    {
      if (IS_THUNAI (c))
        {
	  if (c != XK_tamil_thunai_A)
	    {
	      if (pthunai)
		tamil_char2mdu (op, -1);
	      pthunai = c;
	    }
          else
            {
              fprintf (op, "+A");
	      pthunai = curchar = lthunai = 0;
            }
	}
      else if (c == XK_tamil_o)
        {
          pthunai = 0;
          curchar = c;
        }
      else if (IS_UYIR (c))
        {
          fprintf (op, "%s", madurai [c - 128]);
          pthunai = curchar = lthunai = 0;
        }
      else if ((c - XK_tamil_ka)%6 == 0)
        {
          curchar = c;
        }
      else
	{
	  pthunai = curchar = lthunai = 0;
	  fprintf (op, "%s", madurai [c - 128]);
	}
    }
  else
    {
      if (c == XK_tamil_thunai_A || c == XK_tamil_La)
	{
	  lthunai = c;
	  tamil_char2mdu (op, -1);
	}
      else
	{
	  tamil_char2mdu (op, -1);
	  tamil_char2mdu (op, c);
	}
    }
}

void
#ifdef __STDC__
fn_tamil2madurai (char* ifile, char* ofile)
#else
fn_tamil2madurai (ifile, ofile)
  char* ifile;
  char* ofile;
#endif /* __STDC__ */
{
  FILE* ip = 0;
  FILE* op = 0;

  if (ifile && (ip = fopen (ifile, "r")) == 0)
    {
      fprintf (stderr, "could not open input file %s\n", ifile);
      fprintf (stderr, "using stdin instead\n");
    }

  if (ofile && (op = fopen (ofile, "w")) == 0)
    {
      fprintf (stderr, "could not open output file %s\n", ofile);
      fprintf (stderr, "using stdout instead\n");
    }

  fp_tamil2madurai (ip, op);

  if (op)
    fclose (op);
  if (ip)
    fclose (ip);
}

void
#ifdef __STDC__
fp_tamil2madurai (FILE* ip, FILE* op)
#else
fp_tamil2madurai (ip, op)
     FILE* ip;
     FILE* op;
#endif /* __STDC__ */
{
  unsigned char buf [1024];
  int rc; /* read count */

  if (ip == 0)
    ip = stdin;
  
  if (op == 0)
    op = stdout;
  
  while ((rc = fread (buf, 1, 1024, ip)) > 0)
    buf_tamil2madurai (buf, rc, op);
  buf_tamil2madurai (0, 0, op);
  
  if (rc == -1)
    {
      perror ("read");
    }
}
  

int
#ifdef __STDC__
buf_tamil2madurai (unsigned char* buf, int rc, FILE* op)
#else
buf_tamil2madurai (buf, rc, op)
  unsigned char* buf;
  int   rc;
  FILE* op;
#endif /* __STDC__ */
{
  static int tamilmode = 0;
  int i = 0;
  for (i = 0; i < rc; i++)
    if (tamilmode)
      {
	if ('a' <= buf [i] && buf [i] <= 'z'
	    || 'A' <= buf [i] && buf [i] <= 'Z')
	  {
	    tamilmode = 0;
	    tamil_char2mdu (op, -1);
	    fprintf (op, "\\et%c", buf [i]);
	  }
	else if (buf [i] >= 128)
	  tamil_char2mdu (op, buf [i]);
	else
	  {
	    tamil_char2mdu (op, -1);
	    fputc (buf [i], op);
	  }
      }
    else if (buf [i] >= 128)
      {
	fprintf (op, "\\bt");
	tamil_char2mdu (op, buf [i]);
	tamilmode = 1;
      }
    else
      fputc (buf [i], op);
  
  /* if rc is 0, check to see end tamilmode */
  if (rc == 0 && tamilmode)
    {
      tamil_char2mdu (op, -1);  
      fprintf (op, "\\et");
      tamilmode = 0;
    }
  if (i < rc)
    return -1;
  return 0;
}  

String
#ifdef __STDC__
str_tamil2madurai (String string, int can_delete)
#else
str_tamil2madurai (string, can_delete)
     String string;
#endif
{
    /* convert to madurai */
    FILE* tp;
    char  tmpfile [L_tmpnam];
    long len;
    
    tmpnam (tmpfile);
    tp = fopen (tmpfile, "w+");
    unlink (tmpfile);
    tp && buf_tamil2madurai (string, strlen (string), tp) != -1
	&& buf_tamil2madurai (0, 0, tp);
    fseek (tp, 0L, 2);
    len = ftell (tp);
    
    if (can_delete)
	XtFree (string);
    string = XtMalloc (sizeof (unsigned char) *  (len + 1));
    fseek (tp, 0L, 0);
    read (fileno (tp), string, sizeof (unsigned char)*len);
    string [len] = '\0';
    fclose (tp);
    return string;
}

String
#ifdef __STDC__
str_madurai2tamil (String string, int can_delete)
#else
str_madurai2tamil (string, can_delete)
     String string;
#endif
{      
    /* convert string to tamil */
    FILE* tp;
    char  tmpfile [L_tmpnam];
    int len;
    
    tmpnam (tmpfile);
    tp = fopen (tmpfile, "w+");
    unlink (tmpfile);
    if (tp == 0 || buf_madurai2tamil (string, strlen (string), tp) == -1)
	fprintf (stderr, "buf_madurai2tamil error\n");
    fseek (tp, 0L, 2);
    len = ftell (tp);
    
    if (can_delete)
	XtFree (string);
    string = XtMalloc (sizeof (unsigned char) * (len + 1));
    fseek (tp, 0L, 0);
    read (fileno (tp), string, sizeof (unsigned char) * len);
    fclose (tp);
    
    return string;
}

#include "ps.h"

static void
#ifdef __STDC__
print_octal_string (FILE* ofp, unsigned char* s, unsigned char* p)
#else
print_octal_string (ofp, s, p)
     FILE* ofp;
     unsigned char* s;
     unsigned char* p;
#endif
{
  if (s == p)
    return;
  fputc ('(', ofp);
  while (s != p)
    {
      if (*s >= 128)
	*s &= 0x7f;
      if (is_ps_special (*s))
	{
	  fputc ('\\', ofp);
	  fputc (*s, ofp);
	}
      else if (*s < 32 || *s > '~')
	fprintf (ofp, "\\%03o", (int)*s);
      else
	fputc (*s, ofp);

      s++;
    }
  fputc (')', ofp);
}

static int
#ifdef __STDC__
is_ps_special (unsigned char c)
#else
is_ps_special (c)
     unsigned char c;
#endif
{
  switch (c)
    {
    case '%': case '(': case ')': case '\\': case '{': case '}':
    case '[': case ']': case '<': case '>':
      return 1;
    }
  return 0;
}

void
#ifdef __STDC__
fp_tamil2ps (FILE* ifp, FILE* ofp)
#else
fp_tamil2ps (ifp, ofp)
     FILE* ifp;
     FILE* ofp;
#endif /* __STDC__ */
{
  unsigned char buf [1024];
  int  bufsz = 0;
  int  pageno = 1; /* current page number */
  int  ymove = 0;
  int  font = -1; /* -1 == none, 0 == english, 1 == tamil */
  char h = 0; /*'p' print (non-mei string)
                'P' print (mei letter) i.e. put a dot on top of the letter
                    the dot is printed using ({)p in /P definition.
                'h' print (non-mei string) and a space
                'H' print (mei letter) and a space
                's' just print a space
              */

  /* if you change the value of DELTA, accordingly change /s{25 w}B definition
     also in ps.h @ line 45.
     DELTA is the width for one space char */
#define DELTA 25

  if (ifp == 0)
    ifp = stdin;
  
  if (ofp == 0)
    ofp = stdout;

  print_ps_prolog (ofp);
  fprintf (ofp, "%%%%Page: %d %d\nbop", pageno, pageno);
  pageno++;

  while ((bufsz = fread (buf, sizeof (unsigned char), 1023, ifp)) > 0)
    {
      unsigned char* s = buf;
      buf [bufsz] = 0;

      while (*s)
	{
	  unsigned char* p = s;

	  while (*p && *p < 128 && !isspace (*p))
	    p++;
	  if (p != s)
	    {
	      /* then we are ready to print p-s chars as English */

	      if (h) { fputc (h, ofp); h=0; }

              /* if the current font is not english, set it to english */
	      if (font != 0)
		{
		  fprintf (ofp, " Fb");
		  font = 0;
		}
	      print_octal_string (ofp, s, p);
	      h = 'p';
	      s = p;
	      continue;
	    }
	  while (*p >= 128 &&
		 (*p < XK_tamil_k
		  || *p > XK_tamil_n
		  || (*p - XK_tamil_k)%6))
	    p++;
	  if (p != s)
	    {
	      /* we are ready to print p-s chars as Tamil */
	      /* no mei letters is printed in this if-arm */

	      if (h) { fputc (h, ofp); h=0; }

              /* if the current font is not Tamil, set it to Tamil */
	      if (font != 1)
		{
		  font = 1;
		  fprintf (ofp, " Fa");
		}
	      print_octal_string (ofp, s, p);
	      h = 'p';
	      s = p;
	      continue;
	    }
	  if (*s >= XK_tamil_k
	      && *s <= XK_tamil_n
	      && (*s - XK_tamil_k) % 6 == 0)
	    {
	      /* You come here to print a mei letter */

	      if (h) { fputc (h, ofp); h=0; }

              /* if the current font is not Tamil, set it to Tamil */
	      if (font != 1)
		{
		  font = 1;
		  fprintf (ofp, " Fa");
		}
	      *s &= 0x7f;
	      ++*s;
	      fprintf (ofp, "(\\%03o)", (int)*s);
	      h = 'P';
	      s++;
	      continue;
	    }
	  if (*s == '')
	    {
	      s++;
	      if (h) { fputc (h, ofp); h=0; }
	      fprintf (ofp, "\neop\n%%%%Page: %d %d\nbop",
		       pageno, pageno);
	      ymove = 0;
	      font = -1;
	      pageno++;
	      continue;
	    }
	      
	  while (*p && *p == '\n')
	    p++;
	  if (p != s)
	    {
	      ymove += (p-s);
	      if (ymove > 54)
		{
		  if (h) { fputc (h, ofp); h=0; }
		  fprintf (ofp, "\neop\n%%%%Page: %d %d\nbop",
			   pageno, pageno);
		  ymove = 0;
		  font  = -1;
		  pageno++;
		}
	      else
		{
		  if (h == 'p' && (p-s) == 1) 
		    fputs ("n\n", ofp);
		  else
		    {
		      if (h) fputc (h, ofp);
		      fprintf (ofp, " %d b\n", (int)(p-s));
		    }
		  h = 0;
		}
	      s = p;
	      continue;
	    }

	  {
	    int ns = 0;
	    while (*p && (*p == ' ' || *p == '\t'))
	      {
		if (*p == ' ')
		  ns++;
		else
		  ns += 4;
		p++;
	      }
	    if (ns == 1)
	      {
		if (h == 'p')
		  {
		    fputc ('h', ofp);
		    h = 0;
		  }
		else if (h == 'P')
		  {
		    fputc ('H', ofp);
		    h = 0;
		  }
		else
		  {
		    if (h) { fputc (h, ofp); h=0; }
		    fputs (" s", ofp);
		  }
		s = p;
		continue;
	      }
	    else if (ns > 1)
	      {
		if (h) { fputc (h, ofp); h=0; }
		ns *= DELTA;
		fprintf (ofp, " %d w", ns);
		s = p;
		continue;
	      }
	  }
	  
	  s++;
	}
    }
  if (h) { fputc (h, ofp); h=0; }

  /* Following if statement is commented out because eop
     is not generated when the input file has only one line
     and it ends without an newline */
  /* if (pageno > 0 && ymove != 0) */
    fprintf (ofp, "\neop\n");

  print_ps_trailer (ofp);
}

#define is_valid_char(c) ('a' <= (c) && (c) <= 'z' \
                             || 'A' <= (c) && (c) <= 'Z' \
			     || (c) == '+')

void
#ifdef __STDC__
XtAppMainLoop (XtAppContext app_context)
#else
XtAppMainLoop (app_context)
     XtAppContext app_context;
#endif /* __STDC__ */
{
  int    tamilmode = 0;
  XEvent lastkey;
  XEvent esckey;
  char   escbuf [8];
  int    lastchar = -1;
  KeySym modlist [1];
  int    modcount = 0;
  int    i;
  
#ifndef NEED_APPINITIALIZE
  {
    extern void init_ascii_src ();

    init_ascii_src ();
  }
#endif

   for (;;)
    {
      XEvent ev;
      Display* d;

      XtAppNextEvent (app_context, &ev);
    
      if (ev.type == KeyPress)
	{
	  char    buf [64];
	  KeySym  ks;
          int     count;

	  count = XLookupString (&ev.xkey, buf, 64, &ks, 0);
	  d = ev.xkey.display;
      

#ifdef DEBUG
	  buf [count] = 0;
	  fprintf (stderr, "(len ks kc state string) = %d %d %d 0x%x %s\n",
		   count, ks, ev.xkey.keycode, ev.xkey.state, buf);
#endif
          if (ks == XK_Escape)
	    {
	      if (lastchar >= 0)
		XtDispatchEvent (&lastkey);
	      if (tamilmode == 0)
		{
		  esckey = ev;
		  escbuf [0] = buf [0];
		  tamilmode = ks;
		}
	      else
		{
		  XRebindKeysym (d, tamilmode, 0, 0, escbuf, 1);
		  tamilmode = 0;
		}
	      continue;
	    }
	  if (count == 1)
	    {
	      if (tamilmode && is_valid_char (buf [0]))
		{
		  char* ts;
		  if (lastchar < 0)
		    {
		      lastkey = ev;
		      lastchar = ks;
		      continue;
		    }
		  else if (ts = map_keys (lastchar, ks))
		    {
		      int len = strlen (ts);
		      
#ifdef DEBUG
		      fprintf (stderr, "%s\n", ts);
#endif
		      XRebindKeysym (d, tamilmode, 0, 0, ts, len);
		      XtDispatchEvent (&esckey); 
		      lastchar = -1;
		      continue;
		    }
		  else
		    {
		      XtDispatchEvent (&lastkey);
		      lastchar = -1;
		    }
		}
	      else if (lastchar >= 0)
		{
		  XtDispatchEvent (&lastkey);
		  lastchar = -1;
		}
	    }
	}
      XtDispatchEvent (&ev);
    }
}

void
XtMainLoop ()
{
    XtAppMainLoop ((XtAppContext) _XtDefaultAppContext ());
}


#ifdef NEED_APPINITIALIZE

Widget
#ifdef __STDC__
XtAppInitialize (
     XtAppContext*     appcon,
     String            appcl,
     XrmOptionDescRec* opts,
     Cardinal          nopts,
     Cardinal*         ac,
     String*           av,
     String*           res,
     ArgList           args,
     Cardinal          nargs)
#else
XtAppInitialize (appcon, appcl, opts, nopts, ac, av, res, args, nargs)
     XtAppContext*     appcon;
     String            appcl;
     XrmOptionDescRec* opts;
     Cardinal          nopts;
     Cardinal*         ac;
     String*           av;
     String*           res;
     ArgList           args;
     Cardinal          nargs;
#endif /* __STDC__ */
{
  extern void init_ascii_src ();

  void init_ascii_src ();
  extern Widget _XtAppInitialize (
#ifdef __STDC__
				  XtAppContext*,
				  String,
				  XrmOptionDescRec*,
				  Cardinal,
				  Cardinal*,
				  String*,
				  String*,
				  ArgList,
				  Cardinal
#endif
				  );
  return _XtAppInitialize (appcon, appcl, opts, nopts,
			   ac, av, res, args, nargs);
}

Widget
#ifdef __STDC__
XtInitialize(String            name,
	     String            classname,
	     XrmOptionDescRec* options,
	     Cardinal          num_options,
	     Cardinal*         argc)
#else
XtInitialize(name, classname, options, num_options, argc, argv)
     String            name;
     String            classname;
     XrmOptionDescRec* options;
     Cardinal          num_options;
     Cardinal*         argc;
     String*           argv;
#endif /* __STDC__ */
{
    XtAppContext app_con;

    return XtAppInitialize(&app_con, classname, options, num_options,
			   argc, argv, NULL, NULL, (Cardinal) 0);
}

#endif /* NEED_APPINITIALIZE */

static char*
#ifdef __STDC__
map_keys (int p, int q)
#else
map_keys (p, q)
     int p;
     int q;
#endif /* __STDC__ */
{
  static char ts [4];
  switch (q)
    {
    case 'a':
      switch (p)
	{
	case 'a':
	  ts [0] = XK_tamil_a;
	  break;
	case 'k':
	  ts [0] = XK_tamil_ka;
	  break;
	case 'f':
	  ts [0] = XK_tamil_nga;
	  break;
	case 's':
	  ts [0] = XK_tamil_sa;
	  break;
	case 'g':
	  ts [0] = XK_tamil_gna;
	  break;
	case 't':
	  ts [0] = XK_tamil_ta;
	  break;
	case 'N':
	  ts [0] = XK_tamil_Na;
	  break;
	case 'd':
	  ts [0] = XK_tamil_tha;
	  break;
	case 'b':
	  ts [0] = XK_tamil_nna;
	  break;
	case 'p':
	  ts [0] = XK_tamil_pa;
	  break;
	case 'm':
	  ts [0] = XK_tamil_ma;
	  break;
	case 'y':
	  ts [0] = XK_tamil_ya;
	  break;
	case 'r':
	  ts [0] = XK_tamil_ra;
	  break;
	case 'l':
	  ts [0] = XK_tamil_la;
	  break;
	case 'v':
	  ts [0] = XK_tamil_va;
	  break;
	case 'z':
	  ts [0] = XK_tamil_zha;
	  break;
	case 'L':
	  ts [0] = XK_tamil_La;
	  break;
	case 'R':
	  ts [0] = XK_tamil_Ra;
	  break;
	case 'n':
	  ts [0] = XK_tamil_na;
	  break;
	default:
	  return 0;
	}
      ts [1] = 0;
      return ts;
    case 'A':
      switch (p)
	{
	case 'a':
	  ts [0] = XK_tamil_A;
	  ts [1] = 0;
	  return ts;
	case '+':
	  ts [0] = XK_tamil_thunai_A;
	  ts [1] = 0;
	  return ts;
	default:
	  if (map_keys (p, 'a'))
	    {
	      ts [1] = XK_tamil_thunai_A;
	      ts [2] = 0;
	      return ts;
	    }
	}
      return 0;
    case 'i':
      switch (p)
	{
	case 'i':
	  ts [0] = XK_tamil_i;
	  ts [1] = 0;
	  return ts;
	case 'a':
	  ts [0] = XK_tamil_ai;
	  ts [1] = 0;
	  return ts;
	default:
	  if (map_keys (p, 'a'))
	    {
	      ts [0] += 1;
	      return ts;
	    }
	}
      return 0;
    case 'I':
      switch (p)
	{
	case 'i':
	  ts [0] = XK_tamil_I;
	  ts [1] = 0;
	  return ts;
	case 'a':
	  ts [0] = XK_tamil_ai;
	  ts [1] = 0;
	  return ts;
	default:
	  if (map_keys (p, 'a'))
	    {
	      ts [0] += 2;
	      return ts;
	    }
	}
      return 0;
    case 'u':
      switch (p)
	{
	case 'u':
	  ts [0] = XK_tamil_u;
	  ts [1] = 0;
	  return ts;
	case 'a':
	case 'o':
	  ts [0] = XK_tamil_o;
	  ts [1] = XK_tamil_La;
	  ts [2] = 0;
	  return ts;
	default:
	  if (map_keys (p, 'a'))
	    {
	      ts [0] += 3;
	      return ts;
	    }
	}
      return 0;
    case 'U':
      switch (p)
	{
	case 'u':
	  ts [0] = XK_tamil_U;
	  ts [1] = 0;
	  return ts;
	case 'a':
	case 'o':
	  ts [0] = XK_tamil_o;
	  ts [1] = XK_tamil_La;
	  ts [2] = 0;
	  return ts;
	default:
	  if (map_keys (p, 'a'))
	    {
	      ts [0] += 4;
	      return ts;
	    }
	}
      return 0;
    case 'e':
      switch (p)
	{
	case 'e':
	  ts [0] = XK_tamil_e;
	  ts [1] = 0;
	  return ts;
	case 'a':
	  return 0;
	case '+':
	  ts [0] = XK_tamil_thunai_e;
	  ts [1] = 0;
	  return ts;
	default:
	  if (map_keys (p, 'a'))
	    {
	      ts [1] = ts [0];
	      ts [0] = XK_tamil_thunai_e;
	      ts [2] = 0;
	      return ts;
	    }
	}
      return 0;
    case 'E':
      switch (p)
	{
	case 'e':
	  ts [0] = XK_tamil_E;
	  ts [1] = 0;
	  return ts;
	case 'a':
	  return 0;
	case '+':
	  ts [0] = XK_tamil_thunai_E;
	  ts [1] = 0;
	  return ts;
	default:
	  if (map_keys (p, 'a'))
	    {
	      ts [1] = ts [0];
	      ts [0] = XK_tamil_thunai_E;
	      ts [2] = 0;
	      return ts;
	    }
	}
      return 0;
    case 'y':
      switch (p)
	{
	case 'y':
	  ts [0] = XK_tamil_y;
	  ts [1] = 0;
	  return ts;
	case 'a':
	  ts [0] = XK_tamil_ai;
	  ts [1] = 0;
	  return ts;
	case '+':
	  ts [0] = XK_tamil_thunai_ai;
	  ts [1] = 0;
	  return ts;
	default:
	  if (map_keys (p, 'a'))
	    {
	      ts [1] = ts [0];
	      ts [0] = XK_tamil_thunai_ai;
	      ts [2] = 0;
	      return ts;
	    }
	}
      return 0;
    case 'Y':
      switch (p)
	{
	case 'a':
	  ts [0] = XK_tamil_ai;
	  ts [1] = 0;
	  return ts;
	case '+':
	  ts [0] = XK_tamil_thunai_ai;
	  ts [1] = 0;
	  return ts;
	default:
	  if (map_keys (p, 'a'))
	    {
	      ts [1] = ts [0];
	      ts [0] = XK_tamil_thunai_ai;
	      ts [2] = 0;
	      return ts;
	    }
	}
      return 0;
    case 'o':
      switch (p)
	{
	case 'o':
	  ts [0] = XK_tamil_o;
	  ts [1] = 0;
	  return ts;
	default:
	  if (map_keys (p, 'e'))
	    {
	      ts [2] = XK_tamil_thunai_A;
	      ts [3] = 0;
	      return ts;
	    }
	}
      return 0;
    case 'O':
      switch (p)
	{
	case 'o':
	  ts [0] = XK_tamil_O;
	  ts [1] = 0;
	  return ts;
	default:
	  if (map_keys (p, 'E'))
	    {
	      ts [2] = XK_tamil_thunai_A;
	      ts [3] = 0;
	      return ts;
	    }
	}
      return 0;
    case 'w':
      switch (p)
	{
	case 'a':
	case 'o':
	  ts [0] = XK_tamil_o;
	  ts [1] = XK_tamil_La;
	  ts [2] = 0;
	  return ts;
	default:
	  if (map_keys (p, 'e'))
	    {
	      ts [2] = XK_tamil_La;
	      ts [3] = 0;
	      return ts;
	    }
	}
      return 0;
    case 'k':
    case 'q':
    case 'h':
      if (p == 'a')
	{
	  ts [0] = XK_tamil_aghu;
	  ts [1] = 0;
	  return ts;
	}
    default:
      if (map_keys (p, 'a'))
	{
	  ts [0]--;
	  return ts;
	}
      break;
    }
  return 0;
}
