/* Copyright (C) 1994 Groupe BULL. See file COPYRIGHT for details */
/*
 *
 * $Id: kts_protocol.c,v 1.2 1994/12/15 13:39:08 beust Exp $
 */

#include <stdio.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "kts_main.h"

/***************************************************************************
 * Private
 ***************************************************************************/

static void
kts_readProtocolFile(GV gv, FILE *f, int length)
{
   char *buf = (char *) malloc(length + 2), *pbuf;
   char executable[256], *pex;
   char mode[256], *pmode;
   char krs[256], *pkrs;
   int nread, i;
   Bool isObserved;

#define IS_BLANK(c) (' ' == c || '\t' == c || '\n' == c)
#define SKIP_BLANKS while (*pbuf && IS_BLANK(*pbuf)) pbuf++
#define SKIP_TO(c) while (*pbuf && (IS_BLANK(c) || c != *pbuf)) pbuf++;
#define RETRIEVE_NEXT_STRING(var) while (*pbuf && ! IS_BLANK(*pbuf)) *var++ = *pbuf++;

   nread = fread(buf, 1, length, f);
   buf[length] = '\0';
   pbuf = buf;
   ASSERT(nread == length);
   while ('\0' != *pbuf) {

      /* skip blank until an @ is found */
      SKIP_TO('@');
      ASSERT('@' == *pbuf);
      pbuf++;
      SKIP_BLANKS;

      /* store the executable name */
      pex = executable;
      RETRIEVE_NEXT_STRING(pex);
      *pex = '\0';
      SKIP_BLANKS;

      /*
      ** At most two sections following one for observe, one for handle
      */
      while (*pbuf && *pbuf != '@') {
	 /* next keyword  : observe or handle */
	 pmode = mode;
	 RETRIEVE_NEXT_STRING(pmode);
	 *pmode = '\0';
	 SKIP_TO('{');
	 pbuf++;
	 SKIP_BLANKS;

	 /* retrieve all the krs'es for this keyword */
	 while (*pbuf && '}' != *pbuf) {
	    pkrs = krs;
	    RETRIEVE_NEXT_STRING(pkrs);
	    *pkrs = '\0';
	    SKIP_BLANKS;
	    /*
	    ** Enter this in the Protocols database
	    */
	    isObserved = ((mode[0] == 'o' || mode[0] == 'O') ? True : False);
	    kts_addVirtualClient(gv, executable, krs, isObserved);
	 }

	 if ('}' == *pbuf) {
	    pbuf++;
	    SKIP_BLANKS;
	 }
      }
   }
   free(buf);
}

static void
kts_clearProtocolFiles(GV gv)
{
   DataBase db = gv -> connections;

   DB_Rewind(db);
   while (! DB_EndOfDataBase(db)) {
      Kts_Connection c = (Kts_Connection) DB_NextEntry(db);
      if (True == c -> isVirtual)
	 DB_RemoveEntry(db, c);
   }
}

/***************************************************************************
 * Public
 ***************************************************************************/

void
kts_readProtocolFiles(GV gv)
{
   char *files[] = { "protocols" };
   int filesCount = 1;
   DataBase db = gv -> connections;

   /*
   ** Remove existing protocols (virtual clients)...
   */
   kts_clearProtocolFiles(gv);

   /*
   ** ... and rebuild our table from scratch
   */
   while (filesCount > 0) {
      char *file = files[--filesCount];
      FILE *f;
      struct stat st;

      if (0 != stat(file, & st)) {
	 fprintf(stderr,
		 "*** ktserver warning : couldn't find protocol file '%s'\n", file);
      }
      else {
	 f = fopen(file, "r");
	 kts_readProtocolFile(gv, f, st.st_size);
	 fclose(f);
      }
   }
}

void
kts_displayProtocols(GV gv)
{
   DataBase db = gv -> connections;

   printf("--------------------------------------------------\n");
   DB_Rewind(db);
   while (!DB_EndOfDataBase(db)) {
      Kts_Connection c = DB_NextEntry(db);
      if (True == c -> isVirtual) {
	 printf("%s\n", c -> virtual.executable);
	 printf("   %s '%s'\n",
		(True == c -> virtual.isObserved ? "observing" : "handling"),
		c -> virtual.krs);
      }
   }
   printf("--------------------------------------------------\n");
}

/*
** This struct is only here to be used in the next two functions.
** DB_LocateEntry's callback function can only be given one argument
** whereas two things are needed : krlName and whether this krl
** is observed or not.
*/
struct tmpStruct {
   char *krlName;
   Bool isObserved;
};

static Bool
findVirtualClient(Kts_Connection conn, struct tmpStruct *krlInfo)
{
   char *krlName = krlInfo -> krlName;
   Bool isObserved = krlInfo -> isObserved;

   /*
   ** The matching client *must* be an observer. If the message
   ** was a request, the client would have sent a LAUNCH packet
   ** to the server, and we would not be in that portion of code.
   */
   if (True == conn -> isVirtual) {
      if (kt_regexpMatch(conn -> virtual.krs, krlName) &&
	  isObserved == conn -> virtual.isObserved)
             return True;
      else
             return False;
   }
   else return False;
}


Bool
kts_checkVirtualClients(GV gv, char *krlName, Bool isObserved)
{
   Bool result = False;
   DataBase db = gv -> connections;
   Kts_Connection vclient;
   struct stat st;
   int pid;
   struct tmpStruct krlInfo;
   char *argv[2], tempName[128], rmTempName[128], var[128];
   sprintf(tempName, "/tmp/.ktfork%d", getpid());
   sprintf(rmTempName, "rm %s", tempName);

   /*
   ** Try to find a virtual client that would match this krl
   */
   krlInfo.krlName = krlName;
   krlInfo.isObserved = isObserved;
   vclient = DB_LocateEntry(db,
			    (void *) ((unsigned long) findVirtualClient),
			    (void *) ((unsigned long) & krlInfo));

   /*
   ** If we found one, fork/exec it. Otherwise, do nothing
   */
   if (NULL != vclient) {
      switch(pid = fork()) {
	 case -1 :
	    printf("*** couldn't fork '%s'\n", vclient -> virtual.executable);
	    break;
	 case 0 :    /* child */
	    signal(SIGHUP, SIG_IGN);   /* survive if Dad dies */
	    argv[0] = vclient -> virtual.executable;
	    argv[1] = NULL;
	    /*
	    ** Set the environment variable to tell this client
	    ** it was launched automatically for a certain operation
	    */
	    sprintf(var, "%s=%s", VARIABLE_KTSERV_OPERATION, krlName);
	    putenv(var);
	    DP(printf("launching '%s %s'\n", var, vclient->virtual.executable));
	    execvp(vclient -> virtual.executable, argv);
	    system(rmTempName);   /* should not be executed */
	    break;
	 default :
	    sleep(2);
	    if (stat(tempName, & st) != 0) {
	       DP(printf("%s successfully forked\n",
			 vclient -> virtual.executable));
	       result = True;
	    }
	    else {
	       fprintf(stderr,"*** exec failed for '%s'\n",
		       vclient -> virtual.executable);
	       system(rmTempName);
	    }
      }
   }
   else {   /* no client found */
      
   }

   return result;
   
}

#ifdef A
Bool
kts_checkProtocolsAndLaunch(GV gv, char *krlName)
{
   Bool result = False;
   DataBase db = gv -> protocols;
   Kts_Protocol prot;
   struct stat st;
   int pid;
   char *argv[2], tempName[128], rmTempName[128], var[128];
   sprintf(tempName, "/tmp/.ktfork%d", getpid());
   sprintf(rmTempName, "rm %s", tempName);

   prot = DB_LocateEntry(db, (void *) findProtocol, krlName);
   if (NULL != prot) {
      switch(pid = fork()) {
	 case -1 :
	    printf("*** couldn't fork '%s'\n", prot -> executable);
	    break;
	 case 0 :    /* child */
	    signal(SIGHUP, SIG_IGN);   /* survive if Dad dies */
	    argv[0] = prot -> executable;
	    argv[1] = NULL;
	    /*
	    ** Set the environment variable to tell this client
	    ** it was launched automatically for a certain operation
	    */
	    sprintf(var, "%s=%s", VARIABLE_KTSERV_OPERATION, krlName);
	    putenv(var);
	    DP(printf("launching '%s %s'\n", var, prot->executable));
	    execvp(prot -> executable, argv);
	    system(rmTempName);   /* should not be executed */
	    break;
	 default :
	    sleep(2);
	    if (stat(tempName, & st) != 0) {
	       DP(printf("%s successfully forked\n", prot -> executable));
	       result = True;
	    }
	    else {
	       fprintf(stderr,"*** exec failed for '%s'\n",prot -> executable);
	       system(rmTempName);
	    }
      }
   }
   else {   /* no client found */
      
   }

   return result;
}
#endif
