/* Copyright (C) 1993 Groupe BULL. See file COPYRIGHT for details */
/*
 *
 * $Id: datalist.c,v 1.1 1994/08/19 14:30:14 beust Exp beust $
 */

#include <stdio.h>
#include "database.h"

/*
#define DEBUG_WANT_COUNT 0
*/

/***********************************************************************
 * List implementation
 **********************************************************************/

struct _List
{
    struct _Node *Head;
    struct _Node *Tail;
    struct _Node *TailPred;
    int node;
};

struct _Node
{
    struct _Node *Succ;
    struct _Node *Pred;
};

static void
db_NewList(struct _List *list)
{
    list->Head = (struct _Node *)&list->Tail;
    list->Tail = NULL;
    list->TailPred = (struct _Node *)list;
}

static void
db_AddHead(struct _List *list,struct _Node *node)
{
    node->Succ=list->Head;
    node->Pred=(struct _Node *)list;
    list->Head->Pred=node;
    list->Head=node;
#ifdef DEBUG_WANT_COUNT
    list -> count++;
#endif /* COUNT */
}

static void
db_AddTail(struct _List *list,struct _Node *node)
{
    node->Succ=(struct _Node *)&list->Tail;
    node->Pred=list->TailPred;
    list->TailPred->Succ=node;
    list->TailPred=node;
#ifdef DEBUG_WANT_COUNT
    list -> count++;
#endif /* COUNT */
}

static struct _Node *
db_RemHead(struct _List *list)
{
    struct _Node *node=NULL;

    if (list->Head->Succ)
    {	
    	node=list->Head;
    
	list->Head->Succ->Pred=(struct _Node *)list;
	list->Head=list->Head->Succ;
#ifdef DEBUG_WANT_COUNT
	list -> count--;
#endif /* COUNT */
    }
    
    return node;
}

static struct _Node *
db_RemTail(struct _List *list)
{
    struct _Node *node=NULL;
    
    if (list->TailPred->Pred)
    {
    	node=list->TailPred;
    
    	list->TailPred->Pred->Succ=list->TailPred;
    	list->TailPred=list->TailPred->Pred;
#ifdef DEBUG_WANT_COUNT
	list -> count--;
#endif /* COUNT */
    }
    
    return node;
}

static void
db_Remove(struct _Node *node)
{
    node->Succ->Pred=node->Pred;
    node->Pred->Succ=node->Succ;
}

/***********************************************************************
 * Datalist implementation
 **********************************************************************/

/***********************************************************************
 * Private
 **********************************************************************/

struct DataNode {
   struct _Node n;
   Generic data;
};

typedef struct _DataBase {
   void *freeFunction;  /* function to free a generic entry */
   struct _List *list;
   struct _Node *current;
} DB;

static Bool
db_findEntry(Generic aa, Generic b)
{
/*
   struct DataNode *aa = (struct DataNode *) a;
   printf("   db_findEntry comparing %x and %x\n", aa->data, b);
*/
   return (aa == b ? True : False);
}

static struct DataNode *
db_nextEntry(DataBase db)
{
   struct DataNode *result;

   struct DataNode *current = (struct DataNode *) db -> current;
   if (current -> n.Succ == db -> list -> Head)
      result = NULL;
   else {
      result = current;
      current = (struct DataNode *) current -> n.Succ;
      db -> current = (struct _Node *) current;
   }

   return result;
}

static struct DataNode *
db_locateEntry(DataBase db,
	       Bool (* findFunction)(Generic a, void *userData),void *userData)
{
   struct DataNode *result = NULL;
   struct DataNode *dn = NULL;
   int running = 1;

   DB_Rewind(db);
   while (1 == running) {
      if (DB_EndOfDataBase(db))
	 running = 0;
      else {
	 dn = db_nextEntry(db);
	 if (True == findFunction(dn -> data, userData)) {
	    running = 0;
	    result = dn;
	 }
      }
   }

   return result;
}

static void
db_dumpDataBase(DataBase db)
{
   struct DataNode *dn = (struct DataNode *) db -> list -> Head;
   while (NULL != dn -> n.Succ) {
      printf("%x (data=%x) ", dn, dn -> data);
      dn = (struct DataNode *) dn -> n.Succ;
   }
   printf("\n");
}

/***********************************************************************
 * Public
 **********************************************************************/

DataBase
DB_NewDataBase(int size, void *freeFunction)
{
   DataBase result;

   NEW(result, DB);
   NEW(result -> list, struct _List);
   db_NewList(result -> list);
   result -> current = NULL;
   result -> freeFunction = freeFunction;

   return result;
}

Bool
DB_EndOfDataBase(DataBase db)
{
   return (NULL == db -> current -> Succ);
}

void
DB_ClearDataBase(DataBase db)
{
   printf("clearing\n");
}

void
DB_DestroyDataBase(DataBase db)
{
   struct _Node *n = db -> list -> Head;
   void (*f)();
   f = db -> freeFunction;
   while (0 != n -> Succ) {
      struct DataNode *dn = (struct DataNode *) n;
      if (NULL != f) (*f)(dn -> data);
      n = n -> Succ;
      free(dn);
   }

   free(db -> list);
   free(db);
}

Bool
DB_AddEntry(DataBase db, Generic entry)
{
   struct DataNode *dn;

   NEW(dn, struct DataNode);
   dn -> data = entry;
   db_AddTail(db -> list, (struct _Node *) dn);
}

Bool
DB_RemoveEntry(DataBase db, Generic entry)
{
   struct DataNode *dn;
   int found = 0;
   Bool result;

   dn = db_locateEntry(db, db_findEntry, entry);
   if (NULL != dn) {
      void (*f)();
      f = db -> freeFunction;
      db_Remove((struct _Node *) dn);
      if (NULL != f) (*f)(dn -> data);
      free(dn);
      result = True;
   }
   else {
      printf("removeentry, not found\n");
      result = False;
   }
   return result;
/*
   DB_Rewind(db);
   while (! DB_EndOfDataBase(db) && ! found) {
      dn = db_nextEntry(db);
      if (entry == dn -> data)
	 found = 1;
   }

   if (found) {
      db_Remove((struct _Node *) dn);
      printf("removed an entry\n");
      result = True;
   }
   else {
      printf("removeentry, entry not found\n");
      result = False;
   }

   return result;
*/

}

Bool
DB_RemoveNthEntry(DataBase db, int n)
{
   Generic gen;

   DB_Rewind(db);
/*
   if (n >= db -> number) {
      db_Err( "DB_RemoveNthEntry : n is too big\n");
      return False;
   }
*/

   gen = DB_NextEntry(db);
   while (n--) gen = DB_NextEntry(db);
   return DB_RemoveEntry(db, gen);
}

Bool
DB_ReplaceEntry(DataBase db, Generic old, Generic new)
{
   Bool result;
   struct DataNode *dn;

   dn = db_locateEntry(db, db_findEntry, old);
   if (NULL != dn) {
      void (*f)() = db -> freeFunction;
      dn -> data = new;
      if (NULL != f) (*f)(old);
      result = True;
   }
   else {
      printf("error replaceentry\n");
      result = False;
   }
}

void
DB_Rewind(DataBase db)
{
   db -> current = db -> list -> Head;
}

Generic
DB_NextEntry(DataBase db)
{
   struct DataNode *dn = db_nextEntry(db);
   return dn -> data;
}

Generic
DB_NthEntry(DataBase db, int n)
{
   Generic result;
   DB_Rewind(db);
   while (n-- >= 0) result = DB_NextEntry(db);

   return result;
}

int
DB_Count(DataBase db)
{
#ifdef DEBUG_WANT_COUNT
   return db -> count;
#else
   int result = 0;
   struct _Node *n = db -> list -> Head;
   while (0 != n -> Succ) {
      result++;
      n = n -> Succ;
   }

   return result;
#endif /* COUNT */
}

void
DB_Sort(DataBase db, int (* compareFunction)(Generic a, Generic b))
{
   Generic *bytes;
   char **pbytes;
   struct _Node *n = db -> list -> Head;
   int dbCount = DB_Count(db);

/*
** In order to be able to use qsort, I must gather all the entries in a
** contiguous area of memory. Then I call qsort, clear the database and
** add the sorted entries one by one
*/

/* Allocate a memory chunk large enough to contain the database... */
   bytes = (Generic *) malloc(dbCount * sizeof(Generic));
   pbytes = (char **) bytes;

/* Fill it */
   while (0 != n -> Succ) {
      *pbytes = ((struct DataNode *) n) -> data;
      pbytes++;
      n = n -> Succ;
   }

/* Sort it */
   qsort(& bytes[0], dbCount, sizeof(Generic), compareFunction);

/* Make it the new list */

   free(bytes);
}

void
DB_DisplayDataBase(DataBase db, void (* displayFunction)(Generic a))
{
   printf("display not implemented\n");
}

Generic
DB_LocateEntry(DataBase db,
	       Bool (* findFunction)(Generic a, void *userData),void *userData)
{
   struct DataNode *dn;
   Generic result;

   dn = db_locateEntry(db, findFunction, userData);
   if (NULL != dn)
      result = dn -> data;
   else
      result = NULL;

   return result;
}

