/*
 * Copyright (C) 1994 Communications and Systems Specialists, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a 
 * copy of this software and associated documentation files (the 
 * "Software"), to deal in the Software without restriction, including 
 * without limitation the rights to use, copy, modify, merge, publish, 
 * distribute, sublicense, and/or sell copies of the Software, and to 
 * permit  persons to whom the Software is furnished to do so, subject to 
 * the following conditions:
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. 
 * IN NO EVENT SHALL COMMUNICATIONS AND SYSTEMS SPECIALISTS, INC. BE 
 * LIABLE FOR ANY CLAIM,  DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION 
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Communications and 
 * Systems Specialists, Inc. shall not be used in advertising or otherwise
 * to promote the sale, use or other dealings in this Software without 
 * prior written authorization from Communications and Systems 
 * Specialists, Inc.
 */

#include <ctype.h>
#include <stdio.h>
#include <Xm/Xm.h>
#include <Xm/AtomMgr.h>
#include <Xm/Form.h>
#include <Xm/LabelG.h>
#include <Xm/MwmUtil.h>
#include <Xm/PushB.h>
#include <Xm/SeparatoG.h>

#include "XmsgXm.h"

#define XmsgXmLOCK_PROCESS
#define XmsgXmUNLOCK_PROCESS

typedef struct _XmsgXmInternalLookupStruct
{
   XrmQuark               name_quark;
   XrmQuark               class_quark;
   _XmsgConst _XmsgString label_str;
   _XmsgConst _XmsgString lookup_str;
}
XmsgXmInternalLookupStruct;

typedef struct _XmsgXmVirtualStruct
{
   Widget       top_modal_cascade_widget;
   unsigned int wait_count;
}
XmsgXmVirtualStruct;

typedef struct _XmsgXmDialogStruct
{
   Widget             dialog_widget;
   Widget             shell_widget;
   Widget             message_widget;
   Widget             option_names_widget;
   Widget             option_values_widget;
   Widget             message_container_widget;
   Widget             icon_widget;
   Widget             okay_widget;
   Widget             separator_widget;
   Pixmap             pixmap;
   Boolean            image_cache;
   Boolean            wait;
   Boolean*           blocking_ptr;
   XmsgMessageStruct* message_ptr;
}
XmsgXmDialogStruct;



typedef struct _XmsgXmPackageData
{
   XmsgXmInternalLookupStruct* lookup_ptr;
   Cardinal                    lookup_total;
   XContext                    xcontext;
}
XmsgXmPackageData;

static XmsgXmPackageData package_data;


static void XmsgXmActivateCB(
#if NeedFunctionPrototypes
   Widget    w,
   XtPointer cd,
   XtPointer cbd  
#endif
);

static void XmsgXmDelayPopupTO(
#if NeedFunctionPrototypes
    XtPointer     cd,
    XtIntervalId* interval_id_ptr  
#endif
);

static void XmsgXmDestroyVirtualCB(
#if NeedFunctionPrototypes
   Widget    w,
   XtPointer cd,
   XtPointer cbd  
#endif
);

static XmsgXmDialogStruct* XmsgXmGetDialog(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr  
#endif
);

static Widget XmsgXmGetParent(
#if NeedFunctionPrototypes
   XmsgMessageStruct*  message_ptr  
#endif
);

static void XmsgXmPopup(
#if NeedFunctionPrototypes
   XmsgMessageStruct*  message_ptr  
#endif
);

static void XmsgXmPositionDialog(
#if NeedFunctionPrototypes
   XmsgXmDialogStruct* dialog_ptr,
   XmsgMessageStruct*  message_ptr  
#endif
);

static void XmsgXmUpdateIcon(
#if NeedFunctionPrototypes
   XmsgXmDialogStruct* dialog_ptr,
   XmsgMessageStruct*  message_ptr  
#endif
);

static void XmsgXmUpdateMessage(
#if NeedFunctionPrototypes
   XmsgXmDialogStruct* dialog_ptr,
   XmsgMessageStruct*  message_ptr   
#endif
);

static void XmsgXmUpdateModality(
#if NeedFunctionPrototypes
   XmsgXmDialogStruct* dialog_ptr,
   XmsgMessageStruct*  message_ptr   
#endif
);

static void XmsgXmUpdateOptionals(
#if NeedFunctionPrototypes
   XmsgXmDialogStruct* dialog_ptr,
   XmsgMessageStruct*  message_ptr   
#endif
);

static void XmsgXmUpdateTitle(
#if NeedFunctionPrototypes
   XmsgXmDialogStruct* dialog_ptr,
   XmsgMessageStruct*  message_ptr   
#endif
);



/*-------------------------------
-- Private function definitions
-------------------------------*/
#if NeedFunctionPrototypes
static void XmsgXmActivateCB(
   Widget    w,
   XtPointer cd,
   XtPointer cbd )
#else
static void XmsgXmActivateCB( w, cd, cbd )
   Widget    w;
   XtPointer cd;
   XtPointer cbd;
#endif
{
   XmsgXmDialogStruct* dialog_ptr = (XmsgXmDialogStruct*) cd;
   Screen* screen_ptr;
   Widget virtual_widget;
   int    status;

   /*-------------------------------------------------------------------
   -- Unmap the dialog before destroying it.  It must be popped down
   -- first because the phase 2 destroy does not happen until the end
   -- of XtDispatchEvent().  If a nested XtDispatchEvent() is called
   -- before, the dialog could still be on the screen which will cause
   -- bad things (e.g., core dumps)
   -------------------------------------------------------------------*/
   XUnmapWindow( XtDisplay( dialog_ptr->shell_widget ), 
                 XtWindow ( dialog_ptr->shell_widget ) );
   XFlush( XtDisplay( dialog_ptr->shell_widget ) );

   XtVaGetValues( dialog_ptr->shell_widget, XmNscreen, &screen_ptr, NULL );

   if (dialog_ptr->image_cache)
   {
      XmDestroyPixmap( screen_ptr, dialog_ptr->pixmap );
   }
   virtual_widget = XtParent( dialog_ptr->shell_widget );

   if (dialog_ptr->wait)
   {
      XmsgXmVirtualStruct* virtual_ptr;

      if (dialog_ptr->blocking_ptr)
      {
         *dialog_ptr->blocking_ptr = False;
      }
      status = XFindContext( XtDisplay( w ),
                             (XID) XtParent( dialog_ptr->shell_widget ),
                             package_data.xcontext, (XPointer*) &virtual_ptr );
      if (status == 0)
      {
         virtual_ptr->wait_count -= 1;
         if (virtual_ptr->wait_count == 0)
         {
            XtRemoveGrab( virtual_widget );
         }
         dialog_ptr->wait = False;
      }
      else
      {
         /*----------------------------------------------------------------
         -- This error is not simple printed to stderr because it *could*
         -- cause an infinitly recursive error if it was dispatched
         -- through xmsg
         ----------------------------------------------------------------*/
         fprintf( stderr, "Xmsg internal error in module %s at line %u\n",
                  __FILE__, (unsigned int) __LINE__ );

         fprintf( stderr, "Missing entry in X context manager, "
                          "pop-up wait disabled\n" );
      }
   }

   XtDestroyWidget( dialog_ptr->shell_widget );

   /*----------------------------------------------------------------------
   -- XmsgFreeMessage will only perform the free operation is the pointer
   -- is not null and is marked as freeable
   ----------------------------------------------------------------------*/
   XmsgFreeMessage( dialog_ptr->message_ptr );
   XtFree( (char*) dialog_ptr );
}


#if NeedFunctionPrototypes
static void XmsgXmDelayPopupTO(
    XtPointer     cd,
    XtIntervalId* interval_id_ptr )
#else
static void XmsgXmDelayPopupTO( cd, interval_id_ptr )
    XtPointer     cd;
    XtIntervalId* interval_id_ptr;
#endif
{
   XmsgMessageStruct* message_ptr;

   message_ptr = (XmsgMessageStruct*) cd;

   XmsgXmPopup( message_ptr );
}


#if NeedFunctionPrototypes
static void XmsgXmDestroyVirtualCB(
   Widget    w,
   XtPointer cd,
   XtPointer cbd )
#else
static void XmsgXmDestroyVirtualCB( w, cd, cbd )
   Widget    w;
   XtPointer cd;
   XtPointer cbd;
#endif
{
   /*---------------------------------------------------------------
   -- Remove the virtual struct reference from the context manager
   -- and free the actual data structure
   ---------------------------------------------------------------*/
   XDeleteContext( XtDisplay( w ), (XID) w, package_data.xcontext );
   XtFree( (char*) cd );
}


#if NeedFunctionPrototypes
static XmsgXmDialogStruct* XmsgXmGetDialog(
   XmsgMessageStruct* message_ptr )
#else
static XmsgXmDialogStruct* XmsgXmGetDialog( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   Widget parent_widget;
   int width;
   XmsgXmDialogStruct* dialog_ptr;
   XmString cs;
   _XmsgConst _XmsgString database_str;

   parent_widget = XmsgXmGetParent( message_ptr );

   dialog_ptr = XtNew( XmsgXmDialogStruct );

   memset( dialog_ptr, 0, sizeof( XmsgXmDialogStruct ) );
   dialog_ptr->blocking_ptr = NULL;

   /*------------------------------------------------------------
   -- This is needed so that the memory can be cleaned up later
   -- if it is located on the heap
   ------------------------------------------------------------*/
   dialog_ptr->message_ptr = message_ptr;

   dialog_ptr->shell_widget = XtVaCreatePopupShell(
      "xmsgShell", topLevelShellWidgetClass, parent_widget,
      XmNallowShellResize, True,
      XmNmwmDecorations,   MWM_DECOR_BORDER | MWM_DECOR_TITLE,
      NULL );

   dialog_ptr->dialog_widget = XtVaCreateManagedWidget( 
      "xmsg", xmFormWidgetClass, dialog_ptr->shell_widget,
      XmNresizePolicy, XmRESIZE_ANY,
      XmNnoResize,     True,
      NULL );

   dialog_ptr->message_container_widget = XtVaCreateManagedWidget(
      "xmsgMessageContainer", xmFormWidgetClass, dialog_ptr->dialog_widget,
      XmNtopAttachment,    XmATTACH_FORM,
      XmNtopOffset,        10,
      XmNleftAttachment,   XmATTACH_FORM,
      XmNleftOffset,       10,
      XmNrightAttachment,  XmATTACH_FORM,
      XmNrightOffset,      10,
      NULL );
      
   dialog_ptr->icon_widget = XtVaCreateManagedWidget(
      "xmsgIcon", xmLabelGadgetClass, dialog_ptr->message_container_widget,
      XmNtopAttachment,    XmATTACH_FORM,
      XmNtopOffset,        10,
      XmNleftAttachment,   XmATTACH_FORM,
      XmNleftOffset,       10,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNbottomOffset,     10,
      NULL );

   dialog_ptr->message_widget = XtVaCreateManagedWidget(
      "xmsgMessage", xmLabelGadgetClass, dialog_ptr->message_container_widget,
      XmNtopAttachment,    XmATTACH_FORM,
      XmNtopOffset,        10,
      XmNleftAttachment,   XmATTACH_WIDGET,
      XmNleftWidget,       dialog_ptr->icon_widget,
      XmNleftOffset,       20,
      XmNrightAttachment,  XmATTACH_FORM,
      XmNrightOffset,      10,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNbottomOffset,     10,
      NULL );

   dialog_ptr->option_names_widget = XtVaCreateManagedWidget(
      "xmsgOptionNames",  xmLabelGadgetClass, dialog_ptr->dialog_widget,
      XmNtopAttachment,    XmATTACH_WIDGET,
      XmNtopWidget,        dialog_ptr->message_container_widget,
      XmNtopOffset,        0,
      XmNrightAttachment,  XmATTACH_POSITION,
      XmNrightPosition,    50,
      XmNrightOffset,      5,
      XmNalignment,        XmALIGNMENT_END,
      NULL );

   dialog_ptr->option_values_widget = XtVaCreateManagedWidget(
      "xmsgOptionValues", xmLabelGadgetClass, dialog_ptr->dialog_widget,
      XmNtopAttachment,    XmATTACH_WIDGET,
      XmNtopWidget,        dialog_ptr->message_container_widget,
      XmNtopOffset,        0,
      XmNleftAttachment,   XmATTACH_POSITION,
      XmNleftPosition,     50,
      XmNleftOffset,       5,
      XmNalignment,        XmALIGNMENT_BEGINNING,
      NULL );

   dialog_ptr->separator_widget = XtVaCreateManagedWidget(
      "xmsgSeparator", xmSeparatorGadgetClass, dialog_ptr->dialog_widget,
      XmNtopAttachment,    XmATTACH_WIDGET,
      XmNtopWidget,        dialog_ptr->option_names_widget,
      XmNtopOffset,        10,
      XmNleftAttachment,   XmATTACH_FORM,
      XmNleftOffset,       0,
      XmNrightAttachment,  XmATTACH_FORM,
      XmNrightOffset,      0,
      NULL );

   database_str = XmsgGetMsgDatabaseString( 
      message_ptr,
      XrmPermStringToQuark( "popupOkLabel" ),
      XrmPermStringToQuark( "PopupOkLabel" ) );

   if (database_str)
   {
      cs = XmStringCreateLtoR( (char*) database_str, "" );
   }
   else
   {
      cs = XmStringCreateLtoR( "OK", "" );
   }
   dialog_ptr->okay_widget = XtVaCreateManagedWidget(
      "xmsgOk", xmPushButtonWidgetClass, dialog_ptr->dialog_widget,
      XmNlabelString,      cs,
      XmNtopAttachment,    XmATTACH_WIDGET,
      XmNtopWidget,        dialog_ptr->separator_widget,
      XmNtopOffset,        10,
      XmNleftAttachment,   XmATTACH_POSITION,
      XmNleftPosition,     50,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNbottomOffset,     10,
      NULL );
   XmStringFree( cs );

   XtAddCallback( dialog_ptr->okay_widget, XmNactivateCallback, 
                  XmsgXmActivateCB, dialog_ptr );
                  

   XtVaSetValues( 
      dialog_ptr->dialog_widget, 
      XmNdefaultButton, dialog_ptr->okay_widget, 
      NULL );

   {
      XtGeometryResult result;
      XtWidgetGeometry geometry;

      XtQueryGeometry( dialog_ptr->okay_widget, NULL, &geometry );

      XtVaSetValues( 
         dialog_ptr->okay_widget, 
         XmNleftOffset, -((int)geometry.width/2) - (int)geometry.border_width, 
         NULL );
   }

   XtRealizeWidget( dialog_ptr->shell_widget );

   return dialog_ptr;
}


#if NeedFunctionPrototypes
static Widget XmsgXmGetParent(
   XmsgMessageStruct*  message_ptr )
#else
static Widget XmsgXmGetParent( message_ptr )
   XmsgMessageStruct*  message_ptr;
#endif
{
   Widget toplevel_widget;
   Widget virtual_widget;
   Widget tmp_widget;

   /*--------------------------------------------------------------
   -- Find the top level shell associated with the message widget
   --------------------------------------------------------------*/
   toplevel_widget = XmsgGetMsgWidget( message_ptr );
   tmp_widget = XtParent( toplevel_widget );
   while (tmp_widget)
   {
      toplevel_widget = tmp_widget;
      tmp_widget = XtParent( toplevel_widget );
   }

   virtual_widget = XtNameToWidget( toplevel_widget, "xmsgVirtual" );
   if (!virtual_widget)
   {
      XmsgXmVirtualStruct* virtual_ptr;

      virtual_widget = XtVaCreatePopupShell(
         "xmsgVirtual", topLevelShellWidgetClass, toplevel_widget,
         XmNallowShellResize, True,
         XmNmwmDecorations,   MWM_DECOR_BORDER | MWM_DECOR_TITLE,
         NULL );

      virtual_ptr = XtNew( XmsgXmVirtualStruct );
      virtual_ptr->wait_count = 0;
      virtual_ptr->top_modal_cascade_widget = virtual_widget;

      XtAddCallback( virtual_widget, XmNdestroyCallback, 
                     XmsgXmDestroyVirtualCB, (XtPointer) virtual_ptr );

      /*--------------------------------------------------------
      -- The only error returned by XSaveContext is no memory!
      -- If that is the case, it will be reported soon enough
      -- somewhere else
      --------------------------------------------------------*/
      XSaveContext( XtDisplay( virtual_widget ), (XID) virtual_widget,
                    package_data.xcontext, (XPointer) virtual_ptr );
   }
   return virtual_widget;
}


#if NeedFunctionPrototypes
static void XmsgXmPopup(
   XmsgMessageStruct*  message_ptr )
#else
static void XmsgXmPopup( message_ptr )
   XmsgMessageStruct*  message_ptr;
#endif
{
   XmsgXmDialogStruct* dialog_ptr;
   XtGrabKind          grab_kind;
   
   dialog_ptr = XmsgXmGetDialog( message_ptr );

   XmsgXmUpdateIcon     ( dialog_ptr, message_ptr );
   XmsgXmUpdateMessage  ( dialog_ptr, message_ptr );
   XmsgXmUpdateOptionals( dialog_ptr, message_ptr );
   XmsgXmUpdateTitle    ( dialog_ptr, message_ptr );
   XmsgXmPositionDialog ( dialog_ptr, message_ptr );
   XmsgXmUpdateModality ( dialog_ptr, message_ptr );

   if (dialog_ptr->wait)
   {
      XtAppContext app_context;
      XEvent       event;
      XmsgXmVirtualStruct* virtual_ptr;
      Boolean      blocking;

      XFindContext( XtDisplay( dialog_ptr->shell_widget ),
                    (XID) XtParent( dialog_ptr->shell_widget ),
                    package_data.xcontext, (XPointer*) &virtual_ptr );
      
      if (virtual_ptr->wait_count == 0)
      {
         XtAddGrab( virtual_ptr->top_modal_cascade_widget, True, False );
      }
      virtual_ptr->wait_count += 1;
      XtPopup( dialog_ptr->shell_widget, XtGrabNonexclusive );

      app_context = XtWidgetToApplicationContext( dialog_ptr->dialog_widget );

      blocking = True;
      dialog_ptr->blocking_ptr = &blocking;
      while (blocking)
      {
         XtAppNextEvent( app_context, &event );
         XtDispatchEvent( &event );
      }
   }
   else
   {
      XtPopup( dialog_ptr->shell_widget, XtGrabNonexclusive );
   }
}


#if NeedFunctionPrototypes
static void XmsgXmPositionDialog(
    XmsgXmDialogStruct* dialog_ptr,
    XmsgMessageStruct*  message_ptr )
#else
static void XmsgXmPositionDialog( dialog_ptr, message_ptr )
    XmsgXmDialogStruct* dialog_ptr;
    XmsgMessageStruct*  message_ptr;
#endif
{
   Widget   requestor_widget;
   Widget   requestor_shell_widget;
   Atom     WM_TRANSIENT_FOR;
   Atom     WINDOW;
   Window   requestor_shell_window;
   Display* display_ptr;
   XmsgPosition position;
   Dimension cover_height;
   Dimension cover_width;
   Dimension dialog_height;
   Dimension dialog_width;
   Dimension root_height;
   Dimension root_width;
   Position  cover_x;
   Position  cover_y;
   Position  dialog_x;
   Position  dialog_y;
   Screen*   screen_ptr;
   WidgetList widget_list;

   requestor_widget =  XmsgGetMsgWidget( message_ptr );

   requestor_shell_widget = requestor_widget;
   while (!XtIsWMShell( requestor_shell_widget ))
   {
      requestor_shell_widget = XtParent( requestor_shell_widget );
   }

   display_ptr = XtDisplay( requestor_widget );

   WM_TRANSIENT_FOR = XmInternAtom( display_ptr, "WM_TRANSIENT_FOR", False );

   if (XtIsRealized( requestor_shell_widget ))
   {
      WINDOW = XmInternAtom( display_ptr, "WINDOW", False );

      requestor_shell_window = XtWindow( requestor_shell_widget );

      XChangeProperty( display_ptr, XtWindow( dialog_ptr->shell_widget ),
                       WM_TRANSIENT_FOR, WINDOW, 32, PropModeReplace,
                       (unsigned char*) &requestor_shell_window, 1 );
   }
   else
   {
      XDeleteProperty( display_ptr, XtWindow( dialog_ptr->shell_widget ),
                       WM_TRANSIENT_FOR );
   }

   position = XmsgGetMsgPosition( message_ptr );

   if (position == XmsgPositionDATABASE)
   {
      _XmsgConst _XmsgString database_str;
      String            str;
      int               i;

      database_str = XmsgGetMsgDatabaseString( 
         message_ptr,
         XrmPermStringToQuark( "popupPosition" ),
         XrmPermStringToQuark( "PopupPosition" ) );

      if (database_str)
      {
          str = XtNewString( database_str );

          XmsgXmLOCK_PROCESS

          for (i = 0; str[ i ] != 0; i++)
          {
             str[ i ] = (char) toupper( (int) str[ i ] );
          }
          XmsgXmUNLOCK_PROCESS

          if (strcmp( str, "WIDGET" ) == 0)
          {
             position = XmsgPositionWIDGET;
          }
          else if (strcmp( str, "SHELL" ) == 0)
          {
             position = XmsgPositionSHELL;
          }
          else if (strcmp( str, "SCREEN" ) == 0)
          {
             position = XmsgPositionSCREEN;
          }
          else
          {
             position = XmsgPositionDEFAULT;
          }
          XtFree( str );
      }
      else
      {
          position = XmsgPositionDEFAULT;
      }
   }

   XtVaGetValues( dialog_ptr->shell_widget,
                  XmNheight, &dialog_height,
                  XmNwidth,  &dialog_width,
                  XmNscreen, &screen_ptr,
                  NULL );

   root_height = HeightOfScreen( screen_ptr );
   root_width  = WidthOfScreen ( screen_ptr );

   switch( position )
   {
      case XmsgPositionWIDGET:
      {
         XtVaGetValues( requestor_widget,
                        XmNheight, &cover_height,
                        XmNwidth,  &cover_width,
                        NULL );
         XtTranslateCoords( requestor_widget, 0, 0, &cover_x, &cover_y );
      }
      break;

      case XmsgPositionSHELL:
      default:
      {
         XtVaGetValues( requestor_shell_widget,
                        XmNheight, &cover_height,
                        XmNwidth,  &cover_width,
                        NULL );
         XtTranslateCoords( requestor_shell_widget, 0, 0, &cover_x, &cover_y );
      }
      break;

      case XmsgPositionSCREEN:
      {
         cover_x      = 0;
         cover_y      = 0;
         cover_height = root_height;
         cover_width  = root_width;
      }
      break;
   }

   /*-------------------------------------------------------
   -- Calculate the desired position for the message popup
   -------------------------------------------------------*/
   dialog_x = cover_x + (Position) (cover_width  - dialog_width ) / 2;
   dialog_y = cover_y + (Position) (cover_height - dialog_height) / 2;

   /*------------------------------------------------------------------------
   -- If the message will be off of the screen, force it back on the screen
   ------------------------------------------------------------------------*/
   if ((Position)(dialog_x + dialog_width) > (Position)root_width)
   {
      dialog_x = root_width - dialog_width;
   }
   if ((Position)(dialog_y + dialog_height) > (Position)root_height)
   {
      dialog_y = root_height - dialog_height;
   }
   if (dialog_x < (Position)0)
   {
      dialog_x = 0;
   }
   if (dialog_y < (Position)0)
   {
      dialog_y = 0;
   }

   /*----------------------------------------------------
   -- Move the message popup to the calculated position
   ----------------------------------------------------*/
   XtVaSetValues( dialog_ptr->shell_widget, 
                  XmNx, dialog_x, 
                  XmNy, dialog_y,
                  NULL );
}


#if NeedFunctionPrototypes
static void XmsgXmUpdateIcon(
   XmsgXmDialogStruct* dialog_ptr,
   XmsgMessageStruct*  message_ptr )
#else
static void XmsgXmUpdateIcon( dialog_ptr, message_ptr )
   XmsgXmDialogStruct* dialog_ptr;
   XmsgMessageStruct*  message_ptr;
#endif
{
   Screen* screen_ptr;
   Pixel   foreground_pixel;
   Pixel   background_pixel;
   char    buffer[ 30 ];
   XmString cs;

   dialog_ptr->image_cache = False;

   XtVaGetValues( dialog_ptr->dialog_widget, 
                  XmNscreen,     &screen_ptr, 
                  XmNforeground, &foreground_pixel,
                  XmNbackground, &background_pixel,
                  NULL );

   sprintf( buffer, "Xmsg%s", XmsgGetMsgCategory( message_ptr ) );

   dialog_ptr->pixmap = XmGetPixmap( screen_ptr, buffer, foreground_pixel, 
                                     background_pixel );

   if (dialog_ptr->pixmap != XmUNSPECIFIED_PIXMAP)
   {
      dialog_ptr->image_cache = True;
      XtVaSetValues( dialog_ptr->icon_widget, 
                     XmNlabelPixmap, dialog_ptr->pixmap,
                     XmNlabelType,   XmPIXMAP,
                     NULL );
   }
   else
   {
      cs = XmStringCreateLocalized( buffer );
      XtVaSetValues( dialog_ptr->icon_widget, 
                     XmNlabelString, cs,
                     XmNlabelType,   XmSTRING,
                     NULL );
      XmStringFree( cs );
   }
}


#if NeedFunctionPrototypes
static void XmsgXmUpdateMessage(
   XmsgXmDialogStruct* dialog_ptr,
   XmsgMessageStruct*  message_ptr  )
#else
static void XmsgXmUpdateMessage( dialog_ptr, message_ptr )
   XmsgXmDialogStruct* dialog_ptr;
   XmsgMessageStruct*  message_ptr;
#endif
{
   XmString cs;
   _XmsgConst _XmsgString text_str;

   text_str = XmsgGetMsgText( message_ptr );
   cs = XmStringCreateLtoR( (String) text_str, "" );
   XtVaSetValues( 
      dialog_ptr->message_widget,
      XmNlabelString, cs,
      NULL );
   XmStringFree( cs );
}


#if NeedFunctionPrototypes
static void XmsgXmUpdateModality(
   XmsgXmDialogStruct* dialog_ptr,
   XmsgMessageStruct*  message_ptr  )
#else
static void XmsgXmUpdateModality( dialog_ptr, message_ptr )
   XmsgXmDialogStruct* dialog_ptr;
   XmsgMessageStruct*  message_ptr;
#endif
{
   _XmsgConst _XmsgString category_str;
   _XmsgConst _XmsgString database_str;

   category_str = XmsgGetMsgCategory( message_ptr );

   if (strcmp( category_str, XmsgCATEGORY_FATAL ) == 0)
   {
      dialog_ptr->wait  = True;
   }
   else
   {
      database_str = XmsgGetMsgDatabaseString( 
         message_ptr, 
         XrmPermStringToQuark( "popupModal" ),
         XrmPermStringToQuark( "PopupModal" ) );

      if (database_str)
      {
         dialog_ptr->wait = XmsgCvtStringToBoolean( database_str, True );
      }
   }
}


#if NeedFunctionPrototypes
static void XmsgXmUpdateOptionals(
   XmsgXmDialogStruct* dialog_ptr,
   XmsgMessageStruct*  message_ptr  )
#else
static void XmsgXmUpdateOptionals( dialog_ptr, message_ptr )
   XmsgXmDialogStruct* dialog_ptr;
   XmsgMessageStruct*  message_ptr;
#endif
{
   char  names_str[ 1000 ];
   char  values_str[ 1000 ];
   char* names_ptr  = names_str;
   char* values_ptr = values_str;
   _XmsgConst _XmsgString database_str;
   XmString cs;
   Display* display_ptr;
   int i;
   Boolean  converted_value;
   Boolean  do_lookup;
   XrmValue value_as_str;
   XrmValue value_as_boolean;
   String tmp_str;

   display_ptr = XtDisplay( dialog_ptr->dialog_widget );

   for (i = 0; i < package_data.lookup_total; i++)
   {
      database_str = XmsgGetMsgDatabaseString( 
         message_ptr, 
         package_data.lookup_ptr[ i ].name_quark,
         package_data.lookup_ptr[ i ].class_quark );

      do_lookup = XmsgCvtStringToBoolean( database_str, False );

      if (do_lookup)
      {
         strcpy( names_ptr, package_data.lookup_ptr[ i ].label_str );
         names_ptr += strlen( names_ptr );
         *names_ptr = '\n';
         names_ptr++;

         strcpy( values_ptr, package_data.lookup_ptr[ i ].lookup_str );
         values_ptr += strlen( values_ptr );
         *values_ptr = '\n';
         values_ptr++;
      }
   }

   if (names_ptr == names_str)
   {
      /* Maybe the options label should be unmanaged? */
   }
   else
   {
      names_ptr--;
      values_ptr--;
   }
   *names_ptr  = '\0';
   *values_ptr = '\0';

   tmp_str = XtNewString( values_str );
   XmsgTranslate( message_ptr, tmp_str, values_str );
   XtFree( tmp_str );

   cs = XmStringCreateLtoR( names_str, "" );
   XtVaSetValues( 
      dialog_ptr->option_names_widget,
      XmNlabelString, cs,
      NULL );  
   XmStringFree( cs );

   cs = XmStringCreateLtoR( values_str, "" );
   XtVaSetValues( 
      dialog_ptr->option_values_widget,
      XmNlabelString, cs,
      NULL );
   XmStringFree( cs );
}


#if NeedFunctionPrototypes
static void XmsgXmUpdateTitle(
   XmsgXmDialogStruct* dialog_ptr,
   XmsgMessageStruct*  message_ptr  )
#else
static void XmsgXmUpdateTitle( dialog_ptr, message_ptr )
   XmsgXmDialogStruct* dialog_ptr;
   XmsgMessageStruct*  message_ptr;
#endif
{
   _XmsgConst _XmsgString database_str;
   char              title_str[ 1000 ];

   database_str = XmsgGetMsgDatabaseString( 
      message_ptr, 
      XrmPermStringToQuark( "popupTitle" ), 
      XrmPermStringToQuark( "PopupTitle" ) );

   if (!database_str)
   {
      database_str = "@m";
   }

   XmsgTranslate( message_ptr, database_str, title_str );

   XtVaSetValues(
      dialog_ptr->shell_widget,
      XmNtitle, title_str,
      NULL );
}


/*------------------------------
-- Public function definitions
------------------------------*/
#if NeedFunctionPrototypes
Boolean XmsgXmCacheBitmap( 
   _XmsgConst _XmsgString bitmap_name_str,
   _XmsgConst _XmsgString bitmap_bits,
   int                    height,
   int                    width )
#else
Boolean XmsgXmCacheBitmap( bitmap_name_str, bitmap_bits, height, width )
   _XmsgConst _XmsgString bitmap_name_str;
   _XmsgConst _XmsgString bitmap_bits;
   int                    height;
   int                    width;
#endif
{
   XImage* image_ptr;
   Boolean cached;

   image_ptr = XtNew( XImage );
   memset( (void*) image_ptr, 0, sizeof( XImage ) );

   image_ptr->width            = width;
   image_ptr->height           = height;
   image_ptr->format           = XYBitmap;
   image_ptr->data             = (String) bitmap_bits;
   image_ptr->byte_order       = MSBFirst;
   image_ptr->bitmap_unit      = 8;
   image_ptr->bitmap_bit_order = LSBFirst;
   image_ptr->bitmap_pad       = 8;
   image_ptr->depth            = 1;
   image_ptr->bytes_per_line   = (width-1)/8 + 1;

   cached = XmInstallImage( image_ptr, (String) bitmap_name_str );

   if (!cached)
   {
      XtFree( (char*) image_ptr );
   }
   return cached;
}


#if NeedFunctionPrototypes
void XmsgXmDispatchPopup(
   XmsgMessageStruct* message_ptr )
#else
void XmsgXmDispatchPopup( message_ptr )
   XmsgMessageStruct* message_ptr;
#endif
{
   Boolean             delay_protocol;
   XmsgMessageStruct*  new_message_ptr;
   Widget              widget;

   delay_protocol = XmsgGetMsgDelayProtocol( message_ptr );

   new_message_ptr = XmsgCopyMessage( message_ptr );
   if (delay_protocol)
   {
      widget = XmsgGetMsgWidget( message_ptr );
  
      XtAppAddTimeOut( XtWidgetToApplicationContext( widget ),
                       (unsigned long) 10, XmsgXmDelayPopupTO, 
                       new_message_ptr );
   }
   else
   {
      XmsgXmPopup( new_message_ptr );
   }
}


#if NeedFunctionPrototypes
void XmsgXmInstall(
   Widget              w,
   unsigned int        control,
   XmsgXmLookupStruct* lookup_ptr,
   Cardinal            lookup_total )
#else
void XmsgXmInstall( w, control, lookup_ptr, lookup_total )
   Widget              w;
   unsigned int        control;
   XmsgXmLookupStruct* lookup_ptr;
   Cardinal            lookup_total;
#endif
{
#  include <bitmaps/xm_error>
#  include <bitmaps/xm_information>
#  include <bitmaps/xm_warning>
#  include "xmsg_debug.xbm"
#  include "xmsg_fatal.xbm"

   static XmsgXmLookupStruct default_lookup_table[] =
   {
      { "popupAppClass"   , "Application Class:", "@A" },
      { "popupAppName"    , "Application Name:" , "@a" },
      { "popupErrno"      , "Error Number:"     , "@e" },
      { "popupErrorText"  , "Error Text:"       , "@E" },
      { "popupHost"       , "Client Host:"      , "@H" },
      { "popupMsgCategory", "Message Category:" , "@m" },
      { "popupMsgClass"   , "Message Class:"    , "@c" },
      { "popupMsgName"    , "Message Name:"     , "@n" },
      { "popupMsgType"    , "Message Type:"     , "@t" },
      { "popupPid"        , "Process ID:"       , "@P" },
      { "popupSourceName" , "Source Name:"      , "@C" },
      { "popupSourceLine" , "Source Line:"      , "@L" },
      { "popupTime"       , "Time Generated:"   , "@T" },
      { "popupLogin"      , "Login Name:"       , "@U" },
      { "popupUser"       , "User Name:"        , "@u" },
      { "popupXDisplay"   , "X Display Server:" , "@D" }      
   };
   static Boolean initialized;

   int i;
   String tmp_str;

   XmsgXmLOCK_PROCESS

   if (!initialized)
   {
      initialized = True;

      package_data.xcontext = XUniqueContext();

      XmsgXmCacheBitmap( "Xmsg" XmsgCATEGORY_DEBUG, (char*)xmsg_debug_bits, 
                         xmsg_debug_height, xmsg_debug_width );
      XmsgXmCacheBitmap( "Xmsg" XmsgCATEGORY_ERROR, xm_error_bits, 
                         xm_error_height, xm_error_width );
      XmsgXmCacheBitmap( "Xmsg" XmsgCATEGORY_FATAL, (char*)xmsg_fatal_bits, 
                         xmsg_fatal_height, xmsg_fatal_width );
      XmsgXmCacheBitmap( "Xmsg" XmsgCATEGORY_INFO, xm_information_bits, 
                         xm_information_height, xm_information_width );
      XmsgXmCacheBitmap( "Xmsg" XmsgCATEGORY_WARNING, xm_warning_bits, 
                         xm_warning_height, xm_warning_width );
      
      if (!lookup_ptr)
      {
         lookup_ptr   = default_lookup_table;
         lookup_total = XtNumber( default_lookup_table );
      }

      tmp_str = XtMalloc( 100 );

      package_data.lookup_ptr = (XmsgXmInternalLookupStruct*) 
         XtMalloc( sizeof( XmsgXmInternalLookupStruct ) * lookup_total );

      for (i = 0; i < lookup_total; i++)
      {
         strcpy( tmp_str, lookup_ptr[ i ].name_str );
         tmp_str[ 0 ] = tolower( tmp_str[ 0 ] );
         package_data.lookup_ptr[i].name_quark = XrmStringToQuark( tmp_str );

         tmp_str[ 0 ] = toupper( tmp_str[ 0 ] );
         package_data.lookup_ptr[i].class_quark = XrmStringToQuark( tmp_str );

         package_data.lookup_ptr[i].label_str = 
            XtNewString( lookup_ptr[ i ].label_str  );

         package_data.lookup_ptr[i].lookup_str = 
            XtNewString( lookup_ptr[ i ].lookup_str );
      }
      package_data.lookup_total = lookup_total;

      XtFree( (char*) tmp_str );

      XmsgInstall( w, control & (~XmsgCONTROL_INSTALL_STDERR) );

      XmsgAddHandler( "popup", XmsgXmDispatchPopup, 0, XmsgShouldPopup );

      if (control & XmsgCONTROL_INSTALL_STDERR)
      {
         XmsgAddHandler( "stderr", XmsgHandlerStderr, 0, NULL );
      }
   }

   XmsgXmUNLOCK_PROCESS
}


