/*
 Copyright (C) 2000 Gerald L. Gay

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public License
 version 2 as published by the Free Software Foundation.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.

 You should have received a copy of the GNU Library General Public
 License along with this library; see the file COPYING.  If not,
 write to the Free Software Foundation, Inc., 675 Mass Ave,
 Cambridge, MA 02139, USA.
*/

#include <Xarm/ScrollBar.h>
#include <Xarm/Callback.h>
#include <Xarm/XarmTreeList.h>
#include <Xarm/Xarm.h>

#if defined(XARM_HAS_XBAE)

static char treeListTranslations[] = "#override  <Btn3Down>:  SelectCell(b3,b3,b3)\n"
                                     "           <Btn1Down>:  SelectCell()";

const int XarmTreeList::indentIncr = 18;

XarmTreeList::~XarmTreeList()
{
    // Delete all the items.

    deleteAllItems(rootNode);
    rootNode = NULL;
}

void XarmTreeList::deleteAllItems(XarmTreeItem *item)
{
    XarmTreeItem *nitem;

    while (item != NULL) {
        if (item->children != NULL) deleteAllItems(item->children);
        nitem = item->next;
        delete [] (char *)item;
        item = nitem;
    }
}


void XarmTreeList::treeInit()
{
    XColor boxColor;

    maxIndent      = 0;
    maxTextLen     = 0;
    numVisible     = 0;
    fontCharHeight = 0;
    fontCharWidth  = 0;
    scrWidth       = 0;
    hsbWidth       = 0;
    lastRowClicked = -1;
    numSelected    = 0;
    fid            = 0;
    disabled       = 0;

    selectCBList    = NULL;
    defActionCBList = NULL;
    expandCBList    = NULL;
    rcCBList        = NULL;
    dragCBList      = NULL;

    XtTranslations trans = XtParseTranslationTable(treeListTranslations);

    XtOverrideTranslations(widget(), trans);

    //**********************************************************************
    // The following line deserves a little explanation.
    // The XarmWidget<> template provides a way for an application
    // to convert a widget obtained from some source other than Xarm
    // into an Xarm object. In this case, the Xbae Matrix widget
    // has an internal vertical and horizontal scrollbar that it
    // manages and unmanages as needed. I want to get the width of
    // this scrollbar to adjust the size of each Pixmap to fill
    // the width that the widget is provided. The Matrix (and
    // therefore the TreeList object) has member functions
    // verticalScrollBar() and horizontalScrollBar() that return
    // the internally created widgets. I can convert them into Xarm
    // objects in several ways:
    //
    // XarmWidget<ScrollBarClass> sb = verticalScrollBar();
    //
    // or
    //
    // XarmWidget<ScrollBarClass> sb(verticalScrollBar());
    //
    // All I want to do with this object is to get it's width.
    // So I combine the 2 steps into one.
    // Note that the XarmWidget<> template does not destroy the
    // widget that it wraps on destruction (this would be a bad thing!).
    //**********************************************************************

    sbWidth  =  XarmWidget<ScrollBarClass>(verticalScrollBar()).width();

    // We're using grey80 for now...
    boxColor.red = boxColor.green = boxColor.blue = 204 << 8;
    boxColor.flags = DoRed | DoGreen | DoBlue;

    if (XAllocColor(display(), colormap(), &boxColor) == 0)
        boxPixel = BlackPixelOfScreen(screen());
    else
        boxPixel = boxColor.pixel;

    // The default (empty) width will be twice the indentIncr
    // to allow for an icon plus the expand/colapse box.
    // The height will be indentIncr until the font size is determined.
    reqWidth  = indentIncr * 2;
    reqHeight = indentIncr;

    // Prime the item list...
    rootNode = NULL;

    // Let's set all the properties we want just in case someone tried to override them.

    columns(1);
    // We set rows to 10 right now until we can figure out the right value...
    rows(10);
    cellHighlightThickness(0);
    cellShadowThickness(0);
    cellMarginWidth(0);
    cellMarginHeight(0);
    shadowThickness(0);
    gridType(XmGRID_NONE);
    fixedRows(0);
    fixedColumns(0);

    Width defWidths[2] = { 30, 0 };

    columnWidths(defWidths);

    ::addCallback(this, &XarmTreeList::drawCell,      widget(), XmNdrawCellCallback);
    ::addCallback(this, &XarmTreeList::selectCell,    widget(), XmNenterCellCallback);
    ::addCallback(this, &XarmTreeList::selectCell,    widget(), XmNselectCellCallback);
    ::addCallback(this, &XarmTreeList::resize,        widget(), XmNresizeCallback);
    ::addCallback(this, &XarmTreeList::defaultAction, widget(), XmNdefaultActionCallback);

    ::addEventHandler(this, &XarmTreeList::mouseMotion,  widget(), Button1MotionMask, false);
}

XarmTreeList::XarmTreeList(Widget w_parent,
			   ArgList arglist = NULL,
			   Cardinal cnt = 0,
			   _XtString name = NULL)
{
    Matrix *myMatrix;

    if (name == NULL) name = "XarmTreeList";
    myMatrix = new Matrix(w_parent, arglist, cnt, name);
    widget(*myMatrix);
    treeInit();
}

XarmTreeList::XarmTreeList(_XtString name, Widget w_parent)
{
    Matrix *myMatrix;

    if (name == NULL) name = "XarmTreeList";
    myMatrix = new Matrix(w_parent, NULL, 0, name);
    widget(*myMatrix);
    treeInit();
}

XarmTreeList::XarmTreeList(Widget w_parent, XarmArg &args, _XtString name)
{  
    Matrix *myMatrix;

    if (name == NULL) name = "XarmTreeList";
    myMatrix = new Matrix(w_parent, args.getArgs(), args.count(), name);
    widget(*myMatrix);
    if (args.hasTypedArgs()) args.setTypedArgs(widget());
    treeInit();
}

void XarmTreeList::addCB(XaTLCallbackType ct, FakeTLInfo *ci)
{
    if (ci == NULL) return;

    XaTLCallbackList *tlcl = new XaTLCallbackList;

    if (tlcl == NULL) return;

    tlcl->ci = ci;

    switch (ct) {
        case XaNselectCallback:         tlcl->next = selectCBList;    selectCBList = tlcl;    break;
        case XaNdefaultActionCallback:  tlcl->next = defActionCBList; defActionCBList = tlcl; break;
        case XaNexpandCallback:         tlcl->next = expandCBList;    expandCBList = tlcl;    break;
        case XaNrightClickCallback:     tlcl->next = rcCBList;        rcCBList = tlcl;        break;
        case XaNprocessDragCallback:    tlcl->next = dragCBList;      dragCBList = tlcl;      break;
        default:                        delete tlcl;
    }
}

XarmTreeItem *XarmTreeList::addItemLast(XarmTreeItem *parent,
					const char   *text,
					Pixmap        openIcon,
					Pixmap        closedIcon,
					XtPointer     closure)
{
    if (text == NULL) return NULL;
    XarmTreeItem *item = (XarmTreeItem *)new char[sizeof(XarmTreeItem) + strlen(text)];
    if (item == NULL) return NULL;

    item->userData   = closure;
    item->openIcon   = openIcon;
    item->closedIcon = closedIcon;
    item->isExpanded = false;
    item->isSelected = false;
    item->indent     = 0;
    item->prev = item->next = item->parent = item->children = NULL;
    strcpy(item->text, text);

    XarmTreeItem *result = addItemLast(parent, item);

    if (result == NULL) delete [] (char *)item;

    return result;
}

XarmTreeItem *XarmTreeList::addItemLast(XarmTreeItem *parent, XarmTreeItem *item)
{
    if (item == NULL) return NULL;
    if (rootNode == NULL) {
        if (parent != NULL) return NULL;
	rootNode = item;
	item->parent = NULL;
    } else {
        if (parent == NULL) return NULL;
	// Walk the parent up and make sure he's in our tree!
	if (parent != rootNode) {
	    XarmTreeItem *pp = parent->parent;
	    while ((pp != NULL) && (pp != rootNode)) pp = pp->parent;
	    if (pp == NULL) return NULL;
	}
	if (parent->children == NULL) {
	    parent->children = item;
	    item->parent = parent;
	    item->indent = parent->indent + indentIncr;
	    if (item->indent > maxIndent) maxIndent = item->indent;
	} else {
	    XarmTreeItem *curChild = parent->children;
	    while ((curChild != NULL) && (curChild->next != NULL)) curChild = curChild->next;
	    if (curChild == NULL) return NULL;
	    curChild->next = item;
	    item->prev = curChild;
	    item->parent = curChild->parent;
	    item->indent = curChild->indent;
	}
    }
    queryTextLen(item);
    refreshTree();
    return item;
}

void XarmTreeList::queryTextLen(const XarmTreeItem *item)
{

    int dirRet, faRet, fdRet;
    XCharStruct lenRet;
    int calcLen;

    if (fid == 0) return;

    XQueryTextExtents(display(), fid, item->text, strlen(item->text), &dirRet, &faRet, &fdRet, &lenRet);

    calcLen = lenRet.width + item->indent + (indentIncr * 2);
    if (calcLen > reqWidth) reqWidth = calcLen;

    if (lenRet.width > maxTextLen) {
        maxTextLen = lenRet.width;
    }
}

XarmTreeItem *XarmTreeList::addItemFirst(XarmTreeItem *parent,
                                         const char   *text,
                                         Pixmap        openIcon,
                                         Pixmap        closedIcon,
                                         XtPointer     closure)
{
    if (text == NULL) return NULL;
    XarmTreeItem *item = (XarmTreeItem *)new char[sizeof(XarmTreeItem) + strlen(text)];
    if (item == NULL) return NULL;

    item->userData   = closure;
    item->openIcon   = openIcon;
    item->closedIcon = closedIcon;
    item->isExpanded = false;
    item->isSelected = false;
    item->indent     = 0;
    item->prev = item->next = item->parent = item->children = NULL;
    strcpy(item->text, text);

    XarmTreeItem *result = addItemFirst(parent, item);

    if (result == NULL) delete [] (char *)item;

    return result;
}

XarmTreeItem *XarmTreeList::addItemFirst(XarmTreeItem *parent, XarmTreeItem *item)
{
    if (item == NULL) return NULL;
    if (rootNode == NULL) {
        if (parent != NULL) return NULL;
	rootNode = item;
	item->parent = NULL;
    } else {
        if (parent == NULL) return NULL;
	// Walk the parent up and make sure he's in our tree!
	if (parent != rootNode) {
	    XarmTreeItem *pp = parent->parent;
	    while ((pp != NULL) && (pp != rootNode)) pp = pp->parent;
	    if (pp == NULL) return NULL;
	}
	if (parent->children == NULL) {
	    parent->children = item;
	    item->parent = parent;
	    item->indent = parent->indent + indentIncr;
	    if (item->indent > maxIndent) maxIndent = item->indent;
	} else {
	    XarmTreeItem *curChild = parent->children;
            parent->children = item;
            item->next = curChild;
	    item->prev = NULL;
	    item->parent = curChild->parent;
	    item->indent = curChild->indent;
	}
    }
    queryTextLen(item);
    refreshTree();
    return item;
}

XarmTreeItem *XarmTreeList::addItemBefore(XarmTreeItem *other,
                                          const char   *text,
                                          Pixmap        openIcon,
                                          Pixmap        closedIcon,
                                          XtPointer     closure)
{
    if (text == NULL)  return NULL;
    if (other == NULL) return NULL;
    XarmTreeItem *item = (XarmTreeItem *)new char[sizeof(XarmTreeItem) + strlen(text)];
    if (item == NULL) return NULL;

    item->userData   = closure;
    item->openIcon   = openIcon;
    item->closedIcon = closedIcon;
    item->isExpanded = false;
    item->isSelected = false;
    item->indent     = 0;
    item->prev = item->next = item->parent = item->children = NULL;
    strcpy(item->text, text);

    XarmTreeItem *result = addItemBefore(other, item);

    if (result == NULL) delete [] (char *)item;

    return result;
}

XarmTreeItem *XarmTreeList::addItemBefore(XarmTreeItem *other, XarmTreeItem *item)
{
    if (item == NULL)      return NULL;
    if (other == NULL)     return NULL;
    if (other == rootNode) return NULL; // Cannot add before root!

    // Walk the parent up and make sure he's in our tree!
    if (other != rootNode) {
        XarmTreeItem *pp = other->parent;
        while ((pp != NULL) && (pp != rootNode)) pp = pp->parent;
        if (pp == NULL) return NULL;
    }

    if (other->prev == NULL) {
        other->parent->children = item;
    } else {
        other->prev->next = item;
    }

    item->next              = other;
    item->prev              = other->prev;
    other->prev             = item;
    item->indent            = other->indent;
    item->parent            = other->parent;

    queryTextLen(item);
    refreshTree();
    return item;
}

XarmTreeItem *XarmTreeList::addItemAfter(XarmTreeItem *other,
                                         const char   *text,
                                         Pixmap        openIcon,
                                         Pixmap        closedIcon,
                                         XtPointer     closure)
{
    if (text == NULL)  return NULL;
    if (other == NULL) return NULL;
    XarmTreeItem *item = (XarmTreeItem *)new char[sizeof(XarmTreeItem) + strlen(text)];
    if (item == NULL) return NULL;

    item->userData   = closure;
    item->openIcon   = openIcon;
    item->closedIcon = closedIcon;
    item->isExpanded = false;
    item->isSelected = false;
    item->indent     = 0;
    item->prev = item->next = item->parent = item->children = NULL;
    strcpy(item->text, text);

    XarmTreeItem *result = addItemAfter(other, item);

    if (result == NULL) delete [] (char *)item;

    return result;
}

XarmTreeItem *XarmTreeList::addItemAfter(XarmTreeItem *other, XarmTreeItem *item)
{
    if (item == NULL)      return NULL;
    if (other == NULL)     return NULL;
    if (other == rootNode) return NULL; // Cannot add after root!

    // Walk the parent up and make sure he's in our tree!
    if (other != rootNode) {
        XarmTreeItem *pp = other->parent;
        while ((pp != NULL) && (pp != rootNode)) pp = pp->parent;
        if (pp == NULL) return NULL;
    }

    if (other->parent == NULL) return NULL;

    if (other->next != NULL) other->next->prev = item;

    item->prev = other;
    item->next = other->next;
    other->next = item;
    item->parent = other->parent;
    item->indent = other->indent;

    queryTextLen(item);
    refreshTree();
    return item;
}

XarmTreeItem *XarmTreeList::getItemAt(int row)
{
    int endRow = 0;

    return getItemAt(rootNode, row, endRow);
}

XarmTreeItem *XarmTreeList::getItemAt(XarmTreeItem *curNode, int gRow, int &cRow)
{
    if (curNode == NULL) return NULL;

    if (cRow == gRow) return curNode;

    XarmTreeItem *item;
    XarmTreeItem *item2;

    // If this node is expanded, go down into it's children...

    if ((curNode->children != NULL) && (curNode->isExpanded))
        item = getItemAt(curNode->children, gRow, ++cRow);

    // If we found the row, return it...
    if (cRow == gRow) return item;

    // If this node has any siblings, walk down to the end...

    item = curNode;

    while (item->next != NULL) {
        item = item->next;
	if (++cRow == gRow) return item;
	if ((item->children != NULL) && (item->isExpanded))
	    item2 = getItemAt(item->children, gRow, ++cRow);
	if (cRow == gRow) return item2;
    }

    return NULL;
}

XarmTreeItem *XarmTreeList::getLastVisible()
{
    XarmTreeItem *item = rootNode;

    if (item == NULL) return NULL;

    do {

        // Walk down to the end of the current list...

        while (item->next != NULL) item = item->next;

	// If the end node is expanded, traverse it...

	if ((item->children != NULL) && (item->isExpanded)) {
	    item = item->children;
	}

    } while (item->next != NULL);

    return item;
}

int XarmTreeList::getNumVisible(XarmTreeItem *item)
{

    int nv = 0;

    if (item == NULL) return 0;

    if (item->indent > maxIndent) maxIndent = item->indent;

    do {

        queryTextLen(item);

        ++nv;

	if ((item->children != NULL) && item->isExpanded)
	    nv += getNumVisible(item->children);

	item = item->next;

    } while (item != NULL);

    return nv;
}

void XarmTreeList::selectRange(int first, int last)
{
    if (first > last) return;

    for (int x = first; x <= last; ++x) {

        XarmTreeItem *item;

	item = getItemAt(x);
	if (item != NULL) item->isSelected = true;
    }
}

void XarmTreeList::deselectAll(XarmTreeItem *item)
{
    if (item == NULL) return;

    do {

        item->isSelected = false;

	if (item->children != NULL) deselectAll(item->children);

	item = item->next;

    } while (item != NULL);
}

void XarmTreeList::mouseMotion(Widget, XtPointer, XEvent *ev, Boolean *)
{

    if (ev->type != MotionNotify) return;

    if (dragCBList == NULL) return;

    // Ignore it until we're at least 3 pixels away
    // from where we started.

    int dx = ev->xmotion.x - lastBPEvent.xbutton.x;
    int dy = ev->xmotion.y - lastBPEvent.xbutton.y;

    if (dx < 0) dx = -dx;
    if (dy < 0) dy = -dy;

    if ((dx < 3) && (dy < 3)) return;

    int rowNum = (lastBPEvent.xbutton.y / reqHeight) + topRow();

    XarmTreeItem *item = getItemAt(lastRowClicked);

    if (item == NULL) return;

    callCB(dragCBList, item, &lastBPEvent);
}

void XarmTreeList::defaultAction(Widget, XtPointer, XtPointer cdata)
{
    XbaeMatrixDefaultActionCallbackStruct *dac = (XbaeMatrixDefaultActionCallbackStruct *)cdata;

    activeRow = dac->row;

    if (dac->event->type == ButtonPressMask) {

        XarmTreeItem *item = getItemAt(dac->row);

	if (item == NULL) return;

	// Only do the double-click action if clicked on 
	// the text of the item.

	int xpos = dac->event->xbutton.x;

	if (xpos < item->indent) return;

	xpos -= item->indent;

	if ((item == rootNode) && (xpos > indentIncr)) {
	    callCB(defActionCBList, item);
	    return;
	}

	if (xpos > indentIncr) {
	    callCB(defActionCBList, item);
	}
    }
}

void XarmTreeList::removeItem(XarmTreeItem *item)
{

    if (item == NULL) return;

    // This is easy!
    if (item == rootNode) {
        rootNode = NULL;
        refreshTree();
    }

    // Walk the parent up and make sure he's in our tree!
    XarmTreeItem *parent = item->parent;
    if (parent != rootNode) {
        XarmTreeItem *pp = parent->parent;
        while ((pp != NULL) && (pp != rootNode)) pp = pp->parent;
        if (pp == NULL) return;
    }

    if (item->prev != NULL) {
        item->prev->next = item->next;
    } else {
        item->parent->children = item->next;
    }

    item->parent = NULL;
    item->prev   = NULL;
    item->next   = NULL;

    refreshTree();
}

void XarmTreeList::deleteItem(XarmTreeItem *item)
{
   if (item == NULL) return;

   removeItem(item);

   deleteAllItems(item);
}

typedef void (FakeTLTarget::*p_tlf)(Widget, XtPointer, XtPointer);

void XarmTreeList::callCB(XaTLCallbackList *acb, XarmTreeItem *item, XEvent *ev)
{
    XarmTreeDragCallbackStruct xtdcs;
    XtPointer ptr;

    if (ev != NULL) {
        xtdcs.item  = item;
        xtdcs.event = ev;
        ptr = (XtPointer)&xtdcs;
    } else {
        ptr = (XtPointer)item;
    }

    while (acb != NULL) {
        FakeTLTarget *obj = acb->ci->_target;
        p_tlf         mf  = acb->ci->_callback;

        (obj->*(mf))(widget(), acb->ci->_closure, ptr);

        acb = acb->next;
    }
}

void XarmTreeList::drawCell(Widget, XtPointer, XtPointer cdata)
{
    XbaeMatrixDrawCellCallbackStruct *dcc = (XbaeMatrixDrawCellCallbackStruct *)cdata;

    activeRow = dcc->row;

    int myWidth = (scrWidth > reqWidth) ? scrWidth : reqWidth;

    // Figure out how big to make the pixmap, 18x18 for box + 18 for icon + some for text...
    Pixmap drawMap = XCreatePixmap(display(),
				   window(),
				   myWidth,
				   reqHeight,
				   depth());

    unsigned long valuemask;
    XGCValues     values;
    GC            drawGC;
    XarmTreeItem *item = getItemAt(dcc->row);

    int iconX, iconY;

    values.foreground = background();
    values.font       = fid;
    valuemask = GCForeground | GCFont;

    drawGC = XCreateGC(display(), drawMap, valuemask, &values);

    XFillRectangle(display(), drawMap, drawGC, 0, 0, myWidth, reqHeight);

    if (item != NULL) {

        // Draw the lines...
        values.foreground = boxPixel;
	values.line_style = LineSolid;
	valuemask = GCForeground | GCLineStyle;
	XChangeGC(display(), drawGC, valuemask, &values);

	if (item != rootNode) {
	    // we first need to figure out how many indented lines
	    // we need to draw.
            XarmTreeItem *ti = item;
            iconX = item->indent - (indentIncr * 2) + (indentIncr / 2);

	    while (ti->parent != NULL) {
                if (ti->parent->next != NULL)
                    XDrawLine(display(), drawMap, drawGC, iconX, 0, iconX, reqHeight - 1);
                ti = ti->parent;
                iconX -= indentIncr;
	    }

	    // Now draw the connecting lines.
	    // We always draw the top line
	    iconX = item->indent + (indentIncr / 2) - indentIncr;
	    XDrawLine(display(), drawMap, drawGC, iconX, 0, iconX, reqHeight / 2);
	    // And we always draw the line to our item
	    XDrawLine(display(), drawMap, drawGC,
		      iconX, reqHeight / 2,
		      iconX + (indentIncr / 2), reqHeight / 2);
	    // If we're not last, draw the bottom line too...
	    if (item->next != NULL)
	        XDrawLine(display(), drawMap, drawGC, iconX, (reqHeight / 2) - 1,
			  iconX, reqHeight - 1);

	    // If this item has children, draw the box...

	    if (item->children != NULL) {

	        values.foreground = background();
		values.line_style = LineSolid;
		valuemask = GCForeground | GCLineStyle;
		XChangeGC(display(), drawGC, valuemask, &values);

		XFillRectangle(display(), drawMap, drawGC, iconX - 4, (reqHeight / 2) - 4, 8, 8);

	        values.foreground = foreground();
		valuemask = GCForeground;
		XChangeGC(display(), drawGC, valuemask, &values);

		XDrawRectangle(display(), drawMap, drawGC, iconX - 4, (reqHeight / 2) - 4,
			       8, 8);

		// Draw the minus part...
		XDrawLine(display(), drawMap, drawGC, iconX - 2, (reqHeight / 2),
			  iconX + 2, reqHeight / 2);

		// If it's not expanded, turn it into a plus!
		if (!item->isExpanded)
		    XDrawLine(display(), drawMap, drawGC, iconX, (reqHeight / 2) - 2,
			      iconX, (reqHeight / 2) + 2);
	    }
	}

        values.foreground = foreground();
	valuemask = GCForeground;
	XChangeGC(display(), drawGC, valuemask, &values);

	iconX = ((indentIncr - 16) / 2) + item->indent;
        iconY = (reqHeight  - 16) / 2;

        Pixmap ipm;

        if (item->isExpanded) ipm = item->openIcon;
        else                  ipm = item->closedIcon;

        if (ipm != XmUNSPECIFIED_PIXMAP) {
            XCopyArea(display(), ipm, drawMap, drawGC, 0, 0, 16, 16, iconX, iconY);
        }

        if (item->isSelected) {
            values.foreground = background();
            values.background = highlightColor();
            valuemask = GCForeground | GCBackground;
        } else {
            values.foreground = foreground();
            values.background = background();
            valuemask = GCForeground | GCBackground;
        }

        XChangeGC(display(), drawGC, valuemask, &values);

	// Re-use iconX/Y (sorry)
	iconX = item->indent + (indentIncr * 2) - indentIncr;
	iconY = ((reqHeight - fontCharHeight) / 2) + fontAscent;
	XDrawImageString(display(), drawMap, drawGC, iconX, iconY, item->text, strlen(item->text));

    }

    XFreeGC(display(), drawGC);
    dcc->type = XbaePixmap;
    dcc->pixmap = drawMap;
}

void XarmTreeList::selectCell(Widget, XtPointer, XtPointer cdata)
{

    XbaeMatrixSelectCellCallbackStruct *scc = (XbaeMatrixSelectCellCallbackStruct *)cdata;

    int    idmy1, idmy2;
    Window wdmy1, wdmy2;

    unsigned int bmask;
    int          xpos;
    int          ypos;
    int          saveRow = lastRowClicked;
    bool         doRefreshAll = false;

    XQueryPointer(display(), window(),
		  &wdmy1, &wdmy2,
		  &idmy1, &idmy2, &xpos, &ypos,
		  &bmask);

    if (!(bmask & (Button1Mask|Button3Mask))) return;

    XarmTreeItem *item = getItemAt(scc->row);

    if (item == NULL) return;

    activeRow = scc->row;

    // Find out if this was a right click

    if (bmask & Button3Mask) {
        callCB(rcCBList, item);
        return;
    }

    if ((scc->event != NULL) && (scc->event->type == ButtonPress))
        memcpy(&lastBPEvent, scc->event, sizeof(XEvent));

    // Adjust xpos if the horizontal scrollbar is managed
    XarmWidget<ScrollBarClass> hb = horizontalScrollBar();
    if (hb.managed()) {
        xpos += hb.value();
    }

    // Figure out where on the cell we got clicked

    if (xpos < (item->indent - indentIncr)) {
        return;
    }

    lastRowClicked = scc->row;

    xpos -= item->indent;

    if ((item == rootNode) && (xpos < indentIncr)) {
	return;
    }

    if (xpos < indentIncr) {
        if ((item != rootNode) && (item->children != NULL)) {
	    item->isExpanded = !item->isExpanded;
	    // Expanding clears the selection
	    lastRowClicked = -1;
	    deselectAll();
	    numSelected = 0;
            callCB(expandCBList, item);
	    refreshTree();
	}
	return;
    }

    // Determine what kind of click we got...

    if (bmask & ShiftMask) {
        // Shift-Click: Select between here and lastRowClicked
        if (numSelected) {
	    deselectAll();
	    doRefreshAll = true;
	}
	int high, low;

	if (saveRow < lastRowClicked) { high = lastRowClicked; low = saveRow; }
	else                          { high = saveRow; low = lastRowClicked; }
	selectRange(low, high);
	numSelected = high - low + 1;
	if (numSelected > 1) doRefreshAll = true;
	// Shift-Click doesn't change lastRowClicked
	lastRowClicked = saveRow;
    } else if (bmask & ControlMask) {
        // Control-Click: Toggle this item
        item->isSelected = !item->isSelected;
	if (item->isSelected) ++numSelected;
	else                  --numSelected;
    } else {
        // Regular Click: Clear all selections and select this guy
        if (numSelected) {
	    deselectAll();
	    doRefreshAll = true;
	}
	item->isSelected = true;
	numSelected = 1;
    }

    callCB(selectCBList, item);

    if (doRefreshAll) refreshTree();
    else              refreshCell(scc->row, 0);
}

void XarmTreeList::resize(Widget, XtPointer, XtPointer cdata)
{
    static int mH = 0;
    static int mW = 0;
    static bool inResize = false;

    XbaeMatrixResizeCallbackStruct *rcs = (XbaeMatrixResizeCallbackStruct *)cdata;

    // Don't do anything on the first resize

    if ((mH == 0) && (mW == 0)) {
        mH = rcs->height;
	mW = rcs->width;
	return;
    }

    // Don't do anything until the Matrix has figured out what it wants
    if ((rcs->height == mH) && (rcs->width == mW)) {
        return;
    }

    // Lock out mW/mH
    mW = mH = -1;

    if (inResize) return;

    inResize = true;

    // Calculate how many rows will fit vertically and how wide our
    // column needs to be to occupy the entire width.
    if (fontCharHeight == 0) fontInit();

    int myHeight = (fontCharHeight > reqHeight) ? fontCharHeight : reqHeight;

    int needRows = rcs->height / myHeight;
    scrWidth = rcs->width;

    // Subtract sbWidth pixels to leave room
    // for the scroll bar.

    scrWidth -= sbWidth;

    visibleRows(needRows + 2);

    if (needRows > numVisible) rows(needRows);
    else                       rows(numVisible);

    Width myColumnWidths[2] = { 0, 0 };

    int myWidth = (scrWidth > reqWidth) ? scrWidth : reqWidth;

    myWidth /= fontCharWidth;

    myColumnWidths[0] = myWidth;
    columnWidths(myColumnWidths);

    inResize = false;
}

void XarmTreeList::showTree(XarmTreeItem *item)
{
    if (item == NULL) return;

    while (item != NULL) {
        for (int x = 0; x < (item->indent / 5); ++x) cout << " ";
	cout << item->text << endl;
	if ((item->children != NULL) && item->isExpanded) showTree(item->children);
	item = item->next;
    }
}

void XarmTreeList::expandTree(XarmTreeItem *branch)
{
    if (branch) {
        branch->isExpanded = true;
	refreshTree();
    }
}

void XarmTreeList::fontInit()
{

    XmFontContext ctx;

    fontCharWidth  = 0;
    fontCharHeight = 0;
    fontAscent     = 0;
    Boolean bResult = XmFontListInitFontContext(&ctx, fontList());

    if (bResult == True) {
        XmFontListEntry fle = XmFontListNextEntry(ctx);

	while (fle != NULL) {
	    XmFontType xft;
	    XtPointer xtp = XmFontListEntryGetFont(fle, &xft);
	    fle = XmFontListNextEntry(ctx);
	    if (xft == XmFONT_IS_FONT) {
	        XFontStruct *xfs = (XFontStruct *)xtp;
		// We use the same formula the Matrix uses
		// to calculate the font width/height.
		fontCharWidth  = (xfs->max_bounds.width + xfs->min_bounds.width) / 2;
		fontCharHeight = xfs->max_bounds.descent + xfs->max_bounds.ascent;
		fontAscent     = xfs->max_bounds.ascent;
		fid = xfs->fid;
		break;
	    }
	}
	XmFontListFreeFontContext(ctx);
    }
}

void XarmTreeList::refreshTree()
{

    if (disabled) return;

    // Recalculate the width/height per-cell
    // requirements and tell the matrix to redraw.

    int           charsPerIcon      = 0;
    Width         myColumnWidths[2] = { 0, 0 };

    disableRedisplay();
    fontInit();

    maxTextLen = 0;
    maxIndent  = 0;
    reqWidth   = 0;

    // Recalculate the number of visible items.
    numVisible = getNumVisible();

    // OK, we know how tall and wide a character is in the current font
    // and we know the max string length we currently have. The Matrix wants
    // a width in characters so we need to figure out how many character cells
    // the icon and expand box take up. Then we calculate the size of a cell
    // in pixels for use in creating our pixmaps.

    if (fontCharWidth > 0) {
        charsPerIcon = ((indentIncr * 2) + (fontCharWidth - 1)) / fontCharWidth;
	charsPerIcon = ((maxIndent / indentIncr) - 1) * charsPerIcon;
    }

    int myWidth;

//    reqWidth = maxTextLen + maxIndent;

    myWidth = (scrWidth > reqWidth) ? scrWidth : reqWidth;

    myWidth += (2 * indentIncr) + fontCharWidth - 1;

    myWidth /= fontCharWidth;

    if (myWidth < 4) myWidth = 4;

    myColumnWidths[0] = myWidth - 3;

    columnWidths(myColumnWidths);

    if (fontCharHeight > indentIncr) reqHeight = fontCharHeight;
    else                             reqHeight = indentIncr;

    if (visibleRows() > numVisible) rows(visibleRows());
    else                            rows(numVisible);

    enableRedisplay();
    refresh();
}

#endif
