/*
 * Copyright (c) 1999, Gerald L. Gay
 * 
 * You may distribute this source code under the terms of the license
 * in the COPYING file.
 * 
 */

/* Include(s) */
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <math.h>
#if defined(XARM_HAS_NEWHEADERS)
#include <iostream>
#else
#include <iostream.h>
#endif
#include <Xarm/AppContext.h>
#include <Xarm/PushB.h>
#include <Xarm/MessageB.h>
#include <Xarm/Form.h>
#include <Xarm/Dial.h>
#include <Xarm/Label.h>
#include <Xarm/Scale.h>
#include <Xarm/Callback.h>
#include <Xarm/GLWidgets.h>

#if defined(XARM_HAS_NAMESPACES)
using namespace std;
#endif

#if defined(XARM_HAS_GL)

//****************************************
//* If you have a fast machine, you may  *
//* want to modify MAX_FPS and see how   *
//* it does. Don't set MIN_FPS below 1   *
//* unless you want to fix the code!     *
//****************************************

#define MIN_FPS  1
#define MAX_FPS  41
#define DEF_FPS  24

/* Declaration(s) */
class Application : public AppContext
{
private:

    Form           *myForm;
    GLDrawingArea  *cubes[2];
    PushButton     *quitB;
    Dial           *fpsDial;
    Label          *fpsLabel;
    Scale          *fpsScale;
    QuestionDialog *exitDiag;

    GLXContext      ctx[2];
    GLint           Black, Red, Green, Blue;
    GLfloat         xrot, yrot, zrot;
    unsigned long   delay;

    /* Callbacks */
    void onExit(Widget,XtPointer,XtPointer);
    void onInc(Widget,XtPointer,XtPointer);
    void exit(Widget,XtPointer,XtPointer);
    void onRealize(Widget,XtPointer,XtPointer);

    /* Timer Procs */
    void  next_frame(XtPointer, XtIntervalId *);

    /* Regular Member Functions */
    void  draw_cube();
    void  init_display();
    void  doDisplay(GLDrawingArea *, GLXContext);
    GLint alloc_color(Colormap cmap, int red, int green, int blue);
    void  translate_pixels(CoreClass *to, CoreClass *from, ...);
    void  first_frame();
  
public:
    Application(char *app_class,int &argc_in_out, char **argv_in_out);

};

int main (int argc, char **argv)
{
    // initialize application 
    Application app("XarmGLDemo",argc,argv);

    // realize application
    app.realize();

    // start the main loop 
    app.mainLoop();

    return EXIT_SUCCESS;
}

Application::Application(char *app_class, int &argc_in_out, char **argv_in_out):
   AppContext(app_class, NULL, 0, argc_in_out, argv_in_out)
{
    XarmArg args;

    delay = 1000/DEF_FPS;

    // install window manager message handler(s)
    Atom proto = addWMProtocol(XA_WM_DELETE_WINDOW);
    ::addWMProtocolCallback(this, &Application::onExit, widget(), proto);

    title(app_class);

    // install window(s) into the application window 
    exitDiag = new QuestionDialog(*this, "Do you really want to quit?");
    exitDiag->dialogTitle("Exit Dialog");
    XtUnmanageChild(exitDiag->getChild(XmDIALOG_HELP_BUTTON));
    ::addCallback(this, &Application::exit, exitDiag, XmNokCallback);

    myForm = new Form("myForm", *this);

    args(XmNtopAttachment,    XmATTACH_FORM)
        (XmNtopOffset,        10)
        (XmNleftAttachment,   XmATTACH_FORM)
        (XmNleftOffset,       10)
        (XmNbottomAttachment, XmATTACH_FORM)
        (XmNbottomOffset,     10)
        (GLwNrgba,            True)
        (GLwNinstallColormap, True)
        (GLwNdoublebuffer,    True)
        (XmNwidth,            400)
        (XmNheight,           400);

    cubes[0] = new GLDrawingArea(*myForm, args, "cubes0");

    args.reset();
    args(XmNtopAttachment,    XmATTACH_FORM)
        (XmNtopOffset,        10)
        (XmNleftAttachment,   XmATTACH_WIDGET)
        (XmNleftWidget,       cubes[0])
        (XmNleftOffset,       10)
        (XmNrightAttachment,  XmATTACH_FORM)
        (XmNrightOffset,      10)
        (GLwNrgba,            True)
        (GLwNinstallColormap, True)
        (GLwNdoublebuffer,    True)
        (XmNwidth,            200)
        (XmNheight,           200);

    cubes[1] = new GLDrawingArea(*myForm, args, "cubes1");

    args.reset();
    args(XmNtopAttachment,       XmATTACH_WIDGET)
        (XmNtopWidget,           cubes[1])
        (XmNtopOffset,           5)
        (XmNleftAttachment,      XmATTACH_WIDGET)
        (XmNleftWidget,          cubes[0])
        (XmNleftOffset,          10)
        (XmNrightAttachment,     XmATTACH_FORM)
        (XmNrightOffset,         10);

    fpsLabel = new Label(*myForm, args, "fpsLabel");
    fpsLabel->labelString("Frames Per Second");

    args.reset();
    args(XmNtopAttachment,       XmATTACH_WIDGET)
        (XmNtopWidget,           fpsLabel)
        (XmNtopOffset,           5)
        (XmNleftAttachment,      XmATTACH_WIDGET)
        (XmNleftWidget,          cubes[0])
        (XmNleftOffset,          10)
        (XmNrightAttachment,     XmATTACH_FORM)
        (XmNrightOffset,         10)
        (XmNheight,              150)
        (XmNminimum,             MIN_FPS)
        (XmNmaximum,             MAX_FPS)
        (XmNvalue,               DEF_FPS)
        (XmNtickIncrement,       5)
        (XmNtickSize,            5)
        (XmNarcMargin,           5)
        (XmNborderMargin,        0)
        (XmNprocessingDirection, XmMAX_ON_RIGHT)
        (XmNfill,                True)
        (XmNshowValue,           True)
        (XmNshowHub,             True)
        (XmNstyle,               PointerStyle)
        (XmNhighlightThickness,  0)
        (XmNforeground, XmRString, "blue",   4)
        (XmNbackground, XmRString, "grey50", 6);

    fpsDial = new Dial(*myForm, args, "fpsDial");

    args.reset();
    args(XmNtopAttachment,       XmATTACH_WIDGET)
        (XmNtopWidget,           fpsDial)
        (XmNtopOffset,           5)
        (XmNleftAttachment,      XmATTACH_WIDGET)
        (XmNleftOffset,          10)
        (XmNleftWidget,          cubes[0])
        (XmNrightAttachment,     XmATTACH_FORM)
        (XmNrightOffset,         10)
        (XmNprocessingDirection, XmMAX_ON_RIGHT)
        (XmNminimum,             MIN_FPS)
        (XmNmaximum,             MAX_FPS)
        (XmNvalue,               DEF_FPS)
        (XmNorientation,         XmHORIZONTAL)
        (XmNhighlightThickness,  0);

    fpsScale = new Scale(*myForm, args, "fpsScale");
    fpsScale->manage();

    args.reset();
    args(XmNtopAttachment,      XmATTACH_WIDGET)
        (XmNtopWidget,          fpsScale)
        (XmNtopOffset,          5)
        (XmNleftAttachment,     XmATTACH_WIDGET)
        (XmNleftOffset,         10)
        (XmNleftWidget,         cubes[0])
        (XmNrightAttachment,    XmATTACH_FORM)
        (XmNrightOffset,        10)
        (XmNbottomAttachment,   XmATTACH_FORM)
        (XmNbottomOffset,       10)
        (XmNhighlightThickness, 0);

    quitB = new PushButton(*myForm, args, "quit");
    quitB->labelString("Quit");

    ::addCallback(this, &Application::onExit,    quitB,    XmNactivateCallback);
    ::addCallback(this, &Application::onInc,     fpsScale, XmNvalueChangedCallback);
    ::addCallback(this, &Application::onInc,     fpsScale, XmNdragCallback);
    ::addCallback(this, &Application::onRealize, cubes[0], GLwNginitCallback);

    myForm->manage();
}

void Application::onInc(Widget, XtPointer, XtPointer)
{
    int newFPS = fpsScale->value();
    fpsDial->value(newFPS);
    delay = 1000 / newFPS;
}

void Application::onRealize(Widget, XtPointer, XtPointer)
{

    Boolean rgba, doublebuffer, cmap_installed;
    XVisualInfo *vi;

    rgba = cubes[0]->rgba();
    cmap_installed = cubes[0]->installColormap();
    doublebuffer = cubes[0]->doublebuffer();
    vi = cubes[0]->visualInfo();

    /* create visual contexts */
    ctx[0] = glXCreateContext(display(), vi, NULL, GL_FALSE);
    ctx[1] = glXCreateContext(display(), vi, NULL, GL_FALSE);
    cubes[0]->makeCurrent(ctx[0]);

    if (rgba) {
        Black = Red = Green = Blue = 0;

        if (cmap_installed) {
            /* In RGBA mode, the Mesa widgets will have their own color map.
               Adjust the colors of the other widgets so that--even if the rest
               of the screen has wrong colors--all application widgets have the
               right colors.  */

            translate_pixels (cubes[0], quitB, XtNbackground, XtNforeground, XtNborder, NULL);
            translate_pixels (cubes[0], myForm, XtNbackground, XtNborder, NULL);

            /* Finally warp the pointer into the mesa widget, to make sure that
               the user sees the right colors at the beginning.  */

            XWarpPointer (*this, None, cubes[0]->window(), 0, 0, 0, 0, 0, 0);
        }
    } else {
        /* Allocate a few colors for use in color index mode.  */

        Colormap cmap;
        cmap = DefaultColormap (display(), DefaultScreen (display()));
        Black = alloc_color (cmap, 0x0000, 0x0000, 0x0000);
        Red   = alloc_color (cmap, 0xffff, 0x0000, 0x0000);
        Green = alloc_color (cmap, 0x0000, 0xffff, 0x0000);
        Blue  = alloc_color (cmap, 0x0000, 0x0000, 0xffff);
    }

    init_display();
    first_frame();
}

void Application::init_display()
{

    xrot = yrot = zrot = 0.0;

    for (int nct = 0; nct < (sizeof(ctx)/sizeof(ctx[0])); ++nct) {

        cubes[nct]->makeCurrent(ctx[nct]);

        glClearColor (0.0, 0.0, 0.0, 0.0);
        glClearIndex (Black);
        glClearDepth (10.0);

        glMatrixMode (GL_PROJECTION);
        glLoadIdentity ();
        glFrustum (-1.0, 1.0, -1.0, 1.0, 1.0, 10.0);
        glTranslatef (0.0, 0.0, -3.0);
    
        glMatrixMode (GL_MODELVIEW);
        glLoadIdentity ();
        
        glCullFace (GL_BACK);
        glEnable (GL_CULL_FACE);
        
        glShadeModel (GL_FLAT);
    }
}

void Application::first_frame()
{
    ::addTimeOut(this, &Application::next_frame, appContext(), delay);
}

void Application::next_frame(XtPointer, XtIntervalId *)
{
    first_frame();
    for (int curCube = 0; curCube < (sizeof(cubes)/sizeof(cubes[0])); ++curCube) {
        doDisplay(cubes[curCube], ctx[curCube]);
    }
}

void Application::doDisplay(GLDrawingArea *gda, GLXContext cx)
{
    int direction;

    /* Spin the small cube backwards! */

    if (gda == cubes[0]) direction = 1;
    else                 direction = -1;

    gda->makeCurrent(cx);
    glClear (GL_COLOR_BUFFER_BIT);
    glPushMatrix ();
    {
        glRotatef (xrot * direction, 1.0, 0.0, 0.0);
        glRotatef (yrot * direction, 0.0, 1.0, 0.0);
        glRotatef (zrot * direction, 0.0, 0.0, 1.0);
        draw_cube ();
    }
    glPopMatrix ();
    glFinish ();
    gda->swapBuffers();

    xrot += 1.0;
    yrot += 0.7;
    zrot -= 0.3;
}

void Application::draw_cube()
{
    /* X faces */
    glIndexi (Red);
    glColor3f (1.0, 0.0, 0.0);
    glBegin (GL_POLYGON);
    {
        glVertex3f (1.0, 1.0, 1.0);
        glVertex3f (1.0, -1.0, 1.0);
        glVertex3f (1.0, -1.0, -1.0);
        glVertex3f (1.0, 1.0, -1.0);
    }
    glEnd ();

    glBegin (GL_POLYGON);
    {
        glVertex3f (-1.0, 1.0, 1.0);
        glVertex3f (-1.0, 1.0, -1.0);
        glVertex3f (-1.0, -1.0, -1.0);
        glVertex3f (-1.0, -1.0, 1.0);
    }
    glEnd ();

    /* Y faces */
    glIndexi (Green);
    glColor3f (0.0, 1.0, 0.0);
    glBegin (GL_POLYGON);
    {
        glVertex3f (1.0, 1.0, 1.0);
        glVertex3f (1.0, 1.0, -1.0);
        glVertex3f (-1.0, 1.0, -1.0);
        glVertex3f (-1.0, 1.0, 1.0);
    }
    glEnd ();

    glBegin (GL_POLYGON);
    {
        glVertex3f (1.0, -1.0, 1.0);
        glVertex3f (-1.0, -1.0, 1.0);
        glVertex3f (-1.0, -1.0, -1.0);
        glVertex3f (1.0, -1.0, -1.0);
    }
    glEnd ();

    /* Z faces */
    glIndexi (Blue);
    glColor3f (0.0, 0.0, 1.0);
    glBegin (GL_POLYGON);
    {
        glVertex3f (1.0, 1.0, 1.0);
        glVertex3f (-1.0, 1.0, 1.0);
        glVertex3f (-1.0, -1.0, 1.0);
        glVertex3f (1.0, -1.0, 1.0);
    }
    glEnd ();

    glBegin (GL_POLYGON);
    {
        glVertex3f (1.0, 1.0, -1.0);
        glVertex3f (1.0, -1.0, -1.0);
        glVertex3f (-1.0, -1.0, -1.0);
        glVertex3f (-1.0, 1.0, -1.0);
    }
    glEnd ();
}

GLint Application::alloc_color(Colormap cmap, int red, int green, int blue)
{

    XColor xcolor;

    xcolor.red = red;
    xcolor.green = green;
    xcolor.blue = blue;
    xcolor.flags = DoRed | DoGreen | DoBlue;

    if (!XAllocColor (display(), cmap, &xcolor)) {
        cerr << "Couldn't allocate color!" << endl;
        exit(NULL, NULL, NULL);
    }

    return xcolor.pixel;
}

/*
 * We pass in CoreClass pointers because we need colormap()
 * which is defined in Core.
 *
 * This function shows some of the auto-cast functionality
 * of Xarm widgets.
 *
 */

void Application::translate_pixels(CoreClass *to, CoreClass *from, ...)
{
    /* This is rather inefficient, but we don't mind for the moment, */
    /*  because it works.  */

    va_list ap;
    char *name;
    Colormap from_cmap, to_cmap;
    XColor xcolor;

    from_cmap = from->colormap();
    to_cmap   = to->colormap();

    va_start (ap, from);
    for (name = va_arg (ap, char *); name != NULL; name = va_arg (ap, char *)) {
        XtVaGetValues (*from, name, &xcolor.pixel, NULL);
        XQueryColor (*from, from_cmap, &xcolor);
        if (!XAllocColor (*to, to_cmap, &xcolor))
            XtAppWarning (appContext(), "Couldn't allocate color!\n");
        else
            XtVaSetValues (from->widget(), name, xcolor.pixel, NULL);
    }
    va_end (ap);
}

void Application::exit(Widget, XtPointer, XtPointer)
{
    quit();
}

void Application::onExit(Widget, XtPointer, XtPointer)
{
    exitDiag->manage();
}

#else

int main()
{
    cout << "Sorry, you need OpenGL or Mesa to run this example." << endl;

    return 0;
}

#endif
