/*
 Copyright (C) 1999 Gerald L. Gay

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public License
 version 2 as published by the Free Software Foundation.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.

 You should have received a copy of the GNU Library General Public
 License along with this library; see the file COPYING.  If not,
 write to the Free Software Foundation, Inc., 675 Mass Ave,
 Cambridge, MA 02139, USA.
*/

/* Include(s) */
#include "setup.h"

// All the actual install code...

InstallingPanel::InstallingPanel(Widget parent, XarmArg &formArgs, char *name)
  : SetupSubpanel(parent, formArgs, name)
{
    XarmArg args;

    setTitle(" Installing Xarm 1.4.0");

    args(XmNleftAttachment,  XmATTACH_FORM)
        (XmNtopAttachment,   XmATTACH_WIDGET)
        (XmNtopWidget,       titleLabel)
        (XmNrightAttachment, XmATTACH_FORM)
        (XmNrightOffset,     5)
        (XmNleftOffset,      5)
        (XmNtopOffset,       20);

    currAction = new Label(*myForm, args, "currAction");
    currAction->labelString("Extracting the Xarm tar file...");

    args.reset();
    args(XmNbottomAttachment,  XmATTACH_FORM)
        (XmNleftAttachment,    XmATTACH_FORM)
        (XmNrightAttachment,   XmATTACH_FORM)
        (XmNleftOffset,        5)
        (XmNrightOffset,       5)
        (XmNbottomOffset,      0)
        (XmNorientation,       XmHORIZONTAL)
        (XmNtickIncrement,     1)
        (XmNshowPercentage,    True)
        (XmNshowMinMax,        False)
        (XmNminimum,           0)
        (XmNmaximum,           100)
        (XmNgaugeMargin,       0)
        (XmNheight,            30)
        (XmNshowValue,         True);

    gauge = new Gauge(*myForm, args, "progressBar");

    args.reset();
    args(XmNleftAttachment,        XmATTACH_FORM)
        (XmNrightAttachment,       XmATTACH_FORM)
        (XmNtopAttachment,         XmATTACH_WIDGET)
        (XmNtopWidget,             currAction)
        (XmNtopOffset,             20)
        (XmNleftOffset,            7)
        (XmNrightOffset,           5)
        (XmNmarginWidth,           3)
        (XmNmarginHeight,          3);

    myFrame = new Frame(*myForm, args, "monitorFrame");

    args.reset();
    args(XmNeditable,              False)
        (XmNcursorPositionVisible, False)
        (XmNeditMode,              XmMULTI_LINE_EDIT)
        (XmNwordWrap,              True)
        (XmNscrollHorizontal,      False)
        (XmNscrollVertical,        True)
        (XmNrows,                  20);

    monitor = new ScrolledText(*myFrame, args, "installMonitor");

    myFrame->manage();

    args.reset();
    args(XmNtopAttachment,      XmATTACH_WIDGET)
        (XmNtopWidget,          monitor)
        (XmNtopOffset,          10)
        (XmNleftAttachment,     XmATTACH_FORM)
        (XmNleftOffset,         60)
        (XmNrightAttachment,    XmATTACH_FORM)
        (XmNrightOffset,        60)
        (XmNbottomAttachment,   XmATTACH_WIDGET)
        (XmNbottomWidget,       gauge)
        (XmNbottomOffset,       10)
        (XmNshadowThickness,    0)
        (XmNhighlightThickness, 0);

    waitHere = new ToggleButton(*myForm, args, "waitToggle");
    waitHere->labelString("Stay on this screen when finished");

}

void InstallingPanel::setOptions(const XarmOptions &options)
{
    // I could probably let the default copy constructor
    // do this for me but I never did trust those things :-)
    myOptions.extractDir    = options.extractDir;
    myOptions.wantConfigure = options.wantConfigure;
    myOptions.wantShared    = options.wantShared;
    myOptions.wantStatic    = options.wantStatic;
    myOptions.wantCDE       = options.wantCDE;
    myOptions.wantXbae      = options.wantXbae;
    myOptions.xbaeDir       = options.xbaeDir;
    myOptions.wantGL        = options.wantGL;
    myOptions.glDir         = options.glDir;
    myOptions.wantExamples  = options.wantExamples;
    myOptions.installDir    = options.installDir;
    myOptions.otherOptions  = options.otherOptions;
    myOptions.wantCompile   = options.wantCompile;
    myOptions.wantInstall   = options.wantInstall;
}

void InstallingPanel::manage()
{
    myForm->manage();

    // Figure out how much work we have to do...

    totalProgress = XarmFileCount;

    if (myOptions.wantConfigure) {
        totalProgress += XarmConfCount;
	if (myOptions.wantCompile) {
	    if (myOptions.wantShared && myOptions.wantStatic) totalProgress += XarmMakeAllCount;
	    else if (myOptions.wantShared)                    totalProgress += XarmMakeNoStaticCount;
	    else                                              totalProgress += XarmMakeNoSharedCount;

	    if (myOptions.wantExamples) totalProgress += XarmExamplesCount;
	    if (myOptions.wantInstall)  {
	        totalProgress += XarmMakeInstallCount;
		if (myOptions.wantExamples) totalProgress += XarmMakeInstallExamples;
	    }
	}
    }

    // Set the gauge min/max
    gauge->maximum(totalProgress);
    currProgress = 0;

    // Start the state machine...
    state = 0;
    busy = false;
    ::addWorkProc(this, &InstallingPanel::installWork, theApp->appContext());
}

Boolean InstallingPanel::installWork(XtPointer)
{
    if (busy) return False;

    busy = true;

    switch (state) {
        case 0: extractTarball(); break;
        case 1: configureXarm();  break;
        case 2: compile();        break;
        case 3: makeInstall();    break;
        case 4: if (waitHere->set()) theApp->almostDone();
                else                 theApp->done();
	        gauge->unmanage();
                return True;
    }

    return False;
}

void InstallingPanel::extractTarball()
{
    struct stat statBuf;

    // Change directory to where we want to extract...

    if (chdir(myOptions.extractDir.c_str()) != 0) {
        theApp->fatal("Setup couldn't change directory to\n"
		      "the desired source destination!");
	return;
    }

    // Stat the application file...

    if (stat(setupFile.c_str(), &statBuf) != 0) {
        theApp->fatal("Setup couldn't get file information\n"
		      "for the Setup binary!");
	return;
    }

    int ifd = open(setupFile.c_str(), O_RDONLY);

    if (ifd == -1) {
        theApp->fatal("Setup couldn't extract the Xarm\n"
		      "source code!");
	return;
    }

    int ofd = creat(XarmTarballName, 0644);

    if (ofd == -1) {
      theApp->fatal("Setup couldn't create the source tar file!");
      return;
    }

    // Seek to the beginning of the tarball...

    if (lseek(ifd, statBuf.st_size - XarmTarballSize, SEEK_SET) == -1) {
        theApp->fatal("Setup couldn't extract the Xarm\n"
		      "source code!");
	return;
    }

    // Read 1K at a time...

    unsigned char buf[1024];

    size_t soFar = 0;

    while (soFar < XarmTarballSize) {
        size_t thisIn = read(ifd, buf, 1024);

	if (thisIn == -1) {
	    theApp->fatal("An error occured extracting the Xarm source code!");
	    return;
	}

	soFar += thisIn;

	size_t totalOut = 0;

	do {
	    size_t thisOut = write(ofd, buf + totalOut, thisIn);

	    if (thisOut == -1) {
	        theApp->fatal("An error occured creating the Xarm tar file!");
		return;
	    }

	    totalOut += thisOut;
	    thisIn -= thisOut;
	} while (thisIn > 0);
    }

    close(ifd);
    close(ofd);

    // The tarball has been created!
    // Let's try to extract it...
    // I don't use "tar xvfz" just on the off-chance
    // that someone doesn't have GNU tar.  They
    // really need to have gzip though!

    string command = "gzip -cd ";
    command += XarmTarballName;
    command += " | tar xvf - 2>&1";

    workFP = popen(command.c_str(), "r");

    if (workFP == NULL) {
        theApp->fatal("Setup couldn't extract the Xarm source code.\n"
		      "Make sure gzip and tar are in your path.");
	return;
    }

    // Watch the tar process...

    ::addInput(this,
	       &InstallingPanel::inputWatcher,
	       theApp->appContext(),
	       fileno(workFP),
	       (XtPointer)XtInputReadMask);
}

// States 1, 2 and 3 are configure, make, make install.
// Define error strings for each...

static char *errMessages[] = {
  "An error occured in the configure process!",
  "An error occured compiling the Xarm library!",
  "An error occured installing Xarm!\n"
  "Check permissions on the target\n"
  "directory and try again." };

void InstallingPanel::inputWatcher(XtPointer, int *, XtInputId *iid)
{

    int ch = fgetc(workFP);

    if ((ch == EOF) || feof(workFP)) {
        ::removeInput(*iid);
	if (pclose(workFP) != 0) {
	    if ((state > 0) && (state < 4))
	        theApp->fatal(errMessages[state - 1]);
	    else
	        theApp->fatal("An error occured in the child process");
	} else {
	    busy = false;
	    ++state;
	}
	return;
    }

    // Add one to the progress meter for every newline

    if (ch == '\n') {
        if (++currProgress < totalProgress)
	    gauge->value(currProgress);
    }

    char tBuf[2];

    tBuf[0] = ch;
    tBuf[1] = '\0';

    XmTextPosition tp = monitor->getLastPosition();
    monitor->insert(tp, tBuf);
}

void InstallingPanel::configureXarm()
{
    if (!myOptions.wantConfigure) {
        busy = false;
	++state;
	return;
    }

    string confCommand;

    // Build the configure command from the options...

    confCommand = "./configure";

    confCommand += " --with-shared=";
    if (myOptions.wantShared) confCommand += "yes";
    else                      confCommand += "no";

    confCommand += " --with-static=";
    if (myOptions.wantStatic) confCommand += "yes";
    else                      confCommand += "no";

    confCommand += " --with-cde=";
    if (myOptions.wantCDE) confCommand += "yes";
    else                   confCommand += "no";

    confCommand += " --with-xbae=";
    if (myOptions.wantXbae) {
        if (myOptions.xbaeDir.length() > 0) confCommand += myOptions.xbaeDir;
	else                                confCommand += "yes";
    } else {
        confCommand += "no";
    }

    confCommand += " --with-gl=";
    if (myOptions.wantGL) {
        if (myOptions.glDir.length() > 0) confCommand += myOptions.glDir;
	else                              confCommand += "yes";
    } else {
        confCommand += "no";
    }

    confCommand += " --enable-examples=";
    if (myOptions.wantExamples) confCommand += "yes";
    else                        confCommand += "no";

    if (myOptions.installDir.length() > 0) {
        confCommand += " --prefix=";
	confCommand += myOptions.installDir;
    }

    if (myOptions.otherOptions.length() > 0) {
        confCommand += " ";
	confCommand += myOptions.otherOptions;
    }

    confCommand += " 2>&1";

    // At this point we are one directory up from
    // the Xarm source directory.  Go down there!
    // We need to extract the directory name out
    // of the tar file name just in case we're 
    // still in beta. If we're in beta then the
    // name will be something like Xarm-1.4.0beta6.tar.gz

    string foo1 = XarmTarballName;

    int len = foo1.find(".tar");

    if (len <= 0) {
        theApp->fatal("An error occured parsing the Xarm file name");
	return;
    }

    string foo2 = foo1.substr(0, len);

    if (chdir(foo2.c_str()) != 0) {
        theApp->fatal("Setup couldn't change to the Xarm source directory!");
	return;
    }

    workFP = popen(confCommand.c_str(), "r");

    if (workFP == NULL) {
        theApp->fatal("Setup couldn't start the configure process");
	return;
    }

    // Put the configure command into the output
    // stream incase someone wants to see how we
    // configured the library...
    confCommand += "\n";
    XmTextPosition tp = monitor->getLastPosition();
    monitor->insert(tp, confCommand.c_str());


    // Watch configure...

    currAction->labelString("Configuring the Xarm library...");

    ::addInput(this,
	       &InstallingPanel::inputWatcher,
	       theApp->appContext(),
	       fileno(workFP),
	       (XtPointer)XtInputReadMask);

}

void InstallingPanel::compile()
{
    if (!myOptions.wantConfigure || !myOptions.wantCompile) {
        ++state;
	busy = false;
	return;
    }

    // We are already in the source directory
    // and if we got this far then the configure worked.
    // All we have to do is start the compile!

    workFP = popen("make 2>&1", "r");

    if (workFP == NULL) {
        theApp->fatal("Setup couldn't start the compile process!");
	return;
    }

    currAction->labelString("Compiling Xarm...");

    // Watch the action!

    ::addInput(this,
	       &InstallingPanel::inputWatcher,
	       theApp->appContext(),
	       fileno(workFP),
	       (XtPointer)XtInputReadMask);
}

void InstallingPanel::makeInstall()
{
    if (!myOptions.wantConfigure ||
	!myOptions.wantCompile   ||
	!myOptions.wantInstall) {
        ++state;
	busy = false;
	return;
    }

    // This one is pretty easy...

    workFP = popen("make install 2>&1", "r");

    if (workFP == NULL) {
        theApp->fatal("Setup couldn't install Xarm!\n"
		      "Check permissions on the target\n"
		      "directory and try again!");
	return;
    }

    currAction->labelString("Installing Xarm...");

    // Watch the action!

    ::addInput(this,
	       &InstallingPanel::inputWatcher,
	       theApp->appContext(),
	       fileno(workFP),
	       (XtPointer)XtInputReadMask);
}
