#ifdef __solaris__
#define _XOPEN_SOURCE
#include <math.h>
#undef _XOPEN_SOURCE
#else
#include <math.h>
#endif

#include <stdio.h>
#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <Xm/Xm.h>
#include <Xm/XmP.h>

#include <Xmext/Dial.h>
#include <Xmext/DialP.h>

#define SQR(x) ((x) * (x))

#if XmVersion > 1001
#include <Xm/DrawP.h>
#endif

/*  Resource list for Dial  */

static XtResource resources[] = 
{
   {
     XmNminimum, XmCMinimum, XmRInt, sizeof(int),
     XtOffset (XmDialWidget, dial.minimum),
     XmRImmediate, (caddr_t) 0
   },

   {
     XmNmaximum, XmCMaximum, XmRInt, sizeof(int),
     XtOffset (XmDialWidget, dial.maximum),
     XmRImmediate, (caddr_t) 100 
   },

   {
     XmNvalue, XmCValue, XmRInt, sizeof(int),
     XtOffset (XmDialWidget, dial.value),
     XmRImmediate, (caddr_t) 0
   },

   {
     XmNtickIncrement, XmCTickIncrement, XmRInt, sizeof(int),
     XtOffset (XmDialWidget, dial.tick_increment),
     XmRImmediate, (caddr_t) 0
   },

   {
     XmNtickSize, XmCTickSize, XmRInt, sizeof(int),
     XtOffset (XmDialWidget, dial.tick_size),
     XmRImmediate, (caddr_t) 3
   },

   {
     XmNarcMargin, XmCArcMargin, XmRInt, sizeof(int),
     XtOffset (XmDialWidget, dial.arc_margin),
     XmRImmediate, (caddr_t) 7
   },

   {
     XmNborderMargin, XmCBorderMargin, XmRInt, sizeof(int),
     XtOffset (XmDialWidget, dial.border_margin),
     XmRImmediate, (caddr_t) 4
   },

   {
     XmNprocessingDirection, XmCProcessingDirection,
     XmRProcessingDirection, sizeof(unsigned char),
     XtOffset (XmDialWidget, dial.processing_direction),
     XmRCallProc, (caddr_t) XmMAX_ON_RIGHT
   },

   {
     XmNfill, XmCFill, XmRBoolean, sizeof(Boolean),
     XtOffset (XmDialWidget, dial.fill),
     XmRImmediate, (caddr_t) True
   },

   {
     XmNshowValue, XmCShowValue, XmRBoolean, sizeof(Boolean),
     XtOffset (XmDialWidget, dial.show_value),
     XmRImmediate, (caddr_t) False
   },

   {
     XmNshowMinMax, XmCShowMinMax, XmRBoolean, sizeof(Boolean),
     XtOffset (XmDialWidget, dial.show_min_max),
     XmRImmediate, (caddr_t) False
   },

   {
     XmNshowHub, XmCShowHub, XmRBoolean, sizeof(Boolean),
     XtOffset (XmDialWidget, dial.show_hub),
     XmRImmediate, (caddr_t) False
   },

   {
     XmNshowPercentage, XmCShowPercentage, XmRBoolean, sizeof(Boolean),
     XtOffset (XmDialWidget, dial.show_percentage),
     XmRImmediate, (caddr_t) False
   },

   {
      XmNfontList, XmCFontList, XmRFontList, sizeof(XmFontList),
      XtOffset(XmDialWidget, dial.fontlist),
      XmRString, (caddr_t) "fixed"
   },

   {
     XmNstyle, XmCStyle, XmRDialStyle, sizeof(DialStyle),
     XtOffset (XmDialWidget, dial.style),
     XmRImmediate, (caddr_t) MeterStyle
   }
};


/*  Static routine definitions  */

/*  Setup  */
static void    Initialize          (XmDialWidget request, XmDialWidget new);
static void    ClassPartInitialize (WidgetClass  wc);
static void    GetDialGC           (XmDialWidget dialw);
static void    LoadFontMetrics     (XmDialWidget dialw);
static void    Destroy             (XmDialWidget dialw);

/* Type converter for enumerated style type */
static Boolean CvtStringToStyle(Display *display, XrmValuePtr args, Cardinal *num_args, 
                                XrmValuePtr from, XrmValuePtr to, XtPointer *data);

/* Events */
static void    Resize    (XmDialWidget dialw);
static void    Redisplay (XmDialWidget dialw, XEvent *event, Region region);
static Boolean SetValues (XmDialWidget current, XmDialWidget request, XmDialWidget new);

/* Drawing functions */
static void    DrawDial(XmDialWidget dialw);
static void    SetDial(XmDialWidget dialw);

static void    FillArc(Display *display, Window window, GC gc, int x, int y, 
                       int xradius, int yradius, int start_angle, int angle);
static void    DrawMeter(Display *display, Window window, GC gc, int x, int y, 
                         int xradius, int yradius, int start_angle, int angle);
static void    FillMeter(Display *display, Window window, GC gc, int x, int y, 
                         int xradius, int yradius, int xinner_radius, int yinner_radius, int start_angle, int angle);
static void    DrawPointer(Display *display, Window window, XmDialWidget dialw, int start_angle, int angle);
static void    DrawTickMarks(Display *display, Window window, XmDialWidget dialw);
static void    DrawHub(Display *display, Window window, XmDialWidget dialw);

/*  The Dial class record definition  */

externaldef (xmdialclassrec) XmDialClassRec xmDialClassRec=
{
   {
      (WidgetClass) &xmPrimitiveClassRec, /* superclass                   */    
      "XmDial",                           /* class_name                   */    
      sizeof(XmDialRec),                  /* widget_size                  */    
      NULL,                               /* class_initialize             */    
      ClassPartInitialize,                /* class_part_initialize        */
      FALSE,                              /* class_inited                 */    
      (XtInitProc) Initialize,            /* initialize                   */    
      NULL,                               /* initialize_hook              */
      (XtRealizeProc) _XtInherit,         /* realize                      */    
      NULL,                               /* actions                      */    
      0,                                  /* num_actions                  */    
      resources,                          /* resources                    */    
      XtNumber(resources),                /* num_resources                */    
      NULLQUARK,                          /* xrm_class                    */    
      TRUE,                               /* compress_motion              */    
      XtExposeCompressMaximal,            /* compress_exposure            */    
      TRUE,                               /* compress_enterleave          */
      FALSE,                              /* visible_interest             */    
      (XtWidgetProc) Destroy,             /* destroy                      */    
      (XtWidgetProc) Resize,              /* resize                       */
      (XtExposeProc) Redisplay,           /* expose                       */    
      (XtSetValuesFunc) SetValues,        /* set_values                   */    
      NULL,                               /* set_values_hook              */
      XtInheritSetValuesAlmost,           /* set_values_almost            */
      NULL,                               /* get_values_hook              */
      NULL,                               /* accept_focus                 */    
      XtVersion,                          /* version                      */
      NULL,                               /* callback private             */
      NULL,                               /* tm_table                     */
      NULL,                               /* query_geometry               */
      NULL,                               /* display_accelerator          */
      NULL,                               /* extension                    */
   },

   {
      (XtWidgetProc)_XtInherit,           /* Primitive border_highlight   */
      (XtWidgetProc)_XtInherit,           /* Primitive border_unhighlight */
      NULL,                               /* translations                 */
      NULL,                               /* arm_and_activate             */
      NULL,                               /* get resources                */
      0,                                  /* num get_resources            */
      NULL,                               /* extension                    */
   }

};

externaldef(xmdialwidgetclass) WidgetClass xmDialWidgetClass =
              (WidgetClass) &xmDialClassRec;


/************************************************************************
 *
 *  ClassPartInitialize
 *     Set up the fast subclassing for the widget
 *
 ************************************************************************/
static void ClassPartInitialize (WidgetClass wc)
{
    XtSetTypeConverter(XmRString, XmRDialStyle,    CvtStringToStyle,
        NULL, 0, XtCacheNone, NULL);
}

      
/************************************************************************
 *
 *  Initialize
 *     The main widget instance initialization routine.
 *
 ************************************************************************/
static void Initialize (XmDialWidget request, XmDialWidget new)
{
   /*
    *  Check the data put into the new widget from .Xdefaults
    *  or through the arg list.
    */
    if (new->dial.style != MeterStyle   &&
        new->dial.style != PointerStyle  ) {
       _XmWarning ((Widget)new, "Invalid style");
       new->dial.style = MeterStyle;
    }

    if (new->dial.minimum >= new->dial.maximum) {
        _XmWarning ((Widget)new, "XmNminimum is greater than XmNmaximum");
        new->dial.minimum = 0;
    }

    if (new->dial.minimum >= new->dial.maximum) {
        _XmWarning ((Widget)new, "XmNmaximum is less than XmNminimum");
        new->dial.maximum = 100;
    }

    if (new->dial.value < new->dial.minimum) {
        _XmWarning ((Widget)new, "XmNvalue is less than XmNminimum");
        new->dial.value = new->dial.minimum;
    }

    if (new->dial.value > new->dial.maximum) {
        _XmWarning ((Widget)new, "XmNvalue is greater than XmNmaximum");
        new->dial.value = new->dial.maximum;
    }

    if (new->dial.tick_increment >  new->dial.maximum - new->dial.minimum) {
        _XmWarning ((Widget)new, "XmNtickIncremnt is greater than Dial range");    
        new->dial.tick_increment = 0;
    }

    /*  Set up a geometry for the widget if it is currently 0.  */

    if (request->core.width == 0) 
        new->core.width = 70;
    if (request->core.height == 0) 
        new->core.height = 50;

    new->dial.fontlist = XmFontListCopy(new->dial.fontlist);
    LoadFontMetrics(new);

    /*  Set the internal dial variables and call resize to generate  */
    /*  the dial drawing variable set.                               */
    Resize(new);

    /*  Get the drawing graphics contexts.  */
    GetDialGC (new);
}


/************************************************************************
 *
 *  GetDialGC
 *     Get the graphics context used for drawing the dial.
 *
 ************************************************************************/
static void GetDialGC(XmDialWidget dialw)
{
    XGCValues values;
    XtGCMask  valueMask = GCForeground | GCBackground | GCFillStyle | GCFont;

    values.foreground = dialw->primitive.foreground;
    values.background = dialw->core.background_pixel;
    values.fill_style = FillSolid;
    values.font = dialw->dial.font->fid;
    dialw->dial.foreground_GC = XtGetGC((Widget) dialw, valueMask, &values);

    values.foreground = dialw->core.background_pixel;
    values.background = dialw->primitive.foreground;
    dialw->dial.background_GC = XtGetGC((Widget) dialw, valueMask, &values);

    values.foreground = dialw->core.background_pixel;
    values.background = dialw->core.background_pixel;
    dialw->dial.text_erase_GC = XtGetGC((Widget) dialw, valueMask, &values);


}


/************************************************************************
 *
 *  Redisplay
 *     General redisplay function called on exposure events.
 *
 ************************************************************************/
static void Redisplay (XmDialWidget dialw, XEvent *event, Region region)
{
    int prev;
    
    DrawDial(dialw);
    prev = dialw->dial.prev_value;
    dialw->dial.prev_value = dialw->dial.minimum;
    SetDial(dialw);
    dialw->dial.prev_value = prev;

    if (dialw->primitive.shadow_thickness > 0)
#if XmVersion == 1001
       _XmDrawShadow(XtDisplay((Widget)dialw), XtWindow((Widget)dialw), 
                dialw->primitive.bottom_shadow_GC,
                dialw->primitive.top_shadow_GC,
                dialw->primitive.shadow_thickness,
                dialw->primitive.highlight_thickness,
                dialw->primitive.highlight_thickness,
                dialw->core.width-2 * dialw->primitive.highlight_thickness,
                dialw->core.height-2 * dialw->primitive.highlight_thickness);
#else
       _XmDrawShadows(XtDisplay((Widget)dialw), XtWindow((Widget)dialw), 
                dialw->primitive.top_shadow_GC,
                dialw->primitive.bottom_shadow_GC,
                dialw->primitive.highlight_thickness,
                dialw->primitive.highlight_thickness,
                dialw->core.width-2 * dialw->primitive.highlight_thickness,
                dialw->core.height-2 * dialw->primitive.highlight_thickness,
                dialw->primitive.shadow_thickness,
                XmSHADOW_IN);
#endif

}

/************************************************************************
 *
 *  Resize
 *    Calculate the drawing rectangles and draw rectangles.
 *
 ************************************************************************/
static void Resize(XmDialWidget dialw)
{
    dialw->dial.border_offset = dialw->dial.border_margin +
                                dialw->primitive.highlight_thickness +
                                dialw->primitive.shadow_thickness;
 
    dialw->dial.arc_offset = dialw->dial.border_offset + 
                             dialw->dial.arc_margin;

    if (dialw->dial.show_min_max || dialw->dial.show_value)
        dialw->dial.text_height = dialw->dial.font->ascent + 
                                  dialw->dial.font->descent;
    else
        dialw->dial.text_height = 0;

    dialw->dial.xorigin = dialw->core.width / 2;
    dialw->dial.yorigin = (dialw->core.height - 
                          dialw->dial.arc_offset) - 
                          dialw->dial.text_height;
    dialw->dial.xradius = dialw->dial.xorigin - 
                          dialw->dial.arc_offset;
    dialw->dial.yradius = dialw->dial.yorigin - 
                          dialw->dial.arc_offset;
    dialw->dial.xinner_radius = dialw->dial.xradius * 0.20;
    dialw->dial.yinner_radius = dialw->dial.yradius * 0.20;

}

/************************************************************************
 *
 *  Destroy
 *    Clean up allocated resources when the widget is destroyed.
 *
 ************************************************************************/
static void Destroy(XmDialWidget dialw)
{
    XmFontListFree(dialw->dial.fontlist);
    XtReleaseGC((Widget) dialw, dialw->dial.foreground_GC);
    XtReleaseGC((Widget) dialw, dialw->dial.background_GC);
    XtReleaseGC((Widget) dialw, dialw->dial.text_erase_GC);
}

/************************************************************************
 *
 *  SetValues
 *
 ************************************************************************/
/* ARGSUSED */
static Boolean SetValues (XmDialWidget current, XmDialWidget request, XmDialWidget new)
{
    Boolean redrawFlag = False;

    /*  Check the data put into the new widget.  */
    if (new->dial.style != MeterStyle   &&
        new->dial.style != PointerStyle )
    {
        _XmWarning ((Widget)new, "Invalid style");
        new->dial.style = MeterStyle;
    }

    if (new->dial.minimum >= new->dial.maximum) {
          _XmWarning ((Widget)new, "XmNminimum is greater than XmNmaximum");
        new->dial.minimum = 0;
    }

    if (new->dial.minimum >= new->dial.maximum) {
        new->dial.maximum = 100;
    }

    if (new->dial.value < new->dial.minimum) {
          _XmWarning ((Widget)new, "XmNvalue is less than XmNminimum");
        new->dial.value = new->dial.minimum;
    }

    if (new->dial.value > new->dial.maximum) {
          _XmWarning ((Widget)new, "XmNvalue is greater than XmNmaximum");
        new->dial.value = new->dial.maximum;
    }

    if (new->dial.tick_increment >  new->dial.maximum - new->dial.minimum) {
        _XmWarning ((Widget)new, "XmNtickIncrement is greater than Dial range");    
        new->dial.tick_increment = 0;
    }

    if ((new->dial.processing_direction != XmMAX_ON_RIGHT) &&
        (new->dial.processing_direction != XmMAX_ON_LEFT)) {
        new->dial.processing_direction = XmMAX_ON_RIGHT;
        _XmWarning ((Widget)new, "Invalid Processing Direction");
    }

    /*  See if the GC's need to be regenerated and widget redrawn.  */

    if (new->core.background_pixel != current->core.background_pixel ||
        new->primitive.foreground != current->primitive.foreground   ||
        new->dial.fontlist != current->dial.fontlist) {
        if (new->dial.fontlist != current->dial.fontlist) {
            XmFontListFree(current->dial.fontlist);
            new->dial.fontlist = XmFontListCopy(new->dial.fontlist);
           LoadFontMetrics(new);
        }

        XtReleaseGC((Widget)new, new->dial.foreground_GC);
        XtReleaseGC((Widget)new, new->dial.background_GC);
        GetDialGC(new);
        redrawFlag = True;
    }

    if (new->dial.style                    != current->dial.style                    ||
        new->dial.minimum                  != current->dial.minimum                  ||
        new->dial.maximum                  != current->dial.maximum                  ||
        new->dial.tick_increment           != current->dial.tick_increment           ||
        new->dial.tick_size                != current->dial.tick_size                ||
        new->dial.fill                     != current->dial.fill                     ||
        new->dial.show_value               != current->dial.show_value               ||
        new->dial.show_min_max             != current->dial.show_min_max             ||
        new->dial.show_percentage          != current->dial.show_percentage          ||
        new->dial.show_hub                 != current->dial.show_hub                 ||
        new->dial.arc_margin               != current->dial.arc_margin               ||
        new->dial.border_margin            != current->dial.border_margin            ||
        new->dial.processing_direction     != current->dial.processing_direction     ||
        new->primitive.highlight_thickness != current->primitive.highlight_thickness ||
        new->primitive.shadow_thickness    != current->primitive.shadow_thickness)   {
        redrawFlag = True;
    }
    else if (new->dial.value != current->dial.value && !redrawFlag) {
        new->dial.prev_value = current->dial.value;
        SetDial(new);
    }
    return redrawFlag;
}

/************************************************************************
 *
 *  XmCreateDial
 *    Create an instance of an dial and return the widget id.
 *
 ************************************************************************/
Widget XmCreateDial(Widget parent, char *name, ArgList arglist, Cardinal argcount)
{
    return (XtCreateWidget(name, xmDialWidgetClass, parent, arglist, argcount));
}

/************************************************************************
 *
 *  LoadFontMetrics
 *    Load the font structure in the widget.    
 *
 ************************************************************************/
static void LoadFontMetrics (XmDialWidget dialw)
{
    XmFontContext    context;
    XmStringCharSet  charset;
    XFontStruct     *font;

    if (!XmFontListInitFontContext(&context, dialw->dial.fontlist))
        _XmWarning((Widget) dialw, "XmFontListInitFontContext Failed.");

    if (!XmFontListGetNextFont(context, &charset, &font))
        _XmWarning((Widget) dialw, "XmFontListGetNextFont Failed.");

    dialw->dial.font = font;
    XtFree(charset);
    XmFontListFreeFontContext(context);
}

/************************************************************************
 *
 *  DrawDial
 *    Draw the dial widget
 *
 ************************************************************************/
static void DrawDial(XmDialWidget dialw)
{
    char      left_string [64];
    char      right_string[64];
    char      value_string[64];
    int       text_width;
    int       pct;
    int       nsegments;
    Display  *display = XtDisplay((Widget) dialw);
    Window    window  = XtWindow((Widget) dialw);

    if (dialw->dial.show_min_max) {
        sprintf(left_string, "%d", (dialw->dial.processing_direction == XmMAX_ON_RIGHT) ?
                dialw->dial.minimum : dialw->dial.maximum);
        sprintf(right_string, "%d", (dialw->dial.processing_direction == XmMAX_ON_RIGHT) ?
                dialw->dial.maximum : dialw->dial.minimum);
        text_width  = XTextWidth(dialw->dial.font,right_string,strlen(right_string));
    }
    else
        text_width = 0;

    XClearWindow(display, window);

    XDrawRectangle(display, window, 
                   dialw->dial.foreground_GC,
                   dialw->dial.border_offset, 
                   dialw->dial.border_offset, 
                   (dialw->core.width - dialw->dial.border_offset * 2), 
                   (dialw->core.height - dialw->dial.border_offset * 2));

    DrawMeter(display, window, dialw->dial.foreground_GC,
              dialw->dial.xorigin, dialw->dial.yorigin,
              dialw->dial.xradius, dialw->dial.yradius, 0, 180);

    XDrawLine(display, window, dialw->dial.foreground_GC,
              dialw->dial.arc_offset, dialw->dial.yorigin, 
              dialw->core.width - dialw->dial.arc_offset, 
              dialw->dial.yorigin);

    if (dialw->dial.show_min_max) {
        XDrawString(display, window, dialw->dial.foreground_GC,
                    dialw->dial.arc_offset, 
                    dialw->core.height - dialw->dial.arc_offset, 
                    left_string, strlen(left_string));

        XDrawString(display, window, dialw->dial.foreground_GC,
                    dialw->core.width - text_width - dialw->dial.arc_offset, 
                    dialw->core.height - dialw->dial.arc_offset,
                    right_string, strlen(right_string));
    }

    if (dialw->dial.show_value) {
        if (dialw->dial.show_percentage) {
            pct = (int)(((double)(dialw->dial.value - dialw->dial.minimum) / 
                         (double)(dialw->dial.maximum - (double)dialw->dial.minimum)) * 100.0);
            sprintf(value_string, "%d%%", pct);
        }
        else
            sprintf(value_string, "%d", dialw->dial.value);

        text_width = XTextWidth(dialw->dial.font, 
                                value_string, strlen(value_string));

        XDrawImageString(display, window, dialw->dial.foreground_GC,
                         dialw->core.width / 2 - text_width / 2, 
                         dialw->core.height - dialw->dial.arc_offset,
                         value_string, strlen(value_string));
    }

    if (dialw->dial.tick_increment) 
        DrawTickMarks(display, window, dialw);

    XFlush(display); 
} 

/************************************************************************
 *
 *  SetDial
 *    Fill the dial widget
 *
 ************************************************************************/
static void SetDial(XmDialWidget dialw)
{
    char      value_string[256];
    int       pct;
    int       start;
    int       arc;
    int       cur_arc;
    int       prev_arc;
    int       text_width;
    Display  *display    = XtDisplay((Widget) dialw);
    Window    window     = XtWindow((Widget) dialw);
    GC        gc;

    if (dialw->dial.show_value) {
        /*
        ** Erase the old one
        */
        if (dialw->dial.show_percentage) {
            pct = (int)(((double)(dialw->dial.prev_value - dialw->dial.minimum) / 
                         (double)(dialw->dial.maximum - dialw->dial.minimum)) * 100.0);
            sprintf(value_string, "%d%%", pct);
        }
        else
            sprintf(value_string, "%d", dialw->dial.prev_value);

        text_width = XTextWidth(dialw->dial.font, 
                                value_string, strlen(value_string));

        XDrawImageString(display, window,
                         dialw->dial.text_erase_GC,
                         dialw->core.width / 2 - text_width / 2, 
                         dialw->core.height - dialw->dial.arc_offset,
                         value_string,
                         strlen(value_string));
        /*
        ** Now the new one
        */ 
        if (dialw->dial.show_percentage) {
            pct = (int)(((double)(dialw->dial.value - dialw->dial.minimum) / 
                         (double)(dialw->dial.maximum - dialw->dial.minimum)) * 100.0);
            sprintf(value_string, "%d%%", pct);
        }
        else
            sprintf(value_string, "%d", dialw->dial.value);

        text_width = XTextWidth(dialw->dial.font, 
                                value_string, strlen(value_string));

        XDrawImageString(display, window,
                         dialw->dial.foreground_GC,
                         dialw->core.width / 2 - text_width / 2, 
                         dialw->core.height - dialw->dial.arc_offset,
                         value_string,
                         strlen(value_string));
    }

    if (dialw->dial.processing_direction == XmMAX_ON_RIGHT) {
        cur_arc  = 180.0 * (double)((double)(dialw->dial.value - dialw->dial.minimum) / 
                           (double)(dialw->dial.maximum - dialw->dial.minimum));
        prev_arc = 180.0 * (double)((double)(dialw->dial.prev_value - dialw->dial.minimum) / 
                           (double)(dialw->dial.maximum - dialw->dial.minimum));
    }
    else {
        cur_arc  = 180.0 - (180.0 * (double)((double)(dialw->dial.value - dialw->dial.minimum) / 
                           (double)(dialw->dial.maximum - dialw->dial.minimum)));
        prev_arc = 180.0 - (180.0 * (double)((double)(dialw->dial.prev_value - dialw->dial.minimum)/ 
                           (double)(dialw->dial.maximum - dialw->dial.minimum)));
    }
    arc = -(cur_arc - prev_arc);
    start = 180 - prev_arc;

    switch (dialw->dial.style) { 
        case PointerStyle: 
            DrawPointer(display, window, dialw, start, arc);
            break;
        case MeterStyle:
            gc = dialw->dial.prev_value < dialw->dial.value ? 
                 dialw->dial.foreground_GC : dialw->dial.background_GC; 

            if (dialw->dial.show_hub) {
                FillMeter(display, window, gc, 
                          dialw->dial.xorigin, dialw->dial.yorigin,
                          dialw->dial.xradius - 4, dialw->dial.yradius - 5,
                          dialw->dial.xinner_radius - 1, dialw->dial.yinner_radius - 1,
                          start, arc);
                DrawHub(display, window, dialw);
            }
            else 
                FillArc(display, window, gc, 
                        dialw->dial.xorigin, dialw->dial.yorigin,
                        dialw->dial.xradius - 3, dialw->dial.yradius - 4,
                        start * 64, arc * 64);
            
            break;
        default:
            break;
    }
    if (dialw->dial.tick_increment) 
        DrawTickMarks(display, window, dialw);

    XFlush(display);
}


static void FillArc (Display *display, Window window, GC gc, int x, int y, 
                     int xradius, int yradius, int start_angle, int angle)
{
    XFillArc(display, window, gc, x - xradius, y - yradius, 
             xradius * 2, yradius * 2, start_angle, angle);
}


static void DrawMeter (Display *display, Window window, GC gc, int x, int y, 
                       int xradius, int yradius, int start_angle, int angle)
{
    int i;
    static XPoint *points = NULL;
    double curr;
    double start;
    double end;
    double incr;
    double sin_angle;
    double cos_angle;
    double fx = x;
    double fy = y;
    double fxradius = xradius;
    double fyradius = yradius;

    if (!points)
       points = (XPoint *) XtMalloc(sizeof(XPoint) * 500);

    incr = M_PI / 180.0;
    start = start_angle * incr;
    end = start + angle * incr;

    incr /= 2;

    for (i = 0, curr = start; curr < end; curr += incr, i++)
    {
        sin_angle = sin(curr + PD2);
        cos_angle = cos(curr + PD2);
        points[i].x = fxradius * sin_angle + fx;
        points[i].y = fyradius * cos_angle + fy;
    }

    sin_angle = sin(end + PD2);
    cos_angle = cos(end + PD2);
    points[i].x = fxradius * sin_angle + fx;
    points[i].y = fyradius * cos_angle + fy;
    i++;

    XDrawLines(display, window, gc, points, i, CoordModeOrigin);

}
    
static void FillMeter (Display *display, Window window, GC gc, int x, int y, 
                       int xradius, int yradius, int xinner_radius, int yinner_radius, 
                       int start_angle, int angle)
{
    int i;
    int j;
    static XPoint *points = NULL;
    static XPoint *points2 = NULL;
    double curr;
    double start;
    double end;
    double incr;
    double sin_angle;
    double cos_angle;

    if (!points)
    {
        points = (XPoint *) XtMalloc(sizeof(XPoint) * 380);
        points2 = (XPoint *) XtMalloc(sizeof(XPoint) * 380);
    }

    incr = M_PI / 180.0;
    if (angle < 0) {
        end = start_angle * incr;
        start = end + angle * incr;
    } 
    else {
        start = start_angle * incr;
        end = start + angle * incr;
    }
    
    for (i = 0, curr = start; curr < end; curr += incr, i++) {
        sin_angle = sin(curr + PD2);
        cos_angle = cos(curr + PD2);
        points[i].x = xradius * sin_angle + x;
        points[i].y = yradius * cos_angle + y;
        if (xinner_radius * yinner_radius) {
          points2[i].x = xinner_radius * sin_angle + x;
          points2[i].y = yinner_radius * cos_angle + y;
        }
    }       

    sin_angle = sin(end + PD2);
    cos_angle = cos(end + PD2);
    points[i].x = xradius * sin_angle + x;
    points[i].y = yradius * cos_angle + y;
    if (xinner_radius * yinner_radius) {
      points2[i].x = xinner_radius * sin_angle + x;
      points2[i].y = yinner_radius * cos_angle + y;
    }
    i++;

    if (xinner_radius * yinner_radius) {
      for (j = i - 1; j >= 0; i++, j--)
          points[i] = points2[j];
    }
    XFillPolygon(display, window, gc, points, i, Nonconvex,CoordModeOrigin);
    
}

static void DrawPointer (Display *display, Window window, XmDialWidget dialw, 
                         int start_angle, int angle)
{
    XPoint tri[3];
    int x2;
    int y2;
    int xpnt;
    int ypnt;
    int lxpnt;
    int lypnt;
    int rxpnt;
    int rypnt;
    double old;
    double new;
    double incr;
    double len;
    double hyp;
    double adj;
    double fx = (double)dialw->dial.xorigin;
    double fy = (double)(dialw->dial.yorigin - 1);
    double fxradius = (double)(dialw->dial.xradius - 3);
    double fyradius = (double)(dialw->dial.yradius - 4);
    double pntxradius = fxradius / 5.0;
    double pntyradius = fyradius / 5.0;
    XRectangle rect[] = {0, 0, dialw->core.width, dialw->dial.yorigin};

    XSetClipRectangles(XtDisplay((Widget)(dialw)), dialw->dial.foreground_GC,
                       0, 0, rect, 1, Unsorted);
    XSetClipRectangles(XtDisplay((Widget)(dialw)), dialw->dial.background_GC,
                       0, 0, rect, 1, Unsorted);
    XSetClipRectangles(XtDisplay((Widget)(dialw)), dialw->primitive.top_shadow_GC,
                       0, 0, rect, 1, Unsorted);

    incr = M_PI / 180.0;
    old = start_angle * incr;
    new = old + angle * incr;

    /* Erase the old one */
    x2 = fxradius * sin(old + PD2) + fx;
    y2 = fyradius * cos(old + PD2) + fy;

    xpnt = pntxradius * sin(old + PD2) + fx;
    ypnt = pntyradius * cos(old + PD2) + fy;
    len = sqrt(SQR((int)fx - xpnt) + SQR((int)fy - ypnt)); 
    hyp = sqrt(SQR(len) + SQR(len/2));
    adj = asin(len/hyp);

    rxpnt = (len/2) * sin(old + adj + PD2) + fx;
    rypnt = (len/2) * cos(old + adj + PD2) + fy;

    lxpnt = (len/2) * sin(old - adj + PD2) + fx;
    lypnt = (len/2) * cos(old - adj + PD2) + fy;
    
    if (dialw->dial.fill) {
        if (!dialw->dial.show_hub) {
            tri[0].x = (int)fx;   tri[0].y = (int)fy;
            tri[1].x = lxpnt;     tri[1].y = lypnt;
            tri[2].x = xpnt;      tri[2].y = ypnt;
            XFillPolygon(display, window, dialw->dial.background_GC, tri, 3, Convex,CoordModeOrigin);

            tri[0].x = (int)fx;   tri[0].y = (int)fy;
            tri[1].x = xpnt;      tri[1].y = ypnt;
            tri[2].x = rxpnt;     tri[2].y = rypnt;
            XFillPolygon(display, window, dialw->dial.background_GC, tri, 3, Convex,CoordModeOrigin);
        }
        tri[0].x = xpnt;      tri[0].y = ypnt;
        tri[1].x = lxpnt;     tri[1].y = lypnt;
        tri[2].x = x2;        tri[2].y = y2;
        XFillPolygon(display, window, dialw->dial.background_GC, tri, 3, Convex,CoordModeOrigin);

        tri[0].x = xpnt;      tri[0].y = ypnt;
        tri[1].x = x2;        tri[1].y = y2;
        tri[2].x = rxpnt;     tri[2].y = rypnt;
        XFillPolygon(display, window, dialw->dial.background_GC, tri, 3, Convex,CoordModeOrigin);
    }    
    else {
        XDrawLine(display, window, dialw->dial.background_GC, (int)fx, (int)fy, x2, y2);
        XDrawLine(display, window, dialw->dial.background_GC, lxpnt, lypnt, xpnt, ypnt);
        XDrawLine(display, window, dialw->dial.background_GC, xpnt, ypnt, rxpnt, rypnt);
        XDrawLine(display, window, dialw->dial.background_GC, (int)fx, (int)fy, rxpnt, rypnt);
        XDrawLine(display, window, dialw->dial.background_GC, (int)fx, (int)fy, lxpnt, lypnt);
        XDrawLine(display, window, dialw->dial.background_GC, lxpnt, lypnt, x2, y2);
        XDrawLine(display, window, dialw->dial.background_GC, rxpnt, rypnt, x2, y2);
    }

    /* Draw the new one */
    x2 = fxradius * sin(new + PD2) + fx;
    y2 = fyradius * cos(new + PD2) + fy;

    xpnt = pntxradius * sin(new + PD2) + fx;
    ypnt = pntyradius * cos(new + PD2) + fy;
    len = sqrt(SQR((int)fx - xpnt) + SQR((int)fy - ypnt)); 
    hyp = sqrt(SQR(len) + SQR(len/2));
    adj = asin(len/hyp);

    rxpnt = (len/2) * sin(new + adj + PD2) + fx;
    rypnt = (len/2) * cos(new + adj + PD2) + fy;

    lxpnt = (len/2) * sin(new - adj + PD2) + fx;
    lypnt = (len/2) * cos(new - adj + PD2) + fy;
    
    if (dialw->dial.fill) {
        if (!dialw->dial.show_hub) {
            tri[0].x = (int)fx;   tri[0].y = (int)fy;
            tri[1].x = lxpnt;     tri[1].y = lypnt;
            tri[2].x = xpnt;      tri[2].y = ypnt;
            XFillPolygon(display, window, dialw->dial.foreground_GC, tri, 3, Convex,CoordModeOrigin);

            tri[0].x = (int)fx;   tri[0].y = (int)fy;
            tri[1].x = xpnt;      tri[1].y = ypnt;
            tri[2].x = rxpnt;     tri[2].y = rypnt;
            XFillPolygon(display, window, dialw->primitive.top_shadow_GC, tri, 3, Convex,CoordModeOrigin);
        }
        tri[0].x = xpnt;      tri[0].y = ypnt;
        tri[1].x = lxpnt;     tri[1].y = lypnt;
        tri[2].x = x2;        tri[2].y = y2;
        XFillPolygon(display, window, dialw->primitive.top_shadow_GC, tri, 3, Convex,CoordModeOrigin);

        tri[0].x = xpnt;      tri[0].y = ypnt;
        tri[1].x = x2;        tri[1].y = y2;
        tri[2].x = rxpnt;     tri[2].y = rypnt;
        XFillPolygon(display, window, dialw->dial.foreground_GC, tri, 3, Convex,CoordModeOrigin);
    }    
    else {
        XDrawLine(display, window, dialw->dial.foreground_GC, (int)fx, (int)fy, x2, y2);
        XDrawLine(display, window, dialw->dial.foreground_GC, lxpnt, lypnt, xpnt, ypnt);
        XDrawLine(display, window, dialw->dial.foreground_GC, xpnt, ypnt, rxpnt, rypnt);
        XDrawLine(display, window, dialw->dial.foreground_GC, (int)fx, (int)fy, rxpnt, rypnt);
        XDrawLine(display, window, dialw->dial.foreground_GC, (int)fx, (int)fy, lxpnt, lypnt);
        XDrawLine(display, window, dialw->dial.foreground_GC, lxpnt, lypnt, x2, y2);
        XDrawLine(display, window, dialw->dial.foreground_GC, rxpnt, rypnt, x2, y2);
    }

    rect[0].height = dialw->core.height;

    XSetClipRectangles(XtDisplay((Widget)(dialw)), dialw->dial.foreground_GC,
                       0, 0, rect, 1, Unsorted);
    XSetClipRectangles(XtDisplay((Widget)(dialw)), dialw->dial.background_GC,
                       0, 0, rect, 1, Unsorted);
    XSetClipRectangles(XtDisplay((Widget)(dialw)), dialw->primitive.top_shadow_GC,
                       0, 0, rect, 1, Unsorted);

    if (dialw->dial.show_hub) 
        DrawHub(display, window, dialw);
}

static Boolean CvtStringToStyle (Display *display, XrmValuePtr args, Cardinal *num_args, 
                                 XrmValuePtr from, XrmValuePtr to, XtPointer *data)
{
    static DialStyle retval;
    Boolean convertError = False;

    if (*num_args != 0) {
        XtAppWarningMsg(XtDisplayToApplicationContext(display),
           "cvtStringToStyle", "wrongParameters",
           "XtToolkitError",
           "String to Style converter needs no extra arguments",
           (String *)NULL, (Cardinal *)NULL);
    }

    if (strcasecmp(from->addr, "meter") == 0)
        retval = MeterStyle;
    else if (strcasecmp(from->addr, "pointer") == 0)
        retval = PointerStyle;
    else convertError = True;

    if (convertError) {
        XtDisplayStringConversionWarning(display, (char *)from->addr, "Style");
    } 
    else if (to->addr == NULL) {
        to->addr = (XtPointer) &retval;
        to->size = sizeof(DialStyle);
    } 
    else if (to->size < sizeof(DialStyle)) {
        to->size = sizeof(DialStyle);
        convertError = True;
    } 
    else {
        *(DialStyle *)to->addr = retval;
        to->size = sizeof(DialStyle);
    }

    return !convertError;
}

static void DrawTickMarks(Display *display, Window window, XmDialWidget dialw)
{
    int       i;
    int       nsegments;
    double    angle;
    double    increment;
    XSegment *segments;

    nsegments = ((dialw->dial.maximum - dialw->dial.minimum) / dialw->dial.tick_increment) + 1;
    segments = (XSegment *) XtMalloc(sizeof(XSegment) * nsegments);

    increment = M_PI / (nsegments - 1);
    for (i = 0, angle = 0.0; angle <= M_PI; angle += increment, i++) {
        segments[i].x1 = (dialw->dial.xradius) * sin(angle + PD2) + dialw->dial.xorigin;
        segments[i].y1 = (dialw->dial.yradius) * cos(angle + PD2) + dialw->dial.yorigin;
        segments[i].x2 = (dialw->dial.xradius - dialw->dial.tick_size) * sin(angle + PD2) + dialw->dial.xorigin;
        segments[i].y2 = (dialw->dial.yradius - dialw->dial.tick_size) * cos(angle + PD2) + dialw->dial.yorigin;
    }
    XDrawSegments(display, window, dialw->dial.foreground_GC, segments, i);
    XtFree((char *)segments);
}

static void DrawHub(Display *display, Window window, XmDialWidget dialw)
{
    if (dialw->dial.fill) 
        FillMeter(display, window, dialw->dial.foreground_GC, 
                  dialw->dial.xorigin - 1, dialw->dial.yorigin,
                  dialw->dial.xinner_radius + 1, dialw->dial.yinner_radius + 2,
                  0, 0, 0, 180);
            
    
    else
        DrawMeter(display, window, dialw->dial.foreground_GC, 
                  dialw->dial.xorigin, dialw->dial.yorigin, 
                  dialw->dial.xinner_radius, dialw->dial.yinner_radius, 0, 180);
}

