/*
 Copyright (C) 1998 Gerald L. Gay
 
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public License 
 version 2 as published by the Free Software Foundation.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.

 You should have received a copy of the GNU Library General Public
 License along with this library; see the file COPYING.  If not,
 write to the Free Software Foundation, Inc., 675 Mass Ave, 
 Cambridge, MA 02139, USA.
*/

/* Include(s) */
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <Xarm/Label.h>
#include <Xarm/PushB.h>
#include <Xarm/MessageB.h>
#include <Xarm/DragDrop.h>
#include <Xarm/Callback.h>
#include <Xarm/Xarm.h>

#ifdef XARM_HAS_CDE

#include "xdnd.h"

#include "EmptyTrash.xpm"
#include "FullTrash.xpm"
#include "bill.xpm"
#include "bill.bm"
#include "bill_m.bm"

/* Declaration(s) */

#if !defined(VERSION)
#define VERSION 1
#endif
#if !defined(PATCHLEVEL)
#define PATCHLEVEL 0
#endif
#if !defined(SUBLEVEL)
#define SUBLEVEL 0
#endif

#if !defined(XTAB_VERSION)
#define makestring(str) #str
#define xmakestring(str) makestring(str)
#define XDND_VERSION xmakestring(VERSION) "." xmakestring(PATCHLEVEL) "." xmakestring(SUBLEVEL)
#endif
#if !defined(XDND_MODULE_NAME)
#define XDND_MODULE_NAME "xdnd"
#endif
#if !defined(XDND_COPYRIGHT)
#define XDND_COPYRIGHT "Copyright (c) 1998 by Gerald L. Gay "
#endif

/* Important Note:
 *
 *    For languages other then English modify all the labelString
 * resources here or in the resource file in the main app-defaults 
 * directory.  We can support foreign languages that easily.
 */

/* fallback resources, 
 *
 * The initial app-defaults file should be similar to this.
 *
 */
_XtString fallbacks[] = 
{
   "! " XDND_MODULE_NAME " resources",
   "! Version " XDND_VERSION,
   "! " XDND_COPYRIGHT, 
   "! This resource file was generated by " XDND_MODULE_NAME ".",
   "! ",
   "*Background: grey",
   "*Foreground: black",
   "*HighlightColor: black",
   "*fontList: -*-helvetica-medium-r-*--12-120-*",
   "*" ".dialogTitle: Drag and Drop Example",
   NULL
};

// extra command line options
static XrmOptionDescRec options [] =
{
   { "-g", "*generateResourceFile",  XrmoptionSepArg, NULL },
   { "--generate-resource=", "*generateResourceFile",  XrmoptionSepArg, NULL },
   { "--help",  "*showUsage",  XrmoptionNoArg, "True" }, 
   { "-h",     "*showUsage",  XrmoptionNoArg, "True" },
   { "--version",     "*showVersion",  XrmoptionNoArg, "True" },
   { "-v",     "*showVersion",  XrmoptionNoArg, "True" },
};

#define XtNgenerateResourceFile "generateResourceFile"
#define XtCgenerateResourceFile "GenerateResourceFile"
#define XtNshowUsage "showUsage"
#define XtCshowUsage "ShowUsage"
#define XtNshowVersion "showVersion"
#define XtCshowVersion "ShowVersion"

static XtResource resources[] = 
{
  {
     XtNgenerateResourceFile, 
     XtCgenerateResourceFile,
     XtRString, 
     sizeof(char *), 
     XtOffsetOf(Application, resource_file),
     XtRString, 
     (XtPointer) NULL
  },

  {
     XtNshowUsage, 
     XtCshowUsage,
     XtRBoolean, 
     sizeof(Boolean), 
     XtOffsetOf(Application, show_usage),
     XtRImmediate, 
     (XtPointer) False
  },

  {
     XtNshowVersion, 
     XtCshowVersion,
     XtRBoolean, 
     sizeof(Boolean), 
     XtOffsetOf(Application, show_version),
     XtRImmediate, 
     (XtPointer) False
  },

};

// =========================================================================
// functions
int
main (int argc, char **argv)
{
   // initialize application 
   Application app(Application::basename(argv[0]),argc,argv);

   // realize application
   app.realize();

   // start the main loop 
   app.mainLoop();

   // exit code just in case we get here.
   exit(EXIT_SUCCESS);
   return EXIT_SUCCESS;  /* SOME compilers like a return statement */
}

// ======================================================================= 
// Application member functions 

Application::Application(char *app_class, int &argc_in_out, char **argv_in_out):
   AppContext(app_class, options, XtNumber(options), argc_in_out, argv_in_out, 
              fallbacks)
{
   // check to see if any args are left over
   if(argc_in_out > 1)
   {
      syntax(argc_in_out, argv_in_out);
   }

   // process application resources
   process_resources();

   // install window manager message handler(s)
   Atom proto = addWMProtocol(XA_WM_DELETE_WINDOW);
   addWMProtocolCallback(widget(),
                         proto,
                         (p_msg)&Application::onExit); 

   // set title
   title(app_class);

   // create 
   p_dnddlg = new DndDialog(*this);

}

void
Application::version()
{
   /* 
    * I should use iostreams but I don't have a working version.
    * Hmm, maybe a xmostreams class is needed.  Sure sounds ugly or maybe not.
    */
   printf("%s version %s\n", (argv())[0], XDND_VERSION);
   quit();
}

void
Application::usage()
{
   printf("Usage: %s [OPTION]\n", (argv())[0]);
   printf("  -g FILE  --generate-resource=FILE  Create resource file as FILE.\n");
   printf("  -h  --help  Print this message.\n");
   printf("  -v  --version  Output version info.\n");
   quit();
}

void
Application::process_resources()
{
   get_resources();

   // checked to see if the user wants the version info
   if(show_version == TRUE)
   {
      version();
   }
   // checked to see if the user wants a usage message
   if(show_usage == TRUE)
   {
      usage();
   }
   // checked to see if the user gave us a file name
   if(resource_file != NULL)
   {
      generate_resource_file(resource_file);
   }
}

void
Application::get_resources()
{
   XtGetApplicationResources(widget(), 
                             this, 
                             resources,
                             XtNumber(resources),
                             NULL,
                             0);
}

void
Application::generate_resource_file(char *fname)
{
   // I should use iostreams instead
   FILE *fp = fopen(fname,"w");
   if(fp != NULL)
   {
      // dump contents of fallbacks to the resource file
      for (int i=0; fallbacks[i] != NULL; i++)
      {
         fprintf(fp, "%s\n",fallbacks[i]);
      } 
      fclose(fp);
   }
   quit();
}

void
Application::syntax(int opt_no, char **cmd_opts)
{
   for (int i=1; i< opt_no; i++) 
   {
      fprintf(stderr,"Invalid option: %s\n", cmd_opts[i]);
   }
   quit();
}

char *
Application::basename(char *name)
{
   char *s;

   if((s=strrchr(name,'/'))!=NULL)
   {
      s++;
   }
   else
   {
      s = name;
   }	
   return s;
}

void 
Application::onExit(Widget, XtPointer, XtPointer)
{
   quit();
}

//--------------------
// DndDialog functions

DndDialog::DndDialog(Widget parent)
{
    XarmArg args;

    // Start with an empty trash can

    isTrashEmpty = true;

    theForm = new Form(parent);

    EmptyTrash = loadPixmap(parent, EmptyTrash_xpm);
    FullTrash  = loadPixmap(parent, FullTrash_xpm);
    Bill       = loadPixmap(parent, bill_xpm);

    args(XmNtopAttachment,   XmATTACH_FORM)
        (XmNleftAttachment,  XmATTACH_FORM)
        (XmNrightAttachment, XmATTACH_FORM);

    Header = new Label(*theForm, args, "Trash Bill");

    args.reset();
    args(XmNlabelType,        XmPIXMAP)
        (XmNlabelPixmap,      Bill)
        (XmNtopAttachment,    XmATTACH_WIDGET)
        (XmNtopWidget,        Header->widget())
        (XmNbottomAttachment, XmATTACH_FORM)
        (XmNleftAttachment,   XmATTACH_FORM);

    TheDevil = new Label(*theForm, args, "Bill");

    //
    // Flag the terminal as a drag source.
    //

    addDragSource(this, &DndDialog::dragStartProc, TheDevil->widget(), NULL);

    args.reset();
    args(XmNlabelType,        XmPIXMAP)
        (XmNlabelPixmap,      EmptyTrash)
        (XmNleftAttachment,   XmATTACH_WIDGET)
        (XmNleftWidget,       TheDevil->widget())
        (XmNtopAttachment,    XmATTACH_WIDGET)
        (XmNtopWidget,        Header->widget())
        (XmNbottomAttachment, XmATTACH_FORM);

    Trash = new Label(*theForm, args, "TrashCan");

    //
    // Flag the Trash Can as a drop site
    //
    // This function takes way too many
    // arguments for my tastes but there
    // itn't much choice.
    //

    addDropSite(this,
		&DndDialog::transferProc,
		NULL,
		Trash->widget(),
		XARM_DND_BUFFER_TRANSFER,
		XmDROP_COPY | XmDROP_MOVE,
		NULL,
		0);

    args.reset();
    args(XmNleftAttachment,   XmATTACH_WIDGET)
        (XmNleftWidget,       Trash->widget())
        (XmNrightAttachment,  XmATTACH_FORM)
        (XmNbottomAttachment, XmATTACH_FORM)
        (XmNtopAttachment,    XmATTACH_WIDGET)
        (XmNtopWidget,        Header->widget());

    emptyButton = new PushButton(*theForm, args, "Empty Trash");

    addCallback(this, &DndDialog::emptyTrash, emptyButton->widget(), XmNactivateCallback);

    theForm->manage();
}

//----------------------------------------
// We have the loadPixmap function because
// we want to set the pixmap background to
// whatever the background of the form is.
// That way, we get the transparency look.
// Unfortunately, there is currently no
// easy way to do it in Xarm.  It does sound
// like a good project though  :-)

Pixmap DndDialog::loadPixmap(Widget w, char **pdata)
{
    static Colormap _cmap;
 
    Screen *screen;
    Window root;
    Display *dpy;
    XarmXpmAttributes xpm_attrib;
    Pixmap mask;
    Pixmap pixmap_to_return;
    Pixel Background;
    int Depth;
    XarmXpmColorSymbol XpmTransparentColor[1] = { { NULL, "none", 0 } };
 
    /* The widget might not have a window yet so just use the root window */
    screen = XtScreen(w);
    root = RootWindowOfScreen(screen);
    dpy = XtDisplay(w);
    mask = 0;
 
    XtVaGetValues(w, XmNbackground, &Background,
		  XmNdepth,      &Depth,
		  NULL);
 
    XpmTransparentColor[0].pixel = Background;
 
    /* initialize colormap if it hasn't been done */
    if(_cmap == (Colormap) NULL) {
        XWindowAttributes w_attrib;
	XGetWindowAttributes(dpy,root,&w_attrib);
	_cmap=w_attrib.colormap;
    }
 
    xpm_attrib.colorsymbols = XpmTransparentColor;
    xpm_attrib.numsymbols = 1;
    xpm_attrib.depth = Depth;
    xpm_attrib.colormap=_cmap;
    xpm_attrib.closeness=65535;
    xpm_attrib.valuemask=XarmXpmSize | XarmXpmReturnPixels | XarmXpmDepth | XarmXpmColorSymbols
                         | XarmXpmColormap | XarmXpmCloseness;
    if(_XarmXpmCreatePixmapFromData(dpy,
                                    root,
                                    const_cast<const char **>(pdata),
                                    &pixmap_to_return,
                                    &mask,
                                    &xpm_attrib) != XarmXpmSuccess) {
        pixmap_to_return = XmUNSPECIFIED_PIXMAP;
    }
 
    return pixmap_to_return;
}

void DndDialog::transferProc(Widget w, XtPointer, XtPointer callData)
{

    DtDndTransferCallbackStruct *dtStuff = (DtDndTransferCallbackStruct *)callData;
    DtDndBuffer    *buffers = dtStuff->dropData->data.buffers;
 
    if ((dtStuff->reason != DtCR_DND_TRANSFER_DATA) &&
        (dtStuff->dropData->protocol != DtDND_BUFFER_TRANSFER)) {
        return;
    }
 
    if (strcmp(buffers[0].name, "Bill DND Demo") == 0) {

        if (!isTrashEmpty) return;

	Trash->labelPixmap(FullTrash);
	isTrashEmpty = false;
    } else {

        dtStuff->status = DtDND_FAILURE;

    }
}

bool DndDialog::dragStartProc(Widget w, XEvent *e, XtPointer cdata)
{

    Widget dragWidget;
    XarmArg args;

    dragIcon   = XCreateBitmapFromData(XtDisplay(w), XtWindow(w), (char *)bill_bm_bits, 26, 40);
    dragMask   = XCreateBitmapFromData(XtDisplay(w), XtWindow(w), (char *)bill_m_bm_bits, 26, 40);
    dragWidget = DtDndCreateSourceIcon(w, dragIcon, dragMask);

    args(DtNsourceIcon, dragWidget);

    // There isn't much to do here except call the Xarm procedure.
    // In a real app, you would decide what is being dragged, what
    // protocol to use, etc.  In this demo, we only have one thing
    // to drag and one way to drag it.
    //
    // This is another function with way too many arguments, sorry.

    XarmStartDrag(this,
                  &DndDialog::convertProc,
                  &DndDialog::finishProc,
                  NULL,
                  NULL,
                  w,
                  e,
                  XARM_DND_BUFFER_TRANSFER,
                  1,
                  XmDROP_COPY | XmDROP_MOVE,
                  args.getArgs(),
                  args.count());
}

void DndDialog::convertProc(Widget, XtPointer, XtPointer callData)
{
    DtDndConvertCallbackStruct *dtStuff = (DtDndConvertCallbackStruct *)callData;
    DtDndBuffer    *buffers = dtStuff->dragData->data.buffers;
 
    if (dtStuff == NULL) return;
 
    /*
     * Verify the validity of the callback reason
     */
 
    if (dtStuff->dragData->protocol != DtDND_BUFFER_TRANSFER ||
        (dtStuff->reason != DtCR_DND_CONVERT_DATA &&
         dtStuff->reason != DtCR_DND_CONVERT_DELETE)) {
        dtStuff->status = DtDND_FAILURE;
        return;
    }
 
    if (dtStuff->reason == DtCR_DND_CONVERT_DATA) {
        buffers[0].name = "Bill DND Demo";
    }
}

void DndDialog::finishProc(Widget, XtPointer, XtPointer)
{
  // Nothing to do
}

void DndDialog::emptyTrash(Widget, XtPointer, XtPointer)
{
    Trash->labelPixmap(EmptyTrash);
    isTrashEmpty = true;
}

#else // XARM_HAS_CDE

int main()
{
  cout << "Sorry, this demo currently only works with CDE" << endl;
}

#endif
