//
// Copyright 1994, Cray Research, Inc.
//                 
// Permission to use, copy, modify and distribute this software and
// its accompanying documentation (the "Software") is granted without
// fee, provided that the above copyright notice and this permission
// notice appear in all copies of the Software and all supporting
// documentation, and the name of Cray Research, Inc. not be used in
// advertising or publicity pertaining to distribution of the 
// Software without the prior specific, written permission of Cray
// Research, Inc.  The Software is a proprietary product of Cray
// Research, Inc., and all rights not specifically granted by this
// license shall remain in Cray Research, Inc.  No charge may be made
// for the use or distribution of the Software.  The Software may be
// distributed as a part of a different product for which a fee is
// charged, if (i) that product contains or provides substantial
// functionality that is additional to, or different from, the
// functionality of the Software, and (ii) no separate, special or
// direct charge is made for the Software.
//         
// THE SOFTWARE IS MADE AVAILABLE "AS IS", AND ALL EXPRESS AND
// IMPLIED WARRANTIES, INCLUDING THE IMPLIED WARRANTIES OF FITNESS
// FOR A PARTICULAR PURPOSE, MERCHANTABILITY, AND FREEDOM FROM
// VIOLATION OF THIRD PARTY INTELLECTUAL PROPERTY RIGHTS, ARE HEREBY
// DISCLAIMED AND EXCLUDED BY CRAY RESEARCH, INC.  CRAY RESEARCH,
// INC. WILL NOT BE LIABLE IN ANY EVENT FOR ANY CONSEQUENTIAL,
// SPECIAL, INCIDENTAL, OR INDIRECT DAMAGES ARISING OUT OF OR IN
// CONNECTION WITH THE PERFORMANCE OF THE SOFTWARE OR ITS USE BY ANY
// PERSON, OR ANY FAILURE OR NEGLIGENCE ON THE PART OF CRAY RESEARCH,
// INC., EXCEPT FOR THE GROSS NEGLIGENCE OR WILLFUL MISCONDUCT OF
// CRAY RESEARCH.
// 
// This License Agreement shall be governed by, and interpreted and
// construed in accordance with, the laws of the State of Minnesota,
// without reference to its provisions on the conflicts of laws, and
// excluding the United Nations Convention of the International Sale
// of Goods.
//
static void USMID() { void("%Z%%M%	%I%	%G% %U%"); }
static void RSCID() { void("$Id: TextPage.cc,v 1.7 1994/09/09 19:46:21 prb Exp $"); }
#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include <Cvo/TextPage.h++>

wchar_t Cvo_TextPageLine::nullstring[1] = { 0, };

Cvo_TextPage::Cvo_TextPage()
{   CVO_ENTER
    output = True;

    newlines = False;
    changedlines = False;
    removedlines = False;

    maxlines = 0;
    absolutemax = 0;
    nlines = 0;
    attr = 0;
    start = 0;
    lines = 0;
    CVO_VOID_RETURN
}

Cvo_TextPage::~Cvo_TextPage()
{   CVO_ENTER
    delete [] lines;
    CVO_DONE
}

void
Cvo_TextPage::Draw(CARD n, const XRectangle &clip, int x, int y)
{
    if (n < maxlines) {
	Cvo_TextPageLine &line = Line(n);

	if (line.Attributes())
	    DrawText(line.Attributes(), line.Text(), line.Length(), clip, x, y);
	else
	    DrawText(line.Attribute(), line.Text(), line.Length(), clip, x, y);
    }
}

CARD
Cvo_TextPage::Width(CARD n)
{
    if (n < maxlines) {
	Cvo_TextPageLine &line = Line(n);

	if (line.Attributes())
	    return(WidthText(line.Attributes(), line.Text(), line.Length()));
	else
	    return(WidthText(line.Attribute(), line.Text(), line.Length()));
    }
    return(0);
}

CARD
Cvo_TextPage::Height(CARD n)
{
    if (n < maxlines) {
	Cvo_TextPageLine &line = Line(n);

	if (line.Attributes())
	    return(HeightText(line.Attributes(), line.Text(), line.Length()));
	else
	    return(HeightText(line.Attribute(), line.Text(), line.Length()));
    }
    return(0);
}

void
Cvo_TextPage::SendPageEvent()
{   CVO_ENTER
    Cvo_PageChangedEvent pce;

    if (!win || !output)
	CVO_VOID_RETURN

    pce.newlines = newlines;
    pce.changedlines = changedlines;
    pce.removedlines = removedlines;
    
    newlines = False;
    changedlines = False;
    removedlines = False;

    win->SendEvent(CvoPageChangedEvent, &pce);

    for (int i = 0; i < nlines; ++i)
	Line(i).ClearModified();
    CVO_VOID_RETURN

}

void
Cvo_TextPage::Notify()
{   CVO_ENTER
    if (!win) {
	CVO_VOID_RETURN
    }
    newlines = True;
    changedlines = True;
    removedlines = True;
    SendPageEvent();
    CVO_VOID_RETURN
}

//
//	ClearLines() - remove all the lines in the buffer.
//

void
Cvo_TextPage::ClearLines(int inform)
{   CVO_ENTER
    for (int x = 0; x < nlines; ++x)
	Line(x).Clear();
    nlines = 0;
    changedlines = True;
    removedlines = True;

    if (inform)
	SendPageEvent();
    CVO_VOID_RETURN
}

void
Cvo_TextPage::ReplaceLine(char *s, int line, int inform)
{   CVO_ENTER

    if (line < nlines) {
	static Cvo_CharacterBuffer cb;
	cb.Set(s);

	Line(line).SetText(cb.wcValue(), cb.wcLength(), attr);
	changedlines = True;

	if (inform)
	    SendPageEvent();
    }

    CVO_VOID_RETURN
}

void
Cvo_TextPage::ReplaceLine(wchar_t *s, int line, int inform)
{   CVO_ENTER

    if (line < nlines) {
	int length = 0;
	for (wchar_t *t = s; *t; ++t, ++length)
	    ;

	Line(line).SetText(s, length, attr);
	changedlines = True;

	if (inform)
	    SendPageEvent();
    }

    CVO_VOID_RETURN
}

void
Cvo_TextPage::AppendToLine(char *s, int line, int inform)
{   CVO_ENTER

    if (line < nlines) {
	static Cvo_CharacterBuffer cb;
	cb.Set(s);

	Line(line).AppendText(cb.wcValue(), cb.wcLength(), attr);
	changedlines = True;

	if (inform)
	    SendPageEvent();
    }

    CVO_VOID_RETURN
}

void
Cvo_TextPage::AppendToLine(wchar_t *s, int line, int inform)
{   CVO_ENTER

    if (line < nlines) {
	int length = 0;
	for (wchar_t *t = s; *t; ++t, ++length)
	    ;

	Line(line).AppendText(s, length, attr);
	changedlines = True;

	if (inform)
	    SendPageEvent();
    }

    CVO_VOID_RETURN
}

int
Cvo_TextPage::AppendLine(wchar_t *s, int length)
{   CVO_ENTER

    if (nlines < maxlines) {
	Line(nlines++).SetText(s, length, attr);
	newlines = True;
    } else if (!absolutemax || maxlines < absolutemax) {
	int newmax = maxlines + 64;

	if (absolutemax && newmax > absolutemax)
	    newmax = absolutemax;

	Cvo_TextPageLine *nl = new Cvo_TextPageLine[newmax];

	for (int x = 0; x < maxlines; ++x) {
	    nl[x] = Line(x);
	    Line(x).Zap();
	}
	maxlines = newmax;

	start = 0;
	delete [] lines;
	lines = nl;

	Line(nlines++).SetText(s, length, attr);
	newlines = True;
    } else {
	start = (start + 1) % maxlines;
	Line(nlines-1).SetText(s, length, attr);
	changedlines = True;
	newlines = True;
	removedlines = True;
    }
    CVO_RETURN(nlines - 1)
}

int
Cvo_TextPage::InsertLine(wchar_t *s, int length)
{   CVO_ENTER
    AppendLine(s, length);

    //
    // Okay, now put it in alphabetical order...
    //
    if (nlines == 1) {
	CVO_RETURN(0)
    }

    if (nlines == 2) {
	if (lines[0] > lines[1]) {
	    Cvo_TextPageLine tmp;

	    tmp = Line(0);
	    Line(0) = Line(1);
	    Line(1) = tmp;
	    tmp.Zap();
	    changedlines = True;
	}
	CVO_RETURN(0)
    }
    for (int x = nlines - 2; x >= 0; --x) {
	if (Line(nlines-1) > Line(x))
	    break;
    }

    if (x++ == nlines - 2)
	CVO_RETURN(nlines-1)

    changedlines = True;

    Cvo_TextPageLine tmp;

    tmp = Line(nlines-1);
    for (int i = nlines - 1; i > x; --i) {
	Line(i) = Line(i-1);
    }
    Line(x) = tmp;
    tmp.Zap();
    CVO_RETURN(x)
}

int
Cvo_TextPage::AppendText(char *s, void *data, int inform)
{   CVO_ENTER

    int r = -1;
    if (s && !*s) {
	wchar_t null = 0;
	r = AppendLine(&null, 0);
	Line(r).SetAssociatedData(data);
    } else if (s) {
	char *e = s;

	while (*s) {
	    while (*e && *e != '\r' && *e != '\n')
		++e;

	    static Cvo_CharacterBuffer cb;
	    cb.Set(s, e - s);
	    r = AppendLine(cb.wcValue(), cb.wcLength());
	    Line(r).SetAssociatedData(data);

	    if (*e) {
		//
		// Handle CR-LF sequences
		//
		if (e[0] == '\r' && e[1] == '\n')
		    ++e;
		++e;
	    }
	    s = e;
	}
    }

    if (inform)
	SendPageEvent();
    CVO_RETURN(r)
}

int
Cvo_TextPage::InsertText(char *s, void *data, int inform)
{   CVO_ENTER
    int r = -1;
    if (s) {
	char *e = s;

	while (*s) {
	    while (*e && *e != '\r' && *e != '\n')
		++e;
	    static Cvo_CharacterBuffer cb;
	    cb.Set(s, e - s);
	    r = InsertLine(cb.wcValue(), cb.wcLength());
	    Line(r).SetAssociatedData(data);
            if (*e) {
		//
		// Handle CR-LF sequences
		//
		if (e[0] == '\r' && e[1] == '\n')
		    ++e;
		++e;
	    }
	    s = e;
	}
    }

    if (inform)
	SendPageEvent();
    CVO_RETURN(r)
}

int
Cvo_TextPage::AppendText(wchar_t *s, void *data, int inform)
{   CVO_ENTER
    int r = -1;
    if (s && !*s) {
	r = AppendLine(s, 0);
	Line(r).SetAssociatedData(data);
    } else if (s) {
	wchar_t *e = s;

	while (*s) {
	    while (*e && *e != '\r' && *e != '\n')
		++e;
	    static Cvo_CharacterBuffer cb;
	    cb.Set(s, e - s);
	    r = AppendLine(cb.wcValue(), cb.wcLength());
	    Line(r).SetAssociatedData(data);
	    if (*e) {
		//
		// Handle CR-LF sequences
		//
		if (e[0] == '\r' && e[1] == '\n')
		    ++e;
		++e;
	    }
	    s = e;
	}
    }

    if (inform)
	SendPageEvent();
    CVO_RETURN(r)
}

int
Cvo_TextPage::InsertText(wchar_t *s, void *data, int inform)
{   CVO_ENTER
    int r = -1;
    if (s) {
	wchar_t *e = s;

	while (*s) {
	    while (*e && *e != '\r' && *e != '\n')
		++e;
	    static Cvo_CharacterBuffer cb;
	    cb.Set(s, e - s);
	    r = InsertLine(cb.wcValue(), cb.wcLength());
            Line(r).SetAssociatedData(data);
            if (*e) {
		//
		// Handle CR-LF sequences
		//
		if (e[0] == '\r' && e[1] == '\n')
		    ++e;
		++e;
	    }
	    s = e;
	}
    }

    if (inform)
	SendPageEvent();
    CVO_RETURN(r)
}

void
Cvo_TextPage::DeleteEntry(int index, int inform)
{   CVO_ENTER
    if (index == -1) {
	ClearLines();
    } else if (index < 0 || index >= nlines) {
	CVO_VOID_RETURN
    } else {
	changedlines = True;
	removedlines = True;
	Line(index).Clear();
	--nlines;
	if (index == 0) {
	    start = (start + 1) % maxlines;
	} else if (index < nlines) {
	    while (index < nlines) {
		Line(index) = Line(index+1);
		++index;
	    }
    	    Line(nlines).Zap();
	}
    }
	
    if (inform)
	SendPageEvent();
    CVO_VOID_RETURN
}

void
Cvo_TextPage::AddEntry(int index, int inform)
{
    if (index < 0)
	return;
    if (index >= nlines) {
	if (absolutemax && index >= absolutemax)
	    return;
    	nlines = index + 1;
	if (nlines >= maxlines) {
	    int newmax = (index + 0x40) & ~0x3f;

	    Cvo_TextPageLine *nl = new Cvo_TextPageLine[newmax];

	    for (int x = 0; x < maxlines; ++x) {
		nl[x] = Line(x);
		Line(x).Zap();
	    }
	    maxlines = newmax;

	    start = 0;
	    delete [] lines;
	    lines = nl;
	}

	newlines = True;
    } else {
	if (absolutemax && nlines + 1 >= absolutemax)
	    return;

	if (++nlines >= maxlines) {
	    int newmax = (nlines + 0x40) & ~0x3f;

	    Cvo_TextPageLine *nl = new Cvo_TextPageLine[newmax];

	    for (int x = 0; x < maxlines; ++x) {
		nl[x] = Line(x);
		Line(x).Zap();
	    }
	    maxlines = newmax;

	    start = 0;
	    delete [] lines;
	    lines = nl;
	}

	Cvo_TextPageLine tmp = Line(nlines-1);
	for (int i = nlines - 1; i > index; --i) {
	    Line(i) = Line(i-1);
	}
	Line(index) = tmp;
	tmp.Zap();

	newlines = True;
	changedlines = True;
    }
    if (inform)
	SendPageEvent();
}

void *
Cvo_TextPage::GetLine(CARD i)
{   CVO_ENTER
    if (i < nlines) {
	charbuf.Set(Line(i).Text());
	CVO_RETURN(charbuf.mbValue())
    }
    charbuf.Truncate(0);
    CVO_RETURN(0)
}

void *
Cvo_TextPage::GetLine(CARD i, Cvo_CharacterBuffer *cb)
{   CVO_ENTER
    CVO_RETURN((i < nlines) ? (cb->Set(Line(i).Text()), cb) : 0)
}

void
Cvo_TextPage::SetAttributes(CARD i, CARD f, CARD t, Cvo_TextAttribute *a)
{   CVO_ENTER
    if (i < nlines) {
	Line(i).SetAttribute(a, f, t);
	changedlines = True;
	SendPageEvent();
    }
    CVO_VOID_RETURN
}

void
Cvo_TextPage::SetAttributes(CARD i, Cvo_TextAttribute *a)
{   CVO_ENTER
    if (i < nlines) {
	Line(i).SetAttribute(a);
	changedlines = True;
	SendPageEvent();
    }
    CVO_VOID_RETURN
}

void
Cvo_TextPage::InsertAt(int line, int col, int len, int inform)
{
    while (nlines <= line) {
	wchar_t space = ' ';
	AppendLine(&space, 1);
    }
    Line(line).InsertAt(col, len, attr);
    changedlines = True;
    if (inform)
	SendPageEvent();
}

void
Cvo_TextPage::ReplaceAt(int line, int col, wchar_t *s, int len, int inform)
{
    while (nlines < line) {
	wchar_t space = ' ';
	AppendLine(&space, 1);
    }

    Line(line).ReplaceAt(col, s, len);
    changedlines = True;
    if (inform)
	SendPageEvent();
}

void
Cvo_TextPage::DeleteAt(int line, int col, int len, int inform)
{
    if (line >= nlines)
	return;

    Line(line).DeleteAt(col, len);
    changedlines = True;
    if (inform)
	SendPageEvent();
}

void
Cvo_TextPageLine::SetText(wchar_t *s, int len, Cvo_TextAttribute *attr)
{   CVO_ENTER
    delete [] text;

    if (multiple) {
	delete [] attrs;
	multiple = False;
    }

    modified = True;
    attrs = (Cvo_TextAttribute **)attr;
    length = len;
    text = new wchar_t[length + 1];
    memcpy(text, s, length * sizeof(wchar_t));
    text[length] = 0;
    CVO_VOID_RETURN
}

void *
Cvo_TextPage::GetAssociatedData(CARD i)
{
    if (i < nlines)
        return(Line(i).AssociatedData());
    return(0);
}
void
Cvo_TextPage::SetAssociatedData(CARD i, void *data)
{
    if (i < nlines)
        Line(i).SetAssociatedData(data);
}

void
Cvo_TextPageLine::AppendText(wchar_t *s, int len, Cvo_TextAttribute *attr)
{   CVO_ENTER
    if (!length) {
	SetText(s, len, attr);
	CVO_VOID_RETURN
    }

    wchar_t *nt = new wchar_t[length + len + 1];
    memcpy(nt, text, length * sizeof(wchar_t));
    memcpy(nt+length, s, len * sizeof(wchar_t));
    delete [] text;
    text = nt;

    if (multiple) {
	Cvo_TextAttribute **na = new Cvo_TextAttribute *[length+len];
	memcpy(na, attrs, length * sizeof(Cvo_TextAttribute *));
	delete [] attrs;
	attrs = na;
	for (int i = length; i < length + len; ++i) {
	    attrs[i] = attr;
	}
    }
    length += len;
    text[length] = 0;
    if (!multiple)
	SetAttribute(attr, length - len, length - 1);
    modified = True;
    CVO_VOID_RETURN
}

void
Cvo_TextPageLine::DeleteAt(int c, int len)
{
    if (c + len > length)
	len = length - c;

    if (len <= 0)
	return;

    if (c + len < length)
	memcpy(text + c, text + c + len, (length - (len + c)) * sizeof(wchar_t));
    if (multiple) {
	if (c + len < length)
	    memcpy(attrs + c, attrs + c + len,
		   (length - (len + c)) * sizeof(Cvo_TextAttribute **));
    }
    length -= len;
    text[length] = 0;
    modified = True;
}

void
Cvo_TextPageLine::InsertAt(int c, int len, Cvo_TextAttribute *attr)
{   CVO_ENTER
    if (len <= 0)
	return;

    int i;
    wchar_t *nt;

    if (c == 0) {
	nt = new wchar_t[length + len + 1];
	if (length)
	    memcpy(nt + len, text, length * sizeof(wchar_t));
    } else if (c < length) {
	nt = new wchar_t[length + len + 1];
	memcpy(nt, text, c * sizeof(wchar_t));
	memcpy(nt + c + len, text + c, (length - c) * sizeof(wchar_t));
    } else {
	nt = new wchar_t[c + len + 1];
	memcpy(nt, text, length * sizeof(wchar_t));
	while (length < c)
	    nt[length++] = ' ';
    }
    delete [] text;
    text = nt;
    for (i = 0; i < len; ++i)
	text[i + c] = ' ';

    if (multiple) {
	Cvo_TextAttribute **na;
	if (c == 0) {
	    if (!attr)
		attr = attrs[0];
	    na = new Cvo_TextAttribute *[length + len + 1];
	    if (length)
		memcpy(na + len, attrs, length * sizeof(Cvo_TextAttribute *));
	} else if (c < length) {
	    if (!attr)
		attr = attrs[c];
	    na = new Cvo_TextAttribute *[length + len + 1];
	    memcpy(na, attrs, c * sizeof(Cvo_TextAttribute *));
	    memcpy(na + c + len, attrs + c, (length - c) * sizeof(Cvo_TextAttribute *));
	} else {
	    if (!attr)
		attr = attrs[length - 1];
	    na = new Cvo_TextAttribute *[c + len + 1];
	    memcpy(na, attrs, length * sizeof(Cvo_TextAttribute *));
	    for (int i = length; i < c; ++i)
		    na[i] = attr;
	}
	delete [] attrs;
	attrs = na;
	for (i = c; i < c + len; ++i)
	    attrs[i] = attr;
    } else if (attr)
	SetAttribute(attr, c, c + len);
    length += len;
    text[length] = 0;

    modified = True;
    CVO_VOID_RETURN
}

void
Cvo_TextPageLine::ReplaceAt(int c, wchar_t *s, int len)
{
    if (len < 0) {
	len = 0;
	while (s[len])
		++len;
    }
    if (len == 0)
	return;

    if (length < c + len) {
	InsertAt(length, c + len - length);
    }

    memcpy(text + c, s, len * sizeof(wchar_t));
    modified = True;
}

void
Cvo_TextPageLine::SetAttribute(Cvo_TextAttribute *attr, CARD from, CARD to)
{   CVO_ENTER
    if (from >= length || to < from)
	CVO_VOID_RETURN
    if (to >= length)
	to = length-1;

    if (!multiple) {
	Cvo_TextAttribute *oattr = (Cvo_TextAttribute *)attrs;
	if (oattr == attr)
	    CVO_VOID_RETURN
	attrs = new Cvo_TextAttribute *[length];
	for (int i = 0; i < length; ++i)
	    attrs[i] = oattr;
	multiple = True;
    }
    while (from <= to)
	attrs[from++] = attr;
    modified = True;
    CVO_VOID_RETURN
}

void
Cvo_TextPageLine::SetAttribute(Cvo_TextAttribute *attr)
{   CVO_ENTER

    if (multiple) {
	delete [] attrs;
	multiple = False;
    }

    attrs = (Cvo_TextAttribute **)attr;
    modified = True;
    CVO_VOID_RETURN
}

int
Cvo_TextPageLine::operator <(Cvo_TextPageLine &other)
{   CVO_ENTER
    int i = 0;

    while (text[i] && text[i] == other.text[i])
	++i;
    modified = True;
    CVO_RETURN(text[i] < other.text[i])
}

int
Cvo_TextPageLine::operator >(Cvo_TextPageLine &other)
{   CVO_ENTER
    int i = 0;

    while (text[i] && text[i] == other.text[i])
	++i;
    CVO_RETURN(text[i] > other.text[i])
}

int
Cvo_TextPageLine::operator ==(Cvo_TextPageLine &other)
{   CVO_ENTER
    int i = 0;

    while (text[i] && text[i] == other.text[i])
	++i;
    CVO_RETURN(text[i] == other.text[i])
}
