/*************************************************************
 *                                                           *
 *                     file: cdrom.c                         *
 *                                                           *
 *************************************************************/

/*
 * Copyright (C) 1988, 1989 Sun Microsystems, Inc.
 */

/*
 * This file contains the routine to interface with the cdrom player.
 */

#include <stdio.h>
#include "cdrom.h"

#include <sys/file.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <mntent.h>
#include <string.h>


/*
 *	these lines added for debugging by DGR
 */

#include <errno.h>

/*
 *	end of added lines
 */

CD_ROM::CD_ROM(char *dev_name, int _nx)
{
    fd = -1;
    ejected = 0;
    entries = 0;
    times = 0;

    Open(dev_name, _nx);
}

void
CD_ROM::Open(char *dev_name, int _nx)
{
    ejected = 0;

    if (fd < 0) {
	if (nx = _nx) {
	    if ((fd = open(dev_name, O_RDONLY)) < 0 &&
		errno != ENXIO && errno != EIO) {
		fprintf(stderr, "failed to open device %s: ", dev_name);
		perror("");
	    }
	    if (fd < 0) ejected++;
	} else {
	    if ((fd = open(dev_name, O_RDONLY|O_EXCL)) < 0 &&
		errno != ENXIO && errno != EIO ) {
		fprintf(stderr, "failed to open device %s: ", dev_name);
		perror("");
	    }
	    if (fd < 0) ejected++;
	}
    }
 
    if (fd >= 0 && !ReadToc())
	ejected = 1;
}

int
CD_ROM::PlayTrack(int play_track, int end_track)
{
	if (ejected) return(0);

	struct	cdrom_ti	ti;

	ti.cdti_trk0 = play_track;
	ti.cdti_ind0 = 1;
	ti.cdti_trk1 = end_track;
	ti.cdti_ind1 = 1;
	if (ioctl(fd, CDROMPLAYTRKIND, (struct cdrom_ti *)&ti) < 0) {
		perror("cdrom_play_track:ioctl");
		return 0;
	}
	return 1;
}

int
CD_ROM::Start()
{
	if (ioctl(fd, CDROMSTART) < 0) {
		if (errno != ENXIO && errno != EIO) perror("ioctl");
		return 0;
	}
	return 1;
}

int
CD_ROM::Stop()
{
	if (ioctl(fd, CDROMSTOP) < 0) {
		perror("cdrom_stop:ioctl");
		return 0;
	}
	return 1;
}

int
CD_ROM::Eject()
{
	ejected = 1;
	if (ioctl(fd, CDROMEJECT) < 0) {
		if (errno != ENXIO && errno != EIO) perror("ioctl");
		return 0;
	}
	return 1;
}

static void
PrintTocEntry(cdrom_tocentry *e, Msf *msf)
{
	printf("Trk: %3d  Adr: 0x%x  Ctl: 0x%x  Fmt: 0x%02x  MSF %02d:%02d (%02d:%02d[%2d])  DM: 0x%02x\n",
		e->cdte_track,
		e->cdte_adr,
		e->cdte_ctrl,
		e->cdte_format,
		msf ? msf->Minute() : e->cdte_addr.msf.minute,
		msf ? msf->Second() : e->cdte_addr.msf.second,
		e->cdte_addr.msf.minute,
		e->cdte_addr.msf.second,
		e->cdte_addr.msf.frame,
		e->cdte_datamode);
}

int
CD_ROM::ReadToc()
{
	if (ejected) return(0);

	if (!ReadTocHeader())
		return(0);
	delete [] entries;
	delete [] times;
	entries = new cdrom_tocentry[(LastTrack() + 1) - FirstTrack()];
	times = new Msf[(LastTrack() + 1) - FirstTrack()];
	lead.cdte_format = CDROM_MSF;
	lead.cdte_track = CDROM_LEADOUT;
	ReadTocEntry(&lead);

	total.Set(lead.cdte_addr.msf.minute,
		  lead.cdte_addr.msf.second,
		  lead.cdte_addr.msf.frame);

	for (int i = FirstTrack(); i <= LastTrack(); ++i) {
	    int t = i - FirstTrack();
	    entries[t].cdte_format = CDROM_MSF;
	    entries[t].cdte_track = i;
	    ReadTocEntry(&entries[t]);

	    times[t].Set(entries[t].cdte_addr.msf.minute,
			 entries[t].cdte_addr.msf.second,
			 entries[t].cdte_addr.msf.frame);
	    if (i > FirstTrack())
		times[t-1] = times[t] - times[t-1];
	    if (i == LastTrack()) {
		times[t] = total - times[t];
	    }
	}
}

int
CD_ROM::ReadTocEntry(struct cdrom_tocentry *entry)
{
	if (ejected) return(0);
	if (ioctl(fd, CDROMREADTOCENTRY, entry) < 0) {
		if (errno != ENXIO && errno != EIO) perror("ioctl");
		return 0;
	}
	return 1;
}
	
int
CD_ROM::ReadTocHeader()
{
	if (ejected) return(0);
	if (ioctl(fd, CDROMREADTOCHDR, &hdr) < 0) {
		if (errno != ENXIO && errno != EIO) perror("ioctl");
		return 0;
	}
	return 1;
}
	
int
CD_ROM::Pause()
{
	if (ejected) return(0);
	if (ioctl(fd, CDROMPAUSE) < 0) {
		perror("cdrom_pause:ioctl");
		return 0;
	}
	return 1;
}

int
CD_ROM::Resume()
{
	if (ejected) return(0);
	if (ioctl(fd, CDROMRESUME) < 0) {
		perror("cdrom_resum:ioctl");
		return 0;
	}
	return 1;
}

int
CD_ROM::Volume(int left_vol, int right_vol)
{
	struct	cdrom_volctrl	vol;

	if (left_vol > 255)	left_vol = 255;
	if (left_vol < 0)	left_vol = 0;
	if (right_vol > 255)	right_vol = 255;
	if (right_vol < 0)	right_vol = 0;

	left = vol.channel0 = left_vol;
	right = vol.channel1 = right_vol;
	/* I don't know what these are for */
	vol.channel2 = left_vol;
	vol.channel3 = right_vol;
	if (ioctl(fd, CDROMVOLCTRL, (struct cdrom_volctrl *)&vol) < 0) {
		if (errno != ENXIO && errno != EIO) perror("ioctl");
		return 0;
	}
	return 1;
}

int
CD_ROM::GetRelativeMSF(struct tm *tmp, int *track)
{
    if (ejected) return(0);
    struct cdrom_subchnl    sc;
 
    sc.cdsc_format = CDROM_MSF;
    if (ioctl(fd, CDROMSUBCHNL, (struct cdrom_subchnl *)&sc) < 0) {
        if (errno != ENXIO && errno != EIO) perror("ioctl");
        return -1;
    }
     
    tmp->tm_sec = sc.cdsc_reladdr.msf.second;
    tmp->tm_min = sc.cdsc_reladdr.msf.minute;
 
    *track = sc.cdsc_trk;
 
    if (sc.cdsc_audiostatus == CDROM_AUDIO_PLAY) {
        return 1;
    }
    else {
        return 0;
    }  
}

int
CD_ROM::GetAbsoluteMSF(Msf *msf)
{
    if (ejected) return(0);
    struct cdrom_subchnl    sc;
 
    sc.cdsc_format = CDROM_MSF;
    if (ioctl(fd, CDROMSUBCHNL, (struct cdrom_subchnl *)&sc) < 0) {
        if (errno != ENXIO && errno != EIO) perror("ioctl");
        return 0;
    }
    msf->Set(sc.cdsc_absaddr.msf.minute,
    	     sc.cdsc_absaddr.msf.second,
    	     sc.cdsc_absaddr.msf.frame);
 
    return 1;
}
 
 
int
CD_ROM::GetMSF(struct cdrom_msf *msf)
{
    if (ejected) return(0);
    struct cdrom_subchnl    sc;
 
    sc.cdsc_format = CDROM_MSF;
    if (ioctl(fd, CDROMSUBCHNL, (struct cdrom_subchnl *)&sc) < 0) {
        if (errno != ENXIO && errno != EIO) perror("ioctl");
        return 0;
    }
    msf->cdmsf_min0 = sc.cdsc_absaddr.msf.minute;
    msf->cdmsf_sec0 = sc.cdsc_absaddr.msf.second;
    msf->cdmsf_frame0 = sc.cdsc_absaddr.msf.frame;
 
    msf->cdmsf_min1 = (unsigned char) -1;
    msf->cdmsf_sec1 = (unsigned char) -1;
    msf->cdmsf_frame1 = (unsigned char) -1;
 
    return 1;
}
 
int
CD_ROM::PlayMSF(struct cdrom_msf *msf)
{
    if (ejected) return(0);
    if (ioctl(fd, CDROMPLAYMSF,  (struct cdrom_msf *)msf) < 0) {
        if (errno != ENXIO && errno != EIO) perror("ioctl");
        return(0);
    }  
    return 1;
}

int
CD_ROM::Playing(int *track)
{
    if (ejected) return(0);
    struct cdrom_subchnl    sc;
 
    sc.cdsc_format = CDROM_MSF;
    if (ioctl(fd, CDROMSUBCHNL, (struct cdrom_subchnl *)&sc) < 0) {
        if (errno != ENXIO && errno != EIO) perror("ioctl");
        return 0;
    }
    *track = sc.cdsc_trk;
    if (sc.cdsc_audiostatus == CDROM_AUDIO_PLAY) {
        return 1;
    }
    else {
        return 0;
    } 
}

int
CD_ROM::Paused(int *track)
{
	if (ejected) return(0);
	struct cdrom_subchnl	sc;

	sc.cdsc_format = CDROM_MSF;
	if (ioctl(fd, CDROMSUBCHNL, (struct cdrom_subchnl *)&sc) < 0) {
		perror("cdrom_paused:ioctl");
		return 0;
	}
	*track = sc.cdsc_trk;
	printf("in pause, status is %d\n", sc.cdsc_audiostatus);
	if (sc.cdsc_audiostatus == CDROM_AUDIO_PAUSED) {
		return 1;
	}
	else {
		return 0;
	}
}
	
int
CD_ROM::Mounted(char *rawd)
{
	char buf[MAXPATHLEN], *cp;
	struct stat st;
	dev_t bdevno;
	FILE *fp;
	struct mntent *mnt;

	/*
	 * Get the block device corresponding to the raw device opened,
	 * and find its device number.
	 */
	if (stat(rawd, &st) != 0) {
		(void)fprintf(stderr, "Couldn't stat \"%s\": ", rawd);
		perror("mounted,1");
		return (UNMOUNTED);
	}
	/*
	 * If this is a raw device, we have to build the block device name.
	 */
	if ((st.st_mode & S_IFMT) == S_IFCHR) {
		cp = strchr(rawd, 'r');
		cp = (cp == NULL) ? NULL : ++cp;
		(void)sprintf(buf, "/dev/%s", cp);
		if (stat(buf, &st) != 0) {
			(void)fprintf(stderr, "Couldn't stat \"%s\": ", buf);
			perror("mounted,2");
			return (UNMOUNTED);
		}
	}
	if ((st.st_mode & S_IFMT) != S_IFBLK) {
		/*
		 * What is this thing? Ferget it!
		 */
		return (UNMOUNTED);
	}
	bdevno = st.st_rdev & (dev_t)(~0x07);	/* Mask out partition. */

	/*
	 * Now go through the mtab, looking at all hsfs filesystems.
	 * Compare the device mounted with our bdevno.
	 */
	if ((fp = setmntent(MOUNTED, "r")) == NULL) {
		(void)fprintf(stderr, "Couldn't open \"%s\"\n", MOUNTED);
		return (UNMOUNTED);
	}
	while ((mnt = getmntent(fp)) != NULL) {
		/* avoid obvious excess stat(2)'s */
		if (strcmp(mnt->mnt_type, "hsfs") != 0) {
			continue;
		}
		if (stat(mnt->mnt_fsname, &st) != 0) {
			continue;
		}
		if (((st.st_mode & S_IFMT) == S_IFBLK) &&
		    ((st.st_rdev & (dev_t)(~0x07)) == bdevno)) {
			(void)endmntent(fp);
			return (STILL_MOUNTED);
		}
	}
	(void)endmntent(fp);
	return (UNMOUNTED);
}

int
CD_ROM::GetSubChannel()
{
	if (ejected) return(0);

	sc.cdsc_format = CDROM_MSF;
	if (ioctl(fd, CDROMSUBCHNL, &sc) < 0) {
		perror("cdrom_get_subchnl:ioctl");
		return 0;
	}
	return(1);
}

inline double Fabs(double x)	{ return(x >= 0.0 ? x : -x); }
inline int Abs(int x)		{ return(x >= 0 ? x : -x); }

void
CD_ROM::Fade(double tl, double tr, int tm)
{
	if (left == 0) left = 155;
	if (right == 0) right = 155;

	double dl = (left - 155.0) / 10.0 - tl;
	double dr = (right - 155.0) / 10.0 - tl;
	int ni;

	if (Fabs(dl) > Fabs(dr)) {
		ni = (int)(dl * 10.0 + 0.5);
	} else {
		ni = (int)(dl * 10.0 + 0.5);
	}
	ni = Abs(ni);
	dl /= (double)ni;
	dr /= (double)ni;

	struct timeval tv;

	tm *= 1000000;	// Convert to micro seconds;

	tm /= ni;

	tv.tv_sec = tm / 1000000;
	tv.tv_usec = tm % 1000000;

	Volume(tl + dl * (ni - 1), tr + dr * (ni - 1));
	while (--ni >= 0) {
		struct timeval tv2 = tv;
		select(0, 0, 0, 0, &tv2);
		Volume(tl + dl * ni, tr + dr * ni);
	}
}

Msf
CD_ROM::TrackTime(int track)
{
    if (ejected || FirstTrack() > track || LastTrack() < track) {
	Msf msf = 0;
	return(msf);
    }
    Msf msf;
    msf.Set(entries[track].cdte_addr.msf.minute,
	    entries[track].cdte_addr.msf.second,
	    entries[track].cdte_addr.msf.frame);
    return(msf);
}
