/*
Copyright 1985, 1986, 1987, 1991, 1998  The Open Group

Portions Copyright 2000 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/

/*
 * $Id$
 */

#pragma ident	"@(#)iiimpAux.c 1.14	00/05/24 SMI"


#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <dlfcn.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <sys/param.h>
#ifndef linux
#include <synch.h>
#endif

#include <X11/Xmd.h>
#include <X11/Xlib.h>

#include "Xlcint.h"

#include "iiimpIM.h"
#include "iiimpAux.h"
#include "iiimpAuxP.h"
#include "guiIMPre.h"

#include "XimpIm.h"

#include "trace_message.h"


#define AUX_BASE_DIR		"/usr/lib/im/"

#if defined(__sparcv9)
#define SPARCV9_DIR		"sparcv9/"
#define SPARCV9_DIR_LEN		(8)
#endif /* __sparcv9 */

#define AUX_DIR_SYMBOL		"aux_dir"
#define AUX_CONF_MAGIC		"# IIIM X auxiliary"


#define IS_SPACE(len, ptr)	((0 < (len)) &&				\
				 (('\t' == *(p)) || (' ' == *(p))))
#define IS_EOL(len, ptr)	(((len) <= 0) || ('\n' == *(p)))
#define IS_NOT_EOL(len, ptr)	((0 < (len)) && ('\n' != *(p)))

#define IS_PREEDIT_POSITION(ic)						 \
			(XIMP_CHK_PRESPOTLMASK(ic) &&			 \
			 (!((ic)->core.input_style & XIMPreeditArea)) && \
			 ((ic->core.input_style & XIMPreeditPosition)))

/*
 * AuxCB aux_method
 */
static void	download(XicCommon ic, XPointer call_data);
static void	start(XicCommon ic, XPointer call_data);
static void	draw(XicCommon ic, XPointer call_data);
static void	done(XicCommon ic, XPointer call_data);
static void	switched(XicCommon ic, XPointer call_data);
static void	destroy(XicCommon ic, XPointer call_data);

static AuxCB aux_method[] = {
	NULL,			/* NONE */
	download,		/* AUX_DOWNLOAD */
	start,			/* AUX_START */
	draw,			/* AUX_DRAW */
	done,			/* AUX_DONE */
	switched,		/* AUX_SWITCHED */
	destroy			/* AUX_DESTROY */
};


/*
 * internal method
 */
static aux_handle_t *	aux_load(char * aux_file_name);
static aux_handle_t *	aux_conf_load(char * aux_file_name);
static aux_handle_t *	aux_so_load(char * aux_file_name);
static aux_t *		aux_get(XicCommon ic, XPointer call_data);
static aux_entry_t *	aux_entry_get(const CARD16 *, int);
static aux_im_data_t *	aux_im_get(XicCommon, int);


/*
 * X auxiliary object service method
 */
static void		service_aux_setvalue(aux_t *,
					     const unsigned char *, int);
static int		service_im_id(aux_t *);
static int		service_ic_id(aux_t *);
static void		service_data_set(aux_t *, int, void *);
static void *		service_data_get(aux_t *, int);
static Display *	service_display(aux_t *);
static Window		service_window(aux_t *);
static XPoint *		service_point(aux_t *, XPoint *);
static XPoint *		service_point_caret(aux_t *, XPoint *);
static size_t		service_utf16_mb(const char **, size_t *,
					 char **, size_t *);
static size_t		service_mb_utf16(const char **, size_t *,
					 char **, size_t *);
static unsigned char *	service_compose(const aux_data_t *, int *);
static int		service_compose_size(aux_data_type_t,
					     const unsigned char *);
static aux_data_t *	service_decompose(aux_data_type_t,
					  const unsigned char *);
static void		service_decompose_free(aux_data_t *);
static void		service_register_X_filter(Display *, Window, int, int,
						  Bool (* filter)(Display *,
								  Window,
								  XEvent *,
								  XPointer),
						  XPointer);
static void		service_unregister_X_filter(Display *, Window,
						    Bool (* filter)(Display *,
								    Window,
								    XEvent *,
								    XPointer),
						    XPointer);
static Bool		service_server(aux_t *);
static Window		service_client_window(aux_t *);
static Window		service_focus_window(aux_t *);
static int		service_screen_number(aux_t *);
static int		service_point_screen(aux_t *, XPoint *);
static int		service_point_caret_screen(aux_t *, XPoint *);

static aux_service_t aux_service = {
	service_aux_setvalue,
	service_im_id,
	service_ic_id,
	service_data_set,
	service_data_get,
	service_display,
	service_window,
	service_point,
	service_point_caret,
	service_utf16_mb,
	service_mb_utf16,
	service_compose,
	service_compose_size,
	service_decompose,
	service_decompose_free,
	service_register_X_filter,
	service_unregister_X_filter,
	service_server,
	service_client_window,
	service_focus_window,
	service_screen_number,
	service_point_screen,
	service_point_caret_screen
};


#if defined(ENABLE_TRACE)
static const char *	AuxChangeReasonName[] = {
	"NONE",
	"AUX_DOWNLOAD",
	"AUX_START",
	"AUX_DRAW",
	"AUX_DONE",
	"AUX_SWITCHED",
	"AUX_DESTROY"
};
#endif /* ENABLE_TRACE */


static aux_handle_t *	aux_handle;


/*
 * object downloading
 */
static void
download(XicCommon ic, XPointer call_data)
{
	char *		aux_file_name;
	int		aux_file_name_len;
	char		file_name[MAXPATHLEN];
	char *		dir_name;
	int		dir_name_len;
	char *		p;
	int		len;
	XIMText		text;
	Bool		useUnicode = False; /* default */

	TRACE_MESSAGE('A', ("iiimpAux: download\n"));

	aux_file_name = NULL;

	len = IMStringToXIMText((unsigned char *)call_data, &text,
				useUnicode);

	if (0 < len) {
		aux_file_name = text.string.multi_byte;
	}

#if defined(IIIM_AUX_SO_PATH_ENV)
	p = getenv(IIIM_AUX_SO_PATH_ENV);

	if (NULL != p) {
		aux_file_name = p;
	}

	if (NULL == aux_file_name) {
		aux_file_name = "./aux.so";
	}
#endif /* IIIM_AUX_SO_PATH_ENV */

	if (NULL == aux_file_name) {
		return;
	}

	aux_file_name_len = strlen(aux_file_name);

	/*
	 * may not start with "/"
	 * may not start with "../"
	 * may not contain "/../"
	 * may not end with "/"
	 * may not end with "/."
	 * may not end with "/.."
	 * may not be ".."
	 */

	if (((1 <= aux_file_name_len) &&
	     ('/' == *(aux_file_name + 0))) ||
	    ((3 <= aux_file_name_len) &&
	     ('.' == *(aux_file_name + 0)) &&
	     ('.' == *(aux_file_name + 1)) &&
	     ('/' == *(aux_file_name + 2))) ||
	    (NULL != strstr(aux_file_name, "/../")) ||
	    ((1 <= aux_file_name_len) &&
	     ('/' == *(aux_file_name + aux_file_name_len - 1))) ||
	    ((2 <= aux_file_name_len) &&
	     ('/' == *(aux_file_name + aux_file_name_len - 2)) &&
	     ('.' == *(aux_file_name + aux_file_name_len - 1))) ||
	    ((3 <= aux_file_name_len) &&
	     ('/' == *(aux_file_name + aux_file_name_len - 3)) &&
	     ('.' == *(aux_file_name + aux_file_name_len - 2)) &&
	     ('.' == *(aux_file_name + aux_file_name_len - 1))) ||
	    ((2 == aux_file_name_len) &&
	     ('.' == *(aux_file_name + 0)) &&
	     ('.' == *(aux_file_name + 1)))) {
		return;
	}

	/*
	 * eliminate leading "./"
	 */
	if ((2 <= aux_file_name_len) &&
	    ('.' == *(aux_file_name + 0)) &&
	    ('/' == *(aux_file_name + 1))) {
		aux_file_name += 2;
		aux_file_name_len -= 2;
	}

	dir_name = AUX_BASE_DIR;

#if defined(IIIM_AUX_SO_BASE_ENV)
	{
		char *	p;
		if (NULL != (p = getenv(IIIM_AUX_SO_BASE_ENV))) {
			dir_name = p;
		}
	}
#endif /* IIIM_AUX_SO_BASE_ENV */

	dir_name_len = strlen(dir_name);

#if defined(__sparcv9)
	if (MAXPATHLEN <
	    (dir_name_len + aux_file_name_len + SPARCV9_DIR_LEN + 1)) {
		return;
	}
	p = strrchr(aux_file_name, '/');
	if (NULL == p) {
		p = aux_file_name;
	} else {
		p += 1;
	}
	memcpy(file_name, dir_name, dir_name_len);
	len = (p - aux_file_name);
	memcpy(file_name + dir_name_len, aux_file_name, len);
	len += dir_name_len;
	memcpy(file_name + len, SPARCV9_DIR, SPARCV9_DIR_LEN);
	len += SPARCV9_DIR_LEN;
	memcpy(file_name + len, p, aux_file_name_len - (p - aux_file_name) + 1);

#else /* !__sparcv9 */

	if (MAXPATHLEN <= (dir_name_len + aux_file_name_len + 1)) {
		return;
	}

	memcpy(file_name, dir_name, dir_name_len + 1);
	memcpy(file_name + dir_name_len,
	       aux_file_name, aux_file_name_len + 1);
#endif /* !__sparcv9 */

	(void)aux_load(file_name);

	return;
}


/*
 * IM_AUX_START
 */
static void
start(XicCommon ic, XPointer call_data)
{
	aux_t *		aux;
	int		length;

	TRACE_MESSAGE('A', ("iiimpAux: start\n"));

	if (NULL == (aux = aux_get(ic, call_data))) {
		TRACE_MESSAGE('A', ("iiimpAux: start: failed to get aux\n"));
		return;
	}

	length = service_compose_size(AUX_DATA_START,
				      (unsigned char *)call_data);
	aux->im->ae->dir.method->start(aux,
				       (const unsigned char *)call_data,
				       length);

	return;
}


/*
 * IM_AUX_DRAW
 */
static void
draw(XicCommon ic, XPointer call_data)
{
	int		length;
	aux_t *		aux;

	TRACE_MESSAGE('A', ("iiimpAux: draw\n"));

	if (NULL == (aux = aux_get(ic, call_data))) {
		TRACE_MESSAGE('A', ("iiimpAux: draw: failed to get aux\n"));
		return;
	}

	length = service_compose_size(AUX_DATA_DRAW,
				      (unsigned char *)call_data);
	aux->im->ae->dir.method->draw(aux,
				      (const unsigned char *)call_data,
				      length);
	return;
}


/*
 * IM_AUX_DONE
 */
static void
done(XicCommon ic, XPointer call_data)
{
	int		length;
	aux_t *		aux;

	TRACE_MESSAGE('A', ("iiimpAux: done\n"));

	if (NULL == (aux = aux_get(ic, call_data))) {
		TRACE_MESSAGE('A', ("iiimpAux: done: failed to get aux\n"));
		return;
	}

	length = service_compose_size(AUX_DATA_DONE,
				      (unsigned char *)call_data);
	aux->im->ae->dir.method->done(aux,
				      (const unsigned char *)call_data,
				      length);
	return;
}


/*
 * not defined yet
 */
static void
switched(XicCommon ic, XPointer call_data)
{
	int *		ip;
	int		im_id;
	int		on_off;
	aux_t *		aux;

	TRACE_MESSAGE('A', ("iiimpAux: switched\n"));

	if (NULL == (aux = aux_get(ic, call_data))) {
		TRACE_MESSAGE('A', ("iiimpAux: switched: failed to get aux\n"));
		return;
	}

	ip = (int *)call_data;
	im_id = *(ip + 0);
	on_off = *(ip + 1);

	aux->im->ae->dir.method->switched(aux, im_id, on_off);
	return;
}


/*
 * not defined yet
 */
static void
destroy(XicCommon ic, XPointer call_data)
{
	aux_handle_t *	ah;
	aux_handle_t *	ah_next;
	int		i;

	TRACE_MESSAGE('A', ("iiimpAux: destroy\n"));

	for (ah = aux_handle; NULL != ah; ah = ah_next) {
		for (i = 0; i < ah->ae_num; i++) {
			if (0 == (ah->ae + i)->created) {
				continue;
			}
			(ah->ae + i)->dir.method->destroy(NULL);
			(ah->ae + i)->created = 0;
		}
		ah_next = ah->next;
		Xfree(ah->aux_name.ptr);
		Xfree(ah->file_name);
		dlclose(ah->handle);
		Xfree(ah->ae);
		Xfree(ah);
	}

	aux_handle = NULL;

	return;
}


/*
 *
 */
void
AuxChange(XicCommon xic, AuxChangeReason reason, XPointer call_data)
{
	if ((0 == reason) || (AUX_DESTROY < reason)) {
		TRACE_MESSAGE('A', ("iiimpAux: AuxChange: unknown reason: %d\n", reason));
		return;
	}

#if defined(ENABLE_TRACE)
	if (TRACE_P('A')) {
		int	im_id;
		int	ic_id;
		int	ci;
		int	len;
		char *	p;

		p = (char *)call_data;

		im_id = *((CARD16 *)(p + 0));
		ic_id = *((CARD16 *)(p + 2));
		ci = *((CARD32 *)(p + 4));
		len = *((CARD16 *)(p + 8));
		TRACE_MESSAGE('A', ("AuxChange: %s im=%d ic=%d ci=%d len=%d\n",
				    AuxChangeReasonName[reason],
				    im_id, ic_id, ci, len));
	}
#endif /* ENABLE_TRACE */

	aux_method[reason](xic, call_data);
	return;
}


/*
 *
 */
static aux_handle_t *
aux_load(char * aux_file_name)
{
	int		fd;
	char		buf[64];
	void *		addr;
	int		magic_len;
	int		len;

	TRACE_MESSAGE('A', ("iiimpAux: aux_load\n"));

	if (-1 == (fd = open(aux_file_name, O_RDONLY, 0))) {
		TRACE_MESSAGE('A', ("iiimpAux: aux_load: can not open %s\n",
				    aux_file_name));
		return NULL;
	}

	magic_len = strlen(AUX_CONF_MAGIC);

	len = read(fd, buf, magic_len);

	close(fd);

	if ((len == magic_len) &&
	    (0 == memcmp(buf, AUX_CONF_MAGIC, len))) {
		return aux_conf_load(aux_file_name);
	} else {
		return aux_so_load(aux_file_name);
	}
}


/*
 *
 */
static aux_handle_t *
aux_conf_load(char * aux_file_name)
{
	int		fd;
	struct stat	st_buf;
	void *		addr;
	char *		p;
	char *		pp;
	char *		aux_name;
	int		aux_name_len;
	char *		aux_so;
	int		aux_so_len;
	aux_handle_t *	ah;
	int		len;
	char *		inbuf;
	size_t		inbytesleft;
	char *		outbuf;
	size_t		outbytesleft;
	int		dir_name_len;
	char *		dir_name_last;

	dir_name_last = strrchr(aux_file_name, '/');
	if (NULL == dir_name_last) {
		return NULL;
	}
	dir_name_len = ((dir_name_last - aux_file_name) + 1);

	if (-1 == (fd = open(aux_file_name, O_RDONLY, 0))) {
		return NULL;
	}

	if (0 != fstat(fd, &st_buf)) {
		close(fd);
		return NULL;
	}

	addr = mmap(0, st_buf.st_size,
		    PROT_READ | PROT_WRITE, MAP_PRIVATE, fd, 0);

	close(fd);

	if (MAP_FAILED == addr) {
		return NULL;
	}

	ah = NULL;
	
	for (p = addr, len = st_buf.st_size; 0 < len; ) {
		if ('#' == *p) {
			while (IS_NOT_EOL(len, p)) {
				p++;
				--len;
			}
			if (IS_EOL(len, p)) {
				p++;
				--len;
			}
			continue;
		}

		while (IS_SPACE(len, p)) {
			p++;
			--len;
		}
		if (IS_EOL(len, p)) {
			p++;
			--len;
			continue;
		}

		aux_name = p;
		while ((!(IS_SPACE(len, p))) && IS_NOT_EOL(len, p)) {
			p++;
			--len;
		}
		if (IS_EOL(len, p)) {
			p++;
			--len;
			continue;
		}

		aux_name_len = (p - aux_name);

		while (IS_SPACE(len, p)) {
			p++;
			--len;
		}
		if (IS_EOL(len, p)) {
			p++;
			--len;
			continue;
		}

		aux_so = p;
		while ((!(IS_SPACE(len, p))) && IS_NOT_EOL(len, p)) {
			p++;
			--len;
		}
		aux_so_len = (p - aux_so);

		ah = (aux_handle_t *)Xmalloc(sizeof (aux_handle_t));
		if (NULL == ah) {
			break;
		}

		ah->aux_name.len = (aux_name_len * (sizeof (CARD16)));
		ah->aux_name.ptr = Xmalloc(ah->aux_name.len);
		if (NULL == ah->aux_name.ptr) {
			XFree(ah);
			break;
		}

		inbuf = aux_name;
		inbytesleft = aux_name_len;
		outbuf = (char *)(ah->aux_name.ptr);
		outbytesleft = ah->aux_name.len;
		IIimpConvertToUTF16(inbuf, inbytesleft,
				    &outbuf, &outbytesleft);

		if ('/' == *aux_so) {
			ah->file_name = Xmalloc(aux_so_len + 1);
		} else {
			ah->file_name = Xmalloc(dir_name_len + aux_so_len + 1);
		}
		if (NULL == ah->file_name) {
			Xfree(ah->aux_name.ptr);
			Xfree(ah);
			break;
		}
		if ('/' == *aux_so) {
			memcpy(ah->file_name, aux_so, aux_so_len);
			*(ah->file_name + aux_so_len) = '\0';
		} else {
			memcpy(ah->file_name, aux_file_name, dir_name_len);
			memcpy(ah->file_name + dir_name_len, aux_so, aux_so_len);
			*(ah->file_name + dir_name_len + aux_so_len) = '\0';
		}
		ah->handle = NULL;
		ah->ae = NULL;
		ah->ae_num = 0;
		ah->next = aux_handle;
		aux_handle = ah;
	}

	munmap(addr, st_buf.st_size);

	return ah;
}


/*
 *
 */
static aux_handle_t *
aux_so_load(char * aux_file_name)
{
	void *		handle = (void *)NULL;
	aux_dir_t *	aux_dir;
	aux_dir_t *	ad;
	int		adn;
	aux_handle_t *	ah;
	aux_handle_t *	ah_free;
	int		i;
	char		path[MAXPATHLEN];

	TRACE_MESSAGE('A', ("iiimpAux: aux_so_load\n"));

	/*
	 * check whether the object is already loaded
	 */
	for (ah = aux_handle; NULL != ah; ah = ah->next) {
		if ((0 == strcmp(aux_file_name, ah->file_name)) &&
		    (NULL != ah->handle)) {
			return ah;
		}
	}

	/*
	 * load the object and construct aux_handle_t structure for it
	 */
	handle = dlopen(aux_file_name, RTLD_LAZY);
	if (NULL == handle) {
		TRACE_MESSAGE('A',
			      ("iiimpAux: aux_so_load: failed to dlopen %s\n",
			       aux_file_name));
		return NULL;
	}

	aux_dir = (aux_dir_t *)dlsym(handle, AUX_DIR_SYMBOL);
	if (NULL == aux_dir) {
		dlclose(handle);
		return NULL;
	}

	for (adn = 0, ad = aux_dir; 0 < ad->name.len; ad += 1, adn += 1);

	if (NULL == ah) {
		ah = (aux_handle_t *)Xmalloc(sizeof (aux_handle_t));
		if (NULL == ah) {
			dlclose(handle);
			return NULL;
		}
		memset(ah, 0, sizeof (aux_handle_t));

		ah_free = ah;
	} else {
		ah_free = NULL;
	}

	if (NULL == ah->file_name) {
		ah->file_name = strdup(aux_file_name);
		if (NULL == ah->file_name) {
			Xfree(ah);
			dlclose(handle);
			return NULL;
		}
	}
	ah->handle = handle;
	ah->ae_num = adn;
	ah->ae = Xmalloc(adn * (sizeof (aux_entry_t)));
	if (NULL == ah->ae) {
		if (NULL != ah_free) {
			Xfree(ah->file_name);
			Xfree(ah);
		}
		dlclose(handle);
		return NULL;
	}
	for (i = 0; i < adn; i++) {
		(ah->ae + i)->created = 0;
		memcpy(&((ah->ae + i)->dir), aux_dir + i, sizeof (aux_dir_t));
	}

	ah->next = aux_handle;
	aux_handle = ah;

	return ah;
}


static aux_t *
aux_get(XicCommon ic, XPointer call_data)
{
	int		im_id;
	int		ic_id;
	CARD16 *	aux_name;
	int		aux_name_length;
	aux_t *		aux;
	aux_t *		aux_free;
	aux_entry_t *	ae;
	aux_im_data_t *	aux_im;

	TRACE_MESSAGE('A', ("iiimpAux: aux_get\n"));

	im_id = *((CARD16 *)(call_data + 0));
	ic_id = *((CARD16 *)(call_data + 2));
	aux_name_length = *((CARD16 *)(call_data + 8));
	aux_name = (CARD16 *)(call_data + 10);

#if defined(ENABLE_TRACE)
	if (TRACE_P('s')) {
		int	i;
		int	l;
		l = (aux_name_length / 2);
		TRACE_MESSAGE('s', ("iiimpAux: aux_get: "));
		for (i = 0; i < l; i++) {
			TRACE_MESSAGE('s', ("%04x ", *(aux_name + i)));
		}
		TRACE_MESSAGE('s', ("\n"));
	}
#endif /* ENABLE_TRACE */

	/*
	 * create aux object if it is not created
	 */
	aux = XIC_IIIMP(ic, aux);
	aux_free = NULL;
	if (NULL == aux) {
		if (NULL == (aux = (aux_t *)Xmalloc(sizeof (aux_t)))) {
		  return NULL;
		}
		aux_free = aux;
		aux->ic = ic;
		aux->service = &aux_service;
		aux->im = NULL;
		aux->im_list = NULL;
		XIC_IIIMP(ic, aux) = aux;
	}

	/*
	 * search for aux_im_data object of im_id
	 */
	for (aux_im = aux->im_list; NULL != aux_im; aux_im = aux_im->next) {
		if ((im_id == aux_im->im_id) &&
		    (aux_name_length == aux_im->ae->dir.name.len) && 
		    (0 == memcmp(aux_name, aux_im->ae->dir.name.ptr,
				 aux_im->ae->dir.name.len))) {
			aux->im = aux_im;
			break;
		}
	}

	/*
	 * if no aux_im_data object for im_id is found, create it
	 */
	if (NULL == aux_im) {
		ae = aux_entry_get(aux_name, aux_name_length);
		if (NULL == ae) {
			Xfree(aux_free);
			XIC_IIIMP(ic, aux) = NULL;
			return NULL;
		}

		aux_im = (aux_im_data_t *)Xmalloc(sizeof (aux_im_data_t));
		if (NULL == aux_im) {
			XIC_IIIMP(ic, aux) = NULL;
			Xfree(aux_free);
			return NULL;
		}
		aux_im->im_id = im_id;
		aux_im->ic_id = ic_id;
		aux_im->ae = ae;
		aux_im->data = NULL;
		aux_im->next = aux->im_list;
		aux->im_list = aux_im;
		aux->im = aux_im;

		if (0 == ae->created) {
			if (False == ae->dir.method->create(aux)) {
				XIC_IIIMP(ic, aux) = NULL;
				Xfree(aux_im);
				Xfree(aux_free);
				return NULL;
			}
			ae->created = 1;
		}
	}
	return aux;
}

/*
 *
 */
static aux_entry_t *
aux_entry_get(const CARD16 * name, int length)
{
	aux_handle_t *	ah;
	aux_handle_t *	ah0;
	aux_entry_t *	ae;
	int		i;

	TRACE_MESSAGE('H', ("iiimpAux: aux_entry_get\n"));

	if (0 == length) {
		return NULL;
	}

	for (ah = aux_handle; NULL != ah; ah = ah->next) {
		if ((length == ah->aux_name.len) &&
		    (0 == memcmp(name, ah->aux_name.ptr, length))) {
			ah0 = aux_so_load(ah->file_name);
			if (NULL == ah0) {
				continue;
			} else {
				ah = ah0;
			}
		}
		for (ae = ah->ae, i = ah->ae_num; 0 < i; ae += 1, --i) {
			if ((length == ae->dir.name.len) &&
			    (0 == memcmp(name, ae->dir.name.ptr, length))) {
				return ae;
			}
		}
	}

	return NULL;
}


/*
 *
 */
static aux_im_data_t *
aux_im_get(XicCommon ic, int im_id)
{
	aux_im_data_t *	im;
	aux_t *		aux;

	TRACE_MESSAGE('I', ("iiimpAux: aux_im_get\n"));
	
	aux = XIC_IIIMP(ic, aux);

	for (im = aux->im_list; NULL != im; im = im->next) {
		if (im_id == im->im_id) {
			aux->im = im;
			return im;
		}
	}

	if (NULL == (im = (aux_im_data_t *)Xmalloc(sizeof (aux_im_data_t)))) {
		return NULL;
	}

	im->ae = NULL;
	im->data = NULL;
	im->next = aux->im_list;
	aux->im_list = im;
	aux->im = im;

	return im;
}


/*
 * aux service function
 */

static void
service_aux_setvalue(aux_t * aux, const unsigned char * p, int len)
{
	IMAuxSetValues(aux->ic, (unsigned char *)p, len, NULL);
}


static int
service_im_id(aux_t * aux)
{
	return aux->im->im_id;
}


static int
service_ic_id(aux_t * aux)
{
	return aux->im->ic_id;
}


static void
service_data_set(aux_t * aux, int im_id, void * data)
{
	aux_im_data_t *	aux_im;

	for (aux_im = aux->im; NULL != aux_im; aux_im = aux_im->next) {
		if (im_id == aux_im->im_id) {
			aux_im->data = data;
		}
	}

	return;
}


static void *
service_data_get(aux_t * aux, int im_id)
{
	aux_im_data_t *	aux_im;

	for (aux_im = aux->im; NULL != aux_im; aux_im = aux_im->next) {
		if (im_id == aux_im->im_id) {
			return aux_im->data;
		}
	}

	return NULL;
}


static Display *
service_display(aux_t * aux)
{
	return aux->ic->core.im->core.display;
}


static Window
service_window(aux_t * aux)
{
	if (XIMP_CHK_FOCUSWINMASK(aux->ic)) {
		return aux->ic->core.focus_window;
	} else if (aux->ic->ximp_icpart->value_mask & XIMP_CLIENT_WIN) {
		return aux->ic->core.client_window;
	} else {
		return None;
	}
}


static XPoint *
service_point(aux_t * aux, XPoint * point)
{
	if (IS_PREEDIT_POSITION(aux->ic)) {
		point->x = aux->ic->core.preedit_attr.spot_location.x;
		point->y = aux->ic->core.preedit_attr.spot_location.y;
	} else {
		point->x = -1;
		point->y = -1;
	}

	return point;
}


static XPoint *
service_point_caret(aux_t * aux, XPoint * point)
{
	point->x = -1;
	point->y = -1;

	if (!IS_PREEDIT_POSITION(aux->ic)) {
		return point;
	}

	PreeditCaretPlacementRelative(aux->ic, point);

	if ((-1 == point->x) && (-1 == point->y)) {
		return service_point(aux, point);
	}

	return point;
}


static size_t
service_utf16_mb(
	const char **	inbuf,
	size_t *	inbytesleft,
	char **		outbuf,
	size_t *	outbytesleft)
{
	int	r;

	r = IIimpConvertFromUTF16((char *)(*inbuf), *inbytesleft,
				  outbuf, outbytesleft);
	return (size_t)r;
}


static size_t
service_mb_utf16(
	const char **	inbuf,
	size_t *	inbytesleft,
	char **		outbuf,
	size_t *	outbytesleft)
{
	int	r;

	r = IIimpConvertToUTF16((char *)(*inbuf), *inbytesleft,
				  outbuf, outbytesleft);
	return (size_t)r;
}


static int		padding[4] = {0, 3, 2, 1};


static unsigned char *
service_compose(const aux_data_t * aux_data, int * size)
{
	int		total_size;
	int		i;
	int		j;
	int		l;
	int		string_list_byte;
	unsigned char *	aux_data_area;
	unsigned char *	p;

	if (NULL == aux_data) {
		return NULL;
	}

	total_size = 2 + 2 + 4 + 2;
	total_size += (aux_data->aux_name_length +
		       padding[(2 + aux_data->aux_name_length) %4]);
	switch (aux_data->type) {
	case AUX_DATA_DRAW:
	case AUX_DATA_SETVALUE:
		total_size += 4;
		total_size += ((sizeof (CARD32)) * aux_data->integer_count);
		total_size += 4;
		for (i = 0, string_list_byte = 0;
		     i < aux_data->string_count; i++) {
			l = (aux_data->string_list + i)->length;
			string_list_byte += (2 + l + padding[(2 + l) %4]);
		}
		total_size += string_list_byte;
		break;
	}

	aux_data_area = (unsigned char *)Xmalloc(total_size);
	if (NULL == aux_data_area) {
		return NULL;
	}

	p = aux_data_area;

	*((CARD16 *)p) = aux_data->im;
	p += 2;
	*((CARD16 *)p) = aux_data->ic;
	p += 2;
	*((CARD32 *)p) = aux_data->aux_index;
	p += 4;
	*((CARD16 *)p) = aux_data->aux_name_length;
	p += 2;
	memcpy(p, aux_data->aux_name, aux_data->aux_name_length);
	p += aux_data->aux_name_length;
	for (i = padding[(2 + aux_data->aux_name_length) %4]; 0 < i; --i) {
		*(p++) = 0x00;
	}

	switch (aux_data->type) {
	case AUX_DATA_DRAW:
	case AUX_DATA_SETVALUE:
		l = ((sizeof (CARD32)) * aux_data->integer_count);
		*((CARD32 *)p) = l;
		p += 4;
		memcpy(p, aux_data->integer_list, l);
		p += l;
		*((CARD32 *)p) = string_list_byte;
		p += 4;
		for (i = 0; i < aux_data->string_count; i++) {
			l = (aux_data->string_list + i)->length;
			*((CARD16 *)p) = l;
			p += 2;
			memcpy(p, (aux_data->string_list + i)->ptr, l);
			p += l;
			for (j = padding[(2 + l) %4]; 0 < j; --j) {
				*(p++) = 0x00;
			}
		}
		break;
	}

	if (NULL != size) {
		*size = total_size;
	}

	return aux_data_area;
}


static int
service_compose_size(aux_data_type_t type, const unsigned char *p)
{
	int			aux_name_length;
	int			integer_list_byte;
	int			string_list_byte;
	const unsigned char *	pp;

	if (NULL == p) {
		return 0;
	}

	pp = p;

	pp += (2 + 2 + 4);
	aux_name_length = *((CARD16 *)pp);
	pp += 2;
	aux_name_length += padding[(2 + aux_name_length) %4];
	pp += aux_name_length;

	if (AUX_DATA_DRAW != type) {
		return (pp - p);
	}

	integer_list_byte = *((CARD32 *)pp);
	pp += 4;
	pp += integer_list_byte;

	string_list_byte = *((CARD32 *)pp);
	pp += 4;
	pp += string_list_byte;

	return (pp - p);
}


static aux_data_t *
service_decompose(aux_data_type_t type, const unsigned char * p)
{
	int			integer_list_byte;
	int			string_list_byte;
	aux_data_t *		aux_data;
	int			len;
	int			i;
	const unsigned char *	pp;
	unsigned char *		ptr;

	aux_data = (aux_data_t *)Xmalloc(sizeof (aux_data_t));
	if (NULL == aux_data) {
		return NULL;
	}

	aux_data->type = type;

	aux_data->im = *((CARD16 *)p);
	p += 2;

	aux_data->ic = *((CARD16 *)p);
	p += 2;

	aux_data->aux_index = *((CARD32 *)p);
	p += 4;

	aux_data->aux_name_length = *((CARD16 *)p);
	p += 2;

	aux_data->aux_name =
		(unsigned char *)Xmalloc(aux_data->aux_name_length);
	if (NULL == aux_data->aux_name) {
		Xfree(aux_data);
		return NULL;
	}
	memcpy(aux_data->aux_name, p, aux_data->aux_name_length);
	p += (aux_data->aux_name_length +
		padding[(2 + aux_data->aux_name_length) %4]);

	if (AUX_DATA_DRAW != type) {
		aux_data->integer_count = 0;
		aux_data->integer_list = NULL;
		aux_data->string_count = 0;
		aux_data->string_list = NULL;
		aux_data->string_ptr = NULL;

		return aux_data;
	}

	integer_list_byte = *((CARD32 *)p);
	p += 4;
	if (0 != (integer_list_byte % 4)) {
		Xfree(aux_data->aux_name);
		Xfree(aux_data);
		return NULL;
	}
	aux_data->integer_count = (integer_list_byte / 4);

	aux_data->integer_list = (int *)Xmalloc(integer_list_byte);
	if (NULL == aux_data->integer_list) {
		Xfree(aux_data->aux_name);
		Xfree(aux_data);
		return NULL;
	}
	memcpy(aux_data->integer_list, p, integer_list_byte);
	p += integer_list_byte;

	string_list_byte = *((CARD32 *)p);
	p += 4;

	for (i = 0, pp = p; pp < (p + string_list_byte); i++) {
		len = *((CARD16 *)pp);
		pp += (2 + len + padding[(2 + len) %4]);
	}
	aux_data->string_count = i;

	aux_data->string_list =
		(aux_string_t *)Xmalloc((sizeof (aux_string_t)) *
					aux_data->string_count);
	if (NULL == aux_data->string_list) {
		Xfree(aux_data->aux_name);
		Xfree(aux_data->integer_list);
		Xfree(aux_data);
		return NULL;
	}
	aux_data->string_ptr = (unsigned char *)Xmalloc(string_list_byte);
	if (NULL == aux_data->string_ptr) {
		Xfree(aux_data->aux_name);
		Xfree(aux_data->integer_list);
		Xfree(aux_data->string_list);
		Xfree(aux_data);
		return NULL;
	}
	memcpy(aux_data->string_ptr, p, string_list_byte);

	for (i = 0, ptr = aux_data->string_ptr;
	     ptr < (aux_data->string_ptr + string_list_byte);
	     i++) {
		len = *((CARD16 *)ptr);
		(aux_data->string_list + i)->length = len;
		ptr += 2;

		(aux_data->string_list + i)->ptr = (unsigned char *)ptr;
		ptr += (len + padding[(2 + len) %4]);
	}

	return aux_data;
}


static void
service_decompose_free(aux_data_t * aux_data)
{
	if (NULL == aux_data) {
		return;
	}

	Xfree(aux_data->aux_name);
	if(aux_data->integer_list) Xfree(aux_data->integer_list);
	if(aux_data->string_list) Xfree(aux_data->string_list);
	if(aux_data->string_ptr) Xfree(aux_data->string_ptr);
	Xfree(aux_data);

	return;
}


static void
service_register_X_filter(
	Display *	display,
	Window		window,
	int		start_type,
	int		end_type,
	Bool		(* filter)(Display *, Window, XEvent *, XPointer),
	XPointer	client_data)
{
	_XRegisterFilterByType(display, window, start_type, end_type,
			       filter, client_data);

	return;
}


static void
service_unregister_X_filter(
	Display *	display,
	Window		window,
	Bool		(* filter)(Display *, Window, XEvent *, XPointer),
	XPointer	client_data)
{
	_XUnregisterFilter(display, window, filter, client_data);
}


static Bool
service_server(aux_t * aux)
{
	char *	client_type;

	client_type = XIM_IIIMP(aux->ic->core.im, client_type);

	return ((0 == strcmp(client_type, "Htt XIM Server")) ? True : False);
}


static Window
service_client_window(aux_t * aux)
{
	/* ic is initialized to zero in CreateIC().
	 * there is no need not to test
	 * (aux->ic->ximp_icpart->value_mask & XIMP_CLIENT_WIN)
	 */
	return aux->ic->core.client_window;
}


static Window
service_focus_window(aux_t * aux)
{
	/* ic is initialized to zero in CreateIC().
	 * there is no need not to test
	 * XIMP_CHK_FOCUSWINMASK(aux->ic)
	 */
	return aux->ic->core.focus_window;
}


static int
service_screen_number(aux_t * aux)
{
	return XIC_GUI(aux->ic, screen_number);
}


static int
service_point_screen(aux_t * aux, XPoint * point)
{
	Display *	display;
	Window		window;
	Window		root_window;
	Window		child;
	int		screen_number;
	int		x;
	int		y;
	int		new_x;
	int		new_y;

	display = aux->ic->core.im->core.display;

	screen_number = XIC_GUI(aux->ic, screen_number);

	if (!IS_PREEDIT_POSITION(aux->ic)) {
		point->x = -1;
		point->y = -1;
		return screen_number;
	}

	if (XIMP_CHK_FOCUSWINMASK(aux->ic)) {
		window =  aux->ic->core.focus_window;
	} else if (aux->ic->ximp_icpart->value_mask & XIMP_CLIENT_WIN) {
		window = aux->ic->core.client_window;
	} else {
		return -1;
	}

	root_window = RootWindow(display, screen_number);

	x = aux->ic->core.preedit_attr.spot_location.x;
	y = aux->ic->core.preedit_attr.spot_location.y;

	XTranslateCoordinates(display, window, root_window,
			      x, y,
			      &new_x, &new_y, &child);

	point->x = new_x;
	point->y = new_y;

	return screen_number;
}


static int
service_point_caret_screen(aux_t * aux, XPoint * point)
{
	point->x = -1;
	point->y = -1;

	if (IS_PREEDIT_POSITION(aux->ic)) {

		PreeditCaretPlacement(aux->ic, point);

		if ((-1 == point->x) && (-1 == point->y)) {
			return service_point_screen(aux, point);
		}
	}

	return XIC_GUI(aux->ic, screen_number);
}
