/*
** tifftopnm.c - converts a Tagged Image File to a portable anymap
**
** Derived by Jef Poskanzer from tif2ras.c, which is:
**
** Copyright (c) 1990 by Sun Microsystems, Inc.
**
** Author: Patrick J. Naughton
** naughton@wind.sun.com
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation.
**
** This file is provided AS IS with no warranties of any kind.  The author
** shall have no liability with respect to the infringement of copyrights,
** trade secrets or any patents by this file or any part thereof.  In no
** event will the author be liable for any lost revenue or profits or
** other special, indirect and consequential damages.
*/

/* Implementation note:

  This probably ought to be reimplemented using TIFFRGBAImageGet() 
  from the Tiff library, because that routine automatically decodes
  anything and everything that might be in a Tiff image file now or
  in the future.

  Much of the code below is basically copied from the Tiff library
  because we use the more primitive TIFFReadScanLine() access method.

  -Bryan 00.03.05
*/

#include "pnm.h"
#ifdef VMS
#ifdef SYSV
#undef SYSV
#endif
#include <tiffioP.h>
#endif
#include <tiffio.h>

#define MAXCOLORS 1024
#ifndef PHOTOMETRIC_DEPTH
#define PHOTOMETRIC_DEPTH 32768
#endif

static void
pick_rgba_pixel(xelval * const r_p, xelval * const b_p, xelval * const g_p);
static void
pick_cmyk_pixel(xelval * const r_p, xelval * const b_p, xelval * const g_p);

/* The following form a cursor into the raster buffer */
static u_char* inP;
static int bitsleft;

/* The following describe the structure of the raster buffer */
static int maxval;
static unsigned short bps, spp;

/* The NEXTSAMPLE macro effects the reading of a sample from the raster
   buffer via the above cursor into the variable 'sample'.
*/
#define NEXTSAMPLE \
    { \
    if ( bitsleft == 0 ) \
	{ \
	++inP; \
	bitsleft = 8; \
	} \
    bitsleft -= bps; \
    sample = ( *inP >> bitsleft ) & maxval; \
    }


void
main( argc, argv )
    int argc;
    char* argv[];
    {
    int argn, cols, rows, grayscale, format;
    int numcolors;
    register TIFF* tif;
    int row, i;
    register int col;
    u_char* buf;
    register xel* xP;
    xel* xelrow;
    xel colormap[MAXCOLORS];
    int headerdump;
    register u_char sample;
    unsigned short photomet, planarconfig, inkset;
    unsigned short* redcolormap;
    unsigned short* greencolormap;
    unsigned short* bluecolormap;
    char* usage = "[-headerdump] [tifffile]";


    pnm_init( &argc, argv );

    argn = 1;
    headerdump = 0;

    if ( argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0' )
	{
	if ( pm_keymatch( argv[argn], "-headerdump", 2 ) )
	    headerdump = 1;
	else
	    pm_usage( usage );
	++argn;
	}

    if ( argn != argc )
	{
	tif = TIFFOpen( argv[argn], "r" );
	if ( tif == NULL )
	    pm_error( "error opening TIFF file %s", argv[argn] );
	++argn;
	}
    else
	{
	tif = TIFFFdOpen( 0, "Standard Input", "r" );
	if ( tif == NULL )
	    pm_error( "error opening standard input as TIFF file" );
	}

    if ( argn != argc )
	pm_usage( usage );

    if ( headerdump )
	TIFFPrintDirectory( tif, stderr, TIFFPRINT_NONE );

    if ( ! TIFFGetField( tif, TIFFTAG_BITSPERSAMPLE, &bps ) )
	bps = 1;
    if ( ! TIFFGetField( tif, TIFFTAG_SAMPLESPERPIXEL, &spp ) )
	spp = 1;
    if ( ! TIFFGetField( tif, TIFFTAG_PHOTOMETRIC, &photomet ) )
	pm_error( "error getting photometric" );
    if( spp > 1 ){
	if ( ! TIFFGetField( tif, TIFFTAG_PLANARCONFIG, &planarconfig ) )
	    pm_error( "error getting planarconfig" );
    }else{
	planarconfig = PLANARCONFIG_CONTIG;
    }


    switch ( spp )
	{
	case 1:
	case 3:
	case 4:
	break;

	default:
	pm_error(
	    "can only handle 1-channel gray scale or 1- or 3-channel color" );
	}

    switch( planarconfig )
	{
	case PLANARCONFIG_CONTIG:
	    break;
	case PLANARCONFIG_SEPARATE:
	    if (photomet != PHOTOMETRIC_RGB && photomet != PHOTOMETRIC_SEPARATED)
		pm_error( "can only handle separate planes "
                 "with RGB or SEPARATED data" );
	    break;
	default:
	    pm_error("Unrecongnized PLANARCONFIG tag!\n");
	}

    (void) TIFFGetField( tif, TIFFTAG_IMAGEWIDTH, &cols );
    (void) TIFFGetField( tif, TIFFTAG_IMAGELENGTH, &rows );

    if ( headerdump )
	{
	pm_message( "%dx%dx%d image", cols, rows, bps * spp );
	pm_message( "%d bits/sample, %d samples/pixel", bps, spp );
	}

    maxval = ( 1 << bps ) - 1;
    if ( maxval == 1 && spp == 1 ) {
        if ( headerdump )
          pm_message("monochrome" );
        grayscale = 1;
	} else {
        /* How come we don't deal with the photometric for the monochrome 
           case (make sure it's one we know)?  -Bryan 00.03.04
           */
        switch ( photomet ) {
          case PHOTOMETRIC_MINISBLACK:
            if ( headerdump )
              pm_message( "%d graylevels (min=black)", maxval + 1 );
            grayscale = 1;
            break;

          case PHOTOMETRIC_MINISWHITE:
            if ( headerdump )
              pm_message( "%d graylevels (min=white)", maxval + 1 );
            grayscale = 1;
            break;

          case PHOTOMETRIC_PALETTE:
            if ( headerdump )
              pm_message( "colormapped" );
            if ( ! TIFFGetField( tif, TIFFTAG_COLORMAP, 
                                &redcolormap, &greencolormap, &bluecolormap ) )
              pm_error( "error getting colormaps" );
            numcolors = maxval + 1;
            if ( numcolors > MAXCOLORS )
              pm_error( "too many colors" );
            maxval = PNM_MAXMAXVAL;
            grayscale = 0;
            for ( i = 0; i < numcolors; ++i ) {
                register xelval r, g, b;
                r = (long) redcolormap[i] * PNM_MAXMAXVAL / 65535L;
                g = (long) greencolormap[i] * PNM_MAXMAXVAL / 65535L;
                b = (long) bluecolormap[i] * PNM_MAXMAXVAL / 65535L;
                PPM_ASSIGN( colormap[i], r, g, b );
            }
            break;

          case PHOTOMETRIC_SEPARATED:
            if ( headerdump )
              pm_message( "color separation" );
            if ( TIFFGetField( tif, TIFFTAG_INKNAMES, &inkset ) == 1
                && inkset != INKSET_CMYK)
            if (inkset != INKSET_CMYK) 
               pm_error("This color separation file uses an inkset (%d) ",
                        "we can't handle.  We handle only CMYK.", inkset);
            if (spp != 4) 
              pm_error("This CMYK color separation file is %d samples per "
                       "pixel.  We need 4 samples, though: C, M, Y, and K.  ",
                       spp);
            grayscale = 0;
            break;
            
          case PHOTOMETRIC_RGB:
            if ( headerdump )
              pm_message( "RGB truecolor" );
            grayscale = 0;
            break;

          case PHOTOMETRIC_MASK:
            pm_error( "don't know how to handle PHOTOMETRIC_MASK" );

          case PHOTOMETRIC_DEPTH:
            pm_error( "don't know how to handle PHOTOMETRIC_DEPTH" );

          case PHOTOMETRIC_YCBCR:
            pm_error( "don't know how to handle PHOTOMETRIC_YCBCR" );

          case PHOTOMETRIC_CIELAB:
            pm_error( "don't know how to handle PHOTOMETRIC_CIELAB" );

          case PHOTOMETRIC_LOGL:
            pm_error( "don't know how to handle PHOTOMETRIC_LOGL" );

          case PHOTOMETRIC_LOGLUV:
            pm_error( "don't know how to handle PHOTOMETRIC_LOGLUV" );

          default:
            pm_error( "unknown photometric: %d", photomet );
	    }
	}
    if ( maxval > PNM_MAXMAXVAL )
	pm_error("bits/sample is too large - "
             "try reconfiguring Tifftopnm with PGM_BIGGRAYS\n"
             "    or without PPM_PACKCOLORS" );

    if ( grayscale )
	{
	if ( maxval == 1 )
	    {
	    format = PBM_TYPE;
	    pm_message( "writing PBM file" );
	    }
	else
	    {
	    format = PGM_TYPE;
	    pm_message( "writing PGM file" );
	    }
	}
    else
	{
	format = PPM_TYPE;
	pm_message( "writing PPM file" );
	}

    buf = (u_char*) malloc(TIFFScanlineSize(tif));
    if ( buf == NULL )
	pm_error( "can't allocate memory for scanline buffer" );

    pnm_writepnminit( stdout, cols, rows, (xelval) maxval, format, 0 );
    xelrow = pnm_allocrow( cols );

    for ( row = 0; row < rows; ++row ) {
        if ( TIFFReadScanline( tif, buf, row, 0 ) < 0 )
          pm_error( "bad data read on line %d", row );

        /* Set readout cursor to beginning of raster buffer */
        inP = buf;
        bitsleft = 8;

        /* Position to beginning of pnm output buffer */
        xP = xelrow;

        switch ( photomet ) {
          case PHOTOMETRIC_MINISBLACK:
            for ( col = 0; col < cols; ++col, ++xP ) {
                NEXTSAMPLE
                PNM_ASSIGN1( *xP, sample );
            }
            break;

          case PHOTOMETRIC_MINISWHITE:
            for ( col = 0; col < cols; ++col, ++xP ) {
                NEXTSAMPLE
                sample = maxval - sample;
                PNM_ASSIGN1( *xP, sample );
            }
            break;

          case PHOTOMETRIC_PALETTE:
            for ( col = 0; col < cols; ++col, ++xP ) {
                NEXTSAMPLE
                *xP = colormap[sample];
            }
            break;

          case PHOTOMETRIC_SEPARATED:
            if (planarconfig == PLANARCONFIG_CONTIG) {
                for ( col = 0; col < cols; ++col, ++xP ) {
                    xelval r, g, b;
                    pick_cmyk_pixel(&r, &b, &g);
            
                    PPM_ASSIGN( *xP, r, g, b );
                }
            } else 
                pm_error("Can't handle multiple plane color separation file");
            break;

          case PHOTOMETRIC_RGB:
            if( planarconfig == PLANARCONFIG_CONTIG ) {
                for ( col = 0; col < cols; ++col, ++xP ) {
                    xelval r, g, b;
                    
                    pick_rgba_pixel(&r, &b, &g);
            
                    PPM_ASSIGN( *xP, r, g, b );
                }
            } else {
                /* The input is in separate planes, so the scan line in the 
                   buffer now is just the red.  We need to read twice more
                   to get the green and blue.
                   */
                
                /* First clear the value and assign the reds */
                for ( col = 0; col < cols; ++col, ++xP ) {
                    PPM_ASSIGN( *xP, 0, 0, 0 );
                    NEXTSAMPLE
                    PPM_PUTR( *xP, sample );
                }
                
                /* Next the greens */
                if ( TIFFReadScanline( tif, buf, row, 1 ) < 0 )
                  pm_error( "bad data read on green line %d", row );
                xP = xelrow;
                inP = buf;
                bitsleft = 8;
                for ( col = 0; col < cols; ++col, ++xP ) {
                    NEXTSAMPLE
                    PPM_PUTG( *xP, sample );
                }

                /* And finally the blues */
                if ( TIFFReadScanline( tif, buf, row, 2 ) < 0 )
                  pm_error( "bad data read on green line %d", row );
                xP = xelrow;
                inP = buf;
                bitsleft = 8;
                for ( col = 0; col < cols; ++col, ++xP ) {
                    NEXTSAMPLE
                    PPM_PUTB( *xP, sample );
                }
            }		
            break;
            
          default:
            pm_error( "internal error:  unknown photometric in the picking "
                     "routine: %d", photomet );
        }
        pnm_writepnmrow( stdout, xelrow, cols, (xelval) maxval, format, 0 );
    }

    pm_close( stdout );
    exit( 0 );
    }



static void
pick_rgba_pixel(xelval * const r_p, xelval * const b_p, xelval * const g_p) {

    register u_char sample;

    NEXTSAMPLE
    *r_p = sample;
    NEXTSAMPLE
    *g_p = sample;
    NEXTSAMPLE
    *b_p = sample;
    if ( spp == 4 )
      NEXTSAMPLE /* skip alpha channel */
}


static void
pick_cmyk_pixel(xelval * const r_p, xelval * const b_p, xelval * const g_p) {

    register u_char sample;
    register u_char c, y, m, k;
    

    NEXTSAMPLE
    c = sample;
    NEXTSAMPLE
    y = sample;
    NEXTSAMPLE
    m = sample;
    NEXTSAMPLE
    k = sample;

    /* I don't know why this works.  It seems to me that yellow pigment should
       translate into both red and green light, not just green as you see here.
       But this is the formula used by TIFFRGBAImageGet() in the Tiff library.
       */
    
    *r_p = ((255-k)*(255-c))/255;
    *g_p = ((255-k)*(255-y))/255;
    *b_p = ((255-k)*(255-m))/255;
}


