/* **************************************************************************
 *
 *	net.c	Network Send and Recieve Handlers
 *	-----------------------------------------
 *	Functions:
 *
 *      int NetSendDataToConnection(int condescriptor, char *outbounddata,
 *		int priority)
 *      void NetCloseConnection(long condescriptor)
 *      void NetClearAllConnections()
 *      long NetGetNewDescriptor()
 *      int NetPollSocket(int sockettopoll)
 *      int NetSendRecycleObject(long condescriptor, long object_num)
 *      int NetHandleLogin(long condescriptor, char *arg)
 *	int NetHandleLogout(long condescriptor) 
 *      int NetHandleGlobalMessage(int sendercondescriptor, char *message)
 *      int NetSendRefresh(long condescriptor)
 *      int NetHandleSetInterval(long condescriptor, char *arg)
 *      int NetSendObjectPosition(long condescriptor, long object_num)
 *      int NetSendObjectValues(long condescriptor, long object_num)
 *      int NetSendObjectMaximums(long condescriptor, long object_num)
 *      int NetSendWhoAmI(long condescriptor)
 *      int NetSendEchoToConnection(long condescriptor, char *arg)
 *      int NetHandleSetObjApp(long condescriptor, char *arg)
 *      int NetHandleSetObjAppS(long condescriptor, char *arg)
 *      int NetHandleSetWeapon(long condescriptor, char *arg)
 *      int NetHandleSetShields(long condescriptor, char *arg)
 *      int NetHandleSetCloak(long condescriptor)
 *      int NetHandleSetDmgCtl(long condescriptor)
 *      int NetSendSetName(long condescriptor, long object_num)
 *      int NetSendWeaponValues(long condescriptor, long object_num,
 *		int weapon_num)
 *      int NetHandleSetWeaponsLock(long condescriptor)
 *      int NetHandleSetWeaponsUnlock(long condescriptor)
 *      int NetHandleFireWeapon(long condescriptor, char *arg)
 *      int NetPollSocketRecvData(long condescriptor)
 *
 *	--- 
 *
 * 
 *
 * 
 *
 *
 */

#include "../include/swserv.h"



int NetSendDataToConnection(int condescriptor, char *outbounddata, int priority)
{
	/* Local variables. */
        char stringa[256];
	long object_num;
	long bytes_sent;
        struct timeval timeout;
        fd_set writefds;
	int consocket;
	long conbuf_items;


	/* Make sure outbounddata strlen() is less than MAX_NET_CMD_SIZE. */
	if(strlen(outbounddata) >= (MAX_NET_CMD_SIZE - 1) )
	{
	    sprintf(stringa,
  "NetSendDataToConnection(): outbounddata exceeds %i bytes length, not sent.",
		MAX_NET_CMD_SIZE
            );
            LogAppendLineFormatted(primary_log_fullname, stringa);
	    return(-1);
	}

	/* Make sure there is a newline character. */
	if(strchr(outbounddata, '\n') == NULL)
	{
            sprintf(stringa,
  "NetSendDataToConnection(): outbounddata missing newline character, not sent."
            );
            LogAppendLineFormatted(primary_log_fullname, stringa);
            sprintf(stringa,
                "NetSendDataToConnection(): `%s'",
                outbounddata
            );
            LogAppendLineFormatted(primary_log_fullname, stringa);
 

            return(-1);
	}

        /* Make sure condescriptor is valid. */
        if((condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS))
                return(-1);

	/* Get connection's socket. */
	consocket = connection[condescriptor].socket;
	if(consocket < 0)
		return(-1);


	/* Set bytes_sent to 0. */
	bytes_sent = 0;


	/* Try to send queued data in connection's buffer if any. */
	conbuf_items = connection[condescriptor].conbuf_items;
	if(conbuf_items > MAX_CON_BUF_ITEMS)
		conbuf_items = MAX_CON_BUF_ITEMS;
        if(conbuf_items < 0)
                conbuf_items = 0;
	while(conbuf_items > 0)
	{
	    usleep(50000);

	    errno = 0;

            timeout.tv_sec = 0;
            timeout.tv_usec = 0;
            FD_ZERO(&writefds);
            FD_SET(consocket, &writefds);
            select(consocket + 1, NULL, &writefds, NULL, &timeout);
            /* Check if socket is connected and valid. */
            if(errno == EBADF)
            {
                errno = 0;

		/* Log lost connection. */
                sprintf(stringa,
 "NetSendDataToConnection() Connection %i: Socket invalid and no longer used.",
                      condescriptor
                );
		LogAppendLineFormatted(primary_log_fullname, stringa);

                connection[condescriptor].socket = -1;
                connection[condescriptor].object_num = -1;
                return(-1);
            }
            /* Socket is valid to send. */
            if(FD_ISSET(consocket, &writefds))
            {
		bytes_sent +=
                    send(consocket,
                        connection[condescriptor].conbuf[conbuf_items - 1].buffer,
         strlen(connection[condescriptor].conbuf[conbuf_items - 1].buffer) + 1,
			0
		    );

if(bytes_sent > 0)
fprintf(stderr, "Buffer line %i flushed: \"%s\"\n", conbuf_items,
    connection[condescriptor].conbuf[conbuf_items - 1].buffer);

		/* Log buffer flush. */
		object_num = connection[condescriptor].object_num;
		if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
		{
		    sprintf(stringa,
             "*Not Logged In*: Connection %i: Buffer line %i flushed: \"%s\"",
			condescriptor,
			conbuf_items,
			connection[condescriptor].conbuf[conbuf_items - 1].buffer
		    );
		}
		else
		{
		    sprintf(stringa,
                     "%s(#%i): Connection %i: Buffer line %i flushed: \"%s\"",
			xsw_object[object_num].name,
			object_num,
                        condescriptor,
                        conbuf_items,
                        connection[condescriptor].conbuf[conbuf_items - 1].buffer  
                    );
		}
                LogAppendLineFormatted(primary_log_fullname, stringa);
		conbuf_items--;
            } 
	    else
	    {
		break;
	    }
            usleep(50000);
	}
	/* Set new amount of buffered items. */
	if(conbuf_items < 0)
		conbuf_items = 0;
/*	connection[condescriptor].conbuf_items = conbuf_items; */
	connection[condescriptor].conbuf_items = 0;


	/*
	 *   Check Priority:
	 *
	 *	0 = Highest, force send for up to 5 seconds.
	 *	1 = Moderate, try to send. If cannot then put into connection's
	 *	    buffer.
	 *	2 = Lowest, try to send, If cannot then forget it.
	 */
	switch(priority)
	{
	    case 0:
		break;


	    case 1:
		/* Check if socket is ready for send. */
		errno = 0;
                timeout.tv_sec = 0;
                timeout.tv_usec = 0;
                FD_ZERO(&writefds);
                FD_SET(consocket, &writefds);
                select(consocket + 1, NULL, &writefds, NULL, &timeout);
                /* Check if socket is connected and valid. */
                if(errno == EBADF)
                {
		    errno = 0;

		    /* Log lost connection. */
                    sprintf(stringa,
   "NetSendDataToConnection() Connection %i: Socket invalid and no longer used.",
                          condescriptor
                    );
                    LogAppendLineFormatted(primary_log_fullname, stringa);

                    connection[condescriptor].socket = -1;
                    connection[condescriptor].object_num = -1;
                    return(-1);
                }
		/* Socket is valid to send. */
                if(FD_ISSET(consocket, &writefds))
                {
                    bytes_sent +=
                        send(consocket, outbounddata, strlen(outbounddata) + 1, 0);
                }
		/* Can't send, queue it. */
		else
		{
		    conbuf_items = connection[condescriptor].conbuf_items;
		    if(conbuf_items < MAX_CON_BUF_ITEMS)
		    {
                        strcpy(
                            connection[condescriptor].conbuf[conbuf_items].buffer,
			    outbounddata
			);

			/* Log queue. */
                        sprintf(stringa,
                           "Connection %i: Queued line %i: \"%s\"",
                           condescriptor,
			   conbuf_items + 1,
                           connection[condescriptor].conbuf[conbuf_items].buffer
			);
                        LogAppendLineFormatted(primary_log_fullname, stringa);

                        conbuf_items++;
			connection[condescriptor].conbuf_items = conbuf_items;
		    }
		}
		break;


	    default:
                /* Check if socket is ready for send. */
		errno = 0;
                timeout.tv_sec = 0;
                timeout.tv_usec = 0;
                FD_ZERO(&writefds);
                FD_SET(consocket, &writefds);
                select(consocket + 1, NULL, &writefds, NULL, &timeout);
                /* Check if socket is connected and valid. */
                if(errno == EBADF)
                {
                    errno = 0;

                    /* Log lost connection. */
                    sprintf(stringa,
   "NetSendDataToConnection() Connection %i: Socket invalid and no longer used.",
                          condescriptor
                    );
                    LogAppendLineFormatted(primary_log_fullname, stringa);

                    connection[condescriptor].socket = -1;
                    connection[condescriptor].object_num = -1;
                    return(-1);
                }
		/* Socket is valid to send. */
                if(FD_ISSET(consocket, &writefds))
                {
                    bytes_sent +=
                       send(consocket, outbounddata, strlen(outbounddata) + 1, 0);
                }
		break;

	}

	/* Update number of bytes sent on that connection. */
	connection[condescriptor].bytes_sent += bytes_sent;

	return(0);
}



void NetCloseConnection(long condescriptor)
{
	/* Local variables. */
	char sndbuf[MAX_NET_CMD_SIZE];
	char stringa[1024];
	long x, y;


	/* Make sure condescriptor is valid. */
	if((condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS))
		return;


	/* Verify disconnect to client. */
	if(connection[condescriptor].socket > -1)
	{
	    /* Send disconnect code to client. */
	    sprintf(sndbuf, "%i\n%c", NET_CMD_LOGOUT, '\0');
	    NetSendDataToConnection(condescriptor, sndbuf, 1);

            /* Close the connection. */
            close(connection[condescriptor].socket);

            /* Log disconnect. */
	    sprintf(stringa, "Connection %i: Connection closed.",
		condescriptor);
	    LogAppendLineFormatted(primary_log_fullname, stringa);
	}


	/* Reset connection values. */
	connection[condescriptor].socket = -1;
        connection[condescriptor].conhost[0] = '\0';
        connection[condescriptor].object_num = -1;
        connection[condescriptor].contime = -1;
        connection[condescriptor].bytes_recieved = 0;
        connection[condescriptor].bytes_sent = 0;
        connection[condescriptor].errors_recieved = 0;
        connection[condescriptor].errors_sent = 0;

	connection[condescriptor].badlogins = 0;

	connection[condescriptor].obj_ud_interval = DEFAULT_UPDATE_INTERVAL;
	connection[condescriptor].obj_ud_last = 0;


	/* Refresh global variable total_connections. */
	for(x = 0, y = -1; x < MAX_CONNECTIONS; x++)
	{
           if(connection[x].socket < 0)
                continue;

	   y = x;
        }
	total_connections = (y + 1);

 
	return;
}



void NetClearAllConnections()
{
	/* Local variables. */
	long x;


	/* Close all connections. */
	for(x = 0; x < MAX_CONNECTIONS; x++)
	{
            NetCloseConnection(x);
	}

	return;
}



long NetGetNewDescriptor()
{
	/* Local variables. */
	long x;
	long free_condescriptor;


	/* Search for next available connection descriptor. */
	free_condescriptor = -1;
	for(x = 0; x < MAX_CONNECTIONS; x++)
	{
	   if(connection[x].socket > -1)
		continue;

	   free_condescriptor = x;
	   break;
	}


	/* Increment global variable highest_connection. */
	if(free_condescriptor >= highest_connection)
		highest_connection = free_condescriptor + 1;


	return(free_condescriptor);
}



int NetPollSocket(int sockettopoll)
{
	/* Local variables. */
	char stringa[256];

        struct timeval timeout;
        fd_set readfds;
        int consocket;

	long con_descriptor;

	int sin_size;
	int new_socket;
	struct sockaddr_in foreign_addr;

	char sndbuf[MAX_NET_CMD_SIZE];


	/* Do we have room for new connection? */
	if(total_connections >= MAX_CONNECTIONS)
	{
		return(-2);
	}


	/* Check sockettopoll for data and validility. */

/*
 *   Do not use select() to check for pending connections.
 *   for some reason it caues the new socket fetched from accept()
 *   to lock up.  More investigation is needed into this.  Using
 *   accept() to poll hasn't caused any problems yet.
 */
/*
	timeout.tv_sec = 0;
	timeout.tv_usec = 0;
	FD_ZERO(&readfds);
	FD_SET(sockettopoll, &readfds);
	select(sockettopoll + 1, &readfds, NULL, NULL, &timeout);

	if( !(FD_ISSET(sockettopoll, &readfds)) )
	{
		return(0);
	}
*/

	/* Poll socket using accept(). */
        sin_size = sizeof(struct sockaddr_in);
        new_socket = accept(sockettopoll, (struct sockaddr *)&foreign_addr,
		&sin_size);
	if(new_socket == -1)
		return(new_socket);
	fcntl(new_socket, F_SETFL, O_NONBLOCK);


	/* Log new connection. */
fprintf(stderr, "%s: Incoming connection.\n",
	inet_ntoa(foreign_addr.sin_addr)
);
        sprintf(stringa, "%s: Incoming connection.",
		inet_ntoa(foreign_addr.sin_addr)
        );
        LogAppendLineFormatted(primary_log_fullname, stringa);


	/* Get a new connection descriptor for new connection. */
	con_descriptor = NetGetNewDescriptor();
	if(con_descriptor < 0)
	{
		strcpy(stringa,
		    "Error: Unable to allocate new connection descriptor."
		);
		LogAppendLineFormatted(primary_log_fullname, stringa);
		close(new_socket);
		return(-2);
	}

	/* Adjust total_connections to mark topmost connection as */
	/* needed.                                                */
	if(con_descriptor >= total_connections)
		total_connections = con_descriptor + 1;
	if(total_connections > MAX_CONNECTIONS)
		total_connections = MAX_CONNECTIONS;


	/* Log newly allocated descriptor for connection. */
	sprintf(stringa, "Connection %i: New connection.",
		con_descriptor);
	LogAppendLineFormatted(primary_log_fullname, stringa);


	/* Reset some statistics for the new connection. */
        connection[con_descriptor].contime = time(NULL);
        connection[con_descriptor].bytes_recieved = 0;
        connection[con_descriptor].bytes_sent = 0;
        connection[con_descriptor].errors_recieved = 0;
        connection[con_descriptor].errors_sent = 0;

	connection[con_descriptor].badlogins = 0;

        connection[con_descriptor].obj_ud_interval = DEFAULT_UPDATE_INTERVAL;
	connection[con_descriptor].obj_ud_last = 0;


        /* Add data for members of connection structure. */
        connection[con_descriptor].socket = new_socket;
	sprintf(connection[con_descriptor].conhost, "%s",
		inet_ntoa(foreign_addr.sin_addr));
	connection[con_descriptor].conhost[MAX_HOSTNAME_LENGTH - 1] = '\0';


        /* We don't give connection a new object untill they send us a valid */
	/* name and password.                                                */
        connection[con_descriptor].object_num = -1;


	/* Ask new connection for name and password. */
	sprintf(sndbuf, "%i\n%c", NET_CMD_LOGIN, '\0');
	NetSendDataToConnection(con_descriptor, sndbuf, 1);	


fprintf(stderr, "Connection %i: New connection.\n",
	con_descriptor
);


	return(0);
}



int NetSendRecycleObject(long condescriptor, long object_num)
{
	/* Local variables. */
	char sndbuf[MAX_NET_CMD_SIZE];
	long x;


	/* Make sure object_num is valid. */
	if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
		return(-1);


        /* Format the sndbuf. */
        sprintf(sndbuf, "%i %i\n%c",
                NET_CMD_RECYCLEOBJECT,
                object_num,
		'\0'
        );


        /* If condescriptor is -1 then send to all connections. */
        if(condescriptor < 0)
        {
           /* Send object parameters. */
           for(x = 0; x < total_connections; x++)
           {
               /* Check if connection has a socket. */
               if(connection[x].socket < 0)
                   continue;
          
               /* Check if connection has an object. */
               if(connection[x].object_num < 0)
                   continue;
        
               /* Send the object position to socket of connection. */
               NetSendDataToConnection(x, sndbuf, 1);
           }
        } 
        else
        {  
            /* Check if connection has a socket. */
            if(connection[condescriptor].socket < 0)
                return(-1);
 
            /* Check if connection has an object. */
            if(connection[condescriptor].object_num < 0)
                return(-1);
        
            /* Send the object position to socket of connection. */
            NetSendDataToConnection(condescriptor, sndbuf, 1);
        }
                
                
        return(0);
}



int NetHandleLogin(long condescriptor, char *arg)
{
	/* Local variables. */
	char stringa[256];
	char sndbuf[MAX_NET_CMD_SIZE];
	char name[MAX_NAME_LENGTH];
	char password[MAX_PASSWORD_LENGTH];
	char *strptr;
	long object_num;


        /* Make sure condescriptor is valid. */
        if((condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS))
                return(-1);
	if(connection[condescriptor].socket < 0)
		return(-1);


	/* See if connection condescriptor is already logged in. */
	if(connection[condescriptor].object_num >= 0)
	{
		sprintf(sndbuf, "%i You are already logged in.\n%c",
			NET_CMD_GLOBALMESG, '\0');
		NetSendDataToConnection(condescriptor, sndbuf, 1);
		return(-1);
	}


	/* Argument arg can't be too short. */
	if(strlen(arg) < 1)
		return(-1);


        /* Close socket if too many bad logins. */
        if(connection[condescriptor].badlogins >= MAX_BAD_LOGINS)
        {
            sprintf(stringa,
        	"%i More than %i bad logins, disconnecting...\n%c",
                NET_CMD_GLOBALMESG,
		MAX_BAD_LOGINS,
                '\0'
            );
            NetSendDataToConnection(condescriptor, stringa, 1);
                
fprintf(stderr, "Connection %i: Sent more than %i bad logins.\n",
 condescriptor, MAX_BAD_LOGINS);

            NetCloseConnection(condescriptor);
            sprintf(stringa, "Connection %i: Sent more than %i bad logins.",
                condescriptor, MAX_BAD_LOGINS);
            LogAppendLineFormatted(primary_log_fullname, stringa);
             
            return(-1);
        }


	/* ********************************************************** */
	/* ***                    Parse Argument                  *** */

	if((strptr = strstr(arg, ";")) == NULL)
	{
                sprintf(sndbuf,
              "%i Login syntax incorrect, should be \"<name>;<password>\"\n%c",
                        NET_CMD_GLOBALMESG, '\0'
		);
                NetSendDataToConnection(condescriptor, sndbuf, 1);

                /* Send a rerequest for login. */
                sprintf(sndbuf, "%i\n%c", NET_CMD_LOGIN, '\0');
                NetSendDataToConnection(condescriptor, sndbuf, 1);

		/* Record a bad login. */
		connection[condescriptor].badlogins += 1;

                return(-1);
	}


	/* Get name. */
	strcpy(name, arg);
	strptr = strstr(name, ";");
	*strptr = '\0';

	/* Get password. */
	strptr = strstr(arg, ";");
	strptr += 1;
	strcpy(password, strptr);


	/* Get object number. */
	object_num = MatchObjectByName(name, XSW_OBJ_TYPE_PLAYER);
	if(object_num < 0)
	{
		/* Send bad login message. */
                sprintf(sndbuf,
      "%i Either that player does not exist, or has a different password.\n%c",
                        NET_CMD_GLOBALMESG, '\0'
                );
                NetSendDataToConnection(condescriptor, sndbuf, 1);

                /* Send a rerequest for login. */
                sprintf(sndbuf, "%i\n%c", NET_CMD_LOGIN, '\0');
                NetSendDataToConnection(condescriptor, sndbuf, 1);

		/* Log what happened. */
		sprintf(stringa,
                    "Connection %i: Bad login name: \"%s\"",
		    condescriptor,
		    name
		);
                LogAppendLineFormatted(primary_log_fullname, stringa);

                /* Record a bad login. */
                connection[condescriptor].badlogins += 1;
 
		return(-1);
	}

	/* Get and check password. */
	if( (strcmp(xsw_object[object_num].password, password)) != 0 )
	{
		/* Send bad login message. */
                sprintf(sndbuf,
      "%i Either that player does not exist, or has a different password.\n%c",
                        NET_CMD_GLOBALMESG, '\0'
                );
                NetSendDataToConnection(condescriptor, sndbuf, 1);

		/* Send a rerequest for login. */
	        sprintf(sndbuf, "%i\n%c", NET_CMD_LOGIN, '\0');
        	NetSendDataToConnection(condescriptor, sndbuf, 1);

                /* Log what happened. */
                sprintf(stringa,
		    "Connection %i: %s(#%i): Bad password: \"%s\"",
                    condescriptor,
                    xsw_object[object_num].name,
		    object_num,
		    password
                );
		LogAppendLineFormatted(primary_log_fullname, stringa);

                /* Record a bad login. */
                connection[condescriptor].badlogins += 1;

                return(-1);
	}


	/* ************************************************** */
	/* All checks passed, set connection's object number. */
	connection[condescriptor].object_num = object_num;


        /* Log successful connection. */
        sprintf(stringa,
            "Connection %i: Successfully logged in as: %s(#%i)",
            condescriptor,
            xsw_object[object_num].name,
	    object_num
        );
	LogAppendLineFormatted(primary_log_fullname, stringa);

fprintf(stderr, "Connection %i: Successfully logged in as: %s(#%i)\n",
   condescriptor,
   xsw_object[object_num].name,
   object_num
);

	/* Print welcome message. */
        sprintf(sndbuf, "%i Welcome to %s.\n%c",
                NET_CMD_GLOBALMESG,
                dbtitle,
                object_num,
                '\0'
        );
        NetSendDataToConnection(condescriptor, sndbuf, 1);

	sprintf(sndbuf, "%i You are now connected as %s(#%i).\n%c",
		NET_CMD_GLOBALMESG,
		xsw_object[object_num].name,
		object_num,
		'\0'
	);
        NetSendDataToConnection(condescriptor, sndbuf, 1);


	/* Send refresh to condescriptor. */
	NetSendRefresh(condescriptor);


	return(0);
}



int NetHandleLogout(long condescriptor)
{
	/* Local variables. */
	long object_num;
	char stringa[256];


	/* Is condescriptor valid? */
	if( (condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS) )
		return(-1);


	/* Log what happened. */
	object_num = connection[condescriptor].object_num;
	if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
	{
		/* If never succesfully logged in. */
		sprintf(stringa,
                  "Connection %i: Never logged in and disconnected.",
			condescriptor
		);
		LogAppendLineFormatted(primary_log_fullname, stringa);
	}
        else
	{
                sprintf(stringa,
                        "Connection %i: %s(#%i) disconnected.",
			condescriptor,
			xsw_object[object_num].name,
			object_num
                );
                LogAppendLineFormatted(primary_log_fullname, stringa);
fprintf(stderr,
   "Connection %i: %s(#%i) disconnected.\n",
   condescriptor,
   xsw_object[object_num].name,
   object_num
);
	}

        /* Close the connection. */
        NetCloseConnection(condescriptor);


	return(0);
}



int NetHandleGlobalMessage(int sendercondescriptor, char *message)
{
	/* Local variables. */
	long object_num;
	long x;
	char sndbuf[MAX_NET_CMD_SIZE];
	char work_str[MAX_MESSAGE_SIZE];
	char stringa[MAX_NET_CMD_SIZE];


	/* message must contain data. */
	if(strlen(message) < 1)
		return(-2);

        /* Make sure sendercondescriptor is valid. */   
        if((sendercondescriptor < 0) || (sendercondescriptor >= MAX_CONNECTIONS))
                return(-1);

	/* Make sure sendercondescriptor has a valid object. */
	object_num = connection[sendercondescriptor].object_num;
	if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
		return(-1);


	/* Copy message to work_str. */
	strncpy(work_str, message, MAX_MESSAGE_SIZE);
	work_str[MAX_MESSAGE_SIZE - 1] = '\0';


	/* Format the message. */
	sprintf(sndbuf, "%i %s(#%i): \"%s\"\n%c",
		NET_CMD_GLOBALMESG,
		xsw_object[object_num].name,
		object_num,
		work_str,
		'\0'
	);


        /* Limit sndbuf length. */
	sndbuf[MAX_NET_CMD_SIZE - 1] = '\0';


	/* Send message to all connections. */
	for(x = 0; x < total_connections; x++)
	{
		/* Does connection have a socket? */
		if(connection[x].socket < 0)
			continue;

		/* Does connection have an object? */
		if(connection[x].object_num < 0)
			continue;

		/* Send data. */
		NetSendDataToConnection(x, sndbuf, 1);
	}


	/* Log global message. */
	sprintf(stringa, "%s(#%i): Global Message: \"%s\"",
		xsw_object[object_num].name,
		object_num,
		work_str
	);
	LogAppendLineFormatted(primary_log_fullname, stringa);

fprintf(stderr,
 "%s(#%i): Global Message: \"%s\"\n",
 xsw_object[object_num].name,
 object_num,
 work_str
);


	return(0);
}



int NetSendRefresh(long condescriptor)
{
	/* Local variables. */
	long object_num;
	long object_count;
	char stringa[1024];


	/* Make sure condescriptor is valid. */
	if( (condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS) )
		return(-1);


	/* Make sure condescriptor has a valid socket and is logged in. */
        if(connection[condescriptor].socket < 0)
                return(-1);
	object_num = connection[condescriptor].object_num;

        if( (object_num < 0) || (object_num >= MAX_OBJECTS))
                return(-1);


        /* Log refresh. */
	sprintf(stringa, "%s(#%i): Requested refresh.",
		xsw_object[object_num].name,
		object_num
	);
        LogAppendLineFormatted(primary_log_fullname, stringa);
fprintf(stderr, "%s(#%i): Requested refresh.\n",
 xsw_object[object_num].name,
 object_num
);


	/* Send who am I to condescriptor. */
        NetSendWhoAmI(condescriptor);

	/* Send all object maximums to condescriptor. */
	for(object_count = 0; object_count < total_objects; object_count++)
	{
            NetSendObjectMaximums(condescriptor, object_count);
	    NetSendSetName(condescriptor, object_count);
	}


	return(0);
}



int NetHandleSetInterval(long condescriptor, char *arg)
{
	/* Local variables. */
	long interval;
	char sndbuf[MAX_NET_CMD_SIZE];


        /* Make sure condescriptor is valid. */
        if((condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS))
                return(-1);

        if(connection[condescriptor].socket < 0)
                return(-1);

	/* arg must be atleast 1 character long. */
	if(strlen(arg) < 1)
		return(-1);

        /*
         *   NET_CMD_SETINTERVAL format is as follows:
         *
         *      interval
         */
	sscanf(arg,
		"%i",
		&interval
	);


	/* Make sure interval is within min and max values. */
	if(interval < MIN_OBJECT_UPDATE_INTERVAL)
		interval = MIN_OBJECT_UPDATE_INTERVAL;

	if(interval > MAX_OBJECT_UPDATE_INTERVAL)
		interval = MAX_OBJECT_UPDATE_INTERVAL;


	/* Set new update interval for connection. */
	connection[condescriptor].obj_ud_interval = interval;


	/* Send response. */
        sprintf(sndbuf, "%i Update interval: %i ms.\n%c",
                NET_CMD_GLOBALMESG,
                interval,
		'\0'
        );
        NetSendDataToConnection(condescriptor, sndbuf, 1);
            


	return(0);
}



int NetSendObjectPosition(long condescriptor, long object_num)
{
	/* Local variables. */
	long x;
	char sndbuf[MAX_NET_CMD_SIZE];


	/* Make sure object is valid. */
	if(xsw_object[object_num].type <= XSW_OBJ_TYPE_GARBAGE)
		return(0);


        /*
         *   NET_CMD_SETOBJLOCAPP format is as follows:
         *
         *      object_num,
         *      type, imageset, size,
         *      x, y, z,
         *      heading, pitch,
         *      velocity velocity_heading,
         *      throttle, frame
         */     
        sprintf(sndbuf,
"%i %i\
 %i %i %i\
 %lf %lf %lf\
 %lf %lf\
 %lf %lf\
 %i %i\n%c",

                NET_CMD_SETOBJLOCAPP,
		object_num,

                xsw_object[object_num].type,
                xsw_object[object_num].imageset,
		xsw_object[object_num].size,

                xsw_object[object_num].x,
                xsw_object[object_num].y,
                xsw_object[object_num].z,

                xsw_object[object_num].heading,
                xsw_object[object_num].pitch,

                xsw_object[object_num].velocity,
                xsw_object[object_num].velocity_heading,

                xsw_object[object_num].throttle,
		xsw_object[object_num].animation.current_frame,
		'\0'
        );



        /* If condescriptor is -1 then send to all connections. */
        if(condescriptor < 0)
        {
           /* Send object parameters. */
           for(x = 0; x < total_connections; x++)
           {
               /* Check if connection has a socket. */
               if(connection[x].socket < 0)
                   continue;
 
               /* Check if connection has an object. */
               if(connection[x].object_num < 0)
                   continue;
                
               /* Send the object position to socket of connection. */
               NetSendDataToConnection(x, sndbuf, 2);
           }
        }
        else
        {
            /* Check if connection has a socket. */
            if(connection[condescriptor].socket < 0)
                return(-1);
        
            /* Check if connection has an object. */
            if(connection[condescriptor].object_num < 0)
                return(-1);
                
            /* Send the object position to socket of connection. */
            NetSendDataToConnection(condescriptor, sndbuf, 2);
        }
 

	return(0);
}



int NetSendObjectPositionS(long condescriptor, long object_num)
{
        /* Local variables. */
        long x;
        char sndbuf[MAX_NET_CMD_SIZE];
 
 
        /* Make sure object is valid. */
        if(xsw_object[object_num].type <= XSW_OBJ_TYPE_GARBAGE)
                return(0);
 
 
        /*
         *   NET_CMD_SETOBJLOCAPPS format is as follows:
         * 
         *      object_num,
         *      type, imageset, size,
         *      x, y, z,
         *      heading, pitch,
         *	frame
         */
        sprintf(sndbuf,
"%i %i\
 %i %i %i\
 %lf %lf %lf\
 %lf %lf\
 %i\n%c",
            
                NET_CMD_SETOBJLOCAPPS,
                object_num,
         
                xsw_object[object_num].type,
                xsw_object[object_num].imageset,
                xsw_object[object_num].size,
 
                xsw_object[object_num].x,
                xsw_object[object_num].y,
                xsw_object[object_num].z,

                xsw_object[object_num].heading,
                xsw_object[object_num].pitch,
         
                xsw_object[object_num].animation.current_frame,
                '\0'
        );

 

        /* If condescriptor is -1 then send to all connections. */
        if(condescriptor < 0)
        {
           /* Send object parameters. */
           for(x = 0; x < total_connections; x++)
           {
               /* Check if connection has a socket. */
               if(connection[x].socket < 0)
                   continue;
           
               /* Check if connection has an object. */
               if(connection[x].object_num < 0)

               /* Send the object position to socket of connection. */
               NetSendDataToConnection(x, sndbuf, 2);
           }
        }
        else
        {
            /* Check if connection has a socket. */
            if(connection[condescriptor].socket < 0)
                return(-1);
 
            /* Check if connection has an object. */
            if(connection[condescriptor].object_num < 0)
                return(-1);
                
            /* Send the object position to socket of connection. */
            NetSendDataToConnection(condescriptor, sndbuf, 2);
        }
                
                
        return(0);
}




int NetSendObjectValues(long condescriptor, long object_num)
{
        /* Local variables. */
        long x;
        char sndbuf[MAX_NET_CMD_SIZE];
                

        /* Make sure object is valid. */
        if(xsw_object[object_num].type <= XSW_OBJ_TYPE_GARBAGE)
                return(0);


        /*
         * Format sndbuf:
         *
         *   object_num,
         *   locked_on, intercepting_object, velocity, velocity_heading,
         *   thrust_rev_state, thrust_dir, thrust, throttle, hp, power,
         *   antimatter, shield_state, selected_weapon, cloak_state,
         *   cloak_strength, visibility, damage_control
         *
         */
        sprintf(sndbuf,
"%i %i\
 %i %i %lf %lf\
 %i %lf %lf %i %lf %lf\
 %lf %i %i %i\
 %lf %lf %i\n%c",

		NET_CMD_SETOBJVALUES,
                object_num,

                xsw_object[object_num].locked_on,
                xsw_object[object_num].intercepting_object,
                xsw_object[object_num].velocity,
                xsw_object[object_num].velocity_heading,
        
                xsw_object[object_num].thrust_rev_state,
                xsw_object[object_num].thrust_dir,
                xsw_object[object_num].thrust,
                xsw_object[object_num].throttle,
                xsw_object[object_num].hp,
                xsw_object[object_num].power,

                xsw_object[object_num].antimatter,
                xsw_object[object_num].shield_state,
                xsw_object[object_num].selected_weapon,

                xsw_object[object_num].cloak_state,
                xsw_object[object_num].cloak_strength,
                xsw_object[object_num].visibility,
                xsw_object[object_num].damage_control,
		'\0'
        );

                
        /* If condescriptor is -1 then send to all connections. */
        if(condescriptor < 0)
        { 
           /* Send object parameters. */
           for(x = 0; x < total_connections; x++)
           {
               /* Check if connection has a socket. */
               if(connection[x].socket < 0)
                   continue;
 
               /* Check if connection has an object. */
               if(connection[x].object_num < 0)
                   continue;

               /* Send the object position to socket of connection. */
               NetSendDataToConnection(x, sndbuf, 2);
           }
        }
        else
        {
            /* Check if connection has a socket. */  
            if(connection[condescriptor].socket < 0)
                return(-1);
                
            /* Check if connection has an object. */
            if(connection[condescriptor].object_num < 0)
                return(-1);
        
            /* Send the object position to socket of connection. */
            NetSendDataToConnection(condescriptor, sndbuf, 2);
        }


        return(0);
}



int NetSendObjectMaximums(long condescriptor, long object_num)
{
        /* Local variables. */   
        long x;
        char sndbuf[MAX_NET_CMD_SIZE];

                
        /* Make sure object is valid. */
        if(xsw_object[object_num].type <= XSW_OBJ_TYPE_GARBAGE)
                return(0);


        /*
	 * Format sndbuf:
	 *
	 *
         *   type, imageset, owner, size, scanner_range,
         *   velocity_max, thrust_power, turnrate, hp_max, power_max,
         *   power_purity, core_efficency, antimatter_max, total_weapons, visibility
	 */
        sprintf(sndbuf,
"%i %i\
 %i %i %i %i %lf\
 %lf %lf %lf %lf %lf\
 %lf %lf %lf %i %lf\n%c",

		NET_CMD_SETOBJMAXIMUMS,
                object_num,

                xsw_object[object_num].type,
                xsw_object[object_num].imageset,
                xsw_object[object_num].owner,
                xsw_object[object_num].size,
                xsw_object[object_num].scanner_range,

                xsw_object[object_num].velocity_max,
                xsw_object[object_num].thrust_power,
                xsw_object[object_num].turnrate,
                xsw_object[object_num].hp_max,
                xsw_object[object_num].power_max,

                xsw_object[object_num].power_purity,
                xsw_object[object_num].core_efficency,
                xsw_object[object_num].antimatter_max,
                xsw_object[object_num].total_weapons,
                xsw_object[object_num].visibility,
		'\0'
        );


	/* If condescriptor is -1 then send to all connections. */
	if(condescriptor < 0)
	{
           /* Send object parameters. */
           for(x = 0; x < total_connections; x++)
           { 
               /* Check if connection has a socket. */
               if(connection[x].socket < 0)
                   continue;
        
               /* Check if connection has an object. */
               if(connection[x].object_num < 0)
                   continue;

               /* Send the object position to socket of connection. */
               NetSendDataToConnection(x, sndbuf, 2);
           }
	}
	else
	{
            /* Check if connection has a socket. */
            if(connection[condescriptor].socket < 0)
                return(-1);
                
            /* Check if connection has an object. */
            if(connection[condescriptor].object_num < 0)
		return(-1);

            /* Send the object position to socket of connection. */
            NetSendDataToConnection(condescriptor, sndbuf, 1);
        }


        return(0);
}



int NetSendWhoAmI(long condescriptor)
{
	/* Local variables. */
	char stringa[MAX_NET_CMD_SIZE];
	long object_num;


        /* Make sure condescriptor is valid. */
        if((condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS))
                return(-1);

	/* See if socket exists. */
	if(connection[condescriptor].socket < 0)
		return(-1);

	/* Get object_num. */
	object_num = connection[condescriptor].object_num;

	/* Does connection have an object?    */
	/* No object means not logged in yet. */
	if(object_num < 0)
	{
		sprintf(stringa, "%i You have not logged in.\n%c",
			NET_CMD_WHOAMI, '\0');
		NetSendDataToConnection(condescriptor, stringa, 1);
		return(-1);
	}

	/* Format and send who am I information. */
	sprintf(stringa, "%i %s;%i\n%c",
		NET_CMD_WHOAMI,
		xsw_object[object_num].name,
		object_num,
		'\0'
	);

	NetSendDataToConnection(condescriptor, stringa, 1);

	usleep(50000);
	NetSendObjectMaximums(-1, object_num);

	return(0);
}



int NetSendEchoToConnection(long condescriptor, char *arg)
{
	/* Local variables. */
	char sndbuf[MAX_NET_CMD_SIZE];


        /* Make sure condescriptor is valid. */
        if((condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS))
                return(-1);


	/* Format sndbuf. */
	sprintf(sndbuf, "%i %s\n%c",
		NET_CMD_ECHO,
		arg,
		'\0'
	);

	NetSendDataToConnection(condescriptor, sndbuf, 1);


	return(0);
}



int NetHandleSetObjApp(long condescriptor, char *arg)
{
	/* Local variables. */
	long object_num;

	long type;
	long imageset;
	long size;

	double x, y, z;
	double heading, pitch;

	double velocity;
	double velocity_heading;

        int throttle;
	long current_frame;


        /* Make sure condescriptor is valid. */
        if((condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS))
                return(-1);


        /* Get object_num. */
        object_num = connection[condescriptor].object_num;
	/* Connection must have valid object_num. */
	if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
		return(-1);


        /*
         *   NET_CMD_SETOBJLOCAPP format is as follows:
         *
         *      object_num,
         *      type, imageset, size,
         *      x, y, z,
         *      heading, pitch,
         *      velocity, velocity_heading,
         *      throttle, frame
         */
	sscanf(arg,
"%i\
 %i %i %i\
 %lf %lf %lf\
 %lf %lf\
 %lf %lf\
 %i %i",

		&object_num,

		&type,
		&imageset,
		&size,

		&x,
		&y,
		&z,

		&heading,
		&pitch,

		&velocity,
		&velocity_heading,

                &throttle,
		&current_frame
	);

	/* We regot the object_num from arg above, now check if it */
	/* matches the connection's object_num.                    */
	if(connection[condescriptor].object_num != object_num)
	{
		fprintf(stderr,
       "Warning: Connection %i which owns object %i tried to move object %i.\n",
			condescriptor,
			connection[condescriptor].object_num,
			object_num
		);
		return(-1);
	}


	/* Set object_num's location and apperance. */

	/* Don't set type, imageset, or size. */
/*
	xsw_object[object_num].type = type;
        xsw_object[object_num].imageset = imageset;
	xsw_object[object_num].size = size;
*/

	/* Set these only if object has enough antimatter. */
	if(xsw_object[object_num].antimatter > 0)
	{
	    xsw_object[object_num].x = x;
            xsw_object[object_num].y = y;
            xsw_object[object_num].z = z;
	}

        xsw_object[object_num].heading = heading;
        xsw_object[object_num].pitch = pitch;

        /* Set these only if object has enough antimatter. */
        if(xsw_object[object_num].antimatter > 0)
        {
            xsw_object[object_num].velocity = velocity;
            xsw_object[object_num].velocity_heading = velocity_heading;
	}

        xsw_object[object_num].throttle = throttle;
        xsw_object[object_num].animation.current_frame = current_frame;


	return(0);
}



int NetHandleSetObjAppS(long condescriptor, char *arg)
{
        /* Local variables. */
        long object_num;
        
        long type;
        long imageset;
        long size;
        
        double x, y, z;
        double heading, pitch;
          
        long current_frame;
        
           
        /* Make sure condescriptor is valid. */
        if((condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS))
                return(-1);
                
 
        /* Get object_num. */
        object_num = connection[condescriptor].object_num;
        /* Connection must have valid object_num. */
        if(object_num < 0)
                return(-1); 
        
        
        /*
         *   NET_CMD_SETOBJLOCAPPS format is as follows:
         *
         *      object_num,
         *      type, imageset, size,
         *      x, y, z,
         *      heading, pitch,
         *      current_frame
         */
        sscanf(arg,
"%i\
 %i %i %i\
 %lf %lf %lf\
 %lf %lf\
 %i",

                &object_num,

                &type,
                &imageset,
                &size,  

                &x,
                &y,  
                &z,

                &heading,
                &pitch,

                &current_frame
        ); 
        
        /* We regot the object_num from arg above, now check if it */
        /* matches the connection's object_num.                    */
        if(connection[condescriptor].object_num != object_num)
        {
                fprintf(stderr,
       "Warning: Connection %i which owns object %i tried to move object %i.\n",
                        condescriptor,
                        connection[condescriptor].object_num,
                        object_num
                );
                return(-1);
        }
         

        /* Set object_num's location and apperance. */
/*
        xsw_object[object_num].type = type;
        xsw_object[object_num].imageset = imageset;
        xsw_object[object_num].size = size;
*/

        /* Set these only if object has enough antimatter. */
        if(xsw_object[object_num].antimatter > 0)  
        {
            xsw_object[object_num].x = x;
            xsw_object[object_num].y = y;
            xsw_object[object_num].z = z;
        }

        xsw_object[object_num].heading = heading;   
        xsw_object[object_num].pitch = pitch;
                
        xsw_object[object_num].animation.current_frame = current_frame;


        return(0);
}



int NetHandleSetWeapon(long condescriptor, char *arg)
{
	/* Local variables. */
	long object_num;
	long object_count;
	int selected_weapon;


        /* Make sure condescriptor is valid. */  
        if((condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS))
                return(-1);
        
           
        object_num = connection[condescriptor].object_num;
        /* Connection must have valid object_num. */
        if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
                return(-1);
           
        /*
         *   NET_CMD_SETWEAPONS format is as follows:
         *
         *      selected_weapon
         */
        sscanf(arg, "%i", &selected_weapon);


	/* Make sure selected_weapon is valid. */
	if( (selected_weapon < 0) || (selected_weapon >= MAX_WEAPONS) )
		return(-1);

	if(selected_weapon >= xsw_object[object_num].total_weapons )
		return(-1);


	/* Set selected weapon. */
	xsw_object[object_num].selected_weapon = selected_weapon;


	/* Set last update to 0 so connections get update quickly. */
	objval_update_last = 0;


	return(0);
}



int NetHandleSetShields(long condescriptor, char *arg)
{
	/* Local variables. */
	long object_num;
	int shield_state;
	double shield_frequency;


        /* Make sure condescriptor is valid. */
        if((condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS))
                return(-1);
        
        
        object_num = connection[condescriptor].object_num;
        /* Connection must have valid object_num. */
        if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
                return(-1);

        /*
         *   NET_CMD_SETSHIELDS format is as follows:
         *
         *      shield_state, shield_frequency
         */
        sscanf(arg, "%i %lf",
		&shield_state,
		&shield_frequency
	);


	/* Does object have shields generators? */
	if(xsw_object[object_num].shield_state == -1)
		return(1);


	/* Set shield_frequency. */
	if(shield_frequency < 0)
		shield_frequency = 0;
	else if(shield_frequency > MAX_SHIELD_FREQUENCY)
		shield_frequency = MAX_SHIELD_FREQUENCY;

	xsw_object[object_num].shield_frequency = shield_frequency;


        /* Is the cloak up? */
        if(xsw_object[object_num].cloak_state == 1)
	{
                xsw_object[object_num].shield_state = 0;
		return(1);
	}


	/* Set shields. */
	xsw_object[object_num].shield_state = shield_state;


        /* Set last update to 0 so connections get update quickly. */
        objval_update_last = 0;


	return(0);
}



int NetHandleSetCloak(long condescriptor)
{
        /* Local variables. */
        long object_num;


        /* Make sure condescriptor is valid. */
        if((condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS))
                return(-1);

        object_num = connection[condescriptor].object_num;
        /* Connection must have valid object_num. */
        if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
                return(-1);
 
	/* Does object have cloak abilities? */
	if(xsw_object[object_num].cloak_state == -1)
		return(1);

        /* Are shields up? if so, lower it. */
        if(xsw_object[object_num].shield_state == 1)
		xsw_object[object_num].shield_state = 0;

        /* Set cloak. */
        if(xsw_object[object_num].cloak_state == 0)
	{
                xsw_object[object_num].cloak_state = 1;
        }
	else if(xsw_object[object_num].cloak_state == 1)
	{
                xsw_object[object_num].cloak_state = 0;
	}


        /* Set last update to 0 so connections get update quickly. */
        objval_update_last = 0;


        return(0);
}



int NetHandleSetDmgCtl(long condescriptor)
{
        /* Local variables. */
        long object_num;
 
                
        /* Make sure condescriptor is valid. */
        if((condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS))
                return(-1);
        
        object_num = connection[condescriptor].object_num;
        /* Connection must have valid object_num. */   
        if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
                return(-1);


        if(xsw_object[object_num].damage_control == 0)
	{
                xsw_object[object_num].damage_control = 1;
        }
	else if(xsw_object[object_num].damage_control == 1)
        {
	        xsw_object[object_num].damage_control = 0;
	}


        /* Set last update to 0 so connections get update quickly. */
        objval_update_last = 0;


        return(0);
}



int NetSendSetName(long condescriptor, long object_num)
{
	/* Local variables. */
	char sndbuf[MAX_NET_CMD_SIZE];
	char name[MAX_NAME_LENGTH];


	/* object_num must be valid. */
	if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
		return(-1);

	/* Get name. */
	strncpy(name, xsw_object[object_num].name, MAX_NAME_LENGTH);


	/* Format the data to be sent out. */
	sprintf(sndbuf, "%i %i;%s\n%c",
		NET_CMD_SETNAME,
		object_num,
		xsw_object[object_num].name,
		'\0'
	);


	NetSendDataToConnection(condescriptor, sndbuf, 1);


	return(0);
}



int NetSendWeaponValues(long condescriptor, long object_num, int weapon_num)
{
	/* Local variables. */
	char sndbuf[MAX_NET_CMD_SIZE];


	/* condescriptor must be valid. */
	if( (condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS) )
		return(-1);

        /* object_num must be valid. */
        if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
                return(-1);

	/* weapon_num must be valid. */
	if( (weapon_num < 0) || (weapon_num >= MAX_WEAPONS) )
		return(-1);


        /*
	 *   NET_CMD_SETWEAPONVAL format is as follows:
	 *
	 *	object_num, weapon_num,
	 *	type, emission_type, amount, max,
	 *	power, range, create_power
	 *	delay, last_used
	 */
        sprintf(sndbuf,
"%i\
 %i %i\
 %i %i %i %i\
 %lf %lf %lf\
 %i %i\n%c",
                NET_CMD_SETWEAPONVAL,

		object_num,
		weapon_num,

		xsw_object[object_num].weapons[weapon_num].type,
                xsw_object[object_num].weapons[weapon_num].emission_type,
                xsw_object[object_num].weapons[weapon_num].amount,
                xsw_object[object_num].weapons[weapon_num].max,

                xsw_object[object_num].weapons[weapon_num].power,
                xsw_object[object_num].weapons[weapon_num].range,
                xsw_object[object_num].weapons[weapon_num].create_power,

                xsw_object[object_num].weapons[weapon_num].delay,
                xsw_object[object_num].weapons[weapon_num].last_used,
		'\0'
        );


	/* Send the weapon values to condescriptor. */
        NetSendDataToConnection(condescriptor, sndbuf, 1);


	return(0);
}



int NetHandleSetWeaponsLock(long condescriptor)
{
        /* Local variables. */
	long object_num;
	long target_object;

        
        /* Make sure condescriptor is valid. */
        if((condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS))
                return(-1);
        

        object_num = connection[condescriptor].object_num;
        /* Connection must have valid object_num. */
        if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
                return(-1);


	target_object = xsw_object[object_num].locked_on;
	/* target_object must be greater than -2 and less than */
	/* MAX_OBJECTS.                                        */
	if( (target_object < -1) || (target_object >= MAX_OBJECTS) )
		target_object = -1;


	/* Cycle locked on. */
	xsw_object[object_num].locked_on = MatchWeaponsLock(
		object_num,
		target_object,
		xsw_object[object_num].scanner_range
	);


        /* Set last update to 0 so connections get update quickly. */
        objval_update_last = 0;


	return(0);
}



int NetHandleSetWeaponsUnlock(long condescriptor)
{
	/* Local variables. */
	long object_num;


        /* Make sure condescriptor is valid. */
        if((condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS))
                return(-1);
          
          
        object_num = connection[condescriptor].object_num;
        /* Connection must have valid object_num. */
        if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
                return(-1);


	/* Unlock. */
	xsw_object[object_num].locked_on = -1;

        /* Set last update to 0 so connections get update quickly. */
        objval_update_last = 0;


        return(0);
}



int NetHandleSetIntercept(long condescriptor, char *arg)
{
        /* Local variables. */
        char sndbuf[MAX_NET_CMD_SIZE];
	long object_num;
	char intercept_arg[MAX_NET_CMD_SIZE];


        /* condescriptor must be valid. */
        if( (condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS) )
            return(-1);

                
        /* object_num must be valid. */
	object_num = connection[condescriptor].object_num;
        if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
            return(-1);

 
        /*
         *   NET_CMD_INTERCEPT format is as follows:
         *
         *      argument
         */
	strncpy(intercept_arg, arg, MAX_NET_CMD_SIZE);
	intercept_arg[MAX_NET_CMD_SIZE - 1] = '\0';


	/* Match intercepting object. */
	xsw_object[object_num].intercepting_object =
	    MatchIntercept(object_num, intercept_arg);


	if(0 != strcmp(intercept_arg, "#off"))
	{
	    if( (strlen(intercept_arg) > 0) &&
	        (xsw_object[object_num].intercepting_object < 0)
	    )
	    {
	    sprintf(sndbuf, "%i Intercept: %s: Cannot find that object.\n%c",
		NET_CMD_GLOBALMESG,
		intercept_arg,
		'\0'
	    );
            NetSendDataToConnection(condescriptor, sndbuf, 1);
	    }
	}


	return(0);
}




int NetHandleFireWeapon(long condescriptor, char *arg)
{
	/* Note this function does not really do anything with arg
	 * however we require that input in case of future requirements.
	 */

	/* Local variables. */
	char stringa[1024];
	long object_num;
	double stream_heading;  /* birth_heading for stream weapons. */
	int selected_weapon;
	int objects_created;


        /* Make sure condescriptor is valid. */
        if((condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS))
                return(-1);


        object_num = connection[condescriptor].object_num;
        /* Connection must have valid object_num. */
        if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
                return(-1);


	/* Get object_num's selected weapon. */
        selected_weapon = xsw_object[object_num].selected_weapon;

        /* Check if selected_weapon is valid. */
        if( (selected_weapon < 0) || (selected_weapon >= MAX_WEAPONS) )
                return(-1);
        if( selected_weapon >= xsw_object[object_num].total_weapons )
                return(-1);


        /* Did we wait the minimum delay before we can fire again? */
        if(xsw_object[object_num].weapons[selected_weapon].delay >=
           ( cur_millitime -
           xsw_object[object_num].weapons[selected_weapon].last_used)
        )
        {    
                return(0);
        }


        /* Cannot fire weapon while cloaked. */
        if(xsw_object[object_num].cloak_state >= 1)
        {
/* Send error message. */
                xsw_object[object_num].weapons[selected_weapon].last_used =
                   cur_millitime;
                return(0); 
        }


        /* If weapon is expendable, see if object_num has any left. */
        if(xsw_object[object_num].weapons[selected_weapon].emission_type ==
           WEPEMISSION_PROJECTILE)
        {
           if(xsw_object[object_num].weapons[selected_weapon].amount <= 0)
           {
                return(0);
           }
        }


        /* Does object_num have enough power to create weapon? */
        if(xsw_object[object_num].power <
                xsw_object[object_num].weapons[selected_weapon].create_power)
        {
                return(0);
        }


        /* ******************************************************** */
        /* ***               Create the Weapon                  *** */
        /* Stream weapons. */
        if(xsw_object[object_num].weapons[selected_weapon].emission_type ==
            WEPEMISSION_STREAM)
        {
            objects_created = WepCreate(
               xsw_object[object_num].weapons[selected_weapon].type,
               object_num,
               xsw_object[object_num].x,
               xsw_object[object_num].y,
               xsw_object[object_num].heading,
               xsw_object[object_num].weapons[selected_weapon].power,
               xsw_object[object_num].weapons[selected_weapon].range
            );
        }
        /* Projectile and pulse weapons. */
        else
        {
            objects_created = WepCreate(
               xsw_object[object_num].weapons[selected_weapon].type,
               object_num,
               xsw_object[object_num].x,
               xsw_object[object_num].y,
               xsw_object[object_num].heading,
               xsw_object[object_num].weapons[selected_weapon].power,
               xsw_object[object_num].weapons[selected_weapon].range
            );
        }


	/* Log weapons fire. */
        sprintf(stringa,
 "%s(#%i): Fired: Weapon %i  type %i  at XYZ: %lf %lf %lf",
		xsw_object[object_num].name,
		object_num,
		selected_weapon,
		xsw_object[object_num].weapons[selected_weapon].type,
		xsw_object[object_num].x,
		xsw_object[object_num].y,
		xsw_object[object_num].z
	);
        LogAppendLineFormatted(primary_log_fullname, stringa);


        /* Were any objects created? */ 
        if(objects_created <= 0)
                return(objects_created);


        /* Mark the last time it was fired. */
        xsw_object[object_num].weapons[selected_weapon].last_used = cur_millitime;


        /* Consume power on object_num for the power that was */
        /* needed to create this weapon.                      */
        if(xsw_object[object_num].weapons[selected_weapon].emission_type
                == WEPEMISSION_STREAM)
        {
           xsw_object[object_num].power -=
                xsw_object[object_num].weapons[selected_weapon].create_power;
        }
        else if(xsw_object[object_num].weapons[selected_weapon].emission_type
                == WEPEMISSION_PROJECTILE)
        {
           xsw_object[object_num].power -=
                xsw_object[object_num].weapons[selected_weapon].create_power
                * objects_created;
        }
        else if(xsw_object[object_num].weapons[selected_weapon].emission_type
                == WEPEMISSION_PULSE)   
        {
           xsw_object[object_num].power -=
                xsw_object[object_num].weapons[selected_weapon].create_power
                * objects_created;
        }
        if(xsw_object[object_num].power < 0)
                xsw_object[object_num].power = 0;
   

        /* If weapon expendable, decrease stock. */
        if(xsw_object[object_num].weapons[selected_weapon].emission_type ==
           WEPEMISSION_PROJECTILE)
        {
           xsw_object[object_num].weapons[selected_weapon].amount -=
              objects_created;
           if(xsw_object[object_num].weapons[selected_weapon].amount < 0)
                xsw_object[object_num].weapons[selected_weapon].amount = 0;
           
	}


        /* Set last update to 0 so connections get update quickly. */
        objval_update_last = 0;


	/* Return the number of weapon objects created. */
	return(objects_created);
}




int NetPollSocketRecvData(long condescriptor)
{
	/* Local variables. */
	char recbuf[MAX_NET_OVERFLOW_SIZE];
        long recbuf_cnt;
        char workbuf[MAX_NET_CMD_SIZE];
        long workbuf_cnt;

        char arg[MAX_NET_CMD_SIZE];
        int command;

        long bytes_read;

        char stringa[MAX_NET_CMD_SIZE];
        char *strptr;
	int sockettohandle;

	struct timeval timeout;
	fd_set readfds;


        /* Make sure condescriptor is valid. */
        if((condescriptor < 0) || (condescriptor >= MAX_CONNECTIONS))
                return(-1);


	/* Make sure socket is valid. */
	sockettohandle = connection[condescriptor].socket;
	if(sockettohandle < 0)
		return(-1);


	/* Poll sockettohandle to see if there is incoming data to be read. */
        errno = 0;
        timeout.tv_sec = 0;
        timeout.tv_usec = 0;
        FD_ZERO(&readfds);
        FD_SET(sockettohandle, &readfds);

        select(sockettohandle + 1, &readfds, NULL, NULL, &timeout);

	/* Is socket valid? */ 
	if(errno == EBADF)
	{
            errno = 0;

            /* Log lost connection. */
            sprintf(stringa,
 "NetPollSocketRecvData() Connection %i: Socket invalid and no longer used.",
		condescriptor
	    );
	    LogAppendLineFormatted(primary_log_fullname, stringa);

            connection[condescriptor].socket = -1;
            connection[condescriptor].object_num = -1;
            return(-1);
	}


	/* Check if the socket has data to be read. */
        if( !(FD_ISSET(sockettohandle, &readfds)) )
                return(0);


	/* Fetch incoming data from sockettohandle. */
	errno = 0;
	bytes_read =
	    recv(sockettohandle, recbuf, MAX_NET_CMD_SIZE, 0);


        /* Did we get any data? */
        if(errno == EAGAIN)
	{
		errno = 0;
                return(0);
	}


        /* Increment bytes_received for this condescriptor. */
        connection[condescriptor].bytes_recieved += bytes_read;


        /* Is bytes_read cannot be greater than MAX_NET_OVERFLOW_SIZE? */
        if(bytes_read > MAX_NET_OVERFLOW_SIZE)
                bytes_read = MAX_NET_OVERFLOW_SIZE;


        /* Set null character at the end of recbuf just to be safe. */
        recbuf[MAX_NET_OVERFLOW_SIZE - 1] = '\0';


        /* Set buffer counts to 0. */
        workbuf_cnt = 0;
        recbuf_cnt = 0;
            

        /* *************************************************************** */
        /* ***                  Begin Parsing recbuf                   *** */
        while(recbuf_cnt < bytes_read)
        {
            /* Fetch data for workbuf. */
            workbuf_cnt = 0;
            while(workbuf_cnt < MAX_NET_CMD_SIZE)
            {
                if(recbuf_cnt >= bytes_read)
                {
                    workbuf[workbuf_cnt] = '\0';
                    break;
                }
 
                if(recbuf[recbuf_cnt] == '\0')
                {
                    recbuf_cnt++;
                    continue;
                }
        
                if( (recbuf[recbuf_cnt] == '\n') ||
                    (recbuf[recbuf_cnt] == '\r')
                )
                {
                    workbuf[workbuf_cnt] = '\0';
                    recbuf_cnt++;
                    break;
                }

                workbuf[workbuf_cnt] = recbuf[recbuf_cnt];
            
                recbuf_cnt++;
                workbuf_cnt++;
            }

            if(workbuf[0] == '\0')
                break;


            /* Get command command. */
            if( (command = StringGetNetCommand(workbuf)) < 0)
            {
                continue;
            }

            /* Get argument arg. */
            strncpy(arg, StringGetNetArgument(workbuf), MAX_NET_CMD_SIZE);
            arg[MAX_NET_CMD_SIZE - 1] = '\0';


	    /* See which net command handling function to call. */
	    switch(command)
	    {
	        case NET_CMD_COMMENT:
		    /* Handle comment. */
		    break;

	        case NET_CMD_LOGIN:
	            NetHandleLogin(condescriptor, arg);
		    break;

	        case NET_CMD_LOGOUT:
		    NetHandleLogout(condescriptor);
		    break;

                case NET_CMD_WHOAMI:
                    NetSendWhoAmI(condescriptor);
                    break;

	        case NET_CMD_ECHO:
		    NetSendEchoToConnection(condescriptor, arg);
		    break;

                case NET_CMD_REFRESH:
		    NetSendRefresh(condescriptor);
                    break;

	        case NET_CMD_SETINTERVAL:
                    NetHandleSetInterval(condescriptor, arg);
		    break;

		case NET_CMD_EXEC:
		    /* Call external function ExecHandleInput(). */
		    ExecHandleInput(condescriptor, arg);
		    break;

	        case NET_CMD_GLOBALMESG:
		    NetHandleGlobalMessage(condescriptor, arg);
		    break;

	        case NET_CMD_GROUPMESG:
		    /* Not supported yet. */
		    break;

                case NET_CMD_DIRECTMESG:
                    /* Not supported yet. */
                    break;

	        case NET_CMD_CREATEOBJECT:
		    /* Server doesn't accept this command. */
		    break;

	        case NET_CMD_RECYCLEOBJECT:
		    /* Server dosen't accept this command. */
		    break;

	        case NET_CMD_SETOBJLOCAPP:
		    NetHandleSetObjApp(condescriptor, arg);
		    break;

	        case NET_CMD_SETOBJVALUES:
		    /* Not supported yet. */
		    break;

	        case NET_CMD_SETOBJMAXIMUMS:
		    /* Server dosen't accept this command. */
		    break;

	        case NET_CMD_CREATEANIMATED:
		    /* Server dosen't accept this command. */
		    break;

	        case NET_CMD_SETWEAPON:
		    NetHandleSetWeapon(condescriptor, arg);
		    break;

	        case NET_CMD_SETSHIELDS:
		    NetHandleSetShields(condescriptor, arg);
		    break;

	        case NET_CMD_SETDMGCTL:
		    NetHandleSetDmgCtl(condescriptor);
		    break;

                case NET_CMD_SETCLOAK:
                    NetHandleSetCloak(condescriptor);
                    break;

                case NET_CMD_WEAPONSLOCK:
		    NetHandleSetWeaponsLock(condescriptor);
		    break;

	        case NET_CMD_WEAPONSUNLOCK:
		    NetHandleSetWeaponsUnlock(condescriptor);
		    break;

                case NET_CMD_INTERCEPT:
		    NetHandleSetIntercept(condescriptor, arg);
		    break;

                case NET_CMD_FIREWEAPON:
		    NetHandleFireWeapon(condescriptor, arg);
		    break;

                case NET_CMD_SETNAME:
		    /* Server dosen't handle this... atleast not yet. */
		    break;

                case NET_CMD_SETWEAPONVAL:
                    /* Server dosen't handle this. */
                    break;

                case NET_CMD_SETOBJLOCAPPS:
                    NetHandleSetObjAppS(condescriptor, arg);
                    break;

	    }
	}




	return(0);

}
