/*-
# X-BASED RUBIK'S CUBE(tm)
#
#  Rubik2d.c
#
###
#
#  Copyright (c) 1994 - 99	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Rubik2d */

#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/CoreP.h>
#include "RubikP.h"
#include "Rubik2dP.h"

static void InitializeRubik2D(Widget request, Widget renew);
static void ResizeRubik2D(Rubik2DWidget w);
static void ExposeRubik2D(Widget renew, XEvent * event, Region region);
static Boolean SetValuesRubik2D(Widget current, Widget request, Widget renew);
static void MoveRubik2DTop(Rubik2DWidget w,
			   XEvent * event, char **args, int nArgs);
static void MoveRubik2DLeft(Rubik2DWidget w,
			    XEvent * event, char **args, int nArgs);
static void MoveRubik2DRight(Rubik2DWidget w,
			     XEvent * event, char **args, int nArgs);
static void MoveRubik2DBottom(Rubik2DWidget w,
			      XEvent * event, char **args, int nArgs);
static void ResizePolyhedrons(Rubik2DWidget w);
static void DrawFrame(Rubik2DWidget w, GC gc);
static void DrawOrientLine(Rubik2DWidget w, int orient, int dx, int dy, GC borderGC, int offset);

static char defaultTranslationsRubik2D[] =
"<KeyPress>q: Quit()\n\
   Ctrl<KeyPress>C: Quit()\n\
   <KeyPress>KP_Divide: MoveCcw()\n\
   <KeyPress>Up: MoveTop()\n\
   <KeyPress>KP_8: MoveTop()\n\
   <KeyPress>R8: MoveTop()\n\
   <KeyPress>Left: MoveLeft()\n\
   <KeyPress>KP_4: MoveLeft()\n\
   <KeyPress>R10: MoveLeft()\n\
   <KeyPress>Begin: MoveCw()\n\
   <KeyPress>KP_5: MoveCw()\n\
   <KeyPress>R11: MoveCw()\n\
   <KeyPress>Right: MoveRight()\n\
   <KeyPress>KP_6: MoveRight()\n\
   <KeyPress>R12: MoveRight()\n\
   <KeyPress>Down: MoveBottom()\n\
   <KeyPress>KP_2: MoveBottom()\n\
   <KeyPress>R14: MoveBottom()\n\
   <Btn1Down>: Select()\n\
   <Btn1Up>: Release()\n\
   <KeyPress>p: Practice()\n\
   <Btn2Down>(2+): Practice()\n\
   <Btn2Down>: PracticeMaybe()\n\
   <KeyPress>r: Randomize()\n\
   <Btn3Down>(2+): Randomize()\n\
   <Btn3Down>: RandomizeMaybe()\n\
   <KeyPress>g: Get()\n\
   <KeyPress>w: Write()\n\
   <KeyPress>u: Undo()\n\
   <KeyPress>s: Solve()\n\
   <KeyPress>d: Decrement()\n\
   <KeyPress>i: Increment()\n\
   <KeyPress>x: IncrementX()\n\
   <KeyPress>y: IncrementY()\n\
   <KeyPress>z: IncrementZ()\n\
   <KeyPress>o: Orientize()";

static XtActionsRec actionsListRubik2D[] =
{
	{"Quit", (XtActionProc) QuitRubik},
	{"MoveCcw", (XtActionProc) MoveRubikCcw},
	{"MoveTop", (XtActionProc) MoveRubik2DTop},
	{"MoveLeft", (XtActionProc) MoveRubik2DLeft},
	{"MoveCw", (XtActionProc) MoveRubikCw},
	{"MoveRight", (XtActionProc) MoveRubik2DRight},
	{"MoveBottom", (XtActionProc) MoveRubik2DBottom},
	{"Select", (XtActionProc) SelectRubik},
	{"Release", (XtActionProc) ReleaseRubik},
	{"Practice", (XtActionProc) PracticeRubik},
	{"PracticeMaybe", (XtActionProc) PracticeRubikMaybe},
	{"Randomize", (XtActionProc) RandomizeRubik},
	{"RandomizeMaybe", (XtActionProc) RandomizeRubikMaybe},
	{"Get", (XtActionProc) GetRubik},
	{"Write", (XtActionProc) WriteRubik},
	{"Undo", (XtActionProc) UndoRubik},
	{"Solve", (XtActionProc) SolveRubik},
	{"Decrement", (XtActionProc) DecrementRubik},
	{"Increment", (XtActionProc) IncrementRubik},
	{"IncrementX", (XtActionProc) IncrementXRubik},
	{"IncrementY", (XtActionProc) IncrementYRubik},
	{"IncrementZ", (XtActionProc) IncrementZRubik},
	{"Orientize", (XtActionProc) OrientizeRubik}
};

static XtResource resourcesRubik2D[] =
{
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.username), XtRString, "nobody"},
	{XtNfaceColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[0]), XtRString, "Red"},
	{XtNfaceColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[1]), XtRString, "Yellow"},
	{XtNfaceColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[2]), XtRString, "White"},
	{XtNfaceColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[3]), XtRString, "Green"},
	{XtNfaceColor4, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[4]), XtRString, "Orange"},
	{XtNfaceColor5, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[5]), XtRString, "Blue"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
    XtOffset(RubikWidget, rubik.foreground), XtRString, XtDefaultForeground},
	{XtNpieceBorder, XtCColor, XtRPixel, sizeof (Pixel),
   XtOffset(RubikWidget, rubik.borderColor), XtRString, XtDefaultForeground},
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(RubikWidget, core.width), XtRString, "300"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(RubikWidget, core.height), XtRString, "400"},
	{XtNsizex, XtCSizeX, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.sizex), XtRString, "3"},	/* DEFAULTCUBES */
	{XtNsizey, XtCSizeY, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.sizey), XtRString, "3"},	/* DEFAULTCUBES */
	{XtNsizez, XtCSizeZ, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.sizez), XtRString, "3"},	/* DEFAULTCUBES */
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.orient), XtRString, "FALSE"},	/* DEFAULTORIENT */
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.mono), XtRString, "FALSE"},
	{XtNreverse, XtCReverse, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.reverse), XtRString, "FALSE"},
	{XtNface, XtCFace, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.currentFace), XtRString, "-1"},
	{XtNpos, XtCPos, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.currentPosition), XtRString, "-1"},
	{XtNdirection, XtCDirection, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.currentDirection), XtRString, "-1"},
	{XtNpractice, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.practice), XtRString, "FALSE"},
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.started), XtRString, "FALSE"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(RubikWidget, rubik.select), XtRCallback, NULL}
};

Rubik2DClassRec rubik2dClassRec =
{
	{
		(WidgetClass) & rubikClassRec,	/* superclass */
		"Rubik2D",	/* class name */
		sizeof (Rubik2DRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializeRubik2D,		/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListRubik2D,	/* actions */
		XtNumber(actionsListRubik2D),	/* num actions */
		resourcesRubik2D,	/* resources */
		XtNumber(resourcesRubik2D),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		NULL,		/* destroy */
		(XtWidgetProc) ResizeRubik2D,	/* resize */
		(XtExposeProc) ExposeRubik2D,	/* expose */
		(XtSetValuesFunc) SetValuesRubik2D,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		XtInheritAcceptFocus,	/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsRubik2D,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	},
	{
		0		/* ignore */
	}
};

WidgetClass rubik2dWidgetClass = (WidgetClass) & rubik2dClassRec;

static RowNext rotateToRow[MAXFACES] =	/*CW to min face */
{
	{1, LEFT, TOP},
	{0, BOTTOM, RIGHT},
	{0, RIGHT, BOTTOM},
	{0, TOP, LEFT},
	{1, RIGHT, BOTTOM},
	{0, LEFT, TOP}
};
static int  planeToCube[MAXRECT] =
{6, 0, 6, 1, 2, 3, 6, 4, 6, 6, 5, 6};
static int  cubeToPlane[MAXFACES] =
{1, 3, 4, 5, 7, 10};

static void
InitializeRubik2D(Widget request, Widget renew)
{
	Rubik2DWidget w = (Rubik2DWidget) renew;

	w->rubik.dim = 2;
	ResizeRubik2D(w);
}

static void
ResizeCubie(Rubik2DWidget w)
{
	XPoint      flatlength, flatlengthh;
	int         tempLength;

	w->rubik.delta = 3;
	flatlength.x = w->rubik.sizex + 2 * w->rubik.sizez;
	flatlength.y = 2 * w->rubik.sizey + 2 * w->rubik.sizez;
	flatlengthh.x = 2 * w->rubik.sizex + 2 * w->rubik.sizez;
	flatlengthh.y = w->rubik.sizey + 2 * w->rubik.sizez;
	w->rubik.vertical =	/* w->core.height >= w->core.width */
		(w->core.height / flatlength.y + w->core.width / flatlength.x <=
	     w->core.height / flatlengthh.y + w->core.width / flatlengthh.x);
	if (!w->rubik.vertical)
		flatlength = flatlengthh;
	tempLength = MIN((w->core.height - 2 * w->rubik.delta - 2) / flatlength.y,
		    (w->core.width - 2 * w->rubik.delta - 2) / flatlength.x);
	w->rubik2d.cubeLength = MAX(tempLength - w->rubik.delta + 1, 0);
	w->rubik2d.faceLengthx = w->rubik.sizex * (w->rubik2d.cubeLength +
						   w->rubik.delta - 1);
	w->rubik2d.faceLengthy = w->rubik.sizey * (w->rubik2d.cubeLength +
						   w->rubik.delta - 1);
	w->rubik2d.faceLengthz = w->rubik.sizez * (w->rubik2d.cubeLength +
						   w->rubik.delta - 1);
	w->rubik2d.viewLengthx = w->rubik2d.faceLengthx + w->rubik.delta - 1;
	w->rubik2d.viewLengthy = w->rubik2d.faceLengthy + w->rubik.delta - 1;
	w->rubik2d.viewLengthz = w->rubik2d.faceLengthz + w->rubik.delta - 1;
}

static void
ResizeRubik2D(Rubik2DWidget w)
{
	ResizeCubie(w);
	if (w->rubik.vertical) {
		w->rubik.puzzleSize.x = (w->rubik2d.viewLengthx - 1) +
			2 * (w->rubik2d.viewLengthz - 1) + w->rubik.delta;
		w->rubik.puzzleSize.y = 2 * (w->rubik2d.viewLengthy - 1) +
			2 * (w->rubik2d.viewLengthz - 1) + w->rubik.delta;
	} else {
		w->rubik.puzzleSize.x = 2 * (w->rubik2d.viewLengthx - 1) +
			2 * (w->rubik2d.viewLengthz - 1) + w->rubik.delta;
		w->rubik.puzzleSize.y = (w->rubik2d.viewLengthy - 1) +
			2 * (w->rubik2d.viewLengthz - 1) + w->rubik.delta;
	}
	w->rubik.puzzleOffset.x = ((int) w->core.width - w->rubik.puzzleSize.x)
		/ 2;
	w->rubik.puzzleOffset.y = ((int) w->core.height - w->rubik.puzzleSize.y)
		/ 2;
	ResizePolyhedrons(w);
}

static void
ExposeRubik2D(Widget renew, XEvent * event, Region region)
{
	Rubik2DWidget w = (Rubik2DWidget) renew;

	if (w->core.visible) {
		if (w->rubik.reverse)
			XFillRectangle(XtDisplay(w), XtWindow(w),
				       w->rubik.inverseGC, 0, 0, w->core.width, w->core.height);
		DrawFrame(w, w->rubik.puzzleGC);
		DrawAllPolyhedrons((RubikWidget) w);
	}
}

static      Boolean
SetValuesRubik2D(Widget current, Widget request, Widget renew)
{
	Rubik2DWidget c = (Rubik2DWidget) current, w = (Rubik2DWidget) renew;
	Boolean     redraw = False;

	if (w->rubik.sizex != c->rubik.sizex ||
	    w->rubik.sizey != c->rubik.sizey ||
	    w->rubik.sizez != c->rubik.sizez) {
		ResetPolyhedrons((RubikWidget) w);
		ResizeRubik2D(w);
		redraw = True;
	}
	if (w->rubik2d.cubeLength != c->rubik2d.cubeLength) {
		ResizeRubik2D(w);
		redraw = True;
	}
	return (redraw);
}

static void
MoveRubik2DTop(Rubik2DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveRubikInput((RubikWidget) w, event->xbutton.x, event->xbutton.y, TOP,
		       (int) (event->xkey.state & ControlMask));
}

static void
MoveRubik2DLeft(Rubik2DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveRubikInput((RubikWidget) w, event->xbutton.x, event->xbutton.y, LEFT,
		       (int) (event->xkey.state & ControlMask));
}

static void
MoveRubik2DRight(Rubik2DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveRubikInput((RubikWidget) w, event->xbutton.x, event->xbutton.y, RIGHT,
		       (int) (event->xkey.state & ControlMask));
}

static void
MoveRubik2DBottom(Rubik2DWidget w, XEvent * event, char **args, int nArgs)
{
	MoveRubikInput((RubikWidget) w, event->xbutton.x, event->xbutton.y, BOTTOM,
		       (int) (event->xkey.state & ControlMask));
}

static void
ResizePolyhedrons(Rubik2DWidget w)
{
	ResizeCubie(w);
	w->rubik.orientLineLength = w->rubik2d.cubeLength / 4;
	w->rubik.letterOffset.x = -2;
	w->rubik.letterOffset.y = 4;
}

static void
FacePosition(Rubik2DWidget w, int faceX, int faceY,
	     int *viewLengthx, int *viewLengthy)
{
	int         i;

	*viewLengthx = w->rubik.puzzleOffset.x;
	*viewLengthy = w->rubik.puzzleOffset.y;
	for (i = 0; i < faceX; i++) {
		if (i & 1) {
			*viewLengthx += w->rubik2d.viewLengthx;
		} else {
			*viewLengthx += w->rubik2d.viewLengthz;
		}
	}
	for (i = 0; i < faceY; i++) {
		if (i & 1) {
			*viewLengthy += w->rubik2d.viewLengthy;
		} else {
			*viewLengthy += w->rubik2d.viewLengthz;
		}
	}
}

Boolean
SelectPolyhedrons2D(Rubik2DWidget w, int x, int y, int *face, int *position)
{
	int         faceX, faceY, i, j, sizeOfRow, sizeOfColumn, tempLength;

	x -= w->rubik.puzzleOffset.x;
	y -= w->rubik.puzzleOffset.y;
	/* faceX = x / w->rubik2d.viewLength; */
	for (faceX = 0; faceX <= MAXY; faceX++) {
		if (faceX & 1)
			tempLength = w->rubik2d.viewLengthx;
		else
			tempLength = w->rubik2d.viewLengthz;
		if (x <= tempLength)
			break;
		else
			x -= tempLength;
	}
	/* faceY = y / w->rubik2d.viewLength; */
	for (faceY = 0; faceY <= MAXY; faceY++) {
		if (faceY & 1)
			tempLength = w->rubik2d.viewLengthy;
		else
			tempLength = w->rubik2d.viewLengthz;
		if (y <= tempLength)
			break;
		else
			y -= tempLength;
	}
	i = MAX((x - w->rubik.delta) /
		(w->rubik2d.cubeLength + w->rubik.delta - 1), 0);
	j = MAX((y - w->rubik.delta) /
		(w->rubik2d.cubeLength + w->rubik.delta - 1), 0);
	if ((faceX != 1 && faceY != 1) ||
	    (faceX >= 3 && w->rubik.vertical) ||
	    (faceY >= 3 && !w->rubik.vertical))
		return False;
	if (faceX == 3)
		*face = MAXFACES - 1;
	else
		*face = planeToCube[faceX + faceY * MAXX];
	faceSizes((RubikWidget) w, *face, &sizeOfRow, &sizeOfColumn);
	if (i >= sizeOfRow)
		i = sizeOfRow - 1;
	if (j >= sizeOfColumn)
		j = sizeOfColumn - 1;
	if (faceX == 3) {
		i = sizeOfRow - 1 - i;
		j = sizeOfColumn - 1 - j;
	}
	*position = j * sizeOfRow + i;
	return True;
}

Boolean
NarrowSelection2D(Rubik2DWidget w, int *face, int *position, int *direction)
{
	if (*face == MAXFACES - 1 && *direction < MAXORIENT && !w->rubik.vertical)
		*direction = (*direction + HALF) % MAXORIENT;
	/* Remap to row movement */
	if (*direction == CW || *direction == CCW) {
		int         i, j, sizeOfRow, sizeOfColumn, newFace;

		newFace = rotateToRow[*face].face;
		faceSizes((RubikWidget) w, newFace, &sizeOfRow, &sizeOfColumn);
		*direction = (*direction == CCW) ?
			(rotateToRow[*face].direction + 2) % MAXORIENT :
			rotateToRow[*face].direction;
		if (rotateToRow[*face].sideFace == LEFT ||
		    rotateToRow[*face].sideFace == BOTTOM) {
			i = sizeOfRow - 1;
			j = sizeOfColumn - 1;
		} else {
			i = j = 0;
		}
		*face = newFace;
		*position = j * sizeOfRow + i;
	}
	return True;
}

static void
DrawFrame(Rubik2DWidget w, GC gc)
{
	int         i;
	XPoint      tempLength;
	XPoint      pos[MAXXY + 1], letters;

	tempLength.x = w->rubik.puzzleOffset.x;
	tempLength.y = w->rubik.puzzleOffset.y;
	for (i = 0; i <= MAXY; i++) {
		pos[i] = tempLength;
		if (i & 1) {
			tempLength.x += w->rubik2d.viewLengthx;
			tempLength.y += w->rubik2d.viewLengthy;
		} else {
			tempLength.x += w->rubik2d.viewLengthz;
			tempLength.y += w->rubik2d.viewLengthz;
		}
	}
	XDrawLine(XtDisplay(w), XtWindow(w), gc,
		  pos[1].x, pos[0].y, pos[2].x, pos[0].y);
	XDrawLine(XtDisplay(w), XtWindow(w), gc,
		  pos[3].x, pos[1].y, pos[3].x, pos[2].y);
	XDrawLine(XtDisplay(w), XtWindow(w), gc,
		  pos[1].x, pos[3].y, pos[2].x, pos[3].y);
	XDrawLine(XtDisplay(w), XtWindow(w), gc,
		  pos[0].x, pos[1].y, pos[0].x, pos[2].y);
	letters.x = (pos[0].x + pos[1].x) / 2 - w->rubik.delta;
	letters.y = (pos[0].y + pos[1].y) / 2;
	XDrawString(XtDisplay(w), XtWindow(w), gc,
		    (int) (letters.x + 5 * w->rubik.letterOffset.x),
		    (int) (letters.y + w->rubik.letterOffset.y), "Front", 5);
	letters.x = (pos[2].x + pos[3].x) / 2 - w->rubik.delta;
	letters.y = (pos[2].y + pos[3].y) / 2;
	XDrawString(XtDisplay(w), XtWindow(w), gc,
		    (int) (letters.x + 4 * w->rubik.letterOffset.x),
		    (int) (letters.y + w->rubik.letterOffset.y), "Back", 4);
	if (w->rubik.vertical) {
		XDrawLine(XtDisplay(w), XtWindow(w), gc,
			  pos[1].x, pos[0].y, pos[1].x, pos[4].y);
		XDrawLine(XtDisplay(w), XtWindow(w), gc,
			  pos[2].x, pos[0].y, pos[2].x, pos[4].y);
		XDrawLine(XtDisplay(w), XtWindow(w), gc,
			  pos[0].x, pos[1].y, pos[3].x, pos[1].y);
		XDrawLine(XtDisplay(w), XtWindow(w), gc,
			  pos[0].x, pos[2].y, pos[3].x, pos[2].y);
		XDrawLine(XtDisplay(w), XtWindow(w), gc,
			  pos[1].x, pos[4].y, pos[2].x, pos[4].y);
	} else {
		XDrawLine(XtDisplay(w), XtWindow(w), gc,
			  pos[0].x, pos[1].y, pos[4].x, pos[1].y);
		XDrawLine(XtDisplay(w), XtWindow(w), gc,
			  pos[0].x, pos[2].y, pos[4].x, pos[2].y);
		XDrawLine(XtDisplay(w), XtWindow(w), gc,
			  pos[1].x, pos[0].y, pos[1].x, pos[3].y);
		XDrawLine(XtDisplay(w), XtWindow(w), gc,
			  pos[2].x, pos[0].y, pos[2].x, pos[3].y);
		XDrawLine(XtDisplay(w), XtWindow(w), gc,
			  pos[4].x, pos[1].y, pos[4].x, pos[2].y);
	}
}

void
DrawSquare2D(Rubik2DWidget w, int face, int position, int offset)
{
	GC          faceGC, borderGC;
	int         dx, dy, orient, i, j, sizeOfRow, sizeOfColumn;

	faceSizes((RubikWidget) w, face, &sizeOfRow, &sizeOfColumn);
	i = position % sizeOfRow;
	j = position / sizeOfRow;
	orient = w->rubik.cubeLoc[face][position].rotation;
	if (w->rubik.vertical || face != MAXFACES - 1) {
		FacePosition(w,
			  cubeToPlane[face] % MAXX, cubeToPlane[face] / MAXX,
			     &dx, &dy);
		dx += i * (w->rubik2d.cubeLength + w->rubik.delta - 1);
		dy += j * (w->rubik2d.cubeLength + w->rubik.delta - 1);
	} else {
		FacePosition(w,
			  cubeToPlane[face] / MAXX, cubeToPlane[face] % MAXX,
			     &dx, &dy);
		dx += (sizeOfRow - 1 - i) *
			(w->rubik2d.cubeLength + w->rubik.delta - 1);
		dy += (sizeOfColumn - 1 - j) *
			(w->rubik2d.cubeLength + w->rubik.delta - 1);
		orient = (orient + HALF) % STRT;
	}
	dx += w->rubik.delta - 1;
	dy += w->rubik.delta - 1;
	if (offset) {
		borderGC = w->rubik.faceGC[(int) w->rubik.cubeLoc[face][position].face];
		if (w->rubik.depth < 2 || w->rubik.mono) {
			faceGC = w->rubik.inverseGC;
		} else {
			faceGC = w->rubik.borderGC;
		}
	} else {
		faceGC = w->rubik.faceGC[(int) w->rubik.cubeLoc[face][position].face];
		borderGC = w->rubik.borderGC;
	}
	XFillRectangle(XtDisplay(w), XtWindow(w),
	       faceGC, dx, dy, w->rubik2d.cubeLength, w->rubik2d.cubeLength);
	XDrawRectangle(XtDisplay(w), XtWindow(w),
	     borderGC, dx, dy, w->rubik2d.cubeLength, w->rubik2d.cubeLength);
	if (w->rubik.depth < 2 || w->rubik.mono) {
		int         letterX, letterY;
		char        buf[2];

		(void) sprintf(buf, "%c",
			       w->rubik.faceName[(int) w->rubik.cubeLoc[face][position].face][0]);
		letterX = dx + w->rubik2d.cubeLength / 2 + w->rubik.letterOffset.x;
		letterY = dy + w->rubik2d.cubeLength / 2 + w->rubik.letterOffset.y;
		if (offset) {
			borderGC = w->rubik.borderGC;
		} else {
			borderGC = w->rubik.inverseGC;
		}
		XDrawString(XtDisplay(w), XtWindow(w), borderGC,
			    letterX, letterY, buf, 1);
	}
	if (w->rubik.orient)
		DrawOrientLine(w, orient, dx, dy, borderGC, offset);
}

static void
DrawOrientLine(Rubik2DWidget w, int orient, int dx, int dy, GC borderGC, int offset)
{
	switch (orient) {
		case TOP:
			XDrawLine(XtDisplay(w), XtWindow(w), borderGC,
				  dx + w->rubik2d.cubeLength / 2,
				  dy,
				  dx + w->rubik2d.cubeLength / 2,
				  dy + w->rubik.orientLineLength);
			return;
		case RIGHT:
			XDrawLine(XtDisplay(w), XtWindow(w), borderGC,
				  dx + w->rubik2d.cubeLength,
				  dy + w->rubik2d.cubeLength / 2,
				  dx + w->rubik2d.cubeLength - w->rubik.orientLineLength -
				  1,
				  dy + w->rubik2d.cubeLength / 2);
			return;
		case BOTTOM:
			XDrawLine(XtDisplay(w), XtWindow(w), borderGC,
				  dx + w->rubik2d.cubeLength / 2,
				  dy + w->rubik2d.cubeLength,
				  dx + w->rubik2d.cubeLength / 2,
				  dy + w->rubik2d.cubeLength - w->rubik.orientLineLength -
				  1);
			return;
		case LEFT:
			XDrawLine(XtDisplay(w), XtWindow(w), borderGC,
				  dx,
				  dy + w->rubik2d.cubeLength / 2,
				  dx + w->rubik.orientLineLength,
				  dy + w->rubik2d.cubeLength / 2);
			return;
		default:
			(void) printf("DrawOrientLine: orient %d\n", orient);
	}
}
